/*
 * GNUsound - a sound editor for GNOME.
 * Copyright (C) 2002-2004  Pascal Haakmat <a.haakmat@chello.nl>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/** 
 * @file
 * Import plugin for gmerlin_avdecoder. It supports some more exotic
 * formats like AAC and Musepack.
 */

#include <gnusound.h>

#ifndef HAVE_GMERLIN_AVDEC
#warning "Not building gmerlin_avdec support."
#else

#include <gmerlin/avdec.h>

struct gmerlin_data {
bgav_t * bgav;                /* Decoder instance */
gavl_audio_converter_t * cnv; /* For doing some trivial conversions */
gavl_audio_format_t input_format; /* Format delivered by the decoder */
gavl_audio_format_t output_format; /* Format required by gnusound */
gavl_audio_frame_t * input_frame;
gavl_audio_frame_t * output_frame;
int do_convert;
/*
 * Stream descriptions are read after the file is closed, so we save them
 * here
 */
char * description;
char * audio_description;

};

static struct file_driver gmerlin_driver;

static struct cmd_value *
gmerlin_attach(struct file *file,
              const char *format) {
    struct gmerlin_data *gmerlin_data = mem_calloc(1, sizeof(*gmerlin_data));
    
    if(!gmerlin_data)
        return cmd_new_error_val("Could not allocate parameters");

    file->driver = &gmerlin_driver;
    file->driver_data = gmerlin_data;

    return cmd_new_void_val();
}

static struct cmd_value *
gmerlin_open_for_reading(struct file *file,
                        struct file_params *params) {
    int result;
    gavl_time_t duration;
    const char * desc; /* Description string */
    struct gmerlin_data *gmerlin_data = file->driver_data;
    const gavl_audio_format_t * format;
    /* Create stuff */

    gmerlin_data->bgav = bgav_create();
    gmerlin_data->cnv = gavl_audio_converter_create();

    /* Try to open the file */
        
    result = bgav_open(gmerlin_data->bgav, file->name);
    if(!result) {
        DEBUG("bgav_open failed: %s\n", bgav_get_error(gmerlin_data->bgav));
        return cmd_new_error_val("bgav_open failed: %s\n",
                                 bgav_get_error(gmerlin_data->bgav));
    }

    /*
     *  Now, we can in theory have an arbitrary number of tracks, each
     *  of which contains an arbitrary number of A/V streams. We choose the
     *  first audio stream and the first track which should be correct in
     *  99.9 % of the cases
     */
    
    if(!bgav_num_tracks(gmerlin_data->bgav)) {
        DEBUG("no tracks\n");
        bgav_close(gmerlin_data->bgav);
        return cmd_new_error_val("File contains no tracks");
    }

    if(!bgav_num_audio_streams(gmerlin_data->bgav, 0)) {
        DEBUG("no supported audio streams\n");
        bgav_close(gmerlin_data->bgav);
        return cmd_new_error_val("File contains no supported audio streams");
    }
    
    bgav_select_track(gmerlin_data->bgav, 0);
    bgav_set_audio_stream(gmerlin_data->bgav, 0, BGAV_STREAM_DECODE);

    /* Start codec so we know the precise format */
    bgav_start(gmerlin_data->bgav);

    /* Get the audio format */
    format = bgav_get_audio_format(gmerlin_data->bgav, 0);

    gavl_audio_format_copy(&(gmerlin_data->input_format), format);
    gavl_audio_format_copy(&(gmerlin_data->output_format), format);

    /* Adjust output format */
    gmerlin_data->output_format.interleave_mode = GAVL_INTERLEAVE_ALL;

    switch(gmerlin_data->input_format.sample_format)
      {
      case GAVL_SAMPLE_U8:
        params->sample_type = SAMPLE_TYPE_INT_8;
        gmerlin_data->output_format.sample_format = GAVL_SAMPLE_S8;
        break;
      case GAVL_SAMPLE_S8:
        params->sample_type = SAMPLE_TYPE_INT_8;
        break;
      case GAVL_SAMPLE_U16:
        params->sample_type = SAMPLE_TYPE_INT_16;
        gmerlin_data->output_format.sample_format = GAVL_SAMPLE_S16;
        break;
      case GAVL_SAMPLE_S16:
        params->sample_type = SAMPLE_TYPE_INT_16;
        break;
      case GAVL_SAMPLE_S32:
        params->sample_type = SAMPLE_TYPE_INT_32;
        break;
      case GAVL_SAMPLE_FLOAT:
        params->sample_type = SAMPLE_TYPE_FLOAT_32;
        break;
      case GAVL_SAMPLE_NONE:
        break; /* Should never happen */
      }

    /* Set remaining params */
    
    params->sample_rate = gmerlin_data->input_format.samplerate;
    params->channels    = gmerlin_data->input_format.num_channels;

    duration = bgav_get_duration(gmerlin_data->bgav, 0);
    if(duration == GAVL_TIME_UNDEFINED)
      params->frame_count = -1;
    else
      params->frame_count =
        gavl_time_to_samples(params->sample_rate, duration);

    /* Check if we must convert */

    if(gavl_audio_converter_init(gmerlin_data->cnv,
                                 &(gmerlin_data->input_format),
                                 &(gmerlin_data->output_format)) > 0)
      {
      gmerlin_data->do_convert = 1;
      }
    else
      {
      gmerlin_data->do_convert = 0;
      }

    /* Since we don't know yet, how many samples we should decode
       at once, we set the samples_per_frame of the output format
       to 0 */

    gmerlin_data->output_format.samples_per_frame = 0;

    /* The output frame will be fed with gnusound supplied memory,
       so we can create it right now */

    gmerlin_data->output_frame = gavl_audio_frame_create(NULL);

    /* Get the descriptions */
    desc = bgav_get_description(gmerlin_data->bgav);
    if(desc)
      gmerlin_data->description = strdup(desc);
    else
      gmerlin_data->description = strdup("Unspecified");
    
    desc = bgav_get_audio_description(gmerlin_data->bgav, 0);
    if(desc)
      gmerlin_data->audio_description = strdup(desc);
    else
      gmerlin_data->audio_description = strdup("Unspecified");
    fprintf(stderr, "Description: %s\n", gmerlin_data->description);
    fprintf(stderr, "Audio description: %s\n", gmerlin_data->audio_description);
    
    return cmd_new_void_val();
}

static struct cmd_value *
gmerlin_open(struct file *file,
            const char *mode,
            struct file_params *params) {
    if(mode[0] == 'r')
        return gmerlin_open_for_reading(file, params);
    else if(mode[0] == 'w')
        return cmd_new_error_val("Not implemented");
    else
        return cmd_new_error_val("Unknown mode %s", mode);
}

static long
gmerlin_read(struct file *file,
            void *buf,
            long count) {
    struct gmerlin_data *gmerlin_data = file->driver_data;

    gmerlin_data->output_frame->samples.s_8 = buf;
    if(gmerlin_data->do_convert)
      {
      /* (Re)allocate temporary frame if necessary */
            
      if(count > gmerlin_data->output_format.samples_per_frame)
        {
        if(gmerlin_data->input_frame)
          gavl_audio_frame_destroy(gmerlin_data->input_frame);
        gmerlin_data->output_format.samples_per_frame = count + 128;
        gmerlin_data->input_frame =
          gavl_audio_frame_create(&(gmerlin_data->output_format));
        }

      /* Read data and convert */
      if(!bgav_read_audio(gmerlin_data->bgav,
                          gmerlin_data->input_frame,
                          0, count))
        return 0;
      gavl_audio_convert(gmerlin_data->cnv, gmerlin_data->input_frame,
                         gmerlin_data->output_frame);
      }
    else
      {
      if(!bgav_read_audio(gmerlin_data->bgav,
                          gmerlin_data->output_frame,
                          0, count))
        return 0;
      }
    
    return gmerlin_data->output_frame->valid_samples;
}

static long
gmerlin_write(struct file *file,
             void *buf,
             long count) {
    return -1;
}

static struct cmd_value *
gmerlin_close(struct file *file) {
    struct gmerlin_data *gmerlin_data = file->driver_data;

    bgav_close(gmerlin_data->bgav);
    gavl_audio_converter_destroy(gmerlin_data->cnv);
    if(gmerlin_data->input_frame)
      gavl_audio_frame_destroy(gmerlin_data->input_frame);
    if(gmerlin_data->output_frame)
      {
      gavl_audio_frame_null(gmerlin_data->output_frame);
      gavl_audio_frame_destroy(gmerlin_data->output_frame);
      }
    return cmd_new_void_val();
}

static void 
gmerlin_detach(struct file *file) {
   struct gmerlin_data *gmerlin_data = file->driver_data;
   if(gmerlin_data->description)
     free(gmerlin_data->description);
   if(gmerlin_data->audio_description)
     free(gmerlin_data->audio_description);
   
   mem_free(file->driver_data);
   file->driver = NULL;
}

static int
gmerlin_snprint(struct file *file,
               enum file_property what,
               char *buf,
               int buflen) {
    struct gmerlin_data *gmerlin_data = file->driver_data;

    switch(what) {
    case FILE_FORMAT:
      return snprintf(buf, buflen, "%s", gmerlin_data->description);
      break;
    case FILE_DETAILED_FORMAT:
      return snprintf(buf, buflen, "Format: %s, codec: %s",
                      gmerlin_data->description,
                      gmerlin_data->audio_description);
      
      break;
    default:
        return -1;
    }
}

static const struct file_format *
gmerlin_get_read_formats() {
    return NULL;
}

static const struct file_format *
gmerlin_get_write_formats() {
    return NULL;
}

static struct file_driver gmerlin_driver = {
    "gmerlin_avdecoder",

    gmerlin_attach,
    gmerlin_open,
    gmerlin_read,
    gmerlin_write,
    gmerlin_close,
    gmerlin_detach,
    
    gmerlin_snprint,

    gmerlin_get_read_formats,
    gmerlin_get_write_formats,

    NULL,
    NULL,
    NULL,

    NULL,
    NULL,
    NULL
};

static int
gmerlin_module_init(int id) {
    file_register_driver(&gmerlin_driver);
    return 0;
}

struct gnusound_module manifest = {
    MODULE_MAGIC,
    MODULE_API_VERSION_4,
    "gmerlin_avdecoder import",
    "0.1",
    "Burkhard Plaum",
    "Copyright (C) 2005",
    "GPL",
    NULL,
    MODULE_FLAG_FACELESS,

    gmerlin_module_init,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
};

#endif /* HAVE_GMERLIN_AVDEC */
