// SPDX-FileCopyrightText: Copyright 2015-2025 go-swagger maintainers
// SPDX-License-Identifier: Apache-2.0

package spec

import (
	"encoding/json"
	"fmt"
	"reflect"
	"strconv"
	"strings"

	"github.com/go-openapi/swag/jsonutils"
)

// Responses is a container for the expected responses of an operation.
// The container maps a HTTP response code to the expected response.
// It is not expected from the documentation to necessarily cover all possible HTTP response codes,
// since they may not be known in advance. However, it is expected from the documentation to cover
// a successful operation response and any known errors.
//
// The `default` can be used a default response object for all HTTP codes that are not covered
// individually by the specification.
//
// The `Responses Object` MUST contain at least one response code, and it SHOULD be the response
// for a successful operation call.
//
// For more information: http://goo.gl/8us55a#responsesObject
type Responses struct {
	VendorExtensible
	ResponsesProps
}

// JSONLookup implements an interface to customize json pointer lookup
func (r Responses) JSONLookup(token string) (any, error) {
	if token == "default" {
		return r.Default, nil
	}
	if ex, ok := r.Extensions[token]; ok {
		return &ex, nil
	}
	if i, err := strconv.Atoi(token); err == nil {
		if scr, ok := r.StatusCodeResponses[i]; ok {
			return scr, nil
		}
	}
	return nil, fmt.Errorf("object has no field %q: %w", token, ErrSpec)
}

// UnmarshalJSON hydrates this items instance with the data from JSON
func (r *Responses) UnmarshalJSON(data []byte) error {
	if err := json.Unmarshal(data, &r.ResponsesProps); err != nil {
		return err
	}

	if err := json.Unmarshal(data, &r.VendorExtensible); err != nil {
		return err
	}
	if reflect.DeepEqual(ResponsesProps{}, r.ResponsesProps) {
		r.ResponsesProps = ResponsesProps{}
	}
	return nil
}

// MarshalJSON converts this items object to JSON
func (r Responses) MarshalJSON() ([]byte, error) {
	b1, err := json.Marshal(r.ResponsesProps)
	if err != nil {
		return nil, err
	}
	b2, err := json.Marshal(r.VendorExtensible)
	if err != nil {
		return nil, err
	}
	concated := jsonutils.ConcatJSON(b1, b2)
	return concated, nil
}

// ResponsesProps describes all responses for an operation.
// It tells what is the default response and maps all responses with a
// HTTP status code.
type ResponsesProps struct {
	Default             *Response
	StatusCodeResponses map[int]Response
}

// MarshalJSON marshals responses as JSON
func (r ResponsesProps) MarshalJSON() ([]byte, error) {
	toser := map[string]Response{}
	if r.Default != nil {
		toser["default"] = *r.Default
	}
	for k, v := range r.StatusCodeResponses {
		toser[strconv.Itoa(k)] = v
	}
	return json.Marshal(toser)
}

// UnmarshalJSON unmarshals responses from JSON
func (r *ResponsesProps) UnmarshalJSON(data []byte) error {
	var res map[string]json.RawMessage
	if err := json.Unmarshal(data, &res); err != nil {
		return err
	}

	if v, ok := res["default"]; ok {
		var defaultRes Response
		if err := json.Unmarshal(v, &defaultRes); err != nil {
			return err
		}
		r.Default = &defaultRes
		delete(res, "default")
	}
	for k, v := range res {
		if !strings.HasPrefix(k, "x-") {
			var statusCodeResp Response
			if err := json.Unmarshal(v, &statusCodeResp); err != nil {
				return err
			}
			if nk, err := strconv.Atoi(k); err == nil {
				if r.StatusCodeResponses == nil {
					r.StatusCodeResponses = map[int]Response{}
				}
				r.StatusCodeResponses[nk] = statusCodeResp
			}
		}
	}
	return nil
}
