#include "KeyBag.h"

#include <tc/cli/FormatUtil.h>

ctrtool::KeyBagInitializer::KeyBagInitializer(bool isDev, const tc::Optional<std::string>& fallback_title_key_str, const tc::Optional<tc::io::Path>& seed_db_path, const tc::Optional<std::string>& fallback_seed_str)
{
	// add static data
	addStaticAesKeys(isDev);
	addStaticRsaKeys(isDev);
	addSigHaxSignatures(isDev);

	// import seed database
	if (seed_db_path.isSet())
	{
		auto file_source = std::shared_ptr<tc::io::StreamSource>();
		
		try {
			auto file_stream = std::shared_ptr<tc::io::FileStream>(new tc::io::FileStream(seed_db_path.get(), tc::io::FileMode::Open, tc::io::FileAccess::Read));
			file_source = std::shared_ptr<tc::io::StreamSource>(new tc::io::StreamSource(file_stream));
		}
		catch (tc::io::FileNotFoundException&) {
			throw tc::ArgumentException("ctrtool::KeyBagInitializer", "Failed to open seed database file.");
		}

		importSeedDb(file_source);
	}

	// import fallback keys
	if (fallback_title_key_str.isSet())
	{
		if (importFallbackKey(fallback_title_key, fallback_title_key_str.get()) == false)
		{
			throw tc::ArgumentOutOfRangeException("ctrtool::KeyBagInitializer", "Fallback title key failed to import.");
		}
	}
	if (fallback_seed_str.isSet())
	{
		if (importFallbackKey(fallback_seed, fallback_seed_str.get()) == false)
		{
			throw tc::ArgumentOutOfRangeException("ctrtool::KeyBagInitializer", "Fallback seed failed to import.");
		}
	}
}

void ctrtool::KeyBagInitializer::addStaticAesKeys(bool isDev)
{
	struct DefaultAesKey
	{
		byte_t key_index;
		KeyBag::Aes128Key key;
	};

	static const size_t kDefaultNcchFixedKeyNum = 2;
	const DefaultAesKey default_ncch_fixed_aes_keys[kDefaultNcchFixedKeyNum][2] = 
	{
		{
			{NCCH_APPLICATION_FIXED_KEY, {0}},
			{NCCH_APPLICATION_FIXED_KEY, {0}},
		},
		{
			{NCCH_SYSTEM_FIXED_KEY, {0x52, 0x7c, 0xe6, 0x30, 0xa9, 0xca, 0x30, 0x5f, 0x36, 0x96, 0xf3, 0xcd, 0xe9, 0x54, 0x19, 0x4b}},
			{NCCH_SYSTEM_FIXED_KEY, {0x52, 0x7c, 0xe6, 0x30, 0xa9, 0xca, 0x30, 0x5f, 0x36, 0x96, 0xf3, 0xcd, 0xe9, 0x54, 0x19, 0x4b}},
		}
	};

	static const size_t kDefaultNcchSecureKeyXNum = 4;
	const DefaultAesKey default_ncch_secure_aes_keys_x[kDefaultNcchSecureKeyXNum][2] = 
	{
		{
			// retail
			{NCCH_SECURE_KEY_FW1, {0xb9, 0x8e, 0x95, 0xce, 0xca, 0x3e, 0x4d, 0x17, 0x1f, 0x76, 0xa9, 0x4d, 0xe9, 0x34, 0xc0, 0x53}},
			// dev
			{NCCH_SECURE_KEY_FW1, {0x51, 0x02, 0x07, 0x51, 0x55, 0x07, 0xcb, 0xb1, 0x8e, 0x24, 0x3d, 0xcb, 0x85, 0xe2, 0x3a, 0x1d}},
		},
		{
			// retail
			{NCCH_SECURE_KEY_FW7, {0xce, 0xe7, 0xd8, 0xab, 0x30, 0xc0, 0x0d, 0xae, 0x85, 0x0e, 0xf5, 0xe3, 0x82, 0xac, 0x5a, 0xf3}},
			// dev
			{NCCH_SECURE_KEY_FW7, {0x81, 0x90, 0x7a, 0x4b, 0x6f, 0x1b, 0x47, 0x32, 0x3a, 0x67, 0x79, 0x74, 0xce, 0x4a, 0xd7, 0x1b}},
		},
		{
			// retail
			{NCCH_SECURE_KEY_FW9_3, {0x82, 0xe9, 0xc9, 0xbe, 0xbf, 0xb8, 0xbd, 0xb8, 0x75, 0xec, 0xc0, 0xa0, 0x7d, 0x47, 0x43, 0x74}},
			// dev
			{NCCH_SECURE_KEY_FW9_3, {0x30, 0x4b, 0xf1, 0x46, 0x83, 0x72, 0xee, 0x64, 0x11, 0x5e, 0xbd, 0x40, 0x93, 0xd8, 0x42, 0x76}},
		},
		{
			// retail
			{NCCH_SECURE_KEY_FW9_6, {0x45, 0xad, 0x04, 0x95, 0x39, 0x92, 0xc7, 0xc8, 0x93, 0x72, 0x4a, 0x9a, 0x7b, 0xce, 0x61, 0x82}},
			// dev
			{NCCH_SECURE_KEY_FW9_6, {0x6c, 0x8b, 0x29, 0x44, 0xa0, 0x72, 0x60, 0x35, 0xf9, 0x41, 0xdf, 0xc0, 0x18, 0x52, 0x4f, 0xb6}},
		},
	};

	static const size_t kDefaultCommonKeyNum = 6;
	const DefaultAesKey default_common_aes_keys[kDefaultCommonKeyNum][2] = 
	{
		{
			// retail
			{COMMONKEY_APPLICATION, {0x64, 0xC5, 0xFD, 0x55, 0xDD, 0x3A, 0xD9, 0x88, 0x32, 0x5B, 0xAA, 0xEC, 0x52, 0x43, 0xDB, 0x98}},
			// dev
			{COMMONKEY_APPLICATION, {0x55, 0xA3, 0xF8, 0x72, 0xBD, 0xC8, 0x0C, 0x55, 0x5A, 0x65, 0x43, 0x81, 0x13, 0x9E, 0x15, 0x3B}},
		},
		{
			// retail
			{COMMONKEY_SYSTEM, {0x4A, 0xAA, 0x3D, 0x0E, 0x27, 0xD4, 0xD7, 0x28, 0xD0, 0xB1, 0xB4, 0x33, 0xF0, 0xF9, 0xCB, 0xC8}},
			// dev
			{COMMONKEY_SYSTEM, {0x44, 0x34, 0xED, 0x14, 0x82, 0x0C, 0xA1, 0xEB, 0xAB, 0x82, 0xC1, 0x6E, 0x7B, 0xEF, 0x0C, 0x25}},
		},
		{
			// retail
			{COMMONKEY_UNUSED_2, {0xFB, 0xB0, 0xEF, 0x8C, 0xDB, 0xB0, 0xD8, 0xE4, 0x53, 0xCD, 0x99, 0x34, 0x43, 0x71, 0x69, 0x7F}},
			// dev
			{COMMONKEY_UNUSED_2, {0xF6, 0x2E, 0x3F, 0x95, 0x8E, 0x28, 0xA2, 0x1F, 0x28, 0x9E, 0xEC, 0x71, 0xA8, 0x66, 0x29, 0xDC}},
		},
		{
			// retail
			{COMMONKEY_UNUSED_3, {0x25, 0x95, 0x9B, 0x7A, 0xD0, 0x40, 0x9F, 0x72, 0x68, 0x41, 0x98, 0xBA, 0x2E, 0xCD, 0x7D, 0xC6}},
			// dev
			{COMMONKEY_UNUSED_3, {0x2B, 0x49, 0xCB, 0x6F, 0x99, 0x98, 0xD9, 0xAD, 0x94, 0xF2, 0xED, 0xE7, 0xB5, 0xDA, 0x3E, 0x27}},
		},
		{
			// retail
			{COMMONKEY_UNUSED_4, {0x7A, 0xDA, 0x22, 0xCA, 0xFF, 0xC4, 0x76, 0xCC, 0x82, 0x97, 0xA0, 0xC7, 0xCE, 0xEE, 0xEE, 0xBE}},
			// dev
			{COMMONKEY_UNUSED_4, {0x75, 0x05, 0x52, 0xBF, 0xAA, 0x1C, 0x04, 0x07, 0x55, 0xC8, 0xD5, 0x9A, 0x55, 0xF9, 0xAD, 0x1F}},
		},
		{
			// retail
			{COMMONKEY_UNUSED_5, {0xA5, 0x05, 0x1C, 0xA1, 0xB3, 0x7D, 0xCF, 0x3A, 0xFB, 0xCF, 0x8C, 0xC1, 0xED, 0xD9, 0xCE, 0x02}},
			// dev
			{COMMONKEY_UNUSED_5, {0xAA, 0xDA, 0x4C, 0xA8, 0xF6, 0xE5, 0xA9, 0x77, 0xE0, 0xA0, 0xF9, 0xE4, 0x76, 0xCF, 0x0D, 0x63}},
		}
	};
	
	// ncch fixed keys
	for (size_t i = 0; i < kDefaultNcchFixedKeyNum; i++)
	{
		ncch_fixed_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(default_ncch_fixed_aes_keys[i][isDev].key_index, default_ncch_fixed_aes_keys[i][isDev].key));
	}

	// ncch secure key x
	for (size_t i = 0; i < kDefaultNcchSecureKeyXNum; i++)
	{
		ncch_secure_key_x.insert(std::pair<byte_t, KeyBag::Aes128Key>(default_ncch_secure_aes_keys_x[i][isDev].key_index, default_ncch_secure_aes_keys_x[i][isDev].key));
	}
	
	// ticket common key
	for (size_t i = 0; i < kDefaultCommonKeyNum; i++)
	{
		common_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(default_common_aes_keys[i][isDev].key_index, default_common_aes_keys[i][isDev].key));
	}

	// populate aes keyslots
	enum StaticKeyXSlots
	{
		KEYX_2C_2F,
		KEYX_30_33,
		KEYX_34_37,
		KEYX_38_3B,
		KEYX_3C,
		KEYX_3D,
		KEYX_3E,
		KEYX_3F,
		BROM_KEYX_NUM
	};
	const DefaultAesKey default_brom_static_keyx[BROM_KEYX_NUM][2] = 
	{
		{
			// retail
			{KEYX_2C_2F, {0xB9, 0x8E, 0x95, 0xCE, 0xCA, 0x3E, 0x4D, 0x17, 0x1F, 0x76, 0xA9, 0x4D, 0xE9, 0x34, 0xC0, 0x53}},
			// dev
			{KEYX_2C_2F, {0x51, 0x02, 0x07, 0x51, 0x55, 0x07, 0xCB, 0xB1, 0x8E, 0x24, 0x3D, 0xCB, 0x85, 0xE2, 0x3A, 0x1D}},
		},
		{
			// retail
			{KEYX_30_33, {0xC6, 0x6E, 0x23, 0x12, 0x8F, 0x28, 0x91, 0x33, 0xF0, 0x4C, 0xDB, 0x87, 0x7A, 0x37, 0x49, 0xF2}},
			// dev
			{KEYX_30_33, {0x3F, 0x05, 0x4E, 0x66, 0x3B, 0x3E, 0xF7, 0x28, 0xC8, 0x98, 0x4D, 0x20, 0xC4, 0xAF, 0xD5, 0xA0}},
		},
		{
			// retail
			{KEYX_34_37, {0x6F, 0xBB, 0x01, 0xF8, 0x72, 0xCA, 0xF9, 0xC0, 0x18, 0x34, 0xEE, 0xC0, 0x40, 0x65, 0xEE, 0x53}},
			// dev
			{KEYX_34_37, {0x7B, 0xFB, 0x77, 0xBC, 0xBC, 0x05, 0x9A, 0x06, 0xAC, 0xAD, 0x88, 0xEF, 0x2F, 0xCA, 0xBE, 0xDB}},
		},
		{
			// retail
			{KEYX_38_3B, {0xB5, 0x29, 0x22, 0x1C, 0xDD, 0xB5, 0xDB, 0x5A, 0x1B, 0xF2, 0x6E, 0xFF, 0x20, 0x41, 0xE8, 0x75}},
			// dev
			{KEYX_38_3B, {0x5C, 0x3D, 0x38, 0xAC, 0x17, 0x40, 0x99, 0x4E, 0xFC, 0x8F, 0xD0, 0xBE, 0x8D, 0x80, 0x97, 0xB3}},
		},
		{
			// retail
			{KEYX_3C, {0xC3, 0x5D, 0x6D, 0x15, 0x68, 0x0B, 0x1A, 0xD4, 0xE9, 0x12, 0xA3, 0x41, 0x83, 0x61, 0x21, 0xB3}},
			// dev
			{KEYX_3C, {0x61, 0xBF, 0x11, 0x37, 0x0B, 0x29, 0x2F, 0xFA, 0xB3, 0x88, 0x51, 0xEC, 0x5D, 0xAE, 0x5D, 0xEC}},
		},
		// commonkey
		{
			// retail
			{KEYX_3D, {0x61, 0x70, 0x85, 0x71, 0x9B, 0x7C, 0xFB, 0x31, 0x6D, 0xF4, 0xDF, 0x2E, 0x83, 0x62, 0xC6, 0xE2}},
			// dev
			{KEYX_3D, {0xBD, 0x4F, 0xE7, 0xE7, 0x33, 0xC7, 0x55, 0xFC, 0xE7, 0x54, 0x0E, 0xAB, 0xBD, 0x8A, 0xC3, 0xD3}},
		},
		{
			// retail
			{KEYX_3E, {0x24, 0xBA, 0xF6, 0x28, 0xD0, 0x68, 0x89, 0xBF, 0x28, 0x2D, 0x0A, 0xA3, 0x5D, 0xC5, 0x56, 0x50}},
			// dev
			{KEYX_3E, {0x28, 0x87, 0xA4, 0xD4, 0x28, 0xF6, 0xF2, 0x24, 0xB0, 0x3A, 0xB3, 0x36, 0xE2, 0x2C, 0x61, 0x1E}},
		},
		{
			// retail
			{KEYX_3F, {0xA3, 0x12, 0x33, 0x28, 0x0B, 0xB4, 0xDA, 0xA7, 0x76, 0x13, 0x93, 0xF7, 0x8C, 0x42, 0x49, 0x52}},
			// dev
			{KEYX_3F, {0xBE, 0x66, 0x5D, 0xE6, 0xFB, 0x8C, 0x3F, 0x0A, 0x98, 0x71, 0x96, 0x0A, 0xD7, 0xCF, 0xBE, 0x79}},
		}
	};

	// 0x2C-0x2F
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key_x.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x2C + i, default_brom_static_keyx[KEYX_2C_2F][isDev].key));
	}
	// 0x30-0x33
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key_x.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x30 + i, default_brom_static_keyx[KEYX_30_33][isDev].key));
	}
	// 0x34-0x37
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key_x.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x34 + i, default_brom_static_keyx[KEYX_34_37][isDev].key));
	}
	// 0x38-0x3B
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key_x.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x38 + i, default_brom_static_keyx[KEYX_38_3B][isDev].key));
	}
	// 0x3C
	brom_static_key_x.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x3C, default_brom_static_keyx[KEYX_3C][isDev].key));
	// 0x3D common key
	brom_static_key_x.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x3D, default_brom_static_keyx[KEYX_3D][isDev].key));
	// 0x3E
	brom_static_key_x.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x3E, default_brom_static_keyx[KEYX_3E][isDev].key));
	// 0x3F
	brom_static_key_x.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x3F, default_brom_static_keyx[KEYX_3F][isDev].key));

	enum StaticKeyYSlots
	{
		KEYY_04,
		KEYY_05,
		KEYY_06,
		KEYY_07,
		KEYY_08,
		KEYY_09,
		KEYY_0A,
		KEYY_0B,
		BROM_KEYY_NUM
	};
	const DefaultAesKey default_brom_static_keyy[BROM_KEYY_NUM][2] = 
	{
		{
			// retail
			{KEYY_04, {0xFF, 0x33, 0x88, 0xEC, 0xD2, 0x17, 0x05, 0xBB, 0x33, 0x9E, 0x96, 0x79, 0x86, 0xDC, 0x49, 0x07}},
			// dev
			{KEYY_04, {0x5C, 0xE6, 0xB1, 0xEC, 0x3F, 0x5F, 0x9D, 0x7B, 0x2E, 0x81, 0xE2, 0x21, 0x45, 0xA5, 0x76, 0x8D}},
		},
		{
			// retail
			{KEYY_05, {0x54, 0xEF, 0x03, 0x5F, 0x30, 0x26, 0x0E, 0x0E, 0x9B, 0x5E, 0x00, 0x4F, 0xC9, 0x85, 0xDC, 0x22}},
			// dev
			{KEYY_05, {0x28, 0x5B, 0x2E, 0x23, 0xA3, 0xC3, 0x0E, 0x54, 0x8F, 0x24, 0xEE, 0x8D, 0x28, 0x7C, 0x26, 0x42}},
		},
		{
			// retail
			{KEYY_06, {0x24, 0xB0, 0x5A, 0xAA, 0xAC, 0x0B, 0x09, 0x92, 0x52, 0x03, 0x0C, 0x02, 0xD1, 0x04, 0x03, 0x17}},
			// dev
			{KEYY_06, {0xAE, 0x94, 0xFC, 0x90, 0xBE, 0x6B, 0x6C, 0xD5, 0xF1, 0xB0, 0xCB, 0x55, 0x95, 0x07, 0x22, 0x0E}},
		},
		{
			// retail
			{KEYY_07, {0xE9, 0xAC, 0xC5, 0xAB, 0xD4, 0xAD, 0x3F, 0x06, 0x60, 0xC8, 0x3C, 0x89, 0x34, 0x88, 0x2F, 0x3F}},
			// dev
			{KEYY_07, {0x49, 0x97, 0x4D, 0x47, 0xB6, 0xE2, 0xC9, 0xD9, 0x19, 0x4A, 0x2D, 0x97, 0xFD, 0xF2, 0x83, 0xBE}},
		},
		{
			// retail
			{KEYY_08, {0x48, 0x03, 0x05, 0x01, 0x06, 0xD4, 0x82, 0xDC, 0xD7, 0x5F, 0x85, 0xC5, 0xAA, 0xDF, 0xF9, 0xB3}},
			// dev
			{KEYY_08, {0x25, 0xA6, 0x19, 0x50, 0x4F, 0x07, 0xD0, 0x68, 0x19, 0x03, 0x34, 0xA8, 0x14, 0x09, 0xC2, 0x08}},
		},
		{
			// retail
			{KEYY_09, {0xAF, 0x63, 0x46, 0xEF, 0xDD, 0xDF, 0xA9, 0x80, 0x6E, 0x3C, 0x6B, 0x68, 0x55, 0xB7, 0x89, 0x30}},
			// dev
			{KEYY_09, {0x65, 0x7B, 0xBD, 0x65, 0x2E, 0x8B, 0xF3, 0x0F, 0x37, 0x40, 0xC4, 0x8F, 0xAC, 0x6C, 0xC5, 0x9E}},
		},
		{
			// retail
			{KEYY_0A, {0x0A, 0x87, 0x0A, 0x2C, 0x4B, 0x2F, 0xC3, 0x17, 0x2E, 0x5F, 0x03, 0x35, 0xD8, 0xC5, 0x08, 0x5D}},
			// dev
			{KEYY_0A, {0x14, 0x7A, 0xD1, 0x4A, 0xC2, 0x06, 0xB1, 0x00, 0xE2, 0x00, 0x2A, 0x7B, 0x1A, 0x0D, 0xDD, 0x3D}},
		},
		{
			// retail
			{KEYY_0B, {0xFD, 0xA0, 0x15, 0x2F, 0xCD, 0x6D, 0xDB, 0x31, 0x33, 0xB8, 0x87, 0xBA, 0x72, 0x7C, 0x0A, 0xDA}},
			// dev
			{KEYY_0B, {0xA3, 0xD0, 0x0D, 0x9E, 0x2C, 0x5E, 0xDF, 0x30, 0x86, 0x64, 0x86, 0x61, 0x1C, 0xE0, 0x8D, 0x25}},
		}
	};
	// 0x04
	brom_static_key_y.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x04, default_brom_static_keyy[KEYY_04][isDev].key));
	// 0x05
	brom_static_key_y.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x05, default_brom_static_keyy[KEYY_05][isDev].key));
	// 0x06
	brom_static_key_y.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x06, default_brom_static_keyy[KEYY_06][isDev].key));
	// 0x07
	brom_static_key_y.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x07, default_brom_static_keyy[KEYY_07][isDev].key));
	// 0x08
	brom_static_key_y.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x08, default_brom_static_keyy[KEYY_08][isDev].key));
	// 0x09
	brom_static_key_y.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x09, default_brom_static_keyy[KEYY_09][isDev].key));
	// 0x0A
	brom_static_key_y.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x0A, default_brom_static_keyy[KEYY_0A][isDev].key));
	// 0x0B
	brom_static_key_y.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x0B, default_brom_static_keyy[KEYY_0B][isDev].key));


	enum StaticKeySlots
	{
		KEY_0C_0F,
		KEY_10_13,
		KEY_14,
		KEY_15,
		KEY_16,
		KEY_17,
		KEY_18_1B,
		KEY_1C_1F,
		KEY_20_23,
		KEY_24_27,
		KEY_28,
		KEY_29,
		KEY_2A,
		KEY_2B,
		KEY_2C_2F,
		KEY_30_33,
		KEY_34_37,
		KEY_38_3B,
		KEY_3C,
		KEY_3D,
		KEY_3E,
		KEY_3F,
		BROM_KEY_NUM
	};
	const DefaultAesKey default_brom_static_key[BROM_KEY_NUM][2] = 
	{
		{
			// retail
			{KEY_0C_0F, {0xE7, 0xC9, 0xFF, 0x9D, 0x4F, 0x5B, 0x6F, 0x4D, 0xC5, 0xE2, 0xF5, 0x0E, 0x85, 0x6F, 0x0A, 0xB2}},
			// dev
			{KEY_0C_0F, {0x25, 0xC6, 0x26, 0x59, 0x55, 0xA4, 0xFC, 0x6A, 0xC7, 0xE8, 0x58, 0x08, 0x7B, 0xD3, 0x09, 0x71}},
		},
		{
			// retail
			{KEY_10_13, {0x28, 0x57, 0x13, 0xDB, 0x53, 0x05, 0x1C, 0x08, 0x9B, 0xDF, 0xB3, 0xB6, 0xAA, 0x63, 0x8F, 0xDA}},
			// dev
			{KEY_10_13, {0x29, 0x72, 0xAF, 0xF5, 0x0F, 0xEE, 0x9F, 0x6F, 0x7B, 0x44, 0x3E, 0xC3, 0x4C, 0x7F, 0xDE, 0xAD}},
		},
		{
			// retail
			{KEY_14, {0x2A, 0xF3, 0xBB, 0xD3, 0x2C, 0xD5, 0x9C, 0x06, 0xFD, 0x4A, 0xBE, 0x58, 0x65, 0x19, 0x87, 0xAD}},
			// dev
			{KEY_14, {0x09, 0x71, 0x2F, 0x77, 0xEF, 0x36, 0x7C, 0xC0, 0xC9, 0x68, 0x41, 0x93, 0x8D, 0xC0, 0xB3, 0xA1}},
		},
		{
			// retail
			{KEY_15, {0xBE, 0x28, 0x36, 0x75, 0x1C, 0x73, 0x4B, 0xA8, 0xDA, 0x18, 0xE1, 0x88, 0x7F, 0x88, 0x8B, 0xD6}},
			// dev
			{KEY_15, {0x62, 0xEE, 0x74, 0x6F, 0x91, 0xE0, 0x2B, 0x1D, 0xD2, 0xF1, 0x7D, 0x46, 0xBB, 0xC2, 0xC9, 0xC5}},
		},
		{
			// retail
			{KEY_16, {0xE3, 0xA1, 0x8E, 0xB1, 0xC1, 0xDC, 0x8A, 0x3D, 0x27, 0xC3, 0x96, 0x7E, 0x6E, 0x36, 0x2D, 0xE3}},
			// dev
			{KEY_16, {0x41, 0x97, 0x20, 0xED, 0x83, 0x04, 0xAC, 0x7C, 0x38, 0xE7, 0x30, 0xC6, 0xF4, 0x4E, 0xAC, 0x7C}},
		},
		{
			// retail
			{KEY_17, {0xD0, 0x29, 0x4C, 0xFB, 0x7B, 0xE0, 0xB4, 0xFB, 0x73, 0x24, 0xD9, 0x86, 0xFD, 0x39, 0x93, 0xBB}},
			// dev
			{KEY_17, {0xC8, 0xE1, 0x94, 0x3B, 0x63, 0x9A, 0x02, 0xF1, 0x48, 0xB4, 0xFC, 0x99, 0xDF, 0xCF, 0x64, 0x6D}},
		},
		{
			// retail
			{KEY_18_1B, {0x5D, 0xDD, 0x47, 0x39, 0x03, 0x7B, 0xC6, 0xA8, 0x70, 0xE6, 0x20, 0xB7, 0x0F, 0x67, 0x35, 0x04}},
			// dev
			{KEY_18_1B, {0x76, 0xF8, 0x45, 0x68, 0x11, 0x32, 0xBB, 0x31, 0xB6, 0xCF, 0x9E, 0x2E, 0x39, 0x48, 0x99, 0x3A}},
		},
		{
			// retail
			{KEY_1C_1F, {0x59, 0xF4, 0x39, 0x9C, 0x2F, 0x95, 0xA4, 0x12, 0x8A, 0x1F, 0xE4, 0x9D, 0x4D, 0xB6, 0x86, 0xDD}},
			// dev
			{KEY_1C_1F, {0x18, 0x5C, 0x51, 0x17, 0x0D, 0x68, 0x16, 0xF2, 0xE4, 0xA5, 0x63, 0x22, 0xFA, 0xBB, 0xB3, 0x9D}},
		},
		{
			// retail
			{KEY_20_23, {0x7C, 0x92, 0xF6, 0x27, 0x25, 0x51, 0xC4, 0x61, 0x4D, 0xB0, 0xB3, 0x45, 0xED, 0xD2, 0xE8, 0x69}},
			// dev
			{KEY_20_23, {0xE3, 0x86, 0x81, 0x5A, 0xA0, 0x4F, 0xEE, 0x3A, 0x23, 0xAE, 0x8E, 0x5A, 0xD7, 0xC5, 0x0F, 0x48}},
		},
		{
			// retail
			{KEY_24_27, {0xBB, 0xE8, 0xB4, 0xE0, 0x9D, 0x09, 0x37, 0x81, 0x6B, 0x23, 0x4D, 0x8E, 0xB3, 0xCD, 0x3C, 0xA2}},
			// dev
			{KEY_24_27, {0xD9, 0xC0, 0x1E, 0xC5, 0x68, 0xE9, 0xC5, 0x85, 0x08, 0x27, 0xEE, 0xED, 0x59, 0xCC, 0x10, 0x57}},
		},
		{
			// retail
			{KEY_28, {0x52, 0x18, 0x12, 0x7E, 0x13, 0x3C, 0xE3, 0xB8, 0x5B, 0xB8, 0xC0, 0x18, 0xCE, 0x76, 0xB7, 0xE2}},
			// dev
			{KEY_28, {0x29, 0x62, 0xF3, 0x47, 0xB1, 0xF9, 0x8A, 0x69, 0x7C, 0x68, 0x94, 0xA8, 0xBA, 0xBD, 0x15, 0xA2}},
		},
		{
			// retail
			{KEY_29, {0x4A, 0x42, 0x64, 0xCF, 0x32, 0xE8, 0x41, 0x70, 0x66, 0x6F, 0x29, 0xAC, 0x88, 0xEF, 0x3F, 0x7E}},
			// dev
			{KEY_29, {0xEB, 0xCD, 0xE8, 0x6C, 0x34, 0xBE, 0x2D, 0x9E, 0xB9, 0x5E, 0x18, 0xB0, 0x3D, 0x8A, 0x41, 0x68}},
		},
		{
			// retail
			{KEY_2A, {0x51, 0xAF, 0x6C, 0x4C, 0x8B, 0x13, 0xDA, 0x32, 0x28, 0xBD, 0x29, 0xB3, 0x71, 0xCF, 0x84, 0xE1}},
			// dev
			{KEY_2A, {0x35, 0x08, 0xCF, 0xD3, 0xEA, 0xE5, 0xA4, 0xDA, 0x14, 0xAA, 0xCD, 0xD7, 0x37, 0x26, 0x3F, 0x77}},
		},
		{
			// retail
			{KEY_2B, {0x3E, 0xD6, 0xF5, 0xCF, 0x2C, 0xC3, 0x7C, 0x54, 0x65, 0x50, 0x00, 0xB7, 0xC8, 0xB5, 0x2E, 0x0D}},
			// dev
			{KEY_2B, {0x4F, 0x04, 0xC8, 0xB6, 0x7F, 0xEA, 0x1F, 0x5F, 0x3D, 0x9F, 0xFE, 0xCD, 0x13, 0x95, 0xF9, 0xDD}},
		},
		{
			// retail
			{KEY_2C_2F, {0xB8, 0x7E, 0x64, 0x01, 0x8B, 0x19, 0x0F, 0xFE, 0x04, 0x8A, 0x81, 0x24, 0xC6, 0x45, 0x41, 0x96}},
			// dev
			{KEY_2C_2F, {0x8A, 0xFE, 0x0F, 0x82, 0x69, 0x90, 0x1A, 0xD9, 0x31, 0x9A, 0x4C, 0xA7, 0x0F, 0xB0, 0x97, 0x0C}},
		},
		{
			// retail
			{KEY_30_33, {0x28, 0xC0, 0xD5, 0x9B, 0x73, 0x66, 0x57, 0xBC, 0xDF, 0x50, 0xFF, 0x17, 0x49, 0x79, 0x95, 0x8A}},
			// dev
			{KEY_30_33, {0xFA, 0xD5, 0xB8, 0x49, 0x64, 0x08, 0x96, 0xC3, 0x4E, 0xAC, 0xDB, 0x2C, 0x72, 0xD3, 0x71, 0x3A}},
		},
		{
			// retail
			{KEY_34_37, {0x6E, 0x78, 0xA3, 0xBE, 0x9B, 0xDD, 0xDA, 0x09, 0xBF, 0xD5, 0x69, 0x48, 0x3F, 0x24, 0xFC, 0xE0}},
			// dev
			{KEY_34_37, {0xA9, 0xBE, 0x6A, 0xB4, 0x31, 0x6C, 0xA5, 0x8A, 0x00, 0xC2, 0xA2, 0x3B, 0xE7, 0x57, 0xDF, 0x39}},
		},
		{
			// retail
			{KEY_38_3B, {0x13, 0xE6, 0x2E, 0x5D, 0x6F, 0xB1, 0x65, 0x6B, 0x24, 0xDD, 0x33, 0x4B, 0xF1, 0x54, 0x68, 0xC3}},
			// dev
			{KEY_38_3B, {0xD1, 0xF8, 0xB1, 0x55, 0x29, 0xD6, 0x4B, 0x64, 0xE0, 0xD7, 0x4F, 0x98, 0xB5, 0xF8, 0xCC, 0x24}},
		},
		{
			// retail
			{KEY_3C, {0x85, 0xDB, 0x63, 0x07, 0x7C, 0x50, 0x11, 0x6B, 0x94, 0x90, 0xD4, 0xFA, 0xD6, 0x1A, 0xB2, 0x41}},
			// dev
			{KEY_3C, {0x82, 0x22, 0x6A, 0x9B, 0x55, 0x4A, 0x47, 0xEB, 0xA3, 0x8B, 0xD4, 0x04, 0x98, 0xED, 0x3F, 0x38}},
		},
		{
			// retail
			{KEY_3D, {0xA3, 0x08, 0xEB, 0x30, 0x64, 0x11, 0x42, 0x13, 0xA6, 0x0C, 0x56, 0x15, 0x8F, 0x5C, 0x49, 0x63}},
			// dev
			{KEY_3D, {0xE3, 0x03, 0x88, 0x02, 0xDE, 0x96, 0x9E, 0x1D, 0x5E, 0x5D, 0xB5, 0xB9, 0xF4, 0x54, 0x41, 0xBF}},
		},
		{
			// retail
			{KEY_3E, {0x07, 0x01, 0x98, 0xA2, 0xE4, 0x19, 0x4B, 0x9D, 0x02, 0x27, 0x92, 0x18, 0x35, 0xE4, 0x10, 0x6F}},
			// dev
			{KEY_3E, {0xE9, 0x86, 0xEC, 0xDF, 0x2C, 0x45, 0x07, 0x30, 0xB4, 0x17, 0x61, 0xE0, 0x37, 0x42, 0x58, 0x6D}},
		},
		{
			// retail
			{KEY_3F, {0x8E, 0xB7, 0x46, 0xF5, 0x18, 0xF7, 0xA8, 0xD8, 0x0E, 0x06, 0x91, 0xC8, 0x64, 0x1A, 0x56, 0xC5}},
			// dev
			{KEY_3F, {0xE5, 0x88, 0x3F, 0x09, 0xC9, 0x81, 0x17, 0x9B, 0x76, 0x8C, 0xB1, 0x33, 0x2C, 0xC6, 0xBB, 0xAC}},
		},
	};
	
	// 0x0C-0x0F
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x0C + i, default_brom_static_key[KEY_0C_0F][isDev].key));
	}
	// 0x10-0x13
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x10 + i, default_brom_static_key[KEY_10_13][isDev].key));
	}
	// 0x14
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x14, default_brom_static_key[KEY_14][isDev].key));
	// 0x15
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x15, default_brom_static_key[KEY_15][isDev].key));
	// 0x16
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x16, default_brom_static_key[KEY_16][isDev].key));
	// 0x17
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x17, default_brom_static_key[KEY_17][isDev].key));
	// 0x18-0x1B
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x18 + i, default_brom_static_key[KEY_18_1B][isDev].key));
	}
	// 0x1C-0x1F
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x1C + i, default_brom_static_key[KEY_1C_1F][isDev].key));
	}
	// 0x20-0x23
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x20 + i, default_brom_static_key[KEY_20_23][isDev].key));
	}
	// 0x24-0x27
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x24 + i, default_brom_static_key[KEY_24_27][isDev].key));
	}
	// 0x28
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x28, default_brom_static_key[KEY_28][isDev].key));
	// 0x29
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x29, default_brom_static_key[KEY_29][isDev].key));
	// 0x2A
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x2A, default_brom_static_key[KEY_2A][isDev].key));
	// 0x2B
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x2B, default_brom_static_key[KEY_2B][isDev].key));
	// 0x2C-0x2F
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x2C + i, default_brom_static_key[KEY_2C_2F][isDev].key));
	}
	// 0x30-0x33
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x30 + i, default_brom_static_key[KEY_30_33][isDev].key));
	}
	// 0x34-0x37
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x34 + i, default_brom_static_key[KEY_34_37][isDev].key));
	}
	// 0x38-0x3B
	for (size_t i = 0; i < 4; i++)
	{
		brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x38 + i, default_brom_static_key[KEY_38_3B][isDev].key));
	}
	// 0x3C
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x3C, default_brom_static_key[KEY_3C][isDev].key));
	// 0x3D
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x3D, default_brom_static_key[KEY_3D][isDev].key));
	// 0x3E
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x3E, default_brom_static_key[KEY_3E][isDev].key));
	// 0x3F
	brom_static_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(0x3F, default_brom_static_key[KEY_3F][isDev].key));
	
	// Add Firmware AES keys
	static const size_t kFirmwareAesKeyNum = 3;
	const DefaultAesKey default_firmware_key[kFirmwareAesKeyNum][2] = 
	{
		{
			// retail
			{FIRM_NGC_KEY, {0x07, 0x55, 0x0C, 0x97, 0x0C, 0x3D, 0xBD, 0x9E, 0xDD, 0xA9, 0xFB, 0x5D, 0x4C, 0x7F, 0xB7, 0x13}},
			// dev
			{FIRM_NGC_KEY, {0x4D, 0xAD, 0x21, 0x24, 0xC2, 0xD3, 0x29, 0x73, 0x10, 0x0F, 0xBF, 0xBD, 0x16, 0x04, 0xC6, 0xF1}},
		},
		{
			// retail
			{FIRM_NOR_KEY, {0x19, 0x2F, 0x08, 0x61, 0x03, 0x07, 0x35, 0xF2, 0x60, 0xB0, 0x64, 0x0B, 0xC0, 0x74, 0xEF, 0x93}},
			// dev
			{FIRM_NOR_KEY, {0xC0, 0x85, 0xC3, 0x07, 0x26, 0xFC, 0x4E, 0xD9, 0x1D, 0xD9, 0x75, 0x67, 0xC0, 0x41, 0xA9, 0x9C}},
		},
		{
			// retail
			{FIRM_SD_KEY, {0x85, 0xDB, 0x63, 0x07, 0x7C, 0x50, 0x11, 0x6B, 0x94, 0x90, 0xD4, 0xFA, 0xD6, 0x1A, 0xB2, 0x41}},
			// dev
			{FIRM_SD_KEY, {0x82, 0x22, 0x6A, 0x9B, 0x55, 0x4A, 0x47, 0xEB, 0xA3, 0x8B, 0xD4, 0x04, 0x98, 0xED, 0x3F, 0x38}},
		},
	};
	for (size_t i = 0; i < kFirmwareAesKeyNum; i++)
	{
		firmware_key.insert(std::pair<byte_t, KeyBag::Aes128Key>(default_firmware_key[i][isDev].key_index, default_firmware_key[i][isDev].key));
	}
}

void ctrtool::KeyBagInitializer::addStaticRsaKeys(bool isDev)
{
	auto public_exp = tc::ByteData({0x01,0x00,0x01});

	struct DefaultRsaKey
	{
		byte_t key_index;
		tc::crypto::RsaKey key;
	};

	static const size_t kDefaultRsaKeyNum = 9;
	DefaultRsaKey default_rsa_keys[kDefaultRsaKeyNum][2] = 
	{
		{
			// retail
			{RSAKEY_FIRM_NAND, {tc::cli::FormatUtil::hexStringToBytes("DECFB6FC3D33E955FDAC90E88817B003A16B9AAB72707932A2A08CBB336FB076962EC4E92ED88F92C02D4D410FDE451B253CBE376B458221E64DB1238182B68162B730F4604BC7F7F0170CB575887793526370F00BC6734341EEE4F071ECC8C132C4DCA9991D31B8A47EDD19040F02A81AAFB3489A29295E4984E09411D17EABB2C0447EA11B5E9D0D1AF9029A2E53032D48967C2CA6D7ACF1ED2B18BB01CB13B9ACA6EE5500377C696162890154779F075D26343AA949A5AFF25E0651B71CE0DEDA5C0B9F98C215FDBAD8A99900ABA48E4A169D662AE85664B2B6C093AF4D38A0165CE4BD62C2466BC95A594A7258FDB2CC36873085E8A1045BE0179BD0EC9B"), tc::ByteData(), public_exp}},
			// dev
			{RSAKEY_FIRM_NAND, {tc::cli::FormatUtil::hexStringToBytes("C11E84E4DAD7EDC8A5D9CB0BE93B9DBC1569F795F68FB7912419BE8FFEBBC11D09F0E378A20CC00B8ECDBA0E694C614ABD13EAE223E40B621F9B631B3134B71273646B4E60410A3D5486FA49EC2D77AD1DBC48E7FB07EF48C3BEC3D6E15B3DCD5E6B46086A3A0EAE254944A5400D945318A48F578EF4CDB4D34EE7159B11958500C49294B8BEFECBB750A0BE8AF5FDC80E4BA941136248CD1AF3C86BA5A0DFF19D3DFDE4D17C1F000D99726BA3057F8638BE4D5EAB93DFF3EEA19F2250A87F31AA2B03719B14C43088426FD52C7B03643B14EC4633CC2C959F5C7B83C567DD7CA12DD6EC170B23C7B19A72BA78CB39685CB6B461E198CF1D69F9D7B0A05E9CEB"), tc::ByteData(), public_exp}},
		},
		{
			// retail
			{RSAKEY_FIRM_RECOVERY, {tc::cli::FormatUtil::hexStringToBytes("D29FF5678F4BE3760C5FB1A6F846B4AE7D95CC50ED6515EE5BBDB9F0BCAB7E2B736B7EE1F623C473C7C3F84E680486DBBEBC8084EC5413CA21633CF4437636C05921F5D25953F0C1B38B07C94F03108BE9F9F791F67F15FD9B74F36220DAC235733BCCB3405B41040E41FF440DA9C5F76F9A64F210C6FC7A5BA84486575DB8B3A9949C536CCD81B446A917E3788D93113B638C253A5CF9FE94E6064DC828C83A124D2650D49B8CD9974D64B23D8DC4D3C6B28C1A07F074274ED34E208DB943ADD13A721B2C59745852A041506E1CB64C9D65275528AB0C931289FBCC8799C11C6AE7FC28659C7D4730FE8F605F7A1D6828DCF01D39A148C376BE8869B769FCE5"), tc::ByteData(), public_exp}},
			// dev
			{RSAKEY_FIRM_RECOVERY, {tc::cli::FormatUtil::hexStringToBytes("C0C6E3B52B1EBA2233CA36F0AA3FE3EF69AC8191CB71FA20798E15F88F8725AD92F200D9128B35BEE7A989448C794B929A76DA20B0EE4596C01F51C618F4DA71BB0209FE5577599C68AA6D9A8DF93A921B7F9B9F4B318917225F29EEE3A8A018B33B8822057A324F56037FC90FCC68D0C6E753BE48F5189464E0AC91CAFF95101BA52A73AC196862DEA7701FD00C071EB176F3E6436531B5CFAC0594D40F3B9CBC0F10E30DA06484692A382F3B70135BFB9C37FEB3CB27AC291D4694442AD351E2C3CDE94C21CDE9BB0BB8850CA67F5F9DC930425CAB834471AEB418EAF9EE52EE86249270220AECDFBF3978F0FB024C89555E0EB66FA9A067B1DE847CE2200B"), tc::ByteData(), public_exp}},
		},
		{
			// retail
			{RSAKEY_NCSD_NAND, {tc::cli::FormatUtil::hexStringToBytes("AA948C55BE3CAAFD5B53E5A9F583EB6D02EE4F72365A81CAFAE0E0D95869C5E65DFE389CA16D7AA00DA999341614B88FDDD8C0DB82E929500B6C9C2728AA3B2C9AEE1A5C3080BE97FDFCF55EBC75A9CF06FE7915F43D7A6B50B553B0A7DF112632FAB970C688F0AF36BE9101788D2C13313D28AE17AF6A80114CA6E06092D82C8061FB2814A4938556FF80B7D1AB4DFC2587F81A3E4EB9CD66EF322AA3D3DCDC3C04B94D6A4B47AB43941BEA092ED688EDF845F14163606AFBD881E7A7473656D873034B28E13D120181DA07BC6763476F2D3A52D19A44EF5960F6FB027E0103A4E0F9202791D4F90F867E56015DFCD42D8C913DD90D1565E65614B1E45C6831"), tc::ByteData(), public_exp}},
			// dev
			{RSAKEY_NCSD_NAND, {tc::cli::FormatUtil::hexStringToBytes("C767DA7B195440553720B382CF7496B9AE884AF5711CBA4E1AEC6D62078DDBAA4B594A3D63CF6CC0BF8DFB32430CCE7C0EE24FDF049F3C650A0EA7A8E91AC44414CF845E1C7B9EDADD6C076B7CC4412841A35B8F4A11C126F8A23C12F23DF88D71D478363DBDF6479B8D5BBBBA04A8A24D274881AB2F2121E9049F5706488F6565D7EA2BC6BDFC529EDCE1C385FC0C66CD1B4D2D882AB431F0665B9F0E86CA44529A8160095D54417D16C623A08DA1DC3FBDAE705D862F025B033591B8F4D7A7928DCE3556FD48A54C0A47E440BD31E224AE86A7BDC94C0E4B124B5AC8F6F08F84FAD15C674D0CE39E706119C108F01EFEA58B63A99C5BB9949782EE0DE71653"), tc::ByteData(), public_exp}},
		},
		{
			// retail
			{RSAKEY_CFA_CCI, {tc::cli::FormatUtil::hexStringToBytes("FBDEB82B40930FF6B19A08061B86FED0DF1079173D8CE27ACE8F2345B90A6DED300EC1A892C4BD1ACEA7AC77AA47E5204A4491DF1CFE8628122D66DFBEAD9661EDF2F7417B57886B241E7DECBE986565366599A9FE24678599EE2AAEEEB1811A22E36D756E21BCEF115C61AF0C3000B6A223EDFE7015DA52E1E62DCE34E8AA4CF1D6675657D3DBC090496F4573934E303070F5C98F3125F2C2E7337F4EB6F52ADF2000E579B2D0F917F77E1690400057914478EF1CE08509DAF4147E4BD735D687548F2AB5A76F50D0F7D1F119C9AC227E0511F5F26DEE9227575FE5150D2768BF52657473A6586D7918AC31DDDD808B7524E117E195251629AB6969C828EE5D"), tc::ByteData(), public_exp}},
			// dev
			{RSAKEY_CFA_CCI, {tc::cli::FormatUtil::hexStringToBytes("B90CC4C678F86E300528C1CBD2CFA7805C574D169CAFA6CD01BB8333AD03BB0663D817F5E3DFDA0D3B860EA2804794446FD9977E786AC39393EF02FC229F80778C70921C43B1374C76E0573BAB89FFEFE5BB3EAB9139B8D9660B64289192E9D0B3DFD14BC173B53F56A04010FE152B1FA27ADE31B02640C357FD35CBF0FAFFFB6FDBCD341D512D2D8118FF0C0851D5B44B5616029F4E6ADF066ECB7285E92E43A208780C389C19BD7B747468C42DC1359E653BD899041C8B938E7E927CBBDD60ECE7FE0E9D4F3646E6F15C9470EE675F362B70448DCA09B95867D29FAD1F135474ADA6844428F3DE7E4C202BC5E912E95EFB8D77A9A4D20D3C3824BEF58AB5F5"), tc::ByteData(), public_exp}},
		},
		{
			// retail
			{RSAKEY_ACCESSDESC, {tc::cli::FormatUtil::hexStringToBytes("B1E3E35F013980D156789DB706F71DBF3E2276EDF95DA236B630610596D300B9EDF1D7E01DA04FB7CF5A198775498840EDE36F7C904A644598D704B95A6B45AA7E94C0B3B7DB7B665920B708E2F383A37FE32021A0EBB7280FF32B15A4C9D0AB8939997E765F9E4D1E01228D74A6EB9AA39D45E510616E20FD2375C0C50503C54C024F544B5708B446C32CF1F9526CCD1455A855926DE24A4146EB08C5F3B48D0D5E21EAAF4D274DDE779397E2C76B661FDB2D6EA95F6114177B2B665AB50189F2237525259C869A89FF641D5BCED77E3F2DA8DAB55AC55F5920B0ED1C91FFA327B88ECF8215E549EFE458E15F8F53B9332A5624AAA1D36E471A634419B38EA5"), tc::ByteData(), public_exp}},
			// dev
			{RSAKEY_ACCESSDESC, {tc::cli::FormatUtil::hexStringToBytes("F43C4582FBF8905D07029F2A988B63B7D38F3CE2E0E093BFDF32434DBEF4D17A3A4E5431D773AE994CC41F3C3EF05705A38A455460D88FD91D680D0E2EEFC8E83DC919F3731E2DDA77883ECA5E25704BF77095835424E0C31A75DF613DD142EC351B38D6C1F67E182A8485DD57741F0A2EF6B294A23EE9A1D009F73A998005AF5755EF52FA243E7FD47C41447B067FB95B2E8E96AE46124D6421E50F85CCEB92E5F0F5A742273BECF8E781756F630A8B0D773851E66633BA79DC2F2C8FC32806BB039CDBD1640A66F0F8C12A491D0C6E35BBEAB35C0DE9957C67BE6577EC07C023050A724886E99EFC2515E7C82165E01BD5D50ED31154BB2978BF2A3C3BB6B1"), tc::ByteData(), public_exp}},
		},
		{
			// retail
			{RSAKEY_CRR, {tc::cli::FormatUtil::hexStringToBytes("D00E6756858ABCF213D87FBB29D192349EE42743D1111DFE9593C4783F4A2D2AB1CA6FB0BA32A2242E2BE5726EF0B49768CCDA171514F074D435B4EE04D89CA9ECDBCB29E895F257BABD65473D6CE50DE5103EB5C37302AEC893777CC523220171CCC60BEACAA0FED4CDA949655A3B0D5BB7B2EA329A43856CCA045C900D9E34B9EF802B2E6D96BC31157E1401C000E0197A2E17E58BA37AD2C247779A8B832E9866A1A1020F0299E7D79FDCC852A22823081B87FFE4BA90025015C720A0AB76607D1FC1BF339D66246A803BF4A27F15B10C36EF44E8D036949A7484488789499A0B7E10F27355623D68AC9E5FAE1D54F8BA628E29BC49E7FDEEBD3D7BCCF8BF"), tc::ByteData(), public_exp}},
			// dev
			{RSAKEY_CRR, {tc::cli::FormatUtil::hexStringToBytes("C5F709805FDADCBD460752AA6DCD72B2500977478A4F092AE291B45F04975175C9196F95BB23147D34DF777807E8D11011AFCE02B873A9FB64B76587E567DD673775FD0FE297BC79A8CCF9FA18B2624DF7536B9C0E1AAB90E65286C81C922C6153A901DA4393D0422EDDD54C8C4CE89156ECEE12700B64F00AD6AFF860C2A8267AC8BA559AA144FD094726A0C1999EF124DFA3C2BBFF0745D9D6A4C0FF6C6C787B6D708C7444B095E6C6665E7EBE71C5913473A7D44C0DFCA921499492A16A4D30A3D69F6C60400CEEEBF89922E16FFC3C9623AA1134004EFC2D604145E35D7806B1F1C307B9D347D0F18C26331F6B46DDF3E38464A7F15311E1534E99DBAC53"), tc::ByteData(), public_exp}},
		},
		
		{
			// retail
			{RSAKEY_SECUREINFO, {tc::cli::FormatUtil::hexStringToBytes("B1791A6D1EADD429BA89A1CD433630174BC68730C5E70560197B50D8C4546710A6E8A101BC2CEB0376F005C70CE0B6D6DFFD26DF33468BDBB2391E7EC01AA1A5A091E807DA378676BA390A25429D5961E161D40485A74BB20186BEB11A3572C1C2EA28AB7A1015325C9E712B7DF965EAE6C6FB8BAED76C2A94A6C5ECE40EAF987E06F20F884FD20635A476E9F70ABA5C5B1461520054044593E468270435355AAD5809D1193F5A0728D6DB6B551F77945DC3BE6FAE5BCC0863E476DFA29B36EA853403E616EAA905E07F3A3E7E7077CF166A61D17E4D354C744485D4F67B0EEE32F1C2D5790248E9621A33BAA39B02B02294057FF6B43888E301E55A237C9C0B"), tc::ByteData(), public_exp}},
			// dev
			{RSAKEY_SECUREINFO, {tc::cli::FormatUtil::hexStringToBytes("B1AA6C553CA84D833C2E9756B52BD3701D0FD4D1EEF171F4FD95961D52BF7563B89D2FF5F815E40A76E20F551163E9E98568415A283122E199DEEC771712C678DA0BB4DD50F30C615FA57DEA74D71D1187BFEBC333D7350EDD45981BEF6FB973E8359CE5B0C8FF5C429BA790AEF9B7625604B1B0B244D68634E62F794D9CAFB59A3BFAC88103966F9BDC878B323C37EBCD21C8B9276FFCC84702FF87D1D02F64D436D48501AD70F3A2B10D13EF5594A0238171F94AD20158906013FB6DB6183831DF1144B59649A35308B264C1EF119E1D17179A8744173A73A2F7D9961A79E1F9866EEE6FBBD2DCCF3B0DC4E276D1D0C03798BEC1BCD9646FC4CB46BB5FF555"), tc::ByteData(), public_exp}},
		},
		{
			// retail
			{RSAKEY_LOCALFRIENDCODESEED, {tc::cli::FormatUtil::hexStringToBytes("A3759A3546CFA7FE30EC55A1B64E08E9449D0C72FCD191FD610A288975BCE6A9B21556E9C7670255ADFC3CEE5EDB78259A4B221B71E7E9515B2A6793B21868CE5E5E12FFD86806AF318D56F9549902346A17E7837496A05AAF6EFDE6BED686AAFD7A65A8EBE11C983A15C17AB540C23D9B7CFDD463C5E6DEB77824C629473335B2E937E054EE9FA53DD793CA3EAE4DB60F5A11E70CDFBA03B21E2B31B65906DB5F940BF76E74CAD4AB55D940058F10FE06050C81BB422190BA4F5C5382E1E10FBC949F60695D1303AAE2E0C108424C200B9BAA552D55276E24E5D60457588FF75F0CEC819F6D2D28F31055F83B7662D4E4A69369B5DA6B4023AF07EB9CBFA9C9"), tc::ByteData(), public_exp}},
			// dev
			{RSAKEY_LOCALFRIENDCODESEED, {tc::cli::FormatUtil::hexStringToBytes("985992B9651768F5E79F6AE500CC57418E6B8DF9D4409CEAFD9627C425D616C2BC31CC23206B0A494931AA567E6EAE1C550663A8721DEB16C51E0023E0BABD26166519CEFEDB7F4299CEE5CECCB2C518957E4BDBB6567D7D73ED9CD594CE265D8BCA6635CB60CAA3C1B652212FF4FF45C73BEBF4CDB6463D0761D661C349AFBCF55BAC71CD0668462FD12E82C23FEB57CF622EE7E2721E73707AF4F7B01B8ADFC28D59C2D440CCF14150D391A1F95D9D45BC5A74641E22CE10D62CE35C7DD2B0EBC516645C9ADC6093A44E7500C1BE16E9D7D35FB869E62C143851B31515577FEAFE7450E36D6027AE8C9CAACABE78EAE4A48A655B39650B7F0BDBC073E091BB"), tc::ByteData(), public_exp}},
		},
		{
			// retail
			{RSAKEY_DSP, {tc::cli::FormatUtil::hexStringToBytes("CB6537CA75D0AC0085B52E5C93491BA09B7BAE24D6B3252C707F513439463B55C5DBFB97422A04D15C1363E4EBF4AF3ABFF447D09D25C819F99DE9632FA2B88186A7A1F4D83456228A6BCAA8B30F56D95F070C98139D1C3E59811C33C6AA707E41B8B9AAA0B41251C61E233876A208C332413B57ABCAD004CBCAD4A8EEA93CED2BE1E61196B7C829C93C70BBCD7955A20258CDD17D6476C7A98DA5CCF339E5B51204BE6AF85237EAAC4CF63863F618637A2760E0B78565E9483B107F1D7917498A211CEC01AAA4232FC50AEFE74CE2056A09ED720D3F877B3EF9A30003191751E969EC10E7E0891C2B46611816858458E279F6F4A7286246E94958DCF94765D9"), tc::ByteData(), public_exp}},
			// dev
			{RSAKEY_DSP, {tc::cli::FormatUtil::hexStringToBytes("CB6537CA75D0AC0085B52E5C93491BA09B7BAE24D6B3252C707F513439463B55C5DBFB97422A04D15C1363E4EBF4AF3ABFF447D09D25C819F99DE9632FA2B88186A7A1F4D83456228A6BCAA8B30F56D95F070C98139D1C3E59811C33C6AA707E41B8B9AAA0B41251C61E233876A208C332413B57ABCAD004CBCAD4A8EEA93CED2BE1E61196B7C829C93C70BBCD7955A20258CDD17D6476C7A98DA5CCF339E5B51204BE6AF85237EAAC4CF63863F618637A2760E0B78565E9483B107F1D7917498A211CEC01AAA4232FC50AEFE74CE2056A09ED720D3F877B3EF9A30003191751E969EC10E7E0891C2B46611816858458E279F6F4A7286246E94958DCF94765D9"), tc::ByteData(), public_exp}},
		},
		
	};

	for (size_t i = 0; i < kDefaultRsaKeyNum; i++)
	{
		rsa_key.insert(std::pair<byte_t, tc::crypto::RsaKey>(default_rsa_keys[i][isDev].key_index, default_rsa_keys[i][isDev].key));
	}

	struct BroadonRsaSignerProfile
	{
		std::string issuer;
		KeyBag::BroadOnRsaSignerProfile profile;
	};

	static const size_t kDefaultBroadonProfileNum = 4;
	BroadonRsaSignerProfile default_broadon_profile[kDefaultBroadonProfileNum][2] = 
	{
		// root
		{
			// retail
			{"Root", {tc::ByteData(), {tc::cli::FormatUtil::hexStringToBytes("F8246C58BAE7500301FBB7C2EBE0010571DA922378F0514EC0031DD0D21ED3D07EFC852069B5DE9BB951A8BC90A244926D379295AE9436AAA6A302510C7B1DEDD5FB20869D7F3016F6BE65D383A16DB3321B95351890B17002937EE193F57E99A2474E9D3824C7AEE38541F567E7518C7A0E38E7EBAF41191BCFF17B42A6B4EDE6CE8DE7318F7F5204B3990E226745AFD485B24493008B08C7F6B7E56B02B3E8FE0C9D859CB8B68223B8AB27EE5F6538078B2DB91E2A153E85818072A23B6DD93281054F6FB0F6F5AD283ECA0B7AF35455E03DA7B68326F3EC834AF314048AC6DF20D28508673CAB62A2C7BC131A533E0B66806B1C30664B372331BDC4B0CAD8D11EE7BBD9285548AAEC1F66E821B3C8A0476900C5E688E80CCE3C61D69CBBA137C6604F7A72DD8C7B3E3D51290DAA6A597B081F9D3633A3467A356109ACA7DD7D2E2FB2C1AEB8E20F4892D8B9F8B46F4E3C11F4F47D8B757DFEFEA3899C33595C5EFDEBCBABE8413E3A9A803C69356EB2B2AD5CC4C858455EF5F7B30644B47C64068CDF809F76025A2DB446E03D7CF62F34E702457B02A4CF5D9DD53CA53A7CA629788C67CA08BFECCA43A957AD16C94E1CD875CA107DCE7E0118F0DF6BFEE51DDBD991C26E60CD4858AA592C820075F29F526C917C6FE5403EA7D4A50CEC3B7384DE886E82D2EB4D4E42B5F2B149A81EA7CE7144DC2994CFC44E1F91CBD495"), tc::ByteData(), public_exp}}},
			// dev
			{"Root", {tc::ByteData(), {tc::cli::FormatUtil::hexStringToBytes("D01FE100D43556B24B56DAE971B5A5D384B93003BE1BBF28A2305B060645467D5B0251D2561A274F9E9F9CEC646150AB3D2AE3366866ACA4BAE81AE3D79AA6B04A8BCBA7E6FB648945EBDFDB85BA091FD7D114B5A3A780E3A22E6ECD87B5A4C6F910E4032208814B0CEEA1A17DF739695F617EF63528DB949637A056037F7B32413895C0A8F1982E1565E38EEDC22E590EE2677B8609F48C2E303FBC405CAC18042F822084E4936803DA7F41349248562B8EE12F78F803246330BC7BE7EE724AF458A472E7AB46A1A7C10C2F18FA07C3DDD89806A11C9CC130B247A33C8D47DE67F29E5577B11C43493D5BBA7634A7E4E71531B7DF5981FE24A114554CBD8F005CE1DB35085CCFC77806B6DE254068A26CB5492D4580438FE1E5A9ED75C5ED451DCE789439CCC3BA28A2312A1B8719EF0F73B713950C02591A7462A607F37C0AA7A18FA943A36D752A5F4192F0136100AA9CB41BBE14BEB1F9FC692FDFA09446DE5A9DDE2CA5F68C1C0C21429287CB2DAAA3D263752F73E09FAF4479D2817429F69800AFDE6B592DC19882BDF581CCABF2CB91029EF35C4CFDBBFF49C1FA1B2FE31DE7A560ECB47EBCFE32425B956F81B69917487E3B789151DB2E78B1FD2EBE7E626B3EA165B4FB00CCB751AF507329C4A3939EA6DD9C50A0E7386B0145796B41AF61F78555944F3BC22DC3BD0D00F8798A42B1AAA08320659AC7395AB4F329"), tc::ByteData(), public_exp}}},
		},
		// ca
		{
			// retail
			{"Root-CA00000003", {tc::cli::FormatUtil::hexStringToBytes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{tc::cli::FormatUtil::hexStringToBytes("B279C9E2EEE121C6EAF44FF639F88F078B4B77ED9F9560B0358281B50E55AB721115A177703C7A30FE3AE9EF1C60BC1D974676B23A68CC04B198525BC968F11DE2DB50E4D9E7F071E562DAE2092233E9D363F61DD7C19FF3A4A91E8F6553D471DD7B84B9F1B8CE7335F0F5540563A1EAB83963E09BE901011F99546361287020E9CC0DAB487F140D6626A1836D27111F2068DE4772149151CF69C61BA60EF9D949A0F71F5499F2D39AD28C7005348293C431FFBD33F6BCA60DC7195EA2BCC56D200BAF6D06D09C41DB8DE9C720154CA4832B69C08C69CD3B073A0063602F462D338061A5EA6C915CD5623579C3EB64CE44EF586D14BAAA8834019B3EEBEED379"), tc::ByteData(), public_exp}}},
			// dev
			{"Root-CA00000004", {tc::cli::FormatUtil::hexStringToBytes("000100031949429D1E58A62E7E8B56D1B76AE302FD8B97491F778745F75388C4DD0BEB1DF122FB9642151497764A53CF78151845E42CA8FDE486FD2A4F53F8A1BA008A7485FF73B3BF7E3C980729D0656B693219ADE835EB5FFFFCCB7CBB5E307FE0688B888EF2D2053FB7E791E985FD15EF10D79CCA88D6BB15E8E4714A98EE09BF7B8AF053232B6450E6D5FDFFC20A6D1EA6A23812E1014525D56D4082703B86986959A73CD1A14364D2C2DAEA96B095F76C46E4FF4155465E70EF1ED31053D97011E010CC93E7914013687FA3A802996D1E557B1CCC7A7E8F5865C1742E28E26DEF38A93AB5D82D43ECCCBF0BEF22E1FD57E2864333582FEDEABC012F986DDFC3E9447973470308455BDC57AA170B84427F73A29B48F6DA135F66C745C142A84AFB0E6A5EED85D7B9719936F8CE2B621F395F40DC03BEF8854C1117FF0C128641CC7843B97B4346DB226F6026ACB56C278B8E0EA79A2D65EF798E1078AD80ED4B9604D2F08B2CD64A23A3DB270833B402F80851F35BED3EE4577C6660FBF16D9413E09C917A49D42C6DA375BC27F0230DB98F8973AB027B522CD57EC03D25E8B3FC3494C97FB108FE18C68A4336E46C26B6F280D27E34BE287C3E4687BC9D776B76D928D1B6352EC0347D7294AA9360268D26F5F652064AF240D7D00C7C5EA3C32DE62D9B5C4B4CAB6FD7BD371D57C2166095910E4AD8E9ED181EF761936153892D77000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000526F6F74000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000014341303030303030303400000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000081122A46C9CC2DC4DF2930E4DF3F8C70A078948775AD5E9AA604C5B4D8EAFF2AA1D214676564EFCA28CC00154554A1A3EA1379E9E6CAACED1593FE88D89AC6B8ACCCAB6E207CEB7CCA29809E2980440662B7D4382A15DA43085745A9AAE59AA05BDB32F66869A2DD4295386C87ECDD3508A2CF60D01E23EC2FE698F470D6001549A2F06759131E534C7006057DEF1D18A83F0AC79CFE80FF5A91F2BED4A0837061190A0329902165403C9A908FB615739F3CE33BF1BAEA16C25BCED7963FACC9D24D9C0AD76FC020B2C4B84C10A741A2CC7D9BAC3AACCCA3529BAC316A9AA75D2A26C7D7D288CBA466C5FE5F454AE679744A90A15772DB3B0E47A49AF031D16DBEAB332B0001000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"), {tc::cli::FormatUtil::hexStringToBytes("C9CC2DC4DF2930E4DF3F8C70A078948775AD5E9AA604C5B4D8EAFF2AA1D214676564EFCA28CC00154554A1A3EA1379E9E6CAACED1593FE88D89AC6B8ACCCAB6E207CEB7CCA29809E2980440662B7D4382A15DA43085745A9AAE59AA05BDB32F66869A2DD4295386C87ECDD3508A2CF60D01E23EC2FE698F470D6001549A2F06759131E534C7006057DEF1D18A83F0AC79CFE80FF5A91F2BED4A0837061190A0329902165403C9A908FB615739F3CE33BF1BAEA16C25BCED7963FACC9D24D9C0AD76FC020B2C4B84C10A741A2CC7D9BAC3AACCCA3529BAC316A9AA75D2A26C7D7D288CBA466C5FE5F454AE679744A90A15772DB3B0E47A49AF031D16DBEAB332B"), tc::ByteData(), public_exp}}},
		},
		// tmd
		{
			// retail
			{"Root-CA00000003-CP0000000b", {tc::cli::FormatUtil::hexStringToBytes("000100042EA66C66CFF335797D0497B77A197F9FE51AB5A41375DC73FD9E0B10669B1B9A5B7E8AB28F01B67B6254C14AA1331418F25BA549004C378DD72F0CE63B1F7091AAFE3809B7AC6C2876A61D60516C43A63729162D280BE21BE8E2FE057D8EB6E204242245731AB6FEE30E5335373EEBA970D531BBA2CB222D9684387D5F2A1BF75200CE0656E390CE19135B59E14F0FA5C1281A7386CCD1C8EC3FAD70FBCE74DEEE1FD05F46330B51F9B79E1DDBF4E33F14889D05282924C5F5DC2766EF0627D7EEDC736E67C2E5B93834668072216D1C78B823A072D34FF3ECF9BD11A29AF16C33BD09AFB2D74D534E027C19240D595A68EBB305ACC44AB38AB820C6D426560C000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000526F6F742D43413030303030303033000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000143503030303030303062000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000137A080BA689C590FD0B2F0D4F56B632FB934ED0739517B33A79DE040EE92DC31D37C7F73BF04BD3E44E20AB5A6FEAF5984CC1F6062E9A9FE56C3285DC6F25DDD5D0BF9FE2EFE835DF2634ED937FAB0214D104809CF74B860E6B0483F4CD2DAB2A9602BC56F0D6BD946AED6E0BE4F08F26686BD09EF7DB325F82B18F6AF2ED525BFD828B653FEE6ECE400D5A48FFE22D538BB5335B4153342D4335ACF590D0D30AE2043C7F5AD214FC9C0FE6FA40A5C86506CA6369BCEE44A32D9E695CF00B4FD79ADB568D149C2028A14C9D71B850CA365B37F70B657791FC5D728C4E18FD22557C4062D74771533C70179D3DAE8F92B117E45CB332F3B3C2A22E705CFEC66F6DA3772B0001000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"), {tc::cli::FormatUtil::hexStringToBytes("A689C590FD0B2F0D4F56B632FB934ED0739517B33A79DE040EE92DC31D37C7F73BF04BD3E44E20AB5A6FEAF5984CC1F6062E9A9FE56C3285DC6F25DDD5D0BF9FE2EFE835DF2634ED937FAB0214D104809CF74B860E6B0483F4CD2DAB2A9602BC56F0D6BD946AED6E0BE4F08F26686BD09EF7DB325F82B18F6AF2ED525BFD828B653FEE6ECE400D5A48FFE22D538BB5335B4153342D4335ACF590D0D30AE2043C7F5AD214FC9C0FE6FA40A5C86506CA6369BCEE44A32D9E695CF00B4FD79ADB568D149C2028A14C9D71B850CA365B37F70B657791FC5D728C4E18FD22557C4062D74771533C70179D3DAE8F92B117E45CB332F3B3C2A22E705CFEC66F6DA3772B"), tc::ByteData(), public_exp}}},
			// dev
			{"Root-CA00000004-CP0000000a", {tc::cli::FormatUtil::hexStringToBytes("000100045005D75E6DDEB8783C81E9EF0D17CD58F59426A3FD6F6990E8F83287122EC25CA14B99242337BA91A75B0F7C59FBF7D1892722C4E6AFC7DEC74A6E007F434A888A8215E8DF2B52ED4200BC69B4DA7FEB746C7A2D96565B45597B8FAEB16BDC76C1C80C47F50DA9C3E1FE28501C26A2D1544BD1604A9E8F322AEF315FEA48226722B7CB372FF35F5E616A5344A585E5A08A2E1777572B7A9AF7D2D8C49CD0A054BF8A9DB49FC660617CB8354E257F68682F94B3CC538C426F88C5485CBEC1D0480474965A7E8259AA9FB66146CE5921C6F0C1751F21917F2496CB0C70157AB7BB3A9F5756565C38922EFDC8F170B9AEA1AE36F55E3526630ABAB2050FF00CDCBB000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000526F6F742D4341303030303030303400000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000014350303030303030306100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000018A34D5BAA7F9380289BE89863107AE10C592C2F7CFFBDAADD74F4A2FBACD76F00934206347156D84049729F3E24FA5E19D15B635CD2EF09DE32EE6B6FC8FA328E2E96B99441047D076295DA0D91D80935D0DE8E6BC6AB1427019CFE4996FC9B54794DEBD7C66673A6DD3A77654794EC1C87AA46D978A97DDB11226ED412C2784B218392C710C77419FFAAF60B75D823DD33C3A15BA72D30A5A4D8F80FD673FD26CB29A6EF5039E25F5961846BDA2EC7CBE4384B28FB0DD58E7CAA7D4B373AD781DD73E30993BDBD7E08554A8CA5C9842D7101A22A01B015FB3078B913F4C73FB5A6F1A25E22B002B6E009547F0FBDF0FEA5501D9315F93D830F0F0E3DE23D96E709D9770001000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"), {tc::cli::FormatUtil::hexStringToBytes("AA7F9380289BE89863107AE10C592C2F7CFFBDAADD74F4A2FBACD76F00934206347156D84049729F3E24FA5E19D15B635CD2EF09DE32EE6B6FC8FA328E2E96B99441047D076295DA0D91D80935D0DE8E6BC6AB1427019CFE4996FC9B54794DEBD7C66673A6DD3A77654794EC1C87AA46D978A97DDB11226ED412C2784B218392C710C77419FFAAF60B75D823DD33C3A15BA72D30A5A4D8F80FD673FD26CB29A6EF5039E25F5961846BDA2EC7CBE4384B28FB0DD58E7CAA7D4B373AD781DD73E30993BDBD7E08554A8CA5C9842D7101A22A01B015FB3078B913F4C73FB5A6F1A25E22B002B6E009547F0FBDF0FEA5501D9315F93D830F0F0E3DE23D96E709D977"), tc::ByteData(), public_exp}}},
		},
		// tik
		{
			// retail
			{"Root-CA00000003-XS0000000c", {tc::cli::FormatUtil::hexStringToBytes("00010004919EBE464AD0F552CD1B72E7884910CF55A9F02E50789641D896683DC005BD0AEA87079D8AC284C675065F74C8BF37C88044409502A022980BB8AD48383F6D28A79DE39626CCB2B22A0F19E41032F094B39FF0133146DEC8F6C1A9D55CD28D9E1C47B3D11F4F5426C2C780135A2775D3CA679BC7E834F0E0FB58E68860A71330FC95791793C8FBA935A7A6908F229DEE2A0CA6B9B23B12D495A6FE19D0D72648216878605A66538DBF376899905D3445FC5C727A0E13E0E2C8971C9CFA6C60678875732A4E75523D2F562F12AABD1573BF06C94054AEFA81A71417AF9A4A066D0FFC5AD64BAB28B1FF60661F4437D49E1E0D9412EB4BCACF4CFD6A3408847982000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000526F6F742D43413030303030303033000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000158533030303030303063000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000137A0894AD505BB6C67E2E5BDD6A3BEC43D910C772E9CC290DA58588B77DCC11680BB3E29F4EABBB26E98C2601985C041BB14378E689181AAD770568E928A2B98167EE3E10D072BEEF1FA22FA2AA3E13F11E1836A92A4281EF70AAF4E462998221C6FBB9BDD017E6AC590494E9CEA9859CEB2D2A4C1766F2C33912C58F14A803E36FCCDCCCDC13FD7AE77C7A78D997E6ACC35557E0D3E9EB64B43C92F4C50D67A602DEB391B06661CD32880BD64912AF1CBCB7162A06F02565D3B0ECE4FCECDDAE8A4934DB8EE67F3017986221155D131C6C3F09AB1945C206AC70C942B36F49A1183BCD78B6E4B47C6C5CAC0F8D62F897C6953DD12F28B70C5B7DF751819A98346526250001000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"), {tc::cli::FormatUtil::hexStringToBytes("AD505BB6C67E2E5BDD6A3BEC43D910C772E9CC290DA58588B77DCC11680BB3E29F4EABBB26E98C2601985C041BB14378E689181AAD770568E928A2B98167EE3E10D072BEEF1FA22FA2AA3E13F11E1836A92A4281EF70AAF4E462998221C6FBB9BDD017E6AC590494E9CEA9859CEB2D2A4C1766F2C33912C58F14A803E36FCCDCCCDC13FD7AE77C7A78D997E6ACC35557E0D3E9EB64B43C92F4C50D67A602DEB391B06661CD32880BD64912AF1CBCB7162A06F02565D3B0ECE4FCECDDAE8A4934DB8EE67F3017986221155D131C6C3F09AB1945C206AC70C942B36F49A1183BCD78B6E4B47C6C5CAC0F8D62F897C6953DD12F28B70C5B7DF751819A9834652625"), tc::ByteData(), public_exp}}},
			// dev
			{"Root-CA00000004-XS00000009", {tc::cli::FormatUtil::hexStringToBytes("0001000463805A351A437BA24319BB3A777B7AF35E724B150A06396C5FEC3845B18876268D5EDAE62F14BA02FAD6FC3B2BBE8707638E55BF055AFCFCB347691189DB1CAF4B4376623E30890A9D3BBB3E50BDF7A6C0F7F8BB0DB56ABBC6C350C888BB9DF09BD130646069DD3467A700EBDCF98CB0F7930E81FE98D972458B947E59E2BE4E912D75CA1B8E2EF46D73B16B35B5670D632D51385328191D9DAE8DC661CCEFA4ABE2F3B04C7BE271B5F92CFA55CD888B72CCBE67FADFEF6B533C45D8CBDFB2764146D6C26F2716C507F3F44466A315D277F289DAFDD550CFA49BEACAC97BE5460EED9BFB04A9DA1958D92A208AACC1F48EE914D88AD741D55B9B6422D8AFAEC7000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000526F6F742D4341303030303030303400000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000015853303030303030303900000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000018A347A4C0844CEB7EB0CFF0AEB777698593E4995A954E581738CED681B0BD7709E7F89ADFAD054883F6C3FDDF7B83E00C2681544329EA826C89F0A67442864D3260327DA77A13406659DA3E416B2794034FAA229DD55452DB270A6AA23D19B1661B197DABC70E881791A12AB43C6CCBF5AA7C3ADD36FB35717B20015900D6F69039354131F8C1C0573A35185890B1AD9A0EECE0F47A7DA52748C972AB0D087B62354091142BB11D1AFAF9CD5C1713535271CAE22A78B17F4ACD59D8BA1D7D705F781B9F9D37188ED7CD0D49577469883A6B8E4E1B85DDBE39450589561297599A09A4C82D2FF5CFB47370DB581EB24E776FA47E62DFB705E880425CB87887977F662C5F0001000100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"), {tc::cli::FormatUtil::hexStringToBytes("C0844CEB7EB0CFF0AEB777698593E4995A954E581738CED681B0BD7709E7F89ADFAD054883F6C3FDDF7B83E00C2681544329EA826C89F0A67442864D3260327DA77A13406659DA3E416B2794034FAA229DD55452DB270A6AA23D19B1661B197DABC70E881791A12AB43C6CCBF5AA7C3ADD36FB35717B20015900D6F69039354131F8C1C0573A35185890B1AD9A0EECE0F47A7DA52748C972AB0D087B62354091142BB11D1AFAF9CD5C1713535271CAE22A78B17F4ACD59D8BA1D7D705F781B9F9D37188ED7CD0D49577469883A6B8E4E1B85DDBE39450589561297599A09A4C82D2FF5CFB47370DB581EB24E776FA47E62DFB705E880425CB87887977F662C5F"), tc::ByteData(), public_exp}}},
		}
	};

	for (size_t i = 0; i < kDefaultBroadonProfileNum; i++)
	{
		broadon_rsa_signer.insert(std::pair<std::string, KeyBag::BroadOnRsaSignerProfile>(default_broadon_profile[i][isDev].issuer, default_broadon_profile[i][isDev].profile));
	}
}

void ctrtool::KeyBagInitializer::addSigHaxSignatures(bool isDev)
{
	struct SigHaxSignature
	{
		byte_t key_index;
		KeyBag::Rsa2048Signature signature;
	};

	static const size_t kSigHaxSignatureNum = 3;
	SigHaxSignature sighax_signatures[kSigHaxSignatureNum][2] = 
	{
		{
			// retail
			{RSAKEY_FIRM_NAND, {0xB6, 0x72, 0x45, 0x31, 0xC4, 0x48, 0x65, 0x7A, 0x2A, 0x2E, 0xE3, 0x06,
		0x45, 0x7E, 0x35, 0x0A, 0x10, 0xD5, 0x44, 0xB4, 0x28, 0x59, 0xB0, 0xE5,
		0xB0, 0xBE, 0xD2, 0x75, 0x34, 0xCC, 0xCC, 0x2A, 0x4D, 0x47, 0xED, 0xEA,
		0x60, 0xA7, 0xDD, 0x99, 0x93, 0x99, 0x50, 0xA6, 0x35, 0x7B, 0x1E, 0x35,
		0xDF, 0xC7, 0xFA, 0xC7, 0x73, 0xB7, 0xE1, 0x2E, 0x7C, 0x14, 0x81, 0x23,
		0x4A, 0xF1, 0x41, 0xB3, 0x1C, 0xF0, 0x8E, 0x9F, 0x62, 0x29, 0x3A, 0xA6,
		0xBA, 0xAE, 0x24, 0x6C, 0x15, 0x09, 0x5F, 0x8B, 0x78, 0x40, 0x2A, 0x68,
		0x4D, 0x85, 0x2C, 0x68, 0x05, 0x49, 0xFA, 0x5B, 0x3F, 0x14, 0xD9, 0xE8,
		0x38, 0xA2, 0xFB, 0x9C, 0x09, 0xA1, 0x5A, 0xBB, 0x40, 0xDC, 0xA2, 0x5E,
		0x40, 0xA3, 0xDD, 0xC1, 0xF5, 0x8E, 0x79, 0xCE, 0xC9, 0x01, 0x97, 0x43,
		0x63, 0xA9, 0x46, 0xE9, 0x9B, 0x43, 0x46, 0xE8, 0xA3, 0x72, 0xB6, 0xCD,
		0x55, 0xA7, 0x07, 0xE1, 0xEA, 0xB9, 0xBE, 0xC0, 0x20, 0x0B, 0x5B, 0xA0,
		0xB6, 0x61, 0x23, 0x6A, 0x87, 0x08, 0xD7, 0x04, 0x51, 0x7F, 0x43, 0xC6,
		0xC3, 0x8E, 0xE9, 0x56, 0x01, 0x11, 0xE1, 0x40, 0x5E, 0x5E, 0x8E, 0xD3,
		0x56, 0xC4, 0x9C, 0x4F, 0xF6, 0x82, 0x3D, 0x12, 0x19, 0xAF, 0xAE, 0xEB,
		0x3D, 0xF3, 0xC3, 0x6B, 0x62, 0xBB, 0xA8, 0x8F, 0xC1, 0x5B, 0xA8, 0x64,
		0x8F, 0x93, 0x33, 0xFD, 0x9F, 0xC0, 0x92, 0xB8, 0x14, 0x6C, 0x3D, 0x90,
		0x8F, 0x73, 0x15, 0x5D, 0x48, 0xBE, 0x89, 0xD7, 0x26, 0x12, 0xE1, 0x8E,
		0x4A, 0xA8, 0xEB, 0x9B, 0x7F, 0xD2, 0xA5, 0xF7, 0x32, 0x8C, 0x4E, 0xCB,
		0xFB, 0x00, 0x83, 0x83, 0x3C, 0xBD, 0x5C, 0x98, 0x3A, 0x25, 0xCE, 0xB8,
		0xB9, 0x41, 0xCC, 0x68, 0xEB, 0x01, 0x7C, 0xE8, 0x7F, 0x5D, 0x79, 0x3A,
		0xCA, 0x09, 0xAC, 0xF7}},
			// dev
			{RSAKEY_FIRM_NAND, {0x88, 0x69, 0x7C, 0xDC, 0xA9, 0xD1, 0xEA, 0x31, 0x82, 0x56, 0xFC, 0xD9,
		0xCE, 0xD4, 0x29, 0x64, 0xC1, 0xE9, 0x8A, 0xBC, 0x64, 0x86, 0xB2, 0xF1,
		0x28, 0xEC, 0x02, 0xE7, 0x1C, 0x5A, 0xE3, 0x5D, 0x63, 0xD3, 0xBF, 0x12,
		0x46, 0x13, 0x40, 0x81, 0xAF, 0x68, 0x75, 0x47, 0x87, 0xFC, 0xB9, 0x22,
		0x57, 0x1D, 0x7F, 0x61, 0xA3, 0x0D, 0xE4, 0xFC, 0xFA, 0x82, 0x93, 0xA9,
		0xDA, 0x51, 0x23, 0x96, 0xF1, 0x31, 0x9A, 0x36, 0x49, 0x68, 0x46, 0x4C,
		0xA9, 0x80, 0x6E, 0x0A, 0x52, 0x56, 0x74, 0x86, 0x75, 0x4C, 0xDD, 0xD4,
		0xC3, 0xA6, 0x2B, 0xDC, 0xE2, 0x55, 0xE0, 0xDE, 0xEC, 0x23, 0x01, 0x29,
		0xC1, 0xBA, 0xE1, 0xAE, 0x95, 0xD7, 0x86, 0x86, 0x56, 0x37, 0xC1, 0xE6,
		0x5F, 0xAE, 0x83, 0xED, 0xF8, 0xE7, 0xB0, 0x7D, 0x17, 0xC0, 0xAA, 0xDA,
		0x8F, 0x05, 0x5B, 0x64, 0x0D, 0x45, 0xAB, 0x0B, 0xAC, 0x76, 0xFF, 0x7B,
		0x34, 0x39, 0xF5, 0xA4, 0xBF, 0xE8, 0xF7, 0xE0, 0xE1, 0x03, 0xBC, 0xE9,
		0x95, 0xFA, 0xD9, 0x13, 0xFB, 0x72, 0x9D, 0x3D, 0x03, 0x0B, 0x26, 0x44,
		0xEC, 0x48, 0x39, 0x64, 0x24, 0xE0, 0x56, 0x3A, 0x1B, 0x3E, 0x6A, 0x1F,
		0x68, 0x0B, 0x39, 0xFC, 0x14, 0x61, 0x88, 0x6F, 0xA7, 0xA6, 0x0B, 0x6B,
		0x56, 0xC5, 0xA8, 0x46, 0x55, 0x4A, 0xE6, 0x48, 0xFC, 0x46, 0xE3, 0x0E,
		0x24, 0x67, 0x8F, 0xAF, 0x1D, 0xC3, 0xCE, 0xB1, 0x0C, 0x2A, 0x95, 0x0F,
		0x4F, 0xFA, 0x20, 0x83, 0x23, 0x4E, 0xD8, 0xDC, 0xC3, 0x58, 0x7A, 0x6D,
		0x75, 0x1A, 0x7E, 0x9A, 0xFA, 0x06, 0x15, 0x69, 0x55, 0x08, 0x4F, 0xF2,
		0x72, 0x5B, 0x69, 0x8E, 0xB1, 0x74, 0x54, 0xD9, 0xB0, 0x2B, 0x6B, 0x76,
		0xBE, 0x47, 0xAB, 0xBE, 0x20, 0x62, 0x94, 0x36, 0x69, 0x87, 0xA4, 0xCA,
		0xB4, 0x2C, 0xBD, 0x0B}},
		},
		{
			// retail
			{RSAKEY_FIRM_RECOVERY, {0x37, 0xE9, 0x6B, 0x10, 0xBA, 0xF2, 0x8C, 0x74, 0xA7, 0x10, 0xEF, 0x35,
		0x82, 0x4C, 0x93, 0xF5, 0xFB, 0xB3, 0x41, 0xCE, 0xE4, 0xFB, 0x44, 0x6C,
		0xE4, 0xD2, 0x90, 0xAB, 0xFC, 0xEF, 0xAC, 0xB0, 0x63, 0xA9, 0xB5, 0x5B,
		0x3E, 0x8A, 0x65, 0x51, 0x1D, 0x90, 0x0C, 0x5A, 0x6E, 0x94, 0x03, 0xAA,
		0xB5, 0x94, 0x3C, 0xEF, 0x3A, 0x1E, 0x88, 0x2B, 0x77, 0xD2, 0x34, 0x79,
		0x42, 0xB9, 0xE9, 0xEB, 0x0D, 0x75, 0x66, 0x37, 0x0F, 0x0C, 0xB7, 0x31,
		0x0C, 0x38, 0xCB, 0x4A, 0xC9, 0x40, 0xD1, 0xA6, 0xBB, 0x47, 0x6B, 0xCC,
		0x2C, 0x48, 0x7D, 0x1C, 0x53, 0x21, 0x20, 0xF1, 0xD2, 0xA3, 0x7D, 0xDB,
		0x3E, 0x36, 0xF8, 0xA2, 0x94, 0x5B, 0xD8, 0xB1, 0x6F, 0xB3, 0x54, 0x98,
		0x03, 0x84, 0x99, 0x8E, 0xCC, 0x38, 0x0C, 0xD5, 0xCF, 0x85, 0x30, 0xF1,
		0xDA, 0xD2, 0xFD, 0x74, 0xBA, 0x35, 0xAC, 0xB9, 0xC9, 0xDA, 0x2C, 0x13,
		0x1C, 0xB2, 0x95, 0x73, 0x6A, 0xE7, 0xEF, 0xA0, 0xD2, 0x68, 0xEE, 0x01,
		0x87, 0x2E, 0xF0, 0x33, 0x05, 0x8A, 0xBA, 0x07, 0xB5, 0xC6, 0x84, 0xEA,
		0xD6, 0x0D, 0x76, 0xEA, 0x84, 0xA1, 0x8D, 0x86, 0x63, 0x07, 0xAA, 0xAA,
		0xB7, 0x64, 0x78, 0x6E, 0x39, 0x6F, 0x2F, 0x8B, 0x63, 0x0E, 0x60, 0xE3,
		0x0E, 0x3F, 0x1C, 0xD8, 0xA6, 0x7D, 0x02, 0xF0, 0xA8, 0x81, 0x52, 0xDE,
		0x7A, 0x9E, 0x0D, 0xD5, 0xE6, 0x4A, 0xB7, 0x59, 0x3A, 0x37, 0x01, 0xE4,
		0x84, 0x6B, 0x6F, 0x33, 0x8D, 0x22, 0xFD, 0x45, 0x5D, 0x45, 0xDF, 0x21,
		0x2C, 0x55, 0x77, 0x26, 0x6A, 0xA8, 0xC3, 0x67, 0xAE, 0x6E, 0x4C, 0xE8,
		0x9D, 0xF4, 0x16, 0x91, 0xBF, 0x1F, 0x7F, 0xE5, 0x8F, 0x22, 0x61, 0xF5,
		0xD2, 0x51, 0xDF, 0x36, 0xDE, 0x9F, 0x5A, 0xF1, 0xF3, 0x68, 0xE6, 0x50,
		0xD5, 0x76, 0x81, 0x0B}},
			// dev
			{RSAKEY_FIRM_RECOVERY, {0x18, 0x72, 0x2B, 0xC7, 0x6D, 0xC3, 0x60, 0x2E, 0x2C, 0x01, 0x71, 0xF3,
		0xBC, 0xA1, 0x2A, 0xB4, 0x0E, 0xA6, 0xD1, 0x12, 0xAE, 0xFB, 0xEC, 0xF4,
		0xBE, 0x7A, 0x2A, 0x58, 0xFF, 0x75, 0x90, 0x58, 0xA9, 0x3C, 0x95, 0xCD,
		0xA9, 0xB3, 0xB6, 0x76, 0xD0, 0x9A, 0x4E, 0x4C, 0x9E, 0x84, 0x2E, 0x5C,
		0x68, 0x22, 0x9A, 0x6A, 0x9D, 0x77, 0xFA, 0xC7, 0x64, 0x45, 0xE7, 0x8E,
		0xB5, 0xB3, 0x63, 0xF8, 0xC6, 0x6B, 0x16, 0x6B, 0xE6, 0x5A, 0xFA, 0xE4,
		0x0A, 0x14, 0x85, 0xA3, 0x64, 0xC2, 0xC1, 0x3B, 0x85, 0x5C, 0xEE, 0xDE,
		0x3D, 0xFE, 0xAC, 0xEC, 0x68, 0xDD, 0x6B, 0x86, 0x87, 0xDD, 0x6D, 0xF8,
		0xB6, 0xD3, 0x21, 0x3F, 0x72, 0x25, 0x2E, 0x7C, 0x03, 0xC0, 0x27, 0xEE,
		0x60, 0x79, 0xF9, 0xC5, 0xE0, 0x29, 0x0E, 0x5D, 0xB8, 0xCA, 0x0B, 0xBC,
		0xF3, 0x0F, 0xCA, 0xD7, 0x2E, 0xB6, 0x37, 0xA1, 0x70, 0xC4, 0xA2, 0xF4,
		0x1D, 0x96, 0xBF, 0x7D, 0x51, 0x7A, 0x2F, 0x4F, 0x33, 0x59, 0x30, 0xDC,
		0x5E, 0x97, 0x92, 0xD7, 0x8E, 0xDF, 0xB5, 0x1D, 0xC7, 0x9A, 0xD9, 0xD7,
		0xA4, 0xE7, 0xF1, 0xED, 0x4D, 0x5A, 0x5C, 0x62, 0x1B, 0x62, 0x45, 0xA7,
		0xF1, 0x65, 0x22, 0x56, 0x01, 0x1D, 0xC3, 0x2C, 0x49, 0xB9, 0x55, 0x30,
		0x4A, 0x42, 0x30, 0x09, 0xE2, 0xB7, 0x80, 0x72, 0xCE, 0xBC, 0x12, 0xB3,
		0x85, 0xB7, 0x2F, 0x92, 0x6F, 0x19, 0x31, 0x8D, 0x64, 0x07, 0x5F, 0x09,
		0x27, 0x8F, 0xBA, 0x84, 0x48, 0xFD, 0x24, 0x84, 0xB8, 0x26, 0x54, 0xA5,
		0x5D, 0x06, 0x45, 0x42, 0xA8, 0xF5, 0xD9, 0xF9, 0x82, 0x8C, 0xDA, 0x5E,
		0x60, 0xD3, 0x1A, 0x40, 0xCF, 0x8E, 0xF1, 0x8D, 0x02, 0x73, 0x10, 0xDA,
		0x4F, 0x80, 0x79, 0x88, 0xBC, 0x75, 0x3C, 0x1E, 0xB3, 0xB3, 0xFC, 0x06,
		0x20, 0x7E, 0x84, 0xDE}},
		},
		{
			// retail
			{RSAKEY_NCSD_NAND, {0x6C, 0xF5, 0x2F, 0x89, 0xF3, 0x78, 0x12, 0x0B, 0xFA, 0x4E, 0x10, 0x61,
		0xD7, 0x36, 0x16, 0x34, 0xD9, 0xA2, 0x54, 0xA4, 0xF5, 0x7A, 0xA5, 0xBD,
		0x9F, 0x2C, 0x30, 0x93, 0x4F, 0x0E, 0x68, 0xCB, 0xE6, 0x61, 0x1D, 0x90,
		0xD7, 0x4C, 0xAA, 0xAC, 0xB6, 0xA9, 0x95, 0x56, 0x56, 0x47, 0x33, 0x3D,
		0xC1, 0x70, 0x92, 0xD3, 0x20, 0x13, 0x10, 0x89, 0xCC, 0xCD, 0x63, 0x31,
		0xCB, 0x3A, 0x59, 0x5D, 0x1B, 0xA2, 0x99, 0xA3, 0x2F, 0xF4, 0xD8, 0xE5,
		0xDD, 0x1E, 0xB4, 0x6A, 0x2A, 0x57, 0x93, 0x5F, 0x6F, 0xE6, 0x37, 0x32,
		0x2D, 0x3B, 0xC4, 0xF6, 0x7C, 0xFE, 0xD6, 0xC2, 0x25, 0x4C, 0x08, 0x9C,
		0x62, 0xFA, 0x11, 0xD0, 0x82, 0x4A, 0x84, 0x4C, 0x79, 0xEE, 0x5A, 0x4F,
		0x27, 0x3D, 0x46, 0xC2, 0x3B, 0xBB, 0xF0, 0xA2, 0xAF, 0x6A, 0xCA, 0xDB,
		0xE6, 0x46, 0xF4, 0x6B, 0x86, 0xD1, 0x28, 0x9C, 0x7F, 0xF7, 0xE8, 0x16,
		0xCF, 0xDA, 0x4B, 0xC3, 0x3D, 0xFF, 0x9D, 0x17, 0x5A, 0xC6, 0x9F, 0x72,
		0x40, 0x6C, 0x07, 0x1B, 0x51, 0xF4, 0x5A, 0x1A, 0xCB, 0x87, 0xF1, 0x68,
		0xC1, 0x77, 0xCB, 0x9B, 0xE6, 0xC3, 0x92, 0xF0, 0x34, 0x18, 0x49, 0xAE,
		0x5D, 0x51, 0x0D, 0x26, 0xEE, 0xC1, 0x09, 0x7B, 0xEB, 0xFB, 0x9D, 0x14,
		0x4A, 0x16, 0x47, 0x30, 0x1B, 0xEA, 0xF9, 0x52, 0x0D, 0x22, 0xC5, 0x5A,
		0xF4, 0x6D, 0x49, 0x28, 0x4C, 0xC7, 0xF9, 0xFB, 0xBA, 0x37, 0x1A, 0x6D,
		0x6E, 0x4C, 0x55, 0xF1, 0xE5, 0x36, 0xD6, 0x23, 0x7F, 0xFF, 0x54, 0xB3,
		0xE9, 0xC1, 0x1A, 0x20, 0xCF, 0xCC, 0xAC, 0x0C, 0x6B, 0x06, 0xF6, 0x95,
		0x76, 0x6A, 0xCE, 0xB1, 0x8B, 0xE3, 0x32, 0x99, 0xA9, 0x4C, 0xFC, 0xA7,
		0xE2, 0x58, 0x81, 0x86, 0x52, 0xF7, 0x52, 0x6B, 0x30, 0x6B, 0x52, 0xE0,
		0xAE, 0xD0, 0x42, 0x18}},
			// dev
			{RSAKEY_NCSD_NAND, {0x53, 0xCB, 0x0E, 0x4E, 0xB1, 0xA6, 0xFF, 0x84, 0x28, 0x4B, 0xE0, 0xE7,
		0x38, 0x5A, 0xB4, 0xA6, 0x86, 0xA8, 0xBB, 0xCB, 0xC1, 0x61, 0x02, 0x47,
		0x92, 0x80, 0xE0, 0x58, 0x36, 0x55, 0xD2, 0x71, 0x3F, 0xE5, 0x06, 0xFA,
		0xEE, 0x74, 0xF8, 0xD1, 0x0F, 0x12, 0x20, 0x44, 0x1C, 0xC2, 0xFF, 0x5D,
		0x6D, 0xDE, 0x99, 0xBE, 0x79, 0xC1, 0x9B, 0x38, 0x6C, 0xAF, 0x68, 0xD5,
		0xEB, 0x8C, 0xED, 0x1A, 0xAB, 0x4D, 0x24, 0x3C, 0x5F, 0x39, 0x86, 0x80,
		0xD3, 0x1C, 0xD2, 0xE3, 0xC9, 0xDD, 0x56, 0x70, 0xF2, 0xA8, 0x8D, 0x56,
		0x3B, 0x8F, 0x65, 0xF5, 0xB2, 0x34, 0xFD, 0x2E, 0xBB, 0x3B, 0xE4, 0x4A,
		0x3B, 0x6C, 0x30, 0x27, 0x22, 0xA2, 0xAD, 0xFB, 0x56, 0xAE, 0x3E, 0x1F,
		0x64, 0x17, 0xBD, 0xEC, 0x1E, 0x5A, 0x86, 0xAA, 0xBB, 0xAF, 0xBE, 0x94,
		0x19, 0xAC, 0xA8, 0xFD, 0xCD, 0x45, 0xE2, 0xCD, 0xF1, 0xEB, 0x69, 0x5F,
		0x6E, 0xA8, 0x78, 0x16, 0x12, 0x2D, 0x7B, 0xE9, 0x8E, 0xEF, 0x92, 0xC0,
		0x81, 0x4B, 0x16, 0xB2, 0x15, 0xB3, 0x1D, 0x8C, 0x81, 0x3B, 0xB3, 0x55,
		0xCE, 0xA8, 0x13, 0x8F, 0xB3, 0xBF, 0x23, 0x74, 0x24, 0x68, 0x42, 0xCD,
		0x91, 0xE1, 0xF9, 0xAA, 0xFF, 0x76, 0x87, 0x86, 0x17, 0xCE, 0x02, 0x06,
		0x47, 0x77, 0xAE, 0xA0, 0x87, 0x6A, 0x2C, 0x24, 0x5C, 0x78, 0x43, 0x41,
		0xCD, 0xEE, 0x90, 0xD6, 0x91, 0x74, 0x59, 0x08, 0xA6, 0xFF, 0x9C, 0xE7,
		0x81, 0x16, 0x67, 0x96, 0xF9, 0xF1, 0x23, 0x8F, 0x88, 0x4C, 0x84, 0xD6,
		0xF1, 0xEE, 0xBB, 0x2E, 0x40, 0xB4, 0xBC, 0xA0, 0x0A, 0x7B, 0x1E, 0x91,
		0x3E, 0x09, 0x80, 0xD2, 0x9F, 0xF6, 0x06, 0x1D, 0x8A, 0xA9, 0x44, 0xC6,
		0x63, 0xF2, 0x63, 0x81, 0x27, 0xF7, 0xCC, 0xAB, 0x6F, 0xC7, 0x15, 0x38,
		0x47, 0x1A, 0x51, 0x38}},
		},
		
	};

	for (size_t i = 0; i < kSigHaxSignatureNum; i++)
	{
		rsa_sighax_signature.insert(std::pair<byte_t, KeyBag::Rsa2048Signature>(sighax_signatures[i][isDev].key_index, sighax_signatures[i][isDev].signature));
	}
}

void ctrtool::KeyBagInitializer::importSeedDb(const std::shared_ptr<tc::io::ISource>& seed_db_source)
{
	// validate source exists
	if (seed_db_source == nullptr)
	{
		throw tc::ArgumentNullException("ctrtool::KeyBagInitializer", "SeedDb source is null.");
	}

	// validate source has minimum size
	if (seed_db_source->length() < (sizeof(SeedDbHeader) + sizeof(SeedDbEntry)))
	{
		throw tc::ArgumentNullException("ctrtool::KeyBagInitializer", "SeedDb source is too small.");
	}

	// read header
	auto hdr_data = seed_db_source->pullData(0, sizeof(SeedDbHeader));
	SeedDbHeader* hdr = (SeedDbHeader*)hdr_data.data();

	// check padding
	for (size_t i = 0; i < hdr->padding.size(); i++)
	{
		if (hdr->padding[i] != 0)
		{
			throw tc::ArgumentNullException("ctrtool::KeyBagInitializer", "SeedDb header had invalid padding.");
		}
	}

	// get entry num
	size_t n_entries = hdr->n_entries.unwrap();

	// check size of source fits expected total seeddb size
	if (seed_db_source->length() < static_cast<int64_t>(sizeof(SeedDbHeader) + n_entries * sizeof(SeedDbEntry)))
	{
		throw tc::ArgumentNullException("ctrtool::KeyBagInitializer", "SeedDb source is too small.");
	}

	// read entries
	auto entry_data = seed_db_source->pullData(sizeof(SeedDbHeader), sizeof(SeedDbEntry) * n_entries);
	SeedDbEntry* entry = (SeedDbEntry*)entry_data.data();

	// import entries
	for (size_t i = 0; i < n_entries; i++)
	{
		seed_db.insert(std::pair<uint64_t, KeyBag::Aes128Key>(entry[i].title_id.unwrap(), entry[i].seed));
	}
}

bool ctrtool::KeyBagInitializer::importFallbackKey(tc::Optional<KeyBag::Aes128Key>& key, const std::string& key_str)
{
	// check key_str size
	if (key_str.size() != sizeof(KeyBag::Aes128Key)*2) { return false; }

	// convert to keydata
	auto keydata = tc::cli::FormatUtil::hexStringToBytes(key_str);
	
	// check keydata size
	if (keydata.size() != sizeof(KeyBag::Aes128Key)) { return false; }

	// save key
	KeyBag::Aes128Key tmp;
	memcpy(tmp.data(), keydata.data(), tmp.size());
	key = tmp;

	return true;
}