// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package registry

import (
	"context"
	"crypto/aes"
	"crypto/cipher"
	"encoding/base64"
	"errors"

	"github.com/drone/drone-yaml/yaml"
	"github.com/drone/drone/core"
	"github.com/drone/drone/logger"
	"github.com/drone/drone/plugin/registry/auths"
)

// Encrypted returns a new encrypted registry credentials
// provider that sources credentials from the encrypted strings
// in the yaml file.
func Encrypted() core.RegistryService {
	return new(encrypted)
}

type encrypted struct {
}

func (c *encrypted) List(ctx context.Context, in *core.RegistryArgs) ([]*core.Registry, error) {
	var results []*core.Registry

	for _, match := range in.Pipeline.PullSecrets {
		logger := logger.FromContext(ctx).
			WithField("name", match).
			WithField("kind", "secret")
		logger.Trace("image_pull_secrets: find encrypted secret")

		// lookup the named secret in the manifest. If the
		// secret does not exist, return a nil variable,
		// allowing the next secret controller in the chain
		// to be invoked.
		data, ok := getEncrypted(in.Conf, match)
		if !ok {
			logger.Trace("image_pull_secrets: no matching encrypted secret in yaml")
			return nil, nil
		}

		decoded, err := base64.StdEncoding.DecodeString(string(data))
		if err != nil {
			logger.WithError(err).Trace("image_pull_secrets: cannot decode secret")
			return nil, err
		}

		decrypted, err := decrypt(decoded, []byte(in.Repo.Secret))
		if err != nil {
			logger.WithError(err).Trace("image_pull_secrets: cannot decrypt secret")
			return nil, err
		}

		parsed, err := auths.ParseBytes(decrypted)
		if err != nil {
			logger.WithError(err).Trace("image_pull_secrets: cannot parse decrypted secret")
			return nil, err
		}

		logger.Trace("image_pull_secrets: found encrypted secret")
		results = append(results, parsed...)
	}

	return results, nil
}

func getEncrypted(manifest *yaml.Manifest, match string) (data string, ok bool) {
	for _, resource := range manifest.Resources {
		secret, ok := resource.(*yaml.Secret)
		if !ok {
			continue
		}
		if secret.Name != match {
			continue
		}
		if secret.Data == "" {
			continue
		}
		return secret.Data, true
	}
	return
}

func decrypt(ciphertext []byte, key []byte) (plaintext []byte, err error) {
	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, err
	}

	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, err
	}

	if len(ciphertext) < gcm.NonceSize() {
		return nil, errors.New("malformed ciphertext")
	}

	return gcm.Open(nil,
		ciphertext[:gcm.NonceSize()],
		ciphertext[gcm.NonceSize():],
		nil,
	)
}
