/*
  libreiserfs - a library for manipulating reiserfs partitions
  Copyright (C) 2001-2004 Yury Umanets <torque@ukrpost.net>.
                                                                                                 
  This program is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.
                                                                                                 
  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
  details.
                                                                                                 
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 Temple
  Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <string.h>
#include <stdio.h>

#include <reiserfs/reiserfs.h>
#include <reiserfs/debug.h>

reiserfs_gauge_t *libreiserfs_gauge_create(const char *name, 
					   reiserfs_gauge_handler_t handler,
					   void *data)
{
	reiserfs_gauge_t *gauge;

	if (!(gauge = libreiserfs_calloc(sizeof(*gauge), 0)))
		return NULL;
    
	if (name)
		strncpy(gauge->name, name, sizeof(gauge->name));

	gauge->handler = handler;
	gauge->data = data;

	libreiserfs_gauge_reset(gauge);
	libreiserfs_gauge_enable(gauge);
	libreiserfs_gauge_determined(gauge);
    
	return gauge;
}
    
void libreiserfs_gauge_free(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return);
    
	libreiserfs_free(gauge);
}

void libreiserfs_gauge_reset(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return);
    
	gauge->value = 0;
	gauge->update_header = 1;
	gauge->update_footer = 0;
}

void libreiserfs_gauge_enable(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return);
	gauge->enabled = 1;
}

void libreiserfs_gauge_disable(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return);
	gauge->enabled = 0;
}

void libreiserfs_gauge_set_handler(reiserfs_gauge_t *gauge, 
				   reiserfs_gauge_handler_t handler) 
{
	ASSERT(gauge != NULL, return);
	gauge->handler = handler;
}

reiserfs_gauge_handler_t libreiserfs_gauge_handler(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return NULL);
	return gauge->handler;
}

void libreiserfs_gauge_determined(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return);

	if (gauge->determined)
		return;
	gauge->determined = 1;
}

void libreiserfs_gauge_undetermined(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return);

	if (!gauge->determined)
		return; 
    
	gauge->determined = 0;
	setlinebuf(stderr);
}

int libreiserfs_gauge_is_determined(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return 0);
	return gauge->determined;
}

void libreiserfs_gauge_set_data(reiserfs_gauge_t *gauge, void *data) {
	ASSERT(gauge != NULL, return);
	gauge->data = data;
}

void *libreiserfs_gauge_data(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return NULL);
	return gauge->data;
}

void libreiserfs_gauge_set_name(reiserfs_gauge_t *gauge, const char *name) {
    
	ASSERT(gauge != NULL, return);
	ASSERT(name != NULL, return);
    
	if (!strncmp(gauge->name, name, sizeof(gauge->name)))
		return;
    
	strncpy(gauge->name, name, sizeof(gauge->name));
    
	gauge->update_header = 1;
	libreiserfs_gauge_touch(gauge);
}

char *libreiserfs_gauge_name(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return NULL);
	return gauge->name;
}

void libreiserfs_gauge_set_value(reiserfs_gauge_t *gauge, unsigned int value) {
	ASSERT(gauge != NULL, return);
    
	if (gauge->value == value)
		return;
	
	gauge->value = value;
	libreiserfs_gauge_touch(gauge);
}

unsigned int libreiserfs_gauge_value(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return 0);
    
	return gauge->value;
}

void libreiserfs_gauge_touch(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return);

	if (gauge->handler && gauge->enabled) {
		gauge->handler(gauge->name, gauge->value,
			       gauge->data, gauge->determined,
			       gauge->update_header,
			       gauge->update_footer);
		
		gauge->update_header = 0;
		gauge->update_footer = 0;
	}	
}

void libreiserfs_gauge_done(reiserfs_gauge_t *gauge) {
	ASSERT(gauge != NULL, return);
    
	gauge->value = 100;
    
	gauge->update_footer = 1;
	libreiserfs_gauge_touch(gauge);
}
