/*
	stretch.cc

	Copyright (c) 2004 Tim Goetze <tim@quitte.de>
	
	http://quitte.de/dsp

	time stretching utility for n-channel sound data, employing the
	phase vocoder.

	preferred tab size is 2 spaces.
 */
/*
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
	02111-1307, USA or point your web browser to http://www.gnu.org.
*/

#include "basics.h"

#include <stdio.h>
#include <stdarg.h>
#include <getopt.h>
#include <errno.h>
#include <unistd.h>

#include <sndfile.h>

#include "pvoc.h"

/* /////////////////////////////////////////////////////////////////////// */

static char * self = 0;
static bool chatty = true;
static char error [400];

static void
die (const char * fmt, ...)
{
	va_list list;

	va_start (list, fmt);
	fprintf (stderr, "%s: ", self);
	vfprintf (stderr, fmt, list);
	fprintf (stderr, "\n");
	exit (1);
}

/* // sample conversion ////////////////////////////////////////////////// */

/* i think these #defines are gcc only. other compilers (and big-endian for
 * that matter) are untested. */

#if __BYTE_ORDER == __LITTLE_ENDIAN
	template <class T> 
	inline T le2host (T t) 
		{ 
			return t; 
		}

	template <class T>
	inline T host2le (T t) 
		{ 
			return t; 
		}
#else /* __BIG_ENDIAN, untested. */
	inline short le2host (short i) 
		{
			return ((i >> 8) & 0xFF) | ((i & 0xFF) << 8); 
		}

	inline short host2le (short i) 
		{
			return ((i >> 8) & 0xFF) | ((i & 0xFF) << 8); 
		}
#endif

/* /////////////////////////////////////////////////////////////////////// */

static struct {const char * ext; int fmt;}
formats[] = {
	{"aif", SF_FORMAT_AIFF  },
	{"wav", SF_FORMAT_WAV },
	{"au",  SF_FORMAT_AU  },
	{"snd", SF_FORMAT_AU  },
	{"svx", SF_FORMAT_SVX },
	{"paf", SF_ENDIAN_BIG | SF_FORMAT_PAF },
	{"fap", SF_ENDIAN_LITTLE | SF_FORMAT_PAF  },
	{"nist",SF_FORMAT_NIST  },  
	{"ircam",SF_FORMAT_IRCAM },
	{"sf",  SF_FORMAT_IRCAM },
	{"voc", SF_FORMAT_VOC },
	{"w64", SF_FORMAT_W64 },
	{"raw", SF_FORMAT_RAW },
	{"mat4",SF_FORMAT_MAT4  },
	{"mat5",SF_FORMAT_MAT5  },
	{"mat", SF_FORMAT_MAT4  },
	{"pvf", SF_FORMAT_PVF   },
	{"xi",  SF_FORMAT_XI  },
	{ }
};

class SF
: public SF_INFO
{
	public:
		SNDFILE * file;
		int fd;
		int bits;
		
		SF()
			{
				fd = -1;
				file = 0;
				frames = 0;
				sections = 0;
				seekable = 0;
				bits = 16;
			}

		void init (int _fd, int fs, int _channels)
			{
				fd = _fd;
				samplerate = fs;
				channels = _channels;
			}

		void guess_format (char * ext)
			{
				int i = 0;
				while (1)
				{
					if (!strcasecmp (formats[i].ext, ext))
						break;
				
					if (!formats[++i].ext)
						return;
				}
				
				format = formats[i].fmt;
				
				if (bits == 8)
					format |= SF_FORMAT_PCM_S8;
				else if (bits == 16)
					format |= SF_FORMAT_PCM_16;
				else if (bits == 24)
					format |= SF_FORMAT_PCM_24;
				else if (bits == 32)
					format |= SF_FORMAT_FLOAT;
				else if (bits == 64)
					format |= SF_FORMAT_DOUBLE;

				/* fprintf (stderr, "%s %x %d\n", formats[i].ext, format, bits); */

				/* SF_ENDIAN_FILE is 0 anyway. */
			}

		void init (char * path, SF * sf = 0)
			{
				if (sf == 0)
					file = sf_open (path, SFM_READ, this);
				else
				{
					(*this) = (*sf);

					char * dot = strrchr (path, '.');
					if (dot)
						guess_format (dot + 1);

					file = sf_open (path, SFM_WRITE, this);
				}
				if (file == 0)
				{
					sprintf (error, "error opening '%s'", path);
					throw error;
				}
			}

		~SF()
			{
				if (file) sf_close (file);
			}

		/* read() and write() expect 's' to be a short * buffer the same
		 * size as the 'f' buffer, using it as temp. storage for piped op. */
		int read (int n, float * f, short * s)
			{
				if (fd < 0)
					return sf_readf_float (file, f, n);
				
				int i = ::read (fd, s, sizeof (short) * n * channels);
				i /= sizeof (short);

				for (int j = 0; j < i; ++j)
					f[j] = ((float) le2host (s[j])) / 32768.f;

				return i / channels;
			}

		void write (int n, float * f, short * s)
			{
				if (fd < 0)
				{
					if (sf_writef_float (file, f, n) != n)
						throw "write error";
					return;
				}
				
				for (int i = 0; i < n * channels; ++i)
					s[i] = host2le ((short) (f[i] * 32767.f));

				n *= channels * sizeof (short);
				if (::write (fd, s, n) != n)
					throw "write error";
			}
};

/* /////////////////////////////////////////////////////////////////////// */

class Stretch
{
	public:
		phasevocoder inpv, outpv;
		float * buffer;
		float * frame;
		
		struct {
			int in, out;
		} decfac;

		Stretch (int fs, int fft, int in, int out)
			{
				decfac.in = in;
				decfac.out = out;
		
				frame = new float [fft + 2];

				inpv.init (fs, fft, decfac.in, PVPP_STREAMING);
				outpv.init (fs, fft, decfac.out, PVPP_STREAMING);
			}

		~Stretch()
			{
				delete [] frame;
			}

		/* expects to get decfac.in samples in input[], output will be
		 * decfac.out */
		int stretch (float * buffer)
			{
				inpv.generate_frame (buffer, frame, decfac.in, PVOC_AMP_FREQ);
				return outpv.process_frame (frame, buffer, PVOC_AMP_FREQ);
			}
};

/* /////////////////////////////////////////////////////////////////////// */

void
version()
{
	fprintf (stderr, "stretch " VERSION "\n");
	fprintf (stderr, "Copyright (C) 1981-2004 Regents of the University of California,\n");
	fprintf (stderr, "              2001-2 Richard Dobson, 2004 Tim Goetze\n");
}

void
help()
{
	version();

	fprintf (stderr, "stretch is a sound data time stretching tool.\n");
	fprintf (stderr, "supported sound file formats: ");

	int i = 0;
	while (1)
	{
		if (i)
			fprintf (stderr, ", ");
		fprintf (stderr, formats[i].ext);

		if (!formats[++i].ext)
			break;
	}
	
	fprintf (stderr, "\ntype 'man stretch' for more help.\n");
}

/* /////////////////////////////////////////////////////////////////////// */

int 
main (int argc, char ** argv)
{
	self = argv[0];

	static struct option options [] = {
		{"amplify", 1, 0, 'a'},
		{"fft", 1, 0, 'f'},
		{"times", 1, 0, 't'},
		{"overlap", 1, 0, 'o'},
		{"channels", 1, 0, 'c'},
		{"rate", 1, 0, 'r'},
		{"bits", 1, 0, 'b'},

		{"limit", 0, 0, 'l'},

		{"quiet", 0, 0, 'q'},
		{"version", 0, 0, 'v'},
		{"help", 0, 0, 'h'},
		{ }
	};

	int fft = 1024, overlap = 0, channels = 2, fs = 44100, bits = 16;
	float times = 1.f, amplify = 1.f;
	char * infile = 0, * outfile = 0;
	bool limit = false;

	while (1) 
	{
		int c = getopt_long (argc, argv, "a:f:t:o:c:r:b:lqvh", options, 0);

		if (c < 0)
			break;

		if (c == 'a')
		{
			amplify = atof (optarg);
			if (amplify < .0001 || amplify > 1000)
				die ("amplify by %.2f?", amplify);
		}
		else if (c == 'f')
		{
			fft = atoi (optarg);
			if (fft <= 64)
				die ("fft size too small (%d)\n", fft);
		}
		else if (c == 't')
		{
			times = atof (optarg);
			if (times < .01 || times > 100)
				die ("stretch by %.2f?", times);
		}
		else if (c == 'c')
		{
			channels = atoi (optarg);
			if (channels < 1 || channels > 8)
				die ("%d channels?", channels);
		}
		else if (c == 'r')
		{
			fs = atoi (optarg);
			if (fs < 4000 || fs > 192000)
				die ("samplerate %d?", fs);
		}
		else if (c == 'b')
		{
			bits = atoi (optarg);
			if (bits != 8 && bits != 16 && bits != 24 && bits != 32 && bits != 64)
				die ("depth %d bits?", bits);
		}
		else if (c == 'o')
		{
			overlap = atoi (optarg);
			if (overlap < 1)
				die ("overlap < 1?");
			/* checked later too */
		}
		else if (c == 'l')
			limit = true;
		else if (c == 'q')
			chatty = false;
		else if (c == 'v')
		{
			version();
			return 0;
		}
		else if (c == 'h')
		{
			help();
			return 0;
		}
		else 
			die ("unknown option: %s", optarg);
	}
	
	if (overlap < 1) 
		overlap = fft / 8;
	if (overlap > fft / 2)
		die ("overlap must be less than 1/2 DFT length (%d : %d)", overlap, fft);

	while (optind < argc)
	{
		if (infile == 0)
			infile = argv [optind++];
		else if (outfile == 0)
			outfile = argv [optind++];
		else
			die ("too many arguments");
	}

	if (infile && !strcmp (infile, "-")) infile = 0;
	if (outfile && !strcmp (outfile, "-")) outfile = 0;

	/* // set up ///////////////////////////////////////////////////////////// */

	int stretch = (int) (overlap * times);
	
	if (stretch > fft / 2)
		die ("stretched overlap is > 1/2 DFT length (%d : %d)", stretch, fft);

	struct {
		int in, out;
	} decfac = {overlap, stretch};

	struct {
		SF in, out;
	} sf;
		
	sf.in.bits = sf.out.bits = bits;

	try 
	{
		if (infile)
			sf.in.init (infile);
		else
			sf.in.init (fileno (stdin), fs, channels);
		
		/* infile has the last word on the output format */
		fs = sf.in.samplerate;
		channels = sf.in.channels;

		if (outfile)
			sf.out.init (outfile, &sf.in);
		else
			sf.out.init (fileno (stdout), fs, channels);
	}
	catch (char * s)
	{
		die (s);
	}

	if (chatty)
	{
		fprintf (stderr, "reading %d channel(s) from %s, ", 
				channels, infile ? infile : "<stdin>");
		fprintf (stderr, "writing to %s\n", outfile ? outfile : "<stdout>");
		fprintf (stderr, "DFT length: %d, ", fft);
		fprintf (stderr, "overlap: in=%d, out=%d, ", decfac.in, decfac.out);
		fprintf (stderr, "stretch: %.4f, ", (float) stretch / (float) overlap);
		fprintf (stderr, "amplify: %.2f\n", amplify);

		float diff = fabs (1 - (float) stretch / ((float) overlap * times));

		if (diff > .00001)
			fprintf (stderr, "warning: stretch factor off by %.3f %%\n", diff * 100);
	}

	/* // go ///////////////////////////////////////////////////////////////// */

	try
	{
		Stretch * stretcher [channels];
		
		for (int i = 0; i < channels; ++i)
		{
			stretcher[i] = new Stretch (sf.in.samplerate, fft, decfac.in, decfac.out);
			if (amplify != 1.f)
				stretcher[i]->outpv.scale_synwindow (amplify);
		}

		float interleaved [max (decfac.in, decfac.out) * channels];
		short pipedbuffer [max (decfac.in, decfac.out) * channels];
		float buffer [max (decfac.in, decfac.out)];
		
		int frames = 0;
		do 
		{
			frames = sf.in.read (decfac.in, interleaved, pipedbuffer);

			float * f;
			int write = 0;
			for (int i = 0; i < channels; ++i)
			{
				f = interleaved + i;
				int j;
				
				/* de-interleave */
				for (j = 0; j < frames; ++j, f += channels)
					buffer[j] = *f;

				/* zero tail of final slice */
				memset (buffer + j, 0, sizeof (float) * (decfac.in - frames));

				/* first few frames don't generate output */
				write |= stretcher[i]->stretch (buffer);
				
				f = interleaved + i;
				if (limit)
					for (int j = 0; j < decfac.out; ++j, f += channels)
						*f = clamp (buffer[j], -1, 1);
				else
					for (int j = 0; j < decfac.out; ++j, f += channels)
						*f = buffer[j];
			}

			if (write)
				sf.out.write (decfac.out, interleaved, pipedbuffer);
		} 
		while (frames);
		
		for (int i = 0; i < channels; ++i)
			delete stretcher[i];
	}
	catch (char * s)
	{
		die (s);
	}

	return 0;
}
