#############################################################################
##
## Copyright (C) 2021 Riverbank Computing Limited
## Copyright (C) 2012 Digia Plc
## All rights reserved.
##
## This file is part of the PyQtChart examples.
##
## $QT_BEGIN_LICENSE$
## Licensees holding valid Qt Commercial licenses may use this file in
## accordance with the Qt Commercial License Agreement provided with the
## Software or, alternatively, in accordance with the terms contained in
## a written agreement between you and Digia.
## $QT_END_LICENSE$
##
#############################################################################


import random

from PyQt6.QtCharts import (QAreaSeries, QBarSet, QChart, QChartView,
        QLineSeries, QPieSeries, QScatterSeries, QSplineSeries,
        QStackedBarSeries)
from PyQt6.QtCore import pyqtSlot, QPointF, Qt
from PyQt6.QtGui import QPainter, QPalette
from PyQt6.QtWidgets import (QCheckBox, QComboBox, QGridLayout, QHBoxLayout,
        QLabel, QSizePolicy, QWidget)


class ThemeWidget(QWidget):

    def __init__(self, parent=None):
        super().__init__(parent)

        self.m_charts = []
        self.m_listCount = 3
        self.m_valueMax = 10
        self.m_valueCount = 7
        self.m_dataTable = self.generateRandomData(self.m_listCount,
                self.m_valueMax, self.m_valueCount)
        self.m_themeComboBox = self.createThemeBox()
        self.m_antialiasCheckBox = QCheckBox("Anti-aliasing")
        self.m_animatedComboBox = self.createAnimationBox()
        self.m_legendComboBox = self.createLegendBox()

        self.connectSignals()

        # Create the layout.
        baseLayout = QGridLayout()
        settingsLayout = QHBoxLayout()
        settingsLayout.addWidget(QLabel("Theme:"))
        settingsLayout.addWidget(self.m_themeComboBox)
        settingsLayout.addWidget(QLabel("Animation:"))
        settingsLayout.addWidget(self.m_animatedComboBox)
        settingsLayout.addWidget(QLabel("Legend:"))
        settingsLayout.addWidget(self.m_legendComboBox)
        settingsLayout.addWidget(self.m_antialiasCheckBox)
        settingsLayout.addStretch()
        baseLayout.addLayout(settingsLayout, 0, 0, 1, 3)

        # Create the charts.
        chartView = QChartView(self.createAreaChart())
        baseLayout.addWidget(chartView, 1, 0)
        self.m_charts.append(chartView)

        chartView = QChartView(self.createBarChart(self.m_valueCount))
        baseLayout.addWidget(chartView, 1, 1)
        self.m_charts.append(chartView)

        chartView = QChartView(self.createLineChart())
        baseLayout.addWidget(chartView, 1, 2)
        self.m_charts.append(chartView)

        chartView = QChartView(self.createPieChart())
        # Funny things happen if the pie slice labels no not fit the screen...
        chartView.setSizePolicy(QSizePolicy.Policy.Ignored,
                QSizePolicy.Policy.Ignored)
        baseLayout.addWidget(chartView, 2, 0)
        self.m_charts.append(chartView)

        chartView = QChartView(self.createSplineChart())
        baseLayout.addWidget(chartView, 2, 1)
        self.m_charts.append(chartView)

        chartView = QChartView(self.createScatterChart())
        baseLayout.addWidget(chartView, 2, 2)
        self.m_charts.append(chartView)

        self.setLayout(baseLayout)

        # Set the defaults.
        self.m_antialiasCheckBox.setChecked(True)
        self.updateUI()

    def connectSignals(self):
        self.m_themeComboBox.currentIndexChanged.connect(self.updateUI)
        self.m_antialiasCheckBox.toggled.connect(self.updateUI)
        self.m_animatedComboBox.currentIndexChanged.connect(self.updateUI)
        self.m_legendComboBox.currentIndexChanged.connect(self.updateUI)

    def generateRandomData(self, listCount, valueMax, valueCount):
        random.seed()

        dataTable = []

        for i in range(listCount):
            dataList = []
            yValue = 0.0
            f_valueCount = float(valueCount)

            for j in range(valueCount):
                yValue += random.uniform(0, valueMax) / f_valueCount
                value = QPointF(
                        j + random.random() * self.m_valueMax / f_valueCount,
                        yValue)
                label = "Slice " + str(i) + ":" + str(j)
                dataList.append((value, label))

            dataTable.append(dataList)

        return dataTable

    def createThemeBox(self):
        themeComboBox = QComboBox()

        themeComboBox.addItem("Light", QChart.ChartTheme.ChartThemeLight)
        themeComboBox.addItem("Blue Cerulean",
                QChart.ChartTheme.ChartThemeBlueCerulean)
        themeComboBox.addItem("Dark", QChart.ChartTheme.ChartThemeDark)
        themeComboBox.addItem("Brown Sand",
                QChart.ChartTheme.ChartThemeBrownSand)
        themeComboBox.addItem("Blue NCS", QChart.ChartTheme.ChartThemeBlueNcs)
        themeComboBox.addItem("High Contrast",
                QChart.ChartTheme.ChartThemeHighContrast)
        themeComboBox.addItem("Blue Icy", QChart.ChartTheme.ChartThemeBlueIcy)

        return themeComboBox

    def createAnimationBox(self):
        animationComboBox = QComboBox()

        animationComboBox.addItem("No Animations",
                QChart.AnimationOption.NoAnimation)
        animationComboBox.addItem("GridAxis Animations",
                QChart.AnimationOption.GridAxisAnimations)
        animationComboBox.addItem("Series Animations",
                QChart.AnimationOption.SeriesAnimations)
        animationComboBox.addItem("All Animations",
                QChart.AnimationOption.AllAnimations)

        return animationComboBox

    def createLegendBox(self):
        legendComboBox = QComboBox()

        legendComboBox.addItem("No Legend ", 0)
        legendComboBox.addItem("Legend Top", Qt.AlignmentFlag.AlignTop)
        legendComboBox.addItem("Legend Bottom", Qt.AlignmentFlag.AlignBottom)
        legendComboBox.addItem("Legend Left", Qt.AlignmentFlag.AlignLeft)
        legendComboBox.addItem("Legend Right", Qt.AlignmentFlag.AlignRight)

        return legendComboBox

    def createAreaChart(self):
        chart = QChart()
        chart.setTitle("Area chart")

        # The lower series is initialized to zero values.
        lowerSeries = None
        y_points = []

        for i, data_list in enumerate(self.m_dataTable):
            upperSeries = QLineSeries(chart)
            for j, (value, _) in enumerate(data_list):
                y = value.y()

                if lowerSeries is None:
                    upperSeries.append(QPointF(j, y))
                    y_points.append(y)
                else:
                    new_y = y_points[i] + y
                    upperSeries.append(QPointF(j, new_y))
                    y_points[j] += new_y

            area = QAreaSeries(upperSeries, lowerSeries)
            area.setName("Series " + str(i))
            chart.addSeries(area)
            lowerSeries = upperSeries

        chart.createDefaultAxes()

        return chart

    def createBarChart(self, valueCount):
        chart = QChart()
        chart.setTitle("Bar chart")

        series = QStackedBarSeries(chart)

        for i, data_list in enumerate(self.m_dataTable):
            set = QBarSet("Bar set " + str(i))
            for value, _ in data_list:
                set << value.y()

            series.append(set)

        chart.addSeries(series)
        chart.createDefaultAxes()

        return chart

    def createLineChart(self):
        chart = QChart()
        chart.setTitle("Line chart")

        for i, data_list in enumerate(self.m_dataTable):
            series = QLineSeries(chart)
            for value, _ in data_list:
                series.append(value)

            series.setName("Series " + str(i))
            chart.addSeries(series)

        chart.createDefaultAxes()

        return chart

    def createPieChart(self):
        chart = QChart()
        chart.setTitle("Pie chart")

        pieSize = 1.0 / len(self.m_dataTable)

        for i, data_list in enumerate(self.m_dataTable):
            series = QPieSeries(chart)
            for value, label in data_list:
                slice = series.append(label, value.y())
                if len(series) == 1:
                    slice.setLabelVisible()
                    slice.setExploded()

            hPos = (pieSize / 2) + (i / float(len(self.m_dataTable)))
            series.setPieSize(pieSize)
            series.setHorizontalPosition(hPos)
            series.setVerticalPosition(0.5)

            chart.addSeries(series)

        return chart

    def createSplineChart(self):
        chart = QChart()
        chart.setTitle("Spline chart")

        for i, data_list in enumerate(self.m_dataTable):
            series = QSplineSeries(chart)
            for value, _ in data_list:
                series.append(value)

            series.setName("Series " + str(i))
            chart.addSeries(series)

        chart.createDefaultAxes()

        return chart

    def createScatterChart(self):
        chart = QChart()
        chart.setTitle("Scatter chart")

        for i, data_list in enumerate(self.m_dataTable):
            series = QScatterSeries(chart)
            for value, _ in data_list:
                series.append(value)

            series.setName("Series " + str(i))
            chart.addSeries(series)

        chart.createDefaultAxes()

        return chart

    @pyqtSlot()
    def updateUI(self):
        theme = self.m_themeComboBox.itemData(
                self.m_themeComboBox.currentIndex())

        if self.m_charts[0].chart().theme() != theme:
            for chartView in self.m_charts:
                chartView.chart().setTheme(theme)

            pal = self.window().palette()

            if theme == QChart.ChartTheme.ChartThemeLight:
                pal.setColor(QPalette.ColorRole.Window, 0xf0f0f0)
                pal.setColor(QPalette.ColorRole.WindowText, 0x404044)
            elif theme == QChart.ChartTheme.ChartThemeDark:
                pal.setColor(QPalette.ColorRole.Window, 0x121218)
                pal.setColor(QPalette.ColorRole.WindowText, 0xd6d6d6)
            elif theme == QChart.ChartTheme.ChartThemeBlueCerulean:
                pal.setColor(QPalette.ColorRole.Window, 0x40434a)
                pal.setColor(QPalette.ColorRole.WindowText, 0xd6d6d6)
            elif theme == QChart.ChartTheme.ChartThemeBrownSand:
                pal.setColor(QPalette.ColorRole.Window, 0x9e8965)
                pal.setColor(QPalette.ColorRole.WindowText, 0x404044)
            elif theme == QChart.ChartTheme.ChartThemeBlueNcs:
                pal.setColor(QPalette.ColorRole.Window, 0x018bba)
                pal.setColor(QPalette.ColorRole.WindowText, 0x404044)
            elif theme == QChart.ChartTheme.ChartThemeHighContrast:
                pal.setColor(QPalette.ColorRole.Window, 0xffab03)
                pal.setColor(QPalette.ColorRole.WindowText, 0x181818)
            elif theme == QChart.ChartTheme.ChartThemeBlueIcy:
                pal.setColor(QPalette.ColorRole.Window, 0xcee7f0)
                pal.setColor(QPalette.ColorRole.WindowText, 0x404044)
            else:
                pal.setColor(QPalette.ColorRole.Window, 0xf0f0f0)
                pal.setColor(QPalette.ColorRole.WindowText, 0x404044)

            self.window().setPalette(pal)

        checked = self.m_antialiasCheckBox.isChecked()
        for chartView in self.m_charts:
            chartView.setRenderHint(QPainter.RenderHint.Antialiasing, checked)

        options = QChart.AnimationOption(
                self.m_animatedComboBox.itemData(
                        self.m_animatedComboBox.currentIndex()))

        if self.m_charts[0].chart().animationOptions() != options:
            for chartView in self.m_charts:
                chartView.chart().setAnimationOptions(options)

        alignment = self.m_legendComboBox.itemData(
                self.m_legendComboBox.currentIndex())

        for chartView in self.m_charts:
            legend = chartView.chart().legend()

            if alignment == 0:
                legend.hide()
            else:
                legend.setAlignment(Qt.AlignmentFlag(alignment))
                legend.show()


if __name__ == '__main__':

    import sys

    from PyQt6.QtWidgets import QApplication, QMainWindow

    app = QApplication(sys.argv)

    window = QMainWindow()
    widget = ThemeWidget()
    window.setCentralWidget(widget)
    window.resize(900, 600)
    window.show()

    sys.exit(app.exec())
