
/*
 *  soldevicetypes.cpp
 *
 *  Copyright (C) 2009 David Hubner <hubnerd@ntlworld.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include "soldevicetypes.h"

// ---- Processor

SolProcessorDevice::SolProcessorDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device)
{
    deviceType = Solid::DeviceInterface::Processor;
    setDefaultDeviceText();
}

SolProcessorDevice::SolProcessorDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::Processor;
  
  setDeviceIcon(KIcon("cpu"));
  setDeviceText(i18n("Processors"));
  setDefaultListing(type);
}

void SolProcessorDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolProcessorDevice>(this,QString(),type);
}

void SolProcessorDevice::setDefaultDeviceText() {

  Solid::Processor *prodev = getInterface<Solid::Processor>(); 
  
  if(!prodev) return;
  setText(0,i18n("Processor ") + (QString::number(prodev->number())));
}

QVListLayout *SolProcessorDevice::getInfoPanelLayout() {

  QList<QString> labels;
  const Solid::Processor *prodev = getInterface<Solid::Processor>(); 
  
  if(!prodev) return NULL;
  deviceInfoLayout = new QVListLayout();
  
  QString extensions;
  switch(prodev->instructionSets()) {
    case Solid::Processor::IntelMmx:
      extensions.append(i18n("Intel MMX "));
    case Solid::Processor::IntelSse:
      extensions.append(i18n("Intel SSE "));
    case Solid::Processor::IntelSse2:
      extensions.append(i18n("Intel SSE2 "));
    case Solid::Processor::IntelSse3:
      extensions.append(i18n("Intel SSE3 "));
    case Solid::Processor::IntelSse4:
      extensions.append(i18n("Intel SSE4 "));
    case Solid::Processor::Amd3DNow:
      extensions.append(i18n("AMD 3DNow "));
    case Solid::Processor::AltiVec:
      extensions.append(i18n("ATI IVEC "));
      break;
    default:
      extensions.append(i18n("None"));
      break;
  }
  
  labels << i18n("Processor Number: ")
  << InfoPanel::friendlyString(QString::number(prodev->number())) 
  << i18n("Max Speed: ") 
  << InfoPanel::friendlyString(QString::number(prodev->maxSpeed()))
  << i18n("Supported Instruction Sets: ")
  << extensions;
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// ---- Storage

SolStorageDevice::SolStorageDevice(QTreeWidgetItem *parent, Solid::Device device, storageChildren c) :
  SolDevice(parent, device)
{
  deviceType = Solid::DeviceInterface::StorageDrive;
  setDefaultDeviceText();
   
  if(c == CREATECHILDREN) createDeviceChildren<SolVolumeDevice>(this,device.udi(),Solid::DeviceInterface::StorageVolume);
}

SolStorageDevice::SolStorageDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::StorageDrive;
  
  setDeviceIcon(KIcon("drive-harddisk"));
  setDeviceText(i18n("Storage Drives"));
  setDefaultListing(type);
}

void SolStorageDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolStorageDevice>(this,QString(),type);
}

void SolStorageDevice::setDefaultDeviceText() {
  
  Solid::StorageDrive *stodev = getInterface<Solid::StorageDrive>();
  if(!stodev) return;
  
  QString storageType;
  switch (stodev->driveType()) {
    case Solid::StorageDrive::HardDisk: 
      storageType = i18n("Harddisk");  break;
    case Solid::StorageDrive::CompactFlash:
      storageType = i18n("Compact Flash");  break;
    case Solid::StorageDrive::SmartMedia:
      storageType = i18n("Smart Media");  break;
    case Solid::StorageDrive::SdMmc:
      storageType = i18n("Sd Mmc"); break;
    case Solid::StorageDrive::CdromDrive:
      storageType = i18n("Optical"); break;
    case Solid::StorageDrive::MemoryStick:
      storageType = i18n("Memory Stick"); break;
    case Solid::StorageDrive::Xd:
      storageType = i18n("Xd"); break;
    default:
      storageType = i18n("Unknown"); 
  }
  
  QString deviceText = storageType + i18n(" Drive"); 
  setDeviceText(deviceText);
}

QVListLayout *SolStorageDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::StorageDrive *stodev = getInterface<Solid::StorageDrive>(); 
  
  if(!stodev) return NULL;
  deviceInfoLayout = new QVListLayout();
  
  QString bus;
  switch(stodev->bus()) {
    case Solid::StorageDrive::Ide:
      bus = i18n("IDE"); break;
    case Solid::StorageDrive::Usb:
      bus = i18n("USB"); break;
    case Solid::StorageDrive::Ieee1394:
      bus = i18n("IEEE1394"); break;
    case Solid::StorageDrive::Scsi:
      bus = i18n("SCSI"); break;
    case Solid::StorageDrive::Sata:
      bus = i18n("SATA"); break;
    case Solid::StorageDrive::Platform:
      bus = i18n("Platform"); break;
    default:
      bus = i18n("Unknown"); break;
  }
  
  labels << i18n("Bus: ")
  << bus
  << i18n("Hotpluggable?")
  << InfoPanel::convertTf(stodev->isHotpluggable())
  << i18n("Removable?") 
  << InfoPanel::convertTf(stodev->isRemovable());
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// ---- Network

SolNetworkDevice::SolNetworkDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::NetworkInterface;
  
  setDefaultDeviceText();
  setDefaultDeviceIcon();
}

SolNetworkDevice::SolNetworkDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::NetworkInterface;
  
  setDeviceIcon(KIcon("network-wired"));
  setDeviceText(i18n("Network Interfaces"));
  setDefaultListing(type);
}

void SolNetworkDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolNetworkDevice>(this,QString(),type);
}

void SolNetworkDevice::setDefaultDeviceText() {
    
  Solid::NetworkInterface *netdev = getInterface<Solid::NetworkInterface>(); 
  if(!netdev) return;
      
  QString deviceText = netdev->ifaceName() + " (" + (netdev->isWireless() ? i18n("Wireless") : i18n("Wired")) + ") ";
  setDeviceText(deviceText);
}

void SolNetworkDevice::setDefaultDeviceIcon() {
  
  Solid::NetworkInterface *netdev = getInterface<Solid::NetworkInterface>(); 
  if(!netdev) return;
  
  if(netdev->isWireless() == true) {
    setDeviceIcon(KIcon("network-wireless"));
  } else {
    setDeviceIcon(KIcon("network-wired"));
  }
}

QVListLayout *SolNetworkDevice::getInfoPanelLayout() {
 
  QList<QString> labels;
  Solid::NetworkInterface *netdev = getInterface<Solid::NetworkInterface>(); 
  
  if(!netdev) return NULL;
  deviceInfoLayout = new QVListLayout();
  
  labels << i18n("Hardware Address: ")
  << InfoPanel::friendlyString(netdev->hwAddress())
  << i18n("Wireless?")
  << InfoPanel::convertTf(netdev->isWireless());
  
  deviceInfoLayout->applyQListToLayout(labels); 
  return deviceInfoLayout;
}

// ---- Volume

SolVolumeDevice::SolVolumeDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::StorageVolume;
}

SolVolumeDevice::SolVolumeDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::StorageVolume;
  
  setDefaultListing(type);
}

void SolVolumeDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolVolumeDevice>(this,QString(),type);
}

QVListLayout *SolVolumeDevice::getInfoPanelLayout() {

  QList<QString> labels;
  Solid::StorageVolume *voldev = getInterface<Solid::StorageVolume>(); 
  
  if(!voldev) return NULL;
  deviceInfoLayout = new QVListLayout();
  
  QString usage;
  switch(voldev->usage()) {
    case Solid::StorageVolume::Unused:
      usage = i18n("Unused"); break;
    case Solid::StorageVolume::FileSystem:
      usage = i18n("FileSystem"); break;
    case Solid::StorageVolume::PartitionTable:
      usage = i18n("Partition Table"); break;   
    case Solid::StorageVolume::Raid:
      usage = i18n("Raid"); break;
    case Solid::StorageVolume::Encrypted:
      usage = i18n("Encrypted"); break;
    default: 
      usage = i18n("Unknown"); break;
  }
  
  labels << i18n("FileSystem Type: ") 
  << InfoPanel::friendlyString(voldev->fsType())
  << i18n("Label: ")
  << InfoPanel::friendlyString(voldev->label(),i18n("Not Set"))
  << i18n("Volume Usage: ")
  << usage
  << i18n("UUID: ")
  << InfoPanel::friendlyString(voldev->uuid());

  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}
  
// -- Audio

SolAudioDevice::SolAudioDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::AudioInterface;
}

SolAudioDevice::SolAudioDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::AudioInterface;
  
  setDeviceIcon(KIcon("audio-card"));
  setDeviceText(i18n("Audio Interfaces"));
  setDefaultListing(type);
}

void SolAudioDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  Q_UNUSED(type);
  
  listAlsa();
  listOss();
}

void SolAudioDevice::listAlsa() {
 
  Solid::Predicate alsaPred =
    Solid::Predicate(Solid::DeviceInterface::AudioInterface,"driver", "Alsa",Solid::Predicate::Equals);
  QList<Solid::Device> list = Solid::Device::listFromQuery(alsaPred, QString());
  
  if(list.count() <= 0) return;
  
  createSubItems(ALSA);
  foreach(Solid::Device dev, list) addItem(dev);
}

void SolAudioDevice::listOss() {
  
  Solid::Predicate ossPred =
    Solid::Predicate(Solid::DeviceInterface::AudioInterface,"driver", "OpenSoundSystem",Solid::Predicate::Equals);
  QList<Solid::Device> list = Solid::Device::listFromQuery(ossPred, QString());
  
  if(list.count() <= 0) return;
  
  createSubItems(OSS);
  foreach(Solid::Device dev, list) addItem(dev);
}

void SolAudioDevice::createSubItems(SubMenus menus) {
  
  if(menus == ALSA) {
    alsaSubItem = new SolDevice(this);
    alsaSubItem->setText(0,i18n("Alsa Interfaces"));
  } else {
    ossSubItem = new SolDevice(this);
    ossSubItem->setText(0,i18n("Open Sound System Interfaces"));
  }
}

void SolAudioDevice::addItem(Solid::Device dev) {
  
   Solid::AudioInterface *auddev = getInterface<Solid::AudioInterface>(dev);
   if(!auddev) return;
  
    switch(auddev->driver()) {
      case Solid::AudioInterface::Alsa:
	if(!alsaSubItem) createSubItems(ALSA);
	new SolAudioDevice(alsaSubItem,dev);
	break;
      case Solid::AudioInterface::OpenSoundSystem:
	if(!ossSubItem) createSubItems(OSS);
	new SolAudioDevice(ossSubItem,dev);
	break;
      default:
	new SolAudioDevice(this,dev);
	break;
    }
}

QVListLayout *SolAudioDevice::getInfoPanelLayout() {

  QList<QString> labels;
  Solid::AudioInterface *auddev = getInterface<Solid::AudioInterface>(); 
  
  if(!auddev) return NULL;
  deviceInfoLayout = new QVListLayout();

  QString AiType;
  switch(auddev->deviceType()) {
    case Solid::AudioInterface::AudioControl:
      AiType = i18n("Control"); break;
    case Solid::AudioInterface::AudioInput:
      AiType = i18n("Input"); break;
    case Solid::AudioInterface::AudioOutput:
      AiType = i18n("Output"); break;
    default:
      AiType = i18n("Unknown"); break;
  }
  
  QString ScType;
  switch(auddev->soundcardType()) {
    case Solid::AudioInterface::InternalSoundcard:
      ScType = i18n("Internal Soundcard"); break;
    case Solid::AudioInterface::UsbSoundcard:
      ScType = i18n("Usb Soundcard"); break;
    case Solid::AudioInterface::FirewireSoundcard:
      ScType = i18n("Firewire Soundcard"); break;
    case Solid::AudioInterface::Headset:
      ScType = i18n("Headset"); break;
    case Solid::AudioInterface::Modem:
      ScType = i18n("Modem"); break;
    default:
      ScType = i18n("Unknown"); break;
  }
  
  labels << i18n("Audio Interface Type: ")
  << AiType
  << i18n("SoundCard Type: ")
  << ScType;
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// Button

SolButtonDevice::SolButtonDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::Button;
  
  setDefaultDeviceIcon();
}

SolButtonDevice::SolButtonDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::Button;
  
  setDeviceIcon(KIcon("insert-button"));
  setDeviceText(i18n("Device Buttons"));
  setDefaultListing(type);
}

void SolButtonDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolButtonDevice>(this,QString(),type);
}

void SolButtonDevice::setDefaultDeviceIcon() {
  setDeviceIcon(KIcon("insert-button"));
}

QVListLayout *SolButtonDevice::getInfoPanelLayout() {

  QList<QString> labels;
  Solid::Button *butdev = getInterface<Solid::Button>();
  
  if(!butdev) return NULL;
  deviceInfoLayout = new QVListLayout();
 
  QString type;
  switch(butdev->type()) {
    case Solid::Button::LidButton:
      type = i18n("Lid Button"); break;
    case Solid::Button::PowerButton:
      type = i18n("Power Button"); break;
    case Solid::Button::SleepButton:
      type = i18n("Sleep Button"); break;
    case Solid::Button::TabletButton:
      type = i18n("Tablet Button"); break;
    default:
      type = i18n("Unknown Button"); break;
  }
    
  labels << i18n("Button type: ")
  << type
  << i18n("Has State?")
  << InfoPanel::convertTf(butdev->hasState());
    
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// Media Player

SolMediaPlayerDevice::SolMediaPlayerDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::PortableMediaPlayer;
}

SolMediaPlayerDevice::SolMediaPlayerDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::PortableMediaPlayer;

  setDeviceIcon(KIcon("multimedia-player"));
  setDeviceText(i18n("Multimedia Players"));
  setDefaultListing(type);
}

void SolMediaPlayerDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolMediaPlayerDevice>(this,QString(),type);
}

QVListLayout *SolMediaPlayerDevice::getInfoPanelLayout() {

  QList<QString> labels;
  Solid::PortableMediaPlayer *mpdev = getInterface<Solid::PortableMediaPlayer>(); 
  
  if(!mpdev) return NULL;
  deviceInfoLayout = new QVListLayout();
  
  labels << i18n("Supported Drivers: ")
  << mpdev->supportedDrivers()
  << i18n("Supported Protocols: ")
  << mpdev->supportedProtocols();

  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// Camera

SolCameraDevice::SolCameraDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::Camera;
}

SolCameraDevice::SolCameraDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::Camera;

  setDeviceIcon(KIcon("camera-web"));
  setDeviceText(i18n("Cameras"));
  setDefaultListing(type);
}

void SolCameraDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolVolumeDevice>(this,QString(),type);
}

QVListLayout *SolCameraDevice::getInfoPanelLayout() {

  QList<QString> labels;
  Solid::Camera *camdev = getInterface<Solid::Camera>(); 

  if(!camdev) return NULL;
  deviceInfoLayout = new QVListLayout();
  
  labels << i18n("Supported Drivers: ")
  << camdev->supportedDrivers()
  << i18n("Supported Protocols: ")
  << camdev->supportedProtocols();
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}
  
// Battery

SolBatteryDevice::SolBatteryDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::Battery;
}

SolBatteryDevice::SolBatteryDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::Battery;

  setDeviceIcon(KIcon("battery"));
  setDeviceText(i18n("Batterys"));
  setDefaultListing(type);
}

void SolBatteryDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolBatteryDevice>(this,QString(),type);
}

QVListLayout *SolBatteryDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::Battery *batdev = getInterface<Solid::Battery>(); 

  if(!batdev) return NULL;
  deviceInfoLayout = new QVListLayout();
  
  QString type;
  switch(batdev->type()) {
      case Solid::Battery::PdaBattery:
	type = i18n("PDA"); break;
      case Solid::Battery::UpsBattery:
	type = i18n("UPS"); break;
      case Solid::Battery::MouseBattery:
	type = i18n("Mouse"); break;
      case Solid::Battery::PrimaryBattery:
	type = i18n("Primary"); break;
      case Solid::Battery::KeyboardBattery:
	type = i18n("Keyboard"); break;
      case Solid::Battery::KeyboardMouseBattery:
	type = i18n("Keyboard + Mouse"); break;
      case Solid::Battery::CameraBattery:
	type = i18n("Camera"); break;
      default:
	type = i18n("Unknown");
  }
  
  QString state;
    switch(batdev->type()) {
      case Solid::Battery::Charging:
	state = i18n("Charging"); break;
      case Solid::Battery::Discharging:
	state = i18n("Discharging"); break;
      default:
	state = i18n("No Charge"); 
    }
  
  labels << i18n("Battery Type: ")
  << type
  << i18n("Charge Status: ")
  << state;
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}
  
// Ac Adapter

SolAcAdapterDevice::SolAcAdapterDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::AcAdapter;
}

SolAcAdapterDevice::SolAcAdapterDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::AcAdapter;

  setDeviceIcon(KIcon("kde"));
  setDeviceText(i18n("AC Adapters"));
  setDefaultListing(type);
}

void SolAcAdapterDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolVolumeDevice>(this,QString(),type);
}

QVListLayout *SolAcAdapterDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::AcAdapter *acdev = getInterface<Solid::AcAdapter>(); 
  
  if(!acdev) return NULL;
  deviceInfoLayout = new QVListLayout();
  
  labels << i18n("Is plugged in?")
  << InfoPanel::convertTf(acdev->isPlugged());
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// DVB

SolDvbDevice::SolDvbDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::DvbInterface;
}

SolDvbDevice::SolDvbDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::DvbInterface;

  setDeviceIcon(KIcon("kde"));
  setDeviceText(i18n("Digital Video Broadcasting Devices"));
  setDefaultListing(type);
}

void SolDvbDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolVolumeDevice>(this,QString(),type);
}

// Serial

SolSerialDevice::SolSerialDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::SerialInterface;
}

SolSerialDevice::SolSerialDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::SerialInterface;
  
  setDeviceIcon(KIcon("printer"));
  setDeviceText(i18n("Serial Devices"));
  setDefaultListing(type);
}

void SolSerialDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolSerialDevice>(this,QString(),type);
}

QVListLayout *SolSerialDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::SerialInterface *serdev = getInterface<Solid::SerialInterface>(); 
  
  if(!serdev) return NULL;
  deviceInfoLayout = new QVListLayout();
  
  QString type;
  switch(serdev->serialType()) {
    case Solid::SerialInterface::Platform:
      type = i18n("Platform"); break;
    case Solid::SerialInterface::Usb:
      type = i18n("USB"); break;
    default:
      type = i18n("Unknown"); 
  }
  
  QString port = i18n("Unknown");
  if(serdev->port() != -1) port = QString::number(serdev->port());

  labels << i18n("Serial Type: ")
  << type
  << i18n("Port: ")
  << port;
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

//  Smartcard

SolSmartCardDevice::SolSmartCardDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::SmartCardReader;
}

SolSmartCardDevice::SolSmartCardDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::SmartCardReader;

  setDeviceIcon(KIcon("kde"));
  setDeviceText(i18n("Smart Card Devices"));
  setDefaultListing(type);
}

void SolSmartCardDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolSmartCardDevice>(this,QString(),type);
}

QVListLayout *SolSmartCardDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::SmartCardReader *scdev = getInterface<Solid::SmartCardReader>(); 
  
  if(!scdev) return NULL;
  deviceInfoLayout = new QVListLayout();
  
  QString type;
  switch(scdev->readerType()) {
    case Solid::SmartCardReader::CardReader:
      type = i18n("Card Reader"); break;
    case Solid::SmartCardReader::CryptoToken:
      type = i18n("Crypto Token"); break;
    default:
      type = i18n("Unknown"); 
  }
 
  labels << i18n("SmartCard Type: ")
  << type;
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}

// Video

SolVideoDevice::SolVideoDevice(QTreeWidgetItem *parent, Solid::Device device) :
  SolDevice(parent, device) 
{
  deviceType = Solid::DeviceInterface::Video;
}

SolVideoDevice::SolVideoDevice(Solid::DeviceInterface::Type type) :
  SolDevice(type)
{ 
  deviceType = Solid::DeviceInterface::Video;
  
  setDeviceIcon(KIcon("video-display"));
  setDeviceText(i18n("Video Devices (Not Video Display)"));
  setDefaultListing(type);
}

void SolVideoDevice::setDefaultListing(Solid::DeviceInterface::Type type) {
 
  createDeviceChildren<SolVideoDevice>(this,QString(),type);
}

QVListLayout *SolVideoDevice::getInfoPanelLayout() {
  
  QList<QString> labels;
  Solid::Video *viddev = getInterface<Solid::Video>(); 
  
  if(!viddev) return NULL;
  deviceInfoLayout = new QVListLayout();
  
  labels << i18n("Supported Drivers: ")
  << viddev->supportedDrivers()
  << i18n("Supported Protocols: ")
  << viddev->supportedProtocols();
  
  deviceInfoLayout->applyQListToLayout(labels);
  return deviceInfoLayout;
}