/**
 * ===========================================
 * LibBase : a free Java utility library
 * ===========================================
 *
 * Project Info:  http://reporting.pentaho.org/libbase
 *
 * (C) Copyright 2007,2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * ------------
 * StackableException
 * ------------
 * (C) Copyright 2006, by Pentaho Corporation.
 */



package org.pentaho.reporting.libraries.base.util;

import java.io.PrintStream;
import java.io.PrintWriter;

/**
 * A baseclass for exceptions, which could have parent exceptions. These parent exceptions
 * are raised in a subclass and are now wrapped into a subclass of this Exception.
 * <p/>
 * The parents are printed when this exception is printed. This class exists mainly for
 * debugging reasons, as with them it is easier to detect the root cause of an error.
 * <p/>
 * <!-- In a perfect world there would be no need for such a class :)-->
 *
 * @author Thomas Morgner
 * @noinspection UseOfSystemOutOrSystemErr
 */
public abstract class StackableException extends Exception
{

  /**
   * The parent exception.
   */
  private Exception parent;
  private static final long serialVersionUID = -8649054607849486697L;

  /**
   * Creates a StackableRuntimeException with no message and no parent.
   */
  protected StackableException()
  {
    super();
  }

  /**
   * Creates an exception.
   *
   * @param message the exception message.
   * @param ex      the parent exception.
   */
  protected StackableException(final String message, final Exception ex)
  {
    super(message);
    this.parent = ex;
  }

  /**
   * Creates an exception.
   *
   * @param message the exception message.
   */
  protected StackableException(final String message)
  {
    super(message);
  }

  /**
   * Returns the parent exception (possibly null).
   *
   * @return the parent exception.
   */
  public Exception getParent()
  {
    return this.parent;
  }

  /**
   * Prints the stack trace to the specified stream.
   *
   * @param stream the output stream.
   */
  public void printStackTrace(final PrintStream stream)
  {
    super.printStackTrace(stream);
    if (getParent() != null)
    {
      stream.println("ParentException: ");
      getParent().printStackTrace(stream);
    }
  }

  /**
   * Prints the stack trace to the specified writer.
   *
   * @param writer the writer.
   */
  public void printStackTrace(final PrintWriter writer)
  {
    super.printStackTrace(writer);
    if (getParent() != null)
    {
      writer.println("ParentException: ");
      getParent().printStackTrace(writer);
    }
  }

  /**
   * Prints this <code>Throwable</code> and its backtrace to the
   * standard error stream. This method prints a stack trace for this
   * <code>Throwable</code> object on the error output stream that is
   * the value of the field <code>System.err</code>. The first line of
   * output contains the result of the {@link #toString()} method for
   * this object. Remaining lines represent data previously recorded by
   * the method {@link #fillInStackTrace()}. The format of this
   * information depends on the implementation, but the following
   * example may be regarded as typical:
   * <blockquote><pre>
   * java.lang.NullPointerException
   *         at MyClass.mash(MyClass.java:9)
   *         at MyClass.crunch(MyClass.java:6)
   *         at MyClass.main(MyClass.java:3)
   * </pre></blockquote>
   * This example was produced by running the program:
   * <blockquote><pre>
   * <p/>
   * class MyClass {
   * <p/>
   *     public static void main(String[] argv) {
   *         crunch(null);
   *     }
   *     static void crunch(int[] a) {
   *         mash(a);
   *     }
   * <p/>
   *     static void mash(int[] b) {
   *         System.out.println(b[0]);
   *     }
   * }
   * </pre></blockquote>
   *
   * @see System#err
   */
  public void printStackTrace()
  {
    synchronized (System.err)
    {
      printStackTrace(System.err);
    }
  }
}