/***************************************************************************
 *   Copyright (C) 2003 by Gav Wood                                        *
 *   gav@cs.york.ac.uk                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 ***************************************************************************/

#define __GEDDEI_BUILD

#include <cmath>
using namespace std;

#include "qfactoryexporter.h"

#include "signaltype.h"
#include "bufferdata.h"
#include "subprocessor.h"
#include "buffer.h"
using namespace Geddei;

#include "spectrum.h"
using namespace SignalTypes;

#define BINS 24
#define CEPSTRA BINS

class MFCC : public SubProcessor
{
	uint theMarkers[BINS + 2], bandWidth;

	virtual void processChunk(const BufferDatas &in, BufferDatas &out) const;
	virtual const bool verifyAndSpecifyTypes(const SignalTypeRefs &inTypes, SignalTypeRefs &outTypes);
	virtual void initFromProperties(const Properties &properties);

public:
	MFCC() : SubProcessor("MFCC") {}
};

void MFCC::processChunk(const BufferDatas &ins, BufferDatas &outs) const
{
	static float theBins[BINS];
	float t;
	// mel scaling with BINS triangular-sum-log band bins
	for(uint i = 1; i < BINS - 1; i++)
	{	t = 0;
		for(uint j = theMarkers[i - 1]; j < theMarkers[i]; j++)
			t += abs(ins[0][j]) * float(j - theMarkers[i - 1]) / float(theMarkers[i] - theMarkers[i - 1]);
		for(uint j = theMarkers[i]; j < theMarkers[i + 1]; j++)
			t += abs(ins[0][j]) * float(theMarkers[i + 1] - j) / float(theMarkers[i + 1] - theMarkers[i]);
		theBins[i - 1] = log(t);
	}
	// ceptrum transformation (dct) TODO: use FFTW
	for(uint i = 0; i < CEPSTRA; i++)
	{	t = 0;
		for(uint j = 0; j < BINS; j++)
			t += theBins[j] * cos(M_PI / BINS * (i + 1.0) * (j + 0.5));
		outs[0][i] = t / float(BINS);
	}
}

const float toMel(const float hertz) { return 1127.01048 * log(1.0 + hertz / 700.0); }
const float toHertz(const float mel) { return 700.0 * (exp(mel / 1127.01048 - 1.0)); }

const bool MFCC::verifyAndSpecifyTypes(const SignalTypeRefs &inTypes, SignalTypeRefs &outTypes)
{
	// TODO: should only take a "stepped spectrum" in.
	if(!inTypes[0].isA<Spectrum>()) return false;
	const Spectrum &in = inTypes[0].asA<Spectrum>();
	outTypes[0] = Spectrum(CEPSTRA, in.frequency(), 1);

	float maxMel = toMel(in.nyquist());
	for(uint i = 0; i < BINS + 2; i++)
		theMarkers[i] = int(toHertz(float(i) * maxMel / float(BINS + 2))) / int(in.step());
	return true;
}

void MFCC::initFromProperties(const Properties &)
{
	setupIO(1, 1, 1, 1, 1);
}

EXPORT_CLASS(MFCC, 0,2,0, SubProcessor);
