/***************************************************************************
 *   Copyright (C) 2005 by SUZUKI Tasuku                                   *
 *   tasuku@linux-life.net                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef _KKDDOC_H_
#define _KKDDOC_H_

#include "../kkglobal.h"

#include <qdom.h>
#include <qimage.h>

#include <klistview.h>
#include <kaction.h>
#include <kcommand.h>

#include "kkdtextitem.h"
#include "kkddatabase.h"

namespace KKDesigner
{
	class KKDPaper;
	class KKDCmdBase;

	/**
	 * @english
	 * @short Document class
	 * Include paper(s), item(s).
	 * @endenglish
	 *
	 * @japanese
	 * @short ドキュメントクラス
	 * 用紙やアイテムを含む
	 * @endjapanese
	 */
	class KKDDoc : public QObject, public QPtrList<KKDPaper>
	{
		Q_OBJECT

		Q_PROPERTY( bool locked READ locked WRITE setLocked )

		Q_PROPERTY( QString text READ text WRITE setText )
		Q_PROPERTY( QString fontFamily READ fontFamily WRITE setFontFamily )
		Q_PROPERTY( int fontSize READ fontSize WRITE setFontSize )
		Q_PROPERTY( bool fontBold READ fontBold WRITE setFontBold )
		Q_PROPERTY( bool fontItalic READ fontItalic WRITE setFontItalic )
		Q_PROPERTY( bool fontUnderline READ fontUnderline WRITE setFontUnderline )
		Q_PROPERTY( bool fontStrikeOut READ fontStrikeOut WRITE setFontStrikeOut )
		Q_PROPERTY( QColor color READ color WRITE setColor )
		Q_PROPERTY( Orientation orientation READ orientation WRITE setOrientation )
		Q_PROPERTY( AlignmentFlags align READ align WRITE setAlign )
		Q_PROPERTY( bool imagePrint READ imagePrint WRITE setImagePrint )
		Q_PROPERTY( DataType dataType READ dataType WRITE setDataType )

	public:
		KKDDoc( QWidget* parent = 0, const char* name = 0 );
		KKDDoc( KCommandHistory* cmdHistory, QWidget* parent = 0, const char* name = 0 );
		virtual ~KKDDoc();

		bool readOnly(){ return m_cmdHistory == NULL; }

		bool open( const QString& filename );
		bool save( const QString& filename );
		void close();
		void print( QPainter& p, KKAddressBook::KKAContactData* receiver = 0, const KKAddressBook::KKAContactData* sender = 0 );
		const int currentIndex(){ return m_currentIndex; }
		KKDPaper* current(){ return at( m_currentIndex ); }

		void insertByCommand( KKDPaper* paper, KKDItemBase* item );
		void removeByCommand( KKDPaper* paper, KKDItemBase* item );

		const QPtrList<KKDItemBase> selectedItems() const { return m_selectedItems; }
		bool cutable() const;
		bool copyable() const { return cutable(); }
		bool pastable() const { return ( m_clipboard.count() > 0 ); }
		
		//BEGIN Properties
		bool locked() const;

		QString text() const;
		QString fontFamily() const;
		int fontSize() const;
		bool fontBold() const;
		bool fontItalic() const;
		bool fontUnderline() const;
		bool fontStrikeOut() const;
		AlignmentFlags align() const;
		QColor color() const;
		Orientation orientation() const;
		bool imagePrint();
		KKDDataBase::DataType dataType() const;

		//END Properties

	public slots:
		void setCurrentIndex( int i );
		void setCurrent( KKDPaper* p );

		//BEGIN Actions
		void cut();
		void copy();
		void paste();
		void insert();
		void remove();
		void raise();
		void lower();

		void selectRect( const QRect& rect, bool clearAll = true );
		void selectAll();
		void deselect();
		//END Actions

		//BEGIN Properties
		void setSelected( QPtrList<KKDItemBase> items, bool selected, bool clearAll = true );
		void setSelected( KKDItemBase* item, bool selected, bool clearAll = true, bool emitSignal = true );

		// Location
		void setLocation( const QPoint& l ){ setLocation( l.x(), l.y() ); }
		void setLocation( int x, int y );
		void moveStart();
		void moving();
		void moveEnd();

		// Size
		void setSize( const QSize& s ){ setSize( s.width(), s.height() ); }
		void setSize( int width, int height );
		void resizeStart();
		void resizing();
		void resizeEnd();

		// Base
		void setLocked( bool l );

		// Text
		void setText( const QString& t );
		void setAlign( AlignmentFlags a );
		void setColor( const QColor& c );
		void setFontFamily( const QString& f );
		void setFontSize( int s );
		void setFontBold( bool b );
		void setFontItalic( bool i );
		void setFontUnderline( bool u );
		void setFontStrikeOut( bool s );
		void setOrientation( Orientation o );

		// Data
		void setDataType( KKDDataBase::DataType t );
		
		// Image
		void eraseImage();
		void setImage( const QImage& i );
		void setImage( const QString& p );

		// Paper
		void setImagePrint( bool p );

		//END Properties

		bool imagePrintEnabled();
		bool imageEraseEnabled();
		
	signals:
		void documentChanged();
		void currentIndexChanged();
		void currentChanged( KKDPaper* );

		void inserted();
		void inserted( KKDItemBase*, KKDPaper* );
		void removed();
		void removed( KKDItemBase*, KKDPaper* );

		void positionChanged();

		void pasteEnabledChanged();

		//BEGIN Properties
		void selectedChanged();

		void locationChanged();
		void sizeChanged();
		void lockedChanged();

		void textChanged();
		void alignChanged();
		void colorChanged();
		void fontFamilyChanged();
		void fontSizeChanged();
		void fontBoldChanged();
		void fontItalicChanged();
		void fontUnderlineChanged();
		void fontStrikeOutChanged();
		void orientationChanged();

		void dataTypeChanged();
		
		void imageChanged();
		void imagePrintChanged();
		//END   Properties

	private:
		void clearSelection();

		QPtrList<KKDItemBase> m_clipboard;
		QPtrList<KKDItemBase> m_selectedItems;
		int m_currentIndex;
		KCommandHistory* m_cmdHistory;
		KKDCmdBase* m_cmd;

		void terminate();
	};
}

#endif /* _KKDDOC_H_ */
