# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import gobject
import gtk
import threading

import stock
import utils
from config import config
from xdg_support import get_xdg_pixmap_file

from source import Source, SourceItem
from library import ListenDB

from helper import Dispatcher
from widget.misc import ScrolledWindow, SimpleMenu
from widget.song_view import SortableSongView
from widget.song_menu import SongMenu, SongMenuManager
from widget.dialog import WindowBase

from plugins.webradio import WebRadioManager


class WindowNewRadio(WindowBase):
    def __init__(self, song=None):
        if song:
            title = _("Edit web radio")            
        else:
            title = _("New web radio")
        WindowBase.__init__(self, title, get_xdg_pixmap_file("playlist_radio_big.png"))
        self.set_property("skip-taskbar-hint", True)
        self.set_property('has-separator', True)
        self.label_heading.set_markup("<span size=\"larger\"><b>" + _("Web radio name and url :") + "</b></span>")

        self.entry_name = gtk.Entry()
        self.entry_name.set_size_request(300, -1)
        self.entry_name.set_activates_default(True)
        self.entry_uri = gtk.Entry()
        self.entry_uri.set_size_request(300, -1)
        self.entry_uri.set_activates_default(True)
        if song:
            self.entry_uri.set_text(song.get("uri"))
            self.entry_name.set_text(song.get("title"))
            
        self.add_widget(self.entry_name)
        self.add_widget(self.entry_uri)
        self.show_all()

        response = self.run()
        if response == gtk.RESPONSE_ACCEPT and self.entry_uri.get_text().strip() != "":
            uri = self.entry_uri.get_text().strip()
            name = self.entry_name.get_text().strip()
            tags = {}
            tags["title"] = name
            tags["uri"] = uri
            if song:
                ListenDB.set_property(song, tags)
            else:
                ListenDB.get_or_create_song(tags, "webradio")
        self.destroy()

class FavoriteRadioSongView(SortableSongView):
    show_all_columns = True

    def __init__(self):
        kargs = {
                "title":(_("Web Radio"), gobject.TYPE_STRING)
                }
        super(FavoriteRadioSongView, self).__init__("favoriteradio", **kargs)

        def song_menu_action(song_menu_manager, id_menu, songs):
            if id_menu == "webradio_new":
                WindowNewRadio()
            elif id_menu == "webradio_edit":
                WindowNewRadio(songs[0])
    
        SongMenuManager.connect("action", song_menu_action)
        SongMenuManager.register_item("webradio_new", 80, stock.NEW_RADIO, sep_before=True)
        SongMenuManager.register_item("webradio_edit", 90, stock.EDIT_RADIO)
        for item_menu in [ "play", "enqueue", "delete", "webradio_edit", "webradio_new"]:
            SongMenuManager.allow_item_for_type(item_menu, "webradio")

        self.menu = SongMenu()
        
        self.empty_menu = SimpleMenu()
        def new_radio(*args, **kwargs): WindowNewRadio()
        self.empty_menu.insert_item(None, new_radio, stock.NEW_RADIO)
        self.empty_menu.show_all()
        
        self.set_menu(self.menu, self.empty_menu)
         
        def added(db, type, songs): 
            if type == "webradio": self.get_model().append_songs(songs)

        def delete(db, type, songs): 
            if type == "webradio": self.get_model().remove_songs(songs)


        def change(db, songs):
            #FIXME: this function don't reorder the song !
            if type == "webradio" :self.get_model().change_songs(songs)

        ListenDB.connect("added", added)
        ListenDB.connect("removed", delete)
        ListenDB.connect("simple-changed", change)

        if ListenDB.isloaded():
            self.__on_db_loaded(ListenDB)
        else:
            ListenDB.connect("loaded", self.__on_db_loaded)


    def __on_db_loaded(self, db):
        self.get_model().fill(list(ListenDB.get_songs("webradio")))

    def destroy(self):
        SongMenuManager.unregister_item("webradio_new")
        SongMenuManager.unregister_item("webradio_edit")
        super(FavoriteRadioSongView,self).destroy()

       
class RadioSongView(SortableSongView):
    show_all_columns = True
    def __init__(self, plugin_name):
        kargs = {
                "title":(_("Title"), gobject.TYPE_STRING),
                "radio_person":(_("Person"), gobject.TYPE_STRING),
                "#bitrate":(_("Bitrate"), gobject.TYPE_STRING)
                }
        
        super(RadioSongView, self).__init__("radio", **kargs)
        self.__plugin_name = plugin_name

        self.menu = SongMenu()
        self.set_menu(self.menu)  

        def song_menu_action(song_menu_manager, id_menu, songs):
            if id_menu == "webradio_addfavorite_" + self.__plugin_name:
                [ListenDB.get_or_create_song(song, "webradio") for song in songs ]

        SongMenuManager.connect("action", song_menu_action)
        SongMenuManager.register_item("webradio_addfavorite_" + plugin_name, 70, stock.ADD_FAVORITE, sep_before=True)
        for item_menu in [ "play", "enqueue", "webradio_addfavorite_" + plugin_name]:
            SongMenuManager.allow_item_for_type(item_menu, "volatile_webradio")

    def add_favorite(self, id_menu, songs):
        #model, rows = self.get_selection().get_selected_rows()
        #[ListenDB.get_or_create_song(model[row][0],"webradio") for row in rows ]
        if id_menu == "webradio_addfavorite_" + self.__plugin_name:
            [ListenDB.get_or_create_song(song, "webradio") for song in songs ]

    def destroy(self):
        SongMenuManager.unregister_item("webradio_addfavorite_" + self.__plugin_name)
        super(RadioSongView, self).destroy()
        
        
class RadioBrowser(gtk.HPaned):
    def __init__(self, webradio_plugin):
        super(RadioBrowser, self).__init__()
        
        self.webradio_plugin = webradio_plugin
        
        model_categorie = gtk.ListStore(str, str)
        
        self.tree_categorie = gtk.TreeView(model_categorie)
        self.tree_categorie.insert_column_with_attributes(-1, _("Categorie"), gtk.CellRendererText(), text=1)
        
        
        self.tree_radio = RadioSongView(webradio_plugin.PLUGIN_NAME)
        
        self.tree_categorie.get_selection().connect("changed", self.on_cat_activated)
        #self.tree_radio.connect("row-activated",self.on_radio_activated)
        
        self.pack1(ScrolledWindow(self.tree_categorie), False, True)
        self.pack2(ScrolledWindow(self.tree_radio), False, True)
        
        self.last_update = 0
        
        self.thread = None
        self.set_position(int(config.get("iradio", webradio_plugin.conf_prefix + "_pos")))
        self.connect("notify::position", lambda pane, property_spec: config.set("iradio", webradio_plugin.conf_prefix + "_pos", "%d" % pane.get_position()))
    
    def fill(self):
        if not self.thread:
            self.thread = threading.Thread(target=self.__thread_fill)
            self.thread.setDaemon(True)
            self.thread.start()
        
    def __thread_fill(self):
        if self.webradio_plugin:
            cat = self.webradio_plugin.get_categorie()
            gobject.idle_add(self.__thread_fill_cb, cat)
        else:
            pass # Widget have been deleted or reloaded
        
    def __thread_fill_cb(self, cat):
        self.thread = None
        model = self.tree_categorie.get_model()
        if model:
            model.clear()    
            for key , name in cat:
                model.append((utils.xmlescape(key), utils.xmlescape(name)))
        else:
            # Browser has been destroy but not unref yet
            pass
        #self.tree_categorie.get_selection().select_path(0)
        
    def on_cat_activated(self, treeselection):
        model, iter = treeselection.get_selected()
        if iter:
            key = model[iter][0]
            self.last_key = key
            #self.get_toplevel().window.set_cursor(gtk.gdk.Cursor(gtk.gdk.WATCH))
            self.thread = threading.Thread(target=self.__thread_get_songs, args=(key,))
            self.thread.setDaemon(True)
            self.thread.start()

    def __thread_get_songs(self, key):
        songs = self.webradio_plugin.get_songs_by_categorie(key)
        gobject.idle_add(self.fill_song_view, key, songs)
        
    def fill_song_view(self, key, tags):
        if self.last_key == key:
            ListenDB.full_erase("volatile_webradio")
            #self.get_toplevel().window.set_cursor(None)    
            model = self.tree_radio.get_model()
            model.clear()
            for tag in tags:
                ListenDB.get_or_create_song(tag, "volatile_webradio")
            model.fill(list(ListenDB.get_songs("volatile_webradio")))
        

class RadioWidget(gtk.VBox):
    __browsers = []
    def __init__(self):
        super(RadioWidget, self).__init__(False, 6)   
        
        self.favorite_song_view = FavoriteRadioSongView()
        
        self.__btn_box = gtk.HBox(False, 6)
        self.notebook = gtk.Notebook()

        self.notebook.set_property("show-tabs", False)
        self.notebook.set_property("show-border", False)        
        
        self.pack_start(self.__btn_box, False, False)
        self.pack_start(self.notebook, True, True)
        self.refresh()
        self.notebook.connect("switch-page", self.on_select_page)
        Dispatcher.connect("reload-plugins", self.refresh)
        
    def refresh(self, *args):
        for child in self.__btn_box.get_children():
            self.__btn_box.remove(child)
            child.destroy()
            del child

        self.__browsers = []

        for child in self.notebook.get_children():
            self.notebook.remove(child)
            if isinstance(child, ScrolledWindow):
                child.remove(self.favorite_song_view)
            if isinstance(child, RadioBrowser):
                child.tree_radio.destroy()
                del child.tree_radio
                del child.webradio_plugin
            child.destroy()
            del child

        self.__browsers = [
                     (_("Favorite"), ScrolledWindow(self.favorite_song_view)),
                     ]

        wrm = WebRadioManager()
        for plug in wrm.list(True):
            self.__browsers.append((plug.PLUGIN_NAME, (RadioBrowser(plug()))))

        btn = None
        page_num = 0
        for browser_name, browser in self.__browsers:
            btn = gtk.RadioButton(btn, browser_name)
            btn.set_mode(False)
            btn.connect("clicked", self.on_click)
            btn.page_num = page_num
            self.__btn_box.pack_start(btn, False, False)
            self.notebook.append_page(browser, gtk.Label(browser_name))
            page_num += 1

        self.__btn_box.show_all()
        self.notebook.show_all()
            
    def on_click(self, b):
        self.notebook.set_current_page(b.page_num)
        
    def on_select_page(self, notebook, page, page_num):
        if page_num > 0: 
            self.notebook.get_nth_page(page_num).fill()
        
    def populate(self):
        self.favorite_song_view.get_model().fill(self.favorite_song_view.pl.get_songs())
        

class RadioSourceItem(SourceItem):
    widget_klass = RadioWidget
    label = _("Webradio")
    config_code = "locallibrary"
    stock = stock.SRC_IRADIO

class RadioSource(Source):
    PLUGIN_NAME = "Web Radio support"
    PLUGIN_DESC = _("Allow browse Web Radio")

    load_priority = 0
    display_index = 100

    def __init__(self):
        Source.__init__(self)
        self.items = [RadioSourceItem()]
                                                       

