/*  Window monitor / application launcher
 *  Copyright (C) 2005-2006 UCHINO Satoshi.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "windowmonitor.h"
#include "launcher.h"
#include "conffile.h"
#include <iostream>
#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>

Launcher::Launcher(ClientMgr& _clientMgr, IconMgr& _iconMgr, TmpFileMgr& _tmpFileMgr)
  : clientMgr(_clientMgr), iconMgr(_iconMgr), tmpFileMgr(_tmpFileMgr),
    desktopEntrySection("Desktop Entry")
{
  const string keylist[] = {	// these keys will be sent to clients
    "Type",
    "Version",
    "Encoding",
    "Name",
    "GenericName",
    "Comment",
    "SortOrder",
    "Categories",
    "OnlyShowIn",
    "NotShowIn",
  };
  const string intkeylist[] = {	// only for internal use
    "Icon",
    "Exec",
    "NoDisplay",
  };

  /* create key set */
  for (int i = 0; i < sizeof(keylist)/sizeof(keylist[0]); i++)
    keySet.insert(keylist[i]);
  intKeySet = keySet;
  for (int i = 0; i < sizeof(intkeylist)/sizeof(intkeylist[0]); i++)
    intKeySet.insert(intkeylist[i]);
}

Launcher::~Launcher()
{
}

bool Launcher::launch(unsigned long id)
{
  Launchers::size_type pos = getPos(id);
  if (pos >= launchers.size() || launchers[pos].commandline.empty()) {
    cerr << __FUNCTION__ << ": could not find id " << id << endl;
    return false;
  }

  cout << __FUNCTION__ << ": executing " << launchers[pos].commandline << endl;

  // execute
  if (!g_spawn_command_line_async(launchers[pos].commandline.c_str(), NULL)) {
    cerr << __FUNCTION__ << ": failed to exec " << launchers[pos].commandline << endl;
    return false;
  }

  return true;
}

/* scan all files in the current directory */
bool Launcher::sendMenuItems(const string& prefix, 
			     unsigned long clientPtr,
			     const string& localePref,
			     const ClientMgr::IconPref* iconPref)
{
  DIR *dir;
  dir = opendir(".");
  if (dir == NULL) {
    cerr << __FUNCTION__ << ": failed to opendir" << endl;
    return false;
  }
  struct dirent *dp;
  while ((dp = readdir(dir)) != NULL) {
    string filename = dp->d_name;
    struct stat file_stat;
    if (stat(filename.c_str(), &file_stat) != 0) {
      cerr << __FUNCTION__ << ": failed to stat '" << filename << "'" << endl;
      continue;
    }
    if (S_ISDIR(file_stat.st_mode)) {
      if (prefix.length() == 0 &&	/* top level directory? */
	  strcmp(filename.c_str(), ".") != 0 &&
	  strcmp(filename.c_str(), "..") != 0 &&
	  chdir(filename.c_str()) == 0) {
	string newprefix = filename + "-";
	sendMenuItems(newprefix, clientPtr, localePref, iconPref);
	chdir("..");
      }
    } else if (S_ISREG(file_stat.st_mode)) {
      string::size_type pos;
      pos = filename.find('.');
      if (pos != string::npos &&
	  filename.substr(pos) == ".desktop") {
	sendMenuItem(prefix, filename, clientPtr, localePref, iconPref);
      } else {
	cerr << __FUNCTION__ << ": skip file '" << filename << "'" << endl;
	continue;
      }
    }
  }
  closedir(dir);
  return true;
}

bool Launcher::sendAllMenuItems(unsigned long clientPtr)
{
  const string* localePref = clientMgr.getLocalePref(clientPtr);
  if (localePref == NULL)
    return false;

  const ClientMgr::IconPref* iconPref = clientMgr.getIconPref(clientPtr);

  /* for each directories in data_dirs ... */
  
  vector<string>* dirList = ConfigFile::getXdgDataDirs();
  for (int i = 0; i < dirList->size(); i++) {
    if (chdir((*dirList)[i].c_str()) != 0 ||
	chdir("applications") != 0) {
      cerr << __FUNCTION__ << ": failed to chdir to '" << (*dirList)[i] << "applications'" << endl;
      continue;
    }
    sendMenuItems("", clientPtr, *localePref, iconPref);
  }
  delete dirList;
  return true;
}

bool Launcher::sendMenuIcons(unsigned long clientPtr)
{
  const ClientMgr::IconPref* iconPref = clientMgr.getIconPref(clientPtr);
  if (iconPref)
    for (int pos = launchers.size(); --pos >= 0; )
      sendIcon(getId(pos), launchers[pos].icon, clientPtr, *iconPref);
}

/*
 * send content of a desktop entry
 */
bool Launcher::sendMenuItem(const string &prefix, const string& filename,
			    unsigned long clientPtr,
			    const string& localePref,
			    const ClientMgr::IconPref* iconPref)
{
  string fileid = prefix + filename;
  string tmpfilekey = fileid + "-" + localePref;
  string realpath;
  int tmpfile_fd;	// fd of tmpfile

  // first try to create tmpfile
  tmpfile_fd = tmpFileMgr.createTmpFile(tmpfilekey, realpath);
  if (tmpfile_fd == -1) {
    // check if it already exists.
    tmpfile_fd = tmpFileMgr.lookupTmpFile(tmpfilekey, realpath);
    if (tmpfile_fd != -1) { // found
      /* lookup item */
      Item* item;
      unsigned int id = 0;
      for (int pos = launchers.size(); --pos >= 0; ) {
	if (launchers[pos].desktopfileid == fileid) {
	  item = &launchers[pos];
	  id = getId(pos);
	  break;
	}
      }
      if (id == 0) {	// item not found
	close(tmpfile_fd);
	return false;
      }

      /* send item */
      SendServerMenuItemPath(id, clientPtr, realpath.length(), realpath.c_str());
      close(tmpfile_fd);
      if (iconPref)
	sendIcon(id, item->icon, clientPtr, *iconPref);
      return true;
    }
    return false;
  }

  /* read a desktop entry */
  ConfigFile::SpItemMap itemMap;
  //cout << "sendMenuItem: section = '" << desktopEntrySection << "'" << endl;
  itemMap = ConfigFile::readSection(filename, desktopEntrySection, localePref, intKeySet);
  if (!itemMap) {
    //cout << __FUNCTION__ << " : no item for '" << fileid << "' found" << endl;
    return false;
  }

  if (itemMap->find("Type") == itemMap->end()) {
    //cout << __FUNCTION__ << " : '" << fileid << "' does not have Type item" << endl;
  } else if ((*itemMap)["Type"].body != "Application") {
    //cout << __FUNCTION__ << " : '" << fileid << "' : '" << (*itemMap)["Type"].body << "' is not Application-Type" << endl;
    return false;
  }

  if (itemMap->find("Exec") == itemMap->end()) {
    //cout << __FUNCTION__ << " : '" << fileid << "' does not have Exec item" << endl;
    return false;
  }

  if (itemMap->find("Name") == itemMap->end()) {
    //cout << __FUNCTION__ << " : '" << fileid << "' does not have Name item" << endl;
    return false;
  }

  if (itemMap->find("NoDisplay") != itemMap->end() &&
      (*itemMap)["NoDisplay"].body != "true") {
    cout << __FUNCTION__ << " : '" << fileid << "' has 'NoDisplay' attribute.  Skipping..." << endl;
    return false;
  }

  /* create new Lancher::Item object */
  Item item;
  item.desktopfileid = fileid;
  item.commandline = (*itemMap)["Exec"].body;
  if (itemMap->find("Icon") != itemMap->end())
    item.icon = (*itemMap)["Icon"].body;

  /* expand % */
  for (string::size_type pos = 0;
       (pos = item.commandline.find("%", pos)) != string::npos; pos++) {
    char c = item.commandline.at(pos + 1);
    switch (c) {
    case '%':
      // convert %% to %
      item.commandline = item.commandline.substr(0, pos) + item.commandline.substr(pos + 1);
      break;
    case 'f':
    case 'F':
    case 'u':
    case 'U':
    case 'd':
    case 'D':
    case 'n':
    case 'N':
    case 'i':
    case 'k':
    case 'v':
    case 'm':	// KDE mini-icon (legacy)
      // just ignore
      item.commandline = item.commandline.substr(0, pos) + item.commandline.substr(pos + 2);
      break;
    case 'c':
      // replace %c with "Name"
      item.commandline = item.commandline.substr(0, pos)
	+ (*itemMap)["Name"].body + item.commandline.substr(pos + 2);
      break;
    default:
      cout << __FUNCTION__ << " : '" << fileid << "' uses unsupported format character '" << c << "'." << endl;
      return false;
    }
  }

  /* update launchers */
  unsigned int id = 0;
  for (int pos = launchers.size(); --pos >= 0; ) {
    if (launchers[pos].desktopfileid == item.desktopfileid) {
      launchers[pos] = item;
      id = getId(pos);
      break;
    }
  }
  if (id == 0) {
    launchers.push_back(item);
    id = launchers.size();
  }

  /* compose data */
  FILE *fp = fdopen(tmpfile_fd, "w");
  if (fp == NULL) {
    cerr << __FUNCTION__ << " : failed to open '" << realpath << "'" << endl;
    return false;
  }
  fprintf(fp, "%s\n", fileid.c_str());
  for (set<string>::iterator i = keySet.begin(); i != keySet.end(); i++) {
    ConfigFile::ItemMap::iterator j = itemMap->find(*i);
    if (j != itemMap->end()) {
      if (j->second.hasLang) {
	fprintf(fp, "%s[%s]=%s\n",i->c_str(), j->second.locale.c_str(),
		j->second.body.c_str());
      } else {
	fprintf(fp, "%s=%s\n",i->c_str(), j->second.body.c_str());
      }
    }
  }
  fclose(fp);

  /* send item */
  SendServerMenuItemPath(id, clientPtr, realpath.length(), realpath.c_str());
  if (iconPref)
    sendIcon(id, item.icon, clientPtr, *iconPref);

  return true;
}

bool Launcher::sendIcon(unsigned long id, string& iconname, unsigned long clientPtr, const ClientMgr::IconPref& iconPref)
{
  string filename;
  ClientMgr::IconPref iconInfo;
  //cout << __FUNCTION__ << " : looking up icon '" << iconname << "'" << endl;
  if (iconMgr.lookupIcon(iconname, clientPtr, filename, iconInfo)) {
    //cout << __FUNCTION__ << " : found '" << filename << "'" << endl;
    return SendServerIconPath(id, rfbServerDataMenuIconPath, clientPtr,
			      iconInfo.bpp, iconInfo.size, iconInfo.formatType,
			      filename.length(), filename.c_str());
  }
  cerr << __FUNCTION__ << " : could not find icon '" << iconname << "'" << endl;
  return false;
}
