(* Copyright (C) 2008 Richard W.M. Jones, Red Hat Inc.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*)

open Printf
open Newt_int

exception NewtFailure of string

type color_set = newtColorSet
type colors = newtColors
type flags_sense = newtFlagsSense
type flag =
  | RETURNEXIT
  | HIDDEN
  | SCROLL
  | DISABLED
  | BORDER
  | WRAP
  | NOF12
  | MULTIPLE
  | SELECTED
  | CHECKBOX
  | PASSWORD
  | SHOWCURSOR
type fd_flag =
  | FD_READ
  | FD_WRITE
  | FD_EXCEPT

let parse_flags = List.fold_left (
  fun c flag ->
    c lor match flag with
    | RETURNEXIT -> nEWT_FLAG_RETURNEXIT
    | HIDDEN -> nEWT_FLAG_HIDDEN
    | SCROLL -> nEWT_FLAG_SCROLL
    | DISABLED -> nEWT_FLAG_DISABLED
    | BORDER -> nEWT_FLAG_BORDER
    | WRAP -> nEWT_FLAG_WRAP
    | NOF12 -> nEWT_FLAG_NOF12
    | MULTIPLE -> nEWT_FLAG_MULTIPLE
    | SELECTED -> nEWT_FLAG_SELECTED
    | CHECKBOX -> nEWT_FLAG_CHECKBOX
    | PASSWORD -> nEWT_FLAG_PASSWORD
    | SHOWCURSOR -> nEWT_FLAG_SHOWCURSOR
) 0

let parse_fd_flags = List.fold_left (
  fun c flag ->
    c lor match flag with
    | FD_READ -> nEWT_FD_READ
    | FD_WRITE -> nEWT_FD_WRITE
    | FD_EXCEPT -> nEWT_FD_EXCEPT
) 0

let init () =
  let r = newtInit () in
  if r <> 0 then
    raise (NewtFailure (sprintf "newtInit failed with error %d" r))

let finished () =
  let r = newtFinished () in
  if r <> 0 then
    raise (NewtFailure (sprintf "newtFinished failed with error %d" r))

type ('a, 'b) either = Either of 'a | Or of 'b

let init_and_finish f =
  init ();
  let r =
    try Either (f ())
    with exn -> Or exn in
  ignore (newtFinished ());
  match r with Either r -> r | Or exn -> raise exn

let cls = newtCls
let resize_screen = newtResizeScreen
let wait_for_key = newtWaitForKey
let clear_key_buffer = newtClearKeyBuffer
let delay = newtDelay
let open_window left top width height title =
  let r = newtOpenWindow left top width height title in
  if r <> 0 then
    raise (NewtFailure (sprintf "newtOpenWindow failed with error %d" r))

let centered_window width height title =
  let r = newtCenteredWindow width height title in
  if r <> 0 then
    raise (NewtFailure (sprintf "newtCenteredWindow failed with error %d" r))

let pop_window = newtPopWindow
let pop_window_no_refresh = newtPopWindowNoRefresh
let set_colors = newtSetColors
let set_color = newtSetColor
let refresh = newtRefresh
let suspend = newtSuspend
(*void newtSetSuspendCallback(newtSuspendCallback cb, void * data);
void newtSetHelpCallback(newtCallback cb);*)
let resume = newtResume
let push_help_line = newtPushHelpLine
let redraw_help_line = newtRedrawHelpLine
let pop_help_line = newtPopHelpLine
let draw_root_text = newtDrawRootText
let bell = newtBell
let cursor_on = newtCursorOn
let cursor_off = newtCursorOff
let get_screen_size = newtGetScreenSize

type _component = {
  co : newtComponent;			(* The component. *)
  mutable in_form : bool;	        (* If added to a form, this is set. *)
}
type 'a component = _component

type component_ptr = newtComponent

let component_equals ptr { co = co } = ptr = co

let debug = false

(* We need to call newtComponentDestroy until the component has been
 * added to a form at which point the form manages it and it will be
 * destroyed by the form's own finalizer.
 *)
let component_set_finalizer what co =
  let f { co = co; in_form = in_form } =
    if not in_form then (
      if debug then
	eprintf "newtComponentDestroy %Ld\n%!" (newtComponentAddress co);
      newtComponentDestroy co
    )
  in
  if debug then eprintf "create %s %Ld\n%!" what (newtComponentAddress co);
  let co = { co = co; in_form = false } in
  Gc.finalise f co;
  co

(* The form finalizer must call newtFormDestroy instead, mainly for
 * backwards compatibility since newtComponentDestroy will do the same
 * thing in newer versions of newt.
 *)
let form_set_finalizer form =
  let f { co = co; in_form = in_form } =
    (* NB. forms can be recursively inside other forms. *)
    if not in_form then (
      if debug then
	eprintf "newtFormDestroy %Ld\n%!" (newtComponentAddress co);
      newtFormDestroy co
    )
  in
  if debug then eprintf "create form %Ld\n%!" (newtComponentAddress form);
  let form = { co = form; in_form = false } in
  Gc.finalise f form;
  form

let compact_button left top text =
  component_set_finalizer "compact button" (newtCompactButton left top text)
  
let button left top text =
  component_set_finalizer "button" (newtButton left top text)

let checkbox left top text default seq =
  component_set_finalizer "checkbox"
    (newtCheckbox left top text default seq)

let checkbox_get_value { co = co } = newtCheckboxGetValue co

let checkbox_set_value { co = co } = newtCheckboxSetValue co

let checkbox_set_flags { co = co } flags sense =
  let flags = parse_flags flags in
  newtCheckboxSetFlags co flags sense
      
let radio_button left top text default prev =
  let prev =
    match prev with
    | None -> None
    | Some { co = co } -> Some co in
  component_set_finalizer "radio button"
    (newtRadiobutton left top text default prev)

let radio_get_current { co = co } =
  newtRadioGetCurrent co

let label left top text =
  component_set_finalizer "label" (newtLabel left top text)

let label_set_text { co = co } = newtLabelSetText co

let vertical_scrollbar left top height normalcol thumbcol =
  component_set_finalizer "vertical scrollbar"
    (newtVerticalScrollbar left top height normalcol thumbcol)

let scrollbar_set { co = co } = newtScrollbarSet co

(* Listboxes. *)

type 'a listbox = {
  lbco : [`Component|`Listbox] component;
  mutable data : (int * 'a) list;
  mutable i : int;
}

let get_datum_by_index lb i =
  try List.assoc i lb.data
  with Not_found ->
    assert false (* this is an internal error because the underlying
		    newt listbox has given us back an integer which we
		    never added *)

let get_index_by_datum lb d =
  List.assoc d (List.map (fun (a,b) -> b,a) lb.data)

let add_datum lb d =
  try
    get_index_by_datum lb d
  with
    Not_found ->
      let i = lb.i in
      lb.i <- lb.i+1;
      lb.data <- (i, d) :: lb.data;
      i

let component_of_listbox lb = lb.lbco

let listbox left top height flags =
  let flags = parse_flags flags in
  let lb =
    component_set_finalizer "listbox" (newtListbox left top height flags) in
  { lbco = lb; data = []; i = 1 }

let listbox_get_current lb =
  let i = newtListboxGetCurrent lb.lbco.co in
  match i with
  | None -> None
  | Some i -> Some (get_datum_by_index lb i)

let listbox_set_current lb = newtListboxSetCurrent lb.lbco.co

let listbox_set_current_by_key lb d =
  let i = get_index_by_datum lb d in
  newtListboxSetCurrentByKey lb.lbco.co i

let listbox_set_entry lb = newtListboxSetEntry lb.lbco.co

let listbox_set_width lb = newtListboxSetWidth lb.lbco.co

let listbox_set_data lb row d =
  let i = add_datum lb d in
  newtListboxSetData lb.lbco.co row i

let listbox_append_entry lb text d =
  let i = add_datum lb d in
  let r = newtListboxAppendEntry lb.lbco.co text i in
  if r <> 0 then
    raise (NewtFailure
	     (sprintf "newtListboxAppendEntry failed with error %d" r))

let listbox_insert_entry lb text d after =
  let i = add_datum lb d in
  let after =
    match after with
    | None -> None
    | Some i -> Some (get_index_by_datum lb i) in
  let r = newtListboxInsertEntry lb.lbco.co text i after in
  if r <> 0 then
    raise (NewtFailure
	     (sprintf "newtListboxInsertEntry failed with error %d" r))

let listbox_delete_entry lb d =
  let i = get_index_by_datum lb d in
  let r = newtListboxDeleteEntry lb.lbco.co i in
  if r <> 0 then
    raise (NewtFailure
	     (sprintf "newtListboxDeleteEntry failed with error %d" r))

let listbox_clear lb = newtListboxClear lb.lbco.co

let listbox_get_entry lb row =
  let text, i = newtListboxGetEntry lb.lbco.co row in
  let d = get_datum_by_index lb i in
  text, d

let listbox_get_selection lb =
  let is = newtListboxGetSelection lb.lbco.co in
  Array.map (get_datum_by_index lb) is

let listbox_clear_selection lb = newtListboxClearSelection lb.lbco.co

let listbox_select_item lb d flag =
  let i = get_index_by_datum lb d in
  newtListboxSelectItem lb.lbco.co i flag

let listbox_item_count lb = newtListboxItemCount lb.lbco.co

(*
newtComponent newtCheckboxTree(int left, int top, int height, int flags);
newtComponent newtCheckboxTreeMulti(int left, int top, int height, [string,in]char *seq, int flags);
//void ** newtCheckboxTreeGetSelection(newtComponent co, int *numitems);
//void * newtCheckboxTreeGetCurrent(newtComponent co);
//void newtCheckboxTreeSetCurrent(newtComponent co, void * item);
//void ** newtCheckboxTreeGetMultiSelection(newtComponent co, int *numitems, char seqnum);
/* last item is NEWT_ARG_LAST for all of these */
//int newtCheckboxTreeAddItem(newtComponent co, 
//			    [string,in]char * text, void * data,
//			    int flags, int index, ...);
//int newtCheckboxTreeAddArray(newtComponent co, 
//			     [string,in]char * text, void * data,
//			     int flags, int * indexes);
//int * newtCheckboxTreeFindItem(newtComponent co, void * data);
//void newtCheckboxTreeSetEntry(newtComponent co, void * data,
//			      [string,in]char * text);
void newtCheckboxTreeSetWidth(newtComponent co, int width);
//char newtCheckboxTreeGetEntryValue(newtComponent co, void * data);
//void newtCheckboxTreeSetEntryValue(newtComponent co, void * data,
//				   char value);
*)
 
let textbox_reflowed left top text width flex_down flex_up flags =
  let flags = parse_flags flags in
  component_set_finalizer "textbox_reflowed"
    (newtTextboxReflowed left top text width flex_down flex_up flags)

let textbox left top width height flags =
  let flags = parse_flags flags in
  component_set_finalizer "textbox" (newtTextbox left top width height flags)

let textbox_set_text { co = co } = newtTextboxSetText co

let textbox_set_height { co = co } = newtTextboxSetHeight co

let textbox_get_num_lines { co = co } = newtTextboxGetNumLines co

let reflow_text = newtReflowText

let form vb help flags =
  let vb =
    match vb with
    | None -> None
    | Some { co = co } -> Some co in
  let flags = parse_flags flags in
  form_set_finalizer (newtForm vb help flags)

let form_set_timer { co = co } = newtFormSetTimer co

let form_watch_fd { co = co } fd flags =
  let flags = parse_fd_flags flags in
  newtFormWatchFd co (Obj.magic fd) flags

let form_set_size { co = co } = newtFormSetSize co

let form_get_current { co = co } = newtFormGetCurrent co

let form_set_background { co = co } = newtFormSetBackground co

let form_set_current { co = form } { co = co } = newtFormSetCurrent form co

let form_add_component form co =
  if co.in_form then failwith "component can only be added to a single form";
  newtFormAddComponent form.co co.co;
  co.in_form <- true

let form_add_components form components =
  List.iter (fun co -> form_add_component form co) components

let form_set_height { co = co } = newtFormSetHeight co

let form_set_width { co = co } = newtFormSetWidth co

let run_form { co = co } = newtRunForm co

let form_run { co = co } = newtFormRun co

let form_add_hot_key { co = co } = newtFormAddHotKey co

(*
//typedef int ( *newtEntryFilter )(newtComponent entry, void * data, int ch,
//			       int cursor);
*)

let entry left top initial width flags =
  let flags = parse_flags flags in
  let initial = match initial with None -> "" | Some s -> s in
  component_set_finalizer "entry" (newtEntry left top initial width None flags)

let entry_set { co = co } = newtEntrySet co

(*
//void newtEntrySetFilter(newtComponent co, newtEntryFilter filter, void * data);
*)

let entry_get_value { co = co } = newtEntryGetValue co

let entry_set_flags { co = co } flags flags_sense =
  let flags = parse_flags flags in
  newtEntrySetFlags co flags flags_sense

let scale left top width full_value =
  component_set_finalizer "scale" (newtScale left top width full_value)

let scale_set { co = co } = newtScaleSet co

let component_add_callback { co = co } = newtComponentAddCallback co
let component_takes_focus { co = co } = newtComponentTakesFocus co

(*
#define NEWT_KEY_TAB			'\t'
#define NEWT_KEY_ENTER			'\r'
#define NEWT_KEY_SUSPEND		'\032'			/* ctrl - z*/
#define NEWT_KEY_ESCAPE			''
#define NEWT_KEY_RETURN			NEWT_KEY_ENTER

#define NEWT_KEY_EXTRA_BASE		0x8000
#define NEWT_KEY_UP			NEWT_KEY_EXTRA_BASE + 1
#define NEWT_KEY_DOWN			NEWT_KEY_EXTRA_BASE + 2
#define NEWT_KEY_LEFT			NEWT_KEY_EXTRA_BASE + 4
#define NEWT_KEY_RIGHT			NEWT_KEY_EXTRA_BASE + 5
#define NEWT_KEY_BKSPC			NEWT_KEY_EXTRA_BASE + 6
#define NEWT_KEY_DELETE			NEWT_KEY_EXTRA_BASE + 7
#define NEWT_KEY_HOME			NEWT_KEY_EXTRA_BASE + 8
#define NEWT_KEY_END			NEWT_KEY_EXTRA_BASE + 9
#define NEWT_KEY_UNTAB			NEWT_KEY_EXTRA_BASE + 10
#define NEWT_KEY_PGUP			NEWT_KEY_EXTRA_BASE + 11
#define NEWT_KEY_PGDN			NEWT_KEY_EXTRA_BASE + 12
#define NEWT_KEY_INSERT			NEWT_KEY_EXTRA_BASE + 13

#define NEWT_KEY_F1			NEWT_KEY_EXTRA_BASE + 101
#define NEWT_KEY_F2			NEWT_KEY_EXTRA_BASE + 102
#define NEWT_KEY_F3			NEWT_KEY_EXTRA_BASE + 103
#define NEWT_KEY_F4			NEWT_KEY_EXTRA_BASE + 104
#define NEWT_KEY_F5			NEWT_KEY_EXTRA_BASE + 105
#define NEWT_KEY_F6			NEWT_KEY_EXTRA_BASE + 106
#define NEWT_KEY_F7			NEWT_KEY_EXTRA_BASE + 107
#define NEWT_KEY_F8			NEWT_KEY_EXTRA_BASE + 108
#define NEWT_KEY_F9			NEWT_KEY_EXTRA_BASE + 109
#define NEWT_KEY_F10			NEWT_KEY_EXTRA_BASE + 110
#define NEWT_KEY_F11			NEWT_KEY_EXTRA_BASE + 111
#define NEWT_KEY_F12			NEWT_KEY_EXTRA_BASE + 112

#define NEWT_KEY_RESIZE			NEWT_KEY_EXTRA_BASE + 113

#define NEWT_ANCHOR_LEFT		(1 << 0)
#define NEWT_ANCHOR_RIGHT		(1 << 1)
#define NEWT_ANCHOR_TOP			(1 << 2)
#define NEWT_ANCHOR_BOTTOM		(1 << 3)

#define NEWT_GRID_FLAG_GROWX		(1 << 0)
#define NEWT_GRID_FLAG_GROWY		(1 << 1)

// XXX also need compare() and hash()
typedef [abstract] void * newtGrid;
enum newtGridElement { NEWT_GRID_EMPTY = 0,
		       NEWT_GRID_COMPONENT, NEWT_GRID_SUBGRID };

newtGrid newtCreateGrid(int cols, int rows);
//newtGrid newtGridVStacked(enum newtGridElement type, void * what, ...);
//newtGrid newtGridVCloseStacked(enum newtGridElement type, void * what, ...);
//newtGrid newtGridHStacked(enum newtGridElement type1, void * what1, ...);
//newtGrid newtGridHCloseStacked(enum newtGridElement type1, void * what1, ...);
newtGrid newtGridBasicWindow(newtComponent text, newtGrid middle,
			     newtGrid buttons);
newtGrid newtGridSimpleWindow(newtComponent text, newtComponent middle,
			     newtGrid buttons);
//void newtGridSetField(newtGrid grid, int col, int row, 
//		      enum newtGridElement type, void * val, int padLeft,
//		      int padTop, int padRight, int padBottom, int anchor,
//		      int flags);
void newtGridPlace(newtGrid grid, int left, int top);
#define newtGridDestroy newtGridFree
void newtGridFree(newtGrid grid, int recurse);
void newtGridGetSize(newtGrid grid, int * width, int * height);
void newtGridWrappedWindow(newtGrid grid, [string,in]char * title);
void newtGridWrappedWindowAt(newtGrid grid, [string,in]char * title, int left, int top);
void newtGridAddComponentsToForm(newtGrid grid, newtComponent form, 
				 int recurse);

//newtGrid newtButtonBarv([string,in]char * button1, newtComponent * b1comp, va_list args);
//newtGrid newtButtonBar([string,in]char * button1, newtComponent * b1comp, ...);

//void newtWinMessage([string,in]char * title, [string,in]char * buttonText, [string,in]char * text, ...);
//void newtWinMessagev([string,in]char * title, [string,in]char * buttonText, [string,in]char * text, 
//		     va_list argv);

//int newtWinChoice([string,in]char * title, [string,in]char * button1, [string,in]char * button2, 
//		   [string,in]char * text, ...);
//int newtWinTernary([string,in]char * title, [string,in]char * button1, [string,in]char * button2, 
//		   [string,in]char * button3, [string,in]char * message, ...);

//int newtWinMenu([string,in]char * title, [string,in]char * text, int suggestedWidth, int flexDown, 
//		int flexUp, int maxListHeight, [string,in]char ** items, int * listItem,
//		[string,in]char * button1, ...);

//struct newtWinEntry {
//    char * text;
//    char ** value;		/* may be initialized to set default */
//    int flags;
//};

//int newtWinEntries([string,in]char * title, [string,in]char * text, int suggestedWidth, int flexDown, 
//		   int flexUp, int dataWidth, 
//		   struct newtWinEntry * items, [string,in]char * button1, ...);
	      *)
