require 'puppet/external/nagios'
require 'puppet/external/nagios/base'
require 'puppet/provider/naginator'

module Puppet::Util::NagiosMaker
  # Create a new nagios type, using all of the parameters
  # from the parser.
  def self.create_nagios_type(name)
    name = name.to_sym
    full_name = ("nagios_#{name}").to_sym

    raise(Puppet::DevError, "No nagios type for #{name}") unless nagtype = Nagios::Base.type(name)

    type = Puppet::Type.newtype(full_name) do

      # Generate a file resource if necessary.
      #
      # @see Puppet::Type::File and its properties owner, group and mode.
      def generate
        return nil unless self[:owner] or self[:group] or self[:mode]
        props = { :name => self[:target] }
        [ :owner, :group, :mode ].each do |prop|
          props[prop] = self[prop] if self[prop]
        end
        Puppet::Type.type(:file).new(props)
      end

    end

    type.ensurable

    type.newparam(nagtype.namevar, :namevar => true) do
      desc "The name of this nagios_#{nagtype.name} resource."
    end

    [ :owner, :group, :mode].each do |fileprop|
      type.newparam(fileprop) do
        desc "The desired #{fileprop} of the config file for this nagios_#{nagtype.name} resource.
 
          NOTE: If the target file is explicitly managed by a file resource in your manifest,
          this parameter has no effect. If a parent directory of the target is managed by
          a recursive file resource, this limitation does not apply (i.e., this parameter
          takes precedence, and if purge is used, the target file is exempt)."
      end
    end

    # We deduplicate the parameters because it makes sense to allow Naginator to have dupes.
    nagtype.parameters.uniq.each do |param|
      next if param == nagtype.namevar

      # We can't turn these parameter names into constants, so at least for now they aren't
      # supported.
      next if param.to_s =~ /^[0-9]/

      type.newproperty(param) do
        desc "Nagios configuration file parameter."
      end
    end

    type.newproperty(:target) do
      desc 'The target.'

      defaultto do
        resource.class.defaultprovider.default_target
      end
    end

    target = "/etc/nagios/#{full_name.to_s}.cfg"
    provider = type.provide(:naginator, :parent => Puppet::Provider::Naginator, :default_target => target) {}
    provider.nagios_type

    type.desc "The Nagios type #{name.to_s}.  This resource type is autogenerated using the
      model developed in Naginator, and all of the Nagios types are generated using the
      same code and the same library.

      This type generates Nagios configuration statements in Nagios-parseable configuration
      files.  By default, the statements will be added to `#{target}`, but
      you can send them to a different file by setting their `target` attribute.

      You can purge Nagios resources using the `resources` type, but *only*
      in the default file locations.  This is an architectural limitation.

    "
  end
end
