/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#ifndef RAST_ERROR_H
#define RAST_ERROR_H

/**
 * @file error.h error
 */

#include <stdarg.h>

#include <apr_pools.h>
#include <apr_errno.h>

#include "rast/config.h"

#ifdef HAVE_XMLRPC
#include <xmlrpc.h>
#endif

#include "rast/macros.h"

RAST_EXTERN_C_BEGIN

/**
 * @defgroup error error
 * @{
 */

/** Enum for error types */
typedef enum {
    RAST_ERROR_TYPE_RAST,     /**< Error of Rast */
    RAST_ERROR_TYPE_APR,      /**< Error of APR */
    RAST_ERROR_TYPE_BDB,      /**< Error of Berkeley DB */
    RAST_ERROR_TYPE_XMLRPC,   /**< Error of XML-RPC protocol */
    RAST_ERROR_TYPE_RUBY,     /**< Error of Ruby */
} rast_error_type_t;

/**
 * A structure that represents an error.
 * rast_error_t is allocated from a global pool, so you must call
 * rast_error_destroy() to destroy it.
 */
typedef struct {
    /** The pool associated with the database */
    apr_pool_t *pool;
    /** The type of the error */
    rast_error_type_t type;
    /** The error code */
    int code;
    /** The error message */
    const char *message;
#ifdef RAST_DEBUG
    /** The file name that error occured */
    const char *file;
    /** The line number that error occured */
    int line;
#endif
} rast_error_t;

/** no error */
#define RAST_OK NULL

rast_error_t *rast_error_vcreate(rast_error_type_t type, int code,
                                 const char *fmt, va_list ap);
rast_error_t *rast_error_create(rast_error_type_t type, int code,
                                const char *fmt, ...);

/**
 * Destory the error.
 * @param error The error to destroy
 */
void rast_error_destroy(rast_error_t *error);

#ifdef RAST_DEBUG

rast_error_t *rast_error_vcreate_debug(rast_error_type_t type, int code,
                                       const char *file, int line,
                                       const char *fmt, va_list ap);
rast_error_t *rast_error_create_debug(rast_error_type_t type, int code,
                                      const char *file, int line,
                                      const char *fmt, ...);
#define rast_error_vcreate(type, code, fmt, ap) \
    rast_error_vcreate_debug(type, code, __FILE__, __LINE__, \
                             fmt, ap)
#define rast_error_create(type, code, ...) \
    rast_error_create_debug(type, code, __FILE__, __LINE__, \
                            __VA_ARGS__)

#define rast_error(code, ...) \
    rast_error_create(RAST_ERROR_TYPE_RAST, code, __VA_ARGS__)

#else /* !RAST_DEBUG */

static inline rast_error_t *
rast_error(int code, const char *fmt, ...)
{
    rast_error_t *error;
    va_list ap;

    va_start(ap, fmt);
    error = rast_error_vcreate(RAST_ERROR_TYPE_RAST, code, fmt, ap);
    va_end(ap);
    return error;
}

#endif /* RAST_DEBUG */

#define apr_status_to_rast_error(status) \
    ((status) == APR_SUCCESS ? RAST_OK : \
     rast_error_create(RAST_ERROR_TYPE_APR, status, NULL))
#define db_error_to_rast_error(error) \
    ((error) == 0 ? RAST_OK : \
     rast_error_create(RAST_ERROR_TYPE_BDB, error, NULL))
#define os_error_to_rast_error(error) \
    apr_status_to_rast_error(APR_FROM_OS_ERROR(error))
#ifdef HAVE_XMLRPC
static inline rast_error_t *
xmlrpc_error_to_rast_error(xmlrpc_env *env, const char *message)
{
    if (env->fault_occurred) {
        const char *format;

        if (message[0] == '\0') {
            format = "%s%s";
        }
        else {
            format = "%s: %s";
        }

        return rast_error_create(RAST_ERROR_TYPE_XMLRPC, (int) env->fault_code,
                                 format, message, env->fault_string);
    }

    return RAST_OK;
}
#endif

#define RAST_ERROR_GENERAL 1
#define RAST_ERROR_NOT_IMPLEMENTED 2
#define RAST_ERROR_CLOSED 3
#define RAST_ERROR_EOF 4
#define RAST_ERROR_CURSOR 5
#define RAST_ERROR_BAD_DB 6
#define RAST_ERROR_INVALID_ARGUMENT 7
#define RAST_ERROR_INVALID_QUERY 8
#define RAST_ERROR_EMPTY_QUERY 9
#define RAST_ERROR_UNSUPPORTED_SCHEME 10
#define RAST_ERROR_UNSUPPORTED_FILTER 11
#define RAST_ERROR_INVALID_SQUENCE 12

static inline int
rast_error_is_eof(rast_error_t *error)
{
    return error != NULL &&
        error->type == RAST_ERROR_TYPE_RAST &&
        error->code == RAST_ERROR_EOF;
}

/** @} */

RAST_EXTERN_C_END

#endif

/* vim: set filetype=c sw=4 expandtab : */
