/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#ifndef RAST_FILTER_H
#define RAST_FILTER_H

#include <apr_buckets.h>
#include <apr_hash.h>
#include <iconv.h>

#include "rast/db.h"

rast_error_t *rast_load_filters(const char *dirname);
rast_error_t *rast_unload_filters();

typedef struct rast_filter_selector_t rast_filter_selector_t;

typedef struct rast_filter_t rast_filter_t;

typedef struct rast_filter_chain_t rast_filter_chain_t;

typedef struct {
    int supported_version;
    void *user_data;
    rast_error_t *(*invoke)(rast_filter_t *filter,
                            apr_bucket_brigade *brigade,
                            const char *mime_type);
} rast_filter_module_t;

struct rast_filter_t {
    void *context;
    rast_filter_module_t *filter_module;
    apr_pool_t *pool;
    rast_filter_chain_t *chain;
};

typedef struct {
    rast_filter_t base;
    rast_filter_selector_t *selector;
} rast_mime_filter_t;

typedef struct rast_text_filter_t rast_text_filter_t;
struct rast_text_filter_t {
    rast_filter_t base;
    rast_text_filter_t *next;
};

typedef struct {
    apr_hash_t *mime_filters;
    apr_hash_t *text_filters;
    apr_hash_t *ext_to_mime;
    apr_pool_t *pool;
} rast_filter_map_t;

typedef rast_error_t *rast_filter_initialize_func_t(rast_filter_map_t *);

/**
 * Add mime filter to the filter map.
 * @param map The filter map
 * @param mime_type The mime_type of filter
 * @param filter_module The filter to add
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_filter_map_add_mime_filter(rast_filter_map_t *map,
                                              const char *mime_type,
                                              rast_filter_module_t *
                                              filter_module);

/**
 * Add text filter to the filter map.
 * @param map The filter map
 * @param name The filter name of filter
 * @param filter_module The filter to add
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_filter_map_add_text_filter(rast_filter_map_t *map,
                                              const char *name,
                                              rast_filter_module_t *
                                              filter_module);

/**
 * Add extension relates mime_type to the filter map.
 * @param map The filter map
 * @param ext The extension to add
 * @param mime_type The mime_type to relate ext
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_filter_map_add_extension(rast_filter_map_t *map,
                                            const char *ext,
                                            const char *mime_type);

/**
 * Pass the next mime filter.
 * @param filter The filter
 * @param brigade The brigade that passed
 * @param mime_type The mime_type to specify data type of the brigade
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_mime_filter_pass(rast_filter_t *filter,
                                    apr_bucket_brigade *brigade,
                                    const char *mime_type,
                                    const char *filename);

/**
 * Pass the next text filter.
 * @param filter The filter
 * @param brigade The brigade that passed
 * @param mime_type The mime_type to specify data type of the brigade
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_text_filter_pass(rast_filter_t *base,
                                    apr_bucket_brigade *brigade,
                                    const char *mime_type);

/**
 * Set property to the document.
 * @param filter The filter
 * @param name The property name
 * @param value The property value
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_filter_set_property(rast_filter_t *filter, const char *name,
                                       const rast_value_t *value);

/**
 * Get charset encoding of the database.
 * @param filter The filter
 * @return The charset encoding of the database
 */
const char *rast_filter_db_encoding(rast_filter_t *filter);

/**
 * Creates filter chain.
 * @param doc The document
 * @param final_filters The text filter names in char * array.
 * @param num_final_filters The number of final_filters
 * @param pool The pool to allocate the memory out of
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_filter_chain_create(rast_filter_chain_t **chain,
                                       rast_document_t *doc,
                                       const char **final_filters,
                                       int num_final_filters,
                                       apr_pool_t *pool);

/**
 * Start extracting text and properties with filters.
 * @param filter The filter
 * @param brigade The brigade stored data to extract text and properties
 * @param mime_type The mime-type to specify the file type of the brigade
 * @param filename The filename of the file to extract text and properties
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_filter_chain_invoke(rast_filter_chain_t *chain,
                                       apr_bucket_brigade *brigade,
                                       const char *mime_type,
                                       const char *filename);

typedef struct {
    iconv_t cd;
    char *head_buf;
    int head_buf_nbytes;
    int done;
    apr_pool_t *pool;
} rast_encoding_converter_t;

extern const char * const RAST_JAPANESE_ENCODINGS[];

/**
 * Guess the charset encoding of text.
 * @param candidate_encodings Candidate charset encoding to guess
 * @param text The text to guess
 * @param text_nbytes The length of the text in bytes
 * @param result The guessed charset encoding
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_encoding_converter_guess(const char * const *,
                                            const char *text,
                                            size_t text_nbytes,
                                            const char **result);

/**
 * Create encoding converter.
 * @param result The encoding converter to create
 * @param from_encoding The charset encoding before convertion
 * @param to_encodings The candidate charset encodings converted into
 * @param pool The pool to allocate the memory out of
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_encoding_converter_create(rast_encoding_converter_t **
                                             result,
                                             const char *from_encoding,
                                             const char **to_encodings,
                                             apr_pool_t *pool);

/**
 * Add text to the encoding converter.
 * @param converter The encoding converter
 * @param buf the text to add
 * @param nbytes The length of the buf in bytes
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_encoding_converter_add_text(rast_encoding_converter_t *
                                               converter,
                                               const char *buf,
                                               int buf_nbytes);

/**
 * Convert text to one of the candidate charset encodings.
 * @param converter The encoding converter
 * @param out_buf The buffer to store converted text
 * @param out_buf_nbytes The length of the out_buf in bytes
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_encoding_converter_get_next(rast_encoding_converter_t *
                                               converter,
                                               char *out_buf,
                                               int *out_buf_nbytes);

/**
 * Check whether converter has text to convert or not.
 * @param converter The encoding converter
 * @return 1 if converter has text to convert, 0 otherwise
 */
int rast_encoding_converter_is_done(rast_encoding_converter_t *converter);

/**
 * Convert text to the target charset encoding.
 * @param from_encoding Candidate charset encoding to guess
 * @param to_encodings The text to guess
 * @param text The length of the test_text in bytes
 * @param nbytes The length of the test_text in bytes

 * @param result The guessed charset encoding
 * @return RAST_OK if succeeded, error otherwise
 */
rast_error_t *rast_convert_encoding(const char *from_encoding,
                                    const char *to_encoding,
                                    const char *text, size_t nbytes,
                                    char **out_buf, size_t *out_buf_nbytes,
                                    apr_pool_t *pool);

#endif
