/*
 * LUKS - Linux Unified Key Setup 
 *
 * Copyright (C) 2004-2005, Clemens Fruhwirth <clemens@endorphin.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <libdevmapper.h>



#include <libdevmapper.h>

#include "luks.h"
#include "../lib/libcryptsetup.h"
#include "../lib/internal.h"
#include "../lib/blockdev.h"

static int setup_mapping(const char *cipher, const char *name, 
			 const char *device, unsigned int payloadOffset,
			 const char *key, unsigned int keyLength, 
			 unsigned int sector, unsigned int srcLength, 
			 struct setup_backend *backend,
			 int mode)
{
	struct crypt_options k;
	struct crypt_options *options = &k;

	int r;

	options->offset = sector;
	options->size = payloadOffset;	

	options->cipher = cipher;
	options->key_size = keyLength;
	options->skip = 0; options->flags = 0;
	options->name = name;
	options->device = device;
	
	if (options->size <= options->offset) {
		set_error("Invalid offset");
		return -EINVAL;
	}

	if (mode == O_RDONLY) {
		options->flags |= CRYPT_FLAG_READONLY;
	}

	r = backend->create(0, options, key);

	if (r <= 0)
		set_error(NULL);

	return r;
}

static int clear_mapping(const char *name, struct setup_backend *backend)
{
	struct crypt_options options;
	options.name=name;
	return backend->remove(&options);
}

static int LUKS_endec_template(char *src, unsigned int srcLength, 
			       struct luks_phdr *hdr, 
			       char *key, unsigned int keyLength, 
			       const char *device, 
			       unsigned int sector, struct setup_backend *backend,
			       ssize_t (*func)(int, void *, size_t),
			       int mode)
{
	int devfd;
	char *name;
	char *fullpath;
	char *dmCipherSpec;
	/* FIXME: we are breaking the abstraction here  */
	const char *dmDir = dm_dir(); 
	int r;

	/* Otherwise this function would look even more ugly. C is obsolet. */
#define EXIT_ON_ALLOC_FAIL(label)                                                         \
	if(r < 0) {                                                                       \
		fprintf(stderr,"memory allocation failed in LUKS_endec_template.\n");     \
		goto label;                                                               \
	}

	
	if(dmDir == NULL) {
		fprintf(stderr,"failed to obtain dm_dir from device mapper library.\n");
		return -1;
	}
	r = asprintf(&name,"temporary-cryptsetup-%d",getpid());
	EXIT_ON_ALLOC_FAIL(out1a);
	r = asprintf(&fullpath,"%s/%s",dmDir,name);
	EXIT_ON_ALLOC_FAIL(out1b);
	asprintf(&dmCipherSpec,"%s-%s",hdr->cipherName, hdr->cipherMode);
	EXIT_ON_ALLOC_FAIL(out1c);

	r = setup_mapping(dmCipherSpec,name,device,hdr->payloadOffset,key,keyLength,sector,srcLength,backend,mode);
	if(r < 0) {
		fprintf(stderr,"failed to setup dm-crypt mapping.\n");
		goto out1c;
	}

	devfd = open(fullpath, mode | O_DIRECT | O_SYNC);
	if(devfd == -1) { r = -EIO; goto out2; }

	r = func(devfd,src,srcLength);
	if(r < 0) { r = -EIO; goto out3; }

	r = 0;
 out3:
	close(devfd);
 out2:
	clear_mapping(name,backend);
 out1c:
	free(dmCipherSpec);
 out1b:
	free(fullpath); 
 out1a:
	free(name); 
	return r;
#undef EXIT_ON_ALLOC_FAIL
}

int LUKS_encrypt_to_storage(char *src, unsigned int srcLength, 
			    struct luks_phdr *hdr, 
			    char *key, unsigned int keyLength, 
			    const char *device, 
			    unsigned int sector, struct setup_backend *backend)
{
	
	return LUKS_endec_template(src,srcLength,hdr,key,keyLength, device, sector, backend,	
				   (ssize_t (*)(int, void *, size_t)) write_blockwise, O_RDWR);
}	

int LUKS_decrypt_from_storage(char *dst, unsigned int dstLength, 
			      struct luks_phdr *hdr, 
			      char *key, unsigned int keyLength, 
			      const char *device, 
			      unsigned int sector, struct setup_backend *backend)
{
	return LUKS_endec_template(dst,dstLength,hdr,key,keyLength, device, sector, backend, read_blockwise, O_RDONLY);
}
