/*
 *  (C) 2004  Dominik Brodowski <linux@dominikbrodowski.de>
 *
 *  Licensed under the terms of the GNU GPL License version 2.
 */


#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>

#define _GNU_SOURCE
#include <getopt.h>

#include <linux/threads.h> /* for NR_CPUS */

#include "cpufreq.h"
#include "config.h"

static void proc_cpufreq_output(unsigned int classic)
{
	unsigned int cpu;
	struct cpufreq_policy *policy;
	unsigned int min_pctg = 0;
	unsigned int max_pctg = 0;
	unsigned long min, max;

	printf("          minimum CPU frequency  -  maximum CPU frequency  -  %s\n", classic ? "policy" : "governor");

	for (cpu=0; cpu < NR_CPUS; cpu++) {
		policy = cpufreq_get_policy(cpu);
		if (!policy)
			continue;

		if (cpufreq_get_hardware_limits(cpu, &min, &max)) {
			max = 0;
		} else {
			min_pctg = (policy->min * 100) / max;
			max_pctg = (policy->max * 100) / max;
		}
		printf("CPU%3d    %9lu kHz (%3d %%)  -  %9lu kHz (%3d %%)  -  %s\n",
		       cpu , policy->min, max ? min_pctg : 0, policy->max, max ? max_pctg : 0, policy->governor);

		cpufreq_put_policy(policy);
	}
}

static void print_speed(unsigned long speed)
{
	unsigned long tmp;

	if (speed > 1000000) {
		tmp = speed % 10000;
		if (tmp >= 5000)
			speed += 10000;
		printf ("%u.%02u GHz", ((unsigned int) speed/1000000), 
			((unsigned int) (speed%1000000)/10000));
	} else if (speed > 100000) {
		tmp = speed % 1000;
		if (tmp >= 500)
			speed += 1000;
		printf ("%u MHz", ((unsigned int) speed / 1000));
	} else if (speed > 1000) {
		tmp = speed % 100;
		if (tmp >= 50)
			speed += 100;
		printf ("%u.%01u MHz", ((unsigned int) speed/1000), 
			((unsigned int) (speed%1000)/100));
	} else
		printf ("%lu kHz", speed);

	return;
}

static void debug_output_one(unsigned int cpu)
{
	char *driver;
	struct cpufreq_affected_cpus *cpus;
	struct cpufreq_available_frequencies *freqs;
	unsigned long min, max, freq_kernel, freq_hardware;
	struct cpufreq_policy *policy;
	struct cpufreq_available_governors * governors;

	if (cpufreq_cpu_exists(cpu)) {
		printf("couldn't analyze CPU %d as it doesn't seem to be present\n", cpu);
		return;
	}

	printf("analyzing CPU %d:\n", cpu);

	driver = cpufreq_get_driver(cpu);
	if (!driver) {
		printf("  no or unknown cpufreq driver is active on this CPU\n");
	} else {
		printf("  driver: %s\n", driver);
		cpufreq_put_driver(driver);
	}

	cpus = cpufreq_get_affected_cpus(cpu);
	if (cpus) {
		printf("  CPUs which need to switch frequency at the same time: ");
		while (cpus->next) {
			printf("%d ", cpus->cpu);
			cpus = cpus->next;
		}
		printf("%d\n", cpus->cpu);
		cpufreq_put_affected_cpus(cpus);
	}

	if (!(cpufreq_get_hardware_limits(cpu, &min, &max))) {
		printf("  hardware limits: ");
		print_speed(min);
		printf(" - ");
		print_speed(max);
		printf("\n");
	}

	freqs = cpufreq_get_available_frequencies(cpu);
	if (freqs) {
		printf("  available frequency steps: ");
		while (freqs->next) {
			print_speed(freqs->frequency);
			printf(", ");
			freqs = freqs->next;
		}
		print_speed(freqs->frequency);
		printf("\n");
		cpufreq_put_available_frequencies(freqs);
	}
	
	governors = cpufreq_get_available_governors(cpu);
	if (governors) {
		printf("  available cpufreq governors: ");
		while (governors->next) {
			printf("%s, ", governors->governor);
			governors = governors->next;
		}
		printf("%s\n", governors->governor);
		cpufreq_put_available_governors(governors);
	}

	policy = cpufreq_get_policy(cpu);
	if (policy) {
		printf("  current policy: frequency should be within ");
		print_speed(policy->min);
		printf(" and ");
		print_speed(policy->max);
		printf(".\n                  The governor \"%s\" may"
		       " decide which speed to use\n                  within this range.\n", 
		       policy->governor);
		cpufreq_put_policy(policy);
	}

	freq_kernel = cpufreq_get_freq_kernel(cpu);
	freq_hardware = cpufreq_get_freq_hardware(cpu);

	if (freq_kernel || freq_hardware) {
		printf("  current CPU frequency is ");
		if (freq_hardware) {
			print_speed(freq_hardware);
			printf(" (asserted by call to hardware)");
		}
		else
			print_speed(freq_kernel);
		printf(".\n");
	}
}

static void debug_output(unsigned int cpu, unsigned int all) {
	if (all) {
		for (cpu=0; cpu < NR_CPUS; cpu++) {
			if (cpufreq_cpu_exists(cpu))
				continue;
			debug_output_one(cpu);
		}
	} else
		debug_output_one(cpu);
}


/* --freq / -f */

static int get_freq_kernel(unsigned int cpu, unsigned int human) {
	unsigned long freq = cpufreq_get_freq_kernel(cpu);
	if (!freq)
		return -EINVAL;
	if (human) {
		print_speed(freq);
		printf("\n");
	} else
		printf("%lu\n", freq);
	return 0;
}


/* --hwfreq / -h */

static int get_freq_hardware(unsigned int cpu, unsigned int human) {
	unsigned long freq = cpufreq_get_freq_hardware(cpu);
	if (!freq)
		return -EINVAL;
	if (human) {
		print_speed(freq);
		printf("\n");
	} else
		printf("%lu\n", freq);
	return 0;
}

/* --hwlimits / -l */

static int get_hardware_limits(unsigned int cpu) {
	unsigned long min, max;
	if (cpufreq_get_hardware_limits(cpu, &min, &max))
		return -EINVAL;
	printf("%lu %lu\n", min, max);
	return 0;
}

/* --driver / -d */

static int get_driver(unsigned int cpu) {
	char *driver = cpufreq_get_driver(cpu);
	if (!driver)
		return -EINVAL;
	printf("%s\n", driver);
	cpufreq_put_driver(driver);
	return 0;
}

/* --policy / -p */

static int get_policy(unsigned int cpu) {
	struct cpufreq_policy *policy = cpufreq_get_policy(cpu);
	if (!policy)
		return -EINVAL;
	printf("%lu %lu %s\n", policy->min, policy->max, policy->governor);
	cpufreq_put_policy(policy);
	return 0;
}

/* --governors / -g */

static int get_available_governors(unsigned int cpu) {
	struct cpufreq_available_governors *governors = cpufreq_get_available_governors(cpu);
	if (!governors)
		return -EINVAL;

	while (governors->next) {
		printf("%s ", governors->governor);
		governors = governors->next;
	}
	printf("%s\n", governors->governor);
	cpufreq_put_available_governors(governors);
	return 0;
}


/* --affected-cpus  / -a */

static int get_affected_cpus(unsigned int cpu) {
	struct cpufreq_affected_cpus *cpus = cpufreq_get_affected_cpus(cpu);
	if (!cpus)
		return -EINVAL;

	while (cpus->next) {
		printf("%d ", cpus->cpu);
		cpus = cpus->next;
	}
	printf("%d\n", cpus->cpu);
	cpufreq_put_affected_cpus(cpus);
	return 0;
}

static void print_header(void) {
	printf(PACKAGE_STRING ": cpufreq-info (C) Dominik Brodowski 2004\n");
	printf("Report errors and bugs to "PACKAGE_BUGREPORT", please\n");
}

static void print_help(void) {
	printf("Usage: cpufreq-info [options]\n");
	printf("Options:\n");
	printf("  -c CPU, --cpu CPU    CPU number which information shall be determined about\n");
	printf("  -e, --debug          Prints out debug information\n");
	printf("  -f, --freq           Get frequency the CPU currently runs at, according\n"
	       "                       to the cpufreq core *\n");
	printf("  -w, --hwfreq         Get frequency the CPU currently runs at, by reading\n"
	       "                       it from hardware (only available to root) *\n");
	printf("  -l, --hwlimits       Determine the minimum and maximum CPU frequency allowed *\n");
	printf("  -d, --driver         Determines the used cpufreq kernel driver *\n");
	printf("  -p, --policy         Gets the currently used cpufreq policy *\n");
	printf("  -g, --governors      Determines available cpufreq governors *\n");
	printf("  -a, --affected-cpus  Determines which CPUs can only switch frequency at the\n");
	printf("                       same time *\n");
	printf("  -o, --proc           Prints out information like provided by the /proc/cpufreq\n");
	printf("                       interface in 2.4. and early 2.6. kernels\n");
	printf("  -m, --human          human-readable output for the -f and -w parameters\n");
	printf("  -h, --help           Prints out this screen\n");

	printf("\n");
	printf("If no argument or only the -c, --cpu parameter is given, debug output about\n");
	printf("cpufreq is printed which is useful e.g. for reporting bugs.\n");
	printf("For the arguments marked with *, omitting the -c or --cpu argument is\n");
	printf("equivalent to setting it to zero\n");
}

static struct option info_opts[] = {
	{ .name="cpu",		.has_arg=required_argument,	.flag=NULL,	.val='c'},
	{ .name="debug",	.has_arg=no_argument,		.flag=NULL,	.val='e'},
	{ .name="freq",		.has_arg=no_argument,		.flag=NULL,	.val='f'},
	{ .name="hwfreq",	.has_arg=no_argument,		.flag=NULL,	.val='w'},
	{ .name="hwlimits",	.has_arg=no_argument,		.flag=NULL,	.val='l'},
	{ .name="driver",	.has_arg=no_argument,		.flag=NULL,	.val='d'},
	{ .name="policy",	.has_arg=no_argument,		.flag=NULL,	.val='p'},
	{ .name="governors",	.has_arg=no_argument,		.flag=NULL,	.val='g'},
	{ .name="affected-cpus",.has_arg=no_argument,		.flag=NULL,	.val='a'},
	{ .name="proc",		.has_arg=no_argument,		.flag=NULL,	.val='o'},
	{ .name="human",	.has_arg=no_argument,		.flag=NULL,	.val='m'},
	{ .name="help",		.has_arg=no_argument,		.flag=NULL,	.val='h'},
};

int main(int argc, char **argv) {
	extern char *optarg;
	extern int optind, opterr, optopt;
	int ret = 0, cont = 1;
	unsigned int cpu = 0;
	unsigned int cpu_defined = 0;
	unsigned int human = 0;
	int output_param = 0;

	do {
		ret = getopt_long(argc, argv, "c:hoefwldpgam", info_opts, NULL);
		switch (ret) {
		case '?':
			output_param = '?';
			cont = 0;
			break;
		case 'h':
			output_param = 'h';
			cont = 0;
			break;
		case -1:
			cont = 0;
			break;
		case 'o':
		case 'a':
		case 'g':
		case 'p':
		case 'd':
		case 'l':
		case 'w':
		case 'f':
		case 'e':
			if (output_param) { 
				output_param = -1; 
				cont = 0; 
				break; 
			}
			output_param = ret;
			break;
		case 'c':
			if (cpu_defined) {
				output_param = -1;
				cont = 0;
				break;
			}
			if ((sscanf(optarg, "%d ", &cpu)) != 1) {
				output_param = '?';
				cont = 0;
			}
			cpu_defined = 1;
			break;
		case 'm':
			if (human) {
				output_param = -1;
				cont = 0;
				break;
			}
			human = 1;
			break;
		}
	} while(cont);
	
	switch (output_param) {
	case 'o':
		if (cpu_defined) {
			print_header();
			printf("passed argument can't work with a CPU argument\n");
			return -EINVAL;
		}
		break;
	case 0:
		output_param = 'e';
	}

	ret = 0;

	switch (output_param) {
	case -1:
		print_header();
		printf("more than one output-specific or CPU argument passed\n");
		return -EINVAL;
		break;
	case '?':
		print_header();
		printf("invalid argument\n");
		print_help();
		ret = -EINVAL;
		break;
	case 'h':
		print_header();
		print_help();
		break;
	case 'o':
		proc_cpufreq_output(1);
		break;
	case 'e':
		print_header();
		debug_output(cpu, !(cpu_defined));
		break;
	case 'a':
		ret = get_affected_cpus(cpu);
		break;
	case 'g':
		ret = get_available_governors(cpu);
		break;
	case 'p':
		ret = get_policy(cpu);
		break;
	case 'd':
		ret = get_driver(cpu);
		break;
	case 'l':
		ret = get_hardware_limits(cpu);
		break;
	case 'w':
		ret = get_freq_hardware(cpu, human);
		break;
	case 'f':
		ret = get_freq_kernel(cpu, human);
		break;
	}
	return (ret);
}
