/*
 * Handle module dependencies
 *
 * Copyright 1994, 1995, 1996, 1997: Jacques Gelinas <jack@solucorp.qc.ca>
 * Additional modifications: Bjrn Ekwall <bj0rn@blox.se> February 1999
 *
 * This file is part of the Linux modutils.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <sys/stat.h>
#include <ctype.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <elf.h>
#include ELF_MACHINE_H
extern "C" {
#include "obj.h"
#include "util.h"
};
#include "depmod.h"

#define ALLOC_MODULE	5000

/*
 *	Create the list of all modules encountered in the link
 */
MODULES::MODULES()
{
	/* #Specification: libs / maxumum number
	 * A maximum of 255 libs are allowed during the link
	 */
	tblibs = (char **) xmalloc(255 * sizeof(char *));
	nblib = 0;
	tbmod = (MODULE *) xmalloc(ALLOC_MODULE * sizeof(MODULE));
	nbmod = 0;
}

/*
 *	Finalize the information about a module.
 */
void MODULES::setmod(
	       MODULE * mod,
	       SYMBOL * tbpub[],
	       int nbpub,
	       SYMBOL * tbext[],
	       int nbext,
	       int module_need)
{
	mod->is_load = module_need;
	mod->pub.nb = nbpub;
	mod->ext.nb = nbext;
	if (nbpub > 0) {
		int size = nbpub * sizeof(SYMBOL *);

		mod->pub.tb = (SYMBOL **) xmalloc(size);
		memcpy(mod->pub.tb, tbpub, size);
	} else {
		mod->pub.tb = NULL;
	}

	if (nbext > 0) {
		int size = nbext * sizeof(SYMBOL *);

		mod->ext.tb = (SYMBOL **) xmalloc(size);
		memcpy(mod->ext.tb, tbext, size);
		if (module_need) {
			SYMBOL **ptext = mod->ext.tb;
			for (int i = 0; i < nbext; i++, ptext++)
				(*ptext)->need = 1;
		}
	} else {
		mod->ext.tb = NULL;
	}
}

/*
 *	Create a module entry in the list of modules.
 *	This module is generally not related to any real file.
*/
MODULE *MODULES::setdummy(const char *name)
{
	MODULE *mod = tbmod + nbmod++;

	mod->name = xstrdup(name);
	mod->lib = -1;
	mod->is_load = 0;
	mod->pub.nb = mod->ext.nb = 0;
	mod->ext.tb = NULL;
	mod->pub.tb = NULL;

	return mod;
}

/*
 *	Read the symbols in an object and register them in syms
 *	Return -1 if there is an error.
 */
int MODULES::loadobj(SYMBOLS & syms, const char *objname)
{
	MODULE *mod = tbmod + nbmod++;
	FILE *fp;
	struct obj_file *f;
	struct obj_section *sect;
	struct obj_symbol *sym;
	SYMBOL *tbext[5000];
	int nbext = 0;
	SYMBOL *tbpub[5000];
	int nbpub = 0;
	int ksymtab;
	int mr = 0;
	int i;

	mod->name = xstrdup(objname);
	mod->lib = -1;

	if ((fp = fopen(objname, "r")) == NULL)
		return 1;

	if (!(f = obj_load(fp))) {
		fclose(fp);
		return -1;
	}
	fclose(fp);

	if ((sect = obj_find_section(f, "__ksymtab")) != NULL)
		ksymtab = sect->idx; /* Only in 2.2 (or at least not 2.0) */
	else
		ksymtab = -1;

	for (i = 0; i < HASH_BUCKETS; ++i) {
		for (sym = f->symtab[i]; sym; sym = sym->next) {
			if (sym->secidx == SHN_UNDEF) {
				tbext[nbext++] = syms.add(sym->name,
							  NULL,
							  SYM_NEED,
							  mr,
							  0);
			}
			else if (ksymtab != -1 && sym->secidx == ksymtab &&
				 ELFW (ST_BIND) (sym->info) == STB_GLOBAL) {
				/* A 2.2 module using EXPORT_SYMBOL */
				tbpub[nbpub++] = syms.add(sym->name + 10,
							  mod,
							  SYM_DEFINED,
							  mr,
							  0);
			} else if (ksymtab == -1
			/*
			 * The test below is removed for 2.0 compatibility
			 * since some 2.0-modules (correctly) hides the
			 * symbols it exports via register_symtab()
			 */
			/* && ELFW (ST_BIND) (sym->info) == STB_GLOBAL */) {
				tbpub[nbpub++] = syms.add(sym->name,
							  mod,
							  SYM_DEFINED,
							  mr,
							  (sym->secidx == SHN_COMMON));
			}
		}
	}
	setmod(mod, tbpub, nbpub, tbext, nbext, 1);
	obj_free(f);

	return 0;
}

static int cmp(const void *p1, const void *p2)
{
	MODULE *pt1 = *(MODULE **) p1;
	MODULE *pt2 = *(MODULE **) p2;

	return strcmp(pt1->name, pt2->name);
}

/*
 *	Format the dependancy list of a module into a simple makefile
 */
void MODULES::prtdepend(
		  FILE * fout,
		  const char *dontcare, /* Module we don't want to know about
		  			 * in the dependancy lists
					 */
		  int skipchars,	/* For depmod -a in an image of a tree */
		  int verbose,		/* Print all modules visiteds */
		  int quiet,		/* Don't print errors */
		  int showerror)	/* Shows undefined symbols */
{
	MODULE **tbdep = new MODULE *[nbmod];
	MODULE *ptmod = tbmod;

	for (int i = 0; i < nbmod; i++, ptmod++) {
		if (strcmp(ptmod->name, dontcare) != 0) {
			SYMBOL **ptext = ptmod->ext.tb;
			int nbext = ptmod->ext.nb;
			int nbdepmod = 0;
			int nberr = 0;

			for (int e = 0; e < nbext; e++, ptext++) {
				MODULE *mod = (*ptext)->module;

				if (mod == NULL) {
					if (!quiet) {
						if (nberr == 0) {
							error("*** Unresolved symbols in module %s"
							      ,ptmod->name);
						}
						if (showerror)
							error("\t%s", (*ptext)->name);
					}
					nberr++;
				} else {
					if (strcmp(mod->name, dontcare) != 0) {
						int m;
						for (m = 0; m < nbdepmod; m++) {
							if (tbdep[m] == mod)
								break;
						}
						if (m == nbdepmod)
							tbdep[nbdepmod++] = mod;
					}
				}
			}
			if (nberr == 0 && verbose)
				printf("%s\n", ptmod->name + skipchars);

			/* Sort so it is nicer to look at:-) */
			qsort(tbdep, nbdepmod, sizeof(MODULE *), cmp);
			fprintf(fout, "%s:", ptmod->name + skipchars);
			for (int m = 0; m < nbdepmod; m++) {
				if (m != 0 && (m & 3) == 0)
					fprintf(fout, "\\\n");
				fprintf(fout, "\t%s", tbdep[m]->name + skipchars);
			}
			fprintf(fout, "\n\n");
		}
	}
	delete[]tbdep;
}
