#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>
#include <zlib.h>
#include "global.h"
#include "proto.h"

#ifndef MD
#  define MD MD5
#endif

#if MD == 2
#  include "md2.h"
#elif MD == 4
#  include "md4.h"
#elif MD == 5
#  include "md5.h"
#endif

#if MD == 2
#  define MD_CTX MD2_CTX
#  define MDInit MD2Init
#  define MDUpdate MD2Update
#  define MDFinal MD2Final
#elif MD == 4
#  define MD_CTX MD4_CTX
#  define MDInit MD4Init
#  define MDUpdate MD4Update
#  define MDFinal MD4Final
#elif MD == 5
#  define MD_CTX MD5_CTX
#  define MDInit MD5Init
#  define MDUpdate MD5Update
#  define MDFinal MD5Final
#endif

#define CLOOP_HEADROOM 128
#define CLOOP_PREAMBLE "#!/bin/sh\n" "#V2.0 Format\n" "insmod cloop.o file=$0 && mount -r -t iso9660 /dev/cloop $1\n" "exit $?\n"

typedef unsigned long long __u64;
#define __cpu_to_be64(x) \
        ((__u64)( \
                (__u64)(((__u64)(x) & (__u64)0x00000000000000ffULL) << 56) | \
                (__u64)(((__u64)(x) & (__u64)0x000000000000ff00ULL) << 40) | \
                (__u64)(((__u64)(x) & (__u64)0x0000000000ff0000ULL) << 24) | \
                (__u64)(((__u64)(x) & (__u64)0x00000000ff000000ULL) <<  8) | \
                (__u64)(((__u64)(x) & (__u64)0x000000ff00000000ULL) >>  8) | \
                (__u64)(((__u64)(x) & (__u64)0x0000ff0000000000ULL) >> 24) | \
                (__u64)(((__u64)(x) & (__u64)0x00ff000000000000ULL) >> 40) | \
                (__u64)(((__u64)(x) & (__u64)0xff00000000000000ULL) >> 56) ))

#define __be64_to_cpu(x) __cpu_to_be64(x)


typedef struct link_list_stub {
    struct link_list_stub *prev, *next;
    off_t offset;
    off_t size;
    int count;
    char *content;
} link_list;

typedef struct cloop_header_stub {
    char preamble[CLOOP_HEADROOM];
    unsigned long block_size;
    unsigned long num_blocks;
    loff_t offsets[0];
} cloop_header;

typedef struct index_body_stub {
    loff_t offset;
    unsigned char blockname[16];
} index_body;

link_list **list_array = NULL;
unsigned long blocksize = 0;
unsigned long blockcount = 0;
cloop_header *file_header;

void create_dir()
{
    FILE *fp;
    char *index_buf;
    size_t index_size;
    unsigned char *blockfilename;
    unsigned char *header_md5;
    unsigned long header_tmp_nl;
    loff_t blockoffset_nl;
    index_body blockinfo[2];
    int count, i;
    char subdir_temp[3];

    if (mkdir("/var/tmp/blocks", 0700) != 0) {
        if (errno != EEXIST) {
            perror("mkdir");
            exit(1);
        }
    }
    chdir("/var/tmp/blocks");

    for (i = 0; i < 0x100; i++) {
        sprintf(subdir_temp, "%02x", i);
        if (mkdir(subdir_temp, 0700) != 0) {
            if (errno != EEXIST) {
                perror("mkdir");
                exit(1);
            }
        }
    }

    if ((fp = fopen("index.idx", "r")) == NULL) {
        perror("reading index file");
        exit(1);
    }

    if ((header_md5 = (unsigned char *)malloc(sizeof(unsigned char) * 16)) == NULL) {
        perror("allocate header");
        exit(1);
    }
    if (fread(header_md5, sizeof(unsigned char), 16, fp) < 16) {
        perror("reading header");
        exit(1);
    }
    /* TODO: check header MD5sum */

    /* reading header: blocksize */
    if (fread(&header_tmp_nl, sizeof(unsigned long), 1, fp) == 0) {
        perror("reading header blocksize");
        exit(1);
    }
    blocksize = ntohl(header_tmp_nl);

    /* reading header: blockcount */
    if (fread(&header_tmp_nl, sizeof(unsigned long), 1, fp) == 0) {
        perror("reading header blockcount");
        exit(1);
    }
    blockcount = ntohl(header_tmp_nl);
    list_array = (link_list **)calloc(blockcount + 1, sizeof(link_list *));
    file_header = (cloop_header *)malloc(sizeof(cloop_header) + sizeof(off_t) * (blockcount + 1));
    memset(file_header->preamble, 0, sizeof(file_header->preamble));
    memcpy(file_header->preamble, CLOOP_PREAMBLE, sizeof(CLOOP_PREAMBLE));
    file_header->block_size = htonl(blocksize);
    file_header->num_blocks = htonl(blockcount);

    for (count = 0; count < blockcount; count++) {
        list_array[count] = (link_list *)malloc(sizeof(link_list));
        fread(&blockinfo, sizeof(index_body), 2, fp);
        fseek(fp, -sizeof(index_body), SEEK_CUR);
        list_array[count]->offset = __be64_to_cpu(blockinfo[0].offset);
        list_array[count]->size = __be64_to_cpu(blockinfo[1].offset);
        list_array[count]->size -= list_array[count]->offset;
        list_array[count]->count = count;
//      fprintf(stderr, "DEBUG: data block %d : size=%lld, offset=%lld\n", count, list_array[count]->offset, list_array[count]->size);
        list_array[count]->content = (unsigned char *)malloc(sizeof(unsigned char) * 16);
        file_header->offsets[count] = __cpu_to_be64(__be64_to_cpu(blockinfo[0].offset) + sizeof(cloop_header) + sizeof(off_t) * (blockcount + 1));
        memcpy(list_array[count]->content, blockinfo[0].blockname, sizeof(unsigned char) * 16);
        if (count > 0) {
            list_array[count-1]->next = list_array[count];
            list_array[count]->prev = list_array[count-1];
        }
    }

//    fprintf(stderr, "DEBUG: blockinfo[final].offset = %lu\n", __be64_to_cpu(blockinfo[1].offset));
    file_header->offsets[blockcount] = __cpu_to_be64(__be64_to_cpu(blockinfo[1].offset) + sizeof(cloop_header) + sizeof(off_t) * (blockcount + 1));
    list_array[blockcount] = (link_list *)malloc(sizeof(link_list));
    list_array[blockcount]->offset = __be64_to_cpu(blockinfo[1].offset);
    list_array[blockcount]->size = 0L;
    list_array[blockcount - 1]->next = list_array[blockcount];
    list_array[blockcount]->next = NULL;
    list_array[blockcount]->prev = list_array[blockcount - 1];
    list_array[blockcount]->count = blockcount;

    return;
}

void bintohex(char *dest, unsigned char *src, int size)
{
    int count;

    memset(dest, 0, size * 2 + 1);

    sprintf(dest, "%02x/", src[0]);

    for (count = 0; count < size; count++) {
        sprintf(&dest[count * 2 + 3], "%02x", src[count]);
    }
    return;
}

int main (int argc, char *argv[])
{
    FILE *fp;
    unsigned char *comprbuf;
    unsigned char *uncomprbuf;
    unsigned long comprlen, uncomprlen;
    unsigned char target_filename[32 + 3 + 1];
    int z_err;
    int i, incomplete = 0;
    
    create_dir();
    if ((comprbuf = malloc(sizeof(unsigned char) * (blocksize + blocksize / 1000 + 12 + 4))) == NULL)
        exit(1);
    if ((uncomprbuf = malloc(sizeof(unsigned char) * blocksize)) == NULL)
        exit(1);

    printf("Checking %d blocks [", blockcount);
    for (i = 0 ; i < blockcount ; i++) {
        uncomprlen = blocksize;

        bintohex(target_filename, list_array[i]->content, 16);
    
        if((fp = fopen(target_filename, "r")) == NULL) {
            if (errno != ENOENT)
	        exit(1);
            fprintf(stderr, "%s not found.\n", target_filename);
            incomplete = 1;
        }
        comprlen = fread(comprbuf, sizeof(char), blocksize + blocksize / 1000 + 12 + 4, fp);
        if (comprlen < 0) {
            perror("fread");
	    return 1;
        }
        fclose(fp);
        z_err = uncompress(uncomprbuf, &uncomprlen, comprbuf, comprlen);
    
        if (z_err != Z_OK) {
            fprintf(stderr, "Uncompress error %d\n", z_err);
        }
        if (memcmp(MDString(uncomprbuf, uncomprlen), list_array[i]->content, 16)  != 0) {
            fprintf(stderr, "%s maybe corrupt.\n", target_filename);
        }
        if (i % 1000 == 0) {
            printf(" %d ", i);
        } else if (i % 100 == 0) {
            printf(".");
        }
        fflush(stdout);
        memset(comprbuf, 0, blocksize + blocksize / 1000 + 12 + 4);
        memset(uncomprbuf, 0, blocksize);
    }
    printf("] done.\n");
    free(comprbuf);
    free(uncomprbuf);
    if (incomplete == 0) {
        creat("/var/tmp/blocks/block_complete", 0600);
    }
    return 0;
}
