/*
 * mkmd5files.c
 * Copyright (C) 2005 YAGI, Toshiki <yagi-toshiki@aist.go.jp>
 *
 * This program can be distributed under the terms of the GNU GPL.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <errno.h>
#include "global.h"
#include "proto.h"
#include "zlib.h"


#ifndef MD
#  define MD MD5
#endif

#if MD == 2
#  include "md2.h"
#elif MD == 4
#  include "md4.h"
#elif MD == 5
#  include "md5.h"
#endif

#if MD == 2
#  define MD_CTX MD2_CTX
#  define MDInit MD2Init
#  define MDUpdate MD2Update
#  define MDFinal MD2Final
#elif MD == 4
#  define MD_CTX MD4_CTX
#  define MDInit MD4Init
#  define MDUpdate MD4Update
#  define MDFinal MD4Final
#elif MD == 5
#  define MD_CTX MD5_CTX
#  define MDInit MD5Init
#  define MDUpdate MD5Update
#  define MDFinal MD5Final
#endif

#define BLOCKSIZE ( 64 * 1024 )
#define V1HEADER "Pseudo CLOOP blockfile index Version 1.0"

typedef unsigned long long __u64;
#define __cpu_to_be64(x) \
        ((__u64)( \
                (__u64)(((__u64)(x) & (__u64)0x00000000000000ffULL) << 56) | \
                (__u64)(((__u64)(x) & (__u64)0x000000000000ff00ULL) << 40) | \
                (__u64)(((__u64)(x) & (__u64)0x0000000000ff0000ULL) << 24) | \
                (__u64)(((__u64)(x) & (__u64)0x00000000ff000000ULL) <<  8) | \
                (__u64)(((__u64)(x) & (__u64)0x000000ff00000000ULL) >>  8) | \
                (__u64)(((__u64)(x) & (__u64)0x0000ff0000000000ULL) >> 24) | \
                (__u64)(((__u64)(x) & (__u64)0x00ff000000000000ULL) >> 40) | \
                (__u64)(((__u64)(x) & (__u64)0xff00000000000000ULL) >> 56) ))

#define __be64_to_cpu(x) __cpu_to_be64(x)

typedef struct cb_list_stub {
    unsigned int size;
    unsigned char *md5sum;
    struct cb_list_stub *next;
} cb_list;

const unsigned char *mkdigest(const unsigned char *);
int makeblocks(char*, char *, char *, int);
void makeheader(FILE *, cb_list *, unsigned int, unsigned int);

int main(int argc, char* argv[])
{
    unsigned char *inputfile;
    unsigned char *indexname = "index.idx";
    unsigned char *targetdir;
    unsigned int blocksize;

    if (argc < 4) {
        fprintf(stderr, "Usage: %s inputfile targetdir blocksize\n", argv[0]);
        return 1;
    }
    if ((inputfile = strdup(argv[1])) == NULL) {
        perror("inputfile");
        return 1;
    }
    if ((targetdir = strdup(argv[2])) == NULL) {
        perror("targetdir");
        return 1;
    }
    if ((blocksize = atoi(argv[3])) == 0 || blocksize % 512 != 0) {
        fprintf(stderr, "blocksize must be a multiple of 512.\n");
	return 1;
    }

    return makeblocks(inputfile, indexname, targetdir, blocksize);
}

int makeblocks(char *inputfile, char *indexfile, char *targetdir, int blocksize)
{
    FILE *infp;
    FILE *outfp;
    FILE *indexfp;
    struct stat dirstat;
    unsigned char *buf;
    unsigned char *comprbuf;
    unsigned char outfilename[36];
    int readcount;
    unsigned int comprlen;
    int count;
    int err;
    int z_err;
    int i;
    char subdir[3];
    int maxlen = blocksize + blocksize/1000 + 12;
    cb_list *cbl;
    cb_list **cbp;
    cbp = &cbl;

    if (stat(targetdir, &dirstat) != 0) {
        switch (errno) {
            case ENOENT:
                if (mkdir(targetdir, 0755) == 0)
                    break;
            default:
                perror("stat targetdir");
                return 1;
        }
    }

    if ((infp = fopen(inputfile, "r")) == NULL) {
        perror("opening inputfile");
        return 1;
    }

    if (chdir(targetdir) != 0) {
        perror("chdir to targetdir");
        return 1;
    }

    for (i = 0; i < 0x100 ; i++) {
        sprintf(subdir, "%02x", i);
        mkdir(subdir, 0755);
    }
        
    if ((indexfp = fopen(indexfile, "w")) == NULL) {
        perror("opening indexfile");
        return 1;
    }

    buf = (unsigned char *)malloc(sizeof(unsigned char) * blocksize);
    if (buf == NULL) {
        perror("getting output buffer");
        return 2;
    }
    comprbuf = (unsigned char *)malloc(sizeof(unsigned char) * maxlen);
    if (comprbuf == NULL) {
        perror("getting output buffer");
        return 2;
    }
    count = 0;
    do {
        comprlen = maxlen;
        memset(buf, 0, blocksize);
        memset(comprbuf, 0, maxlen);
        readcount = 0;
        readcount = fread(buf, sizeof(unsigned char), blocksize, infp);
        if (ferror(infp)) {
            perror("reading input");
            return 2;
        }
        if ((z_err = compress2(comprbuf, &comprlen, buf, blocksize, Z_BEST_COMPRESSION)) != Z_OK) {
            fprintf(stderr,"Something wrong with compression %d\n", z_err);
            return 2;
        }
        if ((*cbp = malloc(sizeof(cb_list))) == NULL) {
            perror("alloc header");
            return 4;
        }
        (*cbp)->size = comprlen;
        (*cbp)->md5sum = MDString(buf, blocksize);
        sprintf(outfilename, "%02x/", (*cbp)->md5sum[0]);
        for (i = 0; i < 16; i++) {
            sprintf(outfilename + i * 2 + 3, "%02x", (*cbp)->md5sum[i]);
        }
        (*cbp)->next = NULL;
        cbp=&((*cbp)->next);

        if ((outfp = fopen(outfilename, "w")) == NULL) {
            perror("opening output block");
            return 4;
        }
        if (fwrite(comprbuf, sizeof(unsigned char), comprlen, outfp) != comprlen) {
            perror("writing output block");
            return 4;
        }
        fclose(outfp);
        fprintf(stderr, "%08d:%s\n", count, outfilename);
        count++;
    } while(!feof(infp));
    makeheader(indexfp, cbl, blocksize, count);
    fclose(infp);
    fclose(indexfp);
    return 0;
}

void makeheader(FILE *fp, cb_list *cbl, unsigned int blocksize, unsigned int blockcount)
{
    unsigned int blocksize_nl = htonl(blocksize);
    unsigned int blockcount_nl = htonl(blockcount);
    unsigned int header_size;
    loff_t bytes_so_far;
    loff_t tmp;
    unsigned char padding[16];
    bytes_so_far = 0;

    header_size = 16 + sizeof(unsigned int) * 2 + (sizeof(loff_t) + 16) * blockcount;

    fwrite(MDString(V1HEADER, sizeof(V1HEADER)), 1, 16, fp);
    fwrite(&blocksize_nl, sizeof(unsigned int), 1, fp);
    fwrite(&blockcount_nl, sizeof(unsigned int), 1, fp);

    while (cbl != NULL) {
        tmp = __cpu_to_be64(bytes_so_far);
        fwrite(&tmp, sizeof(tmp), 1, fp);
        fwrite(cbl->md5sum, sizeof(unsigned char), 16, fp);
        bytes_so_far += cbl->size;
        cbl = cbl->next;
    }
    tmp = __cpu_to_be64(bytes_so_far);
    fwrite(&tmp, sizeof(tmp), 1, fp);
    fwrite(padding, sizeof(unsigned char), 16, fp);
    return;
}
