/*
    fs_wrapper: Filesystem wrapper
    Copyright (C) 2005 YAGI, Toshiki <yagi-toshiki@aist.go.jp>

    This program can be distributed under the terms of the GNU GPL.
*/

#include <curl/curl.h>
#include <zlib.h>

#define PROGRAM_NAME "fs_wrapper"
#define PROGRAM_VERSION "0.37"
#define USER_AGENT PROGRAM_NAME "/" PROGRAM_VERSION

#ifdef linux
/* For pread()/pwrite() */
#define _XOPEN_SOURCE 500
#endif

#define CLOOP_HEADROOM 128
#define CLOOP_PREAMBLE "#!/bin/sh\n" "#V2.0 Format\n" "insmod cloop.o file=$0 && mount -r -t iso9660 /dev/cloop $1\n" "exit $?\n"

#include <fuse.h>
#include <stdio.h>
#define __USE_GNU
#include <string.h>
#undef __USE_GNU
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <sys/statvfs.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/timeb.h>
#include <arpa/inet.h>
#include <pthread.h>

#define IS_NOT_IN_QUEUE 1
#define IS_IN_QUEUE 0

#define ERR -1
#define OK 0
#define NOT_FOUND 1
#define CORRUPT 2

#ifdef DEBUG 
#define D_LOG fprintf
#else
#define D_LOG(...) 
#endif

struct MemoryStruct {
    char *memory;
    size_t size;
};

struct worker_arg_t {
    pthread_mutex_t *mutex;
    struct task_queue_stub *task;
};

size_t
WriteMemoryCallback(void *ptr, size_t size, size_t nmemb, void *data)
{
    size_t realsize = size * nmemb;
    struct MemoryStruct *mem = (struct MemoryStruct *)data;

    mem->memory = (char *)realloc(mem->memory, mem->size + realsize + 1);
    if (mem->memory) {
        memcpy(&(mem->memory[mem->size]), ptr, realsize);
        mem->size += realsize;
        mem->memory[mem->size] = 0;
    }

    return realsize;
}

typedef unsigned long long __u64;
#define __cpu_to_be64(x) \
        ((__u64)( \
                (__u64)(((__u64)(x) & (__u64)0x00000000000000ffULL) << 56) | \
                (__u64)(((__u64)(x) & (__u64)0x000000000000ff00ULL) << 40) | \
                (__u64)(((__u64)(x) & (__u64)0x0000000000ff0000ULL) << 24) | \
                (__u64)(((__u64)(x) & (__u64)0x00000000ff000000ULL) <<  8) | \
                (__u64)(((__u64)(x) & (__u64)0x000000ff00000000ULL) >>  8) | \
                (__u64)(((__u64)(x) & (__u64)0x0000ff0000000000ULL) >> 24) | \
                (__u64)(((__u64)(x) & (__u64)0x00ff000000000000ULL) >> 40) | \
                (__u64)(((__u64)(x) & (__u64)0xff00000000000000ULL) >> 56) ))

#define __be64_to_cpu(x) __cpu_to_be64(x)

CURL *http_handler;


typedef struct command_stack_stub {
    char *content;
    struct command_stack_stub *next;
} command_stack;

typedef struct task_list_stub {
    char *content;
    struct task_list_stub *next;
} task_list;

typedef struct task_queue_stub {
    struct task_list_stub *task_queue_top;
    struct task_list_stub *task_queue_bottom;
} task_queue;

task_queue *dlahead_queue;

typedef struct link_list_stub {
    struct link_list_stub *prev, *next;
    off_t offset;
    off_t size;
    int count;
    char *content;
} link_list;

typedef struct cloop_header_stub {
    char preamble[CLOOP_HEADROOM];
    unsigned long block_size;
    unsigned long num_blocks;
    loff_t offsets[0];
} cloop_header;

typedef struct index_body_stub {
    loff_t offset;
    unsigned char blockname[16];
} index_body;

command_stack *stack_top = NULL;
link_list **list_array = NULL;
link_list *list_top = NULL;
unsigned int blocksize = 0;
unsigned int blockcount = 0;
cloop_header *file_header;
int is_cache_all = 0;
pthread_mutex_t wrapper_read_mutex;

static char *target_path;
static char *cache_path;
static char *source_path;

static char      *block_reader      (char *, size_t *);
static char      *http_block_reader (char *, size_t *);
static void       block_eraser      (long);
static void       dlahead           (int);
static void       push_queue        (struct task_queue_stub *, char *);
static void       dlahead_invoker   (struct MemoryStruct *, int);
static void      *dlahead_worker    (void *);
static int        check_block       (char *filename);
static off_t      get_file_size     (void);
static link_list *search_list       (off_t, int, int);
static void       bintohex          (char *, unsigned char *, int);

static int search_tasklist(char *string)
{
    task_list *target;
    if (dlahead_queue == NULL) return IS_NOT_IN_QUEUE;
    target = dlahead_queue->task_queue_bottom;
    while (target != NULL) {
        if (strcmp(string, target->content) == 0)
            return IS_IN_QUEUE;
        target = target->next;
    }
    return IS_NOT_IN_QUEUE;
}

static int push(char *target)
{
    command_stack *temp;
    if ((temp = (command_stack *)malloc(sizeof(command_stack))) == NULL) {
        return -1;
    }
    if ((temp->content = strdup(target)) == NULL) {
        return -1;
    }
    temp->next = stack_top;
    stack_top = temp;
    return 0;
}

static char *pop(void)
{
    char *buf;
    command_stack *temp;
    if (stack_top->content == NULL)
        return NULL;
    if ((buf = strdup(stack_top->content)) == NULL) {
        return NULL;
    }
    temp = stack_top;
    stack_top = stack_top->next;
    free(temp->content);
    free(temp);
    return buf;
}

static int xmp_getattr(const char *path, struct stat *stbuf)
{
    int res = 0;
    off_t filesize = 0;

    memset(stbuf, 0, sizeof(struct stat));
    if (strcmp(path, "/") == 0) {
        stbuf->st_mode = S_IFDIR | 0555;
        stbuf->st_nlink = 2;
    } else if (strcmp(path, target_path) == 0) {
        filesize = get_file_size();
        stbuf->st_mode = S_IFREG | 0444;
        stbuf->st_nlink = 1;
        stbuf->st_size = filesize;
    } else {
        res = -ENOENT;
    }

    return res;
}

static void dlahead(int connection)
{
    CURL *headhandler;
    struct MemoryStruct chunk;
    char *url;
    CURLcode err;
    chunk.memory = NULL;
    chunk.size = 0;

    if (connection <= 0) return;
    url = malloc(strlen(source_path) + strlen("dlahead") + 1);
    memset(url, 0, strlen(source_path) + strlen("dlahead") + 1);
    memcpy(url, source_path, strlen(source_path));
    if (strcat(url, "dlahead") == NULL) {
        return;
    }

    headhandler = curl_easy_init();
    curl_easy_setopt(headhandler, CURLOPT_USERAGENT, USER_AGENT);
    curl_easy_setopt(headhandler, CURLOPT_URL, url);
    curl_easy_setopt(headhandler, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    curl_easy_setopt(headhandler, CURLOPT_WRITEDATA, (void *)&chunk);
    curl_easy_setopt(headhandler, CURLOPT_FAILONERROR, 1);
    curl_easy_setopt(headhandler, CURLOPT_FOLLOWLOCATION, 1);
    curl_easy_setopt(headhandler, CURLOPT_MAXREDIRS, 10);
    curl_easy_setopt(headhandler, CURLOPT_TIMEOUT, 30);
    curl_easy_setopt(headhandler, CURLOPT_NOSIGNAL, 1);
    err = curl_easy_perform(headhandler);
    curl_easy_cleanup(headhandler);
    if (err == CURLE_OK)
        dlahead_invoker(&chunk, connection);
    return;
}

task_queue *make_taskqueue(struct MemoryStruct *chunk)
{
    char *target;
    char *token;
    task_queue *queue;

    target = chunk->memory;
    queue = malloc(sizeof(task_queue));
    queue->task_queue_top = NULL;
    queue->task_queue_bottom = NULL;

    while ((token = strtok(target, "\n")) != NULL) {
        push_queue(queue, token);
        target = NULL;
    }
    return queue;
}

static void push_queue(task_queue *queue, char *token)
{
    task_list *new_item;

    new_item = malloc(sizeof(task_list));
    if (new_item == NULL) {
        fprintf(stderr, "Can't allocate memory.\n");
        exit(1);
    }

    /* creates new task_list item */
    new_item->content = token;
    new_item->next = NULL;

    /* if queue is empty */
    if (queue->task_queue_top == NULL) {
        D_LOG(stderr, "queue is empty.\n");
        queue->task_queue_top = new_item;
        queue->task_queue_bottom = new_item;
    } else {
        D_LOG(stderr, "adding item to queue.\n");
        queue->task_queue_top->next = new_item;
        queue->task_queue_top = new_item;
    }
    return;
}

static task_list *get_queue(task_queue *queue)
{
    task_list *retval;
    if (queue->task_queue_bottom == NULL) {
        return NULL;
    }
    retval = queue->task_queue_bottom;
    queue->task_queue_bottom = queue->task_queue_bottom->next;
    if (queue->task_queue_bottom == NULL) queue->task_queue_top = NULL;
    return retval;
}
    
static task_list *get_one_task(struct worker_arg_t *arg)
{
    task_list *retval;
    pthread_mutex_lock(arg->mutex);
    retval = get_queue(arg->task);
    pthread_mutex_unlock(arg->mutex);
    return retval;
}

static void dlahead_invoker(struct MemoryStruct *chunk, int nthread)
{
    pthread_t *tid;
    int i;
    int error;

    struct worker_arg_t *worker_arg;

    dlahead_queue = make_taskqueue(chunk);
    worker_arg = malloc(sizeof(struct worker_arg_t));
    worker_arg->task = dlahead_queue;
    worker_arg->mutex = malloc(sizeof(pthread_mutex_t));

    pthread_mutex_init(worker_arg->mutex, NULL);
    tid = malloc(sizeof(pthread_t) * nthread);

    for (i = 0; i < nthread; i++) {
        error = pthread_create(&tid[i], NULL, dlahead_worker, worker_arg);
        if (error != 0)
            fprintf(stderr, "Couldn't run thread number %d, errno %d\n", i, error);
        else
            fprintf(stderr, "Thread %d working.\n", i);
        pthread_detach(tid[i]);
    }
    return;
}

static int check_block (char *filename)
{
    FILE *fp;
    unsigned char *comprbuf;
    unsigned char *uncomprbuf;
    unsigned long comprlen, uncomprlen;
    int z_err;
    int err;
    char tmp_filename[1024];
    
    sprintf (tmp_filename, "%s/%s", cache_path, filename);

    if ((fp = fopen(tmp_filename, "r")) == NULL) {
        return NOT_FOUND;
    }
    
    if ((comprbuf = malloc(sizeof(unsigned char) * (blocksize + blocksize / 1000 + 12 + 4))) == NULL)
        return ERR;
    if ((uncomprbuf = malloc(sizeof(unsigned char) * blocksize)) == NULL) {
        free(comprbuf);
        return ERR;
    }

    uncomprlen = blocksize;

    comprlen = fread(comprbuf, sizeof(unsigned char), blocksize + blocksize / 1000 + 12 + 4, fp);
    fclose(fp);
    if (comprlen < 0) {
        free(uncomprbuf);
        free(comprbuf);
        return ERR;
    }
    z_err = uncompress(uncomprbuf, &uncomprlen, comprbuf, comprlen);

    if (z_err == Z_OK)
        err = OK;
    else
        err = CORRUPT;

    free(comprbuf);
    free(uncomprbuf);
    return err;

}

static void *dlahead_worker (void *arg)
{
    CURL *curl;
    CURLcode err;
    char *url;
    task_list *work_list;
    struct worker_arg_t *tasks;
    struct MemoryStruct chunk;
    FILE *fp;

    url = NULL;
    tasks = (struct worker_arg_t *)arg;

    chunk.memory = NULL;
    chunk.size = 0;

    curl = curl_easy_init();
    curl_easy_setopt(curl, CURLOPT_USERAGENT, USER_AGENT);
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&chunk);
    curl_easy_setopt(curl, CURLOPT_FAILONERROR, 1);
    curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
    curl_easy_setopt(curl, CURLOPT_MAXREDIRS, 10);
    curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 30);
    curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);

    while ((work_list = get_one_task(tasks)) != NULL) {
        if (check_block(work_list->content) != OK) {
	    char tmp_filename[1024];
            url = malloc(strlen(work_list->content) + strlen(source_path) + 1);
            if (url == NULL) return NULL;
            memset(url, 0, strlen(work_list->content) + strlen(source_path) + 1);
            memcpy(url, source_path, strlen(source_path));
            strncat(url, work_list->content, strlen(work_list->content));
            curl_easy_setopt(curl, CURLOPT_URL, url);
            err = curl_easy_perform(curl);
            while (err != CURLE_OK) {
                curl_easy_setopt(curl, CURLOPT_FRESH_CONNECT, 1);
                err = curl_easy_perform(curl);
            }
            sprintf (tmp_filename, "%s/%s", cache_path, work_list->content);
            if ((fp = fopen(tmp_filename, "w")) == NULL) {
                exit(1);
	    }
            if (fwrite(chunk.memory, 1, chunk.size, fp) != chunk.size)
                exit(1);
            free(url);
            url = NULL;
            free(work_list);
            free(chunk.memory);
            fclose(fp);
            chunk.memory = NULL;
            chunk.size = 0;
        }
    }
    free(url);
    curl_easy_cleanup(curl);
    return NULL;
}

static off_t get_file_size(void)
{
    return list_array[blockcount]->offset + sizeof(cloop_header) + sizeof(off_t) * (blockcount + 1);
}

static int xmp_readlink(const char *path, char *buf, size_t size)
{
    int res;

    res = readlink(path, buf, size - 1);
    if(res == -1)
        return -errno;

    buf[res] = '\0';
    return 0;
}


static int xmp_getdir(const char *path, fuse_dirh_t h, fuse_dirfil_t filler)
{
    if (strcmp(path, "/") != 0) {
        return -ENOENT;
    }

    filler(h, ".", 0);
    filler(h, "..", 0);
    filler(h, target_path + 1, 0);

    return 0;
}

#if 0
static int xmp_mknod(const char *path, mode_t mode, dev_t rdev)
{
    return -EROFS;
}

static int xmp_mkdir(const char *path, mode_t mode)
{
    return -EROFS;
}

static int xmp_unlink(const char *path)
{
    return -EROFS;
}

static int xmp_rmdir(const char *path)
{
    return -EROFS;
}

static int xmp_symlink(const char *from, const char *to)
{
    return -EROFS;
}

static int xmp_rename(const char *from, const char *to)
{
    return -EROFS;
}

static int xmp_link(const char *from, const char *to)
{
    return -EROFS;
}

static int xmp_chmod(const char *path, mode_t mode)
{
    return -EROFS;
}

static int xmp_chown(const char *path, uid_t uid, gid_t gid)
{
    return -EROFS;
}

static int xmp_truncate(const char *path, off_t size)
{
    return -EROFS;
}
#endif
static int xmp_utime(const char *path, struct utimbuf *buf)
{
    int res;
    
    res = utime(path, buf);
    if(res == -1)
        return -errno;

    return 0;
}


static int xmp_open(const char *path, int flags)
{
    if (strcmp(path, target_path) != 0) {
        return -ENOENT;
    }

    if((flags & 3) != O_RDONLY) {
        return -EROFS;
    }

    return 0;
}

static int xmp_read(const char *path, char *buf, size_t size, off_t offset)
{
    int res;
    size_t size_remain;
    off_t offset_remain;
    off_t header_size;
    int read_length;
    int readcount;
    link_list *source_file;
    char *tmp;
    char *block_buf;
    size_t block_size;
    char source_filename[32 + 3 + 1];
#if 0
    int source_path_len;
    struct timeb tmptime;
#endif

    res = 0;
    pthread_mutex_lock(&wrapper_read_mutex);
    readcount = 0;
    tmp = buf;

    if (offset < sizeof(cloop_header) + sizeof(off_t) * (blockcount + 1)) {
        /* returns cloop header */
	if(offset + size < sizeof(cloop_header) + sizeof(off_t) * (blockcount + 1)) {
	    memcpy(tmp, (unsigned char *)file_header + offset, size);
	    readcount = size;
	    goto READ_DONE;
	}
	header_size =  (sizeof(cloop_header) + sizeof(off_t) * (blockcount + 1)) - offset;
 	D_LOG(stderr, "DEBUG: size of remaining header = %llu\n", header_size);
	memcpy(tmp, (unsigned char *)file_header + offset,header_size);
	readcount += header_size;
	offset += header_size;
	tmp = buf + header_size;
	size -= header_size;
    }

    offset_remain = offset - (sizeof(cloop_header) + sizeof(off_t) * (blockcount + 1));
    D_LOG(stderr, "DEBUG: offset = %llu, offset_remain = %llu\n", offset, offset_remain);
    D_LOG(stderr, "DEBUG: guessed offset in body = %lu\n", sizeof(cloop_header) + sizeof(off_t) * blockcount);
    D_LOG(stderr, "DEBUG: data block to read: %d\n", size);
    source_file = search_list(offset_remain, 0, blockcount);
    size_remain = size;

    do {
        bintohex(source_filename, (unsigned char *)source_file->content, 16);
#if 0
	ftime(&tmptime);
  	fprintf(stderr, "%d.%03d: #%08d(%s) :", tmptime.time, tmptime.millitm, source_file->count, source_filename);
#endif
        block_buf = block_reader(source_filename, &block_size);
	if (block_buf == NULL) {
            return -ESTALE;
        }
	
	if (size_remain + offset_remain > source_file->offset + source_file->size) {
	    read_length = source_file->size - (offset_remain - source_file->offset);
	} else
	    read_length = size_remain;
        D_LOG(stderr, "DEBUG: source_file->size = %lld, offset_remain = %lld, source_file->offset = %lld\n", source_file->size, offset_remain, source_file->offset);
 	D_LOG(stderr, "DEBUG: read_length = %d\n", read_length);
        /* res = pread(fd, tmp, read_length, offset_remain - source_file->offset); */
	memcpy(tmp, block_buf + (offset_remain - source_file->offset), read_length);
        D_LOG(stderr,"DEBUG: %d bytes to read, %d read done.\n", read_length, res);
	free(block_buf);
	source_file = source_file->next;
	offset_remain = source_file->offset;
	tmp += read_length;
	size_remain -= read_length;
	readcount += read_length;
	if (source_file == NULL || source_file->content == NULL) break;
    } while(size_remain > 0);

    READ_DONE:
    pthread_mutex_unlock(&wrapper_read_mutex);
    D_LOG(stderr, "DEBUG: readcount = %d\n", readcount);
    return readcount;
}

static char *block_reader(char *blockname, size_t *size)
{
    FILE *fp = NULL;
    char *string = NULL;
    struct stat filestat;
    size_t ret_fwrite;
    char filename[1024];

    while (search_tasklist(blockname) == IS_IN_QUEUE) {
        usleep(10000);
    }

    sprintf (filename, "%s/%s", cache_path, blockname);
    if ((strcmp(blockname, "index.idx") == 0) && (is_cache_all == 0)) {
        D_LOG(stderr, "force read.\n");
        string = http_block_reader(blockname, size);
        if (string == NULL) {
            fprintf(stderr, "read failed or read-only cache.\n");
            goto READ_FILE;
        }
        fp = fopen(filename, "w");
	if (fp == NULL) {
	    free(string);
	    return NULL;
	}
        if (fwrite(string, sizeof(char), *size, fp) < *size) {
            fclose(fp);
            free(string);
            unlink(filename);
            return NULL;
        }
        fclose(fp);
	return string;
    } else if (stat(filename, &filestat) != 0) {
  	D_LOG(stderr, "(%s) :missed\n", blockname);
        string = http_block_reader(blockname, size);
        if (string == NULL) {
            fprintf(stderr, "HTTP read failed.\n");
            return NULL;
        }
	fp = fopen(filename, "w");
	if (fp == NULL) {
	    fprintf(stderr, "failed to open, errono is : %s, %d\n", filename,errno);
	    free(string);
	    return NULL;
	}
	ret_fwrite = fwrite(string, sizeof(char), *size, fp);
	if (ret_fwrite < *size) {
            fprintf(stderr, "Not enough writes. aborted.\n");
            fclose(fp);
            free(string);
            unlink(filename);
            return NULL;
        }
	fclose(fp);
        return string;
    } else {
        D_LOG(stderr, "%s: hits.\n", blockname);
    }

READ_FILE:
    *size = filestat.st_size;
    fp = fopen(filename, "r");
    string = malloc(sizeof(char) * *size);
    fread(string, sizeof(char), *size, fp);
    fclose(fp);
    return string;
}

static char *http_block_reader(char *blockname, size_t *size)
{
    char *url;
    struct MemoryStruct chunk;
    struct statfs fs_stat;
    double fs_block_free;
    CURLcode err = 0; 

    chunk.memory = NULL;
    chunk.size = 0;

    statfs(cache_path, &fs_stat);
    fs_block_free = (double)fs_stat.f_bfree / (double)fs_stat.f_blocks;

    if (fs_block_free < 0.20) {
        block_eraser((long)(fs_stat.f_blocks / 10 * fs_stat.f_bsize));
    }

    url = malloc(sizeof(char) * (strlen(source_path) + strlen(blockname) + 1));
    memset(url, 0, strlen(source_path) + strlen(blockname) + 1);
    memcpy(url, source_path, strlen(source_path));
    if (strcat(url, blockname) == NULL) {
        fprintf(stderr, "URL invalid\n");
        return NULL;
    }
    if (url) {
        curl_easy_setopt(http_handler, CURLOPT_URL, url);
        curl_easy_setopt(http_handler, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
        curl_easy_setopt(http_handler, CURLOPT_WRITEDATA, (void *)&chunk);
        err = curl_easy_perform(http_handler);

        while (err != CURLE_OK) {
            fprintf(stderr, "Can't get chunk. retry.\n");
            curl_easy_setopt(http_handler, CURLOPT_FRESH_CONNECT, 1);
            err = curl_easy_perform(http_handler);
        }
        *size = chunk.size;
    } else {
        fprintf(stderr, "Type the URL you want to accces on the command line\n");
    }

    if (err == CURLE_OK) {
        push(blockname);
    } else {
        fprintf(stderr, "Can't get chunk. retry.\n");
        sleep(1);
        return http_block_reader(blockname, size);
    }
    /* Clean up the request */
    free(url);
    return chunk.memory;
}

static void block_eraser(long bytes)
{
    char *target;
    long bytes_deleted = 0;
    struct stat filebuf;
    while ((target = pop()) != 0) {
	char filename[1024];
        sprintf (filename, "%s/%s", cache_path, target);
        if (stat(filename, &filebuf)) {
            return;
        }
        bytes_deleted += filebuf.st_size;
        unlink(filename);
        free(target);
        target = NULL;
        if (bytes_deleted >= bytes)
            break;
    }
    return;
}

#if 0
static int xmp_write(const char *path, const char *buf, size_t size,
                     off_t offset)
{
    return -EROFS;
}
#endif
static int xmp_release(const char *path, int flags)
{
    /* Just a stub.  This method is optional and can safely be left
       unimplemented */

    (void) path;
    (void) flags;
    return 0;
}
#if 0
static int xmp_fsync(const char *path, int isdatasync)
{
    /* Just a stub.  This method is optional and can safely be left
       unimplemented */

    (void) path;
    (void) isdatasync;
    return -EROFS;
}
#endif
static struct fuse_operations xmp_oper = {
    .getattr	= xmp_getattr,
    .readlink	= xmp_readlink,
    .getdir	= xmp_getdir,
    .utime	= xmp_utime,
    .open	= xmp_open,
    .read	= xmp_read,
    .release	= xmp_release,
};

static void bintohex(char *dest, unsigned char *src, int size)
{
    int count;

    memset(dest, 0, size * 2 + 1);

    sprintf(dest, "%02x/", src[0]);
    
    for (count = 0; count < size; count++) {
        sprintf(&dest[count * 2 + 3], "%02x", src[count]);
    }
    return;
}

static void create_dir()
{
    FILE *fp;
    char *index_buf;
    size_t index_size;
    unsigned char *header_md5;
    unsigned long header_tmp_nl;
    index_body blockinfo[2];
    int count, i;
    char tmp_subdir[1024];
    char tmp_filename[1024];
    char *ahead_connection_env;
    int ahead_connection;

    ahead_connection = 4;
    if((ahead_connection_env = getenv("MAXCONNECTION")) != NULL) {
        ahead_connection = atoi(ahead_connection_env);
    }

    for (i = 0; i < 0x100; i++) {
        sprintf(tmp_subdir, "%s/%02x", cache_path, i);
        if (mkdir(tmp_subdir, 0777) != 0) {
            if (errno != EEXIST) {
                perror("mkdir");
                exit(1);
            }
        }
    }

    dlahead(ahead_connection);
    sprintf(tmp_filename, "%s/index.idx", cache_path);
    D_LOG(stderr, "DEBUG: index ");
    if (is_cache_all == 0) {
        index_buf = block_reader("index.idx", &index_size);
        if ((fp = fopen(tmp_filename, "w")) == NULL) {
            perror("writing index file");
            exit(1);
        }
        fwrite(index_buf, sizeof(char), index_size, fp);
        fclose(fp);
    }
    
    if ((fp = fopen(tmp_filename, "r")) == NULL) {
        perror("reading index file");
        exit(1);
    }

    if ((header_md5 = (unsigned char *)malloc(sizeof(unsigned char) * 16)) == NULL) {
        perror("allocate header");
	exit(1);
    }
    if (fread(header_md5, sizeof(unsigned char), 16, fp) < 16) {
        perror("reading header");
	exit(1);
    }
    /* TODO: check header MD5sum */

    /* reading header: blocksize */
    if (fread(&header_tmp_nl, sizeof(unsigned int), 1, fp) == 0) {
        perror("reading header blocksize");
	exit(1);
    }
    blocksize = ntohl(header_tmp_nl);

    /* reading header: blockcount */
    if (fread(&header_tmp_nl, sizeof(unsigned int), 1, fp) == 0) {
        perror("reading header blockcount");
	exit(1);
    }
    blockcount = ntohl(header_tmp_nl);
    list_array = (link_list **)calloc(blockcount + 1, sizeof(link_list *));
    file_header = (cloop_header *)malloc(sizeof(cloop_header) + sizeof(off_t) * (blockcount + 1));
    memset(file_header->preamble, 0, sizeof(file_header->preamble));
    memcpy(file_header->preamble, CLOOP_PREAMBLE, sizeof(CLOOP_PREAMBLE));
    file_header->block_size = htonl(blocksize);
    file_header->num_blocks = htonl(blockcount);

    for (count = 0; count < blockcount; count++) {
        list_array[count] = (link_list *)malloc(sizeof(link_list));
        fread(&blockinfo, sizeof(index_body), 2, fp);
	fseek(fp, -sizeof(index_body), SEEK_CUR);
	list_array[count]->offset = __be64_to_cpu(blockinfo[0].offset);
	list_array[count]->size = __be64_to_cpu(blockinfo[1].offset);
	list_array[count]->size -= list_array[count]->offset;
	list_array[count]->count = count;
	D_LOG(stderr, "DEBUG: data block %d : size=%lld, offset=%lld\n", count, list_array[count]->offset, list_array[count]->size);
	list_array[count]->content = (char *)malloc(sizeof(unsigned char) * 16);
	file_header->offsets[count] = __cpu_to_be64(__be64_to_cpu(blockinfo[0].offset) + sizeof(cloop_header) + sizeof(off_t) * (blockcount + 1));
	memcpy(list_array[count]->content, blockinfo[0].blockname, sizeof(unsigned char) * 16);
	if (count > 0) {
	    list_array[count-1]->next = list_array[count];
	    list_array[count]->prev = list_array[count-1];
	}
    }
    D_LOG(stderr, "DEBUG: blockinfo[final].offset = %lld\n", __be64_to_cpu(blockinfo[1].offset));
    file_header->offsets[blockcount] = __cpu_to_be64(__be64_to_cpu(blockinfo[1].offset) + sizeof(cloop_header) + sizeof(off_t) * (blockcount + 1));
    list_array[blockcount] = (link_list *)malloc(sizeof(link_list));
    list_array[blockcount]->offset = __be64_to_cpu(blockinfo[1].offset);
    list_array[blockcount]->size = 0L;
    list_array[blockcount - 1]->next = list_array[blockcount];
    list_array[blockcount]->next = NULL;
    list_array[blockcount]->prev = list_array[blockcount - 1];
    list_array[blockcount]->count = blockcount;

    fclose(fp);

    return;
}

link_list *search_list(off_t key, int head, int tail)
{
    if (list_array[(head + tail) / 2]->offset <= key || head == tail) {
        if(list_array[(head + tail) / 2]->next->offset > key) {
 	    D_LOG(stderr, "DEBUG: Getting block #%d :", (head + tail) / 2);
	    return list_array[(head + tail) / 2];
	} else {
	    return search_list(key, (head + tail) / 2 + 1, tail);
	}
    } else {
        return search_list(key, head, (head + tail) / 2 - 1);
    }
    return NULL;
}

int main(int argc, char *argv[])
{
    char *fuse_argv[3];
    char *p;
    if(getenv("FROMCACHE") != NULL)
        is_cache_all = 1;

    pthread_mutex_init(&wrapper_read_mutex, NULL);
    
    p = strrchr(argv[1], '/');
    /* target_path is needed '/' in the head */
    if (p != NULL) {
        target_path = strdup(p);
    } else {	
        target_path = strdup(argv[1]);
    }
    cache_path = strdup(argv[2]);
    source_path = strdup(argv[3]);
    curl_global_init(CURL_GLOBAL_ALL);
    http_handler = curl_easy_init();
    if (!http_handler) {
        fprintf(stderr, "HTTP handler cannot get.\n");
        return 1;
    }
    curl_easy_setopt(http_handler, CURLOPT_USERAGENT, USER_AGENT);
    curl_easy_setopt(http_handler, CURLOPT_FAILONERROR, 1);
    curl_easy_setopt(http_handler, CURLOPT_FOLLOWLOCATION, 1);
    curl_easy_setopt(http_handler, CURLOPT_MAXREDIRS, 10);
    curl_easy_setopt(http_handler, CURLOPT_CONNECTTIMEOUT, 30);
    curl_easy_setopt(http_handler, CURLOPT_NOSIGNAL, 1);

    create_dir();

    creat("/tmp/.fuse_ready", 0444);
    if ((stack_top = (command_stack *)malloc(sizeof(command_stack))) == NULL) {
         return 1;
    }
    stack_top->content = NULL;
    stack_top->next = NULL;

    fuse_argv[0] = strdup(PROGRAM_NAME);
   
    p = strrchr(argv[1], '/');
    if (p != NULL) {
        fuse_argv[1] = strndup(argv[1], (size_t)(p - argv[1]));
    } else {
        fuse_argv[1] = strdup(argv[1]);
    }

    fuse_argv[2] = strdup("-f");
    fuse_main(3, fuse_argv, &xmp_oper);

    free(fuse_argv[0]);
    free(fuse_argv[1]);
    free(fuse_argv[2]);
    return 0;
}
