/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com.          */
/*                                                                        */
/* NXAGENT, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

/*

Copyright 1993 by Davor Matic

Permission to use, copy, modify, distribute, and sell this software
and its documentation for any purpose is hereby granted without fee,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation.  Davor Matic makes no representations about
the suitability of this software for any purpose.  It is provided "as
is" without express or implied warranty.

*/

#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "X.h"
#include "Xproto.h"
#include "screenint.h"
#include "input.h"
#include "misc.h"
#include "scrnintstr.h"
#include "servermd.h"
#include "opaque.h"

/*
 * NX specific includes and definitions.
 */

#include "Agent.h"
#include "Display.h"
#include "Args.h"
#include "Control.h"
#include "Options.h"
#include "Binder.h"
#ifdef RENDER
#include "Render.h"
#endif

/*
 * From X11/ImUtil.c.
 */

extern int _XGetBitsPerPixel(Display *dpy, int depth);

/*
 * FIXME: Add comments and a comprehensive
 * organization for sparse variable herein.
 */

#ifdef NXAGENT_PACKEDIMAGES

#include NXAGENT_NXPACK_INCLUDE
#include NXAGENT_NXLIB_INCLUDE

int nxagentPackMethod     = -1;
int nxagentPackQuality    = -1;
int nxagentTightThreshold = -1;

#endif

/*
 * Set here the required log level.
 */

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG
#undef  WATCH

#ifdef WATCH
#include "unistd.h"
#endif

char nxagentDisplayName[1024];
char nxagentWindowName[256];
char nxagentDialogName[256];
char nxagentSessionID[256];
char *nxagentOptionFile;

Bool nxagentSynchronize = False;
Bool nxagentFullGeneration = False;
int nxagentDefaultClass = TrueColor;
Bool nxagentUserDefaultClass = False;
int nxagentDefaultDepth;
Bool nxagentUserDefaultDepth = False;
Bool nxagentSoftwareScreenSaver = False;
struct UserGeometry nxagentUserGeometry = {0, 0, 0, 0, 0};
Bool nxagentUserBorderWidth = False;
int nxagentNumScreens = 0;
Bool nxagentDoDirectColormaps = False;
Window nxagentParentWindow = 0;
Bool nxagentIpaq = False;

Bool nxagentResizeDesktopAtStartup = False;

Bool nxagentUseNXTrans   = False;
Bool nxagentForceNXTrans = False;

int nxagentMaxAllowedReset = 0;

int nxagentCleanGet    = 0;
int nxagentCleanAlloc  = 0;
int nxagentCleanFlush  = 0;
int nxagentCleanSend   = 0;
int nxagentCleanImages = 0;

int nxagentImageSplit  = 0;
int nxagentImageMask   = 0;
int nxagentImageFrame  = 0;

unsigned int nxagentImageSplitMethod  = 0;
unsigned int nxagentImageMaskMethod   = 0;

#ifdef NXAGENT_KEYMAP
char *nxagentKeyboard = NULL;
char *nxagentKbtype = NULL;
#endif

#ifdef NXAGENT_EXPOSURES
Bool nxagentAutoExposures = True;
#endif

#ifdef NXAGENT_FORCEBACK
Bool nxagentForceBackingStore = False;
unsigned long nxagentMaxAreaBackS = 786432;
unsigned long nxagentMinAreaBackS = 4096;
unsigned long nxbsQueueTimeout = 5000;
#endif

#ifdef NXAGENT_ONEXIT
Bool nxagentOnce = True;
#endif

#ifdef NXAGENT_SAFEMODE
Bool nxagentSafeMode = False;
#endif

/*
 * When "fast mode" is not set, agent will query the
 * remote X server to get real content of drawables.
 * When fast mode is enabled, agent will save the
 * round-trip by just clearing the drawable to its
 * background.
 */

int nxagentFastCopyAreaEnable = TRUE;
int nxagentFastGetImageEnable = TRUE;

/*
 * Suggested color to be used when guessing
 * the content of a remote window when fast
 * GetImage mode is enabled.
 */

unsigned int nxagentPixelHintEnable = (unsigned int) UNDEFINED;

unsigned int nxagentPixelHint[8] =
{
  (unsigned int) UNDEFINED,
  (unsigned int) UNDEFINED,
  (unsigned int) UNDEFINED,
  (unsigned int) UNDEFINED,
  (unsigned int) UNDEFINED,
  (unsigned int) UNDEFINED,
  (unsigned int) UNDEFINED,
  (unsigned int) UNDEFINED
};

void nxagentProcessOptionsFile(void);
static void nxagentParseOptionString(char*);

static void nxagentSetDefaultPixelHint(Display *dpy);

/*
 * Get the caption to be used for helper dialogs
 * from agent's window name passed as parameter.
 */

static int nxagentGetDialogName();

int ddxProcessArgument(int argc, char *argv[], int i)
{
  /*
   * Ensure that the options are set to their defaults.
   */

  static Bool resetOptions = True;

  if (resetOptions == True)
  {
    nxagentResetOptions();

    resetOptions = False;
  }

  if (!strcmp(argv[i], "-B"))
  {
    #ifdef TEST
    fprintf(stderr, "ddxProcessArgument: Checking the NX binder option.\n");
    #endif

    if (nxagentCheckBinder(argc, argv, i) > 0)
    {
      /*
       * We are going to run the agent with the
       * 'binder' option. Go straight to the
       * proxy loop.
       */

      #ifdef TEST
      fprintf(stderr, "ddxProcessArgument: Entering the NX proxy binder loop.\n");
      #endif

      nxagentBinderLoop();

      /*
       * This will make an exit.
       */

      nxagentBinderExit(0);
    }
    else
    {
      /*
       * Exit with an error.
       */

      nxagentBinderExit(1);
    }
  }

  if (!strcmp(argv[i], "-display"))
  {
    if (++i < argc)
    {
      strncpy(nxagentDisplayName, argv[i], 1023);

      nxagentDisplayName[1023] = '\0';

      return 2;
    }

    return 0;
  }

  if (!strcmp(argv[i], "-id"))
  {
    if (++i < argc)
    {
      strncpy(nxagentSessionID, argv[i], 255);

      *(nxagentSessionID + 255) = '\0';

      return 2;
    }

    return 0;
  }

  /*
   * This had to be '-options' since the beginning
   * but was '-option' by mistake. Now we have to
   * handle the backward compatibility.
   */

  if (!strcmp(argv[i], "-options") || !strcmp(argv[i], "-option"))
  {
    if (++i < argc)
    {
      int size;

      if (nxagentOptionFile)
      {
        xfree(nxagentOptionFile);
        nxagentOptionFile = NULL;
      }

      if ((size = strlen(argv[i])) < 1024)
      {
        if ((nxagentOptionFile = xalloc(size + 1)) == NULL)
        {
          FatalError("malloc failed");
        }

        strncpy(nxagentOptionFile, argv[i], size);

        nxagentOptionFile[size] = '\0';
      }
      else
      {
        /*
         * It is useless to store the file name
         * that has just been truncated.
         */

        #ifdef WARNING
        fprintf(stderr, "ddxProcessArgument: WARNING! Option file name "
                    "too long. It will be ignored.\n");
        #endif
      }

      return 2;
    }

    return 0;
  }

  if (!strcmp(argv[i], "-full")) {
    nxagentFullGeneration = True;
    return 1;
  }

  if (!strcmp(argv[i], "-class")) {
    if (++i < argc) {
      if (!strcmp(argv[i], "StaticGray")) {
        nxagentDefaultClass = StaticGray;
        nxagentUserDefaultClass = True;
        return 2;
      }
      else if (!strcmp(argv[i], "GrayScale")) {
        nxagentDefaultClass = GrayScale;
        nxagentUserDefaultClass = True;
        return 2;
      }
      else if (!strcmp(argv[i], "StaticColor")) {
        nxagentDefaultClass = StaticColor;
        nxagentUserDefaultClass = True;
        return 2;
      }
      else if (!strcmp(argv[i], "PseudoColor")) {
        nxagentDefaultClass = PseudoColor;
        nxagentUserDefaultClass = True;
        return 2;
      }
      else if (!strcmp(argv[i], "TrueColor")) {
        nxagentDefaultClass = TrueColor;
        nxagentUserDefaultClass = True;
        return 2;
      }
      else if (!strcmp(argv[i], "DirectColor")) {
        nxagentDefaultClass = DirectColor;
        nxagentUserDefaultClass = True;
        return 2;
      }
    }
    return 0;
  }

  if (!strcmp(argv[i], "-cc")) {
    if (++i < argc && sscanf(argv[i], "%i", &nxagentDefaultClass) == 1) {
      if (nxagentDefaultClass >= 0 && nxagentDefaultClass <= 5) {
        nxagentUserDefaultClass = True;
        /* lex the OS layer process it as well, so return 0 */
      }
    }
    return 0;
  }

  if (!strcmp(argv[i], "-depth")) {
    if (++i < argc && sscanf(argv[i], "%i", &nxagentDefaultDepth) == 1) {
      if (nxagentDefaultDepth > 0) {
        nxagentUserDefaultDepth = True;
        return 2;
      }
    }
    return 0;
  }

  if (!strcmp(argv[i], "-sss")) {
    nxagentSoftwareScreenSaver = True;
    return 1;
  }
  
  /*
   * Use the provided color for the pixel
   * used to guess the content of GetImage
   * operations.
   */

  if (!strcmp(argv[i], "-hint"))
  {
    if (++i < argc && sscanf(argv[i], "%x", &nxagentPixelHintEnable) == 1)
    {
      return 2;
    }

    return 0;
  }

  if (!strcmp(argv[i], "-geometry"))
  {
    if (++i < argc)
    {
      if (!strcmp(argv[i],"fullscreen"))
      {
        nxagentChangeOption(Fullscreen, True);
      }
      else if (!strcmp(argv[i],"ipaq"))
      {
        nxagentChangeOption(Fullscreen, True);

        nxagentIpaq = True;
      }
      else
      {
        nxagentUserGeometry.flag = XParseGeometry(argv[i], 
                                                      &nxagentUserGeometry.X, 
                                                          &nxagentUserGeometry.Y, 
                                                              &nxagentUserGeometry.Width, 
                                                                  &nxagentUserGeometry.Height);
      }

      if (nxagentUserGeometry.flag || nxagentOption(Fullscreen)) return 2;
    }

    return 0;
  }

  if (!strcmp(argv[i], "-bw"))
  {
    int BorderWidth;

    if (++i < argc && sscanf(argv[i], "%i", &BorderWidth) == 1)
    {
      nxagentChangeOption(BorderWidth, BorderWidth);

      if (nxagentOption(BorderWidth) >= 0)
      {
        nxagentUserBorderWidth = True;

        return 2;
      }
    }

    return 0;
  }

  if (!strcmp(argv[i], "-name"))
  {
    if (++i < argc)
    {
      strncpy(nxagentWindowName, argv[i], 255);

      *(nxagentWindowName + 255) = '\0';

      return 2;
    }

    return 0;
  }

  if (!strcmp(argv[i], "-scrns")) {
    if (++i < argc && sscanf(argv[i], "%i", &nxagentNumScreens) == 1) {
      if (nxagentNumScreens > 0) {
        if (nxagentNumScreens > MAXSCREENS) {
          ErrorF("Maximum number of screens is %d.\n", MAXSCREENS);
          nxagentNumScreens = MAXSCREENS;
        }
        return 2;
      }
    }
    return 0;
  }

  if (!strcmp(argv[i], "-install")) {
    nxagentDoDirectColormaps = True;
    return 1;
  }

  if (!strcmp(argv[i], "-parent")) {
    if (++i < argc) {
      nxagentParentWindow = (XID) strtol (argv[i], (char**)NULL, 0);
      return 2;
    }
  }

  if (!strcmp(argv[i], "-forcenx")) {
    nxagentForceNXTrans = True;
    return 1;
  }

  if (!strcmp(argv[i], "-strict")) {
      nxagentStrictControl = True;
      return 1;
  }

  if (!strcmp(argv[i], "-nostrict")) {
      nxagentStrictControl = False;
      return 1;
  }

#ifdef NXAGENT_EXPOSURES
  if (!strcmp(argv[i], "-noautoexpose")) {
      nxagentAutoExposures = False;
      return 1;
  }
#endif

#ifdef NXAGENT_ONEXIT
  if (!strcmp(argv[i], "-noonce"))
  {
      nxagentOnce = False;
      return 1;
  }
#endif

#ifdef NXAGENT_FORCEBACK
  if (!strcmp(argv[i], "-bs"))
  {
    if (++i < argc)
    {
      if (!strcmp(argv[i], "force"))
      {
        /*
         * nxagentChangeOption(BackingStore, BackingStoreForce);
         */

        fprintf(stderr, "ddxProcessArgument: WARNING: \"force\" value for -bs option is disabled.\n");
        fprintf(stderr, "ddxProcessArgument: WARNING: Setting default value \"when_requested\".\n");

        return 2;
      }
      else if (!strcmp(argv[i], "when_requested"))
      {
        /*
         * nxagentChangeOption(BackingStore, BackingStoreWhenRequested);
         */

        return 2;
      }
      else if (!strcmp(argv[i], "never"))
      {
        /*
         * nxagentChangeOption(BackingStore, BackingStoreNever);
         */

        fprintf(stderr, "ddxProcessArgument: WARNING: \"never\" value for -bs option is disabled.\n");
        fprintf(stderr, "ddxProcessArgument: WARNING: Setting default value \"when_requested\".\n");

        return 2;
      }
    }

    /*
     * ErrorF("Warning: Option -bs without arguments disables backing-store.\n");
     * ErrorF("Warning: Use -bs [force|when_requested|never] to set required mode.\n");
     */

    fprintf(stderr, "ddxProcessArgument: WARNING: Option -bs without arguments is disabled.\n");
    fprintf(stderr, "ddxProcessArgument: WARNING: Setting default value \"when_requested\".\n");

    /*
     * nxagentChangeOption(BackingStore, BackingStoreNever);
     */

    return 1;
  }

  if (!strcmp(argv[i], "-maxbsarea")) {
      if (++i < argc && sscanf(argv[i], "%lu", &nxagentMaxAreaBackS) == 1) {
          if (nxagentMaxAreaBackS > 0) {
              return 2;
          }
      }
      return 0;
  }

  if (!strcmp(argv[i], "-minbsarea")) {
      if (++i < argc && sscanf(argv[i], "%lu", &nxagentMinAreaBackS) == 1) {
          if (nxagentMinAreaBackS > 0) {
              return 2;
          }
      }
      return 0;
  }

  if (!strcmp(argv[i], "-bstimeout")) {
      if (++i < argc && sscanf(argv[i], "%lu", &nxbsQueueTimeout) == 1) {
          if (nxbsQueueTimeout > 0) {
              return 2;
          }
      }
      return 0;
  }
#endif

#ifdef NXAGENT_SAFEMODE
  if (!strcmp(argv[i], "-safe")) {
    nxagentSafeMode = True;
    return 1;
  }
#endif

#ifdef NXAGENT_KEYMAP
  if (!strcmp(argv[i], "-keyboard")) {
    if (++i < argc)
    {
      nxagentKeyboard = argv[i];
      return 2;
    }
    return 0;
  }

  if (!strcmp(argv[i], "-kbtype")) {
    if (++i < argc)
    {
      int size;

      if (nxagentKbtype)
      {
        xfree(nxagentKbtype);
        nxagentKbtype = NULL;
      }

      if ((size = strlen(argv[i])) < 256)
      {
        if ((nxagentKbtype = xalloc(size + 1)) == NULL)
        {
          FatalError("malloc failed");
        }

        strncpy(nxagentKbtype, argv[i], size);

        nxagentKbtype[size] = '\0';
      }
      #ifdef WARNING

      /*
       * Is useless to remember a kbtype
       * that has just been truncated.
       */

      else
      {
        fprintf(stderr, "ddxProcessArgument: Warning option kbtype too long, ignored\n");
      }
      #endif

      return 2;
    }
    return 0;
  }
#endif

#ifdef NXAGENT_FOCUS_KARMA
 if (!strcmp(argv[i], "-limit")) {
    nxagentUsesFocusKarma = True;
    return 1;
  }

  if (!strcmp(argv[i], "-nolimit")) {
    nxagentUsesFocusKarma = False;
    return 1;
  }
#endif

  /*
   * FIXME: -nogetimage and -vfb are useless with
   * the internal fb. Options are left only for
   * compatibility with old versions.
   */

  if (!strcmp(argv[i], "-vfb"))
  {
     return 2;
  }

  if (!strcmp(argv[i], "-extensions"))
  {
     return 1;
  }

#ifdef RENDER
  if (!strcmp(argv[i], "-norender"))
  {
    nxagentRenderEnable = False;

    return 1;
  }
#endif

  /*
   * The original -noreset option, disabling
   * dispatchExceptionAtReset, is the default.
   * Use this option to restore the original
   * behaviour.
   */

  if (!strcmp(argv[i], "-reset"))
  {
    nxagentChangeOption(Reset, True);

    return 1;
  }

  if (!strcmp(argv[i], "-persistent"))
  {
    nxagentChangeOption(Persistent, True);

    return 1;
  }

  if (!strcmp(argv[i], "-nopersistent"))
  {
    nxagentChangeOption(Persistent, False);

    return 1;
  }

  if (!strcmp(argv[i], "-noshmem"))
  {
    nxagentChangeOption(SharedMemory, False);

    return 1;
  }

  if (!strcmp(argv[i], "-noignore"))
  {
    nxagentChangeOption(DeviceControl, True);

    return 1;
  }

  /*
   * Enable pseudo-rootless mode.
   */

  if (!strcmp(argv[i], "-R"))
  {
    nxagentChangeOption(Binder, False);
    nxagentChangeOption(Desktop, False);
    nxagentChangeOption(Rootless, True);

    return 1;
  }

  /*
   * Enable "desktop" mode. This is the
   * default.
   */

  if (!strcmp(argv[i], "-D"))
  {
    nxagentChangeOption(Binder, False);
    nxagentChangeOption(Rootless, False);
    nxagentChangeOption(Desktop, True);

    return 1;
  }

  /*
   * Enable the auto-disconnect timeout.
   */

  if (!strcmp(argv[i], "-timeout"))
  {
    int seconds;

    if (++i < argc && sscanf(argv[i], "%i", &seconds) == 1)
    {
      if (seconds >= 0)
      {
        if (seconds > 0 && seconds < 60)
        {
          seconds = 60;
        }

        nxagentChangeOption(Timeout, seconds);

        return 2;
      }
    }

    return 0;
  }

  if (!strcmp(argv[i], "-query"))

  {
    nxagentChangeOption(Desktop, True);
    nxagentChangeOption(Xdmcp, True);
    nxagentMaxAllowedReset = 0;

    return 0;
  }

  if (!strcmp(argv[i], "-broadcast"))

  {
    nxagentChangeOption(Desktop, True);
    nxagentChangeOption(Xdmcp, True);
    nxagentMaxAllowedReset = 0;

    return 0;
  }

  if (!strcmp(argv[i], "-indirect"))
  {
    nxagentChangeOption(Desktop, True);
    nxagentChangeOption(Xdmcp, True);
    nxagentMaxAllowedReset = 1;

    return 0;
  }

  return 0;
}

static void nxagentParseOptions(char *name, char *value)
{
  int size, argc;
  char *argv[2] = {NULL, NULL};
  extern int nxagentReconnectTrap;

  #ifdef WARNING
  fprintf(stderr, "nxagentParseOptions: Processing option '%s' = '%s'.\n",
              validateString(name), validateString(value));
  #endif

  if (!strcmp(name, "kbtype") ||
          !strcmp(name, "id") ||
              !strcmp(name, "display") ||
                  !strcmp(name, "geometry"))
  {
    argv[1] = value;
    argc = 2;
  }
  else if (!strcmp(name, "R") && !strcmp(value, "1"))
  {
    argc = 1;
  }
  else if (!strcmp(name, "fast") || !strcmp(name, "slow"))
  {
    /*
     * Enable use of "fast" mode in CopyArea
     * and GetImage operations, saving, if
     * possible, the network roundtrips.
     */

    if ((!strcmp(name, "fast") && !strcmp(value, "1")) ||
            (!strcmp(name, "slow") && !strcmp(value, "0")))
    {
      nxagentFastCopyAreaEnable = True;
      nxagentFastGetImageEnable = True;
    }
    else if ((!strcmp(name, "fast") && !strcmp(value, "0")) ||
                 (!strcmp(name, "slow") && !strcmp(value, "1")))
    {
      nxagentFastCopyAreaEnable = False;
      nxagentFastGetImageEnable = False;
    }
    else
    {
      #ifdef WARNING
      fprintf(stderr, "WARNING: Ignored wrong value [%s] for option [%s].\n",
                  validateString(name), validateString(value));
      #endif
    }

    return;
  }
  else if (!strcmp(name, "render"))
  {
    if (nxagentReconnectTrap == True)
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentParseOptions: Ignoring render option at reconnection.\n");
      #endif
    }
    else if (!strcmp(value, "1"))
    {
      nxagentRenderEnable = True;
    }
    else if (!strcmp(value, "0"))
    {
      nxagentRenderEnable = False;
    }
    else
    {
      #ifdef WARNING
      fprintf(stderr, "WARNING: Ignored wrong value [%s] for render option.\n", validateString(value));
      #endif
    }

    return;
  }
  else if (!strcmp(name, "fullscreen"))
  {
    if (nxagentReconnectTrap == True)
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentParseOptions: Ignoring fullscreen option at reconnection.\n");
      #endif
    }
    else if (!strcmp(value, "1"))
    {
      nxagentChangeOption(Fullscreen, True);
    }
    else if (!strcmp(value, "0"))
    {
      nxagentChangeOption(Fullscreen, False);
    }
    else
    {
      #ifdef WARNING
      fprintf(stderr, "WARNING: Ignored wrong value [%s] for fullscreen option.\n", validateString(value));
      #endif
    }

    return;
  }
  else if (!strcmp(name, "resize"))
  {
    if (!strcmp(value, "1"))
    {
      nxagentResizeDesktopAtStartup = True;
    }
    else if (!strcmp(value, "0"))
    {
      nxagentResizeDesktopAtStartup = False;
    }
    else
    {
      #ifdef WARNING
      fprintf(stderr, "WARNING: Ignored wrong value [%s] for resize option.\n", validateString(value));
      #endif
    }

    return;
  }
  else
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentParseOptionString: Warning ignored option '%s' = '%s'.\n",
                validateString(name), validateString(value));
    #endif

    return;
  }

  if ((size = strlen(name) + 1) > 1)
  {
    if ((argv[0] = malloc(size + 1)) == NULL)
    {
      FatalError("Malloc failed");
    }

    *argv[0] = '-';

    memcpy(argv[0] + 1, name, size);
  }

  ddxProcessArgument(argc, argv, 0);
}

static void nxagentParseOptionString(char *string)
{
  char *value = NULL, *option = NULL, *delimiter= NULL;

  /*
   * removing the port specification.
   */

  delimiter = rindex(string, ':');

  if (delimiter)
  {
    *delimiter = 0;
  }
  else
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentParseOptionString: "
                "Warning option file doesn't contain port specification.\n");
    #endif
  }

  while ((option = strtok(option ? NULL : string, ",")))
  {
    delimiter = rindex(option, '=');

    if (delimiter)
    {
      *delimiter = 0;
      value = delimiter + 1;
    }
    else
    {
      value = NULL;
    }

    nxagentParseOptions(option, value);
  }
}

void nxagentProcessOptionsFile()
{
  FILE *file;
  long offset, size, sizeOfFile, maxFileSize = (1 << 20);
  char *data;

  #ifdef WARNING
  fprintf(stderr, "nxagentProcessOptionsFile: Option file is [%s].\n", validateString(nxagentOptionFile));
  #endif

  if (nxagentOptionFile == NULL)
  {
    return;
  }

  if ((file = fopen(nxagentOptionFile, "r")) == NULL)
  {
    fprintf(stderr, "nxagentProcessOptionsFile: WARNING couldn't open option file %s: %s\n",
                validateString(nxagentOptionFile), strerror(errno));
    goto Exit;
  }

  if (fseek(file, 0, SEEK_END))
  {
    perror("nxagentProcessOptionsFile: fseek failed");
    goto CloseFile;
  }

  sizeOfFile = ftell(file);

  if (sizeOfFile == -1)
  {
    perror("nxagentProcessOptionsFile: ftell failed");
    goto CloseFile;
  }
  #ifdef WARNING
  else
  {
    fprintf(stderr, "nxagentProcessOptionsFile: Option file is %ld bytes long.\n", sizeOfFile);
  }
  #endif

  rewind(file);

  if (sizeOfFile > maxFileSize)
  {
    fprintf(stderr, "nxagentProcessOptionsFile: couldn't parse option file: too big\n");
    goto CloseFile;
  }

  if ((data = xalloc(sizeOfFile + 1)) == NULL)
  {
    perror("nxagentProcessOptionsFile: malloc failed");
    goto CloseFile;
  }

  offset = 0;
  size = 0;

  while (1)
  {
    size_t result = fread(data + offset, 1, sizeOfFile, file);

    if (ferror(file))
    {
      perror("nxagentProcessOptionsFile: fread failed");
      goto FreeMem;
    }

    size += result;
    offset += result;

    if (feof(file) || (size == sizeOfFile))
    {
      break;
    }
  }

  if (size != sizeOfFile)
  {
     fprintf(stderr, "nxagentProcessOptionsFile: couldn't read all the option file\n");
     goto FreeMem;
  }

  /*
   * Truncate buffer to first line.
   */

  for (offset = 0; (offset < sizeOfFile) && (data[offset] != '\n'); offset++);

  data[offset] = 0;

  nxagentParseOptionString(data);

FreeMem:
  if (data)
  {
      xfree(data);
  }
CloseFile:
  if (fclose(file))
  {
    fprintf(stderr, "nxagentProcessOptionsFile: WARNING couldn't close option file %s: %s\n",
                validateString(nxagentOptionFile), strerror(errno));
  }
Exit:
  return;
}

/*
 * FIXME: Transport initialization, shouldn't depend upon
 *        argv[], because we call it at every reconnection.
 */

Bool nxagentPostProcessArgs(char* name, Display* dpy, Screen* scr)
{
    Bool useNXTrans = False;

    #ifdef WATCH

    fprintf(stderr, "nxagentPostProcessArgs: Watchpoint 2.\n");

/*
Reply   Total	Cached	Bits In			Bits Out		Bits/Reply	  Ratio
------- -----	------	-------			--------		----------	  -----
N/A
*/

    sleep(30);

    #endif

    if (nxagentOption(Rootless) && nxagentOption(Fullscreen))
    {
      #ifdef WARNING
      fprintf(stderr, "WARNING: Ignoring fullscreen option for rootless session.\n");
      #endif

      nxagentChangeOption(Fullscreen, False);
    }

    /*
     * Ensure we have a valid name for children dialogs.
     */

    nxagentGetDialogName();

    /*
     * Ensure we have a valid name for window name.
     */

    if (*nxagentWindowName == '\0')
    {
      strncpy(nxagentWindowName, "NX", 255);

      *(nxagentWindowName + 255) = '\0';
    }

    /*
     * Set backing-store option according
     * to command line parameter.
     */

    switch (nxagentOption(BackingStore))
    {
      case BackingStoreNever:
      {
        enableBackingStore = False;
        nxagentForceBackingStore = False;

        break;
      }
      case BackingStoreWhenRequested:
      {
        enableBackingStore = True;
        nxagentForceBackingStore = False;

        break;
      }
      case BackingStoreForce:
      {
        enableBackingStore = True;
        nxagentForceBackingStore = True;

        break;
      }
      default:
      {
        nxagentChangeOption(BackingStore, BackingStoreWhenRequested);

        enableBackingStore = True;
        nxagentForceBackingStore = False;

        break;
      }
    }

    #ifdef TEST
    fprintf(stderr, "nxagentPostProcessArgs: Backing store mode set to [%d] with [%d] and [%d].\n",
                nxagentOption(BackingStore), enableBackingStore, nxagentForceBackingStore);
    #endif

    /*
     * Note that use of NX packed images as well as
     * render extension could be later disabled due
     * to the fact that session is running nested
     * in a nxagent server.
     */

    if (nxagentForceNXTrans)
    {
      useNXTrans = True;
    }
    else if ((strncasecmp(name, "nx/", 3) == 0) ||
                 (strncasecmp(name, "nx:", 3) == 0) ||
                     (strncasecmp(name, "nx,", 3) == 0) ||
                         (strcasecmp(name, "nx") == 0))
    {
      useNXTrans = True;
    }

    if (useNXTrans)
    {
      int karmaSize, splitSize;
      int bitsPerPixel = 0;

      unsigned int linkType;
      unsigned int protocolMajor, protocolMinor, protocolPatch;
      unsigned int packMethod, packQuality;
      int dataLevel, streamLevel, deltaLevel;
      unsigned int loadCache, saveCache, startupCache;

      unsigned char supportedMethods[NXNumberOfPackMethods];
      unsigned int entries = NXNumberOfPackMethods;

      NXGetControlParameters(dpy, &linkType, &protocolMajor, &protocolMinor,
                                 &protocolPatch, &karmaSize, &splitSize,
                                     &packMethod, &packQuality, &dataLevel,
                                         &streamLevel, &deltaLevel, &loadCache,
                                             &saveCache, &startupCache);

      /*
       * This option should be used to verify if the NX
       * transport is enabled, assuming that a value of
       * 0 is also defined as LINK_TYPE_NONE. This is a
       * side effect, in reality, but can be considered
       * safe enough.
       */

      nxagentChangeOption(LinkType, linkType);

      if (protocolMajor < 1 ||
              (protocolMajor == 1 &&
                   protocolMinor < 3))
      {
        nxagentChangeOption(SharedMemory, False);
      }

      /*
       * Set the flag so a dialog will be
       * showed at the time timeout is
       * elapsed.
       */

      if (loadCache == 1 && startupCache == 0)
      {
        nxagentSetNoCacheDialogTimeout(linkType);

        nxagentNoCacheDialogEnable = True;
      }

#ifdef NXAGENT_PACKEDIMAGES_DEBUG
      fprintf(stderr, "nxagentPostProcessArgs: Got preferred pack method [%d] and quality [%d] from proxy.\n",
                  packMethod, packQuality);
#endif

      if (nxagentPackMethod == -1)
      {
        nxagentPackMethod = packMethod;
      }

      if (nxagentPackQuality == -1)
      {
        nxagentPackQuality = packQuality;
      }

      if (nxagentTightThreshold == -1)
      {
        nxagentTightThreshold = splitSize;
      }

      if (NXGetUnpackParameters(dpy, &entries, supportedMethods) == 0 ||
              entries != NXNumberOfPackMethods)
      {
        fprintf(stderr, "nxagentPostProcessArgs: NXGetUnpackParameters() failed. Disabling pack of images\n");

        nxagentPackMethod = NO_PACK;
      }

      if (!supportedMethods[nxagentPackMethod] &&
              nxagentPackMethod >= PACK_PNG_JPEG_8_COLORS &&
                  nxagentPackMethod <= PACK_PNG_JPEG_16M_COLORS)
      {
        /*
         * Let's try the same JPEG method without PNG.
         * Arbitrarily assume the corresponding method
         * is at this offset.
         */

        fprintf(stderr, "nxagentPostProcessArgs: Pack method [%d] not supported by proxy.\n",
                    nxagentPackMethod);

        nxagentPackMethod = nxagentPackMethod - 21;
      }

      if (!supportedMethods[nxagentPackMethod])
      {
        fprintf(stderr, "nxagentPostProcessArgs: Pack method [%d] not supported by proxy. Disabling pack of images.\n",
                      nxagentPackMethod);

        nxagentPackMethod = NO_PACK;
      }

      bitsPerPixel = _XGetBitsPerPixel(dpy, DefaultDepth(dpy, DefaultScreen(dpy)));

      nxagentPackMethod = SelectPackMethod(nxagentPackMethod, bitsPerPixel,
                                               bitsPerPixel);

      /*
       * FIXME: There is a problem in cleanup of images when selecting
       *        a method that doesn't reduce the number of colors. This
       *        is the case of JPEG 16M on 24/32 bits. The same problem
       *        arises with JPEG 64K in case of 16 bits, but we cannot
       *        force 32K because this would degrade quality of images
       *        too much.
       */
/*
#define NXAGENT_PACKEDIMAGES_DEBUG

      if (nxagentPackMethod == PACK_JPEG_16M_COLORS &&
              (bitsPerPixel == 24 || bitsPerPixel == 32))
      {
        nxagentPackMethod = PACK_JPEG_2M_COLORS;

        #ifdef NXAGENT_PACKEDIMAGES_DEBUG
        fprintf(stderr, "nxagentPostProcessArgs: Forcing pack method [%d] with [%d] bits per pixel.\n",
                    nxagentPackMethod, bitsPerPixel);
        #endif

        fprintf(stderr, "nxagentPostProcessArgs: Cleanup of images will be applied.\n");
      }
      #ifdef NXAGENT_PACKEDIMAGES_DEBUG
      else if (nxagentPackMethod == PACK_JPEG_64K_COLORS &&
                   bitsPerPixel == 16)
      {
        fprintf(stderr, "nxagentPostProcessArgs: Using pack method [%d] with [%d] bits per pixel.\n",
                    nxagentPackMethod, bitsPerPixel);

        fprintf(stderr, "nxagentPostProcessArgs: No cleanup of images will be applied.\n");
      }
      else
      {
        fprintf(stderr, "nxagentPostProcessArgs: Using pack method [%d] with [%d] bits per pixel.\n",
                    nxagentPackMethod, bitsPerPixel);
      }
      #endif

#undef NXAGENT_PACKEDIMAGES_DEBUG
*/

      if (nxagentStopImgSz == -1)
      {
        nxagentStopImgSz = splitSize;
      }

      if (nxagentStopKarmaSz == -1)
      {
        nxagentStopKarmaSz        = karmaSize;
        nxagentStopKarmaSzDefault = karmaSize;
        nxagentStopKarmaSzCheck   = karmaSize >> 4;
      }

      if (nxagentPackMethod != -1)
      {
        /*
         * Set the unpack geometry for the 'default'
         * client.
         */

        NXSetUnpackGeometry(dpy, 0, scr, NULL);
      }

      NXGetCleanupParameters(dpy, &nxagentCleanGet, &nxagentCleanAlloc,
                                 &nxagentCleanFlush, &nxagentCleanSend, &nxagentCleanImages);

      NXGetImageParameters(dpy, &nxagentImageSplit, &nxagentImageMask, &nxagentImageFrame,
                               &nxagentImageSplitMethod, &nxagentImageMaskMethod);

      /*
       * Agent calls NXlib clean and mask functions
       * by itself, instead of relying on Xlib.
       */

      {
        /*
         * FIXME: These definitions should be
         * available in NXlib.h or NXlibint.h.
         */

        extern int _NXEnableCleanImages;
        extern int _NXEnableImageMask;

        _NXEnableCleanImages = 0;
        _NXEnableImageMask   = 0;
      }

      /*
       * In case of slower links send karma and
       * sync messages based on congestion state
       * reported by NX transport.
       */

      if (nxagentStrictControl == False)
      {
        if (linkType == LINK_TYPE_MODEM ||
                linkType == LINK_TYPE_ISDN ||
                    linkType == LINK_TYPE_ADSL)
        {
          nxagentStrictControl = False;
        }
        else
        {
          nxagentStrictControl = True;
        }
      }
      else
      {
        /*
         * If strict control is enabled reduce
         * by half the number of blocking sync
         * messages.
         */

        if (linkType == LINK_TYPE_MODEM ||
                linkType == LINK_TYPE_ISDN ||
                    linkType == LINK_TYPE_ADSL)
        {
          nxagentSyncModulo = 2;
        }
      }

      if (nxagentStrictControl == False)
      {
        fprintf(stderr, "Info: Disabling strict flow control in agent.\n");
      }

      /*
       * Adjust threshold according to the
       * current policy.
       */

      nxagentAdjustKarma();

      /*
       * Activate shared memory PutImages
       * in X server proxy.
       */

      if (nxagentOption(SharedMemory) == True)
      {
        /*
         * Shared memory on path from agent to
         * X client proxy is not implemented.
         */

        unsigned int enableClient = 0;
        unsigned int enableServer = 1;

        unsigned int clientSegment, serverSegment;

        NXGetShmemParameters(dpy, &enableClient, &enableServer,
                                 &clientSegment, &serverSegment);

        if (enableServer == False)
        {
          fprintf(stderr, "Info: Not using shared memory support in X server.\n");
        }
        else
        {
          fprintf(stderr, "Info: Using shared memory support in X server.\n");
        }
      }
    }
    else
    {
      /*
       * We don't have a proxy on the remote side.
       */

      nxagentPackMethod = -1;

      nxagentStopImgSz = 0;
      nxagentStopKarmaSz = 0;
      nxagentStopKarmaSzDefault = 0;
      nxagentStopKarmaSzCheck = 0;
    }

    /*
     * Resetting the congestion state.
     */

    nxagentCongestion = False;

    /*
     * Disable slow mode if not set selected on the
     * command line.
     */
    
    if (nxagentFastCopyAreaEnable == UNDEFINED)
    {
      nxagentFastCopyAreaEnable = True;
    }

    if (nxagentFastGetImageEnable == UNDEFINED)
    {
      nxagentFastGetImageEnable = True;
    }

    if (nxagentPixelHintEnable == (unsigned int) UNDEFINED)
    {
      nxagentSetDefaultPixelHint(dpy);
    }
    else
    {
      /*
       * Use the suggested color.
       */

      nxagentPixelHint[0] = nxagentPixelHint[1] =
          nxagentPixelHint[2] = nxagentPixelHint[3] =
              nxagentPixelHintEnable;
    }


    /*
     * Why do we have to store this
     * in a separate flag?
     */

    if (nxagentRenderEnable == True)
    {
      nxagentAlphaEnabled = True;
    }
    else
    {
      /*
       * Without RENDER extension many X clients 
       * use lots of get image in order to do
       * compositing between windows and pixmaps.
       * And this can badly fail if fast get image
       * is enabled.
       */

      nxagentAlphaEnabled = False;
      nxagentFastCopyAreaEnable = False;
      nxagentFastGetImageEnable = False;
    }

    #ifdef TEST

    if (nxagentFastGetImageEnable)
    {
      fprintf(stderr, "nxagentPostProcessArgs: Using pixel hints [%x, %x, %x, %x].\n",
                  nxagentPixelHint[0], nxagentPixelHint[1],
                      nxagentPixelHint[2], nxagentPixelHint[3]);
    }

    #endif

    if (nxagentFastCopyAreaEnable == True)
    {
      fprintf(stderr, "Info: Using fast copy area mode in agent.\n");
    }
    else
    {
      fprintf(stderr, "Info: Not using fast copy area mode in agent.\n");
    }

    if (nxagentFastGetImageEnable == True)
    {
      fprintf(stderr, "Info: Using fast get image mode in agent.\n");
    }
    else
    {
      fprintf(stderr, "Info: Not using fast get image mode in agent.\n");
    }

    if (nxagentOption(Rootless) && nxagentOption(Xdmcp))
    {
      FatalError("PANIC! Cannot start a XDMCP session in rootless mode.\n");
    }

    /*
     * We enable server reset only for indirect XDMCP sessions.
     */

    if (nxagentOption(Reset) == True && nxagentMaxAllowedReset == 0)
    {
      extern char dispatchExceptionAtReset;

      #ifdef WARNING
      fprintf(stderr, "nxagentPostProcessArgs: Disabling reset.\n");
      #endif

      nxagentChangeOption(Reset, False);

      dispatchExceptionAtReset = 0;
    }

    /*
     * We skip server reset by default. This should
     * be equivalent to passing the -noreset option
     * to a standard XFree86 server.
     */

    if (nxagentOption(Reset) == False)
    {
      extern char dispatchExceptionAtReset;

      #ifdef TEST
      fprintf(stderr, "nxagentPostProcessArgs: Disabling dispatch of exception at server reset.\n");
      #endif

      dispatchExceptionAtReset = 0;
    }

    /*
     * Check if the user activated the auto-disconect
     * feature.
     */

    if (nxagentOption(Timeout) > 0)
    {
      fprintf(stderr, "Info: Using auto-disconnect timeout of %d seconds.\n",
                  nxagentOption(Timeout));

      ScreenSaverTime = nxagentOption(Timeout) * MILLI_PER_SECOND;
    }

    #ifdef WATCH

    fprintf(stderr, "nxagentPostProcessArgs: Watchpoint 3.\n");

/*
Reply   Total	Cached	Bits In			Bits Out		Bits/Reply	  Ratio
------- -----	------	-------			--------		----------	  -----
#16     1		256 bits (0 KB) ->	12 bits (0 KB) ->	256/1 -> 12/1	= 21.333:1
#233 A  1		256 bits (0 KB) ->	131 bits (0 KB) ->	256/1 -> 131/1	= 1.954:1
#245 A  2		512 bits (0 KB) ->	19 bits (0 KB) ->	256/1 -> 10/1	= 26.947:1
*/

    sleep(30);

    #endif

    return useNXTrans;
}

void ddxUseMsg()
{
  ErrorF("-display string        display name of the real server\n");
  ErrorF("-sync                  sinchronize with the real server\n");
  ErrorF("-full                  utilize full regeneration\n");
  ErrorF("-class string          default visual class\n");
  ErrorF("-depth int             default depth\n");
  ErrorF("-sss                   use software screen saver\n");
  ErrorF("-geometry WxH+X+Y      window size and position\n");
  ErrorF("-bw int                window border width\n");
  ErrorF("-name string           window name\n");
  ErrorF("-scrns int             number of screens to generate\n");
  ErrorF("-install               instal colormaps directly\n");

  ErrorF("The NX system adds the following arguments:\n");
  ErrorF("-forcenx               force use of NX protocol messages assuming communication through nxproxy\n");
#ifdef NXAGENT_STOPBIGREQ
  ErrorF("-imgstop int           wait for confirmation from nxproxy after any X_PutImage bigger than int\n");
#endif
#ifdef NXAGENT_GETIFOCUS
  ErrorF("-getifocus             mask X_GetInputFocus requests and wait for confirmation from nxproxy\n");
#endif
#ifdef NXAGENT_XLIBPARAMS
  ErrorF("-imgframe              let NXlib draw a tiny rectangle around images\n");
  ErrorF("-imgsplit PW,PH,BW,BH  force NXlib to split images to the given width and height\n");
#endif
#ifdef NXAGENT_FORCEBACK
  /*
   * ErrorF("-backingstore          force backing-store on all windows\n");
   */
  ErrorF("-maxbsarea int         maximum area occupied by a window candidate for backing-store\n");
  ErrorF("-bstimeout int         timeout for realizing the backing-store\n");
#endif
#ifdef RENDER
  ErrorF("-norender              disable use of the render extension\n");
#endif
  ErrorF("-nopersistent          disable disconnection/reconnection to the X display on SIGHUP\n");
  ErrorF("-noshmem               disable use of shared memory in path from remote nxproxy to the X server\n");
  ErrorF("-noignore              don't ignore pointer and keyboard configuration changes mandated by clients\n");
  ErrorF("-D                     enable desktop mode\n");
  ErrorF("-R                     enable rootless mode\n");
  ErrorF("-B                     enable proxy binding mode\n");
}

/*
 * FIXME: We brought the number of pixel hints from
 * 4 to 8. Must still reflect the change in the ini-
 * tialization and in the parsing of the command
 * line parameters.
 */

static void nxagentSetDefaultPixelHint(Display *dpy)
{
  int bitsPerPixel = _XGetBitsPerPixel(dpy, DefaultDepth(dpy, DefaultScreen(dpy)));

  switch (bitsPerPixel)
  {
    case 32:
    case 24:
    {
      /*
       * Usually SUN Solaris is shipped with a
       * version of OpenOffice that just uses
       * GetImages to render antialiased text
       * on 32 bits displays.
       */

      #ifdef __sun
      
      nxagentPixelHint[0] = 0xffffff;
      nxagentPixelHint[1] = 0xffffff;
      nxagentPixelHint[2] = 0xffffff;
      nxagentPixelHint[3] = 0xffffff;
      nxagentPixelHint[4] = 0xffffff;

      #else

      nxagentPixelHint[0] = 0xe6e6e6;
      nxagentPixelHint[1] = 0xffffff;
      nxagentPixelHint[2] = 0xb7bbd4;
      nxagentPixelHint[3] = 0xcccccc;
      nxagentPixelHint[4] = 0xc0c0c0;
      nxagentPixelHint[5] = 0xf0f0f0;

      #endif

      break;
    }
    case 16:
    case 15:
    {
      #ifdef __sun
      
      nxagentPixelHint[0] = 0xffff;
      nxagentPixelHint[1] = 0xffff;
      nxagentPixelHint[2] = 0xffff;
      nxagentPixelHint[3] = 0xffff;
      nxagentPixelHint[4] = 0xffff;

      #else

      nxagentPixelHint[0] = 0xe73c;
      nxagentPixelHint[1] = 0xffff;
      nxagentPixelHint[2] = 0xb5fa;
      nxagentPixelHint[3] = 0xb5fa;
      nxagentPixelHint[4] = 0xe73c;

      /*
       * These values were previously
       * reported by Reza.
       *
       * nxagentPixelHint[0] = 0xef7c;
       * nxagentPixelHint[1] = 0xffff;
       * nxagentPixelHint[2] = 0xd676;
       * nxagentPixelHint[3] = 0xd676;
       * nxagentPixelHint[4] = 0xef7c;
       */

      #endif

      break;
    }
    default:
    {
      nxagentPixelHint[0] = 0xff;
      nxagentPixelHint[1] = 0xff;
      nxagentPixelHint[2] = 0xff;
      nxagentPixelHint[3] = 0xff;
      nxagentPixelHint[4] = 0xff;
    }
  }
}

static int nxagentGetDialogName()
{
  strcpy(nxagentDialogName, "NX");

  *(nxagentDialogName + 255) = '\0';

  if (nxagentOption(Rootless) == False)
  {
    return 1;
  }

  if (*nxagentWindowName != '\0')
  {
    char *separator = strchr(nxagentWindowName, ':');

    /*
     * Find the colon in display.
     */

    if ((separator == NULL) || !isdigit(*(separator + 1)))
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetDialogName: PANIC! Can't find port in window name [%s].\n",
                  nxagentWindowName);
      #endif
    }
    else
    {
      /*
       * Find the end of display number.
       */

      separator = strchr(separator, ' ');

      if (separator == NULL)
      {
        #ifdef TEST
        fprintf(stderr, "nxagentGetDialogName: PANIC! Can't find end of port in window name [%s].\n",
                    nxagentWindowName);
        #endif
      }
      else
      {
        strncpy(nxagentDialogName, nxagentWindowName,
                    separator - nxagentWindowName);

        *(nxagentDialogName + (separator - nxagentWindowName)) = '\0';

        #ifdef TEST
        fprintf(stderr, "nxagentGetDialogName: Got dialog name [%s] with base name [%s].\n",
                    nxagentDialogName, nxagentWindowName);
        #endif

        return 1;
      }
    }
  }
  else if (*nxagentSessionID != '\0')
  {
    int length = strlen(nxagentSessionID);

    strcpy(nxagentDialogName, "NX - ");

    if (length > (MD5_LENGTH * 2 + 1) &&
           *(nxagentSessionID + (length - (MD5_LENGTH * 2 + 1))) == '-')
    {
      strncat(nxagentDialogName, nxagentSessionID, length - (MD5_LENGTH * 2 + 1));
    }
    else
    {
      strncat(nxagentDialogName, nxagentSessionID, 250);
    }

    *(nxagentSessionID + 255) = '\0';

    return 1;
  }

  return 0;
}
