/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com.          */
/*                                                                        */
/* NXAGENT, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

/*
 * Used in handling of karma on lost focus.
 */

#include <signal.h>
#include <time.h>
#include <errno.h>

#include "X.h"
#include "Xproto.h"
#include "Xatom.h"
#include "dix.h"
#include "os.h"
#include "osdep.h"
#include "dixstruct.h"
#include "windowstr.h"
#include "inputstr.h"
#include "Agent.h"

/*
 * NX specific includes and definitions.
 */

#include NXAGENT_NXLIB_INCLUDE

#include "Args.h"
#include "Control.h"
#include "Dialog.h"
#include "Handlers.h"

/*
 * Set here the required log level.
 */

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

/*
 * Returns the last signal delivered
 * to the process.
 */

extern int _X11TransSocketCheckSignal(void);

/*
 * The current agent's display.
 */

extern Display *nxagentDisplay;

/*
 * Defined in Display.c
 */

extern Bool nxagentUseNXTrans;

/*
 * Time in milliseconds of first iteration
 * through the dispatcher.
 */

unsigned long nxagentStartTime = -1;

/*
 * Disable new handling of control flow in
 * production code. It seems to worsen the
 * performances in some circumstances.
 */

Bool nxagentStrictControl = True;

/*
 * True if client is asleep.
 */

Bool nxagentSleepingByKarma[MAX_CONNECTIONS];
Bool nxagentSleepingBySync[MAX_CONNECTIONS];

/*
 * ...At the same time reduce number of
 * sync messages by half. This is dirty
 * but seems to help significantly.
 */

unsigned int nxagentSyncCounter = 0;
unsigned int nxagentSyncModulo  = 1;

/*
 * Most of these variable are set in Args.c
 * according to command line parameters or
 * querying the NX transport.
 */

long nxagentKarmaQueue[MAX_CONNECTIONS];

Bool          nxagentUsesFocusKarma    = False;
Bool          nxagentFocusKarmaOnStart = True;
Bool          nxagentFocusKarma        = False;
unsigned long nxagentFocusKarmaTimeOut = 0;

long nxagentStopKarmaSz        = -1;
long nxagentStopKarmaSzDefault = -1;
long nxagentStopKarmaSzCheck   = -1;
long nxagentStopKarmaSleep     = 50000;

/*
 * Used for the streaming of images.
 */

int nxagentStopImgSz = -1;

/*
 * These flags are updated according to ClientMessages
 * received from transport in case the proxy link is
 * temporarily unable to accept more data or when proxy
 * link is going through a reset.
 */

Bool nxagentCongestion = False;
Bool nxagentReset      = False;

/*
 * Count how many XSync() are received from clients.
 * If they exceed a threshold in a given period of
 * time then synchronize with the real display. Used
 * to avoid some ugly side-effects caused by clients
 * performing animations based on time needed by a
 * round-trip.
 */

unsigned int nxagentAnimationCount      = 0;
unsigned long int nxagentAnimationStart = 0;

int nxagentCheckAnimation(unsigned long int now);

/*
 * Used to show an alert in the case a persistent
 * cache was not loaded by NX proxy at startup.
 */

int nxagentNoCacheDialogEnable  = False;
int nxagentNoCacheDialogTimeout = 30000;

/*
 * Referenced by Init.c and Screen.c.
 */

int nxagentClientPrivateIndex;

void nxagentGuessClientHint(ClientPtr client, Atom property, char *data)
{
  #ifdef TEST
  fprintf(stderr, "nxagentGuessClientHint: Client [%d] setting property [%s] as [%s].\n",
              client -> index, validateString(NameForAtom(property)), validateString(data));
  #endif

  if (property == XA_WM_NAME && strstr(data, "OpenOffice.org"))
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentGuessClientHint: Detected OpenOffice as [%d].\n", client -> index);
    #endif

    nxagentClientHint(client) = OPEN_OFFICE;
  }

  if (nxagentClientPriv(client) -> clientHint == UNKNOWN)
  {
    if (property == XA_WM_CLASS && strcmp(data, "nxclient") == 0)
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentGuessClientHint: Detected nxclient as [%d].\n", client -> index);
      #endif

      nxagentClientHint(client) = NXCLIENT;
    }
  }

  if (nxagentClientPriv(client) -> clientHint == NXCLIENT)
  {
    if (property == MakeAtom("WM_WINDOW_ROLE", 14, True) &&
            strncmp(data, "msgBox", 6) == 0)
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentGuessClientHint: Detected nxclient dialog as [%d].\n", client -> index);
      #endif

      nxagentClientHint(client) = NXCLIENT_DIALOG;
    }
  }
}

void nxagentWakeupByReconnect(void)
{
  int i;

  for (i = 1; i < currentMaxClients; i++)
  {
    if ((clients[i]) && !(clients[i] -> clientGone) &&
            (nxagentClientPriv(clients[i]) -> is_ignored))
    {
      AttendClient(clients[i]);

      nxagentSleepingByKarma[i] = False;
      nxagentSleepingBySync[i] = False;

      nxagentClientPriv(clients[i]) -> is_ignored = False;
    }
  }

  return;
}

void nxagentWakeupByDecongestion(struct timeval **timeout)
{
  extern fd_set ClientsWithInput;

  static struct timeval zero;

  int wakeup = 0;

  int i;

  for (i = 1; i < currentMaxClients; i++)
  {
    if (clients[i] && !(clients[i] -> clientGone) &&
            nxagentClientPriv(clients[i]) -> is_ignored)
    {
      if (nxagentSleepingByKarma[i])
      {
        nxWakeByKarma(clients[i]);

        wakeup = 1;
      }
      else if (nxagentSleepingBySync[i])
      {
        nxWakeByGetIFocus(clients[i]);

        wakeup = 1;
      }

      if (wakeup == 1)
      {
        int fd = ((OsCommPtr) clients[i] -> osPrivate) -> fd;

        wakeup = 0;

        if (FD_ISSET(fd, &ClientsWithInput))
        {
          #ifdef TEST
          fprintf(stderr, "nxagentWakeupByDecongestion: Client with descriptor [%d] has input "
                      "after the wakeup.\n", fd);
          #endif

          if (*timeout != NULL)
          {
            #ifdef TEST
            fprintf(stderr, "nxagentWakeupByDecongestion: Former select timeout was [%ld] Ms.\n",
                        (*timeout) -> tv_sec * 1000 + (*timeout) -> tv_usec / 1000);
            #endif

            (*timeout) -> tv_sec  = 0;
            (*timeout) -> tv_usec = 0;
          }
          else
          {
            #ifdef TEST
            fprintf(stderr, "nxagentWakeupByDecongestion: Former select timeout was null.\n");
            #endif

            zero.tv_sec  = 0;
            zero.tv_usec = 0;

            *timeout = &zero;
          }

          #ifdef TEST
          fprintf(stderr, "nxagentWakeupByDecongestion: New select timeout is [%ld] Ms.\n",
                      (*timeout) -> tv_sec * 1000 + (*timeout) -> tv_usec / 1000);
          #endif
        }
      }
    }
  }

  return;
}

Bool nxWakeByError(ClientPtr client)
{
  if (client == NULL)
  {
      ErrorF("nxWakeByError: Bad client pointer provided to function.\n");

      return False;
  }

#ifdef TEST
  fprintf(stderr, "nxWakeByError: Going to wakeup client id [%d].\n",
              client -> index);
#endif

  AttendClient(client);

  nxagentSleepingByKarma[client -> index] = False;
  nxagentSleepingBySync[client -> index] = False;

  nxagentClientPriv(client) -> is_ignored = False;

  if (nxagentClientPriv(client) -> GetIFReply != NULL)
  {
#ifdef TEST
      fprintf(stderr, "nxWakeByError: Going to write pending reply to client id [%d].\n",
                  client -> index);
#endif

      nxagentWriteReplyToClient(client, sizeof(xGetInputFocusReply),
                                (xGetInputFocusReply*) nxagentClientPriv(client) -> GetIFReply);

      /*
       * Used to restart the client here.
       *
       * AttendClient(client);
       */

      xfree(nxagentClientPriv(client) -> GetIFReply);

      nxagentClientPriv(client) -> GetIFReply = NULL;
  }

  return True;
}

Bool nxSleepByBigReq(ClientPtr client)
{
    if (nxagentUseNXTrans)
    {
        if (client == NULL)
        {
            ErrorF("nxSleepByBigReq: Bad client pointer provided to function.\n");

            return False;
        }

#ifdef TEST
        fprintf(stderr, "nxSleepByBigReq: Forcing sleep of client [%d] with agent sequence [%ld].\n",
                    client -> index, NextRequest(nxagentDisplay));
#endif

        nxagentClientPriv(client) -> is_ignored = True;

        IgnoreClient(client);
    }

    return True;
}

Bool nxWakeByBigRequest(ClientPtr client)
{
    if (nxagentUseNXTrans)
    {
        if (client == NULL)
        {
            ErrorF("nxWakeByBigRequest: Bad client pointer provided to function.\n");

            return False;
        }

#ifdef TEST
        fprintf(stderr, "nxWakeByBigRequest: Going to wakeup client id [%d].\n",
                    client -> index);
#endif

        AttendClient(client);
        nxagentClientPriv(client) -> is_ignored = False;
    }

    return True;
}

Bool nxSleepByGetIFocus(ClientPtr client, pointer rep)
{
    if (nxagentUseNXTrans)
    {
        if (client == NULL)
        {
            ErrorF("nxSleepByGetIFocus: Bad client pointer provided to function.\n");

            return False;
        }

#ifdef TEST
        fprintf(stderr, "nxSleepByGetIFocus: Forcing sleep of client [%d] with agent sequence [%ld].\n",
                    client -> index, NextRequest(nxagentDisplay));
#endif

        nxagentClientPriv(client) -> is_ignored = True;
        IgnoreClient(client);

        nxagentClientPriv(client) -> GetIFReply = rep;

        nxagentSleepingBySync[client -> index] = True;

        /*
         *  One synchronization event is enough.
         *  Reset bytes received for this client.
         */

        nxagentKarmaQueue[client -> index] = 0;
    }

    return True;
}

Bool nxWakeByGetIFocus(ClientPtr client)
{
    if (nxagentUseNXTrans)
    {
        if (client == NULL)
        {
            ErrorF("nxWakeByGetIFocus: Bad client pointer provided to function.\n");

            return False;
        }

#ifdef TEST
        fprintf(stderr, "nxWakeByGetIFocus: Going to wakeup client id [%d].\n",
                    client -> index);
#endif

        AttendClient(client);

        nxagentSleepingBySync[client -> index] = False;

        nxagentClientPriv(client) -> is_ignored = False;

        if (nxagentClientPriv(client) -> GetIFReply != NULL)
        {
#ifdef TEST
            fprintf(stderr, "nxWakeByGetIFocus: Going to write reply to client id [%d].\n",
                        client -> index);
#endif

            nxagentWriteReplyToClient(client, sizeof(xGetInputFocusReply),
                                      (xGetInputFocusReply*) 
                                         nxagentClientPriv(client) -> GetIFReply);

            xfree(nxagentClientPriv(client) -> GetIFReply);

            nxagentClientPriv(client) -> GetIFReply = NULL;

            return True;
        }
        else
        {
#ifdef TEST
            fprintf(stderr, "nxWakeByGetIFocus: Pointer to reply for client id [%d] is NULL.\n",
                    client -> index);
#endif

            return False;
        }
    }

    return True;
}

Bool nxSleepByKarma(ClientPtr client)
{
    if (nxagentUseNXTrans)
    {
        if (client == NULL)
        {
            ErrorF("nxSleepByKarma: Bad client pointer provided to function.\n");

            return False;
        }

#ifdef TEST
        fprintf(stderr, "nxSleepByKarma: Forcing sleep of client [%d] with agent sequence [%ld].\n",
                client -> index, NextRequest(nxagentDisplay));
#endif

        nxagentClientPriv(client) -> is_ignored = True;
        IgnoreClient(client);

        nxagentSleepingByKarma[client -> index] = True;
    }

    return True;
}

Bool nxWakeByKarma(ClientPtr client)
{
    if (nxagentUseNXTrans)
    {
        if (client == NULL)
        {
            ErrorF("nxWakeByKarma: Bad client pointer provided to function.\n");

            return False;
        }

#ifdef TEST
        fprintf(stderr, "nxWakeByKarma: Going to wakeup client id [%d].\n",
                    client -> index);
#endif

        AttendClient(client);

        nxagentSleepingByKarma[client -> index] = False;

        nxagentClientPriv(client) -> is_ignored = False;
    }

    return True;
}

int nxagentCheckAnimation(unsigned long int now)
{
    nxagentAnimationCount++;

    if (nxagentAnimationStart == 0)
    {
        nxagentAnimationStart = now;

#ifdef TEST
        fprintf(stderr, "nxagentCheckAnimation: Initialized a new animation start at [%lu].\n",
                    nxagentAnimationStart);
#endif
    }
    else if (now - nxagentAnimationStart > 200)
    {
#ifdef TEST
        fprintf(stderr, "nxagentCheckAnimation: Reset animation start after [%lu] ms.\n",
                    now - nxagentAnimationStart);
#endif

        nxagentAnimationStart = 0;
        nxagentAnimationCount = 0;
    }
    else if (nxagentAnimationCount > 40)
    {
#ifdef TEST
        fprintf(stderr, "nxagentCheckAnimation: Needing synchronization with counter [%d] after [%lu] ms.\n",
                    nxagentAnimationCount, now - nxagentAnimationStart);
#endif

        nxagentAnimationStart = 0;
        nxagentAnimationCount = 0;

        return 1;
    }

#ifdef TEST
    fprintf(stderr, "nxagentCheckAnimation: Nothing to do with sync counter [%d] after [%lu] ms.\n",
                nxagentAnimationCount, now - (nxagentAnimationStart == 0 ? now : nxagentAnimationStart));
#endif

    return 0;
}

/*
 * Replaces ProcGetInputFocus from dix's events.c.
 */

int nxProcGetInputFocus(ClientPtr client)
{
    xGetInputFocusReply *rep;

    REQUEST(xReq);

    FocusClassPtr focus = inputInfo.keyboard -> focus;

    /*
     * Get rid of warning on unused variable stuff.
     */

    stuff = stuff;

    if ((rep = xalloc(sizeof(xGetInputFocusReply))) == NULL)
    {
        FatalError("nxProcGetInputFocus: Can't allocate memory for reply.\n");
    }

#ifdef TEST
    fprintf(stderr, "nxProcGetInputFocus: GetInputFocus request from client [%d].\n",
                client -> index);
#endif

    REQUEST_SIZE_MATCH(xReq);

    rep -> type = X_Reply;
    rep -> length = 0;
    rep -> sequenceNumber = client -> sequence;

    if (focus -> win == NoneWin)
    {
        rep -> focus = None;
    }
    else if (focus -> win == PointerRootWin)
    {
        rep -> focus = PointerRoot;
    }
    else
    {
        rep -> focus = focus -> win -> drawable.id;
    }

    rep -> revertTo = focus -> revert;

    if (nxagentUseNXTrans)
    {
       /*
        * Synchronize with real X server in case clients
        * use round-trips to perform timing of animations.
        * This is the case of window minimization in KDE.
        */

        unsigned long now = GetTimeInMillis();

        if (nxagentCheckAnimation(now) > 0)
        {
#ifdef TEST
            fprintf(stderr, "nxProcGetInputFocus: Synchronizing with real X server.\n");
#endif

            /*
             * Never recur to a round-trip in the first
             * 30 seconds since session startup.
             */

            if (now - nxagentStartTime > 30000)
            {
/*
FIXME: How does it perform without this workaround?

              XSync(nxagentDisplay, False);
*/
            }

#ifdef TEST
            fprintf(stderr, "nxProcGetInputFocus: Sending immediate reply to client [%d].\n",
                        client -> index);
#endif

            nxagentWriteReplyToClient(client, sizeof(xGetInputFocusReply), rep);

            xfree(rep);

            return Success;
        }

        /*
         * If strict control is disabled wait for sync
         * events only if proxy is in congestion state.
         */

        if (nxagentStrictControl == True || nxagentCongestion == True)
        {
            if (++nxagentSyncCounter % nxagentSyncModulo == 0)
            {
                if (nxSleepByGetIFocus(client, (pointer) rep) == True)
                {
                    return Success;
                }

#ifdef TEST
                fprintf(stderr, "nxProcGetInputFocus: Failed to stop client [%d].\n",
                            client -> index);
#endif
            }
        }
    }

#ifdef TEST
    fprintf(stderr, "nxProcGetInputFocus: Sending immediate reply to client [%d].\n",
                client -> index);
#endif

    nxagentWriteReplyToClient(client, sizeof(xGetInputFocusReply), rep);

    xfree(rep);

    return Success;
}

static Bool nxagentCheckCongestionPredicate(Display *display, XEvent *event, XPointer ptr)
{
  return (event -> type == ClientMessage &&
              event -> xclient.data.l[0] == NXCongestionNotify &&
                  event -> xclient.window == 0 && event -> xclient.message_type == 0 &&
                      event -> xclient.format == 32);
}

void nxagentCheckCongestion()
{
    XEvent event;

#ifdef TEST
    fprintf(stderr, "nxagentCheckCongestion: Checking congestion with state [%d].\n",
                nxagentCongestion);
#endif

    /*
     * TODO: Check in the event queue only if a
     * signal was previously delivered by the
     * NX transport. Signal is not sent in cur-
     * rent version of proxy code.
     *
     *
     * if (_X11TransSocketCheckSignal() != SIGURG)
     * {
     *   #ifdef TEST
     *   fprintf(stderr, "nxagentCheckCongestion: Skipping check on event queue.\n",
     *               nxagentCongestion);
     *   #endif
     *
     *   return;
     * }
     */

    /*
     * Calling XCheckIfEvent() will flush our
     * display buffer. 
     */

    while (XCheckIfEvent(nxagentDisplay, &event, nxagentCheckCongestionPredicate, NULL))
    {
        int state = (int) event.xclient.data.l[1];

#ifdef TEST
        fprintf(stderr, "nxagentCheckCongestion: NXCongestionNotify received with state [%d].\n",
                    state);
#endif

        nxagentCongestion = state;

        nxagentAdjustKarma();

        return;
    }

#ifdef TEST
    fprintf(stderr, "nxagentCheckCongestion: NXCongestionNotify event not found in queue.\n");
#endif
}

void nxagentAdjustKarma()
{
    if (nxagentStrictControl == False)
    {
        if (nxagentCongestion == True)
        {
            nxagentStopKarmaSz = nxagentStopKarmaSzDefault;
        }
        else
        {
            nxagentStopKarmaSz = nxagentStopKarmaSzDefault << 1;
        }
    }
}

void nxagentCheckKarma(ClientPtr client, unsigned long now)
{
#ifdef DEBUG
    fprintf(stderr, "nxagentCheckKarma: nxagentFocusKarma = [%d], nxagentFocusKarmaOnStart = [%d].\n",
                nxagentFocusKarma, nxagentFocusKarmaOnStart);

    fprintf(stderr, "nxagentCheckKarma: nxagentStartTime = [%ld], now = [%ld].\n",
                nxagentStartTime, now);

    fprintf(stderr, "nxagentCheckKarma: nxagentStopKarmaSzDefault = [%ld], nxagentStopKarmaSzCheck = [%ld].\n",
                nxagentStopKarmaSzDefault, nxagentStopKarmaSzCheck);

    fprintf(stderr, "nxagentCheckKarma: nxagentStopKarmaSz = [%ld], nxagentKarmaQueue[%d] = [%ld].\n",
                nxagentStopKarmaSz, client -> index, nxagentKarmaQueue[client -> index]);
#endif

    if (client -> index > MAX_CONNECTIONS || nxagentStopKarmaSz <= 0)
    {
#ifdef TEST
        fprintf(stderr, "nxagentCheckKarma: Karma disabled or invalid client index [%d].\n",
                    client -> index);
#endif

        return;
    }

    /*
     * Don't apply karma on focus during the
     * first 120 seconds since session startup.
     */

    if (nxagentFocusKarmaOnStart == True &&
            nxagentStartTime + 120000 < now)
    {
        nxagentFocusKarmaOnStart = False;
    }

    /*
     * If the agent window doesn't have the focus or
     * it is not visible since 30 seconds, then save
     * the wasted bandwidth by sleeping often.
     */

    if (nxagentUsesFocusKarma == True && nxagentFocusKarma == True &&
            nxagentFocusKarmaTimeOut + 30000 < now && nxagentFocusKarmaOnStart == False &&
                    nxagentKarmaQueue[client -> index] > nxagentStopKarmaSzCheck)
    {
        struct timespec req, rem;

        if (ClientIsAsleep(client) == False)
        {
#ifdef TEST
            fprintf(stderr, "nxagentCheckKarma: Suspending client id [%d] by focus with queue of [%ld] bytes.\n",
                        client -> index, nxagentKarmaQueue[client -> index]);
#endif

            nxSleepByKarma(client);

            nxagentKarmaQueue[client -> index] %= nxagentStopKarmaSzCheck;
        }

        req.tv_sec  = 0;
        req.tv_nsec = nxagentStopKarmaSleep * 1000;

#ifdef TEST
        fprintf(stderr, "nxagentCheckKarma: Suspending agent by focus for [%ld] milliseconds.\n",
                    nxagentStopKarmaSleep / 1000);
#endif

        while (nanosleep(&req, &rem) == -1)
        {
            if (errno != EINTR) break;

            req.tv_nsec = rem.tv_nsec;
        }
    }
    else
    {
        if (nxagentKarmaQueue[client -> index] > nxagentStopKarmaSz)
        {
            /*
             * Client has exceeded its karma threshold.
             */

            if (ClientIsAsleep(client) == False)
            {
#ifdef TEST
                fprintf(stderr, "nxagentCheckKarma: Suspending client id [%d] with queue of [%ld] bytes.\n",
                            client -> index, nxagentKarmaQueue[client -> index]);
#endif

                nxSleepByKarma(client);

                nxagentKarmaQueue[client -> index] %= nxagentStopKarmaSz;
            }
        }
        else if (nxagentStrictControl == False &&
                     nxagentKarmaQueue[client -> index] > nxagentStopKarmaSzDefault)
        {
            /*
             * Data is not exceeding the threshold but
             * is probably enough to make a full packet.
             */

#ifdef TEST
            fprintf(stderr, "nxagentCheckKarma: Flushing due to client id [%d] with queue of [%ld] bytes.\n",
                        client -> index, nxagentKarmaQueue[client -> index]);
#endif

            XFlush(nxagentDisplay);

            nxagentKarmaQueue[client -> index] %= nxagentStopKarmaSzDefault;
        }
    }
}

void nxagentSetNoCacheDialogTimeout(int linkType)
{
  switch (linkType)
  {
    case LINK_TYPE_MODEM:
    {
      nxagentNoCacheDialogTimeout = 30000;

      break;
    }
    case LINK_TYPE_ISDN:
    {
      nxagentNoCacheDialogTimeout = 20000;

      break;
    }
    case LINK_TYPE_ADSL:
    {
      nxagentNoCacheDialogTimeout = 10000;

      break;
    }
    default:
    {
      nxagentNoCacheDialogTimeout = 5000;

      break;
    }
  }
}

void nxagentShowNoCacheDialog()
{
  nxagentLaunchDialog(DIALOG_NOCACHE);

  nxagentNoCacheDialogEnable = False;
}
