/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com.          */
/*                                                                        */
/* NXAGENT, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <X11/Xlib.h>

#include "opaque.h"
#include "Args.h"
#include "Dialog.h"

#include NXAGENT_NX_INCLUDE

/*
 * Set here the required log level.
 */

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

int nxagentKillDialogPid = 0;
int nxagentSuspDialogPid = 0;
int nxagentRootlessDialogPid = 0;
int nxagentPulldownDialogPid = 0;
int nxagentSlowModeDialogPid = 0;
int nxagentFastModeDialogPid = 0;
int nxagentEnableRandRModeDialogPid = 0;
int nxagentDisableRandRModeDialogPid = 0;
int nxagentEnableLazyModePid = 0;
int nxagentDisableLazyModePid = 0;

static int nxagentFailedReconnectionDialogPid = 0;

char nxagentPulldownWindow[16];

char nxagentFailedReconnectionMessage[256];

void nxagentResetDialog(int pid)
{
  if (pid == nxagentRootlessDialogPid)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentResetDialog: Resetting rootless dialog pid [%d].\n", nxagentRootlessDialogPid);
    #endif

    nxagentRootlessDialogPid = 0;
  }
  else if (pid == nxagentPulldownDialogPid)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentResetDialog: Resetting pulldown dialog pid [%d].\n", nxagentPulldownDialogPid);
    #endif

    nxagentPulldownDialogPid = 0;
  }
  else if (pid == nxagentKillDialogPid)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentResetDialog: Resetting kill dialog pid [%d].\n", nxagentKillDialogPid);
    #endif

    nxagentKillDialogPid = 0;
  }
  else if (pid == nxagentSuspDialogPid)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentResetDialog: Resetting suspend dialog pid [%d].\n", nxagentSuspDialogPid);
    #endif

    nxagentSuspDialogPid = 0;
  }
  else if (pid == nxagentSlowModeDialogPid)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentResetDialog: Resetting slow mode dialog pid [%d].\n", nxagentSlowModeDialogPid);
    #endif

    nxagentSlowModeDialogPid = 0;
  }
  else if (pid == nxagentFastModeDialogPid)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentResetDialog: Resetting fast mode dialog pid [%d].\n", nxagentFastModeDialogPid);
    #endif

    nxagentFastModeDialogPid = 0;
  }
  else if (pid == nxagentFailedReconnectionDialogPid)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentResetDialog: Resetting Failed Reconnection dialog pid [%d].\n",
                nxagentFailedReconnectionDialogPid);
    #endif

    nxagentFailedReconnectionDialogPid = 0;
  }
  else if (pid == nxagentEnableRandRModeDialogPid)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentResetDialog: Resetting RandR mode dialog pid [%d].\n",
                nxagentEnableRandRModeDialogPid);
    #endif

    nxagentEnableRandRModeDialogPid = 0;
  }
  else if (pid == nxagentDisableRandRModeDialogPid)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentResetDialog: Resetting NoRandR mode dialog pid [%d].\n",
                nxagentDisableRandRModeDialogPid);
    #endif

    nxagentDisableRandRModeDialogPid = 0;
  }
  else if (pid == nxagentEnableLazyModePid)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentResetDialog: Resetting enable lazy mode dialog pid [%d].\n",
                nxagentEnableLazyModePid);
    #endif

    nxagentEnableLazyModePid = 0;
  }
  else if (pid == nxagentDisableLazyModePid)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentResetDialog: Resetting disable lazy mode dialog pid [%d].\n",
                nxagentDisableLazyModePid);
    #endif

    nxagentDisableLazyModePid = 0;
  }
}

void nxagentLaunchDialog(DialogType dialogType)
{
  char dialogDisplay[256];
  sigset_t set, oldSet;
  int *pid;
  char *type;
  char *message;
  int local;

  const char *window = NULL;

  switch (dialogType)
  {
    case DIALOG_KILL_SESSION:
    {
      message = DIALOG_KILL_SESSION_MESSAGE;
      type = DIALOG_KILL_SESSION_TYPE;
      local = DIALOG_KILL_SESSION_LOCAL;
      pid = &nxagentKillDialogPid;

      break;
    }
    case DIALOG_SUSPEND_SESSION:
    {
      message = DIALOG_SUSPEND_SESSION_MESSAGE;
      type = DIALOG_SUSPEND_SESSION_TYPE;
      local = DIALOG_SUSPEND_SESSION_LOCAL;
      pid = &nxagentSuspDialogPid;

      break;
    }
    case DIALOG_ROOTLESS:
    {
      message = DIALOG_ROOTLESS_MESSAGE;
      type = DIALOG_ROOTLESS_TYPE;
      local = DIALOG_ROOTLESS_LOCAL;
      pid = &nxagentRootlessDialogPid;

      break;
    }
    case DIALOG_PULLDOWN:
    {
      message = DIALOG_PULLDOWN_MESSAGE;
      type = DIALOG_PULLDOWN_TYPE;
      local = DIALOG_PULLDOWN_LOCAL;
      pid = &nxagentPulldownDialogPid;
      window = nxagentPulldownWindow;

      break;
    }
    case DIALOG_SLOWMODE:
    {
      message = DIALOG_SLOWMODE_MESSAGE;
      type = DIALOG_SLOWMODE_TYPE;
      local = DIALOG_SLOWMODE_LOCAL;
      pid = &nxagentSlowModeDialogPid;

      break;
    }
    case DIALOG_FASTMODE:
    {
      message = DIALOG_FASTMODE_MESSAGE;
      type = DIALOG_FASTMODE_TYPE;
      local = DIALOG_FASTMODE_LOCAL;
      pid = &nxagentFastModeDialogPid;

      break;
    }
    case DIALOG_FAILED_RECONNECTION:
    {
      message = DIALOG_FAILED_RECONNECTION_MESSAGE;
      type = DIALOG_FAILED_RECONNECTION_TYPE;
      local = DIALOG_FAILED_RECONNECTION_LOCAL;
      pid = &nxagentFailedReconnectionDialogPid;

      break;
    }
    case DIALOG_ENABLE_DESKTOP_RESIZE_MODE:
    {
      message = DIALOG_ENABLE_DESKTOP_RESIZE_MODE_MESSAGE;
      type = DIALOG_ENABLE_DESKTOP_RESIZE_MODE_TYPE;
      local = DIALOG_ENABLE_DESKTOP_RESIZE_MODE_LOCAL;
      pid = &nxagentEnableRandRModeDialogPid;

      break;
    }
    case DIALOG_DISABLE_DESKTOP_RESIZE_MODE:
    {
      message = DIALOG_DISABLE_DESKTOP_RESIZE_MODE_MESSAGE;
      type = DIALOG_DISABLE_DESKTOP_RESIZE_MODE_TYPE;
      local = DIALOG_DISABLE_DESKTOP_RESIZE_MODE_LOCAL;
      pid = &nxagentDisableRandRModeDialogPid;

      break;
    }
    case DIALOG_ENABLE_LAZY_MODE:
    {
      message = DIALOG_ENABLE_LAZY_MODE_MESSAGE;
      type = DIALOG_ENABLE_LAZY_MODE_TYPE;
      local = DIALOG_ENABLE_LAZY_MODE_LOCAL;
      pid = &nxagentEnableLazyModePid;

      break;
    }
    case DIALOG_DISABLE_LAZY_MODE:
    {
      message = DIALOG_DISABLE_LAZY_MODE_MESSAGE;
      type = DIALOG_DISABLE_LAZY_MODE_TYPE;
      local = DIALOG_DISABLE_LAZY_MODE_LOCAL;
      pid = &nxagentDisableLazyModePid;

      break;
    }
    default:
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentLaunchDialog: Unknown Dialog type [%d].\n", dialogType);
      #endif

      return;
    }
  }

  #ifdef TEST
  fprintf(stderr, "nxagentLaunchDialog: Launching dialog type [%d] message [%s].\n", type, message);
  #endif

  if (dialogType == DIALOG_FAILED_RECONNECTION)
  {
    strncpy(dialogDisplay, nxagentDisplayName, 255);
  }
  else
  {
    strcpy(dialogDisplay, ":");
    strncat(dialogDisplay, display, 254);
  }

  *(dialogDisplay + 255) = '\0';

  /*
   * We don't want to receive SIGCHLD
   * before we store the child pid.
   */

  sigemptyset(&set);

  sigaddset(&set, SIGCHLD);

  sigprocmask(SIG_BLOCK, &set, &oldSet);

  *pid = NXTransDialog(nxagentDialogName, message, window,
                           type, local, dialogDisplay);

  #ifdef WARNING
  fprintf(stderr, "nxagentLaunchDialog: Launched dialog %s with pid [%d] on display %s.\n",
              DECODE_DIALOG_TYPE(dialogType), *pid, dialogDisplay);
  #endif

  *dialogDisplay = '\0';

  /*
   * Restore the previous set of
   * blocked signal. 
   */

  sigprocmask(SIG_SETMASK, &oldSet, NULL);
}

void nxagentPulldownDialog(Window wid)
{
  snprintf(nxagentPulldownWindow, 15, "%ld", wid);
  nxagentPulldownWindow[15] = 0;

  #ifdef WARNING
  fprintf(stderr, "nxagentPulldownDialog: Going to launch "
              "pulldown dialog on window [%s].\n", nxagentPulldownWindow);
  #endif

  nxagentLaunchDialog(DIALOG_PULLDOWN);
  nxagentPulldownWindow[0] = 0;
}

void nxagentFailedReconnectionDialog(int alert, char *error)
{
  if (NXTransRunning())
  {
    NXTransAlert(alert, NX_ALERT_REMOTE);

    while (NXTransRunning())
    {
      int ret;
      struct timeval timeout;

      timeout.tv_sec  = 30;
      timeout.tv_usec = 0;

      ret = NXTransContinue(&timeout);
    }
  }
  else
  {
    int pid;
    int status;
    int options = 0;

    snprintf(nxagentFailedReconnectionMessage, 255, "Reconnection Failed: %s", error);

    *(nxagentFailedReconnectionMessage + 255) = '\0';

    nxagentLaunchDialog(DIALOG_FAILED_RECONNECTION);

    while ((pid = waitpid(nxagentFailedReconnectionDialogPid, &status, options)) == -1 &&
               errno == EINTR);

    if (pid == -1)
    {
      if (errno == ECHILD)
      {
        #ifdef WARNING
        fprintf(stderr, "nxagentFailedReconnectionDialog: Got ECHILD waiting for child %d (Failed Reconnection dialog).\n",
                    nxagentFailedReconnectionDialogPid);
        #endif

        nxagentFailedReconnectionDialogPid = 0;
      }
      else
      {
        fprintf(stderr, "nxagentFailedReconnectionDialog: PANIC! Got unexpected error waiting for child %d: %s.\n",
                    nxagentFailedReconnectionDialogPid, strerror(errno));
      }
    }
    else if (pid > 0)
    {
      if (WIFSTOPPED(status))
      {
        #ifdef WARNING
        fprintf(stderr, "nxagentFailedReconnectionDialog: Child process [%d] was stopped "
                    "with signal [%d].\n", pid, (WSTOPSIG(status)));
        #endif
      }
      else
      {
        #ifdef WARNING

        if (WIFEXITED(status))
        {
          fprintf(stderr, "nxagentFailedReconnectionDialog: Child process [%d] exited "
                      "with status [%d].\n", pid, (WEXITSTATUS(status)));
        }
        else if (WIFSIGNALED(status))
        {
          fprintf(stderr, "nxagentFailedReconnectionDialog: Child process [%d] died "
                      "because of signal [%d].\n", pid, (WTERMSIG(status)));
        }

        #endif

        nxagentResetDialog(pid);
      }
    }
    #ifdef WARNING
    else if (pid == 0)
    {
      fprintf(stderr, "nxagentFailedReconnectionDialog: No own child process exited.\n");
    }
    #endif
  }
}

void nxagentKillDialog(DialogType dialogType)
{
  int pid;

  switch (dialogType)
  {
    case DIALOG_KILL_SESSION:
    {
      pid = nxagentKillDialogPid;

      break;
    }
    case DIALOG_SUSPEND_SESSION:
    {
      pid = nxagentSuspDialogPid;

      break;
    }
    case DIALOG_ROOTLESS:
    {
      pid = nxagentRootlessDialogPid;

      break;
    }
    case DIALOG_PULLDOWN:
    {
      pid = nxagentPulldownDialogPid;

      break;
    }
    case DIALOG_SLOWMODE:
    {
      pid = nxagentSlowModeDialogPid;

      break;
    }
    case DIALOG_FASTMODE:
    {
      pid = nxagentFastModeDialogPid;

      break;
    }
    case DIALOG_FAILED_RECONNECTION:
    {
      pid = nxagentFailedReconnectionDialogPid;

      break;
    }
    case DIALOG_ENABLE_DESKTOP_RESIZE_MODE:
    {
      pid = nxagentEnableRandRModeDialogPid;

      break;
    }
    case DIALOG_DISABLE_DESKTOP_RESIZE_MODE:
    {
      pid = nxagentDisableRandRModeDialogPid;

      break;
    }
    case DIALOG_ENABLE_LAZY_MODE:
    {
      pid = nxagentEnableLazyModePid;

      break;
    }
    case DIALOG_DISABLE_LAZY_MODE:
    {
      pid = nxagentDisableLazyModePid;

      break;
    }
    default:
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentKillDialog: Unknown Dialog type [%d].\n", dialogType);
      #endif

      return;
    }
  }

  if (pid > 0)
  {
    if (kill(pid, SIGTERM) == -1)
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentKillDialog: Failed to kill dialog pid [%d]: %s.\n",
                  pid, strerror(errno));
      #endif
    }
  }
  else
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentKillDialog: Cannot kill dialog type %d pid is [%d].\n", dialogType, pid);
    #endif
  }
}
