/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com.          */
/*                                                                        */
/* NXAGENT, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

/*

Copyright 1993 by Davor Matic

Permission to use, copy, modify, distribute, and sell this software
and its documentation for any purpose is hereby granted without fee,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation.  Davor Matic makes no representations about
the suitability of this software for any purpose.  It is provided "as
is" without express or implied warranty.

*/

#include "X.h"
#include "Xproto.h"
#include "miscstruct.h"
#include "dixstruct.h"
#include "fontstruct.h"
#include "gcstruct.h"
#include "scrnintstr.h"
#include "windowstr.h"
#include "pixmapstr.h"
#include "region.h"
#include "servermd.h"

#include "Agent.h"
#include "Args.h"
#include "Display.h"
#include "Options.h"
#include "Screen.h"
#include "GC.h"
#include "Font.h"
#include "GCOps.h"
#include "Drawable.h"
#include "Window.h"
#include "Visual.h"
#include "Control.h"
#include "Trap.h"
#include "Lazy.h"
#include "Reconnect.h"
#include "Events.h"

#ifdef NXAGENT_MVFB
#include "../../fb/fb.h"
#endif

#include NXAGENT_NXLIB_INCLUDE
#include NXAGENT_NXPACK_INCLUDE

/*
 * Set here the required log level.
 */

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG
#undef  DUMP

/*
 * Preferred image compression settings.
 */

extern int nxagentPackMethod;
extern int nxagentPackQuality;
extern int nxagentTightThreshold;
extern int nxagentReconnectTrap;

/*
 * Never pack images having a data size smaller
 * than this threshold.
 */

#define NXAGENT_IMAGE_PACK_THRESHOLD  768

/*
 * Define this if you want to try to guess the
 * background pixel by using the last image got
 * from the server.
 */

#undef NXAGENT_USES_ASYNC_GET_IMAGE

/*
 * Used to reformat image when connecting to
 * displays having different byte order.
 */

extern void BitOrderInvert(unsigned char *, int);
extern void TwoByteSwap( unsigned char *, register int);
extern void FourByteSwap(register unsigned char *, register int);

/*
 * Store the last visual used to unpack the
 * images per each given client.
 */

static VisualID nxagentUnpackVisualId[MAX_CONNECTIONS];

/*
 * Store alpha data set fro the given client.
 */

typedef struct _UnpackAlpha
{
  int entries;
  unsigned char *data;

} UnpackAlphaRec;

typedef UnpackAlphaRec *UnpackAlphaPtr;

static UnpackAlphaRec *nxagentUnpackAlpha[MAX_CONNECTIONS];

/*
 * Predefined visual used for drawables having 
 * 32 bits depth and an associated picture.
 */

static Visual nxagentAlphaVisual;

/*
 * Store data of the last image got from the
 * real X server. Used to guess content of
 * remote window and avoid a round trip.
 */

XImage *nxagentLastGetImage = NULL;

/*
 * True if we are already waiting for a
 * former GetImage reply.
 */

int nxagentLastGetImageClient = UNDEFINED;

/*
 * These are taken from dispatcher.
 */

#define ClientOrder(client) \
    ((client)->swapped ? !ServerOrder() : ServerOrder())

int ServerOrder(void)
{
  int whichbyte = 1;

  if (*((char *) &whichbyte))
      return LSBFirst;

  return MSBFirst;
}

void nxagentImageReformat(char *base, int nbytes, int bpp, int order)
{
  /*
   * This is used whenever we need to swap the image data.
   * If we got an image from a X server having a different
   * endianess, we will need to redormat the image to match
   * our own image-order so that ProcGetImage can return
   * the expected format to the client.
   */

  switch (bpp)
  {
    case 1:
    {
      if (BITMAP_BIT_ORDER != order)
      {
        #ifdef TEST
        fprintf(stderr, "nxagentReformatImage: Bit order invert with size [%d] "
                    "bits per pixel [%d] byte order [%d].\n", nbytes, bpp, order);
        #endif

        BitOrderInvert((unsigned char *) base, nbytes);
      }

      #if IMAGE_BYTE_ORDER != BITMAP_BIT_ORDER && BITMAP_SCANLINE_UNIT != 8

      nxagentImageReformat(base, nbytes, BITMAP_SCANLINE_UNIT, order);

      #endif

      break;
    }
    case 4:
    case 8:
    {
      break;
    }
    case 16:
    {
      if (IMAGE_BYTE_ORDER != order)
      {
        #ifdef TEST
        fprintf(stderr, "nxagentReformatImage: Two bytes swap with size [%d] "
                    "bits per pixel [%d] byte order [%d].\n", nbytes, bpp, order);
        #endif

        TwoByteSwap((unsigned char *) base, nbytes);
      }

      break;
    }
    case 32:
    {
      if (IMAGE_BYTE_ORDER != order)
      {
        #ifdef TEST
        fprintf(stderr, "nxagentReformatImage: Four bytes swap with size [%d] "
                    "bits per pixel [%d] byte order [%d].\n", nbytes, bpp, order);
        #endif

        FourByteSwap((unsigned char *) base, nbytes);
      }

      break;
    }
  }
}

Visual *nxagentImageVisual(DrawablePtr pDrawable, int depth)
{
  /*
   * We always use our default visual, corresponding to
   * depth of the real display. If the drawable is 32 bits
   * or it has been associated to a Picture we can assume
   * that it is using the special RENDER visual having the
   * alpha channel. In this case we assume a fixed color-
   * mask layout. Is this assumption always true?
   */

  if (nxagentAlphaVisual.red_mask == 0 &&
          nxagentAlphaVisual.green_mask == 0 &&
              nxagentAlphaVisual.blue_mask == 0)
  {
    nxagentAlphaVisual.visualid = FakeClientID(0);

    if (ServerOrder() == LSBFirst)
    {
      nxagentAlphaVisual.red_mask = 0x00ff0000;
      nxagentAlphaVisual.green_mask = 0x0000ff00;
      nxagentAlphaVisual.blue_mask = 0x000000ff;
    }
    else
    {
      nxagentAlphaVisual.red_mask = 0x0000ff00;
      nxagentAlphaVisual.green_mask = 0x00ff0000;
      nxagentAlphaVisual.blue_mask = 0xff000000;
    }

    #ifdef TEST
    fprintf(stderr,"nxagentImageVisual: Set Alpha visual id %ld mask (%ld,%ld,%ld).\n",
                nxagentAlphaVisual.visualid, nxagentAlphaVisual.red_mask,
                    nxagentAlphaVisual.green_mask, nxagentAlphaVisual.blue_mask);
    #endif
  }

  if (nxagentDrawablePicture(pDrawable) != NULL || depth == 32)
  {
    return &nxagentAlphaVisual;
  }
  else
  {
    return nxagentDefaultVisual(pDrawable -> pScreen);
  }
}

int nxagentImageLength(int width, int height, int format, int leftPad, int depth)
{
  int line = 0;

  if (format == XYBitmap)
  {
    line = BitmapBytePad(width + leftPad);
  }
  else if (format == XYPixmap)
  {
    line  = BitmapBytePad(width + leftPad);

    line *= depth;
  }
  else if (format == ZPixmap)
  {
    line = PixmapBytePad(width, depth);
  }

  return line * height;
}

int nxagentImagePad(int width, int format, int leftPad, int depth)
{
  int line = 0;

  if (format == XYBitmap)
  {
    line = BitmapBytePad(width + leftPad);
  }
  else if (format == XYPixmap)
  {
    line = BitmapBytePad(width + leftPad);
  }
  else if (format == ZPixmap)
  {
    line = PixmapBytePad(width, depth);
  }

  return line;
}

unsigned char *nxagentImageAlpha(XImage *ximage)
{
  unsigned char *pAlphaData, *pData;

  int dataSize, pos, i;

  /*
   * Use one byte per pixel.
   */

  dataSize = (ximage -> bytes_per_line * ximage -> height) >> 2;

  pAlphaData = xalloc(dataSize);

  if (pAlphaData == NULL)
  {
    return NULL;
  }

  if (ximage -> byte_order == MSBFirst)
  {
    pos = (ServerOrder() == LSBFirst ? 0 : 3);
  }
  else
  {
    pos = (ServerOrder() == LSBFirst ? 3 : 0);
  }

  pData = pAlphaData;

  for (i = 0; i < dataSize; i++, pData++)
  {
    *pData = ximage -> data[(i * 4) + pos];
  }

  return pAlphaData;
}

void nxagentSetUnpackAlpha(DrawablePtr pDrawable, XImage *pImage, ClientPtr pClient)
{
  int entries;

  unsigned char *pAlphaData;

  if (nxagentDrawablePicture(pDrawable) != NULL || pImage -> depth == 32)
  {
    pAlphaData = nxagentImageAlpha(pImage);

    if (!pAlphaData)
    {
      #ifdef PANIC
      fprintf(stderr, "nxagentSetUnpackAlpha: PANIC! Can't allocate data for the alpha channel.\n");
      #endif

      return;
    }

    entries = (pImage -> bytes_per_line * pImage -> height) >> 2;

    if (!nxagentUnpackAlpha[pClient -> index] ||
            nxagentUnpackAlpha[pClient -> index] -> entries != entries ||
                memcmp(nxagentUnpackAlpha[pClient -> index] -> data,
                           pAlphaData, entries) != 0)
    {
      #ifdef DEBUG
      fprintf(stderr, "nxagentSetUnpackAlpha: Sending alpha channel with width [%d] height [%d] "
                  "bytes per line [%d] entries [%d].\n", pImage -> width, pImage -> height,
                      pImage -> bytes_per_line, entries);
      #endif

      NXSetUnpackAlpha(nxagentDisplay, pClient -> index, entries, pAlphaData);

      if (nxagentUnpackAlpha[pClient -> index])
      {
        xfree(nxagentUnpackAlpha[pClient -> index] -> data);
      }
      else
      {
        nxagentUnpackAlpha[pClient -> index] = xalloc(sizeof(UnpackAlphaRec));
      }

      nxagentUnpackAlpha[pClient -> index] -> data = pAlphaData;
      nxagentUnpackAlpha[pClient -> index] -> entries = entries;
    }
    else
    {
      #ifdef DEBUG
      fprintf(stderr, "nxagentSetUnpackAlpha: Matched alpha channel with [%d] entries.\n",
                  entries);
      #endif

      xfree(pAlphaData);
    }
  }
}

void nxagentFillSpans(pDrawable, pGC, nSpans, pPoints, pWidths, fSorted)
     DrawablePtr pDrawable;
     GCPtr pGC;
     int nSpans;
     xPoint *pPoints;
     int *pWidths;
     int fSorted;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to FillSpans on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
     fbFillSpans(nxagentVirtualDrawable(pDrawable), pGC, nSpans, pPoints, pWidths, fSorted);
  }
  else if (nxagentUsesFrameBuffer(wClient((WindowPtr) pDrawable)))
  {
    fbFillSpans(pDrawable, pGC, nSpans, pPoints, pWidths, fSorted);
  }
#else
  ErrorF("nxagent warning: function nxagentFillSpans not implemented\n");
#endif
}

void nxagentSetSpans(pDrawable, pGC, pSrc, pPoints, pWidths, nSpans, fSorted)
     DrawablePtr pDrawable;
     GCPtr pGC;
     char * pSrc;
     xPoint *pPoints;
     int *pWidths;
     int nSpans;
     int fSorted;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to SetSpans on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
     fbSetSpans(nxagentVirtualDrawable(pDrawable), pGC, pSrc, pPoints, pWidths, nSpans, fSorted);
  }
#else
  ErrorF("nxagent warning: function nxagentSetSpans not implemented\n");
#endif
}

void nxagentGetSpans(pDrawable, maxWidth, pPoints, pWidths, nSpans, pBuffer)
     DrawablePtr pDrawable;
     int maxWidth;
     xPoint *pPoints;
     int *pWidths;
     int nSpans;
     char *pBuffer;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: going to GetSpans on mvfb pixmap = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    fbGetSpans(nxagentVirtualDrawable(pDrawable), maxWidth, pPoints, pWidths, nSpans, pBuffer);
  }
#else
  ErrorF("nxagent warning: function nxagentGetSpans not implemented\n");
#endif
}

#ifdef NXAGENT_QUERYBSIZE
void
nxagentQueryBestSize(class, pwidth, pheight, pScreen)
int class;
unsigned short *pwidth;
unsigned short *pheight;
ScreenPtr pScreen;
{
  unsigned width, test;

  switch(class)
  {
    case CursorShape:
      if (*pwidth > pScreen->width)
        *pwidth = pScreen->width;
      if (*pheight > pScreen->height)
        *pheight = pScreen->height;
      break;
    case TileShape:
    case StippleShape:
      width = *pwidth;
      if (!width) break;
      /* Return the closes power of two not less than what they gave me */
      test = 0x80000000;
      /* Find the highest 1 bit in the width given */
      while(!(test & width))
         test >>= 1;
      /* If their number is greater than that, bump up to the next
       *  power of two */
      if((test - 1) & width)
         test <<= 1;
      *pwidth = test;
      /* We don't care what height they use */
      break;
  }
}
#else
void nxagentQueryBestSize(class, pWidth, pHeight, pScreen)
  int class;
  short *pWidth;
  short *pHeight;
  ScreenPtr pScreen;
{
  unsigned int width, height;

  width = *pWidth;
  height = *pHeight;

  XQueryBestSize(nxagentDisplay, class,
                 nxagentDefaultWindows[pScreen->myNum],
                 width, height, &width, &height);

  *pWidth = width;
  *pHeight = height;
}
#endif /* NXAGENT_QUERYBSIZE */

int nxagentRealizePixmap(PixmapPtr pPixmap)
{
  GCPtr pGC;
  int ret;
  PixmapPtr pReal;
  int lazy = nxagentOption(Lazy);

  #ifdef TEST
  fprintf(stderr, "nxagentRealizePixmap: pixmap is %p real %p virtual %p\n",
              (void*) pPixmap, (void*) nxagentRealPixmap(pPixmap), (void*) nxagentVirtualPixmap(pPixmap));
  #endif

  pReal = nxagentRealPixmap(pPixmap);

  if (nxagentPixmapStatus(pReal) == DRAWABLE_SYNCHRONIZED)
  {
    #ifdef TEST
    fprintf(stderr, "nxagentRealizePixmap: Cannot realize pixmap %p id [%ld] status is SYNCHRONIZED.\n",
                (void*) pReal, pReal -> drawable.id);
    #endif

    return 0;
  }

  pGC = CreateGC((DrawablePtr)pReal, 0, 0, &ret);

  if (ret != Success)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentRealizePixmap: WARNING! Failed to create GC bailing out.\n");
    #endif

    return 0;
  }

  nxagentChangeOption(Lazy, False);
  nxagentFBTrap = True;
  nxagentReconnectPixmapData(pReal, pGC);
  nxagentFBTrap = False;
  nxagentChangeOption(Lazy, lazy);

  FreeGC(pGC, 0);

  return 1;
}

void nxagentPutImage(pDrawable, pGC, depth, dstX, dstY, dstWidth,
                         dstHeight, leftPad, format, data)
  DrawablePtr pDrawable;
  GCPtr       pGC;
  int         depth, dstX, dstY, dstWidth, dstHeight;
  int         leftPad, format;
  char        *data;
{
  Visual *pVisual;

  int length;
  int split;

  int x, y, w, h;

  int bytesPerLine;
  int numSubImages;
  int totalHeight;

  ClientPtr client;

  if (nxagentGCTrap == True && nxagentReconnectTrap == False &&
          nxagentFBTrap == False && nxagentShmTrap == False)
  {
    if ((pDrawable) -> type == DRAWABLE_PIXMAP)
    {
      fbPutImage(nxagentVirtualDrawable(pDrawable), pGC, depth,
                     dstX, dstY, dstWidth, dstHeight, leftPad, format, data);
    }

    return;
  }

  if (pDrawable -> type == DRAWABLE_PIXMAP && 
          nxagentReconnectTrap == False && nxagentFBTrap == False &&
              nxagentShmTrap == False)
  {
    DrawablePtr pVirtual = nxagentVirtualDrawable(pDrawable);

    if (pVirtual -> bitsPerPixel == 0)
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentPutImage: WARNING! Virtual pixmap at [%p] has "
                  "invalid bits per pixel.\n", (void *) pVirtual);

      fprintf(stderr, "nxagentPutImage: WARNING! While handling image for pixmap at [%p].\n",
                  (void *) pDrawable);
      #endif

      return;
    }

    fbPutImage(nxagentVirtualDrawable(pDrawable), pGC, depth,
                   dstX, dstY, dstWidth, dstHeight, leftPad, format, data);
  }

  /*
   * Don't try to paint the image on the drawable
   * if it is already dirty and the image covers
   *  the full surface.
   *
   * if (nxagentOption(Lazy) && nxagentDrawableStatus(pDrawable) == DRAWABLE_NOT_SYNCHRONIZED &&
   *         dstX == 0 && dstY == 0 && pDrawable -> width == dstWidth &&
   *            pDrawable -> height == dstHeight)
   * {
   *   return;
   * }
   */

  if (nxagentOption(LinkType) != LINK_TYPE_NONE &&
          NXImageCacheStat < 0)
  {
    NXCacheInit(100);
  }

  /*
   * Get visual according to drawable and depth.
   */

  pVisual = nxagentImageVisual(pDrawable, depth);

  /*
   * Get bytes per line according to format.
   */

  x = dstX;
  y = dstY;
  w = dstWidth;
  h = dstHeight;

  bytesPerLine = nxagentImagePad(w, format, leftPad, depth);

  totalHeight = h;

  length = bytesPerLine * h;

  /*
   * Tell the proxy to split the image even
   * if the size is below the threshold.
   */

  split = (nxagentOption(LinkType) != LINK_TYPE_NONE &&
               NXDisplayError(nxagentDisplay) == 0 &&
                   nxagentReconnectTrap == 0);

  /*
   * If we don't have a valid client then we
   * disable the splitting and assume the X
   * server's internal client.
   */

  client = requestingClient;

  if (client == NULL)
  {
    client = serverClient;

    #ifdef TEST
    fprintf(stderr, "nxagentPutImage: WARNING! Can't split the image with "
                "client [%d] and length [%d].\n", client -> index, length);
    #endif

    split = 0;
  }

  #ifdef TEST

  if (split == 1)
  {
    fprintf(stderr, "nxagentPutImage: Splitting the image with client [%d] "
                "and length [%d].\n", client -> index, length);
  }
  else if (nxagentOption(LinkType) != LINK_TYPE_NONE)
  {
    fprintf(stderr, "nxagentPutImage: WARNING! Not splitting the image with "
                "client [%d] and length [%d].\n", client -> index, length);
  }

  #endif

  /*
   * Send the start/end split sequence only
   * once. The proxy will tell us to suspend
   * the client, if needed.
   */

  if (split == 1)
  {
    if (nxagentOption(Lazy) == 1)
    {
      NXStartSplit(nxagentDisplay, client -> index, NXSplitModeLazy);
    }
    else
    {
      NXStartSplit(nxagentDisplay, client -> index, NXSplitModeDefault);
    }
  }

  h = MIN(NXAGENT_MAX_REQUEST_EFF_SIZE, length) / bytesPerLine;

  numSubImages = totalHeight / h + 1;

  while (numSubImages > 0)
  {
    nxagentPutSubImage(pDrawable, pGC, depth, x, y, w, h,
                           leftPad, format, data, pVisual, client);
    y += h;

    data += h * bytesPerLine;

    if (--numSubImages == 1)
    {
      h = totalHeight % h;

      if (h == 0)
      {
        break;
      }
    }
  }

  if (split == 1)
  {
    NXEndSplit(nxagentDisplay, client -> index);

    /*
     * Now we need to check if all the messages went
     * straight through the output stream or any of
     * them required a split. In this case we will
     * put the client asleep.
     */

    split = nxagentWaitSplitEvent(client -> index);

    if (split == 1)
    {
      if (nxagentOption(Lazy) == 1)
      {
        nxagentApplyPlaceholder(nxagentDrawable(pDrawable), dstX, dstY,
                                    dstWidth, dstHeight, depth);

        nxagentSetDrawableStatus(pDrawable, DRAWABLE_NOT_SYNCHRONIZED);

        return;
      }
    }
  }

  /*
   * We either suspended the client or had the
   * image successfully drawn without requiring
   * a split. Assume that now the drawable is
   * consistent.
   */

  nxagentSetDrawableStatus(pDrawable, DRAWABLE_SYNCHRONIZED);
}

void nxagentPutSubImage(pDrawable, pGC, depth, x, y, w, h, leftPad,
                            format, data, pVisual, client)
  DrawablePtr pDrawable;
  GCPtr       pGC;
  int         depth, x, y, w, h;
  int         leftPad;
  int         format;
  char        *data;
  Visual      *pVisual;
  ClientPtr   client;
{
  NXPackedImage *packedImage = NULL;
  XImage        *plainImage  = NULL;
  unsigned char *md5         = NULL;

  unsigned int packMethod = nxagentPackMethod;

  /*
   * XCreateImage is the place where the leftPad should be passed.
   * The image data is received from our client unmodified. In
   * theory what we would need to do is just creating an appropri-
   * ate XImage structure based on the incoming data and let Xlib
   * do the rest. Probably we don't have to pass leftPad again in
   * the src_x of XPutImage otherwise the src_x would make Xlib
   * to take into account the xoffset field twice. Unfortunately
   * passing the leftPad doesn't work.
   *
   * plainImage = XCreateImage(nxagentDisplay, pVisual,
   *                           depth, format, leftPad, (char *) data,
   *                           w, h, BitmapPad(nxagentDisplay),
   *                           nxagentImagePad(w, format, leftPad, depth));
   */

   if ((plainImage = XCreateImage(nxagentDisplay, pVisual,
                                  depth, format, leftPad, (char *) data,
                                  w, h, BitmapPad(nxagentDisplay),
                                  nxagentImagePad(w, format, leftPad, depth))) == NULL)

  {
    #ifdef WARNING
    fprintf(stderr, "nxagentPutSubImage: WARNING! Failed to create image.\n");
    #endif

    return;
  }

  #ifdef DEBUG
  fprintf(stderr, "nxagentPutSubImage: Handling image with geometry [%d, %d] depth [%d].\n",
              w, h, depth);

  fprintf(stderr, "nxagentPutSubImage: Default pack method is [%d] quality is [%d].\n",
              nxagentPackMethod, nxagentPackQuality);
  #endif

  /*
   * We got image data from the X client with our own
   * server endianess while the ximage was initialized
   * with the endianess of the remote display. Fix the
   * ximage structure to carry to appropriate flags.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentPutSubImage: Image has order [%d] depth [%d] bits-per-pixel [%d].\n",
              plainImage -> byte_order, plainImage -> depth, plainImage -> bits_per_pixel);
  #endif

  nxagentSetImageFormat(plainImage);

  #ifdef TEST
  fprintf(stderr, "nxagentPutSubImage: Image has now order [%d].\n",
              plainImage -> byte_order);
  #endif

  #ifdef TEST
  fprintf(stderr, "nxagentPutSubImage: Image has visual with RGB color masks [%0lx][%0lx][%0lx].\n",
              pVisual -> red_mask, pVisual -> green_mask, pVisual -> blue_mask);
  #endif

  #ifdef TEST
  fprintf(stderr, "nxagentPutSubImage: Server order is [%d] client order is [%d].\n",
              ServerOrder(), ClientOrder(client));
  #endif

  #ifdef TEST
  fprintf(stderr, "nxagentPutSubImage: Display image order is [%d] bitmap order is [%d].\n",
              ImageByteOrder(nxagentDisplay), BitmapBitOrder(nxagentDisplay));
  #endif

  /*
   * Always clean the image to help any future
   * cache method but don't clean drawables that
   * are used in render extension and that use
   * the alpha channel.
   */

  if ((nxagentOption(LinkType) == LINK_TYPE_NONE ||
           (nxagentDrawablePicture(pDrawable) != NULL && depth == 32) ||
                (nxagentOption(LinkType) == LINK_TYPE_LAN &&
                     nxagentPackMethod == NO_PACK)) == 0)
  {
    NXCleanInPlaceImage(plainImage);
  }

  /*
   * Don't pack images with depth 1 or 8 or having
   * a data size less than the given threshold.
   */

  if (nxagentOption(LinkType) != LINK_TYPE_NONE && depth != 1 && depth != 8 &&
          nxagentImageLength(w, h, format, leftPad, depth) >
              NXAGENT_IMAGE_PACK_THRESHOLD)
  {
    if (nxagentUnpackVisualId[client -> index] != pVisual -> visualid)
    {
      nxagentUnpackVisualId[client -> index] = pVisual -> visualid;

      #ifdef DEBUG
      fprintf(stderr, "nxagentPutSubImage: Sending new geometry for client [%d].\n",
                  client -> index);
      #endif
      
      NXSetUnpackGeometry(nxagentDisplay, client -> index,
                              (Screen *) pDrawable->pScreen, pVisual);
    }

    if (NXImageCacheStat > 0)
    {
      packedImage = NXCacheFindImage(plainImage, &packMethod, &md5);

      if (packedImage != NULL)
      {
        nxagentSetUnpackAlpha(pDrawable, plainImage, client);

        NXPutPackedImage(nxagentDisplay, client -> index, nxagentDrawable(pDrawable),
                             nxagentGC(pGC), packedImage, packMethod, depth,
                                 0, 0, x, y, w, h);

        #ifdef DEBUG
        fprintf(stderr, "nxagentPutSubImage: Calling NXPutPackedImage(%s) size is [%d x %d] data size is [%d]\n",
                    (packMethod >= PACK_PNG_8_COLORS && packMethod <= PACK_PNG_16M_COLORS) ?
                        "PNG" : "JPEG", plainImage -> width, plainImage -> height, packedImage -> xoffset);
        #endif

        XFree(plainImage);

        if (md5 != NULL)
        {
          XFree(md5);
        }

        return;
      }
    }

    if (packMethod > 0 &&
            (packMethod > PACK_TIGHT_16M_COLORS ||
                 packMethod < PACK_TIGHT_8_COLORS))
    {
      if (packMethod >= PACK_JPEG_8_COLORS &&
              packMethod <= PACK_JPEG_16M_COLORS)
      {
        #ifdef DEBUG
        fprintf(stderr, "nxagentPutSubImage: encodingJpeg forced by command line parameter\n");
        #endif

        packMethod = PACK_JPEG_16M_COLORS;

        packedImage = NXEncodeJpeg(plainImage, packMethod, nxagentPackQuality);
      }
      else if (packMethod >= PACK_PNG_8_COLORS &&
                   packMethod <= PACK_PNG_16M_COLORS)
      {
        #ifdef DEBUG
        fprintf(stderr, "nxagentPutSubImage: encodingPng forced by command line parameter\n");
        #endif

        packMethod = PACK_PNG_16M_COLORS;

        packedImage = NXEncodePng(plainImage, packMethod , NULL);
      }
      else if (packMethod >= PACK_PNG_JPEG_8_COLORS &&
                   PACK_PNG_JPEG_16M_COLORS)
      {
        #ifdef DEBUG
        fprintf(stderr, "nxagentPutSubImage: Trying to execute deperecated "
                    "NXDynamicSelectPackMethod(). Falling back to XPutImage.\n");
        #endif
      }
    }
  }

  if (packedImage != NULL &&
          (packMethod < PACK_TIGHT_8_COLORS ||
               packMethod > PACK_TIGHT_16M_COLORS))
  {
    #ifdef DEBUG
    fprintf(stderr, "nxagentPutSubImage: Calling NXPutPackedImage(%s) size is [%d x %d] data size is [%d]\n",
                (packMethod >= PACK_PNG_8_COLORS && packMethod <= PACK_PNG_16M_COLORS) ?
                     "PNG" : "JPEG", plainImage -> width, plainImage -> height, packedImage -> xoffset);
    #endif

    nxagentSetUnpackAlpha(pDrawable, plainImage, client);

    NXPutPackedImage(nxagentDisplay, client -> index, nxagentDrawable(pDrawable),
                         nxagentGC(pGC), packedImage, packMethod, depth,
                             0, 0, x, y, w, h);
  }
  else if (packMethod >= PACK_TIGHT_8_COLORS &&
               packMethod <= PACK_TIGHT_16M_COLORS)
  {
    NXEncodeTight(nxagentDisplay, nxagentDrawable(pDrawable), nxagentDefaultVisual(pDrawable->pScreen),
                      nxagentGC(pGC), plainImage, packMethod, 0, 0, x, y, w, h);

    #ifdef DEBUG
    fprintf(stderr, "nxagentPutSubImage: Calling NXPutPackedImage(TIGHT) size is [%d x %d] "
                "data size is [%d].\n", plainImage -> width, plainImage -> height,
                    packedImage -> xoffset);
    #endif

    NXPutPackedImage(nxagentDisplay, client -> index, nxagentDrawable(pDrawable),
                         nxagentGC(pGC), packedImage, packMethod, depth,
                             0, 0, x, y, w, h);
  }
  else if (packedImage == NULL)
  {
    #ifdef DEBUG
    fprintf(stderr, "nxagentPutSubImage: Calling XPutImage, size is [%d x %d] data size is [%d].\n",
                plainImage -> width, plainImage -> height, plainImage -> bytes_per_line *
                    plainImage -> height);
    #endif

    /*
     * Passing the leftPad value in src_x doesn't work.
     *
     * XPutImage(nxagentDisplay, nxagentDrawable(pDrawable),
     *               nxagentGC(pGC), plainImage, leftPad, 0, x, y, w, h);
     */

    XPutImage(nxagentDisplay, nxagentDrawable(pDrawable),
                  nxagentGC(pGC), plainImage, 0, 0, x, y, w, h);
  }

  if (packedImage)
  {
    if (NXImageCacheStat > 0 && md5 != NULL)
    {
      NXCacheAddImage(packedImage, packMethod, md5);
    }

    XFree(packedImage);
  }
  else
  {
    if (md5 != NULL)
    {
      XFree(md5);
    }
  }

  XFree(plainImage);
}

int nxagentGetPixelHint(DrawablePtr pDrawable, int x, int y, int w, int h,
                            unsigned int format, unsigned long planeMask,
                                unsigned long *pixel)
{
  /*
   * Given the hints, try a suitable pixel. At least
   * with OpenOffice, plane mask seems to be always
   * 0xffffffff, even on 16 bpp displays.
   */

  #ifdef DUMP

  {
    XImage *image;

    image = XGetImage(nxagentDisplay, nxagentDrawable(pDrawable),
                          x, y, w, h, planeMask, format);

    if (image != NULL)
    {
      fprintf(stderr, "nxagentGetPixelHint: Matching [%d, %lx] geometry [%d, %d, %d, %d] pixel [0x%lx].\n",
                  format, planeMask, x, y, w, h, XGetPixel(image, 0, 0));

      XDestroyImage(image);
    }
  }

  #endif

  if (format == ZPixmap && planeMask == 0xffffffff)
  {
    if ((w == 9 && h == 7) || (w == 7 && h == 9))
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice tooltip.\n");
      #endif

      *pixel = nxagentPixelHint[0];

      return 1;
    }
    else if ((w == 10 && h <= 10) || (h == 10 && w <= 10))
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice close button.\n");
      #endif

      *pixel = nxagentPixelHint[0];

      return 1;
    }
    else if (w == 11 && h == 11)
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice toolbar.\n");
      #endif

      *pixel = nxagentPixelHint[0];

      return 1;
    }
    else if (w == 16 && h == 16)
    {
      if (x == 18 || y == 31)
      {
        *pixel = nxagentPixelHint[4];

        #ifdef TEST
        fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice 2.0 menu entry.\n");
        #endif
      }
      else if (x == 8)
      {
        #ifdef TEST
        fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice 2.0 menu entry.\n");
        #endif

        *pixel = nxagentPixelHint[4];
      }
      else if (x <= 8)
      {
        #ifdef TEST
        fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice menu entry.\n");
        #endif

        *pixel = nxagentPixelHint[0];
      }
      else if (y == 24 || y == 51)
      {
        #ifdef TEST
        fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice 2.0 toolbar.\n");
        #endif

        *pixel = nxagentPixelHint[5];
      }
      else
      {
        #ifdef TEST
        fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice open dialog.\n");
        #endif

        *pixel = nxagentPixelHint[1];
      }

      return 1;
    }
    else if (w == 18 && h == 18)
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice mouse over close button.\n");
      #endif

      *pixel = nxagentPixelHint[2];

      return 1;
    }
    else if ((w == 24 && h <= 24) || (h == 24 && w <= 24))
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice toolbar.\n");
      #endif

      *pixel = nxagentPixelHint[0];

      return 1;
    }
    else if ((w == 31 && h <= 31) || (h == 31 && w <= 31))
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice mouse over toolbar.\n");
      #endif

      *pixel = nxagentPixelHint[3];

      return 1;
    }
    else if (w == 32 && h == 32)
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice icon.\n");
      #endif

      *pixel = nxagentPixelHint[0];

      return 1;
    }
  }

  return 0;
}

void nxagentGetDefaultHint(DrawablePtr pDrawable, int x, int y, int w, int h,
                               unsigned int format, unsigned long planeMask,
                                   unsigned long *pixel)
{
  /*
   * We were unable to find out if the query involved
   * a well-known graphic element. We default to an
   * appropriate background.
   */

  if (nxagentLastGetImage)
  {
    #ifdef TEST
    fprintf(stderr, "nxagentGetDefaultHint: WARNING! Trying with data from last request.\n");
    #endif

    *pixel = XGetPixel(nxagentLastGetImage, 0, 0);
  }
  else
  {
    /*
     * Trying to set pixel to the window's background
     * often results in a ugly black. Let's consider
     * preferable a solid grey.
     *
     * *pixel = ((WindowPtr) pDrawable) -> background.pixel;
     */

    if (w >= h * 4)
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetDefaultHint: WARNING! Trying with default white pixel.\n");
      #endif

      *pixel = nxagentPixelHint[1];
    }
    else
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetDefaultHint: WARNING! Trying with default grey pixel.\n");
      #endif

      *pixel = nxagentPixelHint[0];
    }
  }
}

void nxagentGetDefaultImage(DrawablePtr pDrawable, int x, int y, int w, int h,
                                unsigned int format, unsigned long planeMask,
                                    unsigned long *hint, char *data)
{
  unsigned long pixel;
  int depth;

  Visual *pVisual;
  XImage *image;

  #ifdef TEST
  fprintf(stderr, "nxagentGetDefaultImage: Called with drawable at [%p] geometry [%d, %d, %d, %d].\n",
              (void *) pDrawable, x, y, w, h);

  fprintf(stderr, "nxagentGetDefaultImage: Format is [%d] plane mask is [%lx].\n",
              format, planeMask);
  #endif

  /*
   * If a hint is provided use the pixel color to
   * fill the background, else try to match a pix-
   * el according to the well-known size of graph-
   * ic elements used by common applications.
   */

  /*
   * FIXME: The hint was used when we were getting
   * the image asynchronously. It could be easily
   * removed.
   */

  if (hint != NULL)
  {
    pixel = *hint;
  }
  else
  {
    if (nxagentGetPixelHint(pDrawable, x, y, w, h, format, planeMask, &pixel) == 0)
    {
      nxagentGetDefaultHint(pDrawable, x, y, w, h, format, planeMask, &pixel);
    }
  }

  pVisual = nxagentImageVisual(pDrawable, pDrawable -> depth);

  if (format == XYPixmap)
  {
    depth = Ones(planeMask);
    depth = MIN(depth, pDrawable -> depth);
  }
  else
  {
    depth = pDrawable -> depth;
  }

  if (!(image = XCreateImage(nxagentDisplay, pVisual,
                             depth, format, 0, data,
                             w, h, BitmapPad(nxagentDisplay),
                             nxagentImagePad(w, format, 0, depth))))
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentGetDefaultImage: WARNING! Failed to create image.\n");
    #endif

    return;
  }

  /*
   * Reuse x and y as indices. Note that image data
   * points to the buffer created in dispatcher.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentGetDefaultImage: Filling image with pixel [0x%lx].\n",
              pixel);
  #endif

  /*
   * FIXME: This could be easily replaced with a
   * PolyFillRectangle using the given pixel.
   */

  for (y = 0; y < h; y++)
  {
    for (x = 0; x < w; x++)
    {
      XPutPixel(image, x, y, pixel);
    }
  }

  nxagentImageReformat(image -> data, image -> bytes_per_line * image -> height, 
                           format == ZPixmap ? BitsPerPixel(image -> depth) : 1, 
                               image -> byte_order);

  /*
   * Don't free the image data as buffer belongs
   * to the caller.
   */

  XFree(image);
}

/*
 * This simpler version will always try to guess
 * the background pixel based on the geometry of
 * the requested area.
 */

#ifndef NXAGENT_USES_ASYNC_GET_IMAGE

void nxagentGetImage(DrawablePtr pDrawable, int x, int y, int w, int h,
                         unsigned int format, unsigned long planeMask, char *data)
{
  XImage *image;

  #ifdef TEST
  fprintf(stderr, "nxagentGetImage: Called with drawable at [%p] geometry [%d, %d, %d, %d].\n",
              (void *) pDrawable, x, y, w, h);

  fprintf(stderr, "nxagentGetImage: Format is [%d] plane mask is [%lx].\n",
              format, planeMask);
  #endif

  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
    #ifdef TEST
    fprintf(stderr, "nxagentGetImage: Going to get image from virtual pixmap at [%p].\n",
                (void *) nxagentVirtualDrawable(pDrawable));
    #endif

    fbGetImage(nxagentVirtualDrawable(pDrawable), x, y, w, h, format, planeMask, data);

    return;
  }
  else if (nxagentFastGetImageEnable &&
               nxagentUsesFrameBuffer(wClient((WindowPtr) pDrawable)))
  {
    fbGetImage(pDrawable, x, y, w, h, format, planeMask, data);

    return;
  }

  /*
   * If fast get image is enabled always try
   * to guess the background.
   */

  if (nxagentFastGetImageEnable)
  {
    nxagentGetDefaultImage(pDrawable, x, y, w, h, format, planeMask, NULL, data);

    return;
  }

  /*
   * If fast get image is disabled, grab the window's content
   * from the real X server. This is really clumsy. In theory
   * X clients should never do that (in fact if agent window
   * is covered it will not work). In practice a few programs
   * and libraries (like surprisingly a famous Linux office
   * automation suite) do, mainly to compose images with the
   * window's backgound. Do they really need this? Why don't
   * they combine content into a Pixmap?
   *
   * Enable the three logs you find below to get the value of
   * the top-left pixel got from the remote server. You can
   * later use the value as a hint when running with "fast"
   * mode enabled.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentGetImage: WARNING! Going to get image from window [%ld] at [%p].\n",
              nxagentWindow((WindowPtr) pDrawable), (void *) pDrawable);

  fprintf(stderr, "nxagentGetImage: WARNING! Geometry [%d, %d, %d, %d] format [%d] plane mask [%lx].\n",
              x, y, w, h, format, planeMask);
  #endif

  image = XGetImage(nxagentDisplay, nxagentDrawable(pDrawable),
                        x, y, w, h, planeMask, format);

  if (!image)
  {
    #ifdef TEST
    fprintf(stderr, "nxagentGetImage: WARNING! Failed to get image. Returning undefined data.\n");
    #endif

    return;
  }

  nxagentImageReformat(image -> data, image -> bytes_per_line * image -> height, 
                           format == ZPixmap ? BitsPerPixel(image -> depth) : 1,
                               image -> byte_order);

  memmove(data, image -> data, image -> bytes_per_line * image -> height);

  /*
   * Image is destroyed. The pointer to the
   * last image got from server will always
   * be NULL.
   */

  XDestroyImage(image);

  return;
}

#else

void nxagentGetImage(DrawablePtr pDrawable, int x, int y, int w, int h,
                         unsigned int format, unsigned long planeMask, char *data)
{
  unsigned long pixel;

  Visual *pVisual;
  XImage *image;

  #ifdef TEST
  fprintf(stderr, "nxagentGetImage: Called with drawable at [%p] geometry [%d, %d, %d, %d].\n",
              (void *) pDrawable, x, y, w, h);

  fprintf(stderr, "nxagentGetImage: Format is [%d] plane mask is [%lx].\n",
              format, planeMask);
  #endif

  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
    #ifdef TEST
    fprintf(stderr, "nxagentGetImage: Going to get image from virtual pixmap at [%p].\n",
                (void *) nxagentVirtualDrawable(pDrawable));
    #endif

    fbGetImage(nxagentVirtualDrawable(pDrawable), x, y, w, h, format, planeMask, data);

    return;
  }
  else if (nxagentFastGetImageEnable &&
               nxagentUsesFrameBuffer(wClient((WindowPtr) pDrawable)))
  {
    fbGetImage(pDrawable, x, y, w, h, format, planeMask, data);

    return;
  }

  /*
   * If fast get image is enabled try
   * to get a hint first.
   */

  if (nxagentFastGetImageEnable)
  {
    if (nxagentGetPixelHint(pDrawable, x, y, w, h, format, planeMask, &pixel))
    {
      nxagentGetDefaultImage(pDrawable, x, y, w, h, format, planeMask, &pixel, data);

      return;
    }
  }

  /*
   * If fast get image is disabled, grab the window's content
   * from the real X server. This is really clumsy. In theory
   * X clients should never do that (in fact if agent window
   * is covered it will not work). In practice a few programs
   * and libraries (like surprisingly a famous Linux office
   * automation suite) do, mainly to compose images with the
   * window's backgound. Do they really need this? Why don't
   * they combine content into a Pixmap?
   */

  if (!nxagentFastGetImageEnable || !nxagentLastGetImage)
  {
    #ifdef TEST
    fprintf(stderr, "nxagentGetImage: WARNING! Going to get image from window [%ld] at [%p].\n",
                nxagentWindow((WindowPtr) pDrawable), (void *) pDrawable);
    #endif

    /*
     * If fast get image is enabled but we don't have
     * a previous image to use as reference, we will
     * have to recur to an expensive round trip. Note
     * anyway that we will only collect one pixel.
     */

    if (nxagentFastGetImageEnable)
    {
      image = XGetImage(nxagentDisplay, nxagentDrawable(pDrawable),
                            x, y, 1, 1, planeMask, format);
    }
    else
    {
      image = XGetImage(nxagentDisplay, nxagentDrawable(pDrawable),
                            x, y, w, h, planeMask, format);
    }

    if (!image)
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetImage: WARNING! Failed to get image. Returning undefined data.\n");
      #endif

      return;
    }

    #ifdef TEST
    fprintf(stderr, "nxagentGetImage: Got pixel [0x%lx] with geometry [%d, %d, %d, %d] "
                "format [%d] mask [%lx].\n", XGetPixel(image, 0, 0), x, y, w, h,
                    format, planeMask);
    #endif

    /*
     * Always save the image. It will be used
     * whenever the agent window is covered.
     */

    #ifdef TEST
    fprintf(stderr, "nxagentGetImage: Saving last image collected from real server.\n");
    #endif

    if (nxagentLastGetImage)
    {
      XDestroyImage(nxagentLastGetImage);
    }

    nxagentLastGetImage = image;

    if (!nxagentFastGetImageEnable)
    {
      nxagentImageReformat(image -> data, image -> bytes_per_line * image -> height, 
                               format == ZPixmap ? BitsPerPixel(image -> depth) : 1,
                                   image -> byte_order);

      memmove(data, image -> data, image -> bytes_per_line * image -> height);

      return;
    }
  }

  /*
   * Fast get image is enabled. We either collected
   * the upper left pixel from the server or we are
   * going to use a previously collected image.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentGetImage: WARNING! Skipping get image from window [%ld] at [%p].\n",
              nxagentWindow((WindowPtr) pDrawable), (void *) pDrawable);
  #endif

  pixel = XGetPixel(nxagentLastGetImage, 0, 0);

  pVisual = nxagentImageVisual(pDrawable, pDrawable -> depth);

  if (!(image = XCreateImage(nxagentDisplay, pVisual,
                             pDrawable -> depth, format, 0, data,
                             w, h, BitmapPad(nxagentDisplay),
                             nxagentImagePad(w, format, 0, pDrawable -> depth))))
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentGetImage: WARNING! Failed to create image.\n");
    #endif

    return;
  }

  /*
   * Guess the content of the new image by filling
   * data with the pixel value got from the last
   * image.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentGetImage: WARNING! Trying with data from last request.\n");
  #endif

  /*
   * Reuse x and y as indices. Note that image data
   * points to the buffer created in dispatcher.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentGetImage: Filling image with pixel [0x%lx].\n",
              pixel);
  #endif

  for (y = 0; y < h; y++)
  {
    for (x = 0; x < w; x++)
    {
      XPutPixel(image, x, y, pixel);
    }
  }

  nxagentImageReformat(image -> data, image -> bytes_per_line * image -> height, 
                           format == ZPixmap ? BitsPerPixel(image -> depth) : 1, 
                               image -> byte_order);

  /*
   * Don't free the image data as buffer belongs
   * to the caller.
   */

  XFree(image);

  /*
   * Finally initiate an aynchronous get image
   * request. A notification will be received
   * as soon as image data is available.
   */

  if (nxagentLastGetImageClient == UNDEFINED)
  {
    NXCollectImage(nxagentDisplay, requestingClient -> index,
                        nxagentDrawable(pDrawable), x, y, 1, 1,
                            planeMask, format);

    nxagentLastGetImageClient = requestingClient -> index;
  }
}

#endif

void nxagentCollectImageEvent(int resource)
{
  XImage *image;

  if (!NXGetCollectedImage(nxagentDisplay, resource, &image))
  {
    #ifdef PANIC
    fprintf(stderr, "nxagentCollectImageEvent: PANIC! Failed to get collected image for resource [%d].\n",
                resource);
    #endif

    return;
  }

  nxagentLastGetImageClient = UNDEFINED;

  if (nxagentLastGetImage)
  {
    XDestroyImage(nxagentLastGetImage);
  }

  nxagentLastGetImage = image;
}

#ifndef NXAGENT_NOEXPOSEOPTIMIZE

static Bool nxagentBitBlitPredicate(display, event, args)
  Display *display;
  XEvent *event;
  char *args;
{
  return (event->type == GraphicsExpose || event->type == NoExpose);
}

#endif

RegionPtr nxagentBitBlitHelper(pGC)
  GC *pGC;
{
  #ifndef NXAGENT_NOEXPOSEOPTIMIZE

  XEvent event;
  RegionPtr pReg, pTmpReg;
  BoxRec Box;
  Bool pending, overlap;

  #endif

  #ifdef TEST
  fprintf(stderr, "nxagentBitBlitHelper: Called for GC at [%p].\n", (void *) pGC);
  #endif

  #ifdef NXAGENT_NOEXPOSEOPTIMIZE

  /*
   * Force NullRegion. We consider enough the graphics
   * expose events generated internally by the nxagent
   * server.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentBitBlitHelper: WARNING! Skipping check on exposures events.\n");
  #endif

  return NullRegion;

  #else

  if (nxagentGCTrap)
  {
    return NullRegion;
  }

  if (!pGC -> graphicsExposures)
  {
    #ifdef TEST
    fprintf(stderr, "nxagentBitBlitHelper: Returning NullRegion with GraphicExposures not selected.\n");
    #endif

    return NullRegion;
  }

  pReg = REGION_CREATE(pGC->pScreen, NULL, 1);
  pTmpReg = REGION_CREATE(pGC->pScreen, NULL, 1);

  if(!pReg || !pTmpReg) return NullRegion;

  pending = True;

  while (pending)
  {
    XIfEvent(nxagentDisplay, &event, nxagentBitBlitPredicate, NULL);

    switch (event.type)
    {
      case NoExpose:
      {
        pending = False;

        #ifdef TEST
        fprintf(stderr, "nxagentBitBlitHelper: Set pending to [%d] because of NoExpose.\n", pending);
        #endif

        break;
      }
      case GraphicsExpose:
      {
        Box.x1 = event.xgraphicsexpose.x;
        Box.y1 = event.xgraphicsexpose.y;
        Box.x2 = event.xgraphicsexpose.x + event.xgraphicsexpose.width;
        Box.y2 = event.xgraphicsexpose.y + event.xgraphicsexpose.height;

        REGION_RESET(pGC->pScreen, pTmpReg, &Box);
        REGION_APPEND(pGC->pScreen, pReg, pTmpReg);

        pending = event.xgraphicsexpose.count;

        #ifdef TEST
        fprintf(stderr, "nxagentBitBlitHelper: Set pending to [%d].\n", pending);
        #endif

        break;
      }
    }
  }

  REGION_DESTROY(pGC->pScreen, pTmpReg);
  REGION_VALIDATE(pGC->pScreen, pReg, &overlap);

  return(pReg);

  #endif
}

RegionPtr nxagentCopyArea(pSrcDrawable, pDstDrawable,
                              pGC, srcx, srcy, width, height, dstx, dsty)
  DrawablePtr pSrcDrawable;
  DrawablePtr pDstDrawable;
  GC *pGC;
  int srcx, srcy;
  int width, height;
  int dstx, dsty;
  
{
  XImage *temporaryImage = NULL;
  PixmapPtr temporaryPixmap = NULL;
  unsigned long planeMask = 0xffffffff;
  int leftPad = 0;
  unsigned int format;

  #ifdef TEST
  fprintf(stderr, "nxagentCopyArea: Image src [%s:%lx], dst [%s:%lx] (%d,%d) -> (%d,%d) size (%d,%d)\n",
              ((pSrcDrawable)->type == DRAWABLE_PIXMAP) ? "PIXMAP" : "WINDOW", 
                  (unsigned long)pSrcDrawable,
                      ((pDstDrawable)->type == DRAWABLE_PIXMAP) ? "PIXMAP" : "WINDOW", 
                          (unsigned long)pDstDrawable,
                              srcx, srcy, dstx, dsty, width, height);
  #endif

  

  if (nxagentGCTrap || nxagentShmTrap)
  {
    if ((pSrcDrawable) -> type == DRAWABLE_PIXMAP &&
            (pDstDrawable) -> type == DRAWABLE_PIXMAP)
    {
      return fbCopyArea(nxagentVirtualDrawable(pSrcDrawable), 
                          nxagentVirtualDrawable(pDstDrawable),
                            pGC, srcx, srcy, width, height, dstx, dsty);
    }
    else if ((pSrcDrawable) -> type == DRAWABLE_PIXMAP)
    {
      if (nxagentUsesFrameBuffer(wClient((WindowPtr) pDstDrawable)))
      {
        return fbCopyArea(nxagentVirtualDrawable(pSrcDrawable), pDstDrawable,
                              pGC, srcx, srcy, width, height, dstx, dsty);
      }
    }
    else if ((pDstDrawable) -> type == DRAWABLE_PIXMAP)
    {
      if (nxagentUsesFrameBuffer(wClient((WindowPtr) pSrcDrawable)))
      {
        return fbCopyArea(pSrcDrawable, nxagentVirtualDrawable(pDstDrawable),
                              pGC, srcx, srcy, width, height, dstx, dsty);
      }
    }
    else
    {
      if (nxagentUsesFrameBuffer(wClient((WindowPtr) pDstDrawable)) ||
              nxagentUsesFrameBuffer(wClient((WindowPtr) pSrcDrawable)))
      {
        return fbCopyArea(pSrcDrawable, pDstDrawable,
                              pGC, srcx, srcy, width, height, dstx, dsty);
      }
    }

    return NullRegion;
  }

  /*
   * While copying the contents of the source drawable to the destination one,
   * we have to be sure that the destination  inherits the status
   * of the source.
   */

  if (nxagentDrawableStatus(pSrcDrawable) == DRAWABLE_SYNCHRONIZED)
  {
    if ((width == pDstDrawable -> width) && (height == pDstDrawable -> height))
    {
      nxagentSetDrawableStatus(pDstDrawable, DRAWABLE_SYNCHRONIZED);
    }
  }
  else
  {
    #ifdef TEST
    fprintf(stderr, "nxagentCopyArea: Setting status of drawable %p to NOT_SYNCHRONIZED.\n",
                (void *) pDstDrawable);
    #endif

    nxagentSetDrawableStatus(pDstDrawable, DRAWABLE_NOT_SYNCHRONIZED);
  }

  XCopyArea(nxagentDisplay, nxagentDrawable(pSrcDrawable), nxagentDrawable(pDstDrawable),
                nxagentGC(pGC), srcx, srcy, width, height, dstx, dsty);

  if (pSrcDrawable -> type == DRAWABLE_PIXMAP &&
          pDstDrawable -> type == DRAWABLE_PIXMAP)
  {

    #ifdef TEST
    fprintf(stderr, "nxagentCopyArea: Going to copy area from virtual pixmap [%lx] to [%lx]\n",
                (unsigned long) nxagentVirtualDrawable(pSrcDrawable),
                    (unsigned long) nxagentVirtualDrawable(pDstDrawable));
    #endif

    return fbCopyArea(nxagentVirtualDrawable(pSrcDrawable), 
                          nxagentVirtualDrawable(pDstDrawable),
                              pGC, srcx, srcy, width, height, dstx, dsty);
  }
  else if ((pSrcDrawable) -> type == DRAWABLE_PIXMAP)
  {
    if (nxagentUsesFrameBuffer(wClient((WindowPtr) pDstDrawable)))
    {
      #ifdef TEST
      fprintf(stderr, "nxagentCopyArea: Going to copy area from virtual pixmap [%lx] to window [%lx]\n",
                  (unsigned long) nxagentVirtualDrawable(pSrcDrawable),
                      (unsigned long) pDstDrawable);
      #endif
      return fbCopyArea(nxagentVirtualDrawable(pSrcDrawable), pDstDrawable,
                            pGC, srcx, srcy, width, height, dstx, dsty);
    }
  }
  else if (pDstDrawable -> type == DRAWABLE_PIXMAP)
  {
    if( (((WindowPtr) pSrcDrawable) -> viewable))
    {
      if (nxagentFastCopyAreaEnable || NXDisplayError(nxagentDisplay))
      {
        /*
         * When fast mode is enabled we ignore the window
         * content, just clean the drawable to a guessed
         * pixel colour and copy it to the pixmap in the
         * framebuffer.
         */

        GCPtr pGCtmp;
        xRectangle rect;
        CARD32 attributes[3];
        short int nrect = 1;
        unsigned long pixel;

        if (nxagentUsesFrameBuffer(wClient((WindowPtr) pSrcDrawable)))
        {
          #ifdef TEST
          fprintf(stderr, "nxagentCopyArea: Going to copy area from window [%lx] to virtual pixmap [%lx]\n",
                      (unsigned long) pSrcDrawable,
                          (unsigned long) nxagentVirtualDrawable(pDstDrawable));
          #endif

          return fbCopyArea(pSrcDrawable, nxagentVirtualDrawable(pDstDrawable),
                                pGC, srcx, srcy, width, height, dstx, dsty);
        }

        format = (pSrcDrawable -> depth == 1 ? XYPixmap : ZPixmap);

        if (nxagentGetPixelHint(pSrcDrawable, srcx, srcy, width, height,
                                    format, planeMask, &pixel) == 0)
        {
          nxagentGetDefaultHint(pSrcDrawable, srcx, srcy, width, height,
                                    format, planeMask, &pixel);
        }

        rect.x = 0;
        rect.y = 0;
        rect.width = width;
        rect.height = height;

        pGCtmp = GetScratchGC(pSrcDrawable -> depth, pSrcDrawable -> pScreen);

        /* Attributes for GCForeground */
        attributes[0] = pixel;
        /* Attributes for GCBackground */
        attributes[1] = pixel;
        /* Attributes for GCFillStyle*/
        attributes[2] = FillSolid;

        ChangeGC(pGCtmp, GCForeground | GCBackground | GCFillStyle, attributes);

        temporaryPixmap = fbCreatePixmap(pSrcDrawable -> pScreen, width, height,
                                             pSrcDrawable -> depth);

        ValidateGC((DrawablePtr) pDstDrawable, pGCtmp);

        fbPolyFillRect((DrawablePtr) temporaryPixmap, pGCtmp, nrect, &rect);

        fbCopyArea((DrawablePtr) temporaryPixmap, nxagentVirtualDrawable(pDstDrawable),
                       pGC, 0, 0, width, height, dstx, dsty);

        fbDestroyPixmap(temporaryPixmap);

        FreeScratchGC(pGCtmp);
      }
      else
      {
        int displayWidth = DisplayWidth(nxagentDisplay, DefaultScreen(nxagentDisplay));
        int displayHeight = DisplayHeight(nxagentDisplay, DefaultScreen(nxagentDisplay));

        format = (pSrcDrawable -> depth == 1 ? XYPixmap : ZPixmap);

        #ifdef NXAGENT_FASTCOPYAREA_DEBUG
        fprintf(stderr, "nxagentCopyArea: We are before calculating dimensions srcx='%d', srcy='%d', width='%d', height='%d'\n",
                    srcx, srcy, width, height);
        #endif

        /*
         * Get rid of the part of image out of the
         * drawable area.
         */

        /*
         * Left border.
         */

        if (srcx < 0)
        {
          srcx = 0;
        }

        /*
         * Top border.
         */

        if (srcy < 0)
        {
          srcy = 0;
        }

        /*
         * Right border.
         */

        if (srcx + width > pSrcDrawable -> width)
        {
          width = pSrcDrawable -> width - srcx;
        }

        /*
         * Bottom border.
         */

        if (srcy + height > pSrcDrawable -> height)
        {
          height = pSrcDrawable -> height - srcy;
        }

        /*
         * Get rid of the part of image out of the
         * nxagent screen.
         */

        /*
         * Left border.
         */

        if (pSrcDrawable -> x + srcx < 0)
        {
          width += (pSrcDrawable -> x + srcx);  
          dstx -= (pSrcDrawable -> x + srcx);
          srcx -= (pSrcDrawable -> x + srcx);
        }

        /*
         * Top border.
         */

        if (pSrcDrawable -> y + srcy < 0)
        {
          height += (pSrcDrawable -> y + srcy);
          dsty -= (pSrcDrawable -> y + srcy);
          srcy -= (pSrcDrawable -> y + srcy);
        }

        /*
         * Right border.
         */

        if (pSrcDrawable -> x + srcx + width > nxagentOption(Width))
        {
          width  = nxagentOption(Width) - (pSrcDrawable -> x + srcx);
        }

        /* 
         * Bottom border.
         */

        if (pSrcDrawable -> y + srcy + height > nxagentOption(Height))
        {
          height = nxagentOption(Height) - (pSrcDrawable -> y + srcy);
        }

        /*
         * Get rid of the part of image out of the
         * X server screen.
         */

        /*
         * Left border.
         */

        if ((nxagentOption(X) + pSrcDrawable -> x + srcx) < 0)
        {
          width += (nxagentOption(X) + pSrcDrawable -> x + srcx);
          dstx -= (nxagentOption(X) + pSrcDrawable -> x + srcx);
          srcx -= (nxagentOption(X) + pSrcDrawable -> x + srcx);
        }

        /*
         * Top border.
         */

        if ((nxagentOption(Y) + pSrcDrawable -> y + srcy) < 0)
        {
          height += (nxagentOption(Y) + pSrcDrawable -> y + srcy);
          dsty -= (nxagentOption(Y) + pSrcDrawable -> y + srcy);
          srcy -= (nxagentOption(Y) + pSrcDrawable -> y + srcy);
        }

        /*
         * Right border.
         */

        if ((nxagentOption(X) + pSrcDrawable -> x + srcx + width) > displayWidth)
        {
          width  = displayWidth - (nxagentOption(X) + pSrcDrawable -> x + srcx);
        }

        /*
         * Bottom border.
         */

        if ((nxagentOption(Y) + pSrcDrawable -> y + srcy + height) > displayHeight)
        {
          height = displayHeight - (nxagentOption(Y) + pSrcDrawable -> y + srcy);
        }

        #ifdef NXAGENT_FASTCOPYAREA_DEBUG
        fprintf(stderr, "nxagentCopyArea: We are after calculating dimensions srcx='%d', srcy='%d', width='%d', height='%d'\n",
                    srcx, srcy, width, height);
        #endif

        if ((width > 0) && (height > 0))
        {
          temporaryImage = XGetImage(nxagentDisplay, nxagentDrawable(pSrcDrawable),
                                       srcx, srcy, width, height, planeMask, format);

          if (temporaryImage)
          {
            temporaryPixmap = fbCreatePixmap(pSrcDrawable->pScreen, width, height, pSrcDrawable->depth);

            fbPutImage((DrawablePtr) temporaryPixmap, pGC, pSrcDrawable->depth, 0, 0, width, height,
                           leftPad, format, temporaryImage->data);

            fbCopyArea((DrawablePtr) temporaryPixmap, nxagentVirtualDrawable(pDstDrawable),
                           pGC, 0, 0, width, height, dstx, dsty);

            XDestroyImage(temporaryImage);

            fbDestroyPixmap(temporaryPixmap);
          }
        }
      }
    }
  }
  else
  {
    if (((WindowPtr) pSrcDrawable) -> viewable && (nxagentUsesFrameBuffer(wClient((WindowPtr) pSrcDrawable)) ||
           nxagentUsesFrameBuffer(wClient((WindowPtr) pSrcDrawable))))
    {
      #ifdef TEST
      fprintf(stderr, "nxagentCopyArea: Going to copy area from window [%lx] to window [%lx]\n",
                  (unsigned long) pSrcDrawable, (unsigned long) pDstDrawable);
      #endif

      return fbCopyArea(pSrcDrawable, pDstDrawable,
                            pGC, srcx, srcy, width, height, dstx, dsty);
    }
  }

  return miHandleExposures(pSrcDrawable, pDstDrawable, pGC, 
                             srcx, srcy, width, height, dstx, dsty, 0);
}

RegionPtr nxagentCopyPlane(pSrcDrawable, pDstDrawable,
                         pGC, srcx, srcy, width, height, dstx, dsty, plane)
  DrawablePtr pSrcDrawable;
  DrawablePtr pDstDrawable;
  GC *pGC;
  int srcx, srcy;
  int width, height;
  int dstx, dsty;
  unsigned long plane;
{
  if (nxagentGCTrap || nxagentShmTrap)
  {
    if ((pSrcDrawable)->type == DRAWABLE_PIXMAP &&
            (pDstDrawable)->type == DRAWABLE_PIXMAP)
    {
      return fbCopyPlane(nxagentVirtualDrawable(pSrcDrawable), 
                           nxagentVirtualDrawable(pDstDrawable),
                             pGC, srcx, srcy, width, height, dstx, dsty, plane);
    }

    return NullRegion;
  }

  /*
   * If the destination pixmap is completely covered by this
   * operation and the source is realized, we can consider this
   * operation as the realization of the destination pixmap.
   * At the moment we are not concerned so much to cover this
   * optimization.
   */

  if (nxagentDrawableStatus(pSrcDrawable) == DRAWABLE_SYNCHRONIZED)
  {
    if ((width == pDstDrawable -> width) && (height == pDstDrawable -> height))
    {
      nxagentSetDrawableStatus(pDstDrawable, DRAWABLE_SYNCHRONIZED);
    }
  }
  else
  {
    #ifdef TEST
    fprintf(stderr, "nxagentCopyPlane: Setting status of drawable %p to NOT_SYNCHRONIZED.\n",
                (void *) pDstDrawable);
    #endif

    nxagentSetDrawableStatus(pDstDrawable, DRAWABLE_NOT_SYNCHRONIZED);
  }

  XCopyPlane(nxagentDisplay,
             nxagentDrawable(pSrcDrawable), nxagentDrawable(pDstDrawable),
             nxagentGC(pGC), srcx, srcy, width, height, dstx, dsty, plane);

#ifdef NXAGENT_MVFB
  if ((pSrcDrawable)->type == DRAWABLE_PIXMAP && (pDstDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: going to copy plane from mvfb pixmap = [%lx] to [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pSrcDrawable), (unsigned long)nxagentVirtualDrawable(pDstDrawable));
#endif


    return fbCopyPlane(nxagentVirtualDrawable(pSrcDrawable), 
                         nxagentVirtualDrawable(pDstDrawable),
                           pGC, srcx, srcy, width, height, dstx, dsty, plane);
  }
#endif

 return miHandleExposures(pSrcDrawable, pDstDrawable, pGC, 
		            srcx, srcy, width, height, dstx, dsty, plane);
}

void nxagentPolyPoint(pDrawable, pGC, mode, nPoints, pPoints)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int mode;
  int nPoints;
  xPoint *pPoints;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      fbPolyPoint(nxagentVirtualDrawable(pDrawable), pGC, mode, nPoints, pPoints);
    }

    return;
  }


#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to poly point on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif


    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
      fprintf(stderr, "GCOps: poly point enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
           XDrawPoints(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
              (XPoint *)pPoints, nPoints, mode);
    }
    else
    {
       XDrawPoints(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                   (XPoint *)pPoints, nPoints, mode);
    }

    fbPolyPoint(nxagentVirtualDrawable(pDrawable), pGC, mode, nPoints, pPoints);

    return;
  }
  else
#endif
    XDrawPoints(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
              (XPoint *)pPoints, nPoints, mode);
}

void nxagentPolylines(pDrawable, pGC, mode, nPoints, pPoints)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int mode;
  int nPoints;
  xPoint *pPoints;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      fbPolyLine(nxagentVirtualDrawable(pDrawable), pGC, mode, nPoints, pPoints);
    }

    return;
  }

  if ((pGC->tileIsPixel == False && nxagentPixmapStatus(pGC->tile.pixmap) == DRAWABLE_NOT_SYNCHRONIZED) ||
          (pGC->stipple && nxagentPixmapStatus(pGC->stipple) == DRAWABLE_NOT_SYNCHRONIZED))
  {
    nxagentSetDrawableStatus(pDrawable, DRAWABLE_NOT_SYNCHRONIZED);
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly line on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif

    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly lines enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
        XDrawLines(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
             (XPoint *)pPoints, nPoints, mode);
    }
    else
    {
      XDrawLines(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                 (XPoint *)pPoints, nPoints, mode);
    }
    fbPolyLine(nxagentVirtualDrawable(pDrawable), pGC, mode, nPoints, pPoints);
    return;
  }
  else
#endif
    XDrawLines(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
               (XPoint *)pPoints, nPoints, mode);
}

void nxagentPolySegment(pDrawable, pGC, nSegments, pSegments)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int nSegments;
  xSegment *pSegments;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miPolySegment(nxagentVirtualDrawable(pDrawable), pGC, nSegments, pSegments);
    }

    return;
  }

  if ((pGC->tileIsPixel == False && nxagentPixmapStatus(pGC->tile.pixmap) == DRAWABLE_NOT_SYNCHRONIZED) ||
          (pGC->stipple && nxagentPixmapStatus(pGC->stipple) == DRAWABLE_NOT_SYNCHRONIZED))
  {
    nxagentSetDrawableStatus(pDrawable, DRAWABLE_NOT_SYNCHRONIZED);
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly segment on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif

    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly segment enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
        XDrawSegments(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                (XSegment *)pSegments, nSegments);
    }
    else
    {
      XDrawSegments(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                    (XSegment *)pSegments, nSegments);
    }
    miPolySegment(nxagentVirtualDrawable(pDrawable), pGC, nSegments, pSegments);
    return;
  }
  else
#endif
    XDrawSegments(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                 (XSegment *)pSegments, nSegments);
}

void nxagentPolyRectangle(pDrawable, pGC, nRectangles, pRectangles)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int nRectangles;
  xRectangle *pRectangles;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miPolyRectangle(nxagentVirtualDrawable(pDrawable), pGC, nRectangles, pRectangles);
    }

    return;
  }

  if ((pGC->tileIsPixel == False && nxagentPixmapStatus(pGC->tile.pixmap) == DRAWABLE_NOT_SYNCHRONIZED) ||
          (pGC->stipple && nxagentPixmapStatus(pGC->stipple) == DRAWABLE_NOT_SYNCHRONIZED))
  {
    nxagentSetDrawableStatus(pDrawable, DRAWABLE_NOT_SYNCHRONIZED);
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly rectangle on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif

    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly rectangle enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
        XDrawRectangles(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                        (XRectangle *)pRectangles, nRectangles);
    }
    else
    {
      XDrawRectangles(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                      (XRectangle *)pRectangles, nRectangles);
    }
    miPolyRectangle(nxagentVirtualDrawable(pDrawable), pGC, nRectangles, pRectangles);
    return;
  }
  else
#endif
    XDrawRectangles(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                    (XRectangle *)pRectangles, nRectangles);
}

void nxagentPolyArc(pDrawable, pGC, nArcs, pArcs)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int nArcs;
  xArc *pArcs;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      fbPolyArc(nxagentVirtualDrawable(pDrawable), pGC, nArcs, pArcs);
    }

    return;
  }

  if ((pGC->tileIsPixel == False && nxagentPixmapStatus(pGC->tile.pixmap) == DRAWABLE_NOT_SYNCHRONIZED) ||
          (pGC->stipple && nxagentPixmapStatus(pGC->stipple) == DRAWABLE_NOT_SYNCHRONIZED))
  {
    nxagentSetDrawableStatus(pDrawable, DRAWABLE_NOT_SYNCHRONIZED);
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly arc on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif

    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly arc enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
          XDrawArcs(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
            (XArc *)pArcs, nArcs);
    }
    else
    {
       XDrawArcs(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                 (XArc *)pArcs, nArcs);
    }
    fbPolyArc(nxagentVirtualDrawable(pDrawable), pGC, nArcs, pArcs);
    return;
  }
  else
#endif
    XDrawArcs(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
              (XArc *)pArcs, nArcs);
}

void nxagentFillPolygon(pDrawable, pGC, shape, mode, nPoints, pPoints)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int shape;
  int mode;
  int nPoints;
  xPoint *pPoints;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miFillPolygon(nxagentVirtualDrawable(pDrawable), pGC, shape, mode, nPoints, pPoints);
    }
    else if (nxagentUsesFrameBuffer(wClient((WindowPtr) pDrawable)))
    {
      miFillPolygon(pDrawable, pGC, shape, mode, nPoints, pPoints);
    }

    return;
  }

  if ((pGC->tileIsPixel == False && nxagentPixmapStatus(pGC->tile.pixmap) == DRAWABLE_NOT_SYNCHRONIZED) ||
          (pGC->stipple && nxagentPixmapStatus(pGC->stipple) == DRAWABLE_NOT_SYNCHRONIZED))
  {
    nxagentSetDrawableStatus(pDrawable, DRAWABLE_NOT_SYNCHRONIZED);
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to fill polygon on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif

    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: fill polygon enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
          XFillPolygon(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                       (XPoint *)pPoints, nPoints, shape, mode);
    }
    else
    {
      XFillPolygon(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                   (XPoint *)pPoints, nPoints, shape, mode);

    }
    miFillPolygon(nxagentVirtualDrawable(pDrawable), pGC, shape, mode, nPoints, pPoints);
    return;
  }
  else
#endif
  {
    XFillPolygon(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                 (XPoint *)pPoints, nPoints, shape, mode);

    if (nxagentUsesFrameBuffer(wClient((WindowPtr) pDrawable)))
    {
      miFillPolygon(pDrawable, pGC, shape, mode, nPoints, pPoints);
    }
  }
}

void nxagentPolyFillRect(pDrawable, pGC, nRectangles, pRectangles)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int nRectangles;
  xRectangle *pRectangles;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      fbPolyFillRect(nxagentVirtualDrawable(pDrawable), pGC, nRectangles, pRectangles);
    }
    else if (nxagentUsesFrameBuffer(wClient((WindowPtr) pDrawable)))
    {
      fbPolyFillRect(pDrawable, pGC, nRectangles, pRectangles);
    }

    return;
  }

  if ((pGC->tileIsPixel == False && nxagentPixmapStatus(pGC->tile.pixmap) == DRAWABLE_NOT_SYNCHRONIZED) ||
          (pGC->stipple && nxagentPixmapStatus(pGC->stipple) == DRAWABLE_NOT_SYNCHRONIZED))
  {
    nxagentSetDrawableStatus(pDrawable, DRAWABLE_NOT_SYNCHRONIZED);
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly fill rect on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif

    if (nxagentPixmapIsVirtual((PixmapPtr) pDrawable))
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly fill rect enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr) pDrawable))
      {
         XFillRectangles(nxagentDisplay, nxagentDrawable((DrawablePtr) nxagentRealPixmap((PixmapPtr) pDrawable)),
                             nxagentGC(pGC), (XRectangle *) pRectangles, nRectangles);
      }
    }
    else
    {
      XFillRectangles(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                      (XRectangle *)pRectangles, nRectangles);
    }

    fbPolyFillRect(nxagentVirtualDrawable(pDrawable), pGC, nRectangles, pRectangles);

    return;
  }
  else
#endif
  {
    XFillRectangles(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                    (XRectangle *)pRectangles, nRectangles);

    if (nxagentUsesFrameBuffer(wClient((WindowPtr) pDrawable)))
    {
      fbPolyFillRect(pDrawable, pGC, nRectangles, pRectangles);
    }
  }
}

void nxagentPolyFillArc(pDrawable, pGC, nArcs, pArcs)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int nArcs;
  xArc *pArcs;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miPolyFillArc(nxagentVirtualDrawable(pDrawable), pGC, nArcs, pArcs);
    }

    return;
  }

  if ((pGC->tileIsPixel == False && nxagentPixmapStatus(pGC->tile.pixmap) == DRAWABLE_NOT_SYNCHRONIZED) ||
          (pGC->stipple && nxagentPixmapStatus(pGC->stipple) == DRAWABLE_NOT_SYNCHRONIZED))
  {
    nxagentSetDrawableStatus(pDrawable, DRAWABLE_NOT_SYNCHRONIZED);
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly fillarc on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif

    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly fill arc enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
         XFillArcs(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                   (XArc *)pArcs, nArcs);
    }
    else
    {
      XFillArcs(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                (XArc *)pArcs, nArcs);
    }
    miPolyFillArc(nxagentVirtualDrawable(pDrawable), pGC, nArcs, pArcs);
    return;
  }
  else
#endif
    XFillArcs(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
            (XArc *)pArcs, nArcs);
}

int nxagentPolyText8(pDrawable, pGC, x, y, count, string)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int count;
  char *string;
{
  int width;

  width = XTextWidth(nxagentFontStruct(pGC->font), string, count);

  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miPolyText8(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    }

    return width + x;
  }

  if ((pGC->tileIsPixel == False && nxagentPixmapStatus(pGC->tile.pixmap) == DRAWABLE_NOT_SYNCHRONIZED) ||
          (pGC->stipple && nxagentPixmapStatus(pGC->stipple) == DRAWABLE_NOT_SYNCHRONIZED))
  {
    nxagentSetDrawableStatus(pDrawable, DRAWABLE_NOT_SYNCHRONIZED);
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly text8 on mvfb pixmap = [%lx] s = [%s]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable), string);
#endif

    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly text8 enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
         XDrawString(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                     x, y, string, count);
    }
    else
    {
      XDrawString(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                  x, y, string, count);
    }
    miPolyText8(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    return width + x;
  }
  else
#endif
    XDrawString(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
              x, y, string, count);

  return width + x;
}

int nxagentPolyText16(pDrawable, pGC, x, y, count, string)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int count;
  unsigned short *string;
{
  int width;

  width = XTextWidth16(nxagentFontStruct(pGC->font), (XChar2b *)string, count);

  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miPolyText16(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    }
    else if (nxagentUsesFrameBuffer(wClient((WindowPtr) pDrawable)))
    {
      miPolyText16(pDrawable, pGC, x, y, count, string);
    }

    return width + x;
  }

  if ((pGC->tileIsPixel == False && nxagentPixmapStatus(pGC->tile.pixmap) == DRAWABLE_NOT_SYNCHRONIZED) ||
          (pGC->stipple && nxagentPixmapStatus(pGC->stipple) == DRAWABLE_NOT_SYNCHRONIZED))
  {
    nxagentSetDrawableStatus(pDrawable, DRAWABLE_NOT_SYNCHRONIZED);
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly text16 on mvfb pixmap = [%lx] s = [%s]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable), string);
#endif

    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly text16 enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
         XDrawString16(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                     x, y, (XChar2b *)string, count);
    }
    else
    {
      XDrawString16(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                    x, y, (XChar2b *)string, count);
    }
    miPolyText16(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    return width + x;
  }
  else
#endif
  {
    XDrawString16(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                  x, y, (XChar2b *)string, count);

    if (nxagentUsesFrameBuffer(wClient((WindowPtr) pDrawable)))
    {
      miPolyText16(pDrawable, pGC, x, y, count, string);
    }
  }

  return width + x;
}

void nxagentImageText8(pDrawable, pGC, x, y, count, string)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int count;
  char *string;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miImageText8(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    }

    return;
  }


#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to image text8 on mvfb pixmap = [%lx] s = [%s]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable), string);
#endif

    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {

#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly image text8 enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
         XDrawImageString(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                          x, y, string, count);
    }
    else
    {
      XDrawImageString(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                       x, y, string, count);
    }
    miImageText8(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    return;
  }
  else
#endif
    XDrawImageString(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                     x, y, string, count);
}

void nxagentImageText16(pDrawable, pGC, x, y, count, string)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int count;
  unsigned short *string;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miImageText16(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    }

    return;
  }


#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to image text16 on mvfb pixmap = [%lx] s = [%s]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable), string);
#endif

    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly image text16 enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
         XDrawImageString16(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                          x, y, (XChar2b *)string, count);
    }
    else
    {
      XDrawImageString16(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                         x, y, (XChar2b *)string, count);
    }
    miImageText16(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    return;
  }
  else
#endif
    XDrawImageString16(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                     x, y, (XChar2b *)string, count);
}

void nxagentImageGlyphBlt(pDrawable, pGC, x, y, nGlyphs, pCharInfo, pGlyphBase)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int nGlyphs;
  CharInfoPtr *pCharInfo;
  pointer pGlyphBase;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to imageGlyphBlt on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    fbImageGlyphBlt(nxagentVirtualDrawable(pDrawable), pGC, x, y, nGlyphs, pCharInfo, pGlyphBase);
  }
#else
  ErrorF("nxagent warning: function nxagentImageGlyphBlt not implemented\n");
#endif
}

void nxagentPolyGlyphBlt(pDrawable, pGC, x, y, nGlyphs, pCharInfo, pGlyphBase)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int nGlyphs;
  CharInfoPtr *pCharInfo;
  pointer pGlyphBase;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to PolyGlyphBlt on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    fbPolyGlyphBlt(nxagentVirtualDrawable(pDrawable), pGC, x, y, nGlyphs, pCharInfo, pGlyphBase);
  }
#else
  ErrorF("nxagent warning: function nxagentPolyGlyphBlt not implemented\n");
#endif
}

void nxagentPushPixels(pGC, pBitmap, pDrawable, width, height, x, y)
  GCPtr pGC;
  PixmapPtr pBitmap;
  DrawablePtr pDrawable;
  int width, height;
  int x, y;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to PushPixels on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                         (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif

    fbPushPixels(pGC, nxagentVirtualPixmap(pBitmap), (DrawablePtr)nxagentVirtualDrawable(pDrawable), width, height, x, y);
  }
#else
  ErrorF("nxagent warning: function nxagentPushPixels not implemented\n");
#endif
}

/*
 * We have to reset the visual cache before
 * connecting to another display, so that a
 * new unpack geometry can be communicated
 * to the new proxy.
 */

void nxagentResetVisualCache()
{
  int i;

  for (i = 0; i < MAX_CONNECTIONS; i++)
  {
    nxagentUnpackVisualId[i] = None;
  }
}

void nxagentResetAlphaCache()
{
  int i;

  for (i = 0; i < MAX_CONNECTIONS; i++)
  {
    if (nxagentUnpackAlpha[i])
    {
      xfree(nxagentUnpackAlpha[i] -> data);

      xfree(nxagentUnpackAlpha[i]);

      nxagentUnpackAlpha[i] = NULL;
    }
  }
}


