/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com.          */
/*                                                                        */
/* NXAGENT, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

/*

Copyright 1993 by Davor Matic

Permission to use, copy, modify, distribute, and sell this software
and its documentation for any purpose is hereby granted without fee,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation.  Davor Matic makes no representations about
the suitability of this software for any purpose.  It is provided "as
is" without express or implied warranty.

*/

#include "X.h"
#include "Xproto.h"
#include "miscstruct.h"
#include "pixmapstr.h"
#include "scrnintstr.h"
#include "regionstr.h"
#include "gc.h"
#include "servermd.h"
#include "mi.h"

#include "../../fb/fb.h"

#include "Agent.h"

#include "Display.h"
#include "Screen.h"
#include "Pixmap.h"
#include "Trap.h"

#include "GC.h"
#include "GCOps.h"
#include "Drawable.h"
#include "Visual.h"
#include "Control.h"
#include NXAGENT_NXLIB_INCLUDE
#include NXAGENT_NXPACK_INCLUDE

RESTYPE  RT_NX_PIXMAP;

/*
 * Set here the required log level.
 */

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

/*
 * Force deallocation of the virtual pixmap.
 */

static Bool nxagentDestroyVirtualPixmap(PixmapPtr pPixmap);

/*
 * The following is just used for persistence.
 */

extern int  nxagentPackMethod;
extern int  nxagentPackQuality;

Bool (*nxagentOldModifyPixmapHeader) (PixmapPtr, int, int, int, int, int, pointer);

#ifdef NXAGENT_STOPBIGREQ
extern Bool nxSleepByBigReq(ClientPtr);
extern Bool nxEndBigReq(ClientPtr);
#endif

static GCPtr nxagentGetTempGC(PixmapPtr);
static Bool nxagentFreeAllTempGC(void);

/*
 * These are temporarily here. Changes must be
 * merged back into the original functions.
 */

extern Bool nxagentScratchGCsDestroy();
extern Bool nxagentScratchGCsCreate();

int nxagentPixmapPrivateIndex;

/* Using a counter to to synchronize a bunch of pixmaps every time.
 */

int PixmapsBeforeExpose = 0;

#ifdef NXAGENT_MVFB
extern Bool nxagentTrue24;
#endif

/*
 * This serves as a tool to check the synchronization
 * between pixmaps in framebuffer and the correspondent
 * pixmaps in the real X server.
 */

#ifdef TEST
Bool nxagentCheckPixmapIntegrity(PixmapPtr pPixmap);
#endif

struct nxagentPixmapPair
{
  Pixmap pixmap;
  PixmapPtr pMap;
};

PixmapPtr nxagentCreatePixmap(pScreen, width, height, depth)
    ScreenPtr   pScreen;
    int         width;
    int         height;
    int         depth;
{
  nxagentPrivPixmapPtr pPixmapPriv;

  PixmapPtr pPixmap;
  PixmapPtr pVirtual;

  /*
   * Create the pixmap structure but do
   * not allocate memory for the data.
   */

  pPixmap = AllocatePixmap(pScreen, 0);

  if (!pPixmap)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentCreatePixmap: WARNING! Failed to create pixmap with "
                "width [%d] height [%d] depth [%d].\n", width, height, depth);
    #endif

    return NullPixmap;
  }

  /*
   * Initialize the core members.
   */

  pPixmap -> drawable.type = DRAWABLE_PIXMAP;
  pPixmap -> drawable.class = 0;
  pPixmap -> drawable.pScreen = pScreen;
  pPixmap -> drawable.depth = depth;
  pPixmap -> drawable.bitsPerPixel = BitsPerPixel(depth);
  pPixmap -> drawable.id = 0;
  pPixmap -> drawable.serialNumber = NEXT_SERIAL_NUMBER;
  pPixmap -> drawable.x = 0;
  pPixmap -> drawable.y = 0;
  pPixmap -> drawable.width = width;
  pPixmap -> drawable.height = height;
  pPixmap -> devKind = 0;
  pPixmap -> refcnt = 1;
  pPixmap -> devPrivate.ptr = NULL;

  /*
   * Create the pixmap in the virtual framebuffer.
   */

  pVirtual = fbCreatePixmap(pScreen, width, height, depth);

  /*
   * Initialize the privates of the real picture.
   */

  pPixmapPriv = nxagentPixmapPriv(pPixmap);

  pPixmapPriv -> isVirtual = False;
  pPixmapPriv -> drawableLazy.status = DRAWABLE_SYNCHRONIZED;
  pPixmapPriv -> drawableLazy.placeholderDrawn = False;

  if (width && height)
  {
    pPixmapPriv -> id = XCreatePixmap(nxagentDisplay,
                                      nxagentDefaultWindows[pScreen -> myNum],
                                      width, height, depth);
  }
  else
  {
    pPixmapPriv -> id = 0;
  }

  pPixmapPriv -> mid = FakeClientID(0);

  AddResource(pPixmapPriv -> mid, RT_NX_PIXMAP, pPixmap);

  pPixmapPriv -> pRealPixmap = pPixmap;
  pPixmapPriv -> pVirtualPixmap = pVirtual;
  pPixmapPriv -> pPicture = NULL;

  /*
   * Check that the virtual pixmap is created with
   * the appropriate bits-per-plane, otherwise free
   * everything and return.
   */

  if (pVirtual == NULL)
  {
    #ifdef PANIC
    fprintf(stderr, "nxagentCreatePixmap: PANIC! Failed to create virtual pixmap with "
                "width [%d] height [%d] depth [%d].\n", width, height, depth);
    #endif

    nxagentDestroyPixmap(pPixmap);

    return NullPixmap;
  }
  else if (pVirtual -> drawable.bitsPerPixel == 0)
  {
    #ifdef WARNING

    fprintf(stderr, "nxagentCreatePixmap: WARNING! Virtual pixmap at [%p] has invalid "
                "bits per pixel.\n", (void *) pVirtual);

    fprintf(stderr, "nxagentCreatePixmap: WARNING! Real pixmap created with width [%d] "
                "height [%d] depth [%d] bits per pixel [%d].\n", pPixmap -> drawable.width,
                    pPixmap -> drawable.height = height, pPixmap -> drawable.depth,
                        pPixmap -> drawable.bitsPerPixel);

    #endif

    if (!nxagentRenderTrap)
    {
      fprintf(stderr, "Warning: Disabling render extension due to missing pixmap format.\n");

      nxagentRenderTrap = 1;
    }

    nxagentDestroyPixmap(pPixmap);

    return NullPixmap;
  }

  /*
   * Initialize the privates of the virtual picture. We
   * could avoid to use a flag and just check the pointer
   * to the virtual pixmap that, if the pixmap is actually
   * virtual, will be NULL. Unfortunately the flag can be
   * changed in nxagentValidateGC(). That code, probably,
   * can be removed in future.
   */

  nxagentPixmapPriv(pVirtual) -> isVirtual = True;

  /*
   * We might distinguish real and virtual pixmaps by
   * checking the pointers to pVirtualPixmap. We should
   * also remove the copy of id and use the one of the
   * real pixmap.
   */
   
  nxagentPixmapPriv(pVirtual) -> id = pPixmapPriv -> id;
  nxagentPixmapPriv(pVirtual) -> mid = 0;

  /*
   * Storing a pointer back to the real pixmap is
   * silly. Unfortunately this is the way it has
   * been originally implemented. See also the
   * comment in destroy of the pixmap.
   */

  nxagentPixmapPriv(pVirtual) -> pRealPixmap = pPixmap;
  nxagentPixmapPriv(pVirtual) -> pVirtualPixmap = NULL;
  nxagentPixmapPriv(pVirtual) -> pPicture = NULL;

  #ifdef TEST
  fprintf(stderr, "nxagentCreatePixmap: Created pixmap at [%p] virtual at [%p] with width [%d] "
              "height [%d] depth [%d].\n", (void *) pPixmap, (void *) pVirtual,
                  width, height, depth);
  #endif

  #ifdef TEST

  if (width && height)
  {
    GCPtr pGCtmp;
    xRectangle rect;
    CARD32 attributes[3];

    rect.x = 0;
    rect.y = 0;
    rect.width = width;
    rect.height = height;

    pGCtmp = GetScratchGC(pPixmap -> drawable.depth, pPixmap -> drawable.pScreen);

    /* Attributes for GCForeground */
    attributes[0] = 0x228b22;
    /* Attributes for GCBackground */
    attributes[1] = 0xffffff;
    /* Attributes for GCFillStyle*/
    attributes[2] = FillSolid;

    ChangeGC(pGCtmp, GCForeground | GCBackground | GCFillStyle , attributes);

    ValidateGC((DrawablePtr)pPixmap, pGCtmp);

    fbPolyFillRect((DrawablePtr) pVirtual, pGCtmp, 1, &rect);

    FreeScratchGC(pGCtmp);

    if (nxagentOption(Lazy) == False)
    {
      nxagentRealizePixmap(pPixmap);
    }
  }

  #endif

  return pPixmap;
}

Bool nxagentDestroyPixmap(pPixmap)
     PixmapPtr pPixmap;
{
  PixmapPtr pVirtual;

  nxagentPrivPixmapPtr pPixmapPriv;

  if (!pPixmap)
  {
    #ifdef PANIC
    fprintf(stderr, "nxagentDestroyPixmap: PANIC! Invalid attempt to destroy "
                "a null pixmap pointer.\n");
    #endif

    return False;
  }

  pPixmapPriv = nxagentPixmapPriv(pPixmap);

  pVirtual = pPixmapPriv -> pVirtualPixmap;

  #ifdef TEST
  fprintf(stderr, "\nnxagentDestroyPixmap: Destroying pixmap at [%p] with virtual at [%p].\n",
              (void *) pPixmap, (void *) pVirtual);
  #endif

  if (pPixmapPriv -> isVirtual)
  {
    int refcnt;

    /*
     * For some pixmaps we receive the destroy only for the
     * virtual. Infact to draw in the framebuffer we can use
     * the virtual pixmap instead of the pointer to the real
     * one. As the virtual pixmap can collect references, we
     * must transfer those references to the real pixmap so
     * we can continue as the destroy had been requested for
     * it.
     */

    pVirtual = pPixmap;
    pPixmap  = pPixmapPriv -> pRealPixmap;

    pPixmapPriv = nxagentPixmapPriv(pPixmap);

    /*
     * Move the references accumulated by the virtual
     * pixmap into the references of the real one.
     */

    refcnt = pVirtual -> refcnt - 1;

    #ifdef TEST
    fprintf(stderr, "nxagentDestroyPixmap: Adding [%d] references to pixmap at [%p].\n",
                refcnt, (void *) pPixmap);
    #endif

    pPixmap -> refcnt += refcnt;

    pVirtual -> refcnt -= refcnt;
  }

  #ifdef TEST
  fprintf(stderr, "nxagentDestroyPixmap: Pixmap has counter [%d] virtual pixmap [%d].\n",
              pPixmap -> refcnt, pVirtual -> refcnt);
  #endif

  --pPixmap -> refcnt;

  #ifdef TEST

  fprintf(stderr, "nxagentDestroyPixmap: Pixmap has now counter [%d] virtual pixmap has [%d].\n",
              pPixmap -> refcnt, pVirtual -> refcnt);

  if (pVirtual -> refcnt != 1)
  {
    fprintf(stderr, "nxagentDestroyPixmap: PANIC! Virtual pixmap does not have references equal to 1.\n");
  }

  #endif

  if (pPixmap -> refcnt)
  {
    return True;
  }

  #ifdef TEST
  fprintf(stderr, "nxagentDestroyPixmap: Managing to destroy the pixmap at [%p]\n",
              (void *) pPixmap);
  #endif

  nxagentDestroyVirtualPixmap(pPixmap);

  /*
   * It seems that a pixmap with 0 width and height
   * is created at the beginning. The id is not
   * assigned in this case.
   */

  if (pPixmapPriv -> id)
  {
    XFreePixmap(nxagentDisplay, pPixmapPriv -> id);
  }

  if (pPixmapPriv -> mid)
  {
    FreeResource(pPixmapPriv -> mid, RT_NONE);
  }

  xfree(pPixmap);

  return True;
}

Bool nxagentDestroyVirtualPixmap(PixmapPtr pPixmap)
{
  PixmapPtr pVirtual;

  pVirtual = nxagentPixmapPriv(pPixmap) -> pVirtualPixmap;

  /*
   * Force the routine to get rid of the virtual
   * pixmap.
   */

  if (pVirtual)
  {
    pVirtual -> refcnt = 1;

    fbDestroyPixmap(pVirtual);
  }

  return True;
}

RegionPtr nxagentPixmapToRegion(pPixmap)
     PixmapPtr pPixmap;
{
  XImage *ximage;
  register RegionPtr pReg, pTmpReg;
  register int x, y;
  unsigned long previousPixel, currentPixel;
  BoxRec Box;
  Bool overlap;

#ifdef NXAGENT_MVFB_DEBUG
  fprintf(stderr, "PixmapToRegion: Pixmap = [%lx] nxagentVirtualPixmap = [%lx]\n",pPixmap,nxagentVirtualPixmap(pPixmap));
#endif

#ifdef NXAGENT_MVFB

  return fbPixmapToRegion(nxagentVirtualPixmap(pPixmap));

#endif

  ximage = XGetImage(nxagentDisplay, nxagentPixmap(pPixmap), 0, 0,
                     pPixmap->drawable.width, pPixmap->drawable.height,
                     1, XYPixmap);

  pReg = REGION_CREATE(pPixmap->drawable.pScreen, NULL, 1);
  pTmpReg = REGION_CREATE(pPixmap->drawable.pScreen, NULL, 1);
  if (!pReg || !pTmpReg) return NullRegion;

  for (y = 0; y < pPixmap->drawable.height; y++) {
    Box.y1 = y;
    Box.y2 = y + 1;
    previousPixel = 0L;
    for (x = 0; x < pPixmap->drawable.width; x++)
    {
      currentPixel = XGetPixel(ximage, x, y);
      if (previousPixel != currentPixel) {
        if (previousPixel == 0L) {
          /* left edge */
          Box.x1 = x;
        }
        else if (currentPixel == 0L) {
          /* right edge */
          Box.x2 = x;
          REGION_RESET(pPixmap->drawable.pScreen, pTmpReg, &Box);
          REGION_APPEND(pPixmap->drawable.pScreen, pReg, pTmpReg);
        }
        previousPixel = currentPixel;
      }
    }
    if (previousPixel != 0L) {
      /* right edge because of the end of pixmap */
      Box.x2 = pPixmap->drawable.width;
      REGION_RESET(pPixmap->drawable.pScreen, pTmpReg, &Box);
      REGION_APPEND(pPixmap->drawable.pScreen, pReg, pTmpReg);
    }
  }

  REGION_DESTROY(pPixmap->drawable.pScreen, pTmpReg);

  XDestroyImage(ximage);

  REGION_VALIDATE(pPixmap->drawable.pScreen, pReg, &overlap);

  return(pReg);
}

Bool nxagentModifyPixmapHeader(pPixmap, width, height, depth,
                                   bitsPerPixel, devKind, pPixData)
    PixmapPtr pPixmap;
    int       width;
    int       height;
    int       depth;
    int       bitsPerPixel;
    int       devKind;
    pointer   pPixData;
{
  PixmapPtr pVirtualPixmap;

  /*
   * See miModifyPixmapHeader() in miscrinit.c. This
   * function is used to recycle the scratch pixmap
   * for this screen. We let it refer to the virtual
   * pixmap.
   */

  if (!pPixmap)
  {
    return False;
  }

  if (nxagentPixmapIsVirtual(pPixmap))
  {
    #ifdef PANIC
    fprintf(stderr, "nxagentModifyPixmapHeader: PANIC! Pixmap at [%p] is virtual.\n",
                (void *) pPixmap);
    #endif

    FatalError("nxagentModifyPixmapHeader: PANIC! Pixmap is virtual.");
  }

  pVirtualPixmap = nxagentVirtualPixmap(pPixmap);

  #ifdef TEST
  fprintf(stderr, "nxagentModifyPixmapHeader: Pixmap has width [%d] height [%d] depth [%d] "
              "bits-per-pixel [%d] devKind [%d] pPixData [%p].\n", pPixmap->drawable.width,
                  pPixmap->drawable.height, pPixmap->drawable.depth, pPixmap->drawable.bitsPerPixel,
                      pPixmap->devKind, (void *) pPixmap->devPrivate.ptr);

  fprintf(stderr, "nxagentModifyPixmapHeader: New parameters are width [%d] height [%d] depth [%d] "
              "bits-per-pixel [%d] devKind [%d] pPixData [%p].\n", width, height, depth,
                  bitsPerPixel, devKind, (void *) pPixData);
  #endif

  if ((width > 0) && (height > 0) && (depth > 0) &&
          (bitsPerPixel > 0) && (devKind > 0) && pPixData)
  {
    pPixmap->drawable.depth = depth;
    pPixmap->drawable.bitsPerPixel = bitsPerPixel;
    pPixmap->drawable.id = 0;
    pPixmap->drawable.serialNumber = NEXT_SERIAL_NUMBER;
    pPixmap->drawable.x = 0;
    pPixmap->drawable.y = 0;
    pPixmap->drawable.width = width;
    pPixmap->drawable.height = height;
    pPixmap->devKind = devKind;
    pPixmap->refcnt = 1;
    pPixmap->devPrivate.ptr = pPixData;

    pVirtualPixmap->drawable.depth = depth;
    pVirtualPixmap->drawable.bitsPerPixel = bitsPerPixel;
    pVirtualPixmap->drawable.id = 0;
    pVirtualPixmap->drawable.serialNumber = NEXT_SERIAL_NUMBER;
    pVirtualPixmap->drawable.x = 0;
    pVirtualPixmap->drawable.y = 0;
    pVirtualPixmap->drawable.width = width;
    pVirtualPixmap->drawable.height = height;
    pVirtualPixmap->devKind = devKind;
    pVirtualPixmap->refcnt = 1;
    pVirtualPixmap->devPrivate.ptr = pPixData;
  }
  else
  {
    if (width > 0)
        pPixmap->drawable.width = width;

    if (height > 0)
        pPixmap->drawable.height = height;

    if (depth > 0)
        pPixmap->drawable.depth = depth;

    if (bitsPerPixel > 0)
        pPixmap->drawable.bitsPerPixel = bitsPerPixel;
    else if ((bitsPerPixel < 0) && (depth > 0))
        pPixmap->drawable.bitsPerPixel = BitsPerPixel(depth);

    if (devKind > 0)
        pPixmap->devKind = devKind;
    else if ((devKind < 0) && ((width > 0) || (depth > 0)))
        pPixmap->devKind = PixmapBytePad(pPixmap->drawable.width,
            pPixmap->drawable.depth);

    if (pPixData)
        pVirtualPixmap->devPrivate.ptr = pPixData;

    if (width > 0)
        pVirtualPixmap->drawable.width = width;

    if (height > 0)
        pVirtualPixmap->drawable.height = height;

    if (depth > 0)
        pVirtualPixmap->drawable.depth = depth;

    if (bitsPerPixel > 0)
        pVirtualPixmap->drawable.bitsPerPixel = bitsPerPixel;
    else if ((bitsPerPixel < 0) && (depth > 0))
        pVirtualPixmap->drawable.bitsPerPixel = BitsPerPixel(depth);

    if (devKind > 0)
        pVirtualPixmap->devKind = devKind;
    else if ((devKind < 0) && ((width > 0) || (depth > 0)))
        pVirtualPixmap->devKind = PixmapBytePad(pVirtualPixmap->drawable.width,
            pVirtualPixmap->drawable.depth);

    if (pPixData)
        pVirtualPixmap->devPrivate.ptr = pPixData;

    #ifdef PANIC

    if (pPixmap->drawable.x != 0 || pPixmap->drawable.y != 0)
    {
      fprintf(stderr, "nxagentModifyPixmapHeader: PANIC! Pixmap at [%p] has x [%d] and y [%d].\n",
                  (void *) pPixmap, pPixmap->drawable.x, pPixmap->drawable.y);

      FatalError("nxagentModifyPixmapHeader: PANIC! Pixmap has x or y greater than zero.");
    }

    #endif

    /*
     * This is a quick patch intended to solve the problem of the
     * clipped glyphs drawn in the virtual frame buffer. We should
     * better investigate why pixmaps are requested with a size
     * smaller than actually required. Note that the buffer seems
     * to be never accessed beyond its boundary. This means that
     * the memory for the glyph is allocated using the right size.
     */

    if (width > 0)
    {
      pPixmap->drawable.width *= 4;
      pVirtualPixmap->drawable.width *= 4;
    }

    if (height > 0)
    {
      pPixmap->drawable.height *= 4;
      pVirtualPixmap->drawable.height *= 4;
    }
  }

  return True;
}

static void nxagentPixmapMatchID(void *p0, XID x1, void *p2)
{
  PixmapPtr pPixmap = (PixmapPtr)p0;
  struct nxagentPixmapPair *pPair = p2;

  if ((pPair -> pMap == NULL) && (nxagentPixmap(pPixmap) == pPair -> pixmap))
  {
    pPair -> pMap = pPixmap;
  }
}

PixmapPtr nxagentPixmapPtr(Pixmap pixmap)
{
  int i;
  struct nxagentPixmapPair pair;

  if (pixmap == None)
  {
    return NULL;
  }

  pair.pixmap = pixmap;
  pair.pMap = NULL;

  for (i = 0; (pair.pMap == NULL) && (i < MAXCLIENTS); i++)
  {
    if (clients[i])
    {
      FindClientResourcesByType(clients[i], RT_NX_PIXMAP,
                                    nxagentPixmapMatchID, &pair);
    }
  }

  #ifdef WARNING
  if (pair.pMap == NULL)
  {
    fprintf(stderr, "nxagentFindPixmap: WARNING! Failed to find "
                "remote pixmap [%ld].\n", pair.pixmap);
  }
  #endif

  return pair.pMap;
}

#ifdef TEST

/*
 * We need libssl in order to
 * calculate the md5 of a buffer,
 * so is better to disable this
 * TEST function even if TEST
 * is enabled.
 */

#if 0
extern char *nxagentStringMD5(char*, int);
#endif
#define nxagentStringMD5(A,B) "MD5 is disabled"
#endif

Bool nxagentReconnectPixmapData(PixmapPtr pPixmap, GCPtr pGC)
{
  extern int nxagentImageLength(int, int, int, int, int);

  int width, height, depth, length;
  unsigned int leftPad, format;
  char *data;

  leftPad = 0;

  width  = pPixmap -> drawable.width;
  height = pPixmap -> drawable.height;
  depth  = pPixmap -> drawable.depth;

  #ifdef TEST
  fprintf(stderr, "nxagentReconnectPixmapData: Reconnecting pixmap with geometry [%d][%d][%d].\n",
              width, height, depth);
  #endif

  format = (depth == 1) ? XYPixmap : ZPixmap;

  length = nxagentImageLength(width, height, format, leftPad, depth);

  if ((data = xalloc(length)) == NULL)
  {
    fprintf(stderr, "nxagentReconnectPixmapData: xalloc failed\n");

    return 0;
  }

  fbGetImage((DrawablePtr)nxagentVirtualPixmap(pPixmap),
              0, 0, width, height, format, AllPlanes, data);

  #ifdef TEST
  fprintf(stderr, "nxagentReconnectPixmapData: Gotten data length [%d] with checksum [0x%s].\n",
              length, nxagentStringMD5(data, length));
  #endif

  nxagentPutImage(pPixmap, pGC, depth, 0, 0, width, height, leftPad, format, data);

  xfree(data);

  return True;
}

/*
 * Reconnection stuff.
 */

int nxagentDestroyNewPixmapResourceType(pointer p, XID id)
{
  /*
   * Address of the destructor is set in Init.c.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentDestroyNewPixmapResourceType: Destroying mirror id [%ld] for pixmap at [%p].\n",
              nxagentPixmapPriv((PixmapPtr) p) -> mid, (void *) p);
  #endif

  nxagentPixmapPriv((PixmapPtr) p) -> mid = None;

  return True;
}

void nxagentDisconnectPixmap(void *p0, XID x1, void* p2)
{
  PixmapPtr pPixmap = (PixmapPtr) p0;

  Bool *pBool;

  pBool = (Bool*) p2;

  #ifdef TEST
  fprintf(stderr, "nxagentDisconnectPixmap: Called with bool [%d] and pixmap at [%p].\n",
              *pBool, (void *) pPixmap);

  fprintf(stderr, "nxagentDisconnectPixmap: Virtual pixmap is [%ld].\n",
              nxagentPixmap(pPixmap));
  #endif

  nxagentPixmap(pPixmap) = None;
}

Bool nxagentDisconnectAllPixmap(void)
{
  Bool r;
  int i;

  #ifdef TEST
  fprintf(stderr, "nxagentDisconnectAllPixmap: Going to iterate through pixmap resources.\n");
  #endif

  for (i = 0, r = 1; i < MAXCLIENTS; r = 1, i++)
  {
    if (clients[i])
    {
      FindClientResourcesByType(clients[i], RT_NX_PIXMAP, nxagentDisconnectPixmap, &r);

      #ifdef WARNING

      if (r == False)
      {
        fprintf(stderr, "nxagentDisconnectAllPixmap: WARNING! Failed to disconnect "
                    "pixmap for client [%d].\n", i);
      }

      #endif
    }
  }

  r = nxagentScratchGCsDestroy();

  return r;
}

void nxagentReconnectPixmap(void *p0, XID x1, void *p2)
{
  PixmapPtr pPixmap = (PixmapPtr) p0;
  Bool *pBool = (Bool*) p2;
  GCPtr pGC;
  nxagentPrivPixmapPtr pPixmapPriv;
  extern ScreenPtr nxagentDummyScreen;


  #ifdef TEST
  fprintf(stderr, "nxagentReconnectPixmap: Called with bool [%d] and pixmap at [%p].\n",
              *pBool, (void *) pPixmap);

  fprintf(stderr, "nxagentReconnectPixmap: Virtual pixmap is at [%p] picture is at [%p].\n",
              (void *) nxagentPixmapPriv(pPixmap) -> pVirtualPixmap,
                  (void *) nxagentPixmapPriv(pPixmap) -> pPicture);
  #endif

  if (!*pBool || !pPixmap)
  {
    return;
  }

  if (pPixmap == nxagentDummyScreen -> pScratchPixmap)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentReconnectPixmap: Ignoring scratch pixmap at [%p].\n", (void*) pPixmap);
    #endif

    return;
  }

  pPixmapPriv = nxagentPixmapPriv(pPixmap);

  if (pPixmap -> drawable.width && pPixmap -> drawable.height)
  {
    pPixmapPriv -> id = XCreatePixmap(nxagentDisplay,
                                      nxagentDefaultWindows[pPixmap -> drawable.pScreen -> myNum],
                                      pPixmap -> drawable.width,
                                      pPixmap -> drawable.height,
                                      pPixmap -> drawable.depth);

    nxagentPixmap(pPixmapPriv -> pVirtualPixmap) = pPixmapPriv -> id;

    #ifdef TEST
    fprintf(stderr, "nxagentReconnectPixmap: Created virtual pixmap with id [%ld] for pixmap at [%p].\n",
                nxagentPixmap(pPixmap), (void *) pPixmap);
    #endif

    if (pPixmap == (PixmapPtr) nxagentDummyScreen -> devPrivate)
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentReconnectPixmap: WARNING! Pixmap is root screen. Returning.\n");
      #endif

      return;
    }

    pGC = nxagentGetTempGC(pPixmap);

    if (pGC == NULL)
    {
      #ifdef PANIC
      fprintf(stderr, "nxagentReconnectPixmap: PANIC! Failed to create the temporary GC. Returning.\n");
      #endif

      *pBool = False;

      return;
    }

    *pBool = nxagentReconnectPixmapData(pPixmap, pGC);

    if (!*pBool)
    {
      #ifdef PANIC
      fprintf(stderr, "nxagentReconnectPixmap: PANIC! Failed to execute the put image request.\n");
      #endif
    }
  }
}

Bool nxagentReconnectAllPixmap(void *p0)
{
  ScreenPtr pScreen;
  int i;
  Bool r;

  #ifdef TEST
  fprintf(stderr, "nxagentReconnectAllPixmap: Going to recreate all pixmaps.\n");
  #endif

  /*
   * Reset the geometry and alpha information
   * used by proxy to unpack the packed images.
   */

  nxagentResetVisualCache();

  nxagentResetAlphaCache();

  nxagentScratchGCsCreate();

  for (i = 0, r = 1; i < MAXCLIENTS; r = 1, i++)
  {
    if (clients[i])
    {
      /*
       * This is a little hack. Probably we
       * don't need this as we are not going
       * to split the images anyway.
       */

      requestingClient = clients[i];

      FindClientResourcesByType(clients[i], RT_NX_PIXMAP, nxagentReconnectPixmap, &r);

      #ifdef WARNING

      if (r == False)
      {
        fprintf(stderr, "nxagentReconnectAllPixmap: WARNING! Failed to recreate "
                    "pixmap for client [%d].\n", i);
      }

      #endif
    }
  }

  for (i = 0, r = 1; i < screenInfo.numScreens; r = 1, i++)
  {
    pScreen = screenInfo.screens[i];

    if (pScreen && pScreen -> pScratchPixmap)
    {
      nxagentReconnectPixmap(pScreen -> pScratchPixmap, 0, &r);

      #ifdef WARNING

      if (r == False)
      {
        fprintf(stderr, "nxagentReconnectAllPixmap: WARNING! Failed to recreate "
                    "scratch pixmap for screen at [%p].\n", (void *) pScreen);
      }

      #endif
    }
  }

  if (!nxagentFreeAllTempGC())
  {
    fprintf(stderr, "nxagentReconnectAllPixmap: free of temporary GCs failed\n");
  }

  return True;
}

typedef struct nxagentGCCache_t{
  int depth;
  GCPtr gc;
}nxagentGCCache_t;

nxagentGCCache_t *nxagentGCCache;
int      nxagentGCCacheSize = 0;

static GCPtr nxagentGetTempGC(PixmapPtr pPixmap)
{
  int i, ret;
  GCPtr pGC;
  nxagentGCCache_t *tmp;

  for (i = 0; i < nxagentGCCacheSize; i++)
  {
      if (pPixmap->drawable.depth == nxagentGCCache[i].depth)
      {
        return nxagentGCCache[i].gc;
      }
  }

  tmp = xrealloc(nxagentGCCache, (nxagentGCCacheSize + 1) * sizeof(nxagentGCCache_t));

  if (!tmp)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentGetTempGC: couldn't allocate memory to recreate Pixmaps\n");
    #endif

    return NULL;
  }

  nxagentGCCache = tmp;

  pGC = CreateGC((DrawablePtr)pPixmap, 0, 0, &ret);

  if (ret != Success)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentGetTempGC: failed to create temporary GC of depth %d\n", pPixmap -> drawable.depth);
    #endif

    return NULL;
  }

  nxagentGCCache[nxagentGCCacheSize].gc = pGC;
  nxagentGCCache[nxagentGCCacheSize].depth = pPixmap->drawable.depth;
  nxagentGCCacheSize++;

  return nxagentGetTempGC(pPixmap);
}

static Bool nxagentFreeAllTempGC(void)
{
  int i, status;

  for (i = 0; i < nxagentGCCacheSize; i++)
  {
    status = FreeGC(nxagentGCCache[i].gc, (XID)0);

    if (status != Success)
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentFreeAllTempGC: Failed to free temporary GC %p: FreeGC returned %d.\n",
                  (void*) nxagentGCCache[i].gc, status);
      #endif
    }
  }

  xfree(nxagentGCCache);
  nxagentGCCache = NULL;
  nxagentGCCacheSize = 0;

  return True;
}

#ifdef TEST

static void nxagentCheckOnePixmapIntegrity(void *p0, XID x1, void *p2)
{
  PixmapPtr pPixmap = (PixmapPtr) p0;
  Bool      *pBool = (Bool*) p2;
  extern ScreenPtr nxagentDummyScreen;

  if (*pBool == False)
  {
    return;
  }

  if (pPixmap == nxagentDummyScreen -> devPrivate)
  {
    #ifdef TEST
    fprintf(stderr, "nxagentCheckOnePixmapIntegrity: pixmap %p is screen.\n", (void*) pPixmap);
    #endif

    return;
  }

  if (pPixmap == nxagentDummyScreen -> PixmapPerDepth[0])
  {
    #ifdef TEST
    fprintf(stderr, "nxagentCheckOnePixmapIntegrity: pixmap %p is default stipple of screen.\n", (void*) pPixmap);
    #endif

    return;
  }

  *pBool = nxagentCheckPixmapIntegrity(pPixmap);
}

Bool nxagentCheckPixmapIntegrity(PixmapPtr pPixmap)
{
  Bool integrity = True;
  XImage *image;
  char *data;
  int format;
  unsigned long plane_mask = AllPlanes;
  unsigned int width, height, length, depth;
  PixmapPtr pVirtual = nxagentVirtualPixmap(pPixmap);
  extern int nxagentImageLength(int, int, int, int, int);
  extern int nxagentImagePad(int, int, int, int);

  width = pPixmap -> drawable.width;
  height = pPixmap -> drawable.height;
  depth = pPixmap -> drawable.depth;
  format = (depth == 1) ? XYPixmap : ZPixmap;

  if (width && height)
  {
     length = nxagentImageLength(width, height, format, 0, depth);
     data = malloc(length);

     if (data == NULL)
     {
       FatalError("nxagentCheckPixmapIntegrity: Failed to allocate a buffer of size %d.\n", length);
     }

     memset(data, 0, length);

     image = XGetImage(nxagentDisplay, nxagentPixmap(pPixmap), 0, 0,
                                   width, height, plane_mask, format);
     if (image == NULL)
     {
       FatalError("XGetImage: Failed.\n");
       return False;
     }

     #ifdef WARNING
     fprintf(stderr, "nxagentCheckPixmapIntegrity: image from X has   %d length  and MD5=[0x%s].\n",
                 length, nxagentStringMD5(image->data, length));
     #endif

     NXCleanInPlaceImage(image);

     #ifdef WARNING
     fprintf(stderr, "nxagentCheckPixmapIntegrity: ximage after clean %d length  and MD5=[0x%s].\n",
                 length, nxagentStringMD5(image->data, length));
     #endif


     fbGetImage((DrawablePtr)pVirtual, 0, 0, width, height, format, plane_mask, data);

     #ifdef WARNING
     fprintf(stderr, "nxagentCheckPixmapIntegrity: image from fb has  %d length and MD5=[0x%s].\n",
                 length, nxagentStringMD5(data, length));
     #endif

     if (image && memcmp(image->data, data, length) != 0)
     {
       #if 0
       XlibGC gc;
       XGCValues value;
       Window win1, win2;
       char *data_backup;
       static int first_time = True;
       #endif
       #ifdef WARNING
       int i;
       char *p, *q;
       #endif

       integrity = False;

       #ifdef WARNING
       for (i = 0, p = image->data, q = data; i < length; i++)
       {
         if (p[i] != q[i])
         {
           fprintf(stderr, "[%d] %d - %d !!!!!!!!!!!!!!!!!!! **************** !!!!!!!!!!!!!!!!!\n", i, p[i], q[i]);
         }
         else
         {
           fprintf(stderr, "[%d] %d - %d\n", i, p[i], q[i]);
         }
       }
       #endif

       NXCleanInPlaceImage(image);

       #ifdef WARNING
       fprintf(stderr, "nxagentCheckPixmapIntegrity: Pixmap %p width %d, height %d, has been realized "
                 "but the data buffer still differs.\n", (void*) pPixmap, width, height);
       fprintf(stderr, "nxagentCheckPixmapIntegrity: bytes_per_line = %d byte pad %d format %d.\n",
                 image -> bytes_per_line, nxagentImagePad(width, height, 0, depth), image->format);

       FatalError("nxagentCheckPixmapIntegrity: image is corrupted!!\n");
       #endif


       #if 0
       if (first_time)
       {
         first_time = False;

         value.foreground = 0xff0000;
         value.background = 0x00000000;
         value.plane_mask = 0xffffffff;
         value.fill_style = FillSolid;

         win1 = XCreateSimpleWindow(nxagentDisplay, DefaultRootWindow(nxagentDisplay), 2, 2,
                                        width, height, 0, 0xFFCC33, 0xFF);

         win2 = XCreateSimpleWindow(nxagentDisplay, DefaultRootWindow(nxagentDisplay), 300, 300,
                                        width, height, 0, 0xFFCC33, 0xFF);

         gc = XCreateGC(nxagentDisplay, win1, GCBackground |
                            GCForeground | GCFillStyle | GCPlaneMask, &value);

         XMapWindow(nxagentDisplay, win1);
         XMapWindow(nxagentDisplay, win2);
         XClearWindow(nxagentDisplay, win1);
         XClearWindow(nxagentDisplay, win2);

         XCopyArea(nxagentDisplay, nxagentPixmap(pPixmap), win1, gc, 0, 0, width, height, 0, 0);

         data_backup = image -> data;
         image -> data = data;

         XPutImage(nxagentDisplay, win2, gc, image, 0, 0, 0, 0, width, height);
         image -> data = data_backup;

         XFreeGC(nxagentDisplay, gc);
       }
       #endif
     }
     else
     {
       #ifdef TEST
       fprintf(stderr, "nxagentCheckPixmapIntegrity: Pixmap %p has been realized "
                   "now remote and frambuffer data are synchronized.\n", (void*) pPixmap);
       #endif
     }

     if (image)
     {
       XDestroyImage(image);
     }

     if (data)
     {
       free(data);
     }
  }
  else
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentCheckPixmapIntegrity: ignored pixmap %p with geometry (%d,%d).\n",
                (void*) pPixmap, width, height);
    #endif
  }

  return integrity;
}

Bool nxagentCheckAllPixmapIntegrity()
{
  int i;
  Bool imageIsGood = True;

  #ifdef TEST
  fprintf(stderr, "nxagentCheckAllPixmapIntegrity\n");
  #endif

  for (i = 0; (i < MAXCLIENTS) && (imageIsGood); i++)
  {
    if (clients[i])
    {
      FindClientResourcesByType(clients[i], RT_NX_PIXMAP,
                                    nxagentCheckOnePixmapIntegrity, &imageIsGood);

    }
  }

  #ifdef TEST
  fprintf(stderr, "nxagentCheckAllPixmapIntegrity: pixmaps integrity = %d.\n", imageIsGood);
  #endif

  return imageIsGood;
}

#endif

static void nxagentRealizeOnePixmap(void *p0, XID x1, void *p2)
{
  PixmapPtr pPixmap = (PixmapPtr) p0;
  Bool nxagentCompleteSynchronization = *(Bool *)p2;
/*
FIXME: For now never assume congestion state here.

  if (PixmapsBeforeExpose >= 10 || NXTransCongestion())
*/
  if (nxagentCompleteSynchronization == False && PixmapsBeforeExpose >= 10)
  {
    return;
  }

  #if 0
  if (pPixmap -> drawable.id == None)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentRealizeOnePixmap: Pixmap %p is not a resource.\n", (void*) pPixmap);
    #endif

    return;
  }
  #endif

  #ifdef TEST
  fprintf(stderr, "nxagentRealizeOnePixmap: Realizing pixmap %p geometry (%d,%d).\n",
            (void*)pPixmap, pPixmap->drawable.width, pPixmap->drawable.height);
  #endif

  if (nxagentRealizePixmap(pPixmap) && nxagentCompleteSynchronization == False)
  {
    PixmapsBeforeExpose++;
  }

  #ifdef TEST
  if (nxagentCheckPixmapIntegrity(pPixmap) == False)
  {
    fprintf(stderr, "nxagentRealizeOnePixmap: Pixmap %p after realization is still not synchronized.\n", (void*) pPixmap);
  }
  #endif
}

static void nxagentRealizeTileAndStipple(void *p0, XID x1, void *p2)
{
  GCPtr pGC = (GCPtr) p0;
  unsigned long mask = 0;
  XGCValues values;
  PixmapPtr tile, stipple;

  if (pGC -> tileIsPixel == False)
  {
    tile = nxagentRealPixmap(pGC -> tile.pixmap);

    if (nxagentPixmapStatus(tile) == DRAWABLE_SYNCHRONIZED)
    {
      values.tile = nxagentPixmap(tile);
      mask |= GCTile;

      #ifdef TEST
      fprintf(stderr, "nxagentRealizeTileAndStipple: Tile %p id [%ld] for GC %p.\n",
                  (void *) tile, tile -> drawable.id, pGC);
      #endif
    }
    else
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentTileAndStipple: WARNING! Tile %p id [%ld] for GC %p has not been realized, status is [%s]\n",
                  (void *) tile, tile -> drawable.id, (void *) pGC,
                      nxagentPixmapStatus(tile) == DRAWABLE_NOT_SYNCHRONIZED ? "NOT_SYNCHRONIZED" :
                          nxagentPixmapStatus(tile) == DRAWABLE_SYNCHRONIZED ? "SYNCHRONIZED" : "UNKNOWN");
      #endif
    }
  }

  if (pGC -> stipple)
  {
    stipple = nxagentRealPixmap(pGC -> stipple);

    if (nxagentPixmapStatus(stipple) == DRAWABLE_SYNCHRONIZED)
    {
      values.stipple = nxagentPixmap(stipple);
      mask |= GCStipple;

      #ifdef TEST
      fprintf(stderr, "nxagentRealizeTileAndStipple: Stipple %p id [%ld] for GC %p.\n",
                  (void *) stipple, stipple -> drawable.id, pGC);
      #endif
    }
    else
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentTileAndStipple: WARNING! Stipple %p id [%ld] for GC %p has not been realized, status is [%s]\n",
                  (void*) stipple, stipple -> drawable.id, (void*) pGC,
                      nxagentPixmapStatus(stipple) == DRAWABLE_NOT_SYNCHRONIZED ? "NOT_SYNCHRONIZED" :
                          nxagentPixmapStatus(stipple) == DRAWABLE_SYNCHRONIZED ? "SYNCHRONIZED" : "UNKNOWN");
      #endif
    }
  }

  if (mask)
  {
    XChangeGC(nxagentDisplay, nxagentGC(pGC), mask, &values);
  }
}

void nxagentSwitchImagePolicy(int sig)
{
  #ifdef WARNING
  fprintf(stderr, "nxagentSwitchImagePolicy: Switching to %s policy.\n",
              nxagentOption(Lazy) ? "EAGER" : "LAZY");
  #endif

  nxagentChangeOption(Lazy, !nxagentOption(Lazy));
}

void nxagentRealizeAllPixmaps(int sig)
{
  int i;
  Bool nxagentCompleteSynchronization = True;
  #ifdef WARNING
  fprintf(stderr, "nxagentRealizeAllPixmaps: Switching to EAGER policy.\n");
  fprintf(stderr, "nxagentRealizeAllPixmaps: Going to realize all pixmaps and GC tiles and stipples.\n");
  #endif

  nxagentChangeOption(Lazy, False);

  for (i = 0; i < MAXCLIENTS; i++)
  {
    if (clients[i])
    {
      FindClientResourcesByType(clients[i], RT_NX_PIXMAP,
                                    nxagentRealizeOnePixmap, &nxagentCompleteSynchronization);
    }
  }

  for (i = 0; i < MAXCLIENTS; i++)
  {
    if (clients[i])
    {
      FindClientResourcesByType(clients[i], RT_GC,
                                    nxagentRealizeTileAndStipple, NULL);
    }
  }
}

void nxagentSynchronizeSomePixmaps()
{
  int i;
  extern void nxagentSendExposeToDirtyWindows();
  #ifdef TEST
  int PixmapsBefore = PixmapsBeforeExpose;
  #endif
  Bool nxagentCompleteSynchronization = False;

/*
FIXME: For now never assume congestion state here.

  for (i = 0; i < MAXCLIENTS && PixmapsBeforeExpose < 10 && !NXTransCongestion(); i++)
*/
  for (i = 0; i < MAXCLIENTS && PixmapsBeforeExpose < 10; i++)
  {
    if (clients[i])
    {
      FindClientResourcesByType(clients[i], RT_NX_PIXMAP,
                                    nxagentRealizeOnePixmap, &nxagentCompleteSynchronization);
    }
  }

  #ifdef TEST
  fprintf(stderr, "nxagentSynchronizeSomePixmaps: Total realized pixmaps %d this time realized %d.\n",
              PixmapsBeforeExpose, PixmapsBeforeExpose - PixmapsBefore);
  #endif

/*
FIXME: For now never assume congestion state here.

  if (PixmapsBeforeExpose == 0 || NXTransCongestion())
*/
  if (PixmapsBeforeExpose == 0)
  {
    return;
  }

  #ifdef WARNING
  fprintf(stderr, "nxagentSynchronizeSomePixmaps: Got threshold of pixmap "
              "to realize, going to expose windows.\n");
  #endif

  for (i = 0; i < MAXCLIENTS; i++)
  {
    if (clients[i])
    {
      FindClientResourcesByType(clients[i], RT_GC,
                                    nxagentRealizeTileAndStipple, NULL);
    }
  }

  for (i = 0; i < MAXCLIENTS; i++)
  {
    if (clients[i])
    {
      FindClientResourcesByType(clients[i], RT_WINDOW,
                                    nxagentSendExposeToDirtyWindows, NULL);
    }
  }

  PixmapsBeforeExpose = 0;
}
