/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com.          */
/*                                                                        */
/* NXAGENT, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <signal.h>

#include "X.h"
#include "Xproto.h"
#include "Xpoll.h"
#include "mi.h"
#include "fb.h"

#include "NX.h"
#include "NXalert.h"

#include "Agent.h"
#include "Drawable.h"
#include "Control.h"
#include "Reconnect.h"
#include "Display.h"
#include "Dialog.h"
#include "Window.h"
#include "Dialog.h"
#include "Args.h"

#include NXAGENT_NXLIB_INCLUDE

/*
 * Set here the required log level.
 */

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

extern void nxagentProcessOptionsFile(void);
extern Bool nxagentReconnectDisplay(void*);
extern Bool nxagentReconnectScreen(void*);
extern Bool nxagentReconnectAllFont(void*);
extern Bool nxagentReconnectAllPixmap(void*);
extern Bool nxagentReconnectAllGC(void*);
extern Bool nxagentReconnectAllCursor(void*);
extern Bool nxagentReconnectAllColormap(void*);
extern Bool nxagentReconnectAllWindow(void*);
extern Bool nxagentReconnectAllGlyphSet(void*);
extern Bool nxagentReconnectAllPictFormat(void*);
extern Bool nxagentReconnectAllPicture(void*);
extern Bool nxagentSetWindowsCursors(void*);
extern void nxagentDeactivatePointerGrab();
extern void nxagentMapRootWindow(void);
extern void nxagentUnmapWindows(void);

extern void nxagentFreeGCList(void);
extern void nxagentFreePropertyList(void);

extern Bool nxagentDisconnectAllPicture(void);
extern Bool nxagentDisconnectAllWindow(void);
extern Bool nxagentDisconnectAllCursor(void);
extern Bool nxagentDisconnectAllPixmap(void);
extern Bool nxagentDisconnectAllGC(void);
extern Bool nxagentDisconnectAllFont(void);
extern Bool nxagentDisconnectDisplay(void);

extern void nxagentBackupDisplayInfo(void);
extern void nxagentCleanBackupDisplayInfo(void);

static char *nxagentGetReconnectError(void);

#define NXAGENT_RECONNECT_DEFAULT_MESSAGE_SIZE  32

static char *nxagentReconnectErrorMessage = NULL;
static int  nxagentReconnectErrorId;

extern Bool nxagentRenderEnable;

int             nxagentReconnectTrap;

enum SESSION_STATE nxagentSessionState = SESSION_STARTING;

#ifdef WARNING

#define DECODE_SESSION_STATE(num) \
         ((num) == SESSION_STARTING ? "SESSION_STARTING" : \
          (num) == SESSION_UP ? "SESSION_UP" : \
          (num) == SESSION_GOING_UP? "SESSION_GOING_UP" : \
          (num) == SESSION_DOWN ? "SESSION_DOWN" : \
          (num) == SESSION_GOING_DOWN? "SESSION_GOING_DOWN" : \
          "UNKNOWN")

#endif

enum RECONNECTION_STEP
{
  DISPLAY_STEP = 0,
  SCREEN_STEP,
  FONT_STEP,
  PIXMAP_STEP,
  GC_STEP,
  CURSOR_STEP,
  COLORMAP_STEP,
  WINDOW_STEP,
  GLYPHSET_STEP,
  PICTFORMAT_STEP,
  PICTURE_STEP,
  STEP_NONE
};

void *reconnectLossyLevel[STEP_NONE];

static enum RECONNECTION_STEP failedStep;

static void nxagentSighupHandler(int signal)
{
  #ifdef TEST
  fprintf(stderr, "nxagentSighupHandler: Handling signal with state [%s] and transport [%d] and generation [%ld].\n",
              DECODE_SESSION_STATE(nxagentSessionState), NXTransRunning(), serverGeneration);
  #endif

  if (signal == SIGHUP)
  {
    if (dispatchException & DE_TERMINATE)
    {
      #ifdef TEST
      fprintf(stderr, "nxagentSighupHandler: Forwarding signal with dispatch exception [%d].\n",
                  dispatchException);
      #endif

      NXTransSignal(SIGHUP, NX_SIGNAL_RAISE);
    }
    else if (nxagentSessionState == SESSION_UP)
    {
      nxagentSessionState = SESSION_GOING_DOWN;

      #ifdef TEST
      fprintf(stderr, "nxagentSighupHandler: Handling signal [%d] by disconnecting the agent.\n",
                  signal);
      fprintf(stderr, "nxagentSighupHandler: Forwarding signal with state [%s] and exception [%d].\n",
                  DECODE_SESSION_STATE(nxagentSessionState), dispatchException);
      #endif

      NXTransSignal(SIGHUP, NX_SIGNAL_RAISE);
    }
    else if (nxagentSessionState == SESSION_STARTING)
    {
      nxagentSetGiveUp();

      #ifdef WARNING
      fprintf(stderr, "nxagentSighupHandler: Handling signal [%d] by terminating the agent.\n", signal);
      #endif
    }
    else if (nxagentSessionState == SESSION_DOWN &&
                 NXTransRunning() == 0)
    {
      nxagentSessionState = SESSION_GOING_UP;

      #ifdef TEST
      fprintf(stderr, "nxagentSighupHandler: Handling signal [%d] by reconnecting the agent.\n",
                  signal);
      #endif
    }
    else
    {
      #ifdef TEST
      fprintf(stderr, "nxagentSighupHandler: Forwarding signal with state [%s] and exception [%d].\n",
                  DECODE_SESSION_STATE(nxagentSessionState), dispatchException);
      #endif

      NXTransSignal(SIGHUP, NX_SIGNAL_RAISE);
    }
  }
  #ifdef TEST
  else
  {
    fprintf(stderr, "nxagentSighupHandler: PANIC! Invalid signal [%d] received in state [%s].\n",
                signal, DECODE_SESSION_STATE(nxagentSessionState));
  }
  #endif
}

void nxagentInitializeRecLossyLevel()
{
  *(int *)reconnectLossyLevel[DISPLAY_STEP]    = 0;
  *(int *)reconnectLossyLevel[SCREEN_STEP]     = 0;
  *(int *)reconnectLossyLevel[FONT_STEP]       = 0;
  *(int *)reconnectLossyLevel[PIXMAP_STEP]     = 0;
  *(int *)reconnectLossyLevel[GC_STEP]         = 0;
  *(int *)reconnectLossyLevel[CURSOR_STEP]     = 0;
  *(int *)reconnectLossyLevel[COLORMAP_STEP]   = 0;
  *(int *)reconnectLossyLevel[WINDOW_STEP]     = 0;
  *(int *)reconnectLossyLevel[GLYPHSET_STEP]   = 0;
  *(int *)reconnectLossyLevel[PICTFORMAT_STEP] = 0;
  *(int *)reconnectLossyLevel[PICTURE_STEP]    = 0;
}

void nxagentInitReconnector(void)
{
  struct sigaction sa;

  #ifdef TEST
  fprintf(stderr, "nxagentInitReconnector: Initializing the SIGHUP handler.\n");
  #endif

  sa.sa_handler = nxagentSighupHandler;
  sigfillset(&sa.sa_mask);
  sa.sa_flags = SA_RESTART;

  sigaction(SIGHUP, &sa, NULL);

  nxagentReconnectTrap = 0;

  reconnectLossyLevel[DISPLAY_STEP]    = xalloc(sizeof(int));
  reconnectLossyLevel[SCREEN_STEP]     = xalloc(sizeof(int));
  reconnectLossyLevel[FONT_STEP]       = xalloc(sizeof(int));
  reconnectLossyLevel[PIXMAP_STEP]     = xalloc(sizeof(int));
  reconnectLossyLevel[GC_STEP]         = xalloc(sizeof(int));
  reconnectLossyLevel[CURSOR_STEP]     = xalloc(sizeof(int));
  reconnectLossyLevel[COLORMAP_STEP]   = xalloc(sizeof(int));
  reconnectLossyLevel[WINDOW_STEP]     = xalloc(sizeof(int));
  reconnectLossyLevel[GLYPHSET_STEP]   = xalloc(sizeof(int));
  reconnectLossyLevel[PICTFORMAT_STEP] = xalloc(sizeof(int));
  reconnectLossyLevel[PICTURE_STEP]    = xalloc(sizeof(int));
}

void nxagentDisconnect(void)
{
  /*
   * We need more work on ensuring that the agent is
   * always left in a consistent state. At the moment
   * we have troubles handling the case when we have
   * begun the disconnect procedure but the X con-
   * nection is closed because of a network failure.
   */

  #ifdef WARNING
  fprintf(stderr, "nxagentDisconnect: Disconnecting session with state [%s].\n",
              DECODE_SESSION_STATE(nxagentSessionState));
  #endif

  nxagentInitializeRecLossyLevel();

  nxagentBackupDisplayInfo();

  if (nxagentOption(Rootless));
  {
    nxagentFreePropertyList();
  }

  if (nxagentRenderEnable)
  {
    nxagentDisconnectAllPicture();
  }

  nxagentDisconnectAllWindow();
  nxagentDisconnectAllCursor();
  nxagentDisconnectAllPixmap();
  nxagentDisconnectAllGC();
  nxagentDisconnectAllFont();
  nxagentDisconnectDisplay();
}

Bool nxagentReconnect(void)
{
  int alert = 0;

  nxagentResizeDesktopAtStartup = False;

  /*
   * We need to zero out every new XID
   * created by the disconnected display.
   */

  nxagentDisconnect();

  /*
   * Set this in order to let the screen
   * function to behave differently at
   * reconnection time.
   */

  nxagentReconnectTrap = True;

  nxagentSetReconnectError(0, NULL);

  nxagentProcessOptionsFile();

  if (!nxagentReconnectDisplay(reconnectLossyLevel[DISPLAY_STEP]))
  {
    failedStep = DISPLAY_STEP;

    goto nxagentReconnectError;
  }

  if (!nxagentReconnectScreen(reconnectLossyLevel[SCREEN_STEP]))
  {
    failedStep = SCREEN_STEP;

    goto nxagentReconnectError;
  }

  /*
   * We opened the display and the agent
   * window. Inform the user that the
   * resume may take a while.
   */

  if (nxagentOption(DisplayLatency) > 200)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentReconnect: Requesting the start reconnection "
                "dialog with latency [%d] Ms.\n", nxagentOption(DisplayLatency));
    #endif

    NXTransAlert(START_RESUME_SESSION_ALERT, NX_ALERT_REMOTE);

    alert = 1;
  }

  if (!nxagentReconnectAllFont(reconnectLossyLevel[FONT_STEP]))
  {
    failedStep = FONT_STEP;

    goto nxagentReconnectError;
  }

  if (!nxagentReconnectAllPixmap(reconnectLossyLevel[PIXMAP_STEP]))
  {
    failedStep = PIXMAP_STEP;

    goto nxagentReconnectError;
  }

  if (!nxagentReconnectAllGC(reconnectLossyLevel[GC_STEP]))
  {
    failedStep = GC_STEP;

    goto nxagentReconnectError;
  }

  if (!nxagentReconnectAllColormap(reconnectLossyLevel[COLORMAP_STEP]))
  {
    failedStep = COLORMAP_STEP;

    goto nxagentReconnectError;
  }

  if (!nxagentReconnectAllWindow(reconnectLossyLevel[WINDOW_STEP]))
  {
    failedStep = WINDOW_STEP;

    goto nxagentReconnectError;
  }

  if (nxagentRenderEnable)
  {
    if (!nxagentReconnectAllGlyphSet(reconnectLossyLevel[GLYPHSET_STEP]))
    {
      failedStep = GLYPHSET_STEP;

      goto nxagentReconnectError;
    }

    if (!nxagentReconnectAllPictFormat(reconnectLossyLevel[PICTFORMAT_STEP]))
    {
      failedStep = PICTFORMAT_STEP;

      goto nxagentReconnectError;
    }

    if (!nxagentReconnectAllPicture(reconnectLossyLevel[PICTURE_STEP]))
    {
      failedStep = PICTURE_STEP;

      goto nxagentReconnectError;
    }
  }

  if (!nxagentReconnectAllCursor(reconnectLossyLevel[CURSOR_STEP]))
  {
    failedStep = CURSOR_STEP;

    goto nxagentReconnectError;
  }

  if (!nxagentSetWindowsCursors(reconnectLossyLevel[WINDOW_STEP]))
  {
    failedStep = WINDOW_STEP;

    goto nxagentReconnectError;
  }

  nxagentDeactivatePointerGrab();

  nxagentWakeupByReconnect();

  nxagentCleanBackupDisplayInfo();

  nxagentFreeGCList();

  nxagentMapRootWindow();

  if (nxagentResizeDesktopAtStartup)
  {
    extern void nxagentRRSetScreenConfig(ScreenPtr, int, int);
    extern ScreenPtr nxagentDummyScreen;

    nxagentRRSetScreenConfig(nxagentDummyScreen, nxagentOption(RootWidth), nxagentOption(RootHeight));

    nxagentResizeDesktopAtStartup = False;
  }

  nxagentReconnectTrap = False;

  /*
   * Get rid of the dialog.
   */

  if (alert == 1)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentReconnect: Displacing the start reconnection dialog.\n");
    #endif

    NXTransAlert(DISPLACE_MESSAGE_ALERT, NX_ALERT_REMOTE);
  }

  if (nxagentSessionState != SESSION_GOING_UP)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentReconnect: Session state is [%s].\n", DECODE_SESSION_STATE(nxagentSessionState));
    #endif

    goto nxagentReconnectError;
  }

  fprintf(stderr, "Info: Reconnection succeded.\n");

  return True;

nxagentReconnectError:

  if (alert == 1)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentReconnect: Displacing the start reconnection dialog.\n");
    #endif

    NXTransAlert(DISPLACE_MESSAGE_ALERT, NX_ALERT_REMOTE);
  }

  if (NXDisplayError(nxagentDisplay) == False)
  {
    nxagentUnmapWindows();

    nxagentFailedReconnectionDialog(nxagentReconnectErrorId, nxagentGetReconnectError());
  }
  else
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentReconnect: Reconnection failed: "
                "display not succesfully opened, cannot launch dialog.\n");
    #endif
  }

  nxagentDisconnect();

  fprintf(stderr, "Info: Reconnection failed: %s\n", nxagentGetReconnectError());

  if (failedStep == FONT_STEP)
  {
    *(int *) reconnectLossyLevel[FONT_STEP] = 1;
  }

  return False;
}

void nxagentSetReconnectError(int id, char *format, ...)
{
  static int size = 0;

  va_list ap;
  int n;

  if (format == NULL)
  {
    nxagentSetReconnectError(id, "");

    return;
  }

  nxagentReconnectErrorId = id;

  while (1)
  {
    va_start (ap, format);

    n = vsnprintf(nxagentReconnectErrorMessage, size, format, ap);

    va_end(ap);

    if (n > -1 && n < size)
    {
      break;
    }
    if (n > -1)
    {
      size = n + 1;
    }
    else
    {
      /*
       * The vsnprintf() in glibc 2.0.6 would return
       * -1 when the output was truncated. See section
       * NOTES on printf(3).
       */

      size = (size ? size * 2 : NXAGENT_RECONNECT_DEFAULT_MESSAGE_SIZE);
    }

    nxagentReconnectErrorMessage = realloc(nxagentReconnectErrorMessage, size);

    if (nxagentReconnectErrorMessage == NULL)
    {
      FatalError("realloc failed");
    }
  }

  return;
}

static char* nxagentGetReconnectError()
{
  if (nxagentReconnectErrorMessage == NULL)
  {
    nxagentSetReconnectError(nxagentReconnectErrorId, "");
  }

  return nxagentReconnectErrorMessage;
}

void nxagentHandleConnectionChanges()
{
  if (nxagentSessionState == SESSION_GOING_DOWN)
  {
    extern void nxagentBreakXConnection();

    fprintf(stderr, "Info: Suspending session on user request.\n");

    nxagentDisconnect();
    nxagentBreakXConnection();
  }
  else if (nxagentSessionState == SESSION_GOING_UP)
  {
    fprintf(stderr, "Info: User asks for restart the agent.\n");

    if (nxagentReconnect())
    {
      nxagentSessionState = SESSION_UP;
    }
    else
    {
      nxagentSessionState = SESSION_DOWN;
    }
  }
}

