/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com.          */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef Agent_H
#define Agent_H

#include <unistd.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/select.h>

#include "Misc.h"
#include "Transport.h"
#include "Proxy.h"

extern Proxy *proxy;

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

class Agent
{
  public:

  //
  // Must be created by passing the fake descriptor that
  // will be used for simulating socket communication
  // betwen the agent and the proxy. I/O will take place
  // by copying data to the agent's read and write buf-
  // fers.
  //

  Agent(int fd[2]);

  ~Agent();

  AgentTransport *getTransport() const
  {
    return transport_;
  }

  void saveReadMask(fd_set *readSet)
  {
    memcpy(&saveRead_, readSet, sizeof(saveRead_));
  }

  void saveWriteMask(fd_set *writeSet)
  {
    memcpy(&saveWrite_, writeSet, sizeof(saveWrite_));
  }

  void clearReadMask(fd_set *readSet)
  {
    FD_CLR(remoteFd_, readSet);
    FD_CLR(localFd_,  readSet);
  }

  void clearWriteMask(fd_set *writeSet)
  {
    FD_CLR(remoteFd_, writeSet);
    FD_CLR(localFd_, writeSet);
  }

  void setRemoteRead(int *setFDs, fd_set *readSet)
  {
    FD_SET(remoteFd_, readSet);

    if (remoteFd_ >= *setFDs)
    {
      *setFDs = remoteFd_ + 1;
    }
  }

  void setRemoteWrite(int *setFDs, fd_set *writeSet)
  {
    FD_SET(remoteFd_, writeSet);

    if (remoteFd_ >= *setFDs)
    {
      *setFDs = remoteFd_ + 1;
    }
  }

  void setLocalRead()
  {
    proxy -> setPending(localFd_);
  }

  void setProxyRead()
  {
    proxy -> setPending(proxy -> getFd());
  }

  void clearLocalRead(fd_set *readSet)
  {
    FD_CLR(localFd_, readSet);
  }

  void clearProxyRead(fd_set *readSet)
  {
    FD_CLR(proxy -> getFd(), readSet);
  }

  void clearRemoteRead(fd_set *readSet)
  {
    FD_CLR(remoteFd_, readSet);
  }

  void clearRemoteWrite(fd_set *writeSet)
  {
    FD_CLR(remoteFd_, writeSet);
  }

  fd_set *getSavedReadMask()
  {
    return &saveRead_;
  }

  fd_set *getSavedWriteMask()
  {
    return &saveWrite_;
  }

  int getRemoteFd() const
  {
    return remoteFd_;
  }

  int getLocalFd() const
  {
    return localFd_;
  }

  int getProxyFd() const
  {
    return proxy -> getFd();
  }

  int isValid() const
  {
    return (transport_ != NULL);
  }

  void setSelected()
  {
    selected_ = 1;
  }

  void resetSelected()
  {
    selected_ = 0;
  }

  int isSelected() const
  {
    return selected_;
  }

  int localReadable()
  {
    return (transport_ -> readable() > 0);
  }

  //
  // Cache the result to avoid multiple calls.
  // It must be always called before querying
  // the other functions.
  //

  void saveChannelState()
  {
    state_ = (proxy != NULL ? proxy ->
                  canRead(localFd_) : 0);
  }

  int remoteCanRead(const fd_set * const readSet)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Agent: remoteCanRead() is " <<
               (FD_ISSET(remoteFd_, readSet) && transport_ -> dequeuable() > 0)
            << " with FD_ISSET() " << (int) FD_ISSET(remoteFd_, readSet)
            << " and dequeuable " << transport_ -> dequeuable()
            << ".\n" << logofs_flush;
    #endif

    return (FD_ISSET(remoteFd_, readSet) &&
                transport_ -> dequeuable() > 0);
  }

  int remoteCanWrite(const fd_set * const writeSet)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Agent: remoteCanWrite() is " <<
               (FD_ISSET(remoteFd_, writeSet) && transport_ ->
               queuable() > 0 && state_ == 1) << " with FD_ISSET() "
            << (int) FD_ISSET(remoteFd_, writeSet) << " queueable "
            << transport_ -> queuable() << " channel can read "
            << state_ << ".\n" << logofs_flush;
    #endif

    return (FD_ISSET(remoteFd_, writeSet) &&
                transport_ -> queuable() > 0 &&
                    state_ == 1);
  }

  int localCanRead()
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Agent: localCanRead() is " <<
               (transport_ -> readable() > 0 && state_ == 1)
            << " with readable " << transport_ -> readable()
            << " channel can read " << state_ << ".\n"
            << logofs_flush;
    #endif

    return (transport_ -> readable() > 0 &&
                state_ == 1);
  }

  int proxyCanRead()
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Agent: proxyCanRead() is " << proxy -> canRead()
            << ".\n" << logofs_flush;
    #endif

    return (proxy -> canRead());
  }

  int enqueueData(const char *data, const int size) const
  {
    return transport_ -> enqueue(data, size);
  }

  int dequeueData(char *data, int size) const
  {
    return transport_ -> dequeue(data, size);
  }

  int dequeuableData() const
  {
    return transport_ -> dequeuable();
  }

  private:

  int remoteFd_;
  int localFd_;

  fd_set saveRead_;
  fd_set saveWrite_;

  int state_;
  int selected_;

  AgentTransport *transport_;
};

#endif /* Agent_H */
