/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <unistd.h>
#include <sys/types.h>
#include <signal.h>
#include <fcntl.h>

#include "NX.h"

#include "Misc.h"

#include "Types.h"
#include "Timestamp.h"

#include "Control.h"
#include "Statistics.h"
#include "Proxy.h"

#include "Keeper.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

#define DISPLAY_LENGTH_LIMIT  256
#define PATH_LENGTH_LIMIT     512

#undef  CLIENT_USES_DASH_DASH_OPTIONS

//
// These are from the main loop.
//

extern int useUnixSocket;

extern int lastDialog;
extern int lastWatchdog;
extern int lastKeeper;

extern void CleanupGlobal();
extern void CleanupSockets();
extern void InstallSignals();

extern void CheckParent(char *name, char *type, int parent);

//
// Close all the unused descriptors and
// install any signal handler that might
// have been disabled in the main process.
//

static void SystemCleanup(char *name);

//
// Release all objects allocated in the
// heap.

static void MemoryCleanup(char *name);

//
// Start a nxclient process in dialog mode.
//

int NXTransDialog(const char *caption, const char *message,
                      const char *window, const char *type, int local,
                         const char* display)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  int pid;

  #ifdef TEST
  *logofs << "NXTransDialog: Going to fork with NX pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  pid = fork();

  if (pid != 0)
  {
    if (pid < 0)
    {
      #ifdef TEST
      *logofs << "NXTransDialog: WARNING! Function fork failed "
              << "with result '" << pid << "'. Error is " << EGET()
              << " '" << ESTR() << "'.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Function fork failed with result '"
           << pid << "'. Error is " << EGET() << " '" << ESTR()
           << "'.\n";
    }
    #ifdef TEST
    else
    {
      *logofs << "NXTransDialog: Created NX dialog process with pid '"
              << pid << "'.\n" << logofs_flush;
    }
    #endif

    return pid;
  }

  #ifdef TEST
  *logofs << "NXTransDialog: Executing child with pid '" << getpid()
          << "' and parent '" << getppid() << "'.\n"
          << logofs_flush;
  #endif

  SystemCleanup("NXTransDialog");

  //
  // Get rid of the unused resources.
  //

  MemoryCleanup("NXTransDialog");

  #ifdef TEST
  *logofs << "NXTransDialog: Running external NX dialog with caption '"
          << caption << "' message '" << message << "' type '"
          << type << "' local '" << local << "' display '"
          << display << "'.\n"
          << logofs_flush;
  #endif

  #ifdef __APPLE__

  char *client = "/Applications/NX Client for OSX.app/Contents/MacOS/nxclient";

  #else

  char *client = "nxclient";

  #endif

  int pulldown = (strcmp(type, "pulldown") == 0);

  for (int i = 0; i < 2; i++)
  {
    #ifdef CLIENT_USES_DASH_DASH_OPTIONS

    if (local != 0)
    {
      if (pulldown)
      {
        execlp(client, client, "--dialog", type, "--caption", caption,
                   "--window", window, "--local", "--display",
                       display, NULL);
      }
      else
      {
        execlp(client, client, "--dialog", type, "--caption", caption,
                   "--message", message, "--local", "--display",
                       display, NULL);
      }
    }
    else
    {
      if (pulldown)
      {
        execlp(client, client, "--dialog", type, "--caption", caption,
                   "--window", window, "--display",
                       display, NULL);
      }
      else
      {
        execlp(client, client, "--dialog", type, "--caption", caption,
                   "--message", message, "--display",
                       display, NULL);
      }
    }

    #else

    if (local != 0)
    {
      if (pulldown)
      {
        execlp(client, client, "-dialog", type, "-caption", caption,
                   "-window", window, "-local", "-display",
                       display, NULL);
      }
      else
      {
        execlp(client, client, "-dialog", type, "-caption", caption,
                   "-message", message, "-local", "-display",
                       display, NULL);
      }
    }
    else
    {
      if (pulldown)
      {
        execlp(client, client, "-dialog", type, "-caption", caption,
                   "-window", window, "-display",
                       display, NULL);
      }
      else
      {
        execlp(client, client, "-dialog", type, "-caption", caption,
                   "-message", message, "-display",
                       display, NULL);
      }
    }

    #endif

    #ifdef WARNING
    *logofs << "NXTransDialog: WARNING! Couldn't start 'nxclient'. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Couldn't start 'nxclient'. Error is "
         << EGET() << " '" << ESTR() << "'.\n";

    char newPath[PATH_LENGTH_LIMIT];

    strcpy(newPath, "/usr/NX/bin:/opt/NX/bin:/usr/local/NX/bin:");

    #ifdef __APPLE__

    strcat(newPath, "/Applications/NX Client for OSX.app/Contents/MacOS:");

    #endif

    #ifdef __CYGWIN32__

    strcat(newPath, ".:");

    #endif

    int newLength = strlen(newPath);

    char *oldPath = getenv("PATH");

    strncpy(newPath + newLength, oldPath, PATH_LENGTH_LIMIT - newLength - 1);

    newPath[PATH_LENGTH_LIMIT - 1] = '\0';

    #ifdef WARNING
    *logofs << "NXTransDialog: WARNING! Trying with path '"
            << newPath << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Trying with path '" << newPath
         << "'.\n";

    //
    // Solaris doesn't seem to have
    // function setenv().
    //

    #ifdef __sun

    char newEnv[PATH_LENGTH_LIMIT];

    sprintf(newEnv,"PATH=%s", newPath);

    putenv(newEnv);

    #else

    setenv("PATH", newPath, 1);

    #endif
  }

  //
  // Hopefully useless.
  //

  exit(0);
}

//
// Start a nxclient process in dialog mode.
//

int NXTransClient(const char* display)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  int pid;

  #ifdef TEST
  *logofs << "NXTransClient: Going to fork with NX pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  pid = fork();

  if (pid != 0)
  {
    if (pid < 0)
    {
      #ifdef TEST
      *logofs << "NXTransClient: WARNING! Function fork failed "
              << "with result '" << pid << "'. Error is " << EGET()
              << " '" << ESTR() << "'.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Function fork failed with result '"
           << pid << "'. Error is " << EGET() << " '" << ESTR()
           << "'.\n";
    }
    #ifdef TEST
    else
    {
      *logofs << "NXTransClient: Created NX client process with pid '"
              << pid << "'.\n" << logofs_flush;
    }
    #endif

    return pid;
  }

  #ifdef TEST
  *logofs << "NXTransClient: Executing child with pid '" << getpid()
          << "' and parent '" << getppid() << "'.\n"
          << logofs_flush;
  #endif

  SystemCleanup("NXTransClient");

  //
  // Get rid of unused resources.
  //

  MemoryCleanup("NXTransClient");

  #ifdef TEST
  *logofs << "NXTransClient: Running external NX client with display '"
          << display << "'.\n" << logofs_flush;
  #endif

  #ifdef __APPLE__

  char *client = "/Applications/NX Client for OSX.app/Contents/MacOS/nxclient";

  #else

  char *client = "nxclient";

  #endif

  //
  // Provide the display in the environment.
  //

  char newDisplay[DISPLAY_LENGTH_LIMIT];

  #ifdef __sun

  snprintf(newDisplay, DISPLAY_LENGTH_LIMIT - 1, "DISPLAY=%s", display);

  newDisplay[DISPLAY_LENGTH_LIMIT - 1] = '\0';

  putenv(newDisplay);

  #else

  strncpy(newDisplay, display, DISPLAY_LENGTH_LIMIT - 1);

  newDisplay[DISPLAY_LENGTH_LIMIT - 1] = '\0';

  setenv("DISPLAY", newDisplay, 1);

  #endif

  for (int i = 0; i < 2; i++)
  {
    execlp(client, client, NULL);

    #ifdef WARNING
    *logofs << "NXTransClient: WARNING! Couldn't start 'nxclient'. Error is "
            << EGET() << " '" << ESTR() << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Couldn't start 'nxclient'. Error is "
         << EGET() << " '" << ESTR() << "'.\n";

    char newPath[PATH_LENGTH_LIMIT];

    strcpy(newPath, "/usr/NX/bin:/opt/NX/bin:/usr/local/NX/bin:");

    #ifdef __APPLE__

    strcat(newPath, "/Applications/NX Client for OSX.app/Contents/MacOS:");

    #endif

    #ifdef __CYGWIN32__

    strcat(newPath, ".:");

    #endif

    int newLength = strlen(newPath);

    char *oldPath = getenv("PATH");

    strncpy(newPath + newLength, oldPath, PATH_LENGTH_LIMIT - newLength - 1);

    newPath[PATH_LENGTH_LIMIT - 1] = '\0';

    #ifdef WARNING
    *logofs << "NXTransClient: WARNING! Trying with path '" << newPath
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Trying with path '" << newPath
         << "'.\n";

    //
    // Solaris doesn't seem to have
    // function setenv().
    //

    #ifdef __sun

    char newEnv[PATH_LENGTH_LIMIT];

    sprintf(newEnv,"PATH=%s", newPath);

    putenv(newEnv);

    #else

    setenv("PATH", newPath, 1);

    #endif
  }

  //
  // Hopefully useless.
  //

  exit(0);
}

//
// Wait until timeout is elapsed
// and kill the parent process.
//

int NXTransWatchdog(int timeout)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  int pid;

  #ifdef TEST
  *logofs << "NXTransWatchdog: Going to fork with NX pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  pid = fork();

  if (pid != 0)
  {
    if (pid < 0)
    {
      #ifdef TEST
      *logofs << "NXTransWatchdog: WARNING! Function fork failed "
              << "with result '" << pid << "'. Error is " << EGET()
              << " '" << ESTR() << "'.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Function fork failed with result '"
           << pid << "'. Error is " << EGET() << " '" << ESTR()
           << "'.\n";
    }
    #ifdef TEST
    else
    {
      *logofs << "NXTransWatchdog: Created NX watchdog process "
              << "with pid '" << pid << "'.\n" << logofs_flush;
    }
    #endif

    return pid;
  }

  int parent = getppid();

  #ifdef TEST
  *logofs << "NXTransWatchdog: Executing child with pid '"
          << getpid() << "' and parent '" << parent
          << "'.\n" << logofs_flush;
  #endif

  SystemCleanup("NXTransWatchdog");

  //
  // Get rid of unused resources.
  //

  MemoryCleanup("NXTransWatchdog");

  //
  // To cancel the watchdog just kill
  // this process.
  //

  T_timestamp startTs = getTimestamp();

  for (;;)
  {
    //
    // Complain if parent is dead and be sure
    // that we never kill the init process.
    //

    CheckParent("NXTransWatchdog", "watchdog", parent);

    if (diffTimestamp(startTs, getTimestamp()) >= timeout * 1000)
    {
      #ifdef TEST
      *logofs << "NXTransWatchdog: Timeout of " << timeout
              << " seconds " << "raised in watchdog.\n"
              << logofs_flush;
      #endif

      //
      // We will just exit. Our parent should be
      // monitoring us and detect that the process
      // is gone.
      //

      HandleCleanup();
    }

    sleep(1);
  }

  //
  // Hopefully useless.
  //

  exit(0);
}

int NXTransKeeper(int caches, int images, const char *root)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (caches == 0 && images == 0)
  {
    #ifdef TEST
    *logofs << "NXTransKeeper: No NX cache house-keeping needed.\n"
            << logofs_flush;
    #endif

    return 0;
  }

  int pid;

  #ifdef TEST
  *logofs << "NXTransKeeper: Going to fork with NX pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  pid = fork();

  if (pid != 0)
  {
    if (pid < 0)
    {
      #ifdef TEST
      *logofs << "NXTransKeeper: WARNING! Function fork failed with "
              << "result '" << pid << "'. Error is " << EGET()
              << " '" << ESTR() << "'.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Function fork failed with result '"
           << pid << "'. Error is " << EGET() << " '" << ESTR()
           << "'.\n";
    }
    #ifdef TEST
    else
    {
      *logofs << "NXTransKeeper: Created NX keeper process with pid '"
              << pid << "'.\n" << logofs_flush;
    }
    #endif

    return pid;
  }

  int parent = getppid();

  #ifdef TEST
  *logofs << "NXTransKeeper: Executing child with pid '" << getpid()
          << "' and parent '" << parent << "'.\n"
          << logofs_flush;
  #endif

  SystemCleanup("NXTransKeeper");

  #ifdef TEST
  *logofs << "NXTransKeeper: Going to run with caches " << caches
          << " images " << images << " and root " << root
          << " at " << strTimestamp() << ".\n" << logofs_flush;
  #endif

  //
  // Create the house-keeper class.
  //

  int timeout = control -> KeeperTimeout;

  Keeper *keeper = new Keeper(caches, images, root, 100, parent);

  if (keeper == NULL)
  {
    #ifdef PANIC
    *logofs << "NXTransKeeper: PANIC! Failed to create the keeper object.\n"
            << logofs_flush;
     #endif

     cerr << "Error" << ": Failed to create the keeper object.\n";

     HandleCleanup();
  }

  //
  // Get rid of unused resources. Root path
  // must be copied in keeper's constructor
  // before control is deleted.
  //
  
  MemoryCleanup("NXTransKeeper");

  //
  // Decrease the priority of this process.
  //
  // The following applies to Cygwin: "Cygwin processes can be 
  // set to IDLE_PRIORITY_CLASS, NORMAL_PRIORITY_CLASS, HIGH_-
  // PRIORITY_CLASS, or REALTIME_PRIORITY_CLASS with the nice
  // call. If you pass a positive number to nice(), then the
  // priority level will decrease by one (within the above list
  // of priorities). A negative number would make it increase
  // by one. It is not possible to change it by more than one
  // at a time without making repeated calls".
  //

  if (nice(5) < 0 && errno != 0)
  {
    #ifdef WARNING
    *logofs << "NXTransKeeper: WARNING! Failed to renice process to +5. "
            << "Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
     #endif

     cerr << "Warning" << ": Failed to renice process to +5. "
          << "Error is " << EGET() << " '" << ESTR()
          << "'.\n";
  }

  //
  // Delay the first run to give priority
  // to session startup.
  //

  #ifdef TEST
  *logofs << "NXTransKeeper: Going to sleep for " << timeout
          << " Ms.\n" << logofs_flush;
  #endif

  usleep(timeout * 1000);

  CheckParent("NXTransKeeper", "keeper", parent);

  //
  // House keeping of persistent caches
  // is performed only once.
  //

  if (caches != 0)
  {
    #ifdef TEST
    *logofs << "NXTransKeeper: Going to cleanup NX cache directories "
            << "at " << strTimestamp() << ".\n" << logofs_flush;
    #endif

    keeper -> keepCaches();
  }

  CheckParent("NXTransKeeper", "keeper", parent);

  //
  // House keeping of images is only
  // performed at X server side.
  //

  if (images == 0)
  {
    HandleCleanup();
  }

  //
  // Run forever until we are killed
  // by our parent.
  //

  for (;;)
  {
    #ifdef TEST
    *logofs << "NXTransKeeper: Going to cleanup NX images "
            << "directories at " << strTimestamp() << ".\n"
            << logofs_flush;
    #endif

    if (keeper -> keepImages() < 0)
    {
      CheckParent("NXTransKeeper", "keeper", parent);

      HandleCleanup();
    }

    #ifdef TEST
    *logofs << "NXTransKeeper: Completed cleanup of NX images "
            << "directories at " << strTimestamp() << ".\n"
            << logofs_flush;

    *logofs << "NXTransKeeper: Going to sleep for " << timeout
            << " Ms.\n" << logofs_flush;
    #endif

    usleep(timeout * 1000);
  }

  //
  // Hopefully useless.
  //

  exit(0);
}

void SystemCleanup(char *name)
{
  #ifdef TEST
  *logofs << name << ": Performing system cleanup in process "
          << "with pid '" << getpid() << "'.\n"
          << logofs_flush;
  #endif

  //
  // Reinstall signals that might
  // have been restored by agents.
  //

  InstallSignals();
}

void MemoryCleanup(char *name)
{
  #ifdef TEST
  *logofs << name << ": Performing memory cleanup in process "
          << "with pid '" << getpid() << "'.\n"
          << logofs_flush;
  #endif

  DisableSignals();

  //
  // Prevent deletion of unix socket
  // and lock file.
  //

  useUnixSocket = 0;

  //
  // Don't let cleanup kill other
  // children.
  //

  lastDialog   = 0;
  lastWatchdog = 0;
  lastKeeper   = 0;

  CleanupSockets();

  CleanupGlobal();

  EnableSignals();
}
