/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

//
// TODO: Needed to send signal to agent.
// Not implemented in production code.
//
// #include <sys/types.h>
// #include <signal.h>
// #include <unistd.h>
//

#include "Socket.h"
#include "Agent.h"

#include "ClientProxy.h"

#include "ClientChannel.h"
#include "GenericChannel.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

ClientProxy::ClientProxy(int proxyFd) : Proxy(proxyFd)
{
  #ifdef DEBUG
  *logofs << "ClientProxy: Created new object at " << this
          << ".\n" << logofs_flush;
  #endif
}

ClientProxy::~ClientProxy()
{
  #ifdef DEBUG
  *logofs << "ClientProxy: Deleted object at " << this
          << ".\n" << logofs_flush;
  #endif
}

//
// Send a notification event of the given type
// to any active X channel.
//

int ClientProxy::handleNotify(T_notification_type type)
{
  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> getType() == channel_x &&
                channels_[channelId] -> getFinish() == 0)
    {
      if (channels_[channelId] -> handleNotify(type) < 0)
      {
        if (handleFinish(channelId) < 0)
        {
          return -1;
        }
      }

      //
      // TODO: Would force agent to check its event
      // queue. This strategy has been tested but
      // not implemented in current production code.
      //
      // if (control -> SessionMode == SESSION_X)
      // {
      //   kill(getppid(), SIGURG);
      // }
      //
    }
  }

  return 1;
}

int ClientProxy::handleNewXConnection(int xClientFd)
{
  int channelId = xClientFd;

  if (channelId >= CONNECTIONS_LIMIT)
  {
    #ifdef PANIC
    *logofs << "ClientProxy: PANIC! Maximum mumber of available "
            << "channels exceeded.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Maximum mumber of available "
         << "channels exceeded.\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "ClientProxy: X client descriptor FD#" << xClientFd
          << " mapped to channel ID#" << channelId << ".\n"
          << logofs_flush;
  #endif

  //
  // Turn queuing off for path proxy-to-X-client.
  //

  if (control -> OptionClientNoDelay == 1)
  {
    SetNoDelay(xClientFd, control -> OptionClientNoDelay);
  }

  //
  // If requested, set the size of the TCP send
  // and receive buffers.
  //

  if (control -> OptionClientSendBuffer != -1)
  {
    SetSendBuffer(xClientFd, control -> OptionClientSendBuffer);
  }

  if (control -> OptionClientReceiveBuffer != -1)
  {
    SetReceiveBuffer(xClientFd, control -> OptionClientReceiveBuffer);
  }

  //
  // Load the persistent cache if this is
  // the first X channel.
  //

  if (handleCheckLoad(load_if_first) < 0)
  {
    return -1;
  }

  if (allocateTransport(xClientFd, channelId) < 0)
  {
    return -1;
  }

  //
  // Starting from protocol level 3 client and server
  // caches are created in proxy and shared between all
  // channels. If remote proxy has older protocol level
  // pointers are NULL and channels must create their
  // own instances.
  //

  channels_[channelId] = new ClientChannel(transports_[channelId], compressor_,
                                               decompressor_);

  if (channels_[channelId] == NULL)
  {
    deallocateTransport(channelId);

    return -1;
  }

  increaseChannels(channelId);

  //
  // Propagate channel stores and caches to the new
  // channel.
  //

  channels_[channelId] -> setOpcodes(opcodeStore_);

  channels_[channelId] -> setStores(clientStore_, serverStore_);

  if (control -> isProtoStep3() == 1)
  {
    channels_[channelId] -> setCaches(clientCache_, serverCache_);
  }

  #ifdef DEBUG
  *logofs << "ClientProxy: Sending code_new_x_connection message.\n"
          << logofs_flush;
  #endif

  if (handleControl(code_new_x_connection, channelId) < 0)
  {
    return -1;
  }

  //
  // Let channel configure itself according
  // to control parameters.
  //

  channels_[channelId] -> handleConfiguration();

  return 1;
}

int ClientProxy::handleNewAgentConnection(Agent *agent)
{
  int xClientFd = agent -> getLocalFd();

  int channelId = xClientFd;

  if (channelId >= CONNECTIONS_LIMIT)
  {
    #ifdef PANIC
    *logofs << "ClientProxy: PANIC! Maximum mumber of available "
            << "channels exceeded.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Maximum mumber of available "
         << "channels exceeded.\n";

    return -1;
  }

  transports_[channelId] = agent -> getTransport();

  return handleNewXConnection(xClientFd);
}

int ClientProxy::handleNewXConnectionFromProxy(int channelId)
{
  #ifdef PANIC
  *logofs << "ClientProxy: PANIC! Function handleNewXConnectionFromProxy() "
          << "shouldn't be called at client side.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleNewXConnectionFromProxy() "
       << "shouldn't be called at client side.\n";

  HandleCleanup();

  return -1;
}

int ClientProxy::checkNewGenericConnection(int newFd, T_channel_type type, char *label)
{
  int channelId = newFd;

  if (channelId >= CONNECTIONS_LIMIT)
  {
    #ifdef PANIC
    *logofs << "ClientProxy: PANIC! Maximum mumber of available "
            << "channels exceeded.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Maximum mumber of available "
         << "channels exceeded.\n";

    return -1;
  }

  return 1;
}

int ClientProxy::handleNewGenericConnection(int newFd, T_channel_type type, char *label)
{
  int channelId = newFd;

  if (checkNewGenericConnection(newFd, type, label) < 0)
  {
    return -1;
  }

  #ifdef TEST
  *logofs << "ClientProxy: " << label << " descriptor FD#"
          << newFd << " mapped to channel ID#" << channelId
          << ".\n" << logofs_flush;
  #endif

  //
  // Turn queuing off for path server-to-proxy.
  //

  SetNoDelay(newFd, 1);

  if (allocateTransport(newFd, channelId) < 0)
  {
    return -1;
  }

  switch (type)
  {
    case channel_cups:
    {
      channels_[channelId] = new CupsChannel(transports_[channelId],
                                                 compressor_, decompressor_);
      break;
    }
    case channel_keybd:
    {
      channels_[channelId] = new KeybdChannel(transports_[channelId],
                                                  compressor_, decompressor_);
      break;
    }
    case channel_samba:
    {
      channels_[channelId] = new SambaChannel(transports_[channelId],
                                                  compressor_, decompressor_);
      break;
    }
    case channel_media:
    {
      channels_[channelId] = new MediaChannel(transports_[channelId],
                                                  compressor_, decompressor_);
      break;
    }
    default:
    {
      channels_[channelId] = new HttpChannel(transports_[channelId],
                                                 compressor_, decompressor_);
      break;
    }
  }

  if (channels_[channelId] == NULL)
  {
    deallocateTransport(channelId);

    return -1;
  }

  #ifdef TEST
  *logofs << "ClientProxy: Accepted new connection to "
          << label << " server.\n" << logofs_flush;
  #endif

  cerr << "Info" << ": Accepted new connection to "
       << label << " server.\n";

  increaseChannels(channelId);

  switch (type)
  {
    case channel_cups:
    {
      #ifdef DEBUG
      *logofs << "ClientProxy: Sending code_new_cups_connection message.\n"
              << logofs_flush;
      #endif

      if (handleControl(code_new_cups_connection, channelId) < 0)
      {
        return -1;
      }

      break;
    }
    case channel_keybd:
    {
      #ifdef DEBUG
      *logofs << "ClientProxy: Sending code_new_keybd_connection message.\n"
              << logofs_flush;
      #endif

      if (handleControl(code_new_keybd_connection, channelId) < 0)
      {
        return -1;
      }

      break;
    }
    case channel_samba:
    {
      #ifdef DEBUG
      *logofs << "ClientProxy: Sending code_new_samba_connection message.\n"
              << logofs_flush;
      #endif

      if (handleControl(code_new_samba_connection, channelId) < 0)
      {
        return -1;
      }

      break;
    }
    case channel_media:
    {
      #ifdef DEBUG
      *logofs << "ClientProxy: Sending code_new_media_connection message.\n"
              << logofs_flush;
      #endif

      if (handleControl(code_new_media_connection, channelId) < 0)
      {
        return -1;
      }

      break;
    }
    default:
    {
      #ifdef DEBUG
      *logofs << "ClientProxy: Sending code_new_http_connection message.\n"
              << logofs_flush;
      #endif

      if (handleControl(code_new_http_connection, channelId) < 0)
      {
        return -1;
      }

      break;
    }
  }

  channels_[channelId] -> handleConfiguration();

  return 1;
}

int ClientProxy::handleNewCupsConnection(int cupsFD)
{
  return handleNewGenericConnection(cupsFD, channel_cups, "cups");
}

int ClientProxy::handleNewCupsConnectionFromProxy(int channelId)
{
  #ifdef PANIC
  *logofs << "ClientProxy: PANIC! Function handleNewCupsConnectionFromProxy() "
          << "shouldn't be called at client side.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleNewCupsConnectionFromProxy() "
       << "shouldn't be called at client side.\n";

  HandleCleanup();

  return -1;
}

int ClientProxy::handleNewKeybdConnection(int keybdFD)
{
  return handleNewGenericConnection(keybdFD, channel_keybd, "embedded keyboard");
}

int ClientProxy::handleNewKeybdConnectionFromProxy(int channelId)
{
  #ifdef PANIC
  *logofs << "ClientProxy: PANIC! Function handleNewKeybdConnectionFromProxy() "
          << "shouldn't be called at client side.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleNewKeybdConnectionFromProxy() "
       << "shouldn't be called at client side.\n";

  HandleCleanup();

  return -1;
}

int ClientProxy::handleNewSambaConnection(int sambaFD)
{
  return handleNewGenericConnection(sambaFD, channel_samba, "SMB");
}

int ClientProxy::handleNewSambaConnectionFromProxy(int channelId)
{
  #ifdef PANIC
  *logofs << "ClientProxy: PANIC! Function handleNewSambaConnectionFromProxy() "
          << "shouldn't be called at client side.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleNewSambaConnectionFromProxy() "
       << "shouldn't be called at client side.\n";

  HandleCleanup();

  return -1;
}

int ClientProxy::handleNewMediaConnection(int mediaFD)
{
  return handleNewGenericConnection(mediaFD, channel_media, "media");
}

int ClientProxy::handleNewMediaConnectionFromProxy(int channelId)
{
  #ifdef PANIC
  *logofs << "ClientProxy: PANIC! Function handleNewMediaConnectionFromProxy() "
          << "shouldn't be called at client side.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleNewMediaConnectionFromProxy() "
       << "shouldn't be called at client side.\n";

  HandleCleanup();

  return -1;
}

int ClientProxy::handleNewHttpConnection(int httpFD)
{
  return handleNewGenericConnection(httpFD, channel_http, "HTTP");
}

int ClientProxy::handleNewHttpConnectionFromProxy(int channelId)
{
  #ifdef PANIC
  *logofs << "ClientProxy: PANIC! Function handleNewHttpConnectionFromProxy() "
          << "shouldn't be called at client side.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleNewHttpConnectionFromProxy() "
       << "shouldn't be called at client side.\n";

  HandleCleanup();

  return -1;
}

int ClientProxy::handleCheckLoad(T_proxy_load type)
{
  int channelCount = 0;

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> getType() == channel_x)
    {
      channelCount++;
    }
  }

  if ((channelCount == 0 && type == load_if_first) ||
          (channelCount > 0 && type == load_if_any))
  {
    #ifdef TEST
    *logofs << "ClientProxy: Going to load content of client store.\n"
            << logofs_flush;
    #endif

    int result = handleLoad();

    if (result == 1)
    {
      #ifdef TEST
      *logofs << "ClientProxy: Sending code_load_request message.\n" << logofs_flush;
      #endif

      if (handleControl(code_load_request) < 0)
      {
        return -1;
      }
 
      priority_ = 1;
    }
    else if (result < 0)
    {
      #ifdef PANIC
      *logofs << "ClientProxy: PANIC! Failed to load content of persistent cache.\n"
              << logofs_flush;
      #endif

      //
      // From protocol version 3, we don't abort proxy
      // connection in case of a corrupted cache. By
      // not sending a load message to the remote peer,
      // both sides will start encoding using empty
      // stores. This behaviour is compatible with old
      // protocol versions.
      //

      if (channelCount == 0 && type == load_if_first)
      {
        if (handleResetStores() < 0)
        {
          #ifdef PANIC
          *logofs << "Proxy: PANIC! Failed to reset message stores.\n"
                  << logofs_flush;
          #endif

          return -1;
        }
      }
      else
      {
        return -1;
      }
    }
  }

  return 0;
}

int ClientProxy::handleCheckSave()
{
  //
  // If no more X channels are remaining
  // then save content of message stores.
  //

  int channelCount = 0;

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> getType() == channel_x)
    {
      channelCount++;

      break;
    }
  }

  if (channelCount == 0)
  {
    int result = handleSave();

    if (result == 1)
    {
      #ifdef TEST
      *logofs << "ClientProxy: Sending code_save_request message.\n"
              << logofs_flush;
      #endif

      if (handleControl(code_save_request) < 0)
      {
        return -1;
      }

      priority_ = 1;

      return 1;
    }
    else if (result < 0)
    {
      #ifdef PANIC
      *logofs << "ClientProxy: PANIC! Failed to save stores to persistent cache.\n"
              << logofs_flush;
      #endif

      return -1;
    }
  }

  return 0;
}

int ClientProxy::handleLoadFromProxy()
{
  #ifdef PANIC
  *logofs << "ClientProxy: PANIC! Invalid load control message "
          << "received in proxy.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Invalid load control message "
       << "received in proxy.\n";

  return -1;
}

int ClientProxy::handleSaveFromProxy()
{
  #ifdef PANIC
  *logofs << "ClientProxy: PANIC! Invalid save control message "
          << "received in proxy.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Invalid save control message "
       << "received in proxy.\n";

  return -1;
}

int ClientProxy::handleSaveStores(ostream *cachefs, md5_state_t *md5StateStream,
                                      md5_state_t *md5StateClient) const
{
  if (clientStore_ -> saveRequestStores(cachefs, md5StateStream, md5StateClient,
                                            use_checksum, discard_data) < 0)
  {
    return -1;
  }
  else if (serverStore_ -> saveReplyStores(cachefs, md5StateStream, md5StateClient,
                                               discard_checksum, use_data) < 0)
  {
    return -1;
  }
  else if (serverStore_ -> saveEventStores(cachefs, md5StateStream, md5StateClient,
                                               discard_checksum, use_data) < 0)
  {
    return -1;
  }

  return 1;
}

int ClientProxy::handleLoadStores(istream *cachefs, md5_state_t *md5StateStream) const
{
  if (clientStore_ -> loadRequestStores(cachefs, md5StateStream,
                                            use_checksum, discard_data) < 0)
  {
    return -1;
  }
  else if (serverStore_ -> loadReplyStores(cachefs, md5StateStream,
                                               discard_checksum, use_data) < 0)
  {
    return -1;
  }
  else if (serverStore_ -> loadEventStores(cachefs, md5StateStream,
                                               discard_checksum, use_data) < 0)
  {
    return -1;
  }

  return 1;
}

