/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include "Misc.h"
#include "Control.h"

#include "Compressor.h"
#include "EncodeBuffer.h"

#define PANIC
#undef  TEST
#undef  DEBUG

Compressor::Compressor(int compressionLevel, int compressionThreshold)
{
  buffer_     = NULL;
  bufferSize_ = 0;

  stream_.zalloc = (alloc_func) 0;
  stream_.zfree  = (free_func) 0;
  stream_.opaque = (voidpf) 0;

  #ifdef TEST
  *logofs << "Compressor: Compression level is "
          << compressionLevel << ".\n" << logofs_flush;
  #endif

  int result = deflateInit2(&stream_, compressionLevel, Z_DEFLATED,
                                15, 9, Z_DEFAULT_STRATEGY);

  if (result != Z_OK)
  {
    #ifdef PANIC
    *logofs << "Compressor: PANIC! Cannot initialize data compression "
            << "library. Error is '" << zError(result) << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot initialize data compression "
         << "library. Error is '" << zError(result) << "'.\n";
  }

  #ifdef TEST
  *logofs << "Compressor: Compression threshold is "
          << compressionThreshold << ".\n" << logofs_flush;
  #endif

  threshold_ = compressionThreshold;
}

Compressor::~Compressor()
{
  if (buffer_)
  {
    delete [] buffer_;

    buffer_     = 0;
    bufferSize_ = 0;
  }

  int result = deflateEnd(&stream_);

  if (result != Z_OK)
  {
    #ifdef PANIC
    *logofs << "Compressor: PANIC! Cannot deinitialize data compression "
            << " library. Error is '" << zError(result) << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot deinitialize data compression "
         << "library. Error is '" << zError(result) << "'.\n";
  }
}

//
// This function compresses and encodes the compressed buffer.
// It returns a pointer to the internal buffer where data was
// compressed.
//

int Compressor::compressBuffer(const unsigned char *uncompressedBuffer, 
                                   const unsigned int uncompressedBufferSize,
                                       unsigned char *&compressedBuffer,
                                           unsigned int &compressedBufferSize,
                                               EncodeBuffer &encodeBuffer)
{
  if (control -> LocalDataCompression == 0 ||
          compressBuffer(uncompressedBuffer, uncompressedBufferSize, 
                             compressedBuffer, compressedBufferSize) <= 0)
  {
    encodeBuffer.encodeBool(0);

    encodeBuffer.encodeMemory(uncompressedBuffer, uncompressedBufferSize);

    return 0;
  }
  else
  {
    encodeBuffer.encodeBool(1);

    if (control -> isProtoStep2() == 1)
    {
      encodeBuffer.encodeValue(compressedBufferSize, 32, 14);
      encodeBuffer.encodeValue(uncompressedBufferSize, 32, 14);
    }
    else
    {
      encodeBuffer.encodeValue(compressedBufferSize, 32);
      encodeBuffer.encodeValue(uncompressedBufferSize, 32);
    }

    encodeBuffer.encodeMemory(compressedBuffer, compressedBufferSize);

    return 1;
  }
}

//
// This function compresses data into a dynamically
// allocated buffer and returns a pointer to it, so
// application must copy data before the next call.
//

int Compressor::compressBuffer(const unsigned char *uncompressedBuffer, 
                                   const unsigned int uncompressedBufferSize,
                                       unsigned char *&compressedBuffer,
                                           unsigned int &compressedBufferSize)
{
  #ifdef DEBUG
  *logofs << "Compressor: Called for buffer at "
          << (void *) uncompressedBuffer << ".\n"
          << logofs_flush;
  #endif

  compressedBufferSize = uncompressedBufferSize;

  if (uncompressedBufferSize < (unsigned int) threshold_)
  {
    #ifdef TEST
    *logofs << "Compressor: Leaving buffer unchanged. Plain size is "
            << uncompressedBufferSize << " while threshold is "
            << (unsigned int) threshold_ << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  //
  // Determine size of temporary buffer. 
  //

  unsigned int newSize = uncompressedBufferSize +
                             (uncompressedBufferSize / 1000) + 12;

  if (buffer_ != NULL &&
          (newSize > bufferSize_ ||
              bufferSize_ >= newSize * 2))
  {
    delete [] buffer_;

    buffer_ = NULL;
    bufferSize_ = 0;
  }

  if (buffer_ == NULL)
  {
    buffer_ = new unsigned char[newSize];

    if (buffer_ == NULL)
    {
      #ifdef PANIC
      *logofs << "Compressor: PANIC! Can't allocate compression buffer of "
              << newSize << " bytes. Error is " << EGET() << " '"
              << ESTR() << "'.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Can't allocate compression buffer of "
           << newSize << " bytes. Error is " << EGET()
           << " '" << ESTR() << "'.\n";

      return 0;
    }

    bufferSize_ = newSize;
  }

  uLongf compressedSize = newSize; 

  int result = compress(buffer_, &compressedSize, uncompressedBuffer,
                            uncompressedBufferSize);

  if (result == Z_OK)
  {
    if (compressedSize > newSize)
    {
      #ifdef PANIC
      *logofs << "Compressor: PANIC! Overflow in compress buffer size. "
              << "Expected size was " << newSize << " while it is "
              << compressedSize << ".\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Overflow in compress buffer size. "
           << "Expected size was " << newSize << " while it is "
           << compressedSize << ".\n";

      return -1;
    }
    else if (compressedSize >= uncompressedBufferSize)
    {
      #ifdef TEST
      *logofs << "Compressor: Leaving buffer unchanged. Plain size is "
              << uncompressedBufferSize << " compressed size is "
              << compressedSize << ".\n" << logofs_flush;
      #endif

      return 0;
    }

    compressedBuffer     = buffer_;
    compressedBufferSize = compressedSize;

    #ifdef TEST
    *logofs << "Compressor: Compressed buffer from "
            << uncompressedBufferSize << " to "
            << compressedSize << " bytes.\n"
            << logofs_flush;
    #endif

    return 1;
  }

  #ifdef PANIC
  *logofs << "Compressor: PANIC! Failed compression of buffer. "
          << "Error is '" << zError(result) << "'.\n"
          << logofs_flush;
  #endif

  cerr << "Error" << ": Failed compression of buffer. "
       << "Error is '" << zError(result) << "'.\n";

  return -1;
}

int Compressor::compress(Bytef *dest, uLongf *destLen, const Bytef *source,
                             uLong sourceLen)
{
  unsigned int old_total_out = stream_.total_out;

  stream_.next_in  = (Bytef *) source;
  stream_.avail_in = (uInt) sourceLen;

  #ifdef MAXSEG_64K

  //
  // Check for source > 64K on 16-bit machine.
  //

  if ((uLong)stream_.avail_in != sourceLen) return Z_BUF_ERROR;

  #endif

  stream_.next_out = dest;
  stream_.avail_out = (uInt)*destLen;

  if ((uLong) stream_.avail_out != *destLen) return Z_BUF_ERROR;

  int result = deflate(&stream_, Z_FINISH);

  if (result != Z_STREAM_END)
  {
    deflateReset(&stream_);

    return (result == Z_OK ? Z_BUF_ERROR : result);
  }

  *destLen = stream_.total_out - old_total_out;

  result = deflateReset(&stream_);

  return result;
}
