/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include "NXpack.h"

#include "Control.h"

#define PANIC
#define WARNING
#undef  TEST

//
// Maximum number of bytes sent for each token.
//

#define TOKEN_BYTES                              16384

//
// Maximum number of tokens that can be spent
// by the client proxy before having to block
// waiting for a reply.
//

#define TOKEN_LIMIT                              8

//
// Kill session if control parameters cannot be
// negotiated before this timeout.
//

#define INIT_TIMEOUT                             60000

//
// How many milliseconds to wait before to restart
// handling of pending messages after a timeout.
//

#define PENDING_TIMEOUT                          0

//
// Write data to the remote proxy if a new frame is
// not produced within this amount of milliseconds.
//

#define FRAME_TIMEOUT                            0

//
// Flush data to the remote proxy this amount of
// milliseconds is elapsed since the last write.
//

#define FLUSH_TIMEOUT                            0

//
// Only send one motion event any N milliseconds.
//

#define MOTION_TIMEOUT                           0

//
// Send a ping to remote proxy if no communication
// takes place within this amount of milliseconds.
//

#define PING_TIMEOUT                             5000

//
// At connection reset, server channel needs to
// sinchronize its sequence counter with sequence
// of next event or reply coming from the X server.
// To do that, it will be able to wait for this
// amount of milliseconds.
//

#define RESET_TIMEOUT                            5000

//
// Close X connection if can't write before this
// timeout.
//

#define CHANNEL_TIMEOUT                          10000

//
// Warn user (or close proxy connection) if don't
// receive any data before this timeout.
//

#define PROXY_TIMEOUT                            120000

//
// Timeout used to wait for a complete message
// to become available.
//

#define RETRY_TIMEOUT                            5

//
// How many milliseconds to wait for the shared
// memory completion event to become available.
//

#define SHMEM_TIMEOUT                            0

//
// Time before we consider concluded the statup
// of X applications making the user's session.
//

#define STARTUP_TIMEOUT                          10000

//
// Before closing down the proxy, wait for the
// given amount of miliseconds to let all the
// running applications to cleanly close down
// their connections.
//

#define CLEANUP_TIMEOUT                          3000

//
// Slow down the client proxy if X server can't
// keep up handling X messages.
//

#define CONGESTION_TIMEOUT                       0

//
// Wait this amount of milliseconds before any
// iteration of the house-keeping process.
//

#define KEEPER_TIMEOUT                           60000

//
// In case of timeout, select can return control
// to program earlier or later of this amount of
// ms. Consider this when calculating if timeout
// is elapsed.
//

#define LATENCY_TIMEOUT                          1

//
// Control memory allocation in transport
// and other classes.
//

#define TRANSPORT_X_BUFFER_SIZE                  32768
#define TRANSPORT_PROXY_BUFFER_SIZE              32768
#define TRANSPORT_GENERIC_BUFFER_SIZE            65536

#define TRANSPORT_X_BUFFER_LIMIT                 524288
#define TRANSPORT_PROXY_BUFFER_LIMIT             327680
#define TRANSPORT_GENERIC_BUFFER_LIMIT           131072

//
// Stop doubling the size and print a warning
// if a buffer reaches this limit.
//

#define TRANSPORT_MAXIMUM_BUFFER_SIZE            524288

//
// Never allow X buffers to exceed this limit.
//

#define TRANSPORT_FORCE_BUFFER_SIZE              65536

//
// Stop reading from proxy or X connections
// if the data stored in buffer exceed these
// limits.
//

#define TRANSPORT_X_BUFFER_THRESHOLD             327680
#define TRANSPORT_PROXY_BUFFER_THRESHOLD         32768
#define TRANSPORT_GENERIC_BUFFER_THRESHOLD       65536

//
// Immediately flush data to the X socket
// if buffer exceeds this limit.
//

#define TRANSPORT_WRITE_THRESHOLD                4096

//
// Write only this amount of bytes on sockets
// at each write. Write the rest when socket
// is reported to be writable.
//

#define TRANSPORT_WRITE_LIMIT                    1048576

//
// Defaults used for socket options.
//

#define OPTION_PROXY_KEEP_ALIVE                  0
#define OPTION_PROXY_LOW_DELAY                   1
#define OPTION_PROXY_CLIENT_NO_DELAY             1
#define OPTION_PROXY_SERVER_NO_DELAY             1
#define OPTION_CLIENT_NO_DELAY                   1
#define OPTION_SERVER_NO_DELAY                   1

#define OPTION_PROXY_RECEIVE_BUFFER              -1
#define OPTION_CLIENT_RECEIVE_BUFFER             -1
#define OPTION_SERVER_RECEIVE_BUFFER             -1

#define OPTION_PROXY_SEND_BUFFER                 -1
#define OPTION_CLIENT_SEND_BUFFER                -1
#define OPTION_SERVER_SEND_BUFFER                -1

#define OPTION_PROXY_RETRY_CONNECT               30
#define OPTION_PROXY_RETRY_ACCEPT                3
#define OPTION_SERVER_RETRY_CONNECT              3

//
// Defaults used for cache persistence.
//

#define PERSISTENT_CACHE_THRESHOLD               102400

#define PERSISTENT_CACHE_ENABLE_LOAD             1
#define PERSISTENT_CACHE_ENABLE_SAVE             1

#define PERSISTENT_CACHE_LOAD_PACKED             1
#define PERSISTENT_CACHE_LOAD_RENDER             1

#define PERSISTENT_CACHE_DISK_LIMIT              33554432

//
// Defaults used for image cache.
//

#define IMAGE_CACHE_ENABLE_LOAD                  1
#define IMAGE_CACHE_ENABLE_SAVE                  1

#define IMAGE_CACHE_DISK_LIMIT                   33554432

//
// Suggested defaults for read length parameters
// used by read buffer classes.
//

#define CLIENT_INITIAL_READ_SIZE                 4096
#define CLIENT_MAXIMUM_READ_SIZE                 32768

#define SERVER_INITIAL_READ_SIZE                 1024
#define SERVER_MAXIMUM_READ_SIZE                 16384

#define PROXY_INITIAL_READ_SIZE                  8192
#define PROXY_MAXIMUM_READ_SIZE                  16384

#define GENERIC_INITIAL_READ_SIZE                4096
#define GENERIC_MAXIMUM_READ_SIZE                8192

//
// Calculate bitrate in given time frames.
// Values are in milliseconds.
//

#define SHORT_BITRATE_TIME_FRAME                 5000
#define LONG_BITRATE_TIME_FRAME                  30000

//
// Bandwidth control. A value of 0 means no
// limit. Values are stored internally in
// bytes per second.
//

#define CLIENT_BITRATE_LIMIT                     0
#define SERVER_BITRATE_LIMIT                     0

//
// CPU limit. Value is ratio between time spent
// in select and time spent handling messages.
// A value of 0 means no limit. By default set
// a limit only at X client side.
//

#define CLIENT_PROCESSOR_LIMIT                   0
#define SERVER_PROCESSOR_LIMIT                   0

//
// Default values for cache control.
//

#define MINIMUM_MESSAGE_SIZE_THRESHOLD           4
#define MAXIMUM_MESSAGE_SIZE_THRESHOLD           262144

#define CLIENT_TOTAL_STORAGE_SIZE_LIMIT          8388608
#define SERVER_TOTAL_STORAGE_SIZE_LIMIT          8388608

#define STORE_TIME_LIMIT                         3600

#define STORE_HITS_LOAD_BONUS                    10
#define STORE_HITS_ADD_BONUS                     20
#define STORE_HITS_LIMIT                         100

#define STORE_HITS_TOUCH                         1
#define STORE_HITS_UNTOUCH                       2

//
// Default parameters for message splitting.
//

#define SPLIT_MODE                               1
#define SPLIT_TIMEOUT                            100
#define SPLIT_DATA_THRESHOLD                     65536
#define SPLIT_DATA_PACKET_LIMIT                  24576
#define SPLIT_TOTAL_STORAGE_SIZE_LIMIT           1048576

//
// Agent and control-flow related parameters.
//

#define AGENT_KARMA_SIZE                         98304
#define AGENT_PACK_METHOD                        35
#define AGENT_PACK_QUALITY                       9
#define AGENT_FLUSH_QUERY                        0
#define AGENT_FLUSH_PRIORITY                     0
#define AGENT_HIDE_RENDER                        0
#define AGENT_TAINT_REPLIES                      1
#define AGENT_TAINT_THRESHOLD                    8

//
// In current version only X server support is
// implemented. Note that use of shared memory
// is negotiated according to options provided
// by the user.
//

#define AGENT_SHMEM_CLIENT                       0
#define AGENT_SHMEM_SERVER                       1

//
// Default size of shared memory segments used
// in MIT-SHM support.
//

#define AGENT_SHMEM_CLIENT_SIZE                  0
#define AGENT_SHMEM_SERVER_SIZE                  2097152

//
// These parameters are propagated by agent
// to the Xlib layer.
//

#define AGENT_CLEAN_GET                          0
#define AGENT_CLEAN_ALLOC                        0
#define AGENT_CLEAN_FLUSH                        0
#define AGENT_CLEAN_SEND                         0
#define AGENT_CLEAN_IMAGES                       1

#define AGENT_IMAGE_SPLIT                        0
#define AGENT_IMAGE_MASK                         0
#define AGENT_IMAGE_FRAME                        0
#define AGENT_IMAGE_SPLIT_METHOD                 0
#define AGENT_IMAGE_MASK_METHOD                  0

//
// Reconnect proxies in case of network failures
// or proxy errors.
//

#define ENABLE_RESTART_ON_FAILURE                0

//
// What do we do at the end of session? If this
// flag is set, we launch a new client letting
// the user run a new NX session.
//

#define ENABLE_RESTART_ON_SHUTDOWN               0

//
// Do we produce a core dump on fatal errors?
//

#define ENABLE_CORE_DUMP_ON_ABORT                0

//
// How do we handle SIGHUP signals? Select one
// of the following options.
//

#define ENABLE_SHUTDOWN_ON_SIGHUP                1
#define ENABLE_RECONFIG_ON_SIGHUP                0
#define ENABLE_RESTART_ON_SIGHUP                 0

//
// Reopen the log file if it exceeds this size.
//

#define LOG_FILE_SIZE_LIMIT                      60000000

//
// Check periodically if we need to truncate the
// log file. By default check every minute.
//

#define FILE_SIZE_CHECK_TIMEOUT                  60000

//
// This file contains stock parameters for NX control class.
// Settings for either 'modem', 'isdn', 'adsl' or 'local'
// are read at startup according to link speed provided by
// the user.
//
// You can override these settings by creating your control
// files in the form control-[type] in user's .nx directory.
// You can reconfigure the link on the fly by sending a -HUP
// signal to the proxy process. This is strictly intended
// to simplify tuning. Test different values at client and
// server side and, if needed, later split the parameter in
// 'Client' and 'Server' counterparts.
//

//
// MODEM 28.8/33.6/56Kbps
//

const char ControlModem[] =
"\
//                                          \n\
// Please, do not edit. This buffer         \n\
// contains sensitive NX parameters.        \n\
//                                          \n\
//                                          \n\
// Scheduling parameters.                   \n\
//                                          \n\
[7]   2048    [TokenBytes]                  \n\
[8]   8       [TokenLimit]                  \n\
//                                          \n\
// Split options.                           \n\
//                                          \n\
[12]  1       [SplitMode]                   \n\
[13]  100     [SplitTimeout]                \n\
[14]  1024    [SplitDataThreshold]          \n\
[15]  768     [SplitDataPacketLimit]        \n\
[16]  1048576 [SplitTotalStorageSizeLimit]  \n\
//                                          \n\
// Timeout parameters.                      \n\
//                                          \n\
[20]  50      [FrameTimeout]                \n\
[21]  200     [FlushTimeout]                \n\
[22]  50      [MotionTimeout]               \n\
[23]  0       [PendingTimeout]              \n\
[24]  0       [ShmemTimeout]                \n\
[25]  5000    [PingTimeout]                 \n\
[26]  60000   [StartupTimeout]              \n\
//                                          \n\
// Transport parameters.                    \n\
//                                          \n\
[30]  4096    [TransportWriteThreshold]     \n\
[31]  327680  [TransportXBufferThreshold]   \n\
[32]  32768   [TransportProxyBufferThre...] \n\
[33]  65536   [TransportGenericBufferTh...] \n\
//                                          \n\
// Read buffers parameters.                 \n\
//                                          \n\
[37]  4096    [ClientInitialReadSize]       \n\
[38]  327680  [ClientMaximumReadSize]       \n\
[39]  1024    [ServerInitialReadSize]       \n\
[40]  327680  [ServerMaximumReadSize]       \n\
[41]  1024    [ProxyInitialReadSize]        \n\
[42]  327680  [ProxyMaximumReadSize]        \n\
[43]  512     [GenericInitialReadSize]      \n\
[44]  1024    [GenericMaximumReadSize]      \n\
//                                          \n\
// Socket options.                          \n\
//                                          \n\
[48] -1       [OptionProxySendBuffer]       \n\
[49] -1       [OptionProxyReceiveBuffer]    \n\
[50]  1       [OptionProxyClientNoDelay]    \n\
[51]  1       [OptionProxyServerNoDelay]    \n\
//                                          \n\
// Agent control parameters.                \n\
//                                          \n\
[55]  12288   [AgentKarmaSize]              \n\
[56]  35      [AgentPackMethod]             \n\
[57]  3       [AgentPackQuality]            \n\
[58]  0       [AgentFlushQueued]            \n\
[59]  0       [AgentFlushPriority]          \n\
[60]  1       [AgentTaintReplies]           \n\
[61]  8       [AgentTaintThreshold]         \n\
//                                          \n\
// Cleanup of buffers.                      \n\
//                                          \n\
[65]  0       [AgentCleanGet]               \n\
[66]  0       [AgentCleanAlloc]             \n\
[67]  0       [AgentCleanFlush]             \n\
[68]  0       [AgentCleanSend]              \n\
[69]  1       [AgentCleanImages]            \n\
//                                          \n\
// Image parameters.                        \n\
//                                          \n\
[73]  0       [AgentImageSplit]             \n\
[74]  0       [AgentImageMask]              \n\
[75]  0       [AgentImageFrame]             \n\
[76]  0       [AgentImageSplitMethod]       \n\
[77]  0       [AgentImageMaskMethod]        \n\
";

//
// ISDN 64/128Kbps.
//

const char ControlIsdn[] =
"\
//                                          \n\
// Please, do not edit. This buffer         \n\
// contains sensitive NX parameters.        \n\
//                                          \n\
//                                          \n\
// Scheduling parameters.                   \n\
//                                          \n\
[7]   4096    [TokenBytes]                  \n\
[8]   8       [TokenLimit]                  \n\
//                                          \n\
// Split options.                           \n\
//                                          \n\
[12]  1       [SplitMode]                   \n\
[13]  100     [SplitTimeout]                \n\
[14]  2048    [SplitDataThreshold]          \n\
[15]  1536    [SplitDataPacketLimit]        \n\
[16]  1048576 [SplitTotalStorageSizeLimit]  \n\
//                                          \n\
// Timeout parameters.                      \n\
//                                          \n\
[20]  20      [FrameTimeout]                \n\
[21]  100     [FlushTimeout]                \n\
[22]  20      [MotionTimeout]               \n\
[23]  0       [PendingTimeout]              \n\
[24]  0       [ShmemTimeout]                \n\
[25]  5000    [PingTimeout]                 \n\
[26]  45000   [StartupTimeout]              \n\
//                                          \n\
// Transport parameters.                    \n\
//                                          \n\
[30]  4096    [TransportWriteThreshold]     \n\
[31]  327680  [TransportXBufferThreshold]   \n\
[32]  32768   [TransportProxyBufferThre...] \n\
[33]  65536   [TransportGenericBufferTh...] \n\
//                                          \n\
// Read buffers parameters.                 \n\
//                                          \n\
[37]  4096    [ClientInitialReadSize]       \n\
[38]  327680  [ClientMaximumReadSize]       \n\
[39]  1024    [ServerInitialReadSize]       \n\
[40]  327680  [ServerMaximumReadSize]       \n\
[41]  2048    [ProxyInitialReadSize]        \n\
[42]  327680  [ProxyMaximumReadSize]        \n\
[43]  1024    [GenericInitialReadSize]      \n\
[44]  2048    [GenericMaximumReadSize]      \n\
//                                          \n\
// Socket options.                          \n\
//                                          \n\
[48] -1       [OptionProxySendBuffer]       \n\
[49] -1       [OptionProxyReceiveBuffer]    \n\
[50]  1       [OptionProxyClientNoDelay]    \n\
[51]  1       [OptionProxyServerNoDelay]    \n\
//                                          \n\
// Agent control parameters.                \n\
//                                          \n\
[55]  24576   [AgentKarmaSize]              \n\
[56]  35      [AgentPackMethod]             \n\
[57]  5       [AgentPackQuality]            \n\
[58]  0       [AgentFlushQueued]            \n\
[59]  0       [AgentFlushPriority]          \n\
[60]  1       [AgentTaintReplies]           \n\
[61]  8       [AgentTaintThreshold]         \n\
//                                          \n\
// Cleanup of buffers.                      \n\
//                                          \n\
[65]  0       [AgentCleanGet]               \n\
[66]  0       [AgentCleanAlloc]             \n\
[67]  0       [AgentCleanFlush]             \n\
[68]  0       [AgentCleanSend]              \n\
[69]  1       [AgentCleanImages]            \n\
//                                          \n\
// Image parameters.                        \n\
//                                          \n\
[73]  0       [AgentImageSplit]             \n\
[74]  0       [AgentImageMask]              \n\
[75]  0       [AgentImageFrame]             \n\
[76]  0       [AgentImageSplitMethod]       \n\
[77]  0       [AgentImageMaskMethod]        \n\
";

//
// ADSL 256/640Kbps or DSL 1/2/34Mbps.
//

const char ControlAdsl[] =
"\
//                                          \n\
// Please, do not edit. This buffer         \n\
// contains sensitive NX parameters.        \n\
//                                          \n\
//                                          \n\
// Scheduling parameters.                   \n\
//                                          \n\
[7]   8192    [TokenBytes]                  \n\
[8]   8       [TokenLimit]                  \n\
//                                          \n\
// Split options.                           \n\
//                                          \n\
[12]  1       [SplitMode]                   \n\
[13]  100     [SplitTimeout]                \n\
[14]  16384   [SplitDataThreshold]          \n\
[15]  6144    [SplitDataPacketLimit]        \n\
[16]  1048576 [SplitTotalStorageSizeLimit]  \n\
//                                          \n\
// Timeout parameters.                      \n\
//                                          \n\
[20]  10      [FrameTimeout]                \n\
[21]  50      [FlushTimeout]                \n\
[22]  10      [MotionTimeout]               \n\
[23]  0       [PendingTimeout]              \n\
[24]  0       [ShmemTimeout]                \n\
[25]  5000    [PingTimeout]                 \n\
[26]  30000   [StartupTimeout]              \n\
//                                          \n\
// Transport parameters.                    \n\
//                                          \n\
[30]  4096    [TransportWriteThreshold]     \n\
[31]  327680  [TransportXBufferThreshold]   \n\
[32]  32768   [TransportProxyBufferThre...] \n\
[33]  65536   [TransportGenericBufferTh...] \n\
//                                          \n\
// Read buffers parameters.                 \n\
//                                          \n\
[37]  4096    [ClientInitialReadSize]       \n\
[38]  327680  [ClientMaximumReadSize]       \n\
[39]  1024    [ServerInitialReadSize]       \n\
[40]  327680  [ServerMaximumReadSize]       \n\
[41]  4096    [ProxyInitialReadSize]        \n\
[42]  327680  [ProxyMaximumReadSize]        \n\
[43]  2048    [GenericInitialReadSize]      \n\
[44]  4096    [GenericMaximumReadSize]      \n\
//                                          \n\
// Socket options.                          \n\
//                                          \n\
[48] -1       [OptionProxySendBuffer]       \n\
[49] -1       [OptionProxyReceiveBuffer]    \n\
[50]  1       [OptionProxyClientNoDelay]    \n\
[51]  1       [OptionProxyServerNoDelay]    \n\
//                                          \n\
// Agent control parameters.                \n\
//                                          \n\
[55]  49152   [AgentKarmaSize]              \n\
[56]  35      [AgentPackMethod]             \n\
[57]  7       [AgentPackQuality]            \n\
[58]  0       [AgentFlushQueued]            \n\
[59]  0       [AgentFlushPriority]          \n\
[60]  1       [AgentTaintReplies]           \n\
[61]  8       [AgentTaintThreshold]         \n\
//                                          \n\
// Cleanup of buffers.                      \n\
//                                          \n\
[65]  0       [AgentCleanGet]               \n\
[66]  0       [AgentCleanAlloc]             \n\
[67]  0       [AgentCleanFlush]             \n\
[68]  0       [AgentCleanSend]              \n\
[69]  1       [AgentCleanImages]            \n\
//                                          \n\
// Image parameters.                        \n\
//                                          \n\
[73]  0       [AgentImageSplit]             \n\
[74]  0       [AgentImageMask]              \n\
[75]  0       [AgentImageFrame]             \n\
[76]  0       [AgentImageSplitMethod]       \n\
[77]  0       [AgentImageMaskMethod]        \n\
";

//
// LOCAL configuration. Includes WAN 1/2Mbps
// and LAN 10/100Mbps.
//

const char ControlLocal[] =
"\
//                                          \n\
// Please, do not edit. This buffer         \n\
// contains sensitive NX parameters.        \n\
//                                          \n\
//                                          \n\
// Scheduling parameters.                   \n\
//                                          \n\
[7]   16384   [TokenBytes]                  \n\
[8]   8       [TokenLimit]                  \n\
//                                          \n\
// Split options.                           \n\
//                                          \n\
[12]  1       [SplitMode]                   \n\
[13]  100     [SplitTimeout]                \n\
[14]  65536   [SplitDataThreshold]          \n\
[15]  24576   [SplitDataPacketLimit]        \n\
[16]  1048576 [SplitTotalStorageSizeLimit]  \n\
//                                          \n\
// Timeout parameters.                      \n\
//                                          \n\
[20]  5       [FrameTimeout]                \n\
[21]  20      [FlushTimeout]                \n\
[22]  5       [MotionTimeout]               \n\
[23]  0       [PendingTimeout]              \n\
[24]  0       [ShmemTimeout]                \n\
[25]  5000    [PingTimeout]                 \n\
[26]  15000   [StartupTimeout]              \n\
//                                          \n\
// Transport parameters.                    \n\
//                                          \n\
[30]  4096    [TransportWriteThreshold]     \n\
[31]  327680  [TransportXBufferThreshold]   \n\
[32]  32768   [TransportProxyBufferThre...] \n\
[33]  65536   [TransportGenericBufferTh...] \n\
//                                          \n\
// Read buffers parameters.                 \n\
//                                          \n\
[37]  4096    [ClientInitialReadSize]       \n\
[38]  327680  [ClientMaximumReadSize]       \n\
[39]  1024    [ServerInitialReadSize]       \n\
[40]  327680  [ServerMaximumReadSize]       \n\
[41]  8192    [ProxyInitialReadSize]        \n\
[42]  327680  [ProxyMaximumReadSize]        \n\
[43]  4096    [GenericInitialReadSize]      \n\
[44]  8192    [GenericMaximumReadSize]      \n\
//                                          \n\
// Socket options.                          \n\
//                                          \n\
[48] -1       [OptionProxySendBuffer]       \n\
[49] -1       [OptionProxyReceiveBuffer]    \n\
[50]  1       [OptionProxyClientNoDelay]    \n\
[51]  1       [OptionProxyServerNoDelay]    \n\
//                                          \n\
// Agent control parameters.                \n\
//                                          \n\
[55]  98304   [AgentKarmaSize]              \n\
[56]  35      [AgentPackMethod]             \n\
[57]  9       [AgentPackQuality]            \n\
[58]  0       [AgentFlushQueued]            \n\
[59]  0       [AgentFlushPriority]          \n\
[60]  1       [AgentTaintReplies]           \n\
[61]  8       [AgentTaintThreshold]         \n\
//                                          \n\
// Cleanup of buffers.                      \n\
//                                          \n\
[65]  0       [AgentCleanGet]               \n\
[66]  0       [AgentCleanAlloc]             \n\
[67]  0       [AgentCleanFlush]             \n\
[68]  0       [AgentCleanSend]              \n\
[69]  1       [AgentCleanImages]            \n\
//                                          \n\
// Image parameters.                        \n\
//                                          \n\
[73]  0       [AgentImageSplit]             \n\
[74]  0       [AgentImageMask]              \n\
[75]  0       [AgentImageFrame]             \n\
[76]  0       [AgentImageSplitMethod]       \n\
[77]  0       [AgentImageMaskMethod]        \n\
";

//
// Set defaults for control. They should be what
// you get in case of 'local' connection.
//

Control::Control()
{
  ProxyMode   = proxy_undefined;
  ProxyStage  = stage_undefined;
  SessionMode = session_undefined;
  FlushPolicy = policy_undefined;
  LinkMode    = link_undefined;

  TokenBytes = TOKEN_BYTES;
  TokenLimit = TOKEN_LIMIT;

  InitTimeout    = INIT_TIMEOUT;
  FrameTimeout   = FRAME_TIMEOUT;
  FlushTimeout   = FLUSH_TIMEOUT;
  MotionTimeout  = MOTION_TIMEOUT;
  PendingTimeout = PENDING_TIMEOUT;
  PingTimeout    = PING_TIMEOUT;

  ResetTimeout   = RESET_TIMEOUT;
  ChannelTimeout = CHANNEL_TIMEOUT;
  ProxyTimeout   = PROXY_TIMEOUT;
  RetryTimeout   = RETRY_TIMEOUT;
  ShmemTimeout   = SHMEM_TIMEOUT;

  StartupTimeout  = STARTUP_TIMEOUT;
  CleanupTimeout  = CLEANUP_TIMEOUT;
  KeeperTimeout   = KEEPER_TIMEOUT;
  LatencyTimeout  = LATENCY_TIMEOUT;

  LogFileSizeLimit     = LOG_FILE_SIZE_LIMIT;
  FileSizeCheckTimeout = FILE_SIZE_CHECK_TIMEOUT;

  EnableRestartOnFailure  = ENABLE_RESTART_ON_FAILURE;
  EnableRestartOnShutdown = ENABLE_RESTART_ON_SHUTDOWN;

  EnableCoreDumpOnAbort  = ENABLE_CORE_DUMP_ON_ABORT;

  EnableShutdownOnSighup = ENABLE_SHUTDOWN_ON_SIGHUP;
  EnableReconfigOnSighup = ENABLE_RECONFIG_ON_SIGHUP;
  EnableRestartOnSighup  = ENABLE_RESTART_ON_SIGHUP;

  //
  // Memory restrictions if any.
  //

  LocalMemoryLevel = -1;

  //
  // Compression must be negotiated between proxies.
  //

  LocalDeltaCompression   = -1;
  RemoteDeltaCompression  = -1;

  LocalDataCompression    = -1;
  LocalStreamCompression  = -1;

  RemoteDataCompression   = -1;
  RemoteStreamCompression = -1;

  LocalDataCompressionLevel     = -1;
  LocalDataCompressionThreshold = -1;
  LocalStreamCompressionLevel   = -1;

  RemoteDataCompressionLevel    = -1;
  RemoteStreamCompressionLevel  = -1;

  //
  // Transport buffers' allocation parameters.
  // 

  TransportXBufferSize            = TRANSPORT_X_BUFFER_SIZE;
  TransportProxyBufferSize        = TRANSPORT_PROXY_BUFFER_SIZE;
  TransportGenericBufferSize      = TRANSPORT_GENERIC_BUFFER_SIZE;

  TransportXBufferLimit           = TRANSPORT_X_BUFFER_LIMIT;
  TransportProxyBufferLimit       = TRANSPORT_PROXY_BUFFER_LIMIT;
  TransportGenericBufferLimit     = TRANSPORT_GENERIC_BUFFER_LIMIT;

  TransportMaximumBufferSize      = TRANSPORT_MAXIMUM_BUFFER_SIZE;
  TransportForceBufferSize        = TRANSPORT_FORCE_BUFFER_SIZE;

  TransportXBufferThreshold       = TRANSPORT_X_BUFFER_THRESHOLD;
  TransportProxyBufferThreshold   = TRANSPORT_PROXY_BUFFER_THRESHOLD;
  TransportGenericBufferThreshold = TRANSPORT_GENERIC_BUFFER_THRESHOLD;

  //
  // Set write policies to X connections.
  //

  TransportWriteLimit     = TRANSPORT_WRITE_LIMIT;
  TransportWriteThreshold = TRANSPORT_WRITE_THRESHOLD;

  //
  // Don't collect statistics by default.
  //

  CollectStatistics = 0;

  //
  // Socket options.
  //

  OptionProxyKeepAlive      = OPTION_PROXY_KEEP_ALIVE;
  OptionProxyLowDelay       = OPTION_PROXY_LOW_DELAY;
  OptionProxyClientNoDelay  = OPTION_PROXY_CLIENT_NO_DELAY;
  OptionProxyServerNoDelay  = OPTION_PROXY_SERVER_NO_DELAY;
  OptionClientNoDelay       = OPTION_CLIENT_NO_DELAY;
  OptionServerNoDelay       = OPTION_SERVER_NO_DELAY;

  OptionProxyReceiveBuffer  = OPTION_PROXY_RECEIVE_BUFFER;
  OptionClientReceiveBuffer = OPTION_CLIENT_RECEIVE_BUFFER;
  OptionServerReceiveBuffer = OPTION_SERVER_RECEIVE_BUFFER;

  OptionProxySendBuffer     = OPTION_PROXY_SEND_BUFFER;
  OptionClientSendBuffer    = OPTION_CLIENT_SEND_BUFFER;
  OptionServerSendBuffer    = OPTION_SERVER_SEND_BUFFER;

  OptionProxyRetryAccept    = OPTION_PROXY_RETRY_ACCEPT;
  OptionProxyRetryConnect   = OPTION_PROXY_RETRY_CONNECT;
  OptionServerRetryConnect  = OPTION_SERVER_RETRY_CONNECT;

  //
  // Base NX directory.
  //

  RootPath = NULL;

  //
  // Set defaults for handling persistent cache.
  //

  PersistentCachePath = NULL;
  PersistentCacheName = NULL;

  PersistentCacheThreshold = PERSISTENT_CACHE_THRESHOLD;

  PersistentCacheEnableLoad = PERSISTENT_CACHE_ENABLE_LOAD;
  PersistentCacheEnableSave = PERSISTENT_CACHE_ENABLE_SAVE;

  PersistentCacheLoadPacked = PERSISTENT_CACHE_LOAD_PACKED;
  PersistentCacheLoadRender = PERSISTENT_CACHE_LOAD_RENDER;

  PersistentCacheDiskLimit = PERSISTENT_CACHE_DISK_LIMIT;

  //
  // Set defaults for image cache.
  //

  ImageCachePath = NULL;

  ImageCacheEnableLoad = IMAGE_CACHE_ENABLE_LOAD;
  ImageCacheEnableSave = IMAGE_CACHE_ENABLE_SAVE;

  ImageCacheDiskLimit = IMAGE_CACHE_DISK_LIMIT;

  //
  // Size of read operations in read buffer classes.
  //

  ClientInitialReadSize  = CLIENT_INITIAL_READ_SIZE;
  ClientMaximumReadSize  = CLIENT_MAXIMUM_READ_SIZE;

  ServerInitialReadSize  = SERVER_INITIAL_READ_SIZE;
  ServerMaximumReadSize  = SERVER_MAXIMUM_READ_SIZE;

  ProxyInitialReadSize   = PROXY_INITIAL_READ_SIZE;
  ProxyMaximumReadSize   = PROXY_MAXIMUM_READ_SIZE;

  GenericInitialReadSize = GENERIC_INITIAL_READ_SIZE;
  GenericMaximumReadSize = GENERIC_MAXIMUM_READ_SIZE;

  ShortBitrateTimeFrame = SHORT_BITRATE_TIME_FRAME;
  LongBitrateTimeFrame  = LONG_BITRATE_TIME_FRAME;

  //
  // Bandwidth control.
  //

  LocalBitrateLimit   = -1;
  LocalProcessorLimit = -1;

  ClientBitrateLimit = CLIENT_BITRATE_LIMIT;
  ServerBitrateLimit = SERVER_BITRATE_LIMIT;

  ClientProcessorLimit = CLIENT_PROCESSOR_LIMIT;
  ServerProcessorLimit = SERVER_PROCESSOR_LIMIT;

  //
  // Default parameters for message handling.
  //

  ClientTotalStorageSizeLimit = CLIENT_TOTAL_STORAGE_SIZE_LIMIT;
  ServerTotalStorageSizeLimit = SERVER_TOTAL_STORAGE_SIZE_LIMIT;

  LocalTotalStorageSizeLimit  = -1;
  RemoteTotalStorageSizeLimit = -1;

  StoreTimeLimit = STORE_TIME_LIMIT;

  StoreHitsLoadBonus = STORE_HITS_LOAD_BONUS;
  StoreHitsAddBonus  = STORE_HITS_ADD_BONUS;
  StoreHitsLimit     = STORE_HITS_LIMIT;

  StoreHitsTouch   = STORE_HITS_TOUCH;
  StoreHitsUntouch = STORE_HITS_UNTOUCH;

  MinimumMessageSizeThreshold = MINIMUM_MESSAGE_SIZE_THRESHOLD;
  MaximumMessageSizeThreshold = MAXIMUM_MESSAGE_SIZE_THRESHOLD;

  SplitMode                  = SPLIT_MODE;
  SplitTimeout               = SPLIT_TIMEOUT;
  SplitDataThreshold         = SPLIT_DATA_THRESHOLD;
  SplitDataPacketLimit       = SPLIT_DATA_PACKET_LIMIT;
  SplitTotalStorageSizeLimit = SPLIT_TOTAL_STORAGE_SIZE_LIMIT;

  AgentKarmaSize      = AGENT_KARMA_SIZE;
  AgentPackMethod     = AGENT_PACK_METHOD;
  AgentPackQuality    = AGENT_PACK_QUALITY;
  AgentFlushQueued    = AGENT_FLUSH_QUERY;
  AgentFlushPriority  = AGENT_FLUSH_PRIORITY;
  AgentHideRender     = AGENT_HIDE_RENDER;
  AgentTaintReplies   = AGENT_TAINT_REPLIES;
  AgentTaintThreshold = AGENT_TAINT_THRESHOLD;

  AgentShmemClient = AGENT_SHMEM_CLIENT;
  AgentShmemServer = AGENT_SHMEM_SERVER;

  AgentShmemClientSize = AGENT_SHMEM_CLIENT_SIZE;
  AgentShmemServerSize = AGENT_SHMEM_SERVER_SIZE;

  AgentCleanGet    = AGENT_CLEAN_GET;
  AgentCleanAlloc  = AGENT_CLEAN_ALLOC;
  AgentCleanFlush  = AGENT_CLEAN_FLUSH;
  AgentCleanSend   = AGENT_CLEAN_SEND;
  AgentCleanImages = AGENT_CLEAN_IMAGES;

  AgentImageSplit       = AGENT_IMAGE_SPLIT;
  AgentImageMask        = AGENT_IMAGE_MASK;
  AgentImageFrame       = AGENT_IMAGE_FRAME;
  AgentImageSplitMethod = AGENT_IMAGE_SPLIT_METHOD;
  AgentImageMaskMethod  = AGENT_IMAGE_MASK_METHOD;

  //
  // Get local version number from compile time
  // settings. Version of remote proxy will be
  // checked at connection time.
  //

  RemoteVersionMajor = -1;
  RemoteVersionMinor = -1;
  RemoteVersionPatch = -1;

  char version[32];

  strcpy(version, VERSION);

  char *value;

  value = strtok(version, ".");

  for (int i = 0; value != NULL && i < 3; i++)
  {
    switch (i)
    {
      case 0:

        LocalVersionMajor = atoi(value);

        break;

      case 1:

        LocalVersionMinor = atoi(value);

        break;

      case 2:

        LocalVersionPatch = atoi(value);

        break;
    }

    value = strtok(NULL, ".");
  }

  #ifdef TEST
  *logofs << "Control: Major version is " << LocalVersionMajor
          << " minor is " << LocalVersionMinor << " patch is "
          << LocalVersionPatch << ".\n" << logofs_flush;
  #endif

  //
  // Initialize local implemented methods later
  // and negotiate remote methods at connection
  // time.
  //

  LocalUnpackMethods  = NULL;
  RemoteUnpackMethods = NULL;

  //
  // Set to 1 those methods which are implemented.
  //

  setLocalUnpackMethods();

  //
  // Set initial value of flow control.
  //

  bytesIn_  = 0;
  bytesOut_ = 0;

  idleTime_  = 0;
  readTime_  = 0;
  writeTime_ = 0;

  //
  // Track the bitrate of I/O operations
  // on the proxy link.
  //

  startShortFrameTs_ = getTimestamp();
  startLongFrameTs_  = getTimestamp();
  startFrameTs_      = getTimestamp();

  bytesInShortFrame_ = 0;
  bytesInLongFrame_  = 0;

  bitrateInShortFrame_ = 0;
  bitrateInLongFrame_  = 0;

  topBitrate_ = 0;

  //
  // Track startup time.
  //

  startupTs_ = getTimestamp();

  isStartup_ = 1;

  //
  // Will have to set the protocol
  // version at the time session
  // is negotiated.
  //

  protoStep1_ = 0;
  protoStep2_ = 0;
  protoStep3_ = 0;
  protoStep4_ = 0;
  protoStep5_ = 0;
  protoStep6_ = 0;

  cacheStep1_ = 0;
  cacheStep1_ = 0;
}

Control::~Control()
{
  if (RootPath != NULL)
  {
    delete [] RootPath;
  }

  if (PersistentCachePath != NULL)
  {
    delete [] PersistentCachePath;
  }

  if (PersistentCacheName != NULL)
  {
    delete [] PersistentCacheName;
  }

  if (LocalUnpackMethods != NULL)
  {
    delete [] LocalUnpackMethods;
  }

  if (RemoteUnpackMethods != NULL)
  {
    delete [] RemoteUnpackMethods;
  }

  if (ImageCachePath != NULL)
  {
    delete [] ImageCachePath;
  }
}

//
// Recalculate current bitrate.
//

void Control::updateBitrate(int bytes)
{
  T_timestamp thisFrameTs = getTimestamp();

  int diffFramesInMs = diffTimestamp(startFrameTs_, thisFrameTs);

  #ifdef DEBUG
  *logofs << "Control: Difference since previous timestamp is "
          << diffFramesInMs << " Ms.\n" << logofs_flush;
  #endif

  if (diffFramesInMs > 0)
  {
    #ifdef DEBUG
    *logofs << "Control: Removing " << diffFramesInMs
            << " Ms in short and long time frame.\n"
            << logofs_flush;
    #endif

    int shortBytesToRemove = (int) (((double) bytesInShortFrame_ * (double) diffFramesInMs) /
                                       (double) ShortBitrateTimeFrame);

    int longBytesToRemove = (int) (((double) bytesInLongFrame_ * (double) diffFramesInMs) /
                                      (double) LongBitrateTimeFrame);

    #ifdef DEBUG
    *logofs << "Control: Removing " << shortBytesToRemove
            << " bytes from " << bytesInShortFrame_
            << " in the short frame.\n" << logofs_flush;
    #endif

    bytesInShortFrame_ -= shortBytesToRemove;

    if (bytesInShortFrame_ < 0)
    {
      #ifdef TEST
      *logofs << "Control: Bytes in short frame are "
              << bytesInShortFrame_ << ". Set to 0.\n"
              << logofs_flush;
      #endif

      bytesInShortFrame_ = 0;
    }

    #ifdef DEBUG
    *logofs << "Control: Removing " << longBytesToRemove
            << " bytes from " << bytesInLongFrame_
            << " in the long frame.\n" << logofs_flush;
    #endif

    bytesInLongFrame_ -= longBytesToRemove;

    if (bytesInLongFrame_ < 0)
    {
      #ifdef TEST
      *logofs << "Control: Bytes in long frame are "
              << bytesInLongFrame_ << ". Set to 0.\n"
              << logofs_flush;
      #endif

      bytesInLongFrame_ = 0;
    }

    int diffStartInMs;

    diffStartInMs = diffTimestamp(thisFrameTs, startShortFrameTs_);

    if (diffStartInMs > ShortBitrateTimeFrame)
    {
      addMsToTimestamp(startShortFrameTs_, diffStartInMs);
    }

    diffStartInMs = diffTimestamp(thisFrameTs, startLongFrameTs_);

    if (diffStartInMs > LongBitrateTimeFrame)
    {
      addMsToTimestamp(startLongFrameTs_, diffStartInMs);
    }

    startFrameTs_ = thisFrameTs;
  }

  #ifdef DEBUG
  *logofs << "Control: Adding " << bytes << " bytes to "
          << bytesInShortFrame_ << " in the short frame.\n"
          << logofs_flush;
  #endif

  bytesInShortFrame_ = bytesInShortFrame_ + bytes;

  #ifdef DEBUG
  *logofs << "Control: Adding " << bytes << " bytes to "
          << bytesInLongFrame_ << " in the long frame.\n"
          << logofs_flush;
  #endif

  bytesInLongFrame_ = bytesInLongFrame_ + bytes;

  bitrateInShortFrame_ = (int) ((double) bytesInShortFrame_ /
                                   ((double) ShortBitrateTimeFrame / 1000));

  bitrateInLongFrame_ = (int) ((double) bytesInLongFrame_ /
                                   ((double) LongBitrateTimeFrame / 1000));

  if (bitrateInShortFrame_ > topBitrate_)
  {
    topBitrate_ = bitrateInShortFrame_;
  }

  #ifdef TEST
  *logofs << "Control: Current bitrate is short " << bitrateInShortFrame_
          << " long " << bitrateInLongFrame_ << " top " << topBitrate_
          << ".\n" << logofs_flush;
  #endif
}

//
// Set here the pack/unpack methods that are
// implemented by this NX proxy.
//

void Control::setLocalUnpackMethods()
{
  LocalUnpackMethods  = new unsigned char[PACK_METHOD_LIMIT];
  RemoteUnpackMethods = new unsigned char[PACK_METHOD_LIMIT];

  for (int i = 0; i < PACK_METHOD_LIMIT; i++)
  {
    LocalUnpackMethods[i]  = 0;
    RemoteUnpackMethods[i] = 0;
  }

  LocalUnpackMethods[NO_PACK]                         = 1;

  LocalUnpackMethods[PACK_MASKED_8_COLORS]            = 1;
  LocalUnpackMethods[PACK_MASKED_64_COLORS]           = 1;
  LocalUnpackMethods[PACK_MASKED_256_COLORS]          = 1;
  LocalUnpackMethods[PACK_MASKED_512_COLORS]          = 1;
  LocalUnpackMethods[PACK_MASKED_4K_COLORS]           = 1;
  LocalUnpackMethods[PACK_MASKED_32K_COLORS]          = 1;
  LocalUnpackMethods[PACK_MASKED_64K_COLORS]          = 1;
  LocalUnpackMethods[PACK_MASKED_256K_COLORS]         = 1;
  LocalUnpackMethods[PACK_MASKED_2M_COLORS]           = 1;
  LocalUnpackMethods[PACK_MASKED_16M_COLORS]          = 1;

  LocalUnpackMethods[PACK_RAW_8_BITS]                 = 1;
  LocalUnpackMethods[PACK_RAW_16_BITS]                = 1;
  LocalUnpackMethods[PACK_RAW_24_BITS]                = 1;

  LocalUnpackMethods[PACK_COLORMAP_256_COLORS]        = 1;

  LocalUnpackMethods[PACK_RDP_PLAIN_256_COLORS]       = 1;
  LocalUnpackMethods[PACK_RDP_COMPRESSED_256_COLORS]  = 1;
  LocalUnpackMethods[PACK_RDP_PLAIN_32K_COLORS]       = 1;
  LocalUnpackMethods[PACK_RDP_COMPRESSED_32K_COLORS]  = 1;
  LocalUnpackMethods[PACK_RDP_PLAIN_64K_COLORS]       = 1;
  LocalUnpackMethods[PACK_RDP_COMPRESSED_64K_COLORS]  = 1;
  LocalUnpackMethods[PACK_RDP_PLAIN_16M_COLORS]       = 1;
  LocalUnpackMethods[PACK_RDP_COMPRESSED_16M_COLORS]  = 1;

  LocalUnpackMethods[PACK_RDP_TEXT]                   = 1;

  LocalUnpackMethods[PACK_RFB_HEXTILE]                = 1;

  LocalUnpackMethods[PACK_RFB_TIGHT_PLAIN]            = 1;
  LocalUnpackMethods[PACK_RFB_TIGHT_COMPRESSED]       = 1;

  LocalUnpackMethods[PACK_TIGHT_8_COLORS]             = 1;
  LocalUnpackMethods[PACK_TIGHT_64_COLORS]            = 1;
  LocalUnpackMethods[PACK_TIGHT_256_COLORS]           = 1;
  LocalUnpackMethods[PACK_TIGHT_512_COLORS]           = 1;
  LocalUnpackMethods[PACK_TIGHT_4K_COLORS]            = 1;
  LocalUnpackMethods[PACK_TIGHT_32K_COLORS]           = 1;
  LocalUnpackMethods[PACK_TIGHT_64K_COLORS]           = 1;
  LocalUnpackMethods[PACK_TIGHT_256K_COLORS]          = 1;
  LocalUnpackMethods[PACK_TIGHT_2M_COLORS]            = 1;
  LocalUnpackMethods[PACK_TIGHT_16M_COLORS]           = 1;

  LocalUnpackMethods[PACK_JPEG_8_COLORS]              = 1;
  LocalUnpackMethods[PACK_JPEG_64_COLORS]             = 1;
  LocalUnpackMethods[PACK_JPEG_256_COLORS]            = 1;
  LocalUnpackMethods[PACK_JPEG_512_COLORS]            = 1;
  LocalUnpackMethods[PACK_JPEG_4K_COLORS]             = 1;
  LocalUnpackMethods[PACK_JPEG_32K_COLORS]            = 1;
  LocalUnpackMethods[PACK_JPEG_64K_COLORS]            = 1;
  LocalUnpackMethods[PACK_JPEG_256K_COLORS]           = 1;
  LocalUnpackMethods[PACK_JPEG_2M_COLORS]             = 1;
  LocalUnpackMethods[PACK_JPEG_16M_COLORS]            = 1;

  LocalUnpackMethods[PACK_PNG_8_COLORS]               = 1;
  LocalUnpackMethods[PACK_PNG_64_COLORS]              = 1;
  LocalUnpackMethods[PACK_PNG_256_COLORS]             = 1;
  LocalUnpackMethods[PACK_PNG_512_COLORS]             = 1;
  LocalUnpackMethods[PACK_PNG_4K_COLORS]              = 1;
  LocalUnpackMethods[PACK_PNG_32K_COLORS]             = 1;
  LocalUnpackMethods[PACK_PNG_64K_COLORS]             = 1;
  LocalUnpackMethods[PACK_PNG_256K_COLORS]            = 1;
  LocalUnpackMethods[PACK_PNG_2M_COLORS]              = 1;
  LocalUnpackMethods[PACK_PNG_16M_COLORS]             = 1;

  LocalUnpackMethods[PACK_PNG_JPEG_8_COLORS]          = 1;
  LocalUnpackMethods[PACK_PNG_JPEG_64_COLORS]         = 1;
  LocalUnpackMethods[PACK_PNG_JPEG_256_COLORS]        = 1;
  LocalUnpackMethods[PACK_PNG_JPEG_512_COLORS]        = 1;
  LocalUnpackMethods[PACK_PNG_JPEG_4K_COLORS]         = 1;
  LocalUnpackMethods[PACK_PNG_JPEG_32K_COLORS]        = 1;
  LocalUnpackMethods[PACK_PNG_JPEG_64K_COLORS]        = 1;
  LocalUnpackMethods[PACK_PNG_JPEG_256K_COLORS]       = 1;
  LocalUnpackMethods[PACK_PNG_JPEG_2M_COLORS]         = 1;
  LocalUnpackMethods[PACK_PNG_JPEG_16M_COLORS]        = 1;
}
