/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef Control_H
#define Control_H

#include "NXpack.h"

#include "Misc.h"
#include "Types.h"
#include "Timestamp.h"

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

//
// Sensible default values set at compile time.
// User can override them by providing a control
// file in his/her .nx directory.
//

extern const char ControlModem[];
extern const char ControlIsdn[];
extern const char ControlAdsl[];
extern const char ControlLocal[];

//
// This is the mode proxy is running.
//

typedef enum
{
  proxy_undefined = -1,
  proxy_client,
  proxy_server,
  proxy_last_tag
}
T_proxy_mode;

//
// Handle advances in the connection
// procedure.
//

typedef enum
{
  stage_undefined,
  stage_initializing,
  stage_connecting,
  stage_connected,
  stage_waiting_forwarder_version,
  stage_waiting_forwarder_options,
  stage_sending_forwarder_options,
  stage_waiting_proxy_version,
  stage_waiting_proxy_options,
  stage_sending_proxy_options,
  stage_waiting_proxy_caches,
  stage_sending_proxy_caches,
  stage_operational
}
T_proxy_stage;

//
// This is a suggestion about the type
// of session that's running.
//

typedef enum
{
  session_undefined = -1,
  session_x,
  session_rdp,
  session_rfb,
  session_application,
  session_last_tag
}
T_session_mode;

//
// Set how data will be written to the peer
// socket.
//

typedef enum
{
  policy_undefined = -1,
  policy_immediate,
  policy_deferred
}
T_flush_policy;

//
// Preferred image streaming policy.
//

typedef enum
{
  split_default = -1,
  split_eager = 1,
  split_lazy,
  split_sync
}
T_split_mode;

//
// Link mode, after negotiation, will be set to
// any of the values defined in the NXproto.h.
//

#define link_undefined  -1;

//
// This class collects functioning parameters, 
// to be configurable at run-time. They are for
// the most part regarding timeouts, transport
// and message stores handling.
//

class Control
{
  public:

  //
  // Does proxy run in client mode or server mode?
  // As soon as we'll have gone through parsing of
  // the command line options the current mode will
  // be propagated to the control class.
  //

  T_proxy_mode ProxyMode;

  //
  // Goes from initializing to operational.
  //

  T_proxy_stage ProxyStage;

  //
  // Hint about type of session currently running.
  //

  T_session_mode SessionMode;

  //
  // Either immediate or defferred flushes.
  //

  T_flush_policy FlushPolicy;

  //
  // Id corresponding to link speed negotiated
  // between proxies.
  //

  int LinkMode;

  //
  // Maximum number of bytes sent for each token.
  //

  int TokenBytes;

  //
  // Maximum number of tokens that can be spent
  // by the client proxy before having to block
  // waiting for a reply.
  //

  int TokenLimit;

  //
  // Kill session if control parameters cannot
  // be negotiated before this timeout.
  //

  int InitTimeout;

  //
  // How many milliseconds to wait before to restart
  // handling of pending messages after a timeout.
  //

  int PendingTimeout;

  //
  // Write data to the remote proxy if a new frame is
  // not produced within this amount of milliseconds.
  //

  int FrameTimeout;

  //
  // Flush data to the remote proxy this amount of
  // milliseconds is elapsed since the last write.
  //

  int FlushTimeout;

  //
  // Enqueue motion notify events in server channel.
  //

  int MotionTimeout;

  //
  // Send a ping to remote proxy if no other data
  // was sent for this amount of milliseconds.
  //

  int PingTimeout;

  //
  // Timeout used by server channel at reset.
  //

  int ResetTimeout;

  //
  // Close the connection if can't write before
  // this timeout.
  //

  int ChannelTimeout;

  //
  // Close connection if can't write before
  // this timeout.
  //

  int ProxyTimeout;

  //
  // Timeout used to wait for a complete message
  // to become available.
  //

  int RetryTimeout;

  //
  // How many milliseconds to wait for the shared
  // memory completion event to become available.
  //

  int ShmemTimeout;

  //
  // How many milliseconds since we connected to the
  // remote proxy is to be considered startup time of
  // the session? All messages that are stored in
  // cache during this period receive an adjuntive
  // hit bonus.
  //

  int StartupTimeout;

  //
  // Wait for applications to complete at the time
  // proxy is shut down.
  //

  int CleanupTimeout;

  //
  // Wait this amount of milliseconds before any
  // iteration of the house-keeping process.
  //

  int KeeperTimeout;

  //
  // Adjust timeout calculations.
  //

  int LatencyTimeout;

  //
  // Maximum allowed size of log file.
  //

  int LogFileSizeLimit;
  int FileSizeCheckTimeout;

  //
  // Do we quit proxy in case of failure or
  // we let it drop the old connection and
  // restart?
  //

  int EnableRestartOnFailure;

  //
  // What do we do at the end of session? If
  // this flag is set we launch a new client
  // letting the user run a new NX session.
  //

  int EnableRestartOnShutdown;

  //
  // Do we generate a core dump and exit in
  // case of program errors?
  //

  int EnableCoreDumpOnAbort;

  //
  // What to do in case of SIGHUP. By default
  // we close down the proxy link and perform
  // a clean shutdown of the session. You can
  // even re-read the session configuration
  // or restart the proxy link.
  //

  int EnableShutdownOnSighup;
  int EnableReconfigOnSighup;
  int EnableRestartOnSighup;

  //
  // Version number of local and remote proxy.
  //

  int LocalVersionMajor;
  int LocalVersionMinor;
  int LocalVersionPatch;

  int RemoteVersionMajor;
  int RemoteVersionMinor;
  int RemoteVersionPatch;

  //
  // Which unpack methods are implemented in proxy?
  //

  unsigned char *LocalUnpackMethods;
  unsigned char *RemoteUnpackMethods;

  //
  // Memory restriction imposed by user.
  //

  int LocalMemoryLevel;

  //
  // Use or not differential compression
  // and caching of X protocol messages.
  //

  int LocalDeltaCompression;
  int RemoteDeltaCompression;

  //
  // Compression of images and replies.
  //

  int LocalDataCompression;
  int LocalDataCompressionLevel;

  int RemoteDataCompression;
  int RemoteDataCompressionLevel;

  //
  // Minimum packet size to be compressed.
  //

  int LocalDataCompressionThreshold;

  //
  // Compress or not data flowing through the proxy
  // link. Level should be one of the ZLIB level as
  // Z_DEFAULT_COMPRESSION or Z_BEST_COMPRESSION.
  //

  int LocalStreamCompression;
  int LocalStreamCompressionLevel;

  int RemoteStreamCompression;
  int RemoteStreamCompressionLevel;

  //
  // Size of read operations in read buffer classes.
  //

  int ClientInitialReadSize;
  int ClientMaximumReadSize;

  int ServerInitialReadSize;
  int ServerMaximumReadSize;

  int ProxyInitialReadSize;
  int ProxyMaximumReadSize;

  int GenericInitialReadSize;
  int GenericMaximumReadSize;

  //
  // Set initial size and resize policy of
  // transport buffers. If maximum size is
  // exceeded, print a warning.
  //

  int TransportXBufferSize;
  int TransportProxyBufferSize;
  int TransportGenericBufferSize;

  int TransportXBufferLimit;
  int TransportProxyBufferLimit;
  int TransportGenericBufferLimit;

  int TransportMaximumBufferSize;
  int TransportForceBufferSize;

  //
  // Stop reading from proxy if channels
  // exceed these buffer limits.
  //

  int TransportXBufferThreshold;
  int TransportProxyBufferThreshold;
  int TransportGenericBufferThreshold;

  //
  // Immediately flush data produced for the
  // X connection if it exceeds this limit.
  //

  int TransportWriteThreshold;

  //
  // Set this to the maximum amount of data that 
  // will be sent in a single write operation.
  //

  int TransportWriteLimit;

  //
  // Socket options.
  //

  int OptionProxyKeepAlive;
  int OptionProxyLowDelay;
  int OptionProxyClientNoDelay;
  int OptionProxyServerNoDelay;
  int OptionClientNoDelay;
  int OptionServerNoDelay;

  int OptionProxyReceiveBuffer;
  int OptionClientReceiveBuffer;
  int OptionServerReceiveBuffer;

  int OptionProxySendBuffer;
  int OptionClientSendBuffer;
  int OptionServerSendBuffer;

  int OptionProxyRetryAccept;
  int OptionProxyRetryConnect;
  int OptionServerRetryConnect;

  //
  // Calculate current bitrate on proxy link
  // using these observation periods. Value
  // is in milliseconds.  
  //

  int ShortBitrateTimeFrame;
  int LongBitrateTimeFrame;

  //
  // Limit bandwidth on proxy link and
  // CPU time spent handling messages.
  //

  int LocalBitrateLimit;
  int LocalProcessorLimit;

  int ClientBitrateLimit;
  int ServerBitrateLimit;
  
  int ClientProcessorLimit;
  int ServerProcessorLimit;
  
  //
  // This is the limit imposed by user on
  // total cache size.
  //

  int ClientTotalStorageSizeLimit;
  int ServerTotalStorageSizeLimit;

  int LocalTotalStorageSizeLimit;
  int RemoteTotalStorageSizeLimit;

  //
  // Discard messages in store older than
  // this amount of seconds.
  //

  int StoreTimeLimit;

  //
  // Any new message in store starts with
  // this amount of hits.
  //

  int StoreHitsAddBonus;

  //
  // Unless it is loaded from persistent
  // cache.
  //

  int StoreHitsLoadBonus;

  //
  // Stop increasing hits at this threshold.
  //

  int StoreHitsLimit;

  //
  // Give a special weight to messages put or
  // taken from cache during startup time.
  //

  int StoreHitsStartup;

  //
  // Weight of touch and untoch operations.
  //

  int StoreHitsTouch;
  int StoreHitsUntouch;

  //
  // Directives on size of messages to cache.
  //

  int MinimumMessageSizeThreshold;
  int MaximumMessageSizeThreshold;

  //
  // Currently selected streaming mode.
  //

  int SplitMode;

  //
  // Any given amount of milliseconds send
  // a new split to the remote side.
  //

  int SplitTimeout;

  //
  // Split messages are sent asynchronously
  // on the link. Messages smaller that this
  // threshold are sent in one shot, and the
  // agent is immediately notified.
  //

  int SplitDataThreshold;

  //
  // This is maximum block size to be used to
  // transfer split messages to the remote
  // peer.
  //

  int SplitDataPacketLimit;

  //
  // Maximum size of temporary storage.
  //

  int SplitTotalStorageSizeLimit;

  //
  // Is statistic output enabled?
  //

  int CollectStatistics;

  //
  // Agent related parameters. Some of these
  // values apply to agent which, at startup,
  // must query the user's settings, others
  // are related to internal optimizations
  // when running proxy in an NX environment.
  //

  int AgentKarmaSize;
  int AgentPackMethod;
  int AgentPackQuality;
  int AgentFlushQueued;
  int AgentFlushPriority;
  int AgentHideRender;
  int AgentTaintReplies;
  int AgentTaintThreshold;

  //
  // Do we allow shared memory image support in
  // client and or server?
  //

  int AgentShmemClient;
  int AgentShmemServer;

  //
  // Default size of shared memory segments used
  // in MIT-SHM support.
  //

  int AgentShmemClientSize;
  int AgentShmemServerSize;

  //
  // Should agent apply clean of pad and unused
  // bytes in X protocol?
  //

  int AgentCleanGet;
  int AgentCleanAlloc;
  int AgentCleanFlush;
  int AgentCleanSend;
  int AgentCleanImages;

  //
  // Should apply special operation on images?
  //

  int AgentImageSplit;
  int AgentImageMask;
  int AgentImageFrame;
  int AgentImageSplitMethod;
  int AgentImageMaskMethod;

  //
  // Usually the user's ".nx" directory.
  //

  char *RootPath;

  //
  // String containing path of cache
  // file selected for load or save.
  //

  char *PersistentCachePath;

  //
  // Name of selected cache file.
  //

  char *PersistentCacheName;

  //
  // Minimum size of cache in memory
  // to proceed to its storage on disk.
  //

  int PersistentCacheThreshold;

  //
  // Is persistent cache enabled?
  //

  int PersistentCacheEnableLoad;
  int PersistentCacheEnableSave;

  //
  // Load packed image and render extension
  // message stores. This currently depends
  // on the type of session.
  //

  int PersistentCacheLoadPacked;
  int PersistentCacheLoadRender;

  //
  // Maximum allowed size of image
  // cache on disk.
  //

  int PersistentCacheDiskLimit;

  //
  // String containing the base path
  // of image cache files.
  //

  char *ImageCachePath;

  //
  // Is image cache enabled?
  //

  int ImageCacheEnableLoad;
  int ImageCacheEnableSave;

  //
  // Maximum allowed size of image
  // cache on disk.
  //

  int ImageCacheDiskLimit;

  //
  // Only constructor, destructor
  // and a few utility functions.
  //

  Control();

  ~Control();

  //
  // Should not leverage control to find channel
  // stores' size limits. As most of values in
  // control, this info must be moved elsewhere.
  //

  int getUpperStorageSizeLimit() const
  {
    return (ClientTotalStorageSizeLimit >
                ServerTotalStorageSizeLimit ?
                    ClientTotalStorageSizeLimit :
                        ServerTotalStorageSizeLimit);
  }

  int getLowerStorageSizeLimit() const
  {
    return (ClientTotalStorageSizeLimit <
                ServerTotalStorageSizeLimit ?
                    ClientTotalStorageSizeLimit :
                        ServerTotalStorageSizeLimit);
  }

  //
  // Track CPU usage.
  //

  void addIdleTime(unsigned int numMs)
  {
    idleTime_ += numMs;
  }

  void subIdleTime(unsigned int numMs)
  {
    idleTime_ -= numMs;
  }

  double getIdleTime() const
  {
    return idleTime_;
  }

  void addReadTime(unsigned int numMs)
  {
    readTime_ += numMs;
  }

  void subReadTime(unsigned int numMs)
  {
    readTime_ -= numMs;
  }

  double getReadTime() const
  {
    return readTime_;
  }

  void addWriteTime(unsigned int numMs)
  {
    writeTime_ += numMs;
  }

  void subWriteTime(unsigned int numMs)
  {
    writeTime_ -= numMs;
  }

  double getWriteTime() const
  {
    return writeTime_;
  }

  //
  // Track network usage.
  //

  void addBytesIn(unsigned int numBytes)
  {
    bytesIn_ += (double) numBytes;
  }

  double getBytesIn()
  {
    return bytesIn_;
  }

  void addBytesOut(unsigned int numBytes)
  {
    bytesOut_ += (double) numBytes;
  }

  double getBytesOut()
  {
    return bytesOut_;
  }

  //
  // Update the current bitrate.
  //

  void updateBitrate(int bytes);

  //
  // Return the current bitrate.
  //

  int getBitrateInShortFrame()
  {
    return bitrateInShortFrame_;
  }

  int getBitrateInLongFrame()
  {
    return bitrateInLongFrame_;
  }

  int getTopBitrate()
  {
    return topBitrate_;
  }

  void resetTopBitrate()
  {
    topBitrate_ = 0;
  }

  int isStartup()
  {
    if (isStartup_ == 0)
    {
      return 0;
    }
    else
    {
      if (StartupTimeout >= LatencyTimeout)
      {
        isStartup_ = ((StartupTimeout - LatencyTimeout -
                           diffTimestamp(startupTs_, getTimestamp())) > 0);
      }
      else
      {
        isStartup_ = 0;
      }
    }

    return isStartup_;
  }

  void setProtoStep1()
  {
    protoStep1_ = 1;
    protoStep2_ = 0;
    protoStep3_ = 0;
    protoStep4_ = 0;
    protoStep5_ = 0;
    protoStep6_ = 0;
  }

  void setProtoStep2()
  {
    protoStep1_ = 0;
    protoStep2_ = 1;
    protoStep3_ = 0;
    protoStep4_ = 0;
    protoStep5_ = 0;
    protoStep6_ = 0;
  }

  void setProtoStep3()
  {
    protoStep1_ = 0;
    protoStep2_ = 1;
    protoStep3_ = 1;
    protoStep4_ = 0;
    protoStep5_ = 0;
    protoStep6_ = 0;
  }

  void setProtoStep4()
  {
    protoStep1_ = 0;
    protoStep2_ = 1;
    protoStep3_ = 1;
    protoStep4_ = 1;
    protoStep5_ = 0;
    protoStep6_ = 0;
  }

  void setProtoStep5()
  {
    protoStep1_ = 0;
    protoStep2_ = 1;
    protoStep3_ = 1;
    protoStep4_ = 1;
    protoStep5_ = 1;
    protoStep6_ = 0;
  }

  void setProtoStep6()
  {
    protoStep1_ = 0;
    protoStep2_ = 1;
    protoStep3_ = 1;
    protoStep4_ = 1;
    protoStep5_ = 1;
    protoStep6_ = 1;
  }

  int isProtoStep1()
  {
    return protoStep1_;
  }

  int isProtoStep2()
  {
    return protoStep2_;
  }

  int isProtoStep3()
  {
    return protoStep3_;
  }

  int isProtoStep4()
  {
    return protoStep4_;
  }

  int isProtoStep5()
  {
    return protoStep5_;
  }

  int isProtoStep6()
  {
    return protoStep6_;
  }

  void setCacheStep1()
  {
    cacheStep1_ = 1;
    cacheStep2_ = 0;
  }

  void setCacheStep2()
  {
    cacheStep1_ = 0;
    cacheStep2_ = 1;
  }

  int isCacheStep1()
  {
    return cacheStep1_;
  }

  int isCacheStep2()
  {
    return cacheStep2_;
  }

  private:

  //
  // Look in Control.cpp.
  //

  void setLocalUnpackMethods();

  //
  // Track CPU and network usage.
  //

  double idleTime_;
  double readTime_;
  double writeTime_;

  double bytesIn_;
  double bytesOut_;

  //
  // Track the current bitrate used for
  // I/O operations on the proxy link.
  //

  T_timestamp startShortFrameTs_;
  T_timestamp startLongFrameTs_;
  T_timestamp startFrameTs_;

  int bytesInShortFrame_;
  int bytesInLongFrame_;

  int bitrateInShortFrame_;
  int bitrateInLongFrame_;

  int topBitrate_;

  //
  // Time since we started the proxy.
  //

  T_timestamp startupTs_;
  int isStartup_;

  //
  // Manage encoding according to
  // protocol versions.
  //

  int protoStep1_;
  int protoStep2_;
  int protoStep3_;
  int protoStep4_;
  int protoStep5_;
  int protoStep6_;

  //
  // Manage the version of cache.
  //

  int cacheStep1_;
  int cacheStep2_;
};

#endif /* Control_H */
