/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include "Misc.h"
#include "Control.h"

#include "Decompressor.h"
#include "DecodeBuffer.h"

#define PANIC
#undef  TEST
#undef  DEBUG

Decompressor::Decompressor()
{
  stream_.zalloc = (alloc_func) 0;
  stream_.zfree  = (free_func) 0;
  stream_.opaque = (void *) 0;

  stream_.next_in = (Bytef *) 0;
  stream_.avail_in = 0;

  int result = inflateInit2(&stream_, 15);

  if (result != Z_OK)
  {
    #ifdef PANIC
    *logofs << "Decompressor: PANIC! Cannot initialize data decompression "
            << "library. Error is '" << zError(result) << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot initialize data decompression "
         << "library. Error is '" << zError(result) << "'.\n";
  }
}

Decompressor::~Decompressor()
{
  int result = inflateEnd(&stream_);

  if (result != Z_OK)
  {
    #ifdef PANIC
    *logofs << "Decompressor: PANIC! Cannot deinitialize data decompression "
            << "library. Error is '" << zError(result) << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot deinitialize data decompression "
         << "library. Error is '" << zError(result) << "'.\n";
  }
}

int Decompressor::decompressBuffer(unsigned char *decompressedBuffer,
                                       unsigned int decompressedSize,
                                           const unsigned char *&compressedBuffer,
                                               unsigned int &compressedSize,
                                                   DecodeBuffer &decodeBuffer)
{
  #ifdef DEBUG
  *logofs << "Decompressor: Called for buffer at "
          << (void *) decompressedBuffer << ".\n"
          << logofs_flush;
  #endif

  unsigned int value;

  decodeBuffer.decodeBool(value);

  if (value == 0)
  {
    memcpy(decompressedBuffer,
               decodeBuffer.decodeMemory(decompressedSize),
                   decompressedSize);

    return 0;
  }

  uLongf checkSize = decompressedSize;

  if (control -> isProtoStep2() == 1)
  {
    decodeBuffer.decodeValue(value, 32, 14);
    compressedSize = value;

    decodeBuffer.decodeValue(value, 32, 14);
    checkSize = value;
  }
  else
  {
    decodeBuffer.decodeValue(value, 32);
    compressedSize = value;

    decodeBuffer.decodeValue(value, 32);
    checkSize = value;
  }

  //
  // If caller needs the original compressed
  // data it must copy this to its own buffer
  // before using any further decode function.
  //

  compressedBuffer = decodeBuffer.decodeMemory(compressedSize);

  int result = decompress(decompressedBuffer, &checkSize,
                              compressedBuffer, compressedSize);

  if (result != Z_OK)
  {
    #ifdef PANIC
    *logofs << "Decompressor: PANIC! Failure decompressing buffer. "
            << "Error is '" << zError(result) << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Failure decompressing buffer. "
         << "Error is '" << zError(result) << "'.\n";

    return -1;
  }
  else if (decompressedSize != checkSize)
  {
    #ifdef PANIC
    *logofs << "Decompressor: PANIC! Expected decompressed size was "
            << decompressedSize << " while it is " << checkSize 
            << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Expected decompressed size was "
         << decompressedSize << " while it is " << checkSize
         << ".\n";

    return -1;
  }

  return 1;
}

//
// This is used to uncompress on-the-fly messages 
// whose data is stored in compressed format.
//

int Decompressor::decompressBuffer(unsigned char *decompressedBuffer,
                                       const unsigned int decompressedSize,
                                           const unsigned char *compressedBuffer,
                                               const unsigned int compressedSize)
{
  #ifdef TEST
  *logofs << "Decompressor: Called for buffer at "
          << (void *) decompressedBuffer << ".\n"
          << logofs_flush;
  #endif

  uLongf checkSize = decompressedSize;

  int result = decompress(decompressedBuffer, &checkSize,
                              compressedBuffer, compressedSize);
  if (result != Z_OK)
  {
    #ifdef PANIC
    *logofs << "Decompressor: PANIC! Failure decompressing buffer. "
            << "Error is '" << zError(result) << "'.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  if (decompressedSize != checkSize)
  {
    #ifdef PANIC
    *logofs << "Decompressor: PANIC! Expected decompressed size was "
            << decompressedSize << " while it is " << checkSize 
            << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Expected decompressed size was "
         << decompressedSize << " while it is " << checkSize
         << ".\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "Decompressor: Decompressed buffer from "
          << compressedSize << " to " << decompressedSize
          << " bytes.\n" << logofs_flush;
  #endif

  return 1;
}


int Decompressor::decompress(Bytef *dest, uLongf *destLen,
                                 const Bytef *source, uLong sourceLen)
{
  stream_.next_in  = (Bytef *) source;
  stream_.avail_in = (uInt) sourceLen;

  unsigned int oldTotalOut = stream_.total_out;

  if ((uLong) stream_.avail_in != sourceLen)
  {
    return Z_BUF_ERROR;
  }

  stream_.next_out  = dest;
  stream_.avail_out = (uInt) *destLen;

  if ((uLong) stream_.avail_out != *destLen)
  {
    return Z_BUF_ERROR;
  }

  int result = inflate(&stream_, Z_FINISH);

  if (result != Z_STREAM_END)
  {
    inflateReset(&stream_);

    return (result == Z_OK ? Z_BUF_ERROR : result);
  }

  *destLen = stream_.total_out - oldTotalOut;

  result = inflateReset(&stream_);

  return result;
}
