/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef GenericChannel_H
#define GenericChannel_H

#include "Channel.h"

#include "Statistics.h"

#include "GenericReadBuffer.h"

//
// Set the verbosity level.
//

#define PANIC
#undef  TEST
#undef  DEBUG

//
// Define this to log a line when a channel
// is created or destroyed.
//

#undef  REFERENCES

//
// This class implements the client
// side compression of X protocol.
//

class GenericChannel : public Channel
{
  public:

  GenericChannel(Transport *transport, Compressor *compressor,
                     Decompressor *decompressor);

  virtual ~GenericChannel();

  virtual int handleRead(EncodeBuffer &encodeBuffer);

  virtual int handleWrite(const unsigned char *message, unsigned int length);

  virtual int handleSplit(EncodeBuffer &encodeBuffer, MessageStore *store, 
                              const unsigned char *buffer, const unsigned int size);

  virtual int handleSplit(EncodeBuffer &encodeBuffer, int packetLimit);

  virtual int handleUnsplit(DecodeBuffer &decodeBuffer, MessageStore *store,
                                unsigned char *buffer, const unsigned int size);

  virtual int handleUnsplit(DecodeBuffer &decodeBuffer);

  virtual int handleAbortSplit(EncodeBuffer &encodeBuffer);

  virtual int handleAbortSplit(DecodeBuffer &decodeBuffer);

  virtual int handleMotion(EncodeBuffer &encodeBuffer, int force)
  {
    return 0;
  }

  virtual int handleConfiguration();

  virtual int handleFinish();

  virtual int handleNotify(T_notification_type type)
  {
    return 0;
  }
    
  virtual int handleReset();

  virtual int needSplit() const
  {
    return 0;
  }

  virtual int needAbort() const
  {
    return 0;
  }

  virtual int needMotion() const
  {
    return 0;
  }

  virtual int needLimit() const
  {
    return (transport_ -> length() >=
                control -> TransportGenericBufferLimit);
  }

  virtual int getLimit() const
  {
    return (control -> TransportGenericBufferLimit);
  }

  virtual T_channel_type getType() const = 0;

  //
  // Used to implement synchronization
  // between proxies.
  //

  virtual int getClientSequence() const
  {
    return -1;
  }

  virtual int getServerSequence() const
  {
    return -1;
  }

  protected:

  //
  // Model generic channels' encoding and decoding
  // policy.
  //

  virtual int isCompressed() = 0;

  //
  // Set flush policy.
  //

  virtual int isPrioritized() = 0;

  virtual int isInteractive() = 0;

  //
  // Generic channel are put in congestion state as
  // soon as the socket is blocked for write.
  //

  virtual int isCongested()
  {
    return (transport_ -> blocked() == 1);
  }

  //
  // Generic channels are considered not reliable.
  //

  virtual int isReliable()
  {
    return 0;
  }

  //
  // Determine if event is user input.
  //

  virtual int isInput(int code)
  {
    return 0;
  }

  //
  // Determine when it's time to give up
  // reading from the channel.
  //

  int checkProxyEvents(int count);

  //
  // Register protocol bits for
  // specific service.
  //

  virtual void addProtocolBits(unsigned int bitsIn, unsigned int bitsOut) = 0;

  //
  // Channel's own read buffer.
  //

  GenericReadBuffer readBuffer_;

  private:

  //
  // Keep track of object's creation
  // and deletion.
  //

  #ifdef REFERENCES

  static int references_;

  #endif
};

class CupsChannel : public GenericChannel
{
  public:

  CupsChannel(Transport *transport, Compressor *compressor,
                   Decompressor *decompressor)

    : GenericChannel(transport, compressor, decompressor)
  {
  }

  virtual ~CupsChannel()
  {
  }

  protected:

  virtual T_channel_type getType() const
  {
    return channel_cups;
  }

  virtual int isCompressed()
  {
    return 1;
  }

  virtual int isPrioritized()
  {
    return 0;
  }

  virtual int isInteractive()
  {
    return 0;
  }

  virtual void addProtocolBits(unsigned int bitsIn,
                                   unsigned int bitsOut)
  {
    if (control -> CollectStatistics)
    {
      statistics -> addSyncBits(bitsIn, bitsOut);
    }
  }
};

class KeybdChannel : public GenericChannel
{
  public:

  KeybdChannel(Transport *transport, Compressor *compressor,
                   Decompressor *decompressor)

    : GenericChannel(transport, compressor, decompressor)
  {
  }

  virtual ~KeybdChannel()
  {
  }

  protected:

  virtual T_channel_type getType() const
  {
    return channel_keybd;
  }

  virtual int isCompressed()
  {
    return 1;
  }

  virtual int isPrioritized()
  {
    return 0;
  }

  virtual int isInteractive()
  {
    return 0;
  }

  virtual void addProtocolBits(unsigned int bitsIn,
                                   unsigned int bitsOut)
  {
    if (control -> CollectStatistics)
    {
      statistics -> addKeybdBits(bitsIn, bitsOut);
    }
  }
};

class SambaChannel : public GenericChannel
{
  public:

  SambaChannel(Transport *transport, Compressor *compressor,
                   Decompressor *decompressor)

    : GenericChannel(transport, compressor, decompressor)
  {
  }

  virtual ~SambaChannel()
  {
  }

  protected:

  virtual T_channel_type getType() const
  {
    return channel_samba;
  }

  virtual int isCompressed()
  {
    return 1;
  }

  virtual int isPrioritized()
  {
    return 0;
  }

  virtual int isInteractive()
  {
    return 0;
  }

  virtual void addProtocolBits(unsigned int bitsIn,
                                   unsigned int bitsOut)
  {
    if (control -> CollectStatistics)
    {
      statistics -> addSambaBits(bitsIn, bitsOut);
    }
  }
};

class MediaChannel : public GenericChannel
{
  public:

  MediaChannel(Transport *transport, Compressor *compressor,
                   Decompressor *decompressor)

    : GenericChannel(transport, compressor, decompressor)
  {
  }

  virtual ~MediaChannel()
  {
  }

  protected:

  virtual T_channel_type getType() const
  {
    return channel_media;
  }

  //
  // Don't try to compress the media data.
  //

  virtual int isCompressed()
  {
    return 0;
  }

  //
  // We reduce latency on media channels
  // by setting them as interactive.
  //

  virtual int isPrioritized()
  {
    return 0;
  }

  virtual int isInteractive()
  {
    return 1;
  }

  virtual void addProtocolBits(unsigned int bitsIn,
                                   unsigned int bitsOut)
  {
    if (control -> CollectStatistics)
    {
      statistics -> addMediaBits(bitsIn, bitsOut);
    }
  }
};

class HttpChannel : public GenericChannel
{
  public:

  HttpChannel(Transport *transport, Compressor *compressor,
                  Decompressor *decompressor)

    : GenericChannel(transport, compressor, decompressor)
  {
  }

  virtual ~HttpChannel()
  {
  }

  protected:

  virtual T_channel_type getType() const
  {
    return channel_http;
  }

  virtual int isCompressed()
  {
    return 1;
  }

  virtual int isPrioritized()
  {
    return 0;
  }

  virtual int isInteractive()
  {
    return 0;
  }

  virtual void addProtocolBits(unsigned int bitsIn,
                                   unsigned int bitsOut)
  {
    if (control -> CollectStatistics)
    {
      statistics -> addHttpBits(bitsIn, bitsOut);
    }
  }
};

#endif /* GenericChannel_H */
