/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#include "Keeper.h"

//
// Sleep once any ONCE entries.
//

#define ONCE  2

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

//
// Define this to know how many messages
// are allocated and deallocated.
//

#undef  REFERENCES

//
// This is used for reference count.
//

#ifdef REFERENCES

int File::references_ = 0;

#endif

bool T_older::operator () (File *a, File *b) const
{
  return a -> compare(b);
}

File::File()
{
  name_ = NULL;

  size_ = 0;
  time_ = 0;

  #ifdef REFERENCES

  references_++;

  *logofs << "File: Created new File at "
          << this << " out of " << references_
          << " allocated references.\n"
          << logofs_flush;

  #endif
}

File::~File()
{
  delete [] name_;

  #ifdef REFERENCES

  references_--;

  *logofs << "Split: Deleted File at " 
          << this << " out of " << references_ 
          << " allocated references.\n" << logofs_flush;
  #endif
}

bool File::compare(File *b) const
{
  if (this -> time_ == b -> time_)
  {
    return (this -> size_ < b -> size_);
  }

  return (this -> time_ < b -> time_);
}

Keeper::Keeper(int caches, int images, const char *root,
                   int sleep, int parent)
{
  caches_ = caches;
  images_ = images;
  sleep_  = sleep;
  parent_ = parent;

  root_ = new char[strlen(root) + 1];

  strcpy(root_, root);

  files_ = new T_files();

  total_ = 0;
}

Keeper::~Keeper()
{
  empty();

  delete files_;

  delete [] root_;
}

int Keeper::keepCaches()
{
  char *rootPath = new char[strlen(root_) + 1];

  if (rootPath == NULL)
  {
    return -1;
  }

  strcpy(rootPath, root_);

  #ifdef TEST
  *logofs << "Keeper: Looking for cache directories in '"
          << rootPath << "'.\n" << logofs_flush;
  #endif

  DIR *rootDir = opendir(rootPath);

  if (rootDir != NULL)
  {
    dirent *dirEntry;

    struct stat fileStat;

    int baseSize = strlen(rootPath);

    int s = 0;

    while (((dirEntry = readdir(rootDir)) != NULL))
    {
      if (s++ % ONCE == 0) usleep(sleep_ * 1000);

      if (strcmp(dirEntry -> d_name, "cache") == 0 ||
              strncmp(dirEntry -> d_name, "cache-", 6) == 0)
      {
        char *dirName = NULL;

        try
        {
          dirName = new char[baseSize + strlen(dirEntry -> d_name) + 2];

          strcpy(dirName, rootPath);
          strcpy(dirName + baseSize, "/");
          strcpy(dirName + baseSize + 1, dirEntry -> d_name);
        }
        catch (exception e)
        {
          #ifdef WARNING
          *logofs << "Keeper: WARNING! Can't check directory entry '"
                  << dirEntry -> d_name << "'.\n" << logofs_flush;
          #endif

          delete [] dirName;

          continue;
        }

        #ifdef TEST
        *logofs << "Keeper: Checking directory '" << dirName
                << "'.\n" << logofs_flush;
        #endif

        if (stat(dirName, &fileStat) == 0 &&
                S_ISDIR(fileStat.st_mode) != 0)
        {
          //
          // Add to repository all the "C-" and
          // "S-" files in the given directory.
          //

          collect(dirName);
        }

        delete [] dirName;
      }
    }

    closedir(rootDir);
  }
  else
  {
    #ifdef WARNING
    *logofs << "Keeper: WARNING! Can't open NX root directory '"
            << rootPath << "'. Error is " << EGET() << " '"
            << ESTR() << "'.\n" << logofs_flush;
     #endif

     cerr << "Warning" << ": Can't open NX root directory '"
          << rootPath << "'. Error is " << EGET() << " '"
          << ESTR() << "'.\n";
  }

  delete [] rootPath;

  //
  // Remove older files.
  //

  cleanup(threshold(caches_));

  //
  // Empty the repository.
  //

  empty();

  return 1;
}

int Keeper::keepImages()
{
  char *rootPath = new char[strlen(root_) + 1];

  if (rootPath == NULL)
  {
    return -1;
  }

  strcpy(rootPath, root_);

  char *imagesPath = new char[strlen(rootPath) + strlen("/images") + 1];

  if (imagesPath == NULL)
  {
    delete [] rootPath;

    return -1;
  }

  strcpy(imagesPath, rootPath);

  strcat(imagesPath, "/images");

  //
  // Check if the cache directory does exist.
  //

  struct stat dirStat;

  if (stat(imagesPath, &dirStat) == -1)
  {
    #ifdef PANIC
    *logofs << "Keeper: PANIC! Can't stat NX images cache directory '"
            << imagesPath << ". Error is " << EGET() << " '"
            << ESTR() << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Can't stat NX images cache directory '"
         << imagesPath << ". Error is " << EGET() << " '"
         << ESTR() << "'.\n";

    delete [] rootPath;
    delete [] imagesPath;

    return -1;
  }

  //
  // Check any of the 16 directories in the
  // images root path.
  //

  char *digitPath = new char[strlen(imagesPath) + 5];

  strcpy(digitPath, imagesPath);

  for (char digit = 0; digit < 16; digit++)
  {
    //
    // Give up if our parent is gone.
    //

    if (parent_ != getppid() || parent_ == 1)
    {
      #ifdef WARNING
      *logofs << "Keeper: WARNING! Parent process appears "
              << "to be dead. Returning.\n"
              << logofs_flush;
      #endif

      return -1;
    }

    sprintf(digitPath + strlen(imagesPath), "/I-%01X", digit);

    //
    // Add to repository all files
    // in the given directory.
    //

    collect(digitPath);
  }

  delete [] rootPath;
  delete [] imagesPath;
  delete [] digitPath;

  //
  // Remove older files.
  //

  cleanup(threshold(images_));

  //
  // Empty the repository.
  //

  empty();

  return 1;
}

int Keeper::collect(const char *path)
{
  #ifdef TEST
  *logofs << "Keeper: Looking for files in directory '"
          << path << "'.\n" << logofs_flush;
  #endif

  DIR *cacheDir = opendir(path);

  if (cacheDir != NULL)
  {
    dirent *dirEntry;

    struct stat fileStat;

    int baseSize = strlen(path);
    int fileSize = baseSize + 3 + MD5_LENGTH * 2 + 1;

    int s = 0;

    while (((dirEntry = readdir(cacheDir)) != NULL))
    {
      if (s++ % ONCE == 0) usleep(sleep_ * 1000);

      if (strlen(dirEntry -> d_name) == (MD5_LENGTH * 2 + 2) &&
              (strncmp(dirEntry -> d_name, "I-", 2) == 0 ||
                  strncmp(dirEntry -> d_name, "S-", 2) == 0 ||
                      strncmp(dirEntry -> d_name, "C-", 2) == 0))
      {
        File *file = NULL;

        char *fileName = NULL;

        try
        {
          fileName = new char[fileSize];

          strcpy(fileName, path);
          strcpy(fileName + baseSize, "/");
          strcpy(fileName + baseSize + 1, dirEntry -> d_name);

          file = new File();

          file -> name_ = fileName;
        }
        catch (exception e)
        {
          #ifdef WARNING
          *logofs << "Keeper: WARNING! Can't add file '"
                  << dirEntry -> d_name << "' to repository.\n"
                  << logofs_flush;
          #endif

          delete [] fileName;
          delete [] file;

          continue;
        }

        #ifdef DEBUG
        *logofs << "Keeper: Adding file '" << file -> name_
                << "'.\n" << logofs_flush;
        #endif

        if (stat(file -> name_, &fileStat) == -1)
        {
          #ifdef PANIC
          *logofs << "Keeper: WARNING! Can't stat NX file '"
                  << file -> name_ << ". Error is " << EGET()
                  << " '" << ESTR() << "'.\n"
                  << logofs_flush;
          #endif

          cerr << "Warning" << ": Can't stat NX file '"
               << file -> name_ << ". Error is " << EGET()
               << " '" << ESTR() << "'.\n";

          delete file;

          return -1;
        }

        file -> size_ = fileStat.st_size;
        file -> time_ = fileStat.st_mtime;

        try
        {
          files_ -> insert(T_files::value_type(file));
        }
        catch (exception e)
        {
          #ifdef WARNING
          *logofs << "Keeper: WARNING! Can't add file '"
                  << dirEntry -> d_name << "' to repository.\n"
                  << logofs_flush;
          #endif

          delete file;
        }

        total_ += file -> size_;
      }
    }

    closedir(cacheDir);
  }
  else
  {
    #ifdef PANIC
    *logofs << "Keeper: WARNING! Can't open NX images subdirectory '"
            << path << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
     #endif

     cerr << "Warning" << ": Can't open NX images subdirectory '"
          << path << ". Error is " << EGET() << " '" << ESTR()
          << "'.\n";
  }

  return 1;
}

int Keeper::cleanup(int threshold)
{
  #ifdef TEST
  *logofs << "Keeper: Checking older files with threshold "
          << threshold << " and " << total_
          << " bytes in repository.\n" << logofs_flush;
  #endif

  for (T_files::iterator i = files_ -> begin();
           i != files_ -> end() && total_ > threshold; i++)
  {
    #ifdef TEST
    *logofs << "Keeper: Removing '" << (*i) -> name_
            << "' with time " << (*i) -> time_ << " and size "
            << (*i) -> size_ << ".\n" << logofs_flush;
    #endif

    unlink((*i) -> name_);

    total_ -= (*i) -> size_;

    delete *i;

    files_ -> erase(i);
  }

  return 1;
}

void Keeper::empty()
{
  #ifdef TEST
  *logofs << "Keeper: Getting rid of files in repository.\n"
          << logofs_flush;
  #endif

  for (T_files::iterator i = files_ -> begin();
           i != files_ -> end(); i++)
  {
    delete *i;
  }

  files_ -> erase(files_ -> begin(), files_ -> end());

  total_ = 0;
}
