/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <errno.h>
#include <signal.h>
#include <setjmp.h>

#include <math.h>
#include <ctype.h>
#include <string.h>
#include <dirent.h>
#include <pwd.h>

#include <fstream.h>
#include <fcntl.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/resource.h>
#include <sys/utsname.h>

#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#ifdef __sun
#include <strings.h>
#include <sys/filio.h>
#endif

//
// MacOSX 10.4 defines socklen_t. This is
// intended to ensure compatibility with
// older versions.
//

#ifdef __APPLE__
#include <AvailabilityMacros.h>
#if MAC_OS_X_VERSION_MAX_ALLOWED <= MAC_OS_X_VERSION_10_3
typedef int socklen_t;
#endif
#endif

#ifdef _AIX
#include <strings.h>
#include <sys/select.h>
#endif

#ifndef __CYGWIN32__
#include <netinet/tcp.h>
#include <sys/un.h>
#endif

//
// NX include files.
//

#include "NX.h"
#include "NXalert.h"

#include "Misc.h"
#include "Control.h"
#include "Socket.h"
#include "Statistics.h"
#include "Auth.h"
#include "Agent.h"

#include "ClientProxy.h"
#include "ServerProxy.h"

#include "Message.h"

//
// System specific defines.
//

#if defined(__EMX__ ) || defined(__CYGWIN32__)

struct sockaddr_un
{
  u_short sun_family;
  char sun_path[108];
};

#endif

//
// HP-UX hides this define.
//

#if defined(hpux) && !defined(RLIM_INFINITY)

#define RLIM_INFINITY  0x7fffffff

#endif

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG
#undef  DUMP

//
// Enable log output in signal handler.
// This is likely to hang the proxy at
// random, at least on Linux.
//

#undef  UNSAFE

//
// Let all logs go to the standard error.
// This is useful to interleave the Xlib
// log output with the proxy output in a
// single file.
//

#undef  MIXED

//
// Define this to override the limits on
// the core dump size.
//

#define COREDUMPS

//
// Upper limit of pre-allocated buffers
// for string parameters.
//

#define DEFAULT_STRING_LENGTH              256

//
// Maximum length of remote options data
// passed by peer proxy at startup.
//

#define DEFAULT_REMOTE_OPTIONS_LENGTH      512

//
// Maximum length of NX display string.
//

#define DEFAULT_DISPLAY_OPTIONS_LENGTH     1024

//
// Maximum number of cache file names
// to send from client to server side.
//

#define DEFAULT_REMOTE_CACHE_ENTRIES       100

//
// Maximum length of remote options
// string that can be received from
// the peer proxy.
//

#define MAXIMUM_REMOTE_OPTIONS_LENGTH      4096

//
// Maximum length of configuration file.
//

#define CONTROL_PARAMETERS_FILE_LENGTH     4096

//
// Lines expected in configuration file.
//

#define CONTROL_PARAMETERS_FILE_ENTRIES    78

//
// Macro is true if we determined our proxy
// mode.
//

#define WE_SET_PROXY_MODE         (control -> ProxyMode != proxy_undefined)

//
// Macro is true if our side is the one that
// should connect to remote.
//

#define WE_INITIATE_CONNECTION    (*connectHost != '\0')

//
// Is true if we must provide our credentials
// to the remote peer.
//

#define WE_PROVIDE_CREDENTIALS    (control -> ProxyMode == proxy_server)

//
// Is true if we listen for a local forwarder
// that will tunnel the traffic through a SSH
// or HTTP link.
//

#define WE_LISTEN_FORWARDER       (control -> ProxyMode == proxy_server && \
                                           listenPort != -1)

//
// Define FLUSH in Misc.h if you want
// immediate flush of log output.
//

ostream *logofs = NULL;

//
// Other stream destriptors used for
// logging.
//

ostream *statofs = NULL;
ostream *errofs  = NULL;

//
// Save standard error's rdbuf here
// and restore it when exiting.
//

static streambuf *errsbuf = NULL;

//
// Allow faults to be recovered by
// jumping back into the main loop.
//

jmp_buf context;

//
// Provide operational parameters.
//

Control *control = NULL;

//
// Collect and print statistics.
//

Statistics *statistics = NULL;

//
// Keep data for X11 authentication.
//

Auth *auth = NULL;

//
// This class makes the hard work.
//

Proxy *proxy = NULL;

//
// Used to handle memory-to-memory
// transport to the X agent.
//

Agent *agent = NULL;

//
// Signal handling functions.
//

void DisableSignals();
void EnableSignals();
void InstallSignals();

static void RestoreSignals();
static void HandleSignal(int signal);

//
// Signal handling utilities.
//

static void InstallSignal(int signal, int action);
static void RestoreSignal(int signal);

static int HandleChildren();

static int HandleChild(int child);
static int WaitChild(int child);
static int CheckChild(int pid, int status);

void CheckParent(char *name, char *type, int parent);

static int CheckAbort();

//
// Timer handling utilities.
//

void SetTimer(int timeout);
void ResetTimer();

static void HandleTimer(int signal);

//
// Return a string literal corresponding to
// the signal number.
//

const char *SignalLabel(int signal);

//
// Return a string literal corresponding to
// the requested flush type.
//

const char *FlushLabel(int type);

//
// Return a string literal corresponding to
// the requested policy type.
//

const char *PolicyLabel(int type);

//
// Return the string literal matching the
// given proxy control code.
//

const char *ControlLabel(int code);

//
// Handle the final close-down of the session.
//

static void HandleShutdown();

//
// Kill a running child.
//

static void KillProcess(int pid, const char *label, int signal, int wait);

//
// Cleanup functions.
//

void CleanupSockets();
void CleanupGlobal();

static void CleanupChildren();
static void CleanupLocal();

//
// Loop forever until the connections
// to the peer proxy is dropped.
//

static void WaitCleanup();

//
// Initialization functions.
//

static int InitBeforeNegotiation();
static int SetupProxyConnection();
static int InitAfterNegotiation();
static int ResetProxyConnection();
static int SetupProxyInstance();
static int SetupAuthInstance();
static int SetupAgentInstance();

static int SetupTCPSocket();
static int SetupUnixSocket();
static int SetupCupsSocket();
static int SetupKeybdSocket();
static int SetupSambaSocket();
static int SetupMediaSocket();
static int SetupHttpSocket();
static int SetupDisplaySocket(int &xServerAddrFamily, sockaddr *&xServerAddr,
                                  unsigned int &xServerAddrLength);

//
// Other convenience functions.
//

static int WaitForRemote(int portNum);
static int ConnectToRemote(const char *const hostName, int portNum);

static int SendProxyOptions(int fd);
static int SendProxyCaches(int fd);
static int ReadProxyVersion(int fd);
static int ReadProxyOptions(int fd);
static int ReadProxyCaches(int fd);
static int ReadForwarderVersion(int fd);
static int ReadForwarderOptions(int fd);

static int ReadRemoteData(int fd, char *buffer, int size, char stop);
static int WriteLocalData(int fd, const char *buffer, int size);

static void PrintVersionInfo();
static void PrintProcessInfo();
static void PrintConnectionInfo();
static void PrintResetInfo();
static void PrintUsageInfo(const char *option, const int error);

//
// This is not static to avoid a warning.
// Option was replaced with loading of NX
// library version.
//

void PrintCopyrightInfo();

static const char *GetOptions(const char *options);
static const char *GetArg(int &argi, int argc, const char **argv);
static int CheckArg(const char *type, const char *name, const char *value);
static int ParseArg(const char *type, const char *name, const char *value);
static int CheckSignal(int signal);

extern "C"
{
  int ParseCommandLineOptions(int argc, const char **argv);
  int ParseEnvironmentOptions(const char *env, int force);
  int ParseBindOptions(char **host, int *port);
}

static int ParseFileOptions(const char *file);
static int ParseRemoteOptions(char *opts);
static int ParseForwarderOptions(char *opts);

//
// These functions are used to parse literal
// values provided by the user and set the
// control parameters accordingly.
//

static int ParseLinkOption(const char *opt);
static int ParseLimitOption(const char *opt);
static int ParseCacheOption(const char *opt);
static int ParseShmemOption(const char *opt);
static int ParseImagesOption(const char *opt);
static int ParsePackOption(const char *opt);

//
// Set host and port where NX proxy is supposed
// to be listening in case such parameters are
// given on the command line.
//

static int ParseHostOption(const char *opt, char *host, int &port);

//
// Determine the interface where to listen for
// the remote proxy connection or the local
// forwarder.
//

static int ParseListenOption(int &interface);

//
// Translate pack method id in a literal.
//

static int ParsePackMethod(const int method, const int quality);

//
// Read control parameters from a simple
// configuration file. Agent will query
// some of these values at startup, after
// having successfully opened the display.
//

static int ParseControlParameters(const char *link);

//
// Try to increase the size of the allowed
// core dumps.
//

static int SetCore();

//
// Set the proxy mode to either client or
// server.
//

static int SetMode(int mode);

//
// Setup defaults used for log and statistics.
//

static int SetLog();

//
// Check if local and remote protocol versions
// are compatible and, eventually, downgrade
// local version to the minimum level that is
// known to work.
//

static int SetVersion();

//
// Setup the listening TCP ports used for the
// additional channels according to user's
// wishes.
//

static int SetPorts();

//
// Set the maximum number of open descriptors.
//

static int SetDescriptors();

//
// Set the path used for choosing the cache.
// It must be selected after determining the
// session type.
//

static int SetCaches();

//
// Initialize, one after the other, all the
// configuration parameters.
//

static int SetParameters();

//
// Initialize the specific parameters.
//

static int SetSession();
static int SetStorage();
static int SetShmem();
static int SetPack();
static int SetImages();
static int SetLimits();

//
// Set up control parameters according to
// expected link speed negotiated between
// proxies.
//

static int SetLink();

static int SetLinkModem();
static int SetLinkIsdn();
static int SetLinkAdsl();
static int SetLinkWan();
static int SetLinkLan();

//
// Adjust compression parameters.
//

static int SetCompression();

static int SetCompressionModem();
static int SetCompressionIsdn();
static int SetCompressionAdsl();
static int SetCompressionWan();
static int SetCompressionLan();

//
// Find out NX paths and files.
//

static char *GetRootPath();
static char *GetCachePath();
static char *GetImagesPath();
static char *GetSessionPath();
static char *GetLastCache(char *list, const char *path);

static int OpenInputFile(char *name, istream *&stream);
static int OpenOutputFile(char *name, ostream *&stream);
static int ReopenOutputFile(char *name, ostream *&stream, int limit);

//
// Perform operations on managed descriptors
// in the main loop.
//

static void handleCheckSessionInLoop();
static void handleCheckLimitsInLoop();

#if defined(TEST) || defined(INFO)
static void handleCheckSelectInLoop(int &setFDs, fd_set &readSet,
                                        fd_set &writeSet, T_timestamp selectTs);
static void handleCheckResultInLoop(int &resultFDs, int &errorFDs, int &setFDs, fd_set &readSet,
                                        fd_set &writeSet, struct timeval &selectTs,
                                            struct timeval &startTs);
static void handleCheckStateInLoop(int &setFDs);
#endif

static void handleCheckSessionInConnect();

static inline void handleSetReadInLoop(fd_set &readSet, int &setFDs, struct timeval &selectTs);
static inline void handleSetWriteInLoop(fd_set &writeSet, int &setFDs, struct timeval &selectTs);
static inline void handleSetListenersInLoop(fd_set &writeSet, int &setFDs);
static inline void handleSetAgentInLoop(int &setFDs, fd_set &readSet, fd_set &writeSet,
                                            struct timeval &selectTs);

static void handleAlertInLoop();
static void handleStatisticsInLoop();
static void handleResetInLoop();
static void handlePingInLoop(int &diffTs);

static inline void handleChannelEventsInLoop();

static void handleAcceptTcpConnectionInLoop(int &fd);
static void handleAcceptUnixConnectionInLoop(int &fd);
static void handleAcceptCupsConnectionInLoop(int &fd);
static void handleAcceptKeybdConnectionInLoop(int &fd);
static void handleAcceptSambaConnectionInLoop(int &fd);
static void handleAcceptMediaConnectionInLoop(int &fd);
static void handleAcceptHttpConnectionInLoop(int &fd);

static inline void handleAgentInLoop(int &resultFDs, int &errorFDs, int &setFDs, fd_set &readSet,
                                         fd_set &writeSet, struct timeval &selectTs);
static inline void handleAgentLateInLoop(int &resultFDs, int &errorFDs, int &setFDs, fd_set &readSet,
                                             fd_set &writeSet, struct timeval &selectTs);

static inline void handleReadableInLoop(int &resultFDs, fd_set &readSet);
static inline void handleWritableInLoop(int &resultFDs, fd_set &writeSet);

//
// Manage the proxy link during the negotiation
// phase.
//

static void handleNegotiationInLoop(int &setFDs, fd_set &readSet,
                                        fd_set &writeSet, T_timestamp &selectTs);

//
// Monitor the size of the log file.
//

static void handleLogReopenInLoop(T_timestamp &logsTs, T_timestamp &nowTs);

//
// Root of directory structure to be created by proxy.
//

static char rootDir[DEFAULT_STRING_LENGTH] = { 0 };

//
// Root of statistics and log files to be created by proxy.
//

static char sessionDir[DEFAULT_STRING_LENGTH] = { 0 };

//
// Log files for errors and statistics. Error log is
// the place where we print also debug informations.
// Both files are closed, deleted and reopened as
// their size exceed the limit set in control class.
// The session log is not reopened, as it is used by
// the NX client and server to track the advance of
// the session.
//

static char logFileName[DEFAULT_STRING_LENGTH]     = { 0 };
static char statFileName[DEFAULT_STRING_LENGTH]    = { 0 };
static char sessionFileName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal representing selected link speed
// parameter. Value is translated in control values
// used by proxies to stay synchronized.
//

static char linkSpeedName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal representing selected
// cache size.
//

static char cacheSizeName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal representing selected
// shared memory segment size.
//

static char shmemSizeName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal of images cache size.
//

static char imagesSizeName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal for bandwidth limit.
//

static char bitrateLimitName[DEFAULT_STRING_LENGTH] = { 0 };

//
// String literal for image packing method.
//

static char packMethodName[DEFAULT_STRING_LENGTH] = { 0 };

//
// Its corresponding value from NXpack.h.
//

static int packMethod  = -1;
static int packQuality = -1;

//
// String literal for session type. Persistent caches
// are searched in directory whose name matches this
// parameter.
//

static char sessionType[DEFAULT_STRING_LENGTH] = { 0 };

//
// Unique id assigned to session. It is used as
// name of directory where all files are placed.
//

static char sessionId[DEFAULT_STRING_LENGTH] = { 0 };

//
// Set if we already parsed the options.
//

static int parsedOptions = 0;
static int parsedCommand = 0;

//
// Buffer data received from the remote proxy at
// session negotiation.
//

static char remoteData[MAXIMUM_REMOTE_OPTIONS_LENGTH] = { 0 };
static int  remotePosition = 0;

//
// Main loop file descriptors.
//

static int tcpFD   = -1;
static int unixFD  = -1;
static int cupsFD  = -1;
static int keybdFD = -1;
static int sambaFD = -1;
static int mediaFD = -1;
static int httpFD  = -1;
static int proxyFD = -1;

//
// Used for internal communication
// with the X agent.
//

static int agentFD[2] = { -1, -1 };

//
// Flags determining which protocols and
// ports are forwarded.
//

int useUnixSocket = 1;

static int useTCPSocket   = 1;
static int useCupsSocket  = 0;
static int useKeybdSocket = 0;
static int useSambaSocket = 0;
static int useMediaSocket = 0;
static int useHttpSocket  = 0;
static int useAgentSocket = 0;

//
// Set by user if he/she wants to modify
// the default TCP_NODELAY option as set
// in control.
//

static int useNoDelay = -1;

//
// Set if user wants to override default
// flush timeout set according to link.
//

static int useFlush = -1;

//
// Set if user wants to hide the RENDER
// extension or wants to short-circuit
// some simple replies at client side.
//

static int useRender = -1;
static int useTaint  = -1;

//
// Name of Unix socket created by the client proxy to
// accept client connections. File must be unlinked
// by cleanup function.
//

static char unixSocketName[DEFAULT_STRING_LENGTH] = { 0 };

//
// Other parameters.
//

static char connectHost[DEFAULT_STRING_LENGTH] = { 0 };
static char acceptHost[DEFAULT_STRING_LENGTH]  = { 0 };
static char listenHost[DEFAULT_STRING_LENGTH]  = { 0 };
static char displayHost[DEFAULT_STRING_LENGTH] = { 0 };
static char authCookie[DEFAULT_STRING_LENGTH]  = { 0 };

static int proxyPort = DEFAULT_NX_PROXY_PORT;
static int xPort     = DEFAULT_NX_X_PORT;

//
// Used to setup the connection the real
// X display socket.
//

static int xServerAddrFamily          = -1;
static sockaddr *xServerAddr          = NULL;
static unsigned int xServerAddrLength = 0;

//
// The port where the local proxy will await
// the peer connection or where the remote
// proxy will be contacted.
//

static int listenPort  = -1;
static int connectPort = -1;

//
// Helper channels are disabled by default.
//

static int cupsPort  = -1;
static int keybdPort = -1;
static int sambaPort = -1;
static int mediaPort = -1;
static int httpPort  = -1;

//
// Host and port where the existing proxy
// is running.
//

static char bindHost[DEFAULT_STRING_LENGTH] = { 0 };
static int  bindPort = -1;

//
// State variables shared between the init
// function and the main loop.
//

T_timestamp startTs;
T_timestamp logsTs;
T_timestamp nowTs;

int diffTs;

//
// This is set to the main proxy process id.
// 

int lastProxy = 0;

//
// Set to last dialog process launched by proxy.
// 

int lastDialog = 0;

//
// Set to watchdog process launched by proxy.
// 

int lastWatchdog = 0;

//
// Set if a cache house-keeper process is run.
// 

int lastKeeper = 0;

//
// Set if shutdown was requested through a signal.
//

static int lastKill = 0;

//
// This is set to the code and local flag of the
// last requested alert.
// 

static struct
{
  int code;
  int local;

} lastAlert;

//
// Manage the current signal masks.
//

static struct
{
  sigset_t saved;

  int blocked;
  int installed;

  int enabled[32];
  int forward[32];

  struct sigaction action[32];

} lastMasks;

//
// Manage the current timer.
//

static struct
{
  struct sigaction action;
  struct itimerval value;
  struct timeval   start;
  struct timeval   next;

} lastTimer;

//
// This is set to last signal received in handler.
//

static int lastSignal = 0;

//
// Set to the last time bytes readable were queried
// by the agent.
//

static T_timestamp lastReadableTs = nullTimestamp();

//
// Here are interfaces declared in NX.h.
//

int NXTransProxy(int fd, int mode, const char* options)
{
  //
  // Let the log temporarily go to the standard
  // error. Be also sure we have a jump context,
  // in the case any subsequent operation will
  // cause a cleanup.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (setjmp(context) == 1)
  {
    #ifdef TEST
    *logofs << "NXTransProxy: Out of the long jump with pid '"
            << lastProxy << "'.\n" << logofs_flush;
    #endif
    
    return -1;
  }

  //
  // Check if have already performed a parsing of
  // parameters, as in the case we are running as
  // a stand-alone process. If needed create the
  // parameters repository
  //

  if (control == NULL)
  {
    control = new Control();
  }

  lastProxy = getpid();

  #ifdef TEST
  *logofs << "NXTransProxy: Main process started with pid '"
          << lastProxy << "'.\n" << logofs_flush;
  #endif

  SetMode(mode);

  if (mode == NX_MODE_CLIENT)
  {
    if (fd != NX_FD_ANY)
    {
      #ifdef TEST

      *logofs << "NXTransProxy: Agent descriptor for X client connections is FD#"
              << fd << ".\n" << logofs_flush;

      *logofs << "NXTransProxy: Disabling listening on further X client connections.\n"
              << logofs_flush;

      #endif

      useTCPSocket   = 0;
      useUnixSocket  = 0;
      useAgentSocket = 1;

      agentFD[1] = fd;
    }
  }
  else if (mode == NX_MODE_SERVER)
  {
    if (fd != NX_FD_ANY)
    {
      #ifdef TEST
      *logofs << "NXTransProxy: PANIC! Agent descriptor for X server connections "
              << "not supported yet.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Agent descriptor for X server connections "
           << "not supported yet.\n";

      return -1;
    }
  }

  const char *env = GetOptions(options);

  if (ParseEnvironmentOptions(env, 0) < 0)
  {
    cerr << "Error" << ": Parsing of proxy options failed.\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "NXTransProxy: Going to run the NX transport loop.\n"
          << logofs_flush;
  #endif

  WaitCleanup();

  //
  // This function should never return.
  //

  exit(0);
}

void NXTransExit(int code)
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  static int recurse;

  if (++recurse > 1)
  {
    #ifdef TEST
    *logofs << "NXTransExit: Aborting parent due to recursion "
            << "through exit.\n" << logofs_flush;
    #endif

    abort();
  }

  #ifdef TEST
  *logofs << "NXTransExit: Parent called exit with code '"
          << code << "'.\n" << logofs_flush;
  #endif

  exit(code);
}

int NXTransParseCommandLine(int argc, const char **argv)
{
  return ParseCommandLineOptions(argc, argv);
}

int NXTransParseEnvironment(const char *env, int force)
{
  return ParseEnvironmentOptions(env, force);
}

void NXTransCleanup()
{
  HandleCleanup();
}

//
// Check the parameters for subsequent
// initialization of the NX transport.
//

int NXTransCreate(int fd, int mode, const char* options)
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  //
  // Be sure we have a jump context, in the
  // case a subsequent operation will cause
  // a cleanup.
  //

  if (setjmp(context) == 1)
  {
    return -1;
  }

  //
  // Create the parameters repository
  //

  if (control != NULL)
  {
    #ifdef PANIC
    *logofs << "NXTransCreate: PANIC! The NX transport seems "
            << "to be already running.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": The NX transport seems "
         << "to be already running.\n";

    return -1;
  }

  control = new Control();

  if (control == NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating the NX transport.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating the NX transport.\n";

    return -1;
  }

  lastProxy = getpid();

  #ifdef TEST
  *logofs << "NXTransCreate: Caller process running with pid '"
          << lastProxy << "'.\n" << logofs_flush;
  #endif

  //
  // Set the local proxy mode an parse the
  // display NX options.
  //

  SetMode(mode);

  const char *env = GetOptions(options);

  if (ParseEnvironmentOptions(env, 0) < 0)
  {
    cerr << "Error" << ": Parsing of NX options failed.\n";

    HandleCleanup();
  }

  //
  // Use the provided descriptor.
  //

  proxyFD = fd;

  #ifdef TEST
  *logofs << "NXTransCreate: Called with NX proxy descriptor '"
          << proxyFD << "'.\n" << logofs_flush;
  #endif

  #ifdef TEST
  *logofs << "NXTransCreate: Creation of the NX transport completed.\n"
          << logofs_flush;
  #endif

  return 1;
}

//
// Tell the proxy to use the descriptor as the internal
// connection to the X client side NX agent. This will
// have the side effect of disabling listening for add-
// itional X client connections.
//

int NXTransAgent(int fd[2])
{
  //
  // Be sure we have a jump context, in the
  // case a subsequent operation will cause
  // a cleanup.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (setjmp(context) == 1)
  {
    return -1;
  }

  if (control == NULL)
  {
    cerr << "Error" << ": Can't set the NX agent without a NX transport.\n";

    return -1;
  }
  else if (control -> ProxyMode != proxy_client)
  {
    #ifdef PANIC
    *logofs << "NXTransAgent: Invalid mode while setting the NX agent.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Invalid mode while setting the NX agent.\n\n";

    return -1;
  }

  useTCPSocket   = 0;
  useUnixSocket  = 0;
  useAgentSocket = 1;

  agentFD[0] = fd[0];
  agentFD[1] = fd[1];

  #ifdef TEST

  *logofs << "NXTransAgent: Internal descriptors for agent are FD#"
          << agentFD[0] << " and FD#" << agentFD[1] << ".\n"
          << logofs_flush;

  *logofs << "NXTransAgent: Disabling listening for further X client "
          << "connections.\n" << logofs_flush;

  #endif

  agent = new Agent(agentFD);

  if (agent == NULL || agent -> isValid() != 1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating the NX memory transport .\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating the NX memory transport.\n";

    HandleCleanup();
  }

  #ifdef TEST
  *logofs << "NXTransAgent: Enabling memory-to-memory transport.\n"
          << logofs_flush;
  #endif

  return 1;
}

//
// Close down the transport and free the
// allocated NX resources.
//

int NXTransDestroy(void)
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (control != NULL)
  {
    //
    // Shut down the proxy descriptor
    // and wait the cleanup to complete.
    //

    if (proxy != NULL)
    {
      #ifdef TEST
      *logofs << "NXTransDestroy: Closing all the X connections.\n"
              << logofs_flush;
      #endif

      proxy -> handleCloseAllXConnections();
    }

    #ifdef TEST
    *logofs << "NXTransDestroy: Waiting for the NX transport to terminate.\n"
            << logofs_flush;
    #endif

    WaitCleanup();
  }
  #ifdef TEST
  else
  {
    *logofs << "NXTransDestroy: The NX transport is not running.\n"
            << logofs_flush;
  }
  #endif

  return 1;
}

//
// Restore the proxy connection in case
// of failure.
//

int NXTransRestore()
{
  //
  // This is presently not supported.
  //

  return -1;

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  #ifdef TEST
  *logofs << "NXTransRestore: Going to restore the NX transport.\n"
          << logofs_flush;
  #endif

  #ifdef TEST
  *logofs << "NXTransRestore: Showing the broken connection alert.\n"
          << logofs_flush;
  #endif

  if (control -> ProxyMode == proxy_client)
  {
    HandleAlert(RESTART_DEAD_PROXY_CONNECTION_CLIENT_ALERT, 1);
  }
  else
  {
    HandleAlert(RESTART_DEAD_PROXY_CONNECTION_SERVER_ALERT, 1);
  }

  handleAlertInLoop();

  #ifdef TEST
  *logofs << "NXTransRestore: Going to restore proxy connection.\n"
          << logofs_flush;
  #endif

  ResetProxyConnection();

  return 1;
}

//
// Assume that the NX transport is valid
// as long as the control class has not
// been destroyed.
//

int NXTransRunning()
{
  return (control != NULL);
}

int NXTransContinue(struct timeval *selectTs)
{
  if (control != NULL)
  {
    fd_set readSet;
    fd_set writeSet;

    int setFDs;
    int errorFDs;
    int resultFDs;

    //
    // Use empty masks and only get the
    // descriptors set by the proxy.
    //

    setFDs = 0;

    FD_ZERO(&readSet);
    FD_ZERO(&writeSet);

    //
    // Run a new loop. If the transport
    // is gone avoid sleeping until the
    // timeout.
    //

    if (NXTransPrepare(&setFDs, &readSet, &writeSet, selectTs) != 0)
    {
      NXTransSelect(&resultFDs, &errorFDs, &setFDs, &readSet, &writeSet, selectTs);

      NXTransExecute(&resultFDs, &errorFDs, &setFDs, &readSet, &writeSet, selectTs);
    }
  }

  return (control != NULL);
}

int NXTransSignal(int signal, int action)
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (control == NULL)
  {
    return 0;
  }

  if (action == NX_SIGNAL_RAISE)
  {
    #ifdef TEST
    *logofs << "NXTransSignal: Raising signal '" << SignalLabel(signal)
            << "' in the proxy handler.\n" << logofs_flush;
    #endif

    HandleSignal(signal);

    return 1;
  }
  else if (signal == NX_SIGNAL_ANY)
  {
    #ifdef TEST
    *logofs << "NXTransSignal: Setting action of all signals to '"
            << action << "'.\n" << logofs_flush;
    #endif

    for (int i = 0; i < 32; i++)
    {
      if (CheckSignal(i) == 1)
      {
        NXTransSignal(i, action);
      }
    }

    return 1;
  }
  else if (CheckSignal(signal) == 1)
  {
    #ifdef TEST
    *logofs << "NXTransSignal: Setting action of signal '"
            << SignalLabel(signal) << "' to '" << action
            << "'.\n" << logofs_flush;
    #endif

    if (action == NX_SIGNAL_ENABLE ||
            action == NX_SIGNAL_FORWARD)
    {
      InstallSignal(signal, action);

      return 1;
    }
    else if (action == NX_SIGNAL_DISABLE)
    {
      RestoreSignal(signal);

      return 1;
    }
  }

  #ifdef WARNING
  *logofs << "NXTransSignal: WARNING! Unable to perform action '"
          << action << "' on signal '" << SignalLabel(signal)
          << "'.\n" << logofs_flush;
  #endif

  cerr << "Warning" << ": Unable to perform action '" << action
       << "' on signal '" << SignalLabel(signal)
       << "'.\n";

  return -1;
}

int NXTransCongestion()
{
  if (control != NULL && proxy != NULL)
  {
    return (proxy -> getCongestion(proxyFD));
  }

  return 0;
}

int NXTransRead(int fd, char *data, int size)
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (control != NULL && agent != NULL &&
          fd == agentFD[0])
  {
    #ifdef DUMP
    *logofs << "NXTransRead: Dequeuing " << size << " bytes "
            << "from FD#" << agentFD[0] << ".\n" << logofs_flush;
    #endif

    int result = agent -> dequeueData(data, size);

    #ifdef DUMP

    if (result < 0 && EGET() == EAGAIN)
    {
      *logofs << "NXTransRead: WARNING! Dequeuing from FD#"
              << agentFD[0] << " would block.\n" << logofs_flush;
    }
    else
    {
      *logofs << "NXTransRead: Dequeued " << result << " bytes "
              << "to FD#" << agentFD[0] << ".\n" << logofs_flush;
    }

    #endif

    return result;
  }
  else
  {
    #ifdef DUMP
    *logofs << "NXTransRead: Reading " << size << " bytes "
            << "from FD#" << fd << ".\n" << logofs_flush;
    #endif

    return read(fd, data, size);
  }
}

int NXTransReadVector(int fd, struct iovec *iovdata, int iovsize)
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (control != NULL && agent != NULL &&
          fd == agentFD[0])
  {
    #if defined(DUMP) || defined(INFO)

    if (control -> ProxyStage == stage_operational &&
            agent -> localReadable() > 0)
    {
      *logofs << "NXTransReadVector: WARNING! Agent has data readable.\n"
              << logofs_flush;
    }

    #endif

    char *base;

    int length;
    int result;

    struct iovec *vector = iovdata;
    int count = iovsize;

    ESET(0);

    int i = 0;
    int total = 0;

    for (;  i < count;  i++, vector++)
    {
      length = vector -> iov_len;
      base = (char *) vector -> iov_base;

      while (length > 0)
      {
        #ifdef DUMP
        *logofs << "NXTransReadVector: Dequeuing " << length
                << " bytes " << "from FD#" << agentFD[0] << ".\n"
                << logofs_flush;
        #endif

        result = agent -> dequeueData(base, length);

        #ifdef DUMP

        if (result < 0 && EGET() == EAGAIN)
        {
          *logofs << "NXTransReadVector: WARNING! Dequeuing from FD#"
                  << agentFD[0] << " would block.\n" << logofs_flush;
        }
        else
        {
          *logofs << "NXTransReadVector: Dequeued " << result
                  << " bytes " << "from FD#" << agentFD[0] << ".\n"
                  << logofs_flush;
        }

        #endif

        if (result < 0 && total == 0)
        {
          return result;
        }
        else if (result <= 0)
        {
          return total;
        }

        ESET(0);

        length -= result;
        total  += result;
        base   += result;
      }
    }

    return total;
  }
  else
  {
    #ifdef DUMP
    *logofs << "NXTransReadVector: Reading vector with "
            << iovsize << " elements from FD#" << fd << ".\n"
            << logofs_flush;
    #endif

    return readv(fd, iovdata, iovsize);
  }
}

int NXTransReadable(int fd, long int *readable)
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  int result = 0;

  if (control == NULL || agent == NULL ||
          fd != agentFD[0])
  {
    result = ioctl(fd, FIONREAD, (void *) readable);

    #ifdef DUMP
    *logofs << "NXTransReadable: Returning " << *readable
            << " bytes as readable from FD#" << fd
            << ".\n" << logofs_flush;
    #endif

    return result;
  }

  //
  // The client might have enqueued data to our side and
  // is now checking for the available events. As the
  // _XEventsQueued() function does not call _XSelect(),
  // we must handle the data or will spin until the next
  // request is written by the client to the transport.
  //
  // Optimize the function to run a new loop only when
  // there is no data readily available and some more
  // can be read from the remote proxy.
  //

  *readable = (long int) agent -> dequeuableData();

  if (*readable == 0 && proxy != NULL && 
          proxy -> canRead() == 1)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "NXTransReadable: WARNING! Trying to "
            << "read to generate new agent data.\n"
            << logofs_flush;
    #endif

    proxy -> setPending(proxyFD);

    //
    // Set the context as the function
    // can cause a cleanup.
    //

    if (setjmp(context) == 1)
    {
      return -1;
    }

    if (proxy -> handleRead() < 0)
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "NXTransReadable: Failure reading "
              << "messages from proxy FD#" << proxyFD
              << ".\n" << logofs_flush;
      #endif

      HandleShutdown();
    }

    //
    // Agent may be gone.
    //

    if (agent != NULL)
    {
      *readable = (long int) agent -> dequeuableData();
    }

    result = 0;
  }

  #ifdef DUMP
  *logofs << "NXTransReadable: Returning " << *readable
          << " bytes as readable from FD#" << fd
          << ".\n" << logofs_flush;
  #endif

  return result;
}

int NXTransWrite(int fd, char *data, int size)
{
  //
  // Be sure we have a valid log file.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (control != NULL && agent != NULL &&
          fd == agentFD[0])
  {
    //
    // Among the other alternatives, this strategy may
    // be worth further investigation. Under this pro-
    // vision, we postpone adding data to the channel
    // as long as there might be congestion events co-
    // ming from the remote.
    //
    // if (proxy != NULL && (proxy -> canRead() ||
    //         proxy -> canRead(agentFD[1]) == 0))
    // {
    //   ...
    // }
    //

    if (proxy != NULL && proxy -> canRead(agentFD[1]) == 0)
    {
      #if defined(DUMP) || defined(INFO)
      *logofs << "NXTransWrite: WARNING! Delayed enqueuing to FD#"
              << agentFD[0] << " with proxy unable to read.\n"
              << logofs_flush;
      #endif

      ESET(EAGAIN);

      return -1;
    }

    #ifdef DUMP
    *logofs << "NXTransWrite: Enqueuing " << size << " bytes "
            << "to FD#" << agentFD[0] << ".\n" << logofs_flush;
    #endif

    int result = agent -> enqueueData(data, size);

    #ifdef DUMP

    if (result < 0 && EGET() == EAGAIN)
    {
      *logofs << "NXTransWrite: WARNING! Enqueuing to FD#"
              << agentFD[0] << " would block.\n" << logofs_flush;
    }
    else
    {
      *logofs << "NXTransWrite: Enqueued " << result << " bytes "
              << "to FD#" << agentFD[0] << ".\n" << logofs_flush;
    }

    #endif

    if (result > 0 && proxy != NULL)
    {
      //
      // Set the context as the function
      // can cause a cleanup.
      //

      if (setjmp(context) == 1)
      {
        return -1;
      }

      proxy -> setPending(agentFD[1]);

      proxy -> handleRead(agentFD[1]);
    }

    return result;
  }
  else
  {
    #ifdef DUMP
    *logofs << "NXTransWrite: Writing " << size << " bytes "
            << "to FD#" << fd << ".\n" << logofs_flush;
    #endif

    return write(fd, data, size);
  }
}

int NXTransWriteVector(int fd, struct iovec *iovdata, int iovsize)
{
  //
  // Be sure we have a valid log file and a
  // jump context because we will later call
  // functions that can perform a cleanup.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  int result = 0;

  if (control != NULL && agent != NULL &&
          fd == agentFD[0])
  {
    //
    // See the comment in NXTransWrite().
    //

    if (proxy != NULL && proxy -> canRead(agentFD[1]) == 0)
    {
      #if defined(DUMP) || defined(INFO)
      *logofs << "NXTransWriteVector: WARNING! Delayed enqueuing to FD#"
              << agentFD[0] << " with proxy unable to read.\n"
              << logofs_flush;
      #endif

      ESET(EAGAIN);

      return -1;
    }

    //
    // Set the context as the function
    // can cause a cleanup.
    //

    if (setjmp(context) == 1)
    {
      return -1;
    }

    char *base;

    int length;

    struct iovec *vector = iovdata;
    int count = iovsize;

    ESET(0);

    int i = 0;
    int total = 0;

    for (;  i < count;  i++, vector++)
    {
      length = vector -> iov_len;
      base = (char *) vector -> iov_base;

      while (length > 0)
      {
        #ifdef DUMP
        *logofs << "NXTransWriteVector: Enqueuing " << length
                << " bytes " << "to FD#" << agentFD[0] << ".\n"
                << logofs_flush;
        #endif

        result = agent -> enqueueData(base, length);

        #ifdef DUMP

        if (result < 0 && EGET() == EAGAIN)
        {
          *logofs << "NXTransWriteVector: WARNING! Enqueuing to FD#"
                  << agentFD[0] << " would block.\n" << logofs_flush;
        }
        else
        {
          *logofs << "NXTransWriteVector: Enqueued " << result
                  << " bytes " << "to FD#" << agentFD[0] << ".\n"
                  << logofs_flush;
        }

        #endif

        if (result < 0 && total == 0)
        {
          goto NXTransWriteVectorResult;
        }
        else if (result <= 0)
        {
          result = total;

          goto NXTransWriteVectorResult;
        }

        ESET(0);

        length -= result;
        total  += result;
        base   += result;
      }
    }

    result = total;

    goto NXTransWriteVectorResult;
  }
  else
  {
    #ifdef DUMP
    *logofs << "NXTransWriteVector: Writing vector with "
            << iovsize << " elements to FD#" << fd << ".\n"
            << logofs_flush;
    #endif

    return writev(fd, iovdata, iovsize);
  }

NXTransWriteVectorResult:

  if (result > 0 && proxy != NULL)
  {
    proxy -> setPending(agentFD[1]);

    proxy -> handleRead(agentFD[1]);
  }

  return result;
}

int NXTransPolicy(int type)
{
  if (control != NULL)
  {
    #if defined(DUMP) || defined(INFO)
    *logofs << "NXTransPolicy: Setting flush policy on "
            << "proxy FD#" << proxyFD << " to '"
            << PolicyLabel(type == NX_POLICY_DEFERRED ?
                   policy_deferred : policy_immediate)
            << "'.\n" << logofs_flush;
    #endif

    control -> FlushPolicy = (type == NX_POLICY_DEFERRED ?
                                  policy_deferred : policy_immediate);

    return 1;
  }

  return 0;
}

int NXTransFlushable()
{
  if (proxy != NULL)
  {
    #if defined(DUMP) || defined(INFO)
    *logofs << "NXTransFlushable: Returning " << proxy ->
               getFlushable(proxyFD) << " as bytes flushable on "
            << "proxy FD#" << proxyFD << ".\n"
            << logofs_flush;
    #endif

    return proxy -> getFlushable(proxyFD);
  }

  return 0;
}

int NXTransFlush(int sync)
{
  if (proxy != NULL)
  {
    if (sync == 1)
    {
      #if defined(DUMP) || defined(INFO)
      *logofs << "NXTransFlush: Requesting a synchronization on "
              << "proxy FD#" << proxyFD << ".\n"
              << logofs_flush;
      #endif

      return proxy -> handleFlush(flush_if_sync);
    }
    else if (proxy -> canFlush() == 1)
    {
      #if defined(DUMP) || defined(INFO)
      *logofs << "NXTransFlush: Requesting an immediate flush on "
              << "proxy FD#" << proxyFD << ".\n"
              << logofs_flush;
      #endif

      return proxy -> handleFlush(flush_if_any);
    }
    #if defined(DUMP) || defined(INFO)
    else
    {
      *logofs << "NXTransFlush: Nothing to flush on proxy FD#"
              << proxyFD << ".\n" << logofs_flush;
    }
    #endif
  }

  return 0;
}

int NXTransAlert(int code, int local)
{
  if (proxy != NULL)
  {
    #if defined(DUMP) || defined(INFO)
    *logofs << "NXTransAlert: Requesting a NX dialog with code "
            << code << " and local " << local << ".\n"
            << logofs_flush;
    #endif

    if (local == 0)
    {
      //
      // Set the context as the function
      // can cause a cleanup.
      //

      if (setjmp(context) == 1)
      {
        return -1;
      }

      proxy -> handleAlert(code);
    }
    else
    {
      //
      // Show the alert at the next loop.
      //

      HandleAlert(code, local);
    }
  }
  #if defined(DUMP) || defined(INFO)
  else
  {
    if (logofs == NULL)
    {
      logofs = &cerr;
    }

    *logofs << "NXTransAlert: Can't request an alert without "
            << "a valid NX transport.\n" << logofs_flush;
  }
  #endif

  return 0;
}

//
// Prepare the file sets and the timeout
// for a later execution of the select().
//

int NXTransPrepare(int *setFDs, fd_set *readSet,
                       fd_set *writeSet, struct timeval *selectTs)
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  //
  // Control is NULL if the NX transport was
  // reset or was never created. If control
  // is valid then prepare to jump back when
  // the transport is destroyed.
  //

  if (control == NULL || setjmp(context) == 1)
  {
    return 0;
  }

  #if defined(TEST) || defined(INFO)
  *logofs << "\nNXTransPrepare: Going to prepare the NX transport.\n"
          << logofs_flush;
  #endif

  if (control -> ProxyStage != stage_operational)
  {
    handleNegotiationInLoop(*setFDs, *readSet, *writeSet, *selectTs);
  }
  else
  {
    #if defined(TEST) || defined(INFO)

    if (isTimestamp(*selectTs) == 0)
    {
      *logofs << "Loop: WARNING! Preparing the select with requested "
              << "timeout of " << selectTs -> tv_sec << " S and "
              << (double) selectTs -> tv_usec / 1000 << " Ms.\n"
              << logofs_flush;
    }
    else
    {
      *logofs << "Loop: Preparing the select with requested "
              << "timeout of " << selectTs -> tv_sec << " S and "
              << (double) selectTs -> tv_usec / 1000 << " Ms.\n"
              << logofs_flush;
    }

    #endif

    //
    // Set descriptors of listening sockets.
    //

    if (control -> ProxyMode == proxy_client)
    {
      handleSetListenersInLoop(*readSet, *setFDs);
    }

    //
    // Set descriptors of both proxy and X
    // connections.
    //

    handleSetReadInLoop(*readSet, *setFDs, *selectTs);

    //
    // Find out which file descriptors have
    // data to write.
    //

    handleSetWriteInLoop(*writeSet, *setFDs, *selectTs);
  }

  //
  // Prepare the masks for handling the memory-
  // to-memory transport. This is required even
  // during session negotiation.
  //

  if (agent != NULL)
  {
    handleSetAgentInLoop(*setFDs, *readSet, *writeSet, *selectTs);
  }

  //
  // Register time spent handling messages.
  //

  nowTs = getTimestamp();

  diffTs = diffTimestamp(startTs, nowTs);

  #ifdef TEST
  *logofs << "Loop: Difference from loop timestamp is "
          << diffTs << " Ms.\n"
          << logofs_flush;
  #endif

  //
  // TODO: Should add read time in two parts
  // otherwise limits are checked before the
  // counters are updated with time spent in
  // the last loop.
  //

  if (control -> ProxyStage == stage_operational)
  {
    control -> addReadTime(diffTs);

    if (control -> CollectStatistics)
    {
      statistics -> addReadTime(diffTs);
    }
  }

  startTs = nowTs;

  #ifdef DEBUG
  *logofs << "Loop: New timestamp is " << strMsTimestamp(startTs)
          << ".\n" << logofs_flush;
  #endif

  return 1;
}

//
// Let's say that it calls select() to find out
// if any of the handled descriptors has data,
// but things are a bit more complex than that.
//

int NXTransSelect(int *resultFDs, int *errorFDs, int *setFDs, fd_set *readSet,
                      fd_set *writeSet, struct timeval *selectTs)
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  //
  // Control is NULL if the NX transport was
  // reset or never created. If control is
  // valid then prepare for jumping back in
  // the case of an error.
  //

  if (control == NULL || setjmp(context) == 1)
  {
    *resultFDs = select(*setFDs, readSet, writeSet, NULL, selectTs);

    *errorFDs = errno;

    return 0;
  }

  #if defined(TEST) || defined(INFO)
  *logofs << "\nNXTransSelect: Going to select the NX descriptors.\n"
          << logofs_flush;
  #endif

  #if defined(TEST) || defined(INFO)

  handleCheckSelectInLoop(*setFDs, *readSet, *writeSet, *selectTs);

  #endif

  //
  // We can skip the select if any of the agent
  // descriptors are ready.
  //

  if (isTimestamp(*selectTs) == 0 &&
          agent != NULL && agent -> isSelected())
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Skipping select at " << strMsTimestamp()
            << " with timeout of " << selectTs -> tv_sec
            << " S and " << (double) selectTs -> tv_usec / 1000
            << " Ms.\n" << logofs_flush;
    #endif

    memset(readSet,  0, sizeof(fd_set));
    memset(writeSet, 0, sizeof(fd_set));

    ESET(0);

    *resultFDs = 0;
    *errorFDs  = 0;
  }
  else
  {
    #if defined(TEST) || defined(INFO)

    if (isTimestamp(*selectTs) == 0)
    {
      *logofs << "Loop: WARNING! Executing the select with requested "
              << "timeout of " << selectTs -> tv_sec << " S and "
              << (double) selectTs -> tv_usec / 1000 << " Ms.\n"
              << logofs_flush;
    }

    #endif

    //
    // Wait for selected sockets or timeout.
    // PGID_USE_PID is specific to HP-UX 10.X.
    //

    ESET(0);

    *resultFDs = select(*setFDs, readSet, writeSet, NULL, selectTs);

    *errorFDs = EGET();
  }

  //
  // Check the result of select.
  //

  #if defined(TEST) || defined(INFO)

  handleCheckResultInLoop(*resultFDs, *errorFDs, *setFDs, *readSet, *writeSet, *selectTs, startTs);

  #endif

  //
  // Get time spent in select. Unfortunately accouting of
  // time statistics was initially done in milliseconds
  // and I never had time to switch to microseconds. This
  // is a real problem on fast machines where each run is
  // unlikely to take more than 500 us. So consider that
  // the results can be inaccurate.
  //

  nowTs = getTimestamp();

  diffTs = diffTimestamp(startTs, nowTs);

  #ifdef TEST
  *logofs << "Loop: Out of select after " << diffTs << " Ms "
          << "at " << strMsTimestamp(nowTs) << " with result "
          << *resultFDs << ".\n" << logofs_flush;
  #endif

  startTs = nowTs;

  #ifdef DEBUG
  *logofs << "Loop: New timestamp is " << strMsTimestamp(startTs)
          << ".\n" << logofs_flush;
  #endif

  if (control -> ProxyStage == stage_operational)
  {
    control -> addIdleTime(diffTs);

    if (control -> CollectStatistics)
    {
      statistics -> addIdleTime(diffTs);
    }
  }

  if (*resultFDs < 0)
  {
    //
    // Check if the call was interrupted or if any of the
    // managed descriptors has become invalid. This can
    // happen to the X11 code, before the descriptor is
    // removed from the managed set.
    //

    #ifdef __sun

    if (*errorFDs == EINTR || *errorFDs == EBADF ||
            *errorFDs == EINVAL)

    #else

    if (*errorFDs == EINTR || *errorFDs == EBADF)

    #endif

    {
      #ifdef TEST

      if (*errorFDs == EINTR)
      {
        *logofs << "Loop: Select failed due to EINTR error.\n"
                << logofs_flush;
      }
      else
      {
        *logofs << "Loop: WARNING! Call to select failed. Error is "
                << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
      }

      #endif
    }
    else
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Call to select failed. Error is "
              << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Call to select failed. Error is "
           << EGET() << " '" << ESTR() << "'.\n";

      HandleCleanup();
    }
  }

  return 1;
}

//
// Perform the required actions on all
// the descriptors having I/O pending.
//

int NXTransExecute(int *resultFDs, int *errorFDs, int *setFDs, fd_set *readSet,
                       fd_set *writeSet, struct timeval *selectTs)
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  //
  // Control is NULL if the NX transport was
  // reset or never created. If control is
  // valid then prepare for jumping back in
  // the case of an error.
  //

  if (control == NULL || setjmp(context) == 1)
  {
    return 0;
  }

  #if defined(TEST) || defined(INFO)
  *logofs << "\nNXTransExecute: Going to execute I/O on the NX descriptors.\n"
          << logofs_flush;
  #endif

  if (control -> ProxyStage == stage_operational)
  {
    //
    // Check if I/O is possible on the proxy and
    // local agent descriptors.
    //

    if (agent != NULL)
    {
      handleAgentInLoop(*resultFDs, *errorFDs, *setFDs, *readSet, *writeSet, *selectTs);
    }

    #ifdef TEST
    *logofs << "Loop: At mark - 1 - difference from loop timestamp is "
            << diffTimestamp(startTs, getTimestamp())
            << " Ms.\n" << logofs_flush;
    #endif

    //
    // Flush any data on newly writable sockets.
    //

    handleWritableInLoop(*resultFDs, *writeSet);

    #ifdef TEST
    *logofs << "Loop: At mark - 2 - difference from loop timestamp is "
            << diffTimestamp(startTs, getTimestamp())
            << " Ms.\n" << logofs_flush;
    #endif

    //
    // Check if any socket has become readable.
    //

    handleReadableInLoop(*resultFDs, *readSet);

    #ifdef TEST
    *logofs << "Loop: At mark - 3 - difference from loop timestamp is "
            << diffTimestamp(startTs, getTimestamp())
            << " Ms.\n" << logofs_flush;
    #endif

    //
    // Handle the scheduled events on channels.
    //
    // - Restart, if possible, any client that was
    //   put to sleep.
    //
    // - Check if there are pointer motion events to
    //   flush. This applies only to X server side.
    //
    // - Check if any channel has exited the conges-
    //   tion state.
    //
    // - Check if there are images that are currently
    //   being streamed.
    //

    handleChannelEventsInLoop();

    #ifdef TEST
    *logofs << "Loop: At mark - 4 - difference from loop timestamp is "
            << diffTimestamp(startTs, getTimestamp())
            << " Ms.\n" << logofs_flush;
    #endif

    //
    // Check if user sent a signal to produce
    // statistics.
    //

    handleStatisticsInLoop();

    //
    // Check if connection is still alive.
    //

    handlePingInLoop(diffTs);

    //
    // We may have flushed the proxy link or
    // handled data coming from the remote.
    // Post-process the masks and select the
    // applicable agent descriptors.
    //

    if (agent != NULL)
    {
      handleAgentLateInLoop(*resultFDs, *errorFDs, *setFDs, *readSet, *writeSet, *selectTs);
    }

    #ifdef TEST
    *logofs << "Loop: At mark - 5 - difference from loop timestamp is "
            << diffTimestamp(startTs, getTimestamp())
            << " Ms.\n" << logofs_flush;
    #endif
  }

  //
  // Check if user sent a signal to restart
  // the proxy link or change configuration.
  //

  handleResetInLoop();

  //
  // Check if we have an alert to show.
  //

  handleAlertInLoop();

  if (control -> ProxyStage == stage_operational)
  {
    //
    // Check if it's time to give up.
    //

    handleCheckSessionInLoop();

    //
    // Check if local proxy is consuming
    // too many resources.
    //

    handleCheckLimitsInLoop();

    //
    // Check coherency of internal state.
    //

    #if defined(TEST) || defined(INFO)

    handleCheckStateInLoop(*setFDs);

    #endif

    #ifdef TEST
    *logofs << "Loop: At mark - 6 - difference from loop timestamp is "
            << diffTimestamp(startTs, getTimestamp())
            << " Ms.\n" << logofs_flush;
    #endif
  }

  //
  // Truncate the logs if needed.
  //

  handleLogReopenInLoop(logsTs, nowTs);

  return 1;
}

//
// Initialize the connection parameters and
// prepare for negotiating the link with the
// remote proxy.
//

int InitBeforeNegotiation()
{
  //
  // Disable limits on core dumps.
  //

  SetCore();

  //
  // Install the signal handlers.
  //

  InstallSignals();

  //
  // Track how much time we spend in initialization.
  //

  startTs = getTimestamp();

  //
  // If not explicitly specified, determine if local
  // mode is client or server according to parameters
  // provided so far.
  //

  if (!WE_SET_PROXY_MODE)
  {
    cerr << "Error" << ": Please specify either the -C or -S option.\n";

    HandleCleanup();
  }

  //
  // Start a watchdog. If initialization cannot
  // be completed before timeout, then clean up
  // everything and exit.
  //

  if (control -> ProxyMode == proxy_client)
  {
    #ifdef TEST
    *logofs << "Loop: Starting watchdog process with timeout of "
            << control -> InitTimeout / 1000 << " seconds.\n"
            << logofs_flush;
    #endif

    lastWatchdog = NXTransWatchdog(control -> InitTimeout / 1000);

    if (lastWatchdog < 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Can't start the NX watchdog process.\n"
              << logofs_flush;
      #endif

      lastWatchdog = 0;
    }
    #ifdef TEST
    else
    {
      *logofs << "Loop: Watchdog started with pid '"
              << lastWatchdog << "'.\n" << logofs_flush;
    }
    #endif
  }

  //
  // Open the log files.
  //

  SetLog();

  //
  // Set cups, multimedia and other
  // auxiliary ports.
  //

  SetPorts();

  //
  // Increase the number of maximum open
  // file descriptors for this process.
  //

  SetDescriptors();

  //
  // Print preliminary info.
  //

  PrintProcessInfo();

  //
  // Set local endianess.
  //

  unsigned int test = 1;

  setHostBigEndian(*((unsigned char *) (&test)) == 0);

  #ifdef TEST
  *logofs << "Loop: Local host is "
          << (hostBigEndian() ? "big endian" : "little endian")
          << ".\n" << logofs_flush;
  #endif

  if (control -> ProxyMode == proxy_client)
  {
    //
    // If running as client proxy, listen on sockets
    // that mimic an X display to which X clients can
    // connect to (e.g. unix:8 and/or localhost:8).
    //

    if (useTCPSocket)
    {
      SetupTCPSocket();
    }

    if (useUnixSocket)
    {
      SetupUnixSocket();
    }

    if (useCupsSocket)
    {
      SetupCupsSocket();
    }

    if (useKeybdSocket)
    {
      SetupKeybdSocket();
    }

    if (useSambaSocket)
    {
      SetupSambaSocket();
    }

    if (useMediaSocket)
    {
      SetupMediaSocket();
    }

    if (useHttpSocket)
    {
      SetupHttpSocket();
    }
  }
  else
  {
    //
    // Disable any listener.
    //

    useUnixSocket  = 0;
    useTCPSocket   = 0;
    useCupsSocket  = 0;
    useKeybdSocket = 0;
    useSambaSocket = 0;
    useMediaSocket = 0;
    useHttpSocket  = 0;

    //
    // Cannot handle internal connections
    // at the X server side.
    //

    useAgentSocket = 0;

    //
    // If we are at server side then get
    // ready to open the local display.
    //

    SetupDisplaySocket(xServerAddrFamily, xServerAddr, xServerAddrLength);

    //
    // Check that user provided for embedded
    // keyboard the same display where appli-
    // cations are going to be run.
    //

    if (keybdPort > 0 && keybdPort != X_TCP_PORT + xPort)
    {
      #ifdef PANIC
      *logofs << "Loop: WARNING! Overriding embedded keyboard "
              << "port with X port '" << X_TCP_PORT + xPort
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Overriding embedded keyboard "
           << "port with X port '" << X_TCP_PORT + xPort
           << "'.\n";

      keybdPort = X_TCP_PORT + xPort;
    }
  }

  //
  // If we are client proxy we need to complete our
  // initializazion phase. We, infact, are the side
  // which tells the link speed and cache size to
  // the remote peer.
  //

  if (control -> ProxyMode == proxy_client)
  {
    SetParameters();
  }

  return 1;
}

int SetupProxyConnection()
{
  if (proxyFD == -1)
  {
    if (WE_INITIATE_CONNECTION)
    {
      if (connectPort < 0)
      {
        connectPort = DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort;
      }

      #ifdef TEST
      *logofs << "Loop: Going to connect to " << connectHost
              << ":" << connectPort << ".\n" << logofs_flush;
      #endif

      proxyFD = ConnectToRemote(connectHost, connectPort);

      #ifdef TEST
      *logofs << "Loop: Connected to remote proxy on FD#"
              << proxyFD << ".\n" << logofs_flush;
      #endif

      cerr << "Info" << ": Connection to remote proxy '" << connectHost
           << ":" << connectPort << "' established.\n";
    }
    else
    {
      if (listenPort < 0)
      {
        listenPort = DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort;
      }

      #ifdef TEST
      *logofs << "Loop: Going to wait for connection on port " 
              << listenPort << ".\n" << logofs_flush;
      #endif

      proxyFD = WaitForRemote(listenPort);

      #ifdef TEST

      if (WE_LISTEN_FORWARDER)
      {
        *logofs << "Loop: Connected to remote forwarder on FD#"
                << proxyFD << ".\n" << logofs_flush;
      }
      else
      {
        *logofs << "Loop: Connected to remote proxy on FD#"
                << proxyFD << ".\n" << logofs_flush;
      }

      #endif

      cerr << "Info" << ": Connection with remote proxy "
           << "established.\n" << logofs_flush;
    }
  }
  #ifdef TEST
  else
  {
    *logofs << "Loop: Using the inherited connection on FD#"
            << proxyFD << ".\n" << logofs_flush;
  }
  #endif

  //
  // Set TCP_NODELAY on proxy descriptor
  // to reduce startup time. Option will
  // later be disabled if needed.
  //

  SetNoDelay(proxyFD, 1);

  //
  // We need non-blocking input since the
  // negotiation phase.
  //

  SetNonBlocking(proxyFD, 1);

  return 1;
}

//
// Create the required proxy and channel classes
// and get ready for handling the encoded traffic.
//

int InitAfterNegotiation()
{
  #ifdef TEST
  *logofs << "Loop: Going to finalize the initialization phase.\n"
          << logofs_flush;
  #endif

  cerr << "Info" << ": Handshaking with remote proxy "
       << "completed.\n" << logofs_flush;

  //
  // If we are server proxy we completed our
  // initializazion phase according to values
  // provided by the client side.
  //

  if (control -> ProxyMode == proxy_server)
  {
    SetParameters();
  }

  //
  // Create the proxy class and the statistics
  // repository and pass all the configuration
  // data we negotiated with the remote peer.
  //

  SetupProxyInstance();

  //
  // We completed both parsing of user's parameters
  // and handlshaking with remote proxy. Now print
  // a brief summary including the most significant
  // control values.
  //

  PrintConnectionInfo();

  //
  // Cancel the initialization watchdog.
  //

  if (lastWatchdog > 0)
  {
    KillProcess(lastWatchdog, "watchdog", SIGTERM, 1);

    lastWatchdog = 0;
    lastSignal   = 0;
  }

  //
  // Start the cache house-keeping process.
  //

  if (control -> LocalTotalStorageSizeLimit > 0)
  {
    if (control -> ProxyMode == proxy_client)
    {
      #ifdef TEST
      *logofs << "Loop: Starting house-keeping process with "
              << "cache " << (control -> ClientTotalStorageSizeLimit > 0 ?
                 control -> PersistentCacheDiskLimit : 0)
              << " and images " << 0 << ".\n" << logofs_flush;
      #endif

      lastKeeper = NXTransKeeper((control -> ClientTotalStorageSizeLimit > 0 ?
                                     control -> PersistentCacheDiskLimit : 0),
                                         0, control -> RootPath);

      if (lastKeeper < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't start the NX keeper process.\n"
                << logofs_flush;
        #endif

        lastKeeper = 0;
      }
      #ifdef TEST
      else
      {
        *logofs << "Loop: Keeper started with pid '"
                << lastKeeper << "'.\n" << logofs_flush;
      }
      #endif
    }
    else
    {
      #ifdef TEST
      *logofs << "Loop: Starting house-keeping process with "
              << "cache " << (control -> ServerTotalStorageSizeLimit > 0 ?
                 control -> PersistentCacheDiskLimit : 0) << " and images "
              << (control -> ImageCacheEnableLoad > 0 ||
                      control -> ImageCacheEnableSave > 0 ?
                          control -> ImageCacheDiskLimit : 0)
              << ".\n" << logofs_flush;
      #endif

      lastKeeper = NXTransKeeper((control -> ServerTotalStorageSizeLimit > 0 ?
                                     control -> PersistentCacheDiskLimit : 0),
                                         (control -> ImageCacheEnableLoad > 0 ||
                                              control -> ImageCacheEnableSave > 0 ?
                                                  control -> ImageCacheDiskLimit : 0),
                                                      control -> RootPath);

      if (lastKeeper < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't start the NX keeper process.\n"
                << logofs_flush;
        #endif

        lastKeeper = 0;
      }
      #ifdef TEST
      else
      {
        *logofs << "Loop: Keeper started with pid '"
                << lastKeeper << "'.\n" << logofs_flush;
      }
      #endif
    }
  }

  //
  // Set the log size check timestamp.
  //

  logsTs = getTimestamp();

  //
  // TODO: Due to the way the new NX transport is working,
  // the accounting of time spent handling messages must
  // be rewritten. In particular, at the moment it only
  // shows the time spent after executing a select, but it
  // doesn't account the time spent in the NXTrans* calls
  // to the encoding or decoding functions that are issued
  // directly by the agent. The result is that the number
  // of KB encoded per second displayed in the proxy sta-
  // tistics is actually much lower than the real through-
  // put generated by the proxy.
  //

  #ifdef TEST
  *logofs << "Loop: Time spent in initialization is "
          << diffTimestamp(startTs, getTimestamp())
          << " Ms.\n" << logofs_flush;
  #endif

  startTs = getTimestamp();

  #ifdef DEBUG
  *logofs << "Loop: New timestamp is " << strMsTimestamp(startTs)
          << ".\n" << logofs_flush;
  #endif

  //
  // We can now start handling binary data from
  // our peer proxy.
  //

  cerr << "Info" << ": Starting X protocol compression.\n";

  return 1;
}

int ResetProxyConnection()
{
  //
  // Send the reset notification to X channels.
  //

  proxy -> handleNotify(notify_begin_reset);

  //
  // Get rid of old persistent cache. In theory
  // path can change to follow a different type
  // of session. Image cache path is determined
  // at startup and will remain the same.
  //

  delete [] control -> PersistentCachePath;

  control -> PersistentCachePath = NULL;

  delete [] control -> PersistentCacheName;

  control -> PersistentCacheName = NULL;

  #ifdef TEST
  *logofs << "Loop: Closing old proxy connection.\n"
          << logofs_flush;
  #endif

  int oldFD = proxyFD;

  close(proxyFD);

  proxyFD = -1;

  #ifdef TEST
  *logofs << "Loop: Creating a new proxy connection.\n"
          << logofs_flush;
  #endif

  //
  // Will set proxyFD to the connected socket.
  //
  // TODO: Resetting the proxy connection will
  // not work. It requires fixing the old code
  // to comply with the new initialization pro-
  // cedure.
  //
  // SetupProxyConnection();
  //
  // NegotiateProxyConnection();
  //

  //
  // Ensure the new socket descriptor has the
  // same number as the old one.
  //

  if (proxyFD != oldFD)
  {
    if (dup2(proxyFD, oldFD) < 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Call to dup2 failed. Error is "
              << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Call to dup2 failed. Error is "
           << EGET() << " '" << ESTR() << "'.\n";

      HandleCleanup();
    }
    else
    {
      close(proxyFD);

      proxyFD = oldFD;
    }
  }

  //
  // TODO: This also has been replaced by the
  // new procedure.
  //
  // NegotiateProxyCache(proxyFD);
  //

  //
  // Inform user about the new cache parameters.
  //

  PrintResetInfo();

  //
  // Reset state from proxy down to channels.
  //

  if (proxy -> handleReset() < 0)
  {
    #ifdef TEST
    *logofs << "Loop: Proxy failure in handleReset().\n"
            << logofs_flush;
    #endif

    HandleCleanup();
  }

  //
  // Set socket options on proxy link.
  //

  proxy -> handleSocketConfiguration();

  //
  // Propagate link configuration. This
  // includes translating some control
  // parameters in 'local' and 'remote'.
  //

  proxy -> handleLinkConfiguration();

  //
  // Further adjust cache parameters according
  // to pack method and session type selected
  // by user.
  //

  proxy -> handleCacheConfiguration();

  //
  // Inform X channels that reset is completed.
  //

  proxy -> handleNotify(notify_end_reset);

  return 1;
}

int SetMode(int mode)
{
  //
  // Set the local proxy mode.
  //

  if (control -> ProxyMode == proxy_undefined)
  {
    if (mode == NX_MODE_CLIENT)
    {
      #ifdef TEST
      *logofs << "Loop: Selected proxy mode is NX_MODE_CLIENT.\n"
              << logofs_flush;
      #endif

      control -> ProxyMode = proxy_client;
    }
    else if (mode == NX_MODE_SERVER)
    {
      #ifdef TEST
      *logofs << "Loop: Selected proxy mode is NX_MODE_SERVER.\n"
              << logofs_flush;
      #endif

      control -> ProxyMode = proxy_server;
    }
    else
    {
      cerr << "Error" << ": Please specify either the -C or -S option.\n";

      HandleCleanup();
    }
  }

  return 1;
}

int SetupProxyInstance()
{
  if (control -> ProxyMode == proxy_client)
  {
    proxy = new ClientProxy(proxyFD);
  }
  else
  {
    proxy = new ServerProxy(proxyFD);
  }

  if (proxy == NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating the NX proxy.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating the NX proxy.\n";

    HandleCleanup();
  }

  //
  // Create the statistics repository.
  //

  statistics = new Statistics(proxy);

  if (statistics == NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating the NX statistics.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating the NX statistics.\n";

    HandleCleanup();
  }

  //
  // If user gave us a proxy cookie than create the
  // X11 authorization repository and find the real
  // cookie to be used for our X display.
  //

  SetupAuthInstance();

  //
  // Inform the X server side proxy about the ports where
  // it will have to forward the network connections.
  //

  if (control -> ProxyMode == proxy_server)
  {
    proxy -> setDisplayInfo(displayHost, xServerAddrFamily,
                                xServerAddr, xServerAddrLength);

    proxy -> setPortInfo(cupsPort, keybdPort, sambaPort,
                             mediaPort, httpPort);
  }

  //
  // Set socket options on proxy link, then propagate link
  // configuration to proxy. This includes translating some
  // control parameters in 'local' and 'remote'. Finally
  // adjust cache parameters according to pack method and
  // session type selected by user.
  //

  if (proxy -> handleSocketConfiguration() < 0 ||
          proxy -> handleLinkConfiguration() < 0 ||
              proxy -> handleCacheConfiguration() < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error configuring the NX transport.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error configuring the NX transport.\n";

    HandleCleanup();
  }

  //
  // Inform the proxy that from now on it can
  // start handling the encoded data.
  //

  proxy -> setOperational();

  //
  // Are we going to use an internal IPC connection
  // with agent? In this case create the connection
  // using the socket descriptor provided by caller.
  //

  SetupAgentInstance();

  #if defined(TEST) || defined(INFO)

  if (proxy -> getFlushable(proxyFD) > 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Proxy FD#" << proxyFD << " has data "
            << "to flush after setup of the NX transport.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Proxy FD#" << proxyFD << " has data "
         << "to flush after setup of the NX transport.\n";

    HandleCleanup();
  }

  #endif

  return 1;
}

int SetupAuthInstance()
{
  //
  // If user gave us a proxy cookie, then create the
  // X11 authorization repository and find the real
  // cookie to be used for our X display.
  //

  if (control -> ProxyMode == proxy_server)
  {
    if (control -> isProtoStep6() == 1 &&
            authCookie != NULL && *authCookie != '\0')
    {
      auth = new Auth(displayHost, authCookie);

      if (auth == NULL || auth -> isValid() != 1)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Error creating the X authorization.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Error creating the X authorization.\n";

        HandleCleanup();
      }
    }
    else
    {
      if (control -> isProtoStep6() == 0)
      {
        #ifdef TEST
        *logofs << "Loop: Remote proxy doesn't support fake "
                << "authentication.\n" << logofs_flush;
        #endif

        cerr << "Info" << ": Remote proxy doesn't support fake "
              << "authentication.\n";
      }
      else
      {
        #ifdef TEST
        *logofs << "Loop: No proxy cookie was provided for "
                << "authentication.\n" << logofs_flush;
        #endif

        cerr << "Info" << ": No proxy cookie was provided for "
             << "authentication.\n";
      }

      #ifdef TEST
      *logofs << "Loop: Forwarding the real X authorization "
              << "cookie.\n" << logofs_flush;
      #endif

      cerr << "Info" << ": Forwarding the real X authorization "
           << "cookie.\n";
    }
  }

  return 1;
}

int SetupAgentInstance()
{
  if (control -> ProxyMode == proxy_client &&
          useAgentSocket == 1)
  {
    //
    // This will temporarily disable signals to safely
    // load the cache, then will send a control packet
    // to the remote end, telling that cache has to be
    // loaded, so it's important that proxy is already
    // set in operational state.
    //

    int result;

    if (agent != NULL)
    {
      result = proxy -> handleNewAgentConnection(agent);
    }
    else
    {
      result = proxy -> handleNewXConnection(agentFD[1]);
    }

    if (result < 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Error creating the NX agent connection.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Error creating the NX agent connection.\n";

      HandleCleanup();
    }
  }

  return 1;
}

int SetupTCPSocket()
{
  //
  // Open TCP socket emulating local display.
  //

  tcpFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (tcpFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(tcpFD) < 0)
  {
    HandleCleanup();
  }

  unsigned int proxyPortTCP = X_TCP_PORT + proxyPort;

  sockaddr_in tcpAddr;

  tcpAddr.sin_family = AF_INET;
  tcpAddr.sin_port = htons(proxyPortTCP);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(tcpFD, (sockaddr *) &tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for TCP port "
            << proxyPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for TCP port "
         << proxyPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(tcpFD, 8) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for TCP port "
            << proxyPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for TCP port "
         << proxyPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  return 1;
}

int SetupUnixSocket()
{
  //
  // Open UNIX domain socket for display.
  //

  unixFD = socket(AF_UNIX, SOCK_STREAM, PF_UNSPEC);

  if (unixFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for UNIX domain"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for UNIX domain"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  sockaddr_un unixAddr;
  unixAddr.sun_family = AF_UNIX;

  struct stat dirStat;

  if ((stat("/tmp/.X11-unix", &dirStat) == -1) && (EGET() == ENOENT))
  {
    mkdir("/tmp/.X11-unix", (0777 | S_ISVTX));
    chmod("/tmp/.X11-unix", (0777 | S_ISVTX));
  }

  sprintf(unixSocketName, "/tmp/.X11-unix/X%d", proxyPort);

  strcpy(unixAddr.sun_path, unixSocketName);

  if (bind(unixFD, (sockaddr *) &unixAddr, sizeof(unixAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for UNIX domain socket "
            << unixSocketName << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ":  Call to bind failed for UNIX domain socket "
         << unixSocketName << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(unixFD, 8) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for UNIX domain socket "
            << unixSocketName << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ":  Call to listen failed for UNIX domain socket "
         << unixSocketName << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  //
  // Let any local user to gain access to socket.
  //

  chmod(unixSocketName, 0777);

  return 1;
}

//
// The following is a dumb copy-paste. I
// will write a generic function someday.
//

int SetupCupsSocket()
{
  cupsFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (cupsFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(cupsFD) < 0)
  {
    HandleCleanup();
  }

  unsigned int cupsPortTCP = cupsPort;

  sockaddr_in tcpAddr;

  tcpAddr.sin_family = AF_INET;
  tcpAddr.sin_port = htons(cupsPortTCP);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(cupsFD, (sockaddr *) &tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for cups TCP port "
            << cupsPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for cups TCP port "
         << cupsPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(cupsFD, 4) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for cups TCP port "
            << cupsPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for cups TCP port "
         << cupsPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  return 1;
}

int SetupKeybdSocket()
{
  keybdFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (keybdFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(keybdFD) < 0)
  {
    HandleCleanup();
  }

  unsigned int keybdPortTCP = keybdPort;

  sockaddr_in tcpAddr;

  tcpAddr.sin_family = AF_INET;
  tcpAddr.sin_port = htons(keybdPortTCP);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(keybdFD, (sockaddr *) &tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for embedded keyboard "
            << "TCP port " << keybdPortTCP << ". Error is " << EGET()
            << " '" << ESTR() << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for embedded keyboard "
         << "TCP port " << keybdPortTCP << ". Error is " << EGET()
         << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  if (listen(keybdFD, 4) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for embedded keyboard "
            << "TCP port " << keybdPortTCP << ". Error is " << EGET()
            << " '" << ESTR() << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for embedded keyboard "
         << "TCP port " << keybdPortTCP << ". Error is " << EGET()
         << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }

  return 1;
}

int SetupSambaSocket()
{
  sambaFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (sambaFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(sambaFD) < 0)
  {
    HandleCleanup();
  }

  unsigned int sambaPortTCP = sambaPort;

  sockaddr_in tcpAddr;

  tcpAddr.sin_family = AF_INET;
  tcpAddr.sin_port = htons(sambaPortTCP);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(sambaFD, (sockaddr *) &tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for samba TCP port "
            << sambaPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for samba TCP port "
         << sambaPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(sambaFD, 8) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for samba TCP port "
            << sambaPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for samba TCP port "
         << sambaPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  return 1;
}

int SetupMediaSocket()
{
  mediaFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (mediaFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(mediaFD) < 0)
  {
    HandleCleanup();
  }

  unsigned int mediaPortTCP = mediaPort;

  sockaddr_in tcpAddr;

  tcpAddr.sin_family = AF_INET;
  tcpAddr.sin_port = htons(mediaPortTCP);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(mediaFD, (sockaddr *) &tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for multimedia TCP port "
            << mediaPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for multimedia TCP port "
         << mediaPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(mediaFD, 8) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for multimedia TCP port "
            << mediaPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for multimedia TCP port "
         << mediaPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  return 1;
}

int SetupHttpSocket()
{
  httpFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (httpFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket"
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket"
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    HandleCleanup();
  }
  else if (SetReuseAddress(httpFD) < 0)
  {
    HandleCleanup();
  }

  unsigned int httpPortTCP = httpPort;

  sockaddr_in tcpAddr;

  tcpAddr.sin_family = AF_INET;
  tcpAddr.sin_port = htons(httpPortTCP);
  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  if (bind(httpFD, (sockaddr *) &tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for http TCP port "
            << httpPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for http TCP port "
         << httpPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  if (listen(httpFD, 8) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for http TCP port "
            << httpPortTCP << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for http TCP port "
         << httpPortTCP << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    HandleCleanup();
  }

  return 1;
}

int SetupDisplaySocket(int &xServerAddrFamily, sockaddr *&xServerAddr,
                           unsigned int &xServerAddrLength)
{
  xServerAddrFamily = AF_INET;
  xServerAddr = NULL;
  xServerAddrLength = 0;

  char *display;

  if (*displayHost == '\0')
  {
    //
    // Assume DISPLAY as the X server to which
    // we will forward the proxied connections.
    // This means that NX parameters have been
    // passed through other means.
    //

    display = getenv("DISPLAY");

    if (display == NULL || *display == '\0')
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Host X server DISPLAY is not set.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Host X server DISPLAY is not set.\n";

      HandleCleanup();
    }
    else if (strncasecmp(display, "nx/nx,", 6) == 0 ||
                 strncasecmp(display, "nx,", 3) == 0 ||
                     strncasecmp(display, "nx:", 3) == 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! NX transport on host X server '"
              << display << "' not supported.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": NX transport on host X server '"
           << display << "' not supported.\n";

      cerr << "Error" << ": Please run the local proxy specifying "
           << "the host X server to connect to.\n";

      HandleCleanup();
    }
    else if (strlen(display) >= DEFAULT_STRING_LENGTH)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Host X server DISPLAY cannot exceed "
              << DEFAULT_STRING_LENGTH << " characters.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Host X server DISPLAY cannot exceed "
           << DEFAULT_STRING_LENGTH << " characters.\n";

      HandleCleanup();
    }

    strcpy(displayHost, display);
  }

  display = new char[strlen(displayHost) + 1];

  if (display == NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Out of memory handling DISPLAY variable.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Out of memory handling DISPLAY variable.\n";

    HandleCleanup();
  }

  strcpy(display, displayHost);

  char *separator = rindex(display, ':');

  if ((separator == NULL) || !isdigit(*(separator + 1)))
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Invalid display '" << display << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Invalid display '" << display << "'.\n";

    HandleCleanup();
  }

  *separator = '\0';

  xPort = atoi(separator + 1);

  #ifdef TEST
  *logofs << "Loop: Using local X display '" << displayHost
          << "' with host '" << display << "' and port '"
          << xPort << "'.\n" << logofs_flush;
  #endif

  if (separator == display || strcmp(display, "unix") == 0)
  {
    //
    // UNIX domain port.
    //

    #ifdef TEST
    *logofs << "Loop: Using real X server on UNIX domain socket.\n"
            << logofs_flush;
    #endif

    sockaddr_un *xServerAddrUNIX = new sockaddr_un;

    xServerAddrFamily = AF_UNIX;
    xServerAddrUNIX -> sun_family = AF_UNIX;

    //
    // The scope of this function is to fill either the sockaddr_un
    // (when the display is set to the Unix Domain socket) or the
    // sockaddr_in structure (when connecting by TCP) only once, so
    // that the structure will be later used at the time the server
    // proxy will try to forward the connection to the X server. We
    // don't need to verify that the socket does exist at the pre-
    // sent moment. The method that forwards the connection will
    // perform the required checks and will retry, if needed. Anyway
    // we need to select the name of the socket, so we check if the
    // well-known directory exists and take that as an indication of
    // where the socket will be created.
    //

    struct stat statInfo;

    char unixSocketDir[DEFAULT_STRING_LENGTH];

    strcpy(unixSocketDir, "/tmp/.X11-unix");

    if (stat(unixSocketDir, &statInfo) < 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Can't determine the location of "
              << "the X display socket.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Can't determine the location of "
           << "the X display socket.\n";

      #ifdef PANIC
      *logofs << "Loop: PANIC! Error " << EGET() << " '" << ESTR()
              << "' checking '" << unixSocketDir << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Error " << EGET() << " '" << ESTR()
           << "' checking '" << unixSocketDir << "'.\n";

      HandleCleanup();
    }
    else
    {
      sprintf(unixSocketName, "%s/X%d", unixSocketDir, xPort);
    }

    strcpy(xServerAddrUNIX -> sun_path, unixSocketName);

    xServerAddr = (sockaddr *) xServerAddrUNIX;
    xServerAddrLength = sizeof(sockaddr_un);
  }
  else
  {
    //
    // TCP port.
    //

    #ifdef TEST
    *logofs << "Loop: Using real X server on TCP port.\n"
            << logofs_flush;
    #endif

    xServerAddrFamily = AF_INET;

    int xServerIPAddr = GetHostAddress(display);

    if (xServerIPAddr == 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Unknown display host '" << display
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Unknown display host '" << display
           << "'.\n";

      HandleCleanup();
    }

    sockaddr_in *xServerAddrTCP = new sockaddr_in;

    xServerAddrTCP -> sin_family = AF_INET;
    xServerAddrTCP -> sin_port = htons(X_TCP_PORT + xPort);
    xServerAddrTCP -> sin_addr.s_addr = xServerIPAddr;

    xServerAddr = (sockaddr *) xServerAddrTCP;
    xServerAddrLength = sizeof(sockaddr_in);
  }

  delete [] display;

  return 1;
}

void HandleShutdown()
{
  if (proxy -> getShutdown() == 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! No shutdown of proxy link "
            << "performed by remote proxy.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Connection with remote peer broken.\n";

    #ifdef TEST
    *logofs << "Loop: Bytes received so far are " << control -> getBytesIn()
            << ".\n" << logofs_flush;
    #endif

    if (control -> EnableRestartOnFailure == 1)
    {
      cerr << "Error" << ": Please check the state of your network "
           << "connection.\n";

      //
      // If not in cleanup phase jump back
      // to the main loop.
      //

      if (lastKill == 0)
      {
        #ifdef TEST
        *logofs << "Loop: Trying to revert to loop context.\n"
                << logofs_flush;
        #endif

        //
        // TODO: A close() in the reconnection
        // procedure seems to be not enough to
        // ensure that remote proxy identifies
        // the link shutdown.
        //

        shutdown(proxyFD, SHUT_RDWR);

        longjmp(context, 1);
      }
    }
    else
    {
      cerr << "Error" << ": Please check the state of your "
           << "network and retry.\n";

      #ifdef TEST
      *logofs << "Loop: Showing the proxy abort dialog.\n"
              << logofs_flush;
      #endif

      if (control -> ProxyMode == proxy_server)
      {
        HandleAlert(ABORT_PROXY_CONNECTION_ALERT, 1);

        handleAlertInLoop();
      }
    }
  }
  else
  {
    #ifdef TEST
    *logofs << "Loop: Finalized the remote proxy shutdown.\n"
            << logofs_flush;
    #endif

    if (control -> getBytesIn() < 1024 &&
            useAgentSocket == 1)
    {
      cerr << "Info" << ": Your session was closed before reaching "
           << "a usable state.\n";
      cerr << "Info" << ": This can be due to local X server "
           << "refusing access to client.\n";
      cerr << "Info" << ": Please check authorization provided "
           << "by remote X application.\n";
    }

    cerr << "Info" << ": Shutting down the link and exiting.\n";
  }

  HandleCleanup();
}


void WaitCleanup()
{
  T_timestamp selectTs;

  while (NXTransRunning())
  {
    setTimestamp(selectTs, control -> PingTimeout);

    NXTransContinue(&selectTs);
  }
}

void KillProcess(int pid, const char *label, int signal, int wait)
{
  if (pid > 0)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Killing the " << label << " process '"
            << pid << "' from process with pid '" << getpid()
            << "' with signal '" << SignalLabel(signal)
            << "'.\n" << logofs_flush;
    #endif

    signal = (signal == 0 ? SIGTERM : signal);

    if (kill(pid, signal) < 0 && EGET() != ESRCH)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Couldn't kill the " << label
              << " process with pid '" << pid << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Couldn't kill the " << label
           << " process with pid '" << pid << "'.\n";
    }

    if (wait == 1)
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "Loop: Waiting for the " << label
              << " process '" << pid << "' to die.\n"
              << logofs_flush;
      #endif

      WaitChild(pid);
    }
  }
  else
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! No " << label << " process "
            << "with pid '" << pid << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": No " << label << " process "
         << "with pid '" << pid << "'.\n";
  }
}

void HandleCleanup()
{
  #ifdef TEST
  *logofs << "Loop: Going to clean up system resources "
          << "in process '" << getpid() << "'.\n"
          << logofs_flush;
  #endif

  //
  // Suspend any signal while cleaning up.
  //

  DisableSignals();

  if (getpid() == lastProxy)
  {
    //
    // Terminate all the children.
    //

    CleanupChildren();
  
    //
    // Close all sockets.
    //

    CleanupSockets();

    //
    // Release the global objects.
    //

    CleanupGlobal();

    //
    // Restore the original signal handlers.
    //

    RestoreSignals();
  }

  //
  // This is our last chance to print a message. If this
  // is the process which created the transport we will
  // jump back into the loop, letting the caller find out
  // that the connection is broken, otherwise we assume
  // that this is a child of the proxy and so we will
  // safely exit.
  //

  #ifdef TEST

  if (getpid() == lastProxy)
  {
    *logofs << "Loop: Reverting to loop context in process with "
            << "pid '" << getpid() << "'.\n" << logofs_flush;
  }
  else
  {
    *logofs << "Loop: Exiting from child process with pid '"
            << getpid() << "'.\n" << logofs_flush;
  }

  #endif

  if (logofs != NULL && logofs != &cerr &&
          *logFileName != '\0')
  {
    *logofs << flush;

    delete logofs;

    //
    // Let the log go again to the standard
    // error.
    //

    logofs = &cerr;
  }

  if (statofs != NULL && statofs != &cerr &&
          *statFileName != '\0')
  {
    *statofs << flush;

    delete statofs;

    statofs = NULL;
  }

  if (errofs != NULL)
  {
    *errofs << flush;

    if (errofs == &cerr)
    {
      errofs = NULL;
    }
    else if (errsbuf != NULL)
    {
      cerr.rdbuf(errsbuf);

      errsbuf = NULL;

      delete errofs;
    }

    errofs = NULL;
  }

  if (getpid() == lastProxy)
  {
    //
    // Reset all values to their default.
    //

    CleanupLocal();

    longjmp(context, 1);
  }
  else
  {
    //
    // Give a last chance to the process
    // to cleanup the control class.
    //

    CleanupGlobal();

    exit(0);
  }
}

void CleanupChildren()
{
  //
  // Remove any watchdog.
  //

  if (lastWatchdog > 0)
  {
    KillProcess(lastWatchdog, "watchdog", SIGTERM, 1);

    lastWatchdog = 0;
    lastSignal   = 0;
  }

  //
  // Kill the cache house-keeping process.
  //

  if (lastKeeper > 0)
  {
    KillProcess(lastKeeper, "house-keeping", SIGTERM, 1);

    lastKeeper = 0;
  }

  //
  // Let any running dialog to continue until it is
  // closed by the user. In general this is the exp-
  // ected behaviour, as for example when we are
  // exiting because the link was abrouptedly shut
  // down.
  //

  if (lastDialog > 0)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: WARNING! Leaving the dialog process '"
            << lastDialog << "' running in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    lastDialog = 0;
  }

  //
  // Give user a chance to start a new session.
  //

  if (control -> EnableRestartOnShutdown == 1)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Respawning nxclient "
            << "on display '" << displayHost
            << "'.\n" << logofs_flush;
    #endif

    NXTransClient(displayHost);
  }
}

void CleanupGlobal()
{
  if (proxy != NULL)
  {
    #ifdef TEST
    *logofs << "Loop: Freeing up proxy in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    delete proxy;

    proxy = NULL;
  }

  if (agent != NULL)
  {
    #ifdef TEST
    *logofs << "Loop: Freeing up agent in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    delete agent;

    agent = NULL;
  }

  if (auth != NULL)
  {
    #ifdef TEST
    *logofs << "Loop: Freeing up auth data in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    delete auth;

    auth = NULL;
  }

  if (statistics != NULL)
  {
    #ifdef TEST
    *logofs << "Loop: Freeing up statistics in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    delete statistics;

    statistics = NULL;
  }

  if (control != NULL)
  {
    #ifdef TEST
    *logofs << "Loop: Freeing up control in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    delete control;

    control = NULL;
  }
}

void CleanupSockets()
{
  if (tcpFD != -1)
  {
    #ifdef TEST
    *logofs << "Loop: Closing TCP listener in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    close(tcpFD);

    tcpFD = -1;
  }

  if (unixFD != -1)
  {
    #ifdef TEST
    *logofs << "Loop: Closing UNIX listener in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    close(unixFD);

    unixFD = -1;
  }

  if (useUnixSocket == 1 && *unixSocketName != '\0')
  {
    #ifdef TEST
    *logofs << "Loop: Going to remove the Unix domain socket '"
            << unixSocketName << "' in process " << "with pid '"
            << getpid() << "'.\n" << logofs_flush;
    #endif

    unlink(unixSocketName);
  }

  if (cupsFD != -1)
  {
    #ifdef TEST
    *logofs << "Loop: Closing cups listener in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    close(cupsFD);

    cupsFD = -1;
  }

  if (keybdFD != -1)
  {
    #ifdef TEST
    *logofs << "Loop: Closing embedded keyboard listener "
            << "in process " << "with pid '" << getpid()
            << "'.\n" << logofs_flush;
    #endif

    close(keybdFD);

    keybdFD = -1;
  }

  if (sambaFD != -1)
  {
    #ifdef TEST
    *logofs << "Loop: Closing samba listener in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    close(sambaFD);

    sambaFD = -1;
  }

  if (mediaFD != -1)
  {
    #ifdef TEST
    *logofs << "Loop: Closing multimedia listener in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    close(mediaFD);

    mediaFD = -1;
  }

  if (httpFD != -1)
  {
    #ifdef TEST
    *logofs << "Loop: Closing http listener in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    close(httpFD);

    httpFD = -1;
  }

  if (proxyFD != -1)
  {
    #ifdef TEST
    *logofs << "Loop: Closing proxy FD in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    close(proxyFD);

    proxyFD = -1;
  }

  if (agentFD[1] != -1)
  {
    #ifdef TEST
    *logofs << "Loop: Closing agent FD in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    close(agentFD[1]);

    agentFD[0] = -1;
    agentFD[1] = -1;
  }
}

void CleanupLocal()
{
  *rootDir    = '\0';
  *sessionDir = '\0';

  *logFileName     = '\0';
  *statFileName    = '\0';
  *sessionFileName = '\0';

  *linkSpeedName  = '\0';
  *cacheSizeName  = '\0';
  *shmemSizeName  = '\0';
  *imagesSizeName = '\0';

  *bitrateLimitName = '\0';

  *packMethodName = '\0';

  packMethod  = -1;
  packQuality = -1;

  *sessionType = '\0';
  *sessionId   = '\0';

  parsedOptions = 0;
  parsedCommand = 0;

  *remoteData = '\0';
  remotePosition = 0;

  tcpFD   = -1;
  unixFD  = -1;
  cupsFD  = -1;
  keybdFD = -1;
  sambaFD = -1;
  mediaFD = -1;
  httpFD  = -1;
  proxyFD = -1;

  agentFD[0] = -1;
  agentFD[1] = -1;

  useUnixSocket  = 1;
  useTCPSocket   = 1;
  useCupsSocket  = 0;
  useKeybdSocket = 0;
  useSambaSocket = 0;
  useMediaSocket = 0;
  useHttpSocket  = 0;
  useAgentSocket = 0;

  useNoDelay = -1;
  useFlush   = -1;
  useRender  = -1;
  useTaint   = -1;

  *unixSocketName = '\0';

  *connectHost = '\0';
  *acceptHost  = '\0';
  *listenHost  = '\0';
  *displayHost = '\0';
  *authCookie  = '\0';

  proxyPort = DEFAULT_NX_PROXY_PORT;
  xPort     = DEFAULT_NX_X_PORT;

  xServerAddrFamily = -1;
  xServerAddr       = NULL;
  xServerAddrLength = 0;

  listenPort  = -1;
  connectPort = -1;

  cupsPort  = -1;
  keybdPort = -1;
  sambaPort = -1;
  mediaPort = -1;
  httpPort  = -1;

  *bindHost = '\0';
  bindPort = -1;

  startTs = nullTimestamp();
  logsTs  = nullTimestamp();
  nowTs   = nullTimestamp();

  diffTs = 0;

  lastProxy    = 0;
  lastDialog   = 0;
  lastWatchdog = 0;
  lastKeeper   = 0;
  lastKill     = 0;

  lastReadableTs = nullTimestamp();

  lastAlert.code  = 0;
  lastAlert.local = 0;

  lastMasks.blocked   = 0;
  lastMasks.installed = 0;

  memset(&lastMasks.saved, 0, sizeof(sigset_t));

  for (int i = 0; i < 32; i++)
  {
    //
    // Need to keep monitoring the children.
    //

    if (i != SIGCHLD)
    {
      lastMasks.enabled[i] = 0;
      lastMasks.forward[i] = 0;

      memset(&lastMasks.action[i], 0, sizeof(struct sigaction));
    }
  }

  lastSignal = 0;

  memset(&lastTimer.action, 0, sizeof(struct sigaction));
  memset(&lastTimer.value,  0, sizeof(struct itimerval));

  lastTimer.start = nullTimestamp();
  lastTimer.next  = nullTimestamp();
}

int CheckAbort()
{
  if (lastSignal != 0)
  {
    #ifdef TEST
    *logofs << "Loop: Aborting the procedure due to signal '"
            << lastSignal << "', '" << SignalLabel(lastSignal)
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Info" << ": Aborting the procedure due to signal '"
         << lastSignal << "', '" << SignalLabel(lastSignal)
         << "'.\n";

    lastSignal = 0;

    return 1;
  }

  return 0;
}

void HandleAbort()
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  *logofs << flush;

  //
  // The current default is to just quit the program.
  // Code has not been updated to deal with the new
  // NX transport loop.
  //

  if (control -> EnableCoreDumpOnAbort == 1)
  {
    cerr << "Error" << ": Generating a core file to help "
         << "investigations.\n";
  }

  if (lastSignal == SIGHUP)
  {
    lastSignal = 0;
  }

  if (control -> EnableCoreDumpOnAbort == 1)
  {
    cerr << flush;

    signal(SIGABRT, SIG_DFL);

    raise(SIGABRT);
  }
  else if (control -> EnableRestartOnFailure == 1)
  {
    //
    // Check if we have already entered the
    // cleanup phase. If not, jump back to
    // the main loop.
    //

    if (lastKill == 0)
    {
      #ifdef TEST
      *logofs << "Loop: Trying to revert to loop context.\n"
              << logofs_flush;
      #endif

      //
      // TODO: A close() in the reconnection
      // procedure seems to be not enough to
      // ensure that remote proxy identifies
      // the link shutdown.
      //

      shutdown(proxyFD, SHUT_RDWR);

      longjmp(context, 1);
    }
  }

  #ifdef TEST
  *logofs << "Loop: Showing the proxy abort dialog.\n"
          << logofs_flush;
  #endif

  if (control -> ProxyMode == proxy_server)
  {
    HandleAlert(ABORT_PROXY_CONNECTION_ALERT, 1);

    handleAlertInLoop();
  }

  HandleCleanup();
}

void HandleAlert(int code, int local)
{
  if (lastAlert.code == 0)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Requesting an alert dialog with code "
            << code << " and local " << local << ".\n"
            << logofs_flush;
    #endif

    lastAlert.code  = code;
    lastAlert.local = local;
  }
  #if defined(TEST) || defined(INFO)
  else
  {
    *logofs << "Loop: WARNING! Alert dialog already requested "
            << "with code " << lastAlert.code << ".\n"
            << logofs_flush;
  }
  #endif

  return;
}

void InstallSignals()
{
  #ifdef TEST
  *logofs << "Loop: Installing signals in process with pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  for (int i = 0; i < 32; i++)
  {
    if (CheckSignal(i) == 1 &&
          lastMasks.enabled[i] == 0)
    {
      InstallSignal(i, NX_SIGNAL_ENABLE);
    }
  }

  lastMasks.installed = 1;
}

void RestoreSignals()
{
  #ifdef TEST
  *logofs << "Loop: Restoring signals in process with pid '"
          << getpid() << "'.\n" << logofs_flush;
  #endif

  if (lastMasks.installed == 1)
  {
    //
    // Need to keep monitoring the children.
    //

    for (int i = 0; i < 32; i++)
    {
      if (lastMasks.enabled[i] == 1 && i != SIGCHLD)
      {
        RestoreSignal(i);
      }
    }
  }

  lastMasks.installed = 0;

  if (lastMasks.blocked == 1)
  {
    EnableSignals();
  }
}

void DisableSignals()
{
  if (lastMasks.blocked == 0)
  {
    sigset_t newMask;

    sigemptyset(&newMask);

    //
    // Block the other signals that may
    // be installed by the agent, that
    // is those signals for which the
    // check returns 2.
    //

    for (int i = 0; i < 32; i++)
    {
      if (CheckSignal(i) > 0)
      {
        #ifdef DUMP
        *logofs << "Loop: Disabling signal " << i << " '"
                << SignalLabel(i) << "' in process with pid '"
                << getpid() << "'.\n" << logofs_flush;
        #endif

        sigaddset(&newMask, i);
      }
    }

    sigprocmask(SIG_BLOCK, &newMask, &lastMasks.saved);

    lastMasks.blocked++;
  }
  else
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Signals were already blocked in "
            << "process with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Signals were already blocked in "
         << "process with pid '" << getpid() << "'.\n";
  }
}

void EnableSignals()
{
  if (lastMasks.blocked == 1)
  {
    #ifdef TEST
    *logofs << "Loop: Enabling signals in process with pid '"
            << getpid() << "'.\n" << logofs_flush;
    #endif

    sigprocmask(SIG_SETMASK, &lastMasks.saved, NULL);

    lastMasks.blocked = 0;
  }
  else
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Signals were not blocked in "
            << "process with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Signals were not blocked in "
         << "process with pid '" << getpid() << "'.\n";
  }
}

void InstallSignal(int signal, int action)
{
  #ifdef TEST

  if (lastMasks.enabled[signal] == 1)
  {
    *logofs << "Loop: Reinstalling handler for signal " << signal
            << " '" << SignalLabel(signal) << "' in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
  }
  else
  {
    *logofs << "Loop: Installing handler for signal " << signal
            << " '" << SignalLabel(signal) << "' in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
  }

  #endif  

  if (signal == SIGALRM && isTimestamp(lastTimer.start))
  {
    ResetTimer();
  }

  struct sigaction newAction;

  newAction.sa_handler = HandleSignal;

  //
  // This field doesn't exist on most OSes except
  // Linux. We keep setting the field to NULL to
  // avoid side-effects in the case the field is
  // a value return.
  //

  #if defined(__linux__)

  newAction.sa_restorer = NULL;

  #endif

  sigemptyset(&(newAction.sa_mask));

  if (signal == SIGCHLD)
  {
    newAction.sa_flags = SA_NOCLDSTOP;
  }
  else
  {
    newAction.sa_flags = 0;
  }

  sigaction(signal, &newAction, &lastMasks.action[signal]);

  lastMasks.enabled[signal] = 1;

  if (action == NX_SIGNAL_FORWARD)
  {
    lastMasks.forward[signal] = 1;
  }
}

void RestoreSignal(int signal)
{
  if (lastMasks.enabled[signal] == 0)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Signal '" << SignalLabel(signal)
            << " not installed in process with pid '"
            << getpid() << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Signal '" << SignalLabel(signal)
         << " not installed in process with pid '"
         << getpid() << "'.\n";

    return;
  }

  #ifdef TEST
  *logofs << "Loop: Restoring handler for signal " << signal
          << " '" << SignalLabel(signal) << "' in process "
          << "with pid '" << getpid() << "'.\n"
          << logofs_flush;
  #endif

  if (signal == SIGALRM && isTimestamp(lastTimer.start))
  {
    ResetTimer();
  }

  sigaction(signal, &lastMasks.action[signal], NULL);

  lastMasks.enabled[signal] = 0;
  lastMasks.forward[signal] = 0;
}

const char *SignalLabel(int signal)
{
  switch (signal)
  {
    case SIGCHLD:
    {
      return "SIGCHLD";
    }
    case SIGUSR1:
    {
      return "SIGUSR1";
    }
    case SIGUSR2:
    {
      return "SIGUSR2";
    }
    case SIGHUP:
    {
      return "SIGHUP";
    }
    case SIGINT:
    {
      return "SIGINT";
    }
    case SIGTERM:
    {
      return "SIGTERM";
    }
    case SIGPIPE:
    {
      return "SIGPIPE";
    }
    case SIGALRM:
    {
      return "SIGALRM";
    }
    case SIGVTALRM:
    {
      return "SIGVTALRM";
    }
    case SIGWINCH:
    {
      return "SIGWINCH";
    }
    case SIGIO:
    {
      return "SIGIO";
    }
    case SIGTSTP:
    {
      return "SIGTSTP";
    }
    case SIGTTIN:
    {
      return "SIGTTIN";
    }
    case SIGTTOU:
    {
      return "SIGTTOU";
    }
    default:
    {
      return "Unknown";
    }
  }
}

void HandleSignal(int signal)
{
  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  #if defined(UNSAFE) && (defined(TEST) || defined(INFO))

  if (lastSignal != 0)
  {
    *logofs << "Loop: WARNING! Last signal is '" << lastSignal
            << "', '" << SignalLabel(signal) << "' and not zero "
            << "in process with pid '" << getpid() << "'.\n"
            << logofs_flush;
  }

  *logofs << "Loop: Signal '" << signal << "', '"
          << SignalLabel(signal) << "' received in process "
          << "with pid '" << getpid() << "'.\n" << logofs_flush;

  #endif

  switch (signal)
  {
    case SIGUSR1:
    {
      if (proxy != NULL && lastSignal == 0)
      {
        lastSignal = SIGUSR1;
      }

      break;
    }
    case SIGUSR2:
    {
      if (proxy != NULL && lastSignal == 0)
      {
        lastSignal = SIGUSR2;
      }

      break;
    }
    case SIGPIPE:
    {
      //
      // It can happen that SIGPIPE is delivered
      // to the proxy even in the case some other
      // descriptor is unexpectedly closed.
      //
      // if (agentFD[1] != -1)
      // {
      //   cerr << "Info" << ": Received signal 'SIGPIPE'. "
      //        << "Closing agent conection.\n";
      //
      //   shutdown(agentFD[1], SHUT_RDWR);
      // }
      //

      break;
    }
    case SIGALRM:
    {
      //
      // Nothing to do. Just allow to wake up
      // the process on blocking operations.
      //

      break;
    }
    case SIGCHLD:
    {
      //
      // Check if any of our children has exited.
      //

      if (HandleChildren() != 0)
      {
        signal = 0;
      }

      //
      // Don't save this signal or it will override
      // any previous signal sent by child before
      // exiting.
      //

      break;
    }

    #ifdef __CYGWIN32__

    case 12:
    {
      //
      // Nothing to do. This signal is what is delivered
      // by the Cygwin library when trying use a shared
      // memory function.
      //

      #ifdef WARNING
      *logofs << "Loop: WARNING! Received signal '12' in "
              << "process with pid '" << getpid() << "'.\n"
              << logofs_flush;

      *logofs << "Loop: WARNING! Please check that the "
              << "cygserver daemon is running.\n"
              << logofs_flush;

      cerr << "Warning" << ": Received signal '12' in "
           << "process with pid '" << getpid() << "'.\n";

      cerr << "Warning" << ": Please check that the "
           << "cygserver daemon is running.\n";

      #endif

      break;
    }

    #endif

    default:
    {
      //
      // Register the signal so we can handle it
      // inside the main loop. We will probably
      // dispose any resource and exit.
      //

      if (getpid() == lastProxy)
      {
        #if defined(UNSAFE) && defined(TEST)
        *logofs << "Loop: Registering end of session request "
                << "due to signal '" << signal << "', '"
                << SignalLabel(signal) << "'.\n"
                << logofs_flush;
        #endif

        lastSignal = signal;
      }
      else
      {
        //
        // This is a child, so exit immediately.
        //

        HandleCleanup();
      }
    }
  }

  if (signal != 0 && lastMasks.forward[signal] == 1)
  {
    #if defined(UNSAFE) && defined(TEST)
    *logofs << "Loop: Forwarding signal '" << signal << "', '"
            << SignalLabel(signal) << "' to previous handler.\n"
            << logofs_flush;
    #endif

    lastMasks.action[signal].sa_handler(signal);
  }
}

int HandleChildren()
{
  //
  // Try to waitpid() for each child because the
  // call might have return ECHILD and so we may
  // have lost any of the processes.
  //
  
  if (lastDialog != 0 && HandleChild(lastDialog) == 1)
  {
    #if defined(UNSAFE) && defined(TEST)
    *logofs << "Loop: Resetting pid of last dialog process "
            << "in handler.\n" << logofs_flush;
    #endif

    lastDialog = 0;

    return 1;
  }

  if (lastWatchdog != 0 && HandleChild(lastWatchdog) == 1)
  {
    #if defined(UNSAFE) && defined(TEST)
    *logofs << "Loop: Watchdog is gone. Setting the last "
            << "signal to SIGTERM.\n" << logofs_flush;
    #endif

    lastSignal = SIGTERM;

    #if defined(UNSAFE) && defined(TEST)
    *logofs << "Loop: Resetting pid of last watchdog process "
            << "in handler.\n" << logofs_flush;
    #endif

    lastWatchdog = 0;

    return 1;
  }

  if (lastKeeper != 0 && HandleChild(lastKeeper) == 1)
  {
    #if defined(UNSAFE) && defined(TEST)
    *logofs << "Loop: Resetting pid of last house-keeping process "
            << "in handler.\n" << logofs_flush;
    #endif

    lastKeeper = 0;

    return 1;
  }

  //
  // This can actually happen either because we
  // reset the pid of the child process as soon
  // as we kill it, or because of a child process
  // of our parent.
  //

  #ifdef TEST
  *logofs << "Loop: Ignoring signal received for the "
          << "unregistered child.\n" << logofs_flush;
  #endif

  return 0;
}

int HandleChild(int child)
{
  int pid;

  int status  = 0;
  int options = WNOHANG | WUNTRACED;

  while ((pid = waitpid(child, &status, options)) &&
             pid == -1 && EGET() == EINTR);

  return CheckChild(pid, status);
}

int WaitChild(int child)
{
  int pid;

  int status  = 0;
  int options = WUNTRACED;

  while ((pid = waitpid(child, &status, options)) &&
             pid == -1 && EGET() == EINTR);

  return (EGET() == ECHILD ? 1 : CheckChild(pid, status));
}

int CheckChild(int pid, int status)
{
  if (pid > 0)
  {
    if (WIFSTOPPED(status))
    {
      #if defined(UNSAFE) && defined(TEST)
      *logofs << "Loop: Child process '" << pid << "' was stopped "
              << "with signal " << (WSTOPSIG(status)) << ".\n"
              << logofs_flush;
      #endif

      return 0;
    }
    else
    {
      #if defined(UNSAFE) && defined(TEST)

      if (WIFEXITED(status))
      {
        *logofs << "Loop: Child process '" << pid << "' exited "
                << "with status " << (WEXITSTATUS(status))
                << ".\n" << logofs_flush;
      }
      else if (WIFSIGNALED(status))
      {
        *logofs << "Loop: Child process '" << pid << "' died "
                << "because of signal " << (WTERMSIG(status))
                << ".\n" << logofs_flush;
      }

      #endif

      return 1;
    }
  }
  else if (pid < 0)
  {
    if (EGET() != ECHILD)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Call to waitpid failed. "
              << "Error is " << EGET() << " '" << ESTR()
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Call to waitpid failed. "
           << "Error is " << EGET() << " '" << ESTR()
           << "'.\n";

      HandleCleanup();
    }

    //
    // This can happen when the waitpid() is
    // blocking, as the SIGCHLD is received
    // within the call.
    //

    #ifdef WARNING
    *logofs << "Loop: WARNING! No more children processes running.\n"
            << logofs_flush;
    #endif

    return 1;
  }

  return 0;
}

void CheckParent(char *name, char *type, int parent)
{
  //
  // Exit if parent is dead.
  //

  if (parent != getppid() || parent == 1)
  {
    #ifdef WARNING
    *logofs << name << ": WARNING! Parent process appears "
            << "to be dead. Exiting " << type << ".\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Parent process appears "
         << "to be dead. Exiting " << type << ".\n";

    HandleCleanup();
  }
}

void HandleTimer(int signal)
{
  if (signal == SIGALRM)
  {
    if (isTimestamp(lastTimer.start))
    {
      #if defined(UNSAFE) && defined(TEST)
      *logofs << "Loop: Timer expired at " << strMsTimestamp()
              << " in process with pid '" << getpid() << "'.\n"
              << logofs_flush;
      #endif

      if (proxy != NULL)
      {
        proxy -> handleTimer();
      }

      ResetTimer();
    }
    else
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Inconsistent timer state "
              << " in process with pid '" << getpid() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Inconsistent timer state "
           << " in process with pid '" << getpid() << "'.\n";
    }
  }
  else
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Inconsistent signal '"
            << signal << "', '" << SignalLabel(signal)
            << "' received in process with pid '"
            << getpid() << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Inconsistent signal '"
         << signal << "', '" << SignalLabel(signal)
         << "' received in process with pid '"
         << getpid() << "'.\n";
  }
}

void SetTimer(int value)
{
  if (isTimestamp(lastTimer.start))
  {
    int diffTs = diffTimestamp(lastTimer.start, getTimestamp());

    if (diffTs > lastTimer.next.tv_usec / 1000 * 2)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Timer missed to expire at "
              << strMsTimestamp() << " in process with pid '"
              << getpid() << "'.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Timer missed to expire at "
           << strMsTimestamp() << " in process with pid '"
           << getpid() << "'.\n";

      HandleTimer(SIGALRM);
    }
    else
    {
      #ifdef TEST
      *logofs << "Loop: Timer already running at "
              << strMsTimestamp() << " in process with pid '"
              << getpid() << "'.\n" << logofs_flush;
      #endif

      return;
    }
  }

  //
  // Save the former handler.
  //

  struct sigaction action;

  action.sa_handler = HandleTimer;

  #if defined(__linux__)

  action.sa_restorer = NULL;

  #endif

  sigemptyset(&action.sa_mask);

  action.sa_flags = 0;

  sigaction(SIGALRM, &action, &lastTimer.action);

  //
  // Start the timer.
  //

  lastTimer.next = getTimestamp(value);

  struct itimerval timer;

  timer.it_interval = lastTimer.next;
  timer.it_value    = lastTimer.next;

  #ifdef TEST
  *logofs << "Loop: Timer set to " << lastTimer.next.tv_sec
          << " S and " << lastTimer.next.tv_usec / 1000
          << " Ms at " << strMsTimestamp() << " in process "
          << "with pid '" << getpid() << "'.\n"
          << logofs_flush;
  #endif

  if (setitimer(ITIMER_REAL, &timer, &lastTimer.value) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to setitimer failed. "
            << "Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to setitimer failed. "
         << "Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    lastTimer.next = nullTimestamp();

    return;
  }

  lastTimer.start = getTimestamp();
}

void ResetTimer()
{
  if (isTimestamp(lastTimer.start) == 0)
  {
    #if defined(UNSAFE) && defined(TEST)
    *logofs << "Loop: Timer not running in process "
            << "with pid '" << getpid() << "'.\n"
            << logofs_flush;
    #endif

    return;
  }

  #if defined(UNSAFE) && defined(TEST)
  *logofs << "Loop: Timer reset at " << strMsTimestamp()
          << " in process with pid '" << getpid()
          << "'.\n" << logofs_flush;
  #endif

  //
  // Restore the old signal mask and timer.
  //

  if (setitimer(ITIMER_REAL, &lastTimer.value, NULL) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to setitimer failed. "
            << "Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to setitimer failed. "
         << "Error is " << EGET() << " '" << ESTR()
         << "'.\n";
  }

  if (sigaction(SIGALRM, &lastTimer.action, NULL) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to sigaction failed. "
            << "Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to sigaction failed. "
         << "Error is " << EGET() << " '" << ESTR()
         << "'.\n";
  }

  lastTimer.start = lastTimer.next = nullTimestamp();
}

const char *FlushLabel(int type)
{
  switch ((T_flush) type)
  {
    case flush_if_needed:
    {
      return "if needed";
    }
    case flush_if_any:
    {
      return "if any";
    }
    case flush_if_sync:
    {
      return "if sync";
    }
    default:
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Unknown flush type '"
              << type << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Unknown flush type '"
           << type << "'.\n";

      HandleCleanup();
    }
  }
}

const char *PolicyLabel(int type)
{
  switch ((T_flush_policy) type)
  {
    case policy_immediate:
    {
      return "immediate";
    }
    case policy_deferred:
    {
      return "deferred";
    }
    default:
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Unknown policy type '"
              << type << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Unknown policy type '"
           << type << "'.\n";

      HandleCleanup();
    }
  }
}

const char *CodeLabel(int code)
{
  switch ((T_proxy_code) code)
  {
    case code_new_x_connection:
    {
      return "code_new_x_connection";
    }
    case code_new_cups_connection:
    {
      return "code_new_cups_connection";
    }
    case code_new_keybd_connection:
    {
      return "code_new_keybd_connection";
    }
    case code_new_samba_connection:
    {
      return "code_new_samba_connection";
    }
    case code_new_media_connection:
    {
      return "code_new_media_connection";
    }
    case code_switch_connection:
    {
      return "code_switch_connection";
    }
    case code_drop_connection:
    {
      return "code_drop_connection";
    }
    case code_finish_connection:
    {
      return "code_finish_connection";
    }
    case code_begin_congestion:
    {
      return "code_begin_congestion";
    }
    case code_end_congestion:
    {
      return "code_end_congestion";
    }
    case code_alert_request:
    {
      return "code_alert_request";
    }
    case code_alert_reply:
    {
      return "code_alert_reply";
    }
    case code_reset_request:
    {
      return "code_reset_request";
    }
    case code_reset_reply:
    {
      return "code_reset_reply";
    }
    case code_load_request:
    {
      return "code_load_request";
    }
    case code_load_reply:
    {
      return "code_load_reply";
    }
    case code_save_request:
    {
      return "code_save_request";
    }
    case code_save_reply:
    {
      return "code_save_reply";
    }
    case code_shutdown_request:
    {
      return "code_shutdown_request";
    }
    case code_shutdown_reply:
    {
      return "code_shutdown_reply";
    }
    case code_token_request:
    {
      return "code_token_request";
    }
    case code_token_reply:
    {
      return "code_token_reply";
    }
    case code_configuration_request:
    {
      return "code_configuration_request";
    }
    case code_configuration_reply:
    {
      return "code_configuration_reply";
    }
    case code_statistics_request:
    {
      return "code_statistics_request";
    }
    case code_statistics_reply:
    {
      return "code_statistics_reply";
    }
    case code_new_http_connection:
    {
      return "code_new_http_connection";
    }
    case code_sync_request:
    {
      return "code_sync_request";
    }
    case code_sync_reply:
    {
      return "code_sync_reply";
    }
    default:
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Unknown control code '"
              << code << "'.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Unknown control code '"
           << code << "'.\n";

      return "Unknown";
    }
  }
}

//
// Open TCP socket to listen for remote proxy and
// block until remote connects. If successful close
// the listening socket and return FD on which the
// other party is connected.
//

int WaitForRemote(int portNum)
{
  char hostLabel[DEFAULT_STRING_LENGTH] = { 0 };

  int retryAccept  = -1;
  int listenIPAddr = -1;

  int proxyFD = -1;
  int newFD   = -1;

  //
  // Get IP address of host to be awaited.
  //

  int acceptIPAddr = 0;

  if (*acceptHost != '\0')
  {
    acceptIPAddr = GetHostAddress(acceptHost);

    if (acceptIPAddr == 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Cannot accept connections from unknown host '"
              << acceptHost << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Cannot accept connections from unknown host '"
           << acceptHost << "'.\n";

      goto WaitForRemoteError;
    }
  }

  proxyFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

  if (proxyFD == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to socket failed for TCP socket. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to socket failed for TCP socket. "
         << "Error is " << EGET() << " '" << ESTR() << "'.\n";

    goto WaitForRemoteError;
  }
  else if (SetReuseAddress(proxyFD) < 0)
  {
    goto WaitForRemoteError;
  }

  listenIPAddr = 0;

  ParseListenOption(listenIPAddr);

  sockaddr_in tcpAddr;

  tcpAddr.sin_family = AF_INET;
  tcpAddr.sin_port = htons(portNum);

  //
  // Quick patch to run on MacOS/X where inet_addr("127.0.0.1")
  // alone seems to fail to return a valid interface. It probably
  // just needs a htonl() or something like that.
  // 
  // TODO: We have to give another look at inet_addr("127.0.0.1")
  // on the Mac.
  //

  #ifdef __APPLE__

  tcpAddr.sin_addr.s_addr = htonl(INADDR_ANY);

  #else

  tcpAddr.sin_addr.s_addr = listenIPAddr;

  #endif

  if (bind(proxyFD, (sockaddr *) &tcpAddr, sizeof(tcpAddr)) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to bind failed for TCP port "
            << portNum << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to bind failed for TCP port "
         << portNum << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    goto WaitForRemoteError;
  }

  if (listen(proxyFD, 4) == -1)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to listen failed for TCP port "
            << portNum << ". Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Call to listen failed for TCP port "
         << portNum << ". Error is " << EGET() << " '" << ESTR()
         << "'.\n";

    goto WaitForRemoteError;
  }

  if (*acceptHost != '\0')
  {
    strcat(hostLabel, "'");
    strcat(hostLabel, acceptHost);
    strcat(hostLabel, "'");
  }
  else
  {
    strcpy(hostLabel, "any host");
  }

  #ifdef TEST
  *logofs << "Loop: Waiting for connection from "
          << hostLabel  << " on port '" << portNum
          << "'.\n" << logofs_flush;
  #endif

  cerr << "Info" << ": Waiting for connection from "
       << hostLabel << " on port '" << portNum
       << "'.\n";

  //
  // How many times to loop waiting for connections
  // from the selected host? Each loop wait for at
  // most 20 seconds so a default value of 3 gives
  // a timeout of 1 minute.
  //
  // TODO: Handling of timeouts and retry attempts
  // must be rewritten.
  //

  retryAccept = control -> OptionProxyRetryAccept;

  for (;;)
  {
    fd_set readSet;

    FD_ZERO(&readSet);
    FD_SET(proxyFD, &readSet);

    T_timestamp selectTs;

    selectTs.tv_sec  = 20;
    selectTs.tv_usec = 0;

    int result = select(proxyFD + 1, &readSet, NULL, NULL, &selectTs);

    if (result == -1)
    {
      if (EGET() == EINTR)
      {
        if (CheckAbort() != 0)
        {
          goto WaitForRemoteError;
        }

        continue;
      }
 
      #ifdef PANIC
      *logofs << "Loop: PANIC! Call to select failed. Error is "
              << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Call to select failed. Error is "
           << EGET() << " '" << ESTR() << "'.\n";

      goto WaitForRemoteError;
    }
    else if (result > 0 && FD_ISSET(proxyFD, &readSet))
    {
      sockaddr_in newAddr;

      size_t addrLen = sizeof(sockaddr_in);

      newFD = accept(proxyFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

      if (newFD == -1)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Call to accept failed. Error is "
                << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Call to accept failed. Error is "
             << EGET() << " '" << ESTR() << "'.\n";

        goto WaitForRemoteError;
      }

      char *connectedHost = inet_ntoa(newAddr.sin_addr);
      unsigned int connectedPort = ntohs(newAddr.sin_port);

      if (*acceptHost == '\0' || (int) newAddr.sin_addr.s_addr == acceptIPAddr)
      {
        #ifdef TEST
        *logofs << "Loop: Accepted connection from '" << connectedHost
                << "' on port '" << connectedPort << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Info" << ": Accepted connection from '" << connectedHost
             << "' on port '" << connectedPort << "'.\n";

        break;
      }
      else
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Refusing connection from '" << connectedHost
                << "' on port '" << connectedPort << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Refusing connection from '" << connectedHost
             << "' on port '" << connectedPort << "'.\n";
      }

      //
      // Not the best way to elude a DOS attack...
      //

      sleep(5);

      close(newFD);
    }

    if (--retryAccept == 0)
    {
      if (*acceptHost == '\0')
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Connection with remote host "
                << "could not be established.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Connection with remote host "
             << "could not be established.\n";
      }
      else
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Connection with remote host '"
                << acceptHost << "' could not be established.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Connection with remote host '"
             << acceptHost << "' could not be established.\n";
      }

      goto WaitForRemoteError;
    }
    else
    {
      handleCheckSessionInConnect();
    }
  }

  close(proxyFD);

  return newFD;

WaitForRemoteError:

  close(proxyFD);

  HandleCleanup();
}

//
// Connect to remote proxy. If successful
// return FD of connection, else return -1.
//

int ConnectToRemote(const char *const hostName, int portNum)
{
  int proxyFD = -1;

  int remoteIPAddr = GetHostAddress(hostName);

  if (remoteIPAddr == 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Unknown remote host '"
            << hostName << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Unknown remote host '"
         << hostName << "'.\n";

    HandleCleanup();
  }

  #ifdef TEST
  *logofs << "Loop: Connecting to remote host '" 
          << hostName << ":" << portNum << "'.\n"
          << logofs_flush;
  #endif

  cerr << "Info" << ": Connecting to remote host '"
       << hostName << ":" << portNum << "'.\n"
       << logofs_flush;

  //
  // How many times we retry to connect to remote
  // host in case of failure?
  //

  int retryConnect = control -> OptionProxyRetryConnect;

  //
  // Show an alert after 20 seconds and use the
  // same timeout to interrupt the connect. The
  // retry timeout is incremental, starting from
  // 100 miliseconds up to 2 seconds.
  //

  int alertTimeout   = 20000;
  int connectTimeout = 20000;
  int retryTimeout   = 100000;

  T_timestamp lastRetry = getTimestamp();

  sockaddr_in addr;

  addr.sin_family = AF_INET;
  addr.sin_port = htons(portNum);
  addr.sin_addr.s_addr = remoteIPAddr;

  for (;;)
  {
    proxyFD = socket(AF_INET, SOCK_STREAM, PF_UNSPEC);

    if (proxyFD == -1)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Call to socket failed. "
              << "Error is " << EGET() << " '" << ESTR()
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Call to socket failed. "
           << "Error is " << EGET() << " '" << ESTR()
           << "'.\n";

      goto ConnectToRemoteError;
    }
    else if (SetReuseAddress(proxyFD) < 0)
    {
      goto ConnectToRemoteError;
    }

    //
    // Ensure operation is timed out
    // if there is a network problem.
    //

    #ifdef DEBUG
    *logofs << "Loop: Timer set to " << connectTimeout / 1000
            << " seconds " << "with retry set to "
            << retryConnect << " in process with pid '"
            << getpid() << "'.\n" << logofs_flush;
    #endif

    SetTimer(connectTimeout);

    int result = connect(proxyFD, (sockaddr *) &addr, sizeof(sockaddr_in));

    int reason = EGET();

    ResetTimer();

    if (result < 0)
    {
      close(proxyFD);

      if (CheckAbort() != 0)
      {
        goto ConnectToRemoteError;
      }
      else if (--retryConnect == 0)
      {
        ESET(reason);

        #ifdef PANIC
        *logofs << "Loop: PANIC! Connection to '" << hostName
                << ":" << portNum << "' failed. Error is "
                << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Connection to '" << hostName
             << ":" << portNum << "' failed. Error is "
             << EGET() << " '" << ESTR() << "'.\n";

        goto ConnectToRemoteError;
      }
      else
      {
        #ifdef TEST
        *logofs << "Loop: Sleeping " << retryTimeout / 1000
                << " Ms before retrying.\n" << logofs_flush;
        #endif

        usleep(retryTimeout);

        retryTimeout <<= 1;

        if (retryTimeout > 2000000)
        {
          retryTimeout = 2000000;
        }
      }

      //
      // Check if it is time to show an alert dialog.
      //

      if (diffTimestamp(lastRetry, getTimestamp()) >=
              (alertTimeout - control -> LatencyTimeout))
      {
        if (lastDialog == 0)
        {
          handleCheckSessionInConnect();

          WaitChild(lastDialog);

          lastRetry = getTimestamp();
        }
      }
      #ifdef TEST
      {
        *logofs << "Loop: Not showing the dialog with "
                << (diffTimestamp(lastRetry, getTimestamp()) / 1000)
                << " seconds elapsed.\n" << logofs_flush;
      }
      #endif

      ESET(reason);

      #ifdef TEST
      *logofs << "Loop: Connection to '" << hostName
              << ":" << portNum << "' failed with error '"
              << ESTR() << "'. Retrying.\n"
              << logofs_flush;
      #endif
    }
    else
    {
      //
      // Connection was successful.
      //

      break;
    }
  }

  return proxyFD;

ConnectToRemoteError:

  if (proxyFD != -1)
  {
    close(proxyFD);
  }

  HandleCleanup();
}

//
// Make a string of options for the remote
// proxy and write it to the descriptor.
// The string includes the local version.
//

int SendProxyOptions(int fd)
{
  char options[DEFAULT_REMOTE_OPTIONS_LENGTH];

  //
  // TODO: Due to a stupid bug in version 1.2.2,
  // we can't send our real version to the remote
  // peer or otherwise it would end up adopting
  // protocol step 2. We add the real version
  // after the 1.2.2 identifier. This data will
  // be ignored by a 'real' 1.2.2.
  //
  // sprintf(options, "NXPROXY-%i.%i.%i", control -> LocalVersionMajor,
  //             control -> LocalVersionMinor, control -> LocalVersionPatch);
  //

  sprintf(options, "NXPROXY-1.2.2-%i.%i.%i", control -> LocalVersionMajor,
              control -> LocalVersionMinor, control -> LocalVersionPatch);

  //
  // If you want to send options from proxy
  // initiating connection use something like
  // this:
  //
  // if (WE_PROVIDE_CREDENTIALS)
  // {
  //   sprintf(options + strlen(options), "%s=%s", option, value);
  // }
  //
  // If you want to send options according to
  // local proxy mode use something like this:
  //
  // if (control -> ProxyMode == proxy_client)
  // {
  //   sprintf(options + strlen(options), "%s=%s", option, value);
  // }
  //

  //
  // Send the authorization cookie if any. We assume
  // user can choose to not provide any auth cookie
  // and allow any connection to be accepted.
  //

  if (WE_PROVIDE_CREDENTIALS && *authCookie != '\0')
  {
    sprintf(options + strlen(options), " cookie=%s,", authCookie);
  }
  else
  {
    sprintf(options + strlen(options), " ");
  }

  //
  // Now link characteristics and compression
  // options. Delta compression, as well as
  // preferred pack method, are imposed by
  // client proxy.
  //

  if (control -> ProxyMode == proxy_client)
  {
    //
    // If protocol level is not 3 we don't
    // have support for PNG decoding at the
    // remote side.
    //

    if (control -> isProtoStep3() == 0 &&
            packMethod >= PACK_PNG_JPEG_8_COLORS &&
                packMethod <= PACK_PNG_JPEG_16M_COLORS)
    {
      //
      // Arbitrarily assume that corresponding
      // JPEG method is at this offset.
      //

      packMethod = packMethod - 21;

      ParsePackMethod(packMethod, packQuality);

      #ifdef WARNING
      *logofs << "Loop: WARNING! Method PNG unsupported by remote. "
              << "Setting preferred method '" << packMethodName
              << "'.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": Method PNG unsupported by remote. "
           << "Setting preferred method '" << packMethodName
           << "'.\n";
    }

    sprintf(options + strlen(options), "link=%s,pack=%s,cache=%s,",
                linkSpeedName, packMethodName, cacheSizeName);

    if (*bitrateLimitName != '\0')
    {
      sprintf(options + strlen(options), "limit=%s,",
                  bitrateLimitName);
    }

    //
    // Starting from protocol level 2 client side
    // suggests the flush timeout to be used by the
    // remote proxy. We know which protocol level
    // is the remote as at client side we are the
    // first to get the remote options.
    //

    if (control -> isProtoStep2() == 1)
    {
      if (useFlush != -1)
      {
        sprintf(options + strlen(options), "flush=%d,", useFlush);
      }
      else
      {
        sprintf(options + strlen(options), "flush=%d,",
                    control -> FlushTimeout);
      }
    }

    //
    // Old proxy versions assume that agent
    // doesn't support the render extension.
    //

    if (control -> SessionMode == session_x &&
            control -> isProtoStep4() == 0)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Not using render extension "
              << "due to lack of alpha support.\n"
              << logofs_flush;
      #endif

      control -> PersistentCacheLoadRender = 0;

      control -> AgentHideRender = 1;
    }

    //
    // Starting from protocol level 3 we let user
    // disable render extension or instruct client
    // proxy to short-circuit simple replies. We
    // also pass the session type to be sure the
    // remote side gets the right value.
    //

    if (control -> isProtoStep3() == 1)
    {
      sprintf(options + strlen(options), "render=%d,taint=%d,",
                  (control -> AgentHideRender == 0),
                      control -> AgentTaintReplies);

      if (*sessionType != '\0')
      {
        sprintf(options + strlen(options), "type=%s,", sessionType);
      }
      else
      {
        sprintf(options + strlen(options), "type=default,");
      }
    }

    //
    // Send image cache parameters.
    //

    if (control -> isProtoStep3() == 1)
    {
      sprintf(options + strlen(options), "images=%s,", imagesSizeName);
    }
    else
    {
      control -> ImageCacheEnableLoad = 0;
      control -> ImageCacheEnableSave = 0;
    }

    sprintf(options + strlen(options), "delta=%d,stream=%d,data=%d ",
                control -> LocalDeltaCompression,
                    control -> LocalStreamCompressionLevel,
                        control -> LocalDataCompressionLevel);
  }
  else
  {
    //
    // If no special compression level was selected,
    // server side will use compression levels set
    // by client.
    //

    if (control -> LocalStreamCompressionLevel < 0)
    {
      sprintf(options + strlen(options), "stream=default,");
    }
    else
    {
      sprintf(options + strlen(options), "stream=%d,",
                  control -> LocalStreamCompressionLevel);
    }

    if (control -> LocalDataCompressionLevel < 0)
    {
      sprintf(options + strlen(options), "data=default ");
    }
    else
    {
      sprintf(options + strlen(options), "data=%d ",
                  control -> LocalDataCompressionLevel);
    }
  }

  #ifdef TEST
  *logofs << "Loop: Sending remote options '"
          << options << "'.\n" << logofs_flush;
  #endif

  return WriteLocalData(fd, options, strlen(options));
}

int ReadProxyVersion(int fd)
{
  #ifdef TEST
  *logofs << "Loop: Going to read the remote proxy version "
          << "from FD#" << fd << ".\n" << logofs_flush;
  #endif

  //
  // Read until the first space in string.
  // We expect the remote version number.
  //

  char options[DEFAULT_REMOTE_OPTIONS_LENGTH];

  int result = ReadRemoteData(fd, options, sizeof(options), ' ');

  if (result <= 0)
  {
    return result;
  }

  #ifdef TEST
  *logofs << "Loop: Received remote version string '"
          << options << "' from FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  if (strncmp(options, "NXPROXY-", strlen("NXPROXY-")) != 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Parse error in remote options string '"
            << options << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Parse error in remote options string '"
         << options << "'.\n";

    return -1;
  }

  //
  // TODO: Due to the same stupid bug, we must
  // read the remote version in a special way.
  //

  int major = -1;
  int minor = -1;
  int patch = -1;

  sscanf(options, "NXPROXY-%i.%i.%i-%i.%i.%i", &(control -> RemoteVersionMajor),
             &(control -> RemoteVersionMinor), &(control -> RemoteVersionPatch),
                 &major, &minor, &patch);

  if (control -> RemoteVersionMajor == 1 &&
          control -> RemoteVersionMinor == 2 &&
              control -> RemoteVersionPatch == 2 &&
                  major != -1 && minor != -1 && patch != -1)
  {
    control -> RemoteVersionMajor = major;
    control -> RemoteVersionMinor = minor;
    control -> RemoteVersionPatch = patch;

    #ifdef TEST
    *logofs << "Loop: Read trailing remote version '" << major
            << "." << minor << "." << patch << "'.\n"
            << logofs_flush;
    #endif

    #ifdef TEST
    *logofs << "Loop: Assuming remote version '" << control -> RemoteVersionMajor
            << "." << control -> RemoteVersionMinor << "." << control -> RemoteVersionPatch
            << "'.\n" << logofs_flush;
    #endif
  }

  if (SetVersion() < 0)
  {
    if (control -> ProxyMode == proxy_server)
    {
      HandleAlert(WRONG_PROXY_VERSION_ALERT, 1);
    }

    handleAlertInLoop();

    #ifdef PANIC
    *logofs << "Loop: PANIC! Remote NXPROXY version number "
            << control -> RemoteVersionMajor << "." << control -> RemoteVersionMinor
            << "." << control -> RemoteVersionPatch << " doesn't match local version "
            << control -> LocalVersionMajor << "." << control -> LocalVersionMinor
            << "." << control -> LocalVersionPatch << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Remote NXPROXY version number "
         << control -> RemoteVersionMajor << "." << control -> RemoteVersionMinor
         << "." << control -> RemoteVersionPatch << " doesn't match local version "
         << control -> LocalVersionMajor << "." << control -> LocalVersionMinor
         << "." << control -> LocalVersionPatch << ".\n";

    return -1;
  }
  else if (control -> RemoteVersionMinor != control -> LocalVersionMinor ||
               control -> RemoteVersionPatch != control -> LocalVersionPatch)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Connected to remote NXPROXY version "
            << control -> RemoteVersionMajor << "." << control -> RemoteVersionMinor
            << "." << control -> RemoteVersionPatch << " with local version "
            << control -> LocalVersionMajor << "." << control -> LocalVersionMinor
            << "." << control -> LocalVersionPatch << ".\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Connected to remote NXPROXY version "
         << control -> RemoteVersionMajor << "." << control -> RemoteVersionMinor
         << "." << control -> RemoteVersionPatch << " with local version "
         << control -> LocalVersionMajor << "." << control -> LocalVersionMinor
         << "." << control -> LocalVersionPatch << ".\n" << logofs_flush;

    if (control -> RemoteVersionMinor > control -> LocalVersionMinor ||
            (control -> RemoteVersionMinor == control -> LocalVersionMinor &&
                 control -> RemoteVersionPatch > control -> LocalVersionPatch))
    {
      cerr << "Warning" << ": Consider checking http://www.nomachine.com/ for updates.\n";
    }
  }

  return 1;
}

int ReadProxyOptions(int fd)
{
  #ifdef TEST
  *logofs << "Loop: Going to read the remote proxy options "
          << "from FD#" << fd << ".\n" << logofs_flush;
  #endif

  char options[DEFAULT_REMOTE_OPTIONS_LENGTH];

  int result = ReadRemoteData(fd, options, sizeof(options), ' ');

  if (result <= 0)
  {
    return result;
  }

  #ifdef TEST
  *logofs << "Loop: Received remote options string '"
          << options << "' from FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  //
  // Get the remote options, delimited by a space character.
  // Note that there will be a further initialization phase
  // at the time proxies negotiate cache file to restore.
  //

  if (ParseRemoteOptions(options) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Couldn't negotiate a valid "
            << "session with remote NX proxy.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Couldn't negotiate a valid "
         << "session with remote NX proxy.\n";

    return -1;
  }

  return 1;
}

int SendProxyCaches(int fd)
{
  #ifdef TEST
  *logofs << "Loop: Synchronizing local and remote caches.\n"
          << logofs_flush;
  #endif

  cerr << "Info" << ": Synchronizing local and remote caches.\n"
       << logofs_flush;

  if (control -> ProxyMode == proxy_client)
  {
    //
    // Prepare a list of caches matching this
    // session type and send it to the remote.
    //

    #ifdef TEST
    *logofs << "Loop: Going to send the list of local caches.\n"
            << logofs_flush;
    #endif

    SetCaches();

    int entries = DEFAULT_REMOTE_CACHE_ENTRIES;

    const char prefix = 'C';

    if (control -> LocalDeltaCompression == 0 ||
            control -> PersistentCacheEnableLoad == 0)
    {
      #ifdef TEST
      *logofs << "Loop: Writing an empty list to FD#" << fd
              << ".\n" << logofs_flush;
      #endif

      return WriteLocalData(fd, "cachelist=none ", strlen("cachelist=none "));
    }

    int count = 0;

    #ifdef TEST
    *logofs << "Loop: Looking for cache files in directory '"
            << control -> PersistentCachePath << "'.\n" << logofs_flush;
    #endif

    DIR *cacheDir = opendir(control -> PersistentCachePath);

    if (cacheDir != NULL)
    {
      dirent *dirEntry;

      int prologue = 0;

      while (((dirEntry = readdir(cacheDir)) != NULL) && (count < entries))
      {
        if (*dirEntry -> d_name == prefix &&
                strlen(dirEntry -> d_name) == (MD5_LENGTH * 2 + 2))
        {
          if (prologue == 0)
          {
            WriteLocalData(fd, "cachelist=", strlen("cachelist="));

            prologue = 1;
          }
          else
          {
            WriteLocalData(fd, ",", strlen(","));
          }

          #ifdef TEST
          *logofs << "Loop: Writing entry '" << control -> PersistentCachePath
                  << "/" << dirEntry -> d_name << "' to FD#" << fd
                  << ".\n" << logofs_flush;
          #endif

          //
          // Write cache file name to the socket,
          // including leading 'C-' or 'S-'.
          //

          WriteLocalData(fd, dirEntry -> d_name, MD5_LENGTH * 2 + 2);

          count++;
        }
      }

      closedir(cacheDir);
    }

    if (count == 0)
    {
      #ifdef TEST
      *logofs << "Loop: Writing an empty list to FD#" << fd
              << ".\n" << logofs_flush;
      #endif

      return WriteLocalData(fd, "cachelist=none ", strlen("cachelist=none "));
    }
    else
    {
      return WriteLocalData(fd, " ", 1);
    }
  }
  else
  {
    //
    // Send back the selected cache name.
    //

    #ifdef TEST
    *logofs << "Loop: Going to send the selected cache.\n"
            << logofs_flush;
    #endif

    char buffer[DEFAULT_STRING_LENGTH];

    if (control -> PersistentCacheName != NULL)
    {
      #ifdef TEST
      *logofs << "Loop: Name of selected cache file is '"
              << control -> PersistentCacheName << "'.\n"
              << logofs_flush;
      #endif

      sprintf(buffer, "cachefile=%s%s ",
                  *(control -> PersistentCacheName) == 'C' ? "S-" : "C-",
                      control -> PersistentCacheName + 2);
    }
    else
    {
      #ifdef TEST
      *logofs << "Loop: No valid cache file was selected.\n"
              << logofs_flush;
      #endif

      sprintf(buffer, "cachefile=none ");
    }

    #ifdef TEST
    *logofs << "Loop: Sending string '" << buffer
            << "' as selected cache file.\n"
            << logofs_flush;
    #endif

    return WriteLocalData(fd, buffer, strlen(buffer));
  }
}

int ReadProxyCaches(int fd)
{
  if (control -> ProxyMode == proxy_client)
  {
    #ifdef TEST
    *logofs << "Loop: Going to receive the selected proxy cache.\n"
            << logofs_flush;
    #endif

    //
    // We will read the name of cache plus the stop character.
    //

    char buffer[DEFAULT_STRING_LENGTH];

    //
    // Leave space for a trailing null.
    //

    int result = ReadRemoteData(fd, buffer, sizeof("cachefile=") + MD5_LENGTH * 2 + 3, ' ');

    if (result <= 0)
    {
      return result;
    }

    char *cacheName = strstr(buffer, "cachefile=");

    if (cacheName == NULL)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Invalid cache file option '"
              << buffer << "' provided by remote proxy.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Invalid cache file option '"
           << buffer << "' provided by remote proxy.\n";

      HandleCleanup();
    }

    cacheName += strlen("cachefile=");

    if (control -> PersistentCacheName != NULL)
    {
      delete [] control -> PersistentCacheName;
    }

    control -> PersistentCacheName = NULL;

    if (strncasecmp(cacheName, "none", strlen("none")) == 0)
    {
      #ifdef TEST
      *logofs << "Loop: No cache file selected by remote proxy.\n"
              << logofs_flush;
      #endif
    }
    else if (strlen(cacheName) != MD5_LENGTH * 2 + 3 ||
                 *(cacheName + MD5_LENGTH * 2 + 2) != ' ')
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Invalid cache file name '"
              << cacheName << "' provided by remote proxy.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Invalid cache file name '"
           << cacheName << "' provided by remote proxy.\n";

      HandleCleanup();
    }
    else
    {
      //
      // It is "C-" + 32 + "\0".
      //

      control -> PersistentCacheName = new char[MD5_LENGTH * 2 + 3];

      *(cacheName + MD5_LENGTH * 2 + 2) = '\0';

      strcpy(control -> PersistentCacheName, cacheName);

      #ifdef TEST
      *logofs << "Loop: Cache file '" << control -> PersistentCacheName
              << "' selected by remote proxy.\n" << logofs_flush;
      #endif
    }
  }
  else
  {
    #ifdef TEST
    *logofs << "Loop: Going to receive the list of remote caches.\n"
            << logofs_flush;
    #endif

    SetCaches();

    int size = ((MD5_LENGTH * 2 + 2) + strlen(",")) * DEFAULT_REMOTE_CACHE_ENTRIES +
                   strlen("cachelist=") + strlen(" ") + 1;

    char *buffer = new char[size];

    int result = ReadRemoteData(fd, buffer, size - 1, ' ');

    if (result <= 0)
    {
      delete [] buffer;

      return result;
    }

    #ifdef TEST
    *logofs << "Loop: Read list of caches from remote side as '"
            << buffer << "'.\n" << logofs_flush;
    #endif

    //
    // Prepare the buffer. What we want is a list
    // like "cache1,cache2,cache2" terminated by
    // null.
    //

    *(buffer + strlen(buffer) - 1) = '\0';

    if (strncasecmp(buffer, "cachelist=", strlen("cachelist=")) != 0)
    {
      #ifdef PANIC
      *logofs << "Loop: Wrong format for list of cache files "
              << "read from FD#" << fd << ".\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Wrong format for list of cache files.\n";

      delete [] buffer;

      return -1;
    }

    control -> PersistentCacheName = GetLastCache(buffer, control -> PersistentCachePath);

    //
    // Get rid of list of caches.
    //

    delete [] buffer;
  }

  return 1;
}

int ReadForwarderVersion(int fd)
{
  #ifdef TEST
  *logofs << "Loop: Going to negotiate the forwarder version.\n"
          << logofs_flush;
  #endif

  //
  // Check if we actually expect the session cookie.
  //

  if (*authCookie == '\0')
  {
    #ifdef TEST
    *logofs << "Loop: No authentication cookie required "
            << "from FD#" << fd << ".\n" << logofs_flush;
    #endif

    return 1;
  }

  char options[DEFAULT_REMOTE_OPTIONS_LENGTH];

  int result = ReadRemoteData(fd, options, sizeof(options), ' ');

  if (result <= 0)
  {
    return result;
  }

  #ifdef TEST
  *logofs << "Loop: Received forwarder version string '" << options
          << "' from FD#" << fd << ".\n" << logofs_flush;
  #endif

  if (strncmp(options, "NXSSH-", strlen("NXSSH-")) != 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Parse error in forwarder options string '"
            << options << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Parse error in forwarder options string '"
         << options << "'.\n";

    return -1;
  }

  //
  // Accept whatever forwarder version.
  //

  sscanf(options, "NXSSH-%i.%i.%i", &(control -> RemoteVersionMajor),
             &(control -> RemoteVersionMinor), &(control -> RemoteVersionPatch));

  #ifdef TEST
  *logofs << "Loop: Read forwarder version '" << control -> RemoteVersionMajor
          << "." << control -> RemoteVersionMinor << "." << control -> RemoteVersionPatch
          << "'.\n" << logofs_flush;
  #endif

  return 1;
}

int ReadForwarderOptions(int fd)
{
  //
  // Get the forwarder cookie.
  //

  if (*authCookie == '\0')
  {
    #ifdef TEST
    *logofs << "Loop: No authentication cookie required "
            << "from FD#" << fd << ".\n" << logofs_flush;
    #endif

    return 1;
  }

  char options[DEFAULT_REMOTE_OPTIONS_LENGTH];

  int result = ReadRemoteData(fd, options, sizeof(options), ' ');

  if (result <= 0)
  {
    return result;
  }

  #ifdef TEST
  *logofs << "Loop: Received forwarder options string '"
          << options << "' from FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  if (ParseForwarderOptions(options) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Couldn't negotiate a valid "
            << "cookie with the NX forwarder.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Couldn't negotiate a valid "
         << "cookie with the NX forwarder.\n";

    return -1;
  }

  return 1;
}

int ReadRemoteData(int fd, char *buffer, int size, char stop)
{
  #ifdef TEST
  *logofs << "Loop: Going to read remote data from FD#"
          << fd << ".\n" << logofs_flush;
  #endif

  if (size >= MAXIMUM_REMOTE_OPTIONS_LENGTH)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Maximum remote options buffer limit exceeded.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Maximum remote options buffer limit exceeded.\n";

    HandleCleanup();
  }

  while (remotePosition < (size - 1))
  {
    int result = read(fd, remoteData + remotePosition, 1);

    if (result <= 0)
    {
      if (result == -1)
      {
        if (EGET() == EAGAIN)
        {
          #ifdef TEST
          *logofs << "Loop: Reading data from FD#" << fd
                  << " would block.\n" << logofs_flush;
          #endif

          return 0;
        }
        else if (EGET() == EINTR)
        {
          if (CheckAbort() != 0)
          {
            return -1;
          }

          continue;
        }
      }

      #ifdef TEST
      *logofs << "Loop: Error reading data from FD#"
              << fd << ".\n" << logofs_flush;
      #endif

      return -1;
    }
    else if (*(remoteData + remotePosition) == stop)
    {
      #ifdef TEST
      *logofs << "Loop: Read stop character from FD#"
              << fd << ".\n" << logofs_flush;
      #endif

      remotePosition++;

      //
      // Copy the fake terminating null
      // in the buffer.
      //

      *(remoteData + remotePosition) = '\0';

      memcpy(buffer, remoteData, remotePosition + 1);

      #ifdef TEST
      *logofs << "Loop: Remote string '" << remoteData
              << "' read from FD#" << fd << ".\n"
              << logofs_flush;
      #endif

      int t = remotePosition;

      remotePosition = 0;

      return t;
    }
    else
    {
      //
      // Make sure string received
      // from far end is printable.
      //

      if (isgraph(*(remoteData + remotePosition)) == 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Non printable character decimal '"
                << (unsigned int) *(remoteData + remotePosition)
                << "' received in remote data from FD#"
                << fd << ".\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Non printable character decimal '"
                << (unsigned int) *(remoteData + remotePosition)
                << "' received in remote data from FD#"
                << fd << ".\n" << logofs_flush;

        *(remoteData + remotePosition) = ' ';
      }

      #ifdef DEBUG
      *logofs << "Loop: Read a further character "
              << "from FD#" << fd << ".\n"
              << logofs_flush;
      #endif

      remotePosition++;
    }
  }

  *(remoteData + remotePosition) = '\0';

  #ifdef TEST
  *logofs << "Loop: Stop character missing from FD#"
          << fd << " after " << remotePosition 
          << " characters read in string " << remoteData
          << ".\n" << logofs_flush;
  #endif

  memcpy(buffer, remoteData, remotePosition);

  remotePosition = 0;

  return -1;
}

int WriteLocalData(int fd, const char *buffer, int size)
{
  int position = 0;

  while (position < size)
  {
    int result = write(fd, buffer + position, size - position);

    if (result <= 0)
    {
      if (result < 0 && EGET() == EINTR)
      {
        continue;
      }

      #ifdef TEST
      *logofs << "Loop: Error writing data to FD#"
              << fd << ".\n" << logofs_flush;
      #endif

      return -1;
    }

    position += result;
  }

  return position;
}

//
// Parse the string passed by calling process in
// the environment. This is not necessarily the
// content of DISPLAY variable, but can be the
// parameters passed when creating the process
// or thread.
//

int ParseEnvironmentOptions(const char *env, int force)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  //
  // Be sure we have a parameters repository
  // and a context to jump into because this
  // can be called before creating the proxy.
  //

  if (control == NULL)
  {
    control = new Control();
  }

  if (setjmp(context) == 1)
  {
    #ifdef TEST
    *logofs << "Loop: Out of the long jump while parsing "
            << "the environment options.\n"
            << logofs_flush;
    #endif
    
    return -1;
  }

  if (force == 0 && parsedOptions == 1)
  {
    #ifdef TEST
    *logofs << "Loop: Skipping a further parse of environment "
            << "options string '" << (env != NULL ? env : "")
            << "'.\n" << logofs_flush;
    #endif

    return 1;
  }

  if (env == NULL || *env == '\0')
  {
    #ifdef TEST
    *logofs << "Loop: Nothing to do with empty environment "
            << "options string '" << (env != NULL ? env : "")
            << "'.\n" << logofs_flush;
    #endif

    return 0;
  }

  #ifdef TEST
  *logofs << "Loop: Going to parse the environment options "
          << "string '" << env << "'.\n"
          << logofs_flush;
  #endif

  parsedOptions = 1;

  //
  // Copy the string passed as parameter
  // because we need to modify it.
  //

  char opts[DEFAULT_DISPLAY_OPTIONS_LENGTH];

  #ifdef VALGRIND

  memset(opts, '\0', DEFAULT_DISPLAY_OPTIONS_LENGTH);

  #endif

  if (strlen(env) >= DEFAULT_DISPLAY_OPTIONS_LENGTH)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Environment options string '" << env
            << "' exceeds length of " << DEFAULT_DISPLAY_OPTIONS_LENGTH
            << " characters.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Environment options string '" << env
         << "' exceeds length of " << DEFAULT_DISPLAY_OPTIONS_LENGTH
         << " characters.\n";

    return -1;
  }

  strcpy(opts, env);

  char *nextOpts = opts;

  //
  // Ensure that DISPLAY environment variable
  // (roughly) follows the X convention for
  // transport notation.
  //

  if (strncasecmp(opts, "nx/nx,:", 7) == 0 ||
          strncasecmp(opts, "nx,:", 4) == 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Parse error in options string '"
            << opts << "' at 'nx,:'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Parse error in options string '"
         << opts << "' at 'nx,:'.\n";

    return -1;
  }
  else if (strncasecmp(opts, "nx/nx,", 6) == 0)
  {
    nextOpts += 6;
  }
  else if (strncasecmp(opts, "nx,", 3) == 0)
  {
    nextOpts += 3;
  }
  else if (strncasecmp(opts, "nx:", 3) == 0)
  {
    nextOpts += 3;
  }
  else if (!force)
  {
    #ifdef TEST
    *logofs << "Loop: Ignoring host X server display string '"
            << opts << "'.\n" << logofs_flush;
    #endif

    return 0;
  }

  //
  // Save here and parse later
  // user specified option file.
  //

  char fileOptions[DEFAULT_STRING_LENGTH] = { 0 };

  //
  // The options string is intended to be a series
  // of name/value tuples in the form name=value
  // separated by the ',' character ended by a ':'
  // followed by remote NX proxy port.
  //

  char *name;
  char *value;

  value = rindex(nextOpts, ':');

  if (value != NULL)
  {
    char *check = value + 1;

    if (*check == '\0' || isdigit(*check) == 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Can't identify NX port in string '"
              << value << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Can't identify NX port in string '"
           << value << "'.\n";

      return -1;
    }

    proxyPort = atoi(check);

    //
    // Get rid of the port specification.
    //

    *value = '\0';
  }
  else if (proxyPort == DEFAULT_NX_PROXY_PORT && force == 0)
  {
    //
    // Complain only if user didn't specify
    // the port on the command line.
    //

    #ifdef PANIC
    *logofs << "Loop: PANIC! Can't identify NX port in string '"
            << opts << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Can't identify NX port in string '"
         << opts << "'.\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "Loop: Parsing options string '"
          << nextOpts << "'.\n" << logofs_flush;
  #endif

  //
  // Now all the other optional parameters.
  //

  name = strtok(nextOpts, "=");

  while (name)
  {
    value = strtok(NULL, ",");

    if (CheckArg("environment", name, value) < 0)
    {
      return -1;
    }

    if (strcasecmp(name, "options") == 0)
    {
      strncpy(fileOptions, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "display") == 0)
    {
      strncpy(displayHost, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "link") == 0)
    {
      if (control -> ProxyMode == proxy_server)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'link' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring option 'link' with value '"
             << value << "' at X server side.\n";
      }
      else if (ParseLinkOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify 'link' option in string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify 'link' option in string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "limit") == 0)
    {
      if (control -> ProxyMode == proxy_server)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'limit' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring option 'limit' with value '"
             << value << "' at X server side.\n";
      }
      else if (ParseLimitOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify option 'limit' in string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify option 'limit' in string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "type") == 0)
    {
      //
      // Type of session, for example "windows" or
      // "unix-kde" or "unix-application", etc.
      //
      // TODO: In protocol level 3 this became a client
      // side option. We must be able to handle it when
      // passed at server side for compatibility with
      // proto 2.
      //

      if (strcasecmp(value, "default") == 0)
      {
        *sessionType = '\0';
      }
      else
      {
        strncpy(sessionType, value, DEFAULT_STRING_LENGTH - 1);
      }
    }
    else if (strcasecmp(name, "listen") == 0)
    {
      if (*connectHost != '\0')
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't handle 'listen' and 'connect' parameters "
                << "at the same time.\n" << logofs_flush;

        *logofs << "Loop: PANIC! Refusing 'listen' parameter with 'connect' being '"
                << connectHost << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't handle 'listen' and 'connect' parameters "
             << "at the same time.\n";

        cerr << "Error" << ": Refusing 'listen' parameter with 'connect' being '"
             << connectHost << "'.\n";

        return -1;
      }

      listenPort = atoi(value);
    }
    else if (strcasecmp(name, "accept") == 0)
    {
      if (*connectHost != '\0')
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't handle 'accept' and 'connect' parameters "
                << "at the same time.\n" << logofs_flush;

        *logofs << "Loop: PANIC! Refusing 'accept' parameter with 'connect' being '"
                << connectHost << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't handle 'accept' and 'connect' parameters "
             << "at the same time.\n";

        cerr << "Error" << ": Refusing 'accept' parameter with 'connect' being '"
             << connectHost << "'.\n";

        return -1;
      }

      strncpy(acceptHost, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "connect") == 0)
    {
      if (*acceptHost != '\0')
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't handle 'connect' and 'accept' parameters "
                << "at the same time.\n" << logofs_flush;

        *logofs << "Loop: PANIC! Refusing 'connect' parameter with 'accept' being '"
                << acceptHost << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't handle 'connect' and 'accept' parameters "
             << "at the same time.\n";

        cerr << "Error" << ": Refusing 'connect' parameter with 'accept' being '"
             << acceptHost << "'.\n";

        return -1;
      }

      strncpy(connectHost, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "port") == 0)
    {
      connectPort = atoi(value);
    }
    else if (strcasecmp(name, "retry") == 0)
    {
      if (atoi(value) > 0)
      {
        control -> OptionProxyRetryConnect  = atoi(value);
        control -> OptionServerRetryConnect = atoi(value);
      }
      else
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Invalid value '" << value
                << "' for option '" << name << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Invalid value '" << value
             << "' for option '" << name << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "session") == 0)
    {
      //
      // Name of session was stored in NX client's
      // configuration and was ignored by NX proxy
      // until 1.5.0. It is now used to pass the
      // name of the session log.
      //

      strncpy(sessionFileName, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "errors") == 0)
    {
      //
      // The old name of the parameter was 'log'
      // but the default name for the file is
      // 'errors' so it more logical to use the
      // same name.
      //

      strncpy(logFileName, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "root") == 0)
    {
      strncpy(rootDir, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "id") == 0)
    {
      strncpy(sessionId, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "stat") == 0)
    {
      control -> CollectStatistics = 1;

      strncpy(statFileName, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "cookie") == 0)
    {
      strncpy(authCookie, value, DEFAULT_STRING_LENGTH - 1);
    }
    else if (strcasecmp(name, "nodelay") == 0)
    {
      useNoDelay = (atoi(value) > 0);
    }
    else if (strcasecmp(name, "flush") == 0)
    {
      useFlush = atoi(value);
    }
    else if (strcasecmp(name, "render") == 0)
    {
      //
      // TODO: In protocol level 3 this became a client
      // side option. We must be able to handle it when
      // passed at server side for compatibility with
      // proto 2.
      //

      useRender = atoi(value);
    }
    else if (strcasecmp(name, "taint") == 0)
    {
      if (control -> ProxyMode == proxy_server)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'taint' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring option 'taint' with value '"
             << value << "' at X server side.\n";
      }
      else
      {
        useTaint = atoi(value);
      }
    }
    else if (strcasecmp(name, "delta") == 0)
    {
      if (control -> ProxyMode == proxy_server)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'delta' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring option 'delta' with value '"
             << value << "' at X server side.\n";
      }
      else
      {
        control -> LocalDeltaCompression = (atoi(value) > 0);
      }
    }
    else if (strcasecmp(name, "data") == 0)
    {
      control -> LocalDataCompressionLevel = atoi(value);

      if (control -> LocalDataCompressionLevel == 0)
      {
        control -> LocalDataCompression = 0;
      }
      else
      {
        control -> LocalDataCompression = 1;
      }
    }
    else if (strcasecmp(name, "stream") == 0)
    {
      control -> LocalStreamCompressionLevel = atoi(value);

      if (control -> LocalStreamCompressionLevel == 0)
      {
        control -> LocalStreamCompression = 0;
      }
      else
      {
        control -> LocalStreamCompression = 1;
      }
    }
    else if (strcasecmp(name, "memory") == 0)
    {
      control -> LocalMemoryLevel = atoi(value);
    }
    else if (strcasecmp(name, "cache") == 0)
    {
      if (control -> ProxyMode == proxy_server)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'cache' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else if (ParseCacheOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify cache size for string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify cache size for string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "images") == 0)
    {
      if (control -> ProxyMode == proxy_server)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'images' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else if (ParseImagesOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify images cache size for string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify images cache size for string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "shmem") == 0)
    {
      if (control -> ProxyMode == proxy_server)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'shmem' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else if (ParseShmemOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify shmem size for string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify shmem size for string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "load") == 0)
    {
      if (control -> ProxyMode == proxy_server)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'load' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else
      {
        control -> PersistentCacheEnableLoad = atoi(value);

        if (control -> PersistentCacheEnableLoad > 0)
        {
          control -> PersistentCacheEnableLoad = 1;
        }
        else
        {
          if (control -> PersistentCacheName != NULL)
          {
            delete [] control -> PersistentCacheName;
          }

          control -> PersistentCacheName = NULL;

          control -> PersistentCacheEnableLoad = 0;
        }
      }
    }
    else if (strcasecmp(name, "save") == 0)
    {
      if (control -> ProxyMode == proxy_server)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'save' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else
      {
        control -> PersistentCacheEnableSave = atoi(value);

        if (control -> PersistentCacheEnableSave > 0)
        {
          control -> PersistentCacheEnableSave = 1;
        }
        else
        {
          if (control -> PersistentCacheName != NULL)
          {
            delete [] control -> PersistentCacheName;
          }

          control -> PersistentCacheName = NULL;

          control -> PersistentCacheEnableSave = 0;
        }
      }
    }
    else if (strcasecmp(name, "cups") == 0)
    {
      cupsPort = atoi(value);
    }
    else if (strcasecmp(name, "sync") == 0)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! No 'sync' channel in current version. "
              << "Assuming 'cups' channel.\n" << logofs_flush;
      #endif

      cerr << "Warning" << ": No 'sync' channel in current version. "
           << "Assuming 'cups' channel.\n";

      cupsPort = atoi(value);
    }
    else if (strcasecmp(name, "keybd") == 0)
    {
      keybdPort = atoi(value);
    }
    else if (strcasecmp(name, "samba") == 0)
    {
      sambaPort = atoi(value);
    }
    else if (strcasecmp(name, "media") == 0)
    {
      mediaPort = atoi(value);
    }
    else if (strcasecmp(name, "http") == 0)
    {
      httpPort = atoi(value);
    }
    else if (strcasecmp(name, "timeout") == 0)
    {
      int timeout = atoi(value);

      if (timeout == 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Disabling timeout on broken proxy connection.\n"
                << logofs_flush;
        #endif

        control -> ProxyTimeout = 0;
      }
      else if ((timeout * 1000) >= control -> PingTimeout)
      {
        control -> ProxyTimeout = timeout * 1000;
      }
      else
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'timeout' with "
                << "invalid value '" << value << "'.\n"
                << logofs_flush;
        #endif
      }
    }
    else if (strcasecmp(name, "cleanup") == 0)
    {
      int cleanup = atoi(value);

      if (cleanup == 0)
      {
        control -> CleanupTimeout = 0;
      }
      else if (cleanup > 0)
      {
        control -> CleanupTimeout = cleanup * 1000;
      }
      else
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'cleanup' with "
                << "invalid value '" << value << "'.\n"
                << logofs_flush;
        #endif
      }
    }
    else if (strcasecmp(name, "pack") == 0)
    {
      if (control -> ProxyMode == proxy_server)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring option 'pack' with value '"
                << value << "' at X server side.\n" << logofs_flush;
        #endif
      }
      else if (ParsePackOption(value) < 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't identify pack method for string '"
                << value << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't identify pack method for string '"
             << value << "'.\n";

        return -1;
      }
    }
    else if (strcasecmp(name, "core") == 0)
    {
      control -> EnableCoreDumpOnAbort = (atoi(value) > 0);
    }
    else if (strcasecmp(name, "kbtype") == 0 ||
                 strcasecmp(name, "geometry") == 0 ||
                     strcasecmp(name, "fullscreen") == 0 ||
                         strcasecmp(name, "rdpcolors") == 0 ||
                             strcasecmp(name, "rdpcache") == 0 ||
                                 strcasecmp(name, "rootless") == 0)
    {
      //
      // Don't do anything. These options
      // are used by agents.
      //
    }
    else
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Ignoring unknown option '"
              << name << "' with value '" << value << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Warning" << ": Ignoring unknown option '"
           << name << "' with value '" << value << "'.\n";
    }

    name = strtok(NULL, "=");

  } // End of while (name) ...

  #ifdef TEST
  *logofs << "Loop: Completed parsing of string '"
          << env << "'.\n" << logofs_flush;
  #endif

  if (*fileOptions != '\0')
  {
    #ifdef TEST
    *logofs << "Loop: Reading options from '" << fileOptions
            << "'.\n" << logofs_flush;
    #endif

    if (ParseFileOptions(fileOptions) < 0)
    {
      return -1;
    }
  }

  //
  // If port where proxy is acting as an X server
  // was not specified assume the same port where
  // proxy is listening for the remote peer.
  //

  if (xPort == DEFAULT_NX_X_PORT)
  {
    xPort = proxyPort;
  }

  return 1;
}

//
// Parse the command line options passed by user when
// running proxy in stand alone mode. Note that passing
// parameters this way is strongly discouraged. These
// command line switch can change (and they do often).
// Please, use the form "option=value" instead and set
// the DISPLAY environment variable.
//

int ParseCommandLineOptions(int argc, const char **argv)
{
  //
  // Be sure log file is valid.
  //

  if (logofs == NULL)
  {
    logofs = &cerr;
  }

  if (setjmp(context) == 1)
  {
    #ifdef TEST
    *logofs << "Loop: Out of the long jump while parsing "
            << "the command line options.\n"
            << logofs_flush;
    #endif
    
    return -1;
  }

  //
  // Be sure we have a parameters repository
  //

  if (control == NULL)
  {
    control = new Control();
  }

  if (parsedCommand == 1)
  {
    #ifdef TEST
    *logofs << "Loop: Skipping a further parse of command line options.\n"
            << logofs_flush;
    #endif

    return 1;
  }

  #ifdef TEST
  *logofs << "Loop: Going to parse the command line options.\n"
          << logofs_flush;
  #endif

  parsedCommand = 1;

  //
  // Print out arguments.
  //

  #ifdef TEST

  *logofs << "Loop: Argc is " << argc << ".\n" << logofs_flush;

  for (int argi = 0; argi < argc; argi++)
  {
    *logofs << "Loop: Argv[" << argi << "] is " << argv[argi]
            << ".\n" << logofs_flush;
  }

  #endif

  //
  // Shall use getopt here.
  //

  for (int argi = 1; argi < argc; argi++)
  {
    const char *nextArg = argv[argi];

    if (*nextArg == '-')
    {
      switch (*(nextArg + 1))
      {
        case 'h':
        {
          PrintUsageInfo(nextArg, 0);

          return -1;
        }
        case 'C':
        {
          //
          // Start proxy in CLIENT mode.
          //

          if (!WE_SET_PROXY_MODE)
          {
            #ifdef TEST
            *logofs << "Loop: Setting local proxy mode to proxy_client.\n"
                    << logofs_flush;
            #endif

            control -> ProxyMode = proxy_client;
          }
          else if (control -> ProxyMode != proxy_client)
          {
            #ifdef PANIC
            *logofs << "Loop: PANIC! Can't redefine local proxy to "
                    << "client mode.\n" << logofs_flush;
            #endif

            cerr << "Error" << ": Can't redefine local proxy to "
                 << "client mode.\n";

            return -1;
          }

          break;
        }
        case 'S':
        {
          //
          // Start proxy in SERVER mode.
          //

          if (!WE_SET_PROXY_MODE)
          {
            #ifdef TEST
            *logofs << "Loop: Setting local proxy mode to proxy_server.\n"
                    << logofs_flush;
            #endif

            control -> ProxyMode = proxy_server;
          }
          else if (control -> ProxyMode != proxy_server)
          {
            #ifdef PANIC
            *logofs << "Loop: PANIC! Can't redefine local proxy to "
                    << "server mode.\n" << logofs_flush;
            #endif

            cerr << "Error" << ": Can't redefine local proxy to "
                 << "server mode.\n";

            return -1;
          }

          break;
        }
        case 'V':
        {
          //
          // Ignore the version argument as the
          // caller should have already managed
          // to load the right library.
          //

          GetArg(argi, argc, argv);

          break;
        }
        case 'v':
        {
          PrintVersionInfo();

          return -1;
        }
        default:
        {
          PrintUsageInfo(nextArg, 1);

          return -1;
        }
      }
    }
    else
    {
      if (nextArg)
      {
        //
        // Try to parse the option as a remote host:port
        // specification as in 'localhost:8'. Such a
        // parameter can be specified at the end of the
        // command line at the connecting side.
        //

        if (ParseHostOption(nextArg, connectHost, connectPort) > 0)
        {
          //
          // Assume port is at a proxied display offset.
          //

          proxyPort = connectPort;

          connectPort += DEFAULT_NX_PROXY_PORT_OFFSET;
        }
        else if (ParseEnvironmentOptions(nextArg, 1) < 0)
        {
          return -1;
        }
      }
    }
  }

  return 1;
}

//
// Set the variable to the values of host and
// port where this proxy is going to hook to
// an existing proxy.
//

int ParseBindOptions(char **host, int *port)
{
  if (*bindHost != '\0')
  {
    *host = bindHost;
    *port = bindPort;

    return 1;
  }
  else
  {
    return 0;
  }
}

//
// Read options from file and merge with environment.
//

int ParseFileOptions(const char *file)
{
  char *fileName;

  if (*file != '/' && *file != '.')
  {
    char *filePath = GetSessionPath();

    if (filePath == NULL)
    {
      cerr << "Error" << ": Cannot determine directory for NX option file.\n";

      HandleCleanup();
    }

    fileName = new char[strlen(filePath) + strlen("/") +
                            strlen(file) + 1];

    strcpy(fileName, filePath);

    strcat(fileName, "/");
    strcat(fileName, file);

    delete [] filePath;
  }
  else
  {
    fileName = new char[strlen(file) + 1];

    strcpy(fileName, file);
  }

  #ifdef TEST
  *logofs << "Loop: Going to read options from file '"
          << fileName << "'.\n" << logofs_flush;
  #endif

  FILE *filePtr = fopen(fileName, "r");

  if (filePtr == NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Can't open options file '" << fileName
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Can't open options file '" << fileName
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    delete [] fileName;

    return -1;
  }

  char format[DEFAULT_STRING_LENGTH];

  sprintf(format, "%%%ds", DEFAULT_DISPLAY_OPTIONS_LENGTH - 1);

  #ifdef TEST
  *logofs << "Loop: Reading with format string " << format
          << " " << "from options file '" << fileName
          << "'.\n" << logofs_flush;
  #endif

  char options[DEFAULT_DISPLAY_OPTIONS_LENGTH];

  #ifdef VALGRIND

  memset(options, '\0', DEFAULT_DISPLAY_OPTIONS_LENGTH);

  #endif

  if (fscanf(filePtr, format, options) <= 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Can't read options from file '" << fileName
            << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Can't read options from file '" << fileName
         << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

    delete [] fileName;

    return -1;
  }

  fclose(filePtr);

  #ifdef TEST
  *logofs << "Loop: Read options '" << options << "' from file '"
          << fileName << "'.\n" << logofs_flush;
  #endif

  if (ParseEnvironmentOptions(options, 1) < 0)
  {
    delete [] fileName;

    return -1;
  }

  delete [] fileName;

  return 1;
}

//
// Parse the option string passed from the
// remote proxy at startup.
//

int ParseRemoteOptions(char *opts)
{
  #ifdef TEST
  *logofs << "Loop: Going to parse the remote options "
          << "string '" << opts << "'.\n"
          << logofs_flush;
  #endif

  char *name;
  char *value;

  //
  // The options string is intended to be a series
  // of name/value tuples in the form name=value
  // separated by the ',' character.
  //

  int hasCookie = 0;
  int hasLink   = 0;
  int hasPack   = 0;
  int hasCache  = 0;
  int hasImages = 0;
  int hasDelta  = 0;
  int hasStream = 0;
  int hasData   = 0;
  int hasLimit  = 0;
  int hasFlush  = 0;
  int hasRender = 0;
  int hasTaint  = 0;
  int hasType   = 0;

  //
  // Get rid of the terminating space.
  //

  if (*(opts + strlen(opts) - 1) == ' ')
  {
    *(opts + strlen(opts) - 1) = '\0';
  }

  name = strtok(opts, "=");

  while (name)
  {
    value = strtok(NULL, ",");

    if (CheckArg("remote", name, value) < 0)
    {
      return -1;
    }

    if (strcasecmp(name, "cookie") == 0)
    {
      if (WE_PROVIDE_CREDENTIALS)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'cookie' with value '"
                << value << "' when initiating connection.\n"
                << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'cookie' with value '"
             << value << "' when initiating connection.\n";
      }
      else if (strncasecmp(authCookie, value, strlen(authCookie)) != 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Authentication cookie '" << value
                << "' doesn't match '" << authCookie << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Authentication cookie '" << value
             << "' doesn't match '" << authCookie << "'.\n";

        return -1;
      }

      hasCookie = 1;
    }
    else if (strcasecmp(name, "link") == 0)
    {
      if (control -> ProxyMode == proxy_client)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'link' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'link' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        if (*linkSpeedName != '\0' && strcasecmp(linkSpeedName, value) != 0)
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! Overriding option 'link' with new value '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Overriding option 'link' with new value '"
               << value << "'.\n";
        }

        if (ParseLinkOption(value) < 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Can't identify remote 'link' option in string '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Can't identify remote 'link' option in string '"
               << value << "'.\n";

          return -1;
        }
      }

      hasLink = 1;
    }
    else if (strcasecmp(name, "pack") == 0)
    {
      if (control -> ProxyMode == proxy_client)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'pack' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'pack' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        if (*packMethodName != '\0' && strcasecmp(packMethodName, value) != 0)
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! Overriding option 'pack' with remote value '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Overriding option 'pack' with remote value '"
               << value << "'.\n";
        }

        if (ParsePackOption(value) < 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Can't identify remote 'pack' option in string '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Can't identify remote 'pack' option in string '"
               << value << "'.\n";

          return -1;
        }
      }

      hasPack = 1;
    }
    else if (strcasecmp(name, "cache") == 0)
    {
      if (control -> ProxyMode == proxy_client)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'cache' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'cache' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        //
        // Cache size is sent as a hint of how much memory
        // the remote proxy is going to consume. A very low
        // powered thin client could choose to refuse the
        // connection.
        //

        if (ParseCacheOption(value) < 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Can't identify remote 'cache' option in string '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Can't identify remote 'cache' option in string '"
               << value << "'.\n";

          return -1;
        }
      }

      hasCache = 1;
    }
    else if (strcasecmp(name, "images") == 0)
    {
      if (control -> ProxyMode == proxy_client)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'images' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'images' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        //
        // Images cache size is sent as a hint.
        // There is no obbligation for the local
        // proxy to use the persistent cache.
        //

        if (ParseImagesOption(value) < 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Can't identify remote 'images' option in string '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Can't identify remote 'images' option in string '"
               << value << "'.\n";

          return -1;
        }
      }

      hasImages = 1;
    }
    else if (strcasecmp(name, "limit") == 0)
    {
      if (control -> ProxyMode == proxy_client)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'limit' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'limit' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        if (*bitrateLimitName != '\0' && strcasecmp(bitrateLimitName, value) != 0)
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! Overriding option 'limit' with new value '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Overriding option 'limit' with new value '"
               << value << "'.\n";
        }

        if (ParseLimitOption(value) < 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Can't identify 'limit' option in string '"
                  << value << "'.\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Can't identify 'limit' option in string '"
               << value << "'.\n";

          return -1;
        }
      }

      hasLimit = 1;
    }
    else if (strcasecmp(name, "flush") == 0)
    {
      if (control -> ProxyMode == proxy_client)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'flush' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'flush' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        //
        // Follow client side suggestion only
        // if user did not specify a value.
        //

        if (useFlush == -1)
        {
          useFlush = atoi(value);
        }
      }

      hasFlush = 1;
    }
    else if (strcasecmp(name, "render") == 0)
    {
      //
      // Became a client side option starting
      // from protocol level 3.
      //

      if (control -> ProxyMode == proxy_client)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'render' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'render' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        //
        // This was not negotiated in protocol
        // level 2. In this case ignore client
        // settings to guarantee compatibility.
        //

        if (useRender == -1)
        {
          useRender = atoi(value);
        }
      }

      hasRender = 1;
    }
    else if (strcasecmp(name, "taint") == 0)
    {
      if (control -> ProxyMode == proxy_client)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'taint' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'taint' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        useTaint = atoi(value);
      }

      hasTaint = 1;
    }
    else if (strcasecmp(name, "type") == 0)
    {
      if (control -> ProxyMode == proxy_client)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'type' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'type' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        //
        // This was not negotiated in protocol
        // level 2. Silently override the local
        // settings. This is safe as old proxy
        // versions don't pass it.
        //

        if (strcasecmp(value, "default") == 0)
        {
          *sessionType = '\0';
        }
        else
        {
          strncpy(sessionType, value, DEFAULT_STRING_LENGTH - 1);
        }
      }

      hasType = 1;
    }
    else if (strcasecmp(name, "delta") == 0)
    {
      if (control -> ProxyMode == proxy_client)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Ignoring remote option 'delta' with value '"
                << value << "' at X client side.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Ignoring remote option 'delta' with value '"
             << value << "' at X client side.\n";
      }
      else
      {
        control -> RemoteDeltaCompression = (atoi(value) > 0);

        //
        // Follow for delta compression the
        // same settings as the client proxy.
        //

        control -> LocalDeltaCompression  = control -> RemoteDeltaCompression;
      }

      hasDelta = 1;
    }
    else if (strcasecmp(name, "stream") == 0)
    {
      //
      // If remote side didn't choose its own
      // stream compression level then assume
      // local settings.
      //

      if (strcasecmp(value, "default") == 0)
      {
        //
        // This applies only at client side.
        //

        control -> RemoteStreamCompression =
            control -> LocalStreamCompression;

        control -> RemoteStreamCompressionLevel =
            control -> LocalStreamCompressionLevel;
      }
      else
      {
        control -> RemoteStreamCompressionLevel = atoi(value);

        if (control -> RemoteStreamCompressionLevel > 0)
        {
          control -> RemoteStreamCompression = 1;
        }
        else
        {
          control -> RemoteStreamCompression = 0;
        }

        if (control -> LocalStreamCompressionLevel < 0)
        {
          control -> LocalStreamCompressionLevel = atoi(value);

          if (control -> LocalStreamCompressionLevel > 0)
          {
            control -> LocalStreamCompression = 1;
          }
          else
          {
            control -> LocalStreamCompression = 0;
          }
        }
      }

      hasStream = 1;
    }
    else if (strcasecmp(name, "data") == 0)
    {
      //
      // Apply the same to data compression level.
      //

      if (strcasecmp(value, "default") == 0)
      {
        control -> RemoteDataCompression =
            control -> LocalDataCompression;

        control -> RemoteDataCompressionLevel =
            control -> LocalDataCompressionLevel;
      }
      else
      {
        control -> RemoteDataCompressionLevel = atoi(value);

        if (control -> RemoteDataCompressionLevel > 0)
        {
          control -> RemoteDataCompression = 1;
        }
        else
        {
          control -> RemoteDataCompression = 0;
        }

        if (control -> LocalDataCompressionLevel < 0)
        {
          control -> LocalDataCompressionLevel = atoi(value);

          if (control -> LocalDataCompressionLevel > 0)
          {
            control -> LocalDataCompression = 1;
          }
          else
          {
            control -> LocalDataCompression = 0;
          }
        }
      }

      hasData = 1;
    }
    else
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Ignoring unknown remote option '"
              << name << "' with value '" << value << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Warning" << ": Ignoring unknown remote option '"
           << name << "' with value '" << value << "'.\n";
    }

    name = strtok(NULL, "=");

  } // End of while (name) ...

  //
  // If we are client side, we need remote 'stream'
  // and 'data' options. If we are server, we need
  // all the above plus 'link' and some others.
  //

  char missing[DEFAULT_STRING_LENGTH];

  *missing = '\0';

  if (control -> ProxyMode == proxy_client)
  {
    if (hasStream == 0)
    {
      strcpy(missing, "stream");
    }
    else if (hasData == 0)
    {
      strcpy(missing, "data");
    }
  }
  else
  {
    if (hasLink == 0)
    {
      strcpy(missing, "link");
    }
    else if (hasCache == 0)
    {
      strcpy(missing, "cache");
    }
    else if (hasPack == 0)
    {
      strcpy(missing, "pack");
    }
    else if (hasDelta == 0)
    {
      strcpy(missing, "delta");
    }
    else if (hasStream == 0)
    {
      strcpy(missing, "stream");
    }
    else if (hasData == 0)
    {
      strcpy(missing, "data");
    }

    if (control -> isProtoStep3() == 1)
    {
      //
      // Don't complain if 'flush', 'render'
      // and 'taint' options are not passed.
      //

      if (hasType == 0)
      {
        strcpy(missing, "type");
      }
      else if (hasImages == 0)
      {
        strcpy(missing, "images");
      }
    }
  }

  if (!WE_PROVIDE_CREDENTIALS)
  {
    //
    // Can be that user doesn't have requested to
    // check the authorization cookie provided by
    // the connecting peer.
    //

    if (hasCookie == 0 && *authCookie != '\0')
    {
      strcpy(missing, "cookie");
    }
  }

  if (*missing != '\0')
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! The remote peer didn't specify the option '"
            << missing << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": The remote peer didn't specify the option '"
         << missing << "'.\n";

    return -1;
  }

  return 1;
}

//
// Parse the cookie provided by the NX proxy
// connection forwarder.
//

int ParseForwarderOptions(char *opts)
{
  #ifdef TEST
  *logofs << "Loop: Going to parse the forwarder options "
          << "string '" << opts << "'.\n"
          << logofs_flush;
  #endif

  char *name;
  char *value;

  int hasCookie = 0;

  //
  // Get rid of the terminating space.
  //

  if (*(opts + strlen(opts) - 1) == ' ')
  {
    *(opts + strlen(opts) - 1) = '\0';
  }

  name = strtok(opts, "=");

  while (name)
  {
    value = strtok(NULL, ",");

    if (CheckArg("forwarder", name, value) < 0)
    {
      return -1;
    }

    if (strcasecmp(name, "cookie") == 0)
    {
      if (strncasecmp(authCookie, value, strlen(authCookie)) != 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! The NX forwarder cookie '" << value
                << "' doesn't match '" << authCookie << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": The NX forwarder cookie '" << value
             << "' doesn't match '" << authCookie << "'.\n";

        return -1;
      }

      hasCookie = 1;
    }
    else
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Ignoring unknown forwarder option '"
              << name << "' with value '" << value << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Warning" << ": Ignoring unknown forwarder option '"
           << name << "' with value '" << value << "'.\n";
    }

    name = strtok(NULL, "=");

  } // End of while (name) ...

  if (hasCookie == 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! The NX forwarder didn't provide "
            << "the authentication cookie.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": The NX forwarder didn't provide "
         << "the authentication cookie.\n";

    return -1;
  }

  return 1;
}

int SetCore()
{
  #ifdef COREDUMPS

  rlimit rlim;

  if (getrlimit(RLIMIT_CORE, &rlim))
  {
    #ifdef TEST
    *logofs << "Cannot read RLIMIT_CORE. Error is '"
            << ESTR() << "'.\n" << logofs_flush;
    #endif

    return -1;
  }

  if (rlim.rlim_cur < rlim.rlim_max)
  {
    rlim.rlim_cur = rlim.rlim_max;

    if (setrlimit(RLIMIT_CORE, &rlim))
    {
      #ifdef TEST
      *logofs << "Loop: Cannot read RLIMIT_CORE. Error is '"
              << ESTR() << "'.\n" << logofs_flush;
      #endif

      return -2;
    }
  }

  #ifdef TEST
  *logofs << "Loop: Set RLIMIT_CORE to "<< rlim.rlim_max
          << ".\n" << logofs_flush;
  #endif

  #endif // #ifdef COREDUMPS

  return 1;
}

char *GetLastCache(char *listBuffer, const char *searchPath)
{
  if (listBuffer == NULL || searchPath == NULL ||
          strncmp(listBuffer, "cachelist=", strlen("cachelist=")) != 0)
  {
    #ifdef TEST
    *logofs << "Loop: Invalid parameters '" << listBuffer << "' and '"
            << (searchPath != NULL ? searchPath : "")
            << "'. Can't select any cache.\n" << logofs_flush;
    #endif

    return NULL;
  }

  char *selectedName = new char[MD5_LENGTH * 2 + 3];

  *selectedName = '\0';

  char *localPrefix;
  char *remotePrefix;

  if (control -> ProxyMode == proxy_client)
  {
    localPrefix  = "C-";
    remotePrefix = "S-";
  }
  else
  {
    localPrefix  = "S-";
    remotePrefix = "C-";
  }

  //
  // Get rid of prefix.
  //

  listBuffer += strlen("cachelist=");

  char *fileName;

  fileName = strtok(listBuffer, ",");

  //
  // It is "/path/to/file" + "/" + "C-" + 32 + "\0".
  //

  char fullPath[strlen(searchPath) + MD5_LENGTH * 2 + 4];

  time_t selectedTime = 0;

  struct stat fileStat;

  while (fileName)
  {
    if (strncmp(fileName, "none", strlen("none")) == 0)
    {
      #ifdef TEST
      *logofs << "Loop: No cache files seem to be available.\n"
              << logofs_flush;
      #endif

      delete [] selectedName;

      return NULL;
    }
    else if (strlen(fileName) != MD5_LENGTH * 2 + 2 ||
                 strncmp(fileName, remotePrefix, 2) != 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Bad cache file name '"
               << fileName << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Bad cache file name '"
           << fileName << "'.\n";

      delete [] selectedName;

      HandleCleanup();
    }

    #ifdef TEST
    *logofs << "Loop: Parsing remote cache name '"
            << fileName << "'.\n" << logofs_flush;
    #endif

    //
    // Prefix, received as "S-", becomes
    // "C-" and viceversa.
    //

    *fileName = *localPrefix;

    strcpy(fullPath, searchPath);
    strcat(fullPath, "/");
    strcat(fullPath, fileName);

    if (stat(fullPath, &fileStat) == 0)
    {
      #ifdef TEST
      *logofs << "Loop: Found a matching cache '"
              << fullPath << "'.\n" << logofs_flush;
      #endif

      if (fileStat.st_mtime >= selectedTime)
      {
        strcpy(selectedName, fileName);

        selectedTime = fileStat.st_mtime;
      }
    }
    #ifdef TEST
    else
    {
      *logofs << "Loop: Can't get stats of file '"
              << fullPath << "'.\n" << logofs_flush;
    }
    #endif

    fileName = strtok(NULL, ",");
  }

  if (*selectedName != '\0')
  {
    return selectedName;
  }
  else
  {
    delete [] selectedName;

    return NULL;
  }
}

char *GetRootPath()
{
  if (*rootDir == '\0')
  {
    strcpy(rootDir, getenv("HOME"));

    if (*rootDir == '\0')
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! No environment variable for HOME.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": No environment variable for HOME.\n";

      return NULL;
    }

    strcat(rootDir, "/.nx");

    //
    // Create NX root directory.
    //

    struct stat dirStat;

    if ((stat(rootDir, &dirStat) == -1) && (EGET() == ENOENT))
    {
      if (mkdir(rootDir, 0777) < 0 && (EGET() != EEXIST))
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't create directory '" << rootDir
                << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Can't create directory '" << rootDir
             << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

        return NULL;
      }
    }

    #ifdef TEST
    *logofs << "Loop: Root of NX directory structure is '"
            << rootDir << "'.\n" << logofs_flush;
    #endif
  }

  char *rootPath = new char[strlen(rootDir) + 1];

  strcpy(rootPath, rootDir);

  return rootPath;
}

char *GetCachePath()
{
  char *rootPath = GetRootPath();

  if (rootPath == NULL)
  {
    return NULL;
  }

  char *cachePath;

  if (*sessionType != '\0')
  {
    cachePath = new char[strlen(rootPath) + strlen("/cache-") +
                             strlen(sessionType) + 1];
  }
  else
  {
    cachePath = new char[strlen(rootPath) + strlen("/cache") + 1];
  }

  strcpy(cachePath, rootPath);

  if (*sessionType != '\0')
  {
    strcat(cachePath, "/cache-");

    strcat(cachePath, sessionType);
  }
  else
  {
    strcat(cachePath, "/cache");
  }

  //
  // Create cache directory if needed.
  //

  struct stat dirStat;

  if ((stat(cachePath, &dirStat) == -1) && (EGET() == ENOENT))
  {
    if (mkdir(cachePath, 0777) < 0 && (EGET() != EEXIST))
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Can't create directory '" << cachePath
              << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't create directory '" << cachePath
           << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

      delete [] rootPath;
      delete [] cachePath;

      return NULL;
    }
  }

  delete [] rootPath;

  return cachePath;
}

char *GetImagesPath()
{
  char *rootPath = GetRootPath();

  if (rootPath == NULL)
  {
    return NULL;
  }

  char *imagesPath = new char[strlen(rootPath) + strlen("/images") + 1];

  strcpy(imagesPath, rootPath);

  strcat(imagesPath, "/images");

  //
  // Create cache directory if needed.
  //

  struct stat dirStat;

  if ((stat(imagesPath, &dirStat) == -1) && (EGET() == ENOENT))
  {
    if (mkdir(imagesPath, 0777) < 0 && (EGET() != EEXIST))
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Can't create directory '" << imagesPath
              << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't create directory '" << imagesPath
           << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

      delete [] rootPath;
      delete [] imagesPath;

      return NULL;
    }
  }

  //
  // Create 16 directories in the path to
  // hold the images whose name begins with
  // the corresponding hexadecimal digit.
  //

  char *digitPath = new char[strlen(imagesPath) + 5];

  strcpy(digitPath, imagesPath);

  //
  // Image paths have format "[path][/I-c][\0]",
  // where c is the first digit of the checksum.
  //

  for (char digit = 0; digit < 16; digit++)
  {
    sprintf(digitPath + strlen(imagesPath), "/I-%01X", digit);

    if ((stat(digitPath, &dirStat) == -1) && (EGET() == ENOENT))
    {
      if (mkdir(digitPath, 0777) < 0 && (EGET() != EEXIST))
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't create directory '" << digitPath
                << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Can't create directory '" << digitPath
             << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

        delete [] rootPath;
        delete [] imagesPath;
        delete [] digitPath;

        return NULL;
      }
    }
  }

  delete [] rootPath;
  delete [] digitPath;

  return imagesPath;
}

char *GetSessionPath()
{
  if (*sessionDir == '\0')
  {
    char *rootPath = GetRootPath();

    if (rootPath == NULL)
    {
      return NULL;
    }

    strcpy(sessionDir, rootPath);

    if (control -> ProxyMode == proxy_client)
    {
      strcat(sessionDir, "/C-");
    }
    else
    {
      strcat(sessionDir, "/S-");
    }

    if (*sessionId == '\0')
    {
      char port[DEFAULT_STRING_LENGTH];

      sprintf(port, "%d", proxyPort);

      strcpy(sessionId, port);
    }

    strcat(sessionDir, sessionId);

    struct stat dirStat;

    if ((stat(sessionDir, &dirStat) == -1) && (EGET() == ENOENT))
    {
      if (mkdir(sessionDir, 0777) < 0 && (EGET() != EEXIST))
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Can't create directory '" << sessionDir
                << ". Error is " << EGET() << " '" << ESTR() << "'.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Can't create directory '" << sessionDir
             << ". Error is " << EGET() << " '" << ESTR() << "'.\n";

        delete [] rootPath;

        return NULL;
      }
    }

    #ifdef TEST
    *logofs << "Loop: Root of NX session is '" << sessionDir
            << "'.\n" << logofs_flush;
    #endif

    delete [] rootPath;
  }

  char *sessionPath = new char[strlen(sessionDir) + 1];

  strcpy(sessionPath, sessionDir);

  return sessionPath;
}

//
// Identify requested link characteristics
// and set control parameters accordingly.
//

int ParseLinkOption(const char *opt)
{
  //
  // Reset user's selection and check if
  // the newly requested value is correct.
  //

  strcpy(linkSpeedName, opt);

  //
  // Normalize user input.
  //

  if (strcasecmp(opt, "33k") == 0 ||
      strcasecmp(opt, "56k") == 0)
  {
    strcpy(linkSpeedName, "modem");
  }
  else if (strcasecmp(opt, "64k")  == 0 ||
           strcasecmp(opt, "128k") == 0)
  {
    strcpy(linkSpeedName, "isdn");
  }
  else if (strcasecmp(opt, "256k") == 0 ||
           strcasecmp(opt, "640k") == 0)
  {
    strcpy(linkSpeedName, "adsl");
  }
  else if (strcasecmp(opt, "1m")  == 0 ||
           strcasecmp(opt, "2m")  == 0 ||
           strcasecmp(opt, "34m") == 0)
  {
    strcpy(linkSpeedName, "wan");
  }
  else if (strcasecmp(opt, "10m")   == 0 ||
           strcasecmp(opt, "100m")  == 0 ||
           strcasecmp(opt, "local") == 0)
  {
    strcpy(linkSpeedName, "lan");
  }

  if (strcasecmp(linkSpeedName, "modem") != 0 &&
      strcasecmp(linkSpeedName, "isdn")  != 0 &&
      strcasecmp(linkSpeedName, "adsl")  != 0 &&
      strcasecmp(linkSpeedName, "wan")   != 0 &&
      strcasecmp(linkSpeedName, "lan")   != 0)
  {
    return -1;
  }

  return 1;
}

int ParsePackOption(const char *opt)
{
  #ifdef DEBUG
  *logofs << "Loop: Pack method is " << packMethod
          << " quality is " << packQuality << ".\n"
          << logofs_flush;
  #endif

  #ifdef DEBUG
  *logofs << "Loop: Parsing pack method '" << opt
          << "'.\n" << logofs_flush;
  #endif

  //
  // We need to check 'png-jpeg' before 'png'
  // otherwise we'll always match the latter.
  //

  if (strcasecmp(opt, "0") == 0 ||
          strcasecmp(opt, "nopack") == 0 ||
              strcasecmp(opt, "no-pack") == 0)
  {
    packMethod = NO_PACK;
  }
  else if (strcasecmp(opt, "8") == 0)
  {
    packMethod = PACK_MASKED_8_COLORS;
  }
  else if (strcasecmp(opt, "64") == 0)
  {
    packMethod = PACK_MASKED_64_COLORS;
  }
  else if (strcasecmp(opt, "256") == 0)
  {
    packMethod = PACK_MASKED_256_COLORS;
  }
  else if (strcasecmp(opt, "512") == 0)
  {
    packMethod = PACK_MASKED_512_COLORS;
  }
  else if (strcasecmp(opt, "4k") == 0)
  {
    packMethod = PACK_MASKED_4K_COLORS;
  }
  else if (strcasecmp(opt, "32k") == 0)
  {
    packMethod = PACK_MASKED_32K_COLORS;
  }
  else if (strcasecmp(opt, "64k") == 0)
  {
    packMethod = PACK_MASKED_64K_COLORS;
  }
  else if (strcasecmp(opt, "256k") == 0)
  {
    packMethod = PACK_MASKED_256K_COLORS;
  }
  else if (strcasecmp(opt, "2m") == 0)
  {
    packMethod = PACK_MASKED_2M_COLORS;
  }
  else if (strcasecmp(opt, "16m") == 0)
  {
    packMethod = PACK_MASKED_16M_COLORS;
  }
  else if (strcasecmp(opt, "256-rdp") == 0)
  {
    packMethod = PACK_RDP_PLAIN_256_COLORS;
  }
  else if (strcasecmp(opt, "256-rdp-compressed") == 0)
  {
    packMethod = PACK_RDP_COMPRESSED_256_COLORS;
  }
  else if (strcasecmp(opt, "32k-rdp") == 0)
  {
    packMethod = PACK_RDP_PLAIN_32K_COLORS;
  }
  else if (strcasecmp(opt, "32k-rdp-compressed") == 0)
  {
    packMethod = PACK_RDP_COMPRESSED_32K_COLORS;
  }
  else if (strcasecmp(opt, "64k-rdp") == 0)
  {
    packMethod = PACK_RDP_PLAIN_64K_COLORS;
  }
  else if (strcasecmp(opt, "64k-rdp-compressed") == 0)
  {
    packMethod = PACK_RDP_COMPRESSED_64K_COLORS;
  }
  else if (strcasecmp(opt, "16m-rdp") == 0)
  {
    packMethod = PACK_RDP_PLAIN_16M_COLORS;
  }
  else if (strcasecmp(opt, "16m-rdp-compressed") == 0)
  {
    packMethod = PACK_RDP_COMPRESSED_16M_COLORS;
  }
  else if (strcasecmp(opt, "rfb-hextile") == 0)
  {
    packMethod = PACK_RFB_HEXTILE;
  }
  else if (strcasecmp(opt, "rfb-tight") == 0)
  {
    packMethod = PACK_RFB_TIGHT_PLAIN;
  }
  else if (strcasecmp(opt, "rfb-tight-compressed") == 0)
  {
    packMethod = PACK_RFB_TIGHT_COMPRESSED;
  }
  else if (strcasecmp(opt, "8-tight") == 0)
  {
    packMethod = PACK_TIGHT_8_COLORS;
  }
  else if (strcasecmp(opt, "64-tight") == 0)
  {
    packMethod = PACK_TIGHT_64_COLORS;
  }
  else if (strcasecmp(opt, "256-tight") == 0)
  {
    packMethod = PACK_TIGHT_256_COLORS;
  }
  else if (strcasecmp(opt, "512-tight") == 0)
  {
    packMethod = PACK_TIGHT_512_COLORS;
  }
  else if (strcasecmp(opt, "4k-tight") == 0)
  {
    packMethod = PACK_TIGHT_4K_COLORS;
  }
  else if (strcasecmp(opt, "32k-tight") == 0)
  {
    packMethod = PACK_TIGHT_32K_COLORS;
  }
  else if (strcasecmp(opt, "64k-tight") == 0)
  {
    packMethod = PACK_TIGHT_64K_COLORS;
  }
  else if (strcasecmp(opt, "256k-tight") == 0)
  {
    packMethod = PACK_TIGHT_256K_COLORS;
  }
  else if (strcasecmp(opt, "2m-tight") == 0)
  {
    packMethod = PACK_TIGHT_2M_COLORS;
  }
  else if (strcasecmp(opt, "16m-tight") == 0)
  {
    packMethod = PACK_TIGHT_16M_COLORS;
  }
  else if (strncasecmp(opt, "8-jpeg", strlen("8-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_8_COLORS;
  }
  else if (strncasecmp(opt, "64-jpeg", strlen("64-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_64_COLORS;
  }
  else if (strncasecmp(opt, "256-jpeg", strlen("256-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_256_COLORS;
  }
  else if (strncasecmp(opt, "512-jpeg", strlen("512-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_512_COLORS;
  }
  else if (strncasecmp(opt, "4k-jpeg", strlen("4k-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_4K_COLORS;
  }
  else if (strncasecmp(opt, "32k-jpeg", strlen("32k-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_32K_COLORS;
  }
  else if (strncasecmp(opt, "64k-jpeg", strlen("64k-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_64K_COLORS;
  }
  else if (strncasecmp(opt, "256k-jpeg", strlen("256k-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_256K_COLORS;
  }
  else if (strncasecmp(opt, "2m-jpeg", strlen("2m-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_2M_COLORS;
  }
  else if (strncasecmp(opt, "16m-jpeg", strlen("16m-jpeg")) == 0)
  {
    packMethod = PACK_JPEG_16M_COLORS;
  }
  else if (strncasecmp(opt, "8-png-jpeg", strlen("8-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_8_COLORS;
  }
  else if (strncasecmp(opt, "64-png-jpeg", strlen("64-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_64_COLORS;
  }
  else if (strncasecmp(opt, "256-png-jpeg", strlen("256-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_256_COLORS;
  }
  else if (strncasecmp(opt, "512-png-jpeg", strlen("512-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_512_COLORS;
  }
  else if (strncasecmp(opt, "4k-png-jpeg", strlen("4k-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_4K_COLORS;
  }
  else if (strncasecmp(opt, "32k-png-jpeg", strlen("32k-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_32K_COLORS;
  }
  else if (strncasecmp(opt, "64k-png-jpeg", strlen("64k-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_64K_COLORS;
  }
  else if (strncasecmp(opt, "256k-png-jpeg", strlen("256k-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_256K_COLORS;
  }
  else if (strncasecmp(opt, "2m-png-jpeg", strlen("2m-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_2M_COLORS;
  }
  else if (strncasecmp(opt, "16m-png-jpeg", strlen("16m-png-jpeg")) == 0)
  {
    packMethod = PACK_PNG_JPEG_16M_COLORS;
  }
  else if (strncasecmp(opt, "8-png", strlen("8-png")) == 0)
  {
    packMethod = PACK_PNG_8_COLORS;
  }
  else if (strncasecmp(opt, "64-png", strlen("64-png")) == 0)
  {
    packMethod = PACK_PNG_64_COLORS;
  }
  else if (strncasecmp(opt, "256-png", strlen("256-png")) == 0)
  {
    packMethod = PACK_PNG_256_COLORS;
  }
  else if (strncasecmp(opt, "512-png", strlen("512-png")) == 0)
  {
    packMethod = PACK_PNG_512_COLORS;
  }
  else if (strncasecmp(opt, "4k-png", strlen("4k-png")) == 0)
  {
    packMethod = PACK_PNG_4K_COLORS;
  }
  else if (strncasecmp(opt, "32k-png", strlen("32k-png")) == 0)
  {
    packMethod = PACK_PNG_32K_COLORS;
  }
  else if (strncasecmp(opt, "64k-png", strlen("64k-png")) == 0)
  {
    packMethod = PACK_PNG_64K_COLORS;
  }
  else if (strncasecmp(opt, "256k-png", strlen("256k-png")) == 0)
  {
    packMethod = PACK_PNG_256K_COLORS;
  }
  else if (strncasecmp(opt, "2m-png", strlen("2m-png")) == 0)
  {
    packMethod = PACK_PNG_2M_COLORS;
  }
  else if (strncasecmp(opt, "16m-png", strlen("16m-png")) == 0)
  {
    packMethod = PACK_PNG_16M_COLORS;
  }
  else
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Invalid pack option with value '"
            << opt << "'. Disabling pack.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Invalid pack option with value '"
         << opt << "'. Disabling pack.\n";

    packMethod = NO_PACK;
  }

  if (packMethod == NO_PACK)
  {
    strcpy(packMethodName, "no-pack");
  }
  else
  {
    strcpy(packMethodName, opt);
  }

  if (packMethod >= PACK_JPEG_8_COLORS &&
          packMethod <= PACK_JPEG_16M_COLORS)
  {
    char *dash = rindex(opt, '-') + 1;

    if (strlen(dash) != 1 || *dash < '0' || *dash > '9')
    {
      if (strcmp(dash, "jpeg") != 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Invalid pack quality with value '"
                << dash << "'. Using default.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Invalid pack quality with value '"
             << dash << "'. Using default.\n";
      }
    }
    else
    {
      packQuality = atoi(dash);

      #ifdef DEBUG
      *logofs << "Loop: Using pack quality '" << packQuality
              << "'.\n" << logofs_flush;
      #endif
    }
  }

  if (packMethod >= PACK_PNG_8_COLORS &&
          packMethod <= PACK_PNG_16M_COLORS)
  {
    char *dash = rindex(opt, '-') + 1;

    if (strlen(dash) != 1 || *dash < '0' || *dash > '9')
    {
      if (strcmp(dash, "png") != 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Invalid pack quality with value '"
                << dash << "'. Using default.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Invalid pack quality with value '"
             << dash << "'. Using default.\n";
      }
    }
    else
    {
      packQuality = atoi(dash);

      #ifdef DEBUG
      *logofs << "Loop: Using pack quality '" << packQuality
              << "'.\n" << logofs_flush;
      #endif
    }
  }

  if (packMethod >= PACK_PNG_JPEG_8_COLORS &&
          packMethod <= PACK_PNG_JPEG_16M_COLORS)
  {
    char *dash = rindex(opt, '-') + 1;

    if (strlen(dash) != 1 || *dash < '0' || *dash > '9')
    {
      if (strcmp(dash, "png") != 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Invalid pack quality with value '"
                << dash << "'. Using default.\n" << logofs_flush;
        #endif

        cerr << "Warning" << ": Invalid pack quality with value '"
             << dash << "'. Using default.\n";
      }
    }
    else
    {
      packQuality = atoi(dash);

      #ifdef DEBUG
      *logofs << "Loop: Using pack quality '" << packQuality
              << "'.\n" << logofs_flush;
      #endif
    }
  }

  return 1;
}

int ParsePackMethod(const int method, const int quality)
{
  switch (method)
  {
    case NO_PACK:
    {
      strcpy(packMethodName, "no-pack");

      break;
    }
    case PACK_MASKED_8_COLORS:
    {
      strcpy(packMethodName, "8");

      break;
    }
    case PACK_MASKED_64_COLORS:
    {
      strcpy(packMethodName, "64");

      break;
    }
    case PACK_MASKED_256_COLORS:
    {
      strcpy(packMethodName, "256");

      break;
    }
    case PACK_MASKED_512_COLORS:
    {
      strcpy(packMethodName, "512");

      break;
    }
    case PACK_MASKED_4K_COLORS:
    {
      strcpy(packMethodName, "4k");

      break;
    }
    case PACK_MASKED_32K_COLORS:
    {
      strcpy(packMethodName, "32k");

      break;
    }
    case PACK_MASKED_64K_COLORS:
    {
      strcpy(packMethodName, "64k");

      break;
    }
    case PACK_MASKED_256K_COLORS:
    {
      strcpy(packMethodName, "256k");

      break;
    }
    case PACK_MASKED_2M_COLORS:
    {
      strcpy(packMethodName, "2m");

      break;
    }
    case PACK_MASKED_16M_COLORS:
    {
      strcpy(packMethodName, "16m");

      break;
    }
    case PACK_RDP_PLAIN_256_COLORS:
    {
      strcpy(packMethodName, "256-rdp");

      break;
    }
    case PACK_RDP_COMPRESSED_256_COLORS:
    {
      strcpy(packMethodName, "256-rdp-compressed");

      break;
    }
    case PACK_RDP_PLAIN_32K_COLORS:
    {
      strcpy(packMethodName, "32k-rdp");

      break;
    }
    case PACK_RDP_COMPRESSED_32K_COLORS:
    {
      strcpy(packMethodName, "32k-rdp-compressed");

      break;
    }
    case PACK_RDP_PLAIN_64K_COLORS:
    {
      strcpy(packMethodName, "64k-rdp");

      break;
    }
    case PACK_RDP_COMPRESSED_64K_COLORS:
    {
      strcpy(packMethodName, "64k-rdp-compressed");

      break;
    }
    case PACK_RDP_PLAIN_16M_COLORS:
    {
      strcpy(packMethodName, "16m-rdp");

      break;
    }
    case PACK_RDP_COMPRESSED_16M_COLORS:
    {
      strcpy(packMethodName, "16m-rdp-compressed");

      break;
    }
    case PACK_RFB_HEXTILE:
    {
      strcpy(packMethodName, "rfb-hextile");

      break;
    }
    case PACK_RFB_TIGHT_PLAIN:
    {
      strcpy(packMethodName, "rfb-tight");

      break;
    }
    case PACK_RFB_TIGHT_COMPRESSED:
    {
      strcpy(packMethodName, "rfb-tight-compressed");

      break;
    }
    case PACK_TIGHT_8_COLORS:
    {
      strcpy(packMethodName, "8-tight");

      break;
    }
    case PACK_TIGHT_64_COLORS:
    {
      strcpy(packMethodName, "64-tight");

      break;
    }
    case PACK_TIGHT_256_COLORS:
    {
      strcpy(packMethodName, "256-tight");

      break;
    }
    case PACK_TIGHT_512_COLORS:
    {
      strcpy(packMethodName, "512-tight");

      break;
    }
    case PACK_TIGHT_4K_COLORS:
    {
      strcpy(packMethodName, "4k-tight");

      break;
    }
    case PACK_TIGHT_32K_COLORS:
    {
      strcpy(packMethodName, "32k-tight");

      break;
    }
    case PACK_TIGHT_64K_COLORS:
    {
      strcpy(packMethodName, "64k-tight");

      break;
    }
    case PACK_TIGHT_256K_COLORS:
    {
      strcpy(packMethodName, "256k-tight");

      break;
    }
    case PACK_TIGHT_2M_COLORS:
    {
      strcpy(packMethodName, "2m-tight");

      break;
    }
    case PACK_TIGHT_16M_COLORS:
    {
      strcpy(packMethodName, "16m-tight");

      break;
    }
    case PACK_JPEG_8_COLORS:
    {
      strcpy(packMethodName, "8-jpeg");

      break;
    }
    case PACK_JPEG_64_COLORS:
    {
      strcpy(packMethodName, "64-jpeg");

      break;
    }
    case PACK_JPEG_256_COLORS:
    {
      strcpy(packMethodName, "256-jpeg");

      break;
    }
    case PACK_JPEG_512_COLORS:
    {
      strcpy(packMethodName, "512-jpeg");

      break;
    }
    case PACK_JPEG_4K_COLORS:
    {
      strcpy(packMethodName, "4k-jpeg");

      break;
    }
    case PACK_JPEG_32K_COLORS:
    {
      strcpy(packMethodName, "32k-jpeg");

      break;
    }
    case PACK_JPEG_64K_COLORS:
    {
      strcpy(packMethodName, "64k-jpeg");

      break;
    }
    case PACK_JPEG_256K_COLORS:
    {
      strcpy(packMethodName, "256k-jpeg");

      break;
    }
    case PACK_JPEG_2M_COLORS:
    {
      strcpy(packMethodName, "2m-jpeg");

      break;
    }
    case PACK_JPEG_16M_COLORS:
    {
      strcpy(packMethodName, "16m-jpeg");

      break;
    }
    case PACK_PNG_8_COLORS:
    {
      strcpy(packMethodName, "8-png");

      break;
    }
    case PACK_PNG_64_COLORS:
    {
      strcpy(packMethodName, "64-png");

      break;
    }
    case PACK_PNG_256_COLORS:
    {
      strcpy(packMethodName, "256-png");

      break;
    }
    case PACK_PNG_512_COLORS:
    {
      strcpy(packMethodName, "512-png");

      break;
    }
    case PACK_PNG_4K_COLORS:
    {
      strcpy(packMethodName, "4k-png");

      break;
    }
    case PACK_PNG_32K_COLORS:
    {
      strcpy(packMethodName, "32k-png");

      break;
    }
    case PACK_PNG_64K_COLORS:
    {
      strcpy(packMethodName, "64k-png");

      break;
    }
    case PACK_PNG_256K_COLORS:
    {
      strcpy(packMethodName, "256k-png");

      break;
    }
    case PACK_PNG_2M_COLORS:
    {
      strcpy(packMethodName, "2m-png");

      break;
    }
    case PACK_PNG_16M_COLORS:
    {
      strcpy(packMethodName, "16m-png");

      break;
    }
    case PACK_PNG_JPEG_8_COLORS:
    {
      strcpy(packMethodName, "8-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_64_COLORS:
    {
      strcpy(packMethodName, "64-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_256_COLORS:
    {
      strcpy(packMethodName, "256-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_512_COLORS:
    {
      strcpy(packMethodName, "512-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_4K_COLORS:
    {
      strcpy(packMethodName, "4k-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_32K_COLORS:
    {
      strcpy(packMethodName, "32k-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_64K_COLORS:
    {
      strcpy(packMethodName, "64k-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_256K_COLORS:
    {
      strcpy(packMethodName, "256k-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_2M_COLORS:
    {
      strcpy(packMethodName, "2m-png-jpeg");

      break;
    }
    case PACK_PNG_JPEG_16M_COLORS:
    {
      strcpy(packMethodName, "16m-png-jpeg");

      break;
    }
    default:
    {
      return -1;
    }
  }

  if (quality < 0 || quality > 9)
  {
    return -1;
  }

  if (packMethod >= PACK_JPEG_8_COLORS &&
          packMethod <= PACK_JPEG_16M_COLORS)
  {
    sprintf(packMethodName + strlen(packMethodName),
                "-%d", quality);
  }
  else if (packMethod >= PACK_PNG_8_COLORS &&
               packMethod <= PACK_PNG_16M_COLORS)
  {
    sprintf(packMethodName + strlen(packMethodName),
                "-%d", quality);
  }
  else if (packMethod >= PACK_PNG_JPEG_8_COLORS &&
               packMethod <= PACK_PNG_JPEG_16M_COLORS)
  {
    sprintf(packMethodName + strlen(packMethodName),
                "-%d", quality);
  }

  packMethod  = method;
  packQuality = quality;

  return 1;
}

int SetLog()
{
  //
  // Base NX directory.
  //

  control -> RootPath = GetRootPath();

  //
  // So far we used stderr (or stdout under
  // WIN32). Now use files selected by user.
  //

  if (*statFileName == '\0')
  {
    strcpy(statFileName, "stats");

    #ifdef TEST
    *logofs << "Loop: Assuming default statistics file '"
            << statFileName << "'.\n" << logofs_flush;
    #endif
  }
  #ifdef TEST
  else
  {
    *logofs << "Loop: Name selected for statistics is '"
            << statFileName << "'.\n" << logofs_flush;
  }
  #endif

  if (OpenOutputFile(statFileName, statofs) < 0)
  {
    HandleCleanup();
  }

  #ifndef MIXED

  if (*logFileName == '\0')
  {
    strcpy(logFileName, "errors");

    #ifdef TEST
    *logofs << "Loop: Assuming default log file name '"
            << logFileName << "'.\n" << logofs_flush;
    #endif
  }
  #ifdef TEST
  else
  {
    *logofs << "Loop: Name selected for log file is '"
            << logFileName << "'.\n" << logofs_flush;
  }
  #endif

  if (OpenOutputFile(logFileName, logofs) < 0)
  {
    HandleCleanup();
  }

  //
  // By default the session log is the standard error
  // of the process. It is anyway required to set the
  // option when running inside SSH, otherwise output
  // will go to the same file as the SSH output. In
  // the NX client this is the 'sshlog' file.
  //

  if (*sessionFileName != '\0')
  {
    #ifdef TEST
    *logofs << "Loop: Name selected for session file is '"
            << sessionFileName << "'.\n" << logofs_flush;
    #endif

    if (errofs != NULL)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Unexpected value for stream errofs.\n"
              << logofs_flush;
      #endif

      cerr << "Warning" << ": Unexpected value for stream errofs.\n";
    }

    if (errsbuf != NULL)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Unexpected value for buffer errsbuf.\n"
              << logofs_flush;
      #endif

      cerr << "Warning" << ": Unexpected value for buffer errsbuf.\n";
    }

    errofs  = NULL;
    errsbuf = NULL;

    if (OpenOutputFile(sessionFileName, errofs) < 0)
    {
      HandleCleanup();
    }

    //
    // Redirect the standard error to the file.
    //

    errsbuf = cerr.rdbuf(errofs -> rdbuf());
  }

  #endif

  //
  // Enable statistics by default.
  //

  control -> CollectStatistics = 1;

  return 1;
}

int SetPorts()
{
  //
  // Three possibilities are given at X client side:
  //
  // Port <= 0: Disable port forwarding.
  // Port == 1: Use default ports.
  // Port >  1: Use the specified port.
  //
  // At X server side, user must explicitly specify the ports
  // where the connections will have to be forwarded. This is
  // both for security reasons and because, when running both
  // proxies on the same host, there is a concrete possibility
  // that, using the default ports, the connection is looped
  // back to itself.
  //

  if (cupsPort <= 0)
  {
    #ifdef TEST
    *logofs << "Loop: Disabling cups connections.\n"
            << logofs_flush;
    #endif

    cupsPort = 0;

    useCupsSocket = 0;
  }
  else
  {
    if (control -> ProxyMode == proxy_client)
    {
      if (cupsPort == 1)
      {
        cupsPort = DEFAULT_NX_CUPS_PORT_OFFSET + proxyPort;
      }

      useCupsSocket = 1;
    }
    else
    {
      if (cupsPort == 1)
      {
        //
        // Use the well-known 631/tcp port of the
        // Internet Printing Protocol.
        //

        cupsPort = 631;
      }

      useCupsSocket = 0;
    }

    #ifdef TEST
    *logofs << "Loop: Using cups port '" << cupsPort
            << "'.\n" << logofs_flush;
    #endif
  }

  if (keybdPort <= 0)
  {
    #ifdef TEST
    *logofs << "Loop: Disabling embedded keyboard connections.\n"
            << logofs_flush;
    #endif

    keybdPort = 0;

    useKeybdSocket = 0;
  }
  else
  {
    if (control -> ProxyMode == proxy_client)
    {
      if (keybdPort == 1)
      {
        keybdPort = DEFAULT_NX_KEYBD_PORT_OFFSET + proxyPort;
      }

      useKeybdSocket = 1;
    }
    else
    {
      //
      // Embedded keyboard must always be connected
      // to the display where session is running.
      // We will later check if user did specify a
      // port that matches our X server.
      //

      if (keybdPort == 1)
      {
        keybdPort = X_TCP_PORT + xPort;
      }

      useKeybdSocket = 0;
    }

    #ifdef TEST
    *logofs << "Loop: Using embedded keyboard port '" << keybdPort
            << "'.\n" << logofs_flush;
    #endif
  }

  if (sambaPort <= 0)
  {
    #ifdef TEST
    *logofs << "Loop: Disabling SMB connections.\n"
            << logofs_flush;
    #endif

    sambaPort = 0;

    useSambaSocket = 0;
  }
  else
  {
    if (control -> ProxyMode == proxy_client)
    {
      if (sambaPort == 1)
      {
        sambaPort = DEFAULT_NX_SAMBA_PORT_OFFSET + proxyPort;
      }

      useSambaSocket = 1;
    }
    else
    {
      if (sambaPort == 1)
      {
        //
        // Assume the 139/tcp port used for SMB
        // over NetBIOS over TCP.
        //

        sambaPort = 139;
      }

      useSambaSocket = 0;
    }

    #ifdef TEST
    *logofs << "Loop: Using SMB port '" << sambaPort
            << "'.\n" << logofs_flush;
    #endif
  }

  if (mediaPort <= 0)
  {
    #ifdef TEST
    *logofs << "Loop: Disabling multimedia connections.\n"
            << logofs_flush;
    #endif

    mediaPort = 0;

    useMediaSocket = 0;
  }
  else
  {
    if (control -> ProxyMode == proxy_client)
    {
      if (mediaPort == 1)
      {
        mediaPort = DEFAULT_NX_MEDIA_PORT_OFFSET + proxyPort;
      }

      useMediaSocket = 1;
    }
    else
    {
      if (mediaPort == 1)
      {
        //
        // We don't have a well-known port to
        // be used for media connections.
        //

        #ifdef PANIC
        *logofs << "Loop: PANIC! No port specified for multimedia connections.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": No port specified for multimedia connections.\n";

        HandleCleanup();
      }

      useMediaSocket = 0;
    }

    #ifdef TEST
    *logofs << "Loop: Using multimedia port '" << mediaPort
            << "'.\n" << logofs_flush;
    #endif
  }

  if (httpPort <= 0)
  {
    #ifdef TEST
    *logofs << "Loop: Disabling HTTP connections.\n"
            << logofs_flush;
    #endif

    httpPort = 0;

    useHttpSocket = 0;
  }
  else
  {
    if (control -> ProxyMode == proxy_client)
    {
      if (httpPort == 1)
      {
        httpPort = DEFAULT_NX_HTTP_PORT_OFFSET + proxyPort;
      }

      useHttpSocket = 1;
    }
    else
    {
      if (httpPort == 1)
      {
        //
        // Use the well-known 80/tcp port.
        //

        httpPort = 80;
      }

      useHttpSocket = 0;
    }

    #ifdef TEST
    *logofs << "Loop: Using HTTP port '" << httpPort
            << "'.\n" << logofs_flush;
    #endif
  }

  return 1;
}

int SetDescriptors()
{
  unsigned int limit = 0;

  #ifdef RLIMIT_NOFILE

  rlimit limits;

  if (getrlimit(RLIMIT_NOFILE, &limits) == 0)
  {
    if (limits.rlim_max == RLIM_INFINITY)
    {
      limit = 0;
    }
    else
    {
      limit = (unsigned int) limits.rlim_max;
    }
  }

  #endif

  #ifdef _SC_OPEN_MAX

  if (limit == 0)
  {
    limit = sysconf(_SC_OPEN_MAX);
  }

  #endif

  #ifdef FD_SETSIZE

  if (limit > FD_SETSIZE)
  {
    limit = FD_SETSIZE;
  }

  #endif

  #ifdef RLIMIT_NOFILE

  if (limits.rlim_cur < limit)
  {
    limits.rlim_cur = limit;

    setrlimit(RLIMIT_NOFILE, &limits);
  }

  #endif

  if (limit == 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Cannot determine number of available "
            << "file descriptors.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot determine number of available "
         << "file descriptors.\n";

    return -1;
  }

  return 1;
}

//
// Find the directory containing the caches
// matching the session type.
//

int SetCaches()
{
  if ((control -> PersistentCachePath = GetCachePath()) == NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error getting or creating the cache path.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error getting or creating the cache path.\n";

    HandleCleanup();
  }

  #ifdef TEST
  *logofs << "Loop: Path of cache files is '" << control -> PersistentCachePath
          << "'.\n" << logofs_flush;
  #endif

  return 1;
}

//
// Initialize all configuration parameters.
//

int SetParameters()
{
  //
  // Find out the type of session.
  //

  SetSession();

  //
  // Initialize the network and compression
  // parameters according to the settings
  // suggested by the user.
  //

  SetLink();

  //
  // Set compression according to link speed.
  //

  SetCompression();

  //
  // Be sure that we have a literal for current
  // cache size. Value will reflect control's
  // default unless we already parsed a 'cache'
  // option. Server side has no control on size
  // of cache but is informed at session nego-
  // tiation about how much memory is going to
  // be used.
  //

  SetStorage();

  //
  // Set size of shared memory segments.
  //

  SetShmem();

  //
  // Make adjustments to cache based
  // on pack method.
  //

  SetPack();

  //
  // Set disk-based image cache.
  //

  SetImages();

  //
  // Set CPU and bandwidth limits.
  //

  SetLimits();

  return 1;
}

//
// According to session type literal determine
// the type of X traffic that is going to be
// transported. Literals should be better
// defined in future proxy revisions.
//

int SetSession()
{
  if (strncmp(sessionType, "unix-kde", strlen("unix-kde")) == 0 ||
          strncmp(sessionType, "unix-gnome", strlen("unix-gnome")) == 0 ||
              strncmp(sessionType, "unix-desktop", strlen("unix-desktop")) == 0 ||
                  strncmp(sessionType, "unix-rootless", strlen("unix-rootless")) == 0)
  {
    control -> SessionMode = session_x;
  }
  else if (strncmp(sessionType, "unix-application", strlen("unix-application")) == 0)
  {
    control -> SessionMode = session_application;
  }
  else if (strncmp(sessionType, "win", strlen("win")) == 0)
  {
    control -> SessionMode = session_rdp;
  }
  else if (strncmp(sessionType, "vnc", strlen("vnc")) == 0)
  {
    control -> SessionMode = session_rfb;
  }
  else
  {
    //
    // If session is not recognized let's
    // assume user knows what is doing.
    //

    control -> SessionMode = session_x;
  }

  //
  // Check if the system administrator has
  // enabled the respawn of the client at
  // the end of session.
  //

  if (control -> ProxyMode == proxy_server)
  {
    struct stat fileStat;

    if (stat("/usr/NX/share/noexit", &fileStat) == 0)
    {
      #ifdef TEST
      *logofs << "Loop: Enabling respawn of client at session shutdown.\n"
              << logofs_flush;
      #endif

      control -> EnableRestartOnShutdown = 1;
    }
  }

  return 1;
}

int SetStorage()
{
  //
  // If differential compression is disabled
  // we don't need a cache at all.
  //

  if (control -> LocalDeltaCompression == 0)
  {
    control -> ClientTotalStorageSizeLimit = 0;
    control -> ServerTotalStorageSizeLimit = 0;
  }

  //
  // Set a a cache size literal.
  //

  int size = control -> getUpperStorageSizeLimit();

  if (size / 1024 > 0)
  {
    sprintf(cacheSizeName, "%dk", size / 1024);
  }
  else
  { 
    sprintf(cacheSizeName, "%d", size);
  }

  if (control -> ProxyMode == proxy_client)
  {
    control -> LocalTotalStorageSizeLimit =
        control -> ClientTotalStorageSizeLimit;

    control -> RemoteTotalStorageSizeLimit =
        control -> ServerTotalStorageSizeLimit;
  }
  else
  {
    control -> LocalTotalStorageSizeLimit =
        control -> ServerTotalStorageSizeLimit;

    control -> RemoteTotalStorageSizeLimit =
        control -> ClientTotalStorageSizeLimit;
  }

  #ifdef DEBUG
  *logofs << "Loop: Storage size limit is "
          << control -> ClientTotalStorageSizeLimit
          << " at client and "
          << control -> ServerTotalStorageSizeLimit
          << " at server.\n"
          << logofs_flush;
  #endif

  #ifdef DEBUG
  *logofs << "Loop: Storage local limit set to "
          << control -> LocalTotalStorageSizeLimit
          << " remote limit set to "
          << control -> RemoteTotalStorageSizeLimit
          << ".\n" << logofs_flush;
  #endif

  //
  // Never reserve for split store more than
  // half the memory available for messages.
  //

  if (control -> SplitTotalStorageSizeLimit > size / 2)
  {
    #ifdef TEST
    *logofs << "Loop: Reducing size of split store to "
            << size / 2 << " bytes.\n"
            << logofs_flush;
    #endif

    control -> SplitTotalStorageSizeLimit = size / 2;
  }

  //
  // Don't load render from persistent
  // cache if extension is hidden or
  // not supported by agent.
  //

  if (control -> AgentHideRender == 1)
  {
    #ifdef TEST
    *logofs << "Loop: Not loading render extension "
            << "from persistent cache.\n"
            << logofs_flush;
    #endif

    control -> PersistentCacheLoadRender = 0;
  }

  return 1;
}

int SetShmem()
{
  //
  // If not set, adjust size of the shared
  // memory segment according to size of
  // the message cache.
  //

  if (*shmemSizeName == '\0')
  {
    int size = control -> getUpperStorageSizeLimit();

    const int mega = 1048576;

    if (size > 0)
    {
      if (size <= 1 * mega)
      {
        size = 0;
      }
      else if (size <= 2 * mega)
      {
        size = 524288;
      }
      else if (size < 4 * mega)
      {
        size = 1048576;
      }
      else
      {
        size = size / 4;
      }

      if (size > 4194304)
      {
        size = 4194304;
      }

      if (size / 1024 > 0)
      {
        sprintf(shmemSizeName, "%dk", size / 1024);
      }
      else
      {
        sprintf(shmemSizeName, "%d", size);
      }

      control -> AgentShmemClientSize = size;
      control -> AgentShmemServerSize = size;
    }
    else
    {
      //
      // The delta compression is disabled.
      // Use a default segment size of 2 MB.
      //

      control -> AgentShmemServerSize = 2 * mega;
    }
  }

  //
  // TODO: In current version only X server
  // support is implemented.
  //
  // if (control -> AgentShmemClientSize >= 524288)
  // {
  //   control -> AgentShmemClient = 1;
  // }
  // else
  // {
  //   control -> AgentShmemClient = 0;
  // }
  //

  if (control -> AgentShmemServerSize >= 524288)
  {
    control -> AgentShmemServer = 1;

    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Set initial shared memory size "
            << "to " << control -> AgentShmemServerSize
            << " bytes.\n" << logofs_flush;
    #endif
  }
  else
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Disabled use of the shared memory "
            << "extension.\n" << logofs_flush;
    #endif

    control -> AgentShmemServer = 0;
  }

  return 1;
}

//
// Adjust pack method according to session type.
//

int SetPack()
{
  #ifdef TEST
  *logofs << "Loop: Setting pack with initial method "
          << packMethod << " and quality " << packQuality
          << ".\n" << logofs_flush;
  #endif

  #ifdef TEST

  int userMethod = packMethod;

  #endif

  //
  // Check if this is a RDP or RFB session and,
  // in this case, use a pack method optimized
  // for such kind of sessions. X sessions can
  // safely use the control default.
  //

  if (control -> SessionMode == session_rdp)
  {
    //
    // If a pack method was not specified, use
    // compressed bitmaps as they seems to offer
    // better performances even on LAN.
    //

    if (packMethod != PACK_RDP_COMPRESSED_256_COLORS &&
            packMethod != PACK_RDP_PLAIN_256_COLORS &&
                packMethod != PACK_RDP_PLAIN_32K_COLORS &&
                    packMethod != PACK_RDP_COMPRESSED_32K_COLORS &&
                        packMethod != PACK_RDP_PLAIN_64K_COLORS &&
                            packMethod != PACK_RDP_COMPRESSED_64K_COLORS &&
                                packMethod != PACK_RDP_PLAIN_16M_COLORS &&
                                    packMethod != PACK_RDP_COMPRESSED_16M_COLORS &&
                                        packMethod != NO_PACK)
    {
      packMethod = PACK_RDP_COMPRESSED_256_COLORS;
    }
  }
  else if (control -> SessionMode == session_rfb)
  {
    if (packMethod != PACK_RFB_HEXTILE &&
            packMethod != PACK_RFB_TIGHT_PLAIN &&
                packMethod != PACK_RFB_TIGHT_COMPRESSED &&
                    packMethod != NO_PACK)
    {
      packMethod = PACK_RFB_TIGHT_PLAIN;
    }
  }
  else if (control -> SessionMode == session_application)
  {
    //
    // At the moment packed images are not
    // supported for normal X applications
    // as this has not been built yet in
    // Xlib.
    //

    packMethod = NO_PACK;
  }

  //
  // Adjust internal settings according
  // to newly selected pack method.
  //

  ParsePackMethod(packMethod, packQuality);

  #ifdef TEST

  if (userMethod != packMethod)
  {
    *logofs << "Loop: WARNING! Overriding option 'pack'. "
            << "Setting preferred method '"
            << packMethodName << "'.\n" << logofs_flush;
  }

  #endif

  control -> AgentPackMethod  = packMethod;
  control -> AgentPackQuality = packQuality;

  //
  // Don't load messages from persistent
  // cache if packed images are disabled.
  //

  if (control -> AgentPackMethod == NO_PACK)
  {
    control -> PersistentCacheLoadPacked = 0;

    #ifdef TEST
    *logofs << "Loop: Not loading packed images "
            << "from persistent cache.\n"
            << logofs_flush;
    #endif

    control -> TokenLimit *= 2;

    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Increasing the available "
            << "tokens to " << control ->
               TokenLimit << ".\n"
            << logofs_flush;
    #endif
  }

  //
  // Note that if we are at X client side, we
  // still ignore which protocol version is
  // supported by remote end. If PNG encoding
  // is requested, we'll later need to check
  // if it's actually supported.
  //

  return 1;
}

//
// Set disk-based image cache according to
// user's wishes.
//

int SetImages()
{
  //
  // Be sure we disable cache in case of session
  // application or with link LAN. This is mainly
  // useful to disable the house-keeping process
  // as in these case we don't have split.
  //

  if (control -> SessionMode != session_x ||
          control -> LinkMode == LINK_TYPE_LAN)
  {
    #ifdef TEST
    *logofs << "Loop: Disabling images cache with session "
            << control -> SessionMode << " and link "
            << control -> LinkMode << ".\n"
            << logofs_flush;
    #endif

    sprintf(imagesSizeName, "0");

    control -> ImageCacheEnableLoad = 0;
    control -> ImageCacheEnableSave = 0;

    return 1;
  }

  int size = control -> ImageCacheDiskLimit;

  if (size / 1024 > 0)
  {
    sprintf(imagesSizeName, "%dk", size / 1024);
  }
  else
  {
    sprintf(imagesSizeName, "%d", size);
  }

  if (size > 0)
  {
    control -> ImageCacheEnableLoad = 1;
    control -> ImageCacheEnableSave = 1;

    if (control -> ProxyMode == proxy_server)
    {
      if ((control -> ImageCachePath = GetImagesPath()) == NULL)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Error getting or creating image cache path.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Error getting or creating image cache path.\n";

        HandleCleanup();
      }

      #ifdef TEST
      *logofs << "Loop: Path of image cache files is '" << control -> ImageCachePath
              << "'.\n" << logofs_flush;
      #endif
    }
  }
  else
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Disabling NX images cache.\n"
            << logofs_flush;
    #endif

    control -> ImageCacheEnableLoad = 0;
    control -> ImageCacheEnableSave = 0;
  }

  return 1;
}

int SetVersion()
{
  //
  // In theory we consider the version numbers to
  // match if the major number matches. Different
  // minor and patch levels should be compatible
  // with each other...
  //

  if (control -> RemoteVersionMajor != 1)
  {
    return -1;
  }

  //
  // ...But in practice we have special cases
  // handled here.
  //

  if (control -> RemoteVersionMinor < 1 ||
          (control -> RemoteVersionMinor == 1 &&
               control -> RemoteVersionPatch < 1))
  {
    return -1;
  }

  if (control -> RemoteVersionMinor >= 2)
  {
    if (control -> RemoteVersionMinor == 2 &&
            control -> RemoteVersionPatch <= 1)
    {
      #ifdef TEST
      *logofs << "Loop: Using NX protocol step 2.\n"
              << logofs_flush;
      #endif

      control -> setProtoStep2();
    }
    else if (control -> RemoteVersionMinor == 2 &&
                 control -> RemoteVersionPatch == 2)
    {
      #ifdef TEST
      *logofs << "Loop: Using NX protocol step 3.\n"
              << logofs_flush;
      #endif

      control -> setProtoStep3();
    }
    else if (control -> RemoteVersionMinor == 3)
    {
      #ifdef TEST
      *logofs << "Loop: Using NX protocol step 4.\n"
              << logofs_flush;
      #endif

      control -> setProtoStep4();
    }
    else if (control -> RemoteVersionMinor == 4)
    {
      #ifdef TEST
      *logofs << "Loop: Using NX protocol step 5.\n"
              << logofs_flush;
      #endif

      control -> setProtoStep5();
    }
    else
    {
      #ifdef TEST
      *logofs << "Loop: Using NX protocol step 6.\n"
              << logofs_flush;
      #endif

      control -> setProtoStep6();
    }
  }
  else
  {
    #ifdef TEST
    *logofs << "Loop: Using NX protocol step 1.\n"
            << logofs_flush;
    #endif

    control -> setProtoStep1();
  }

  return 1;
}

//
// Identify requested link characteristics
// and set control parameters accordingly.
//

int SetLink()
{
  #ifdef TEST
  *logofs << "Loop: Setting link with initial value "
          << linkSpeedName << ".\n" << logofs_flush;
  #endif

  if (*linkSpeedName == '\0')
  {
    strcpy(linkSpeedName, "lan");
  }
  
  #ifdef TEST
  *logofs << "Loop: Link speed is " << linkSpeedName
          << ".\n" << logofs_flush;
  #endif

  if (strcasecmp(linkSpeedName, "modem") == 0)
  {
    SetLinkModem();
  }
  else if (strcasecmp(linkSpeedName, "isdn") == 0)
  {
    SetLinkIsdn();
  }
  else if (strcasecmp(linkSpeedName, "adsl") == 0)
  {
    SetLinkAdsl();
  }
  else if (strcasecmp(linkSpeedName, "wan") == 0)
  {
    SetLinkWan();
  }
  else if (strcasecmp(linkSpeedName, "lan") == 0)
  {
    SetLinkLan();
  }
  else
  {
    return -1;
  }

  //
  // In the case of RDP sessions reduce the
  // karma limit and set size of TCP receive
  // buffer of path agent-to-proxy according
  // to the size of scheduled writes.
  //

  if (control -> SessionMode == session_rdp)
  {
    control -> AgentKarmaSize /= 8;

    if (control -> AgentKarmaSize < 1024)
    {
      control -> AgentKarmaSize = 1024;
    }

    #ifdef TEST
    *logofs << "Loop: Reduced agent karma size to "
            << control -> AgentKarmaSize << ".\n"
            << logofs_flush;
    #endif
  }

  //
  // Override default flush timeout set in control
  // files in case of normal X client connections.
  // Normal X clients use so many replies to make
  // queuing completely useless. Do this at client
  // side as server side will receive our value at
  // session negotiation.
  //

  if (control -> SessionMode == session_application)
  {
    if (control -> ProxyMode == proxy_client && useFlush == -1)
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "Loop: WARNING! Disabling flush timeout "
              << "with session application.\n"
              << logofs_flush;
      #endif

      control -> FrameTimeout = 0;
      control -> FlushTimeout = 0;
    }
  }
  else
  {
    //
    // Don't taint replies if mode
    // is not single X applications.
    //

    control -> AgentTaintReplies = 0;

    #ifdef TEST
    *logofs << "Loop: Overriding taint of replies with "
            << "new value " << control -> AgentTaintReplies
            << ".\n" << logofs_flush;
    #endif
  }

  #if defined(TEST) || defined(INFO)
  *logofs << "Loop: Using flush timeout " << control ->
             FlushTimeout << " with frame timeout "
          << control -> FrameTimeout << ".\n"
          << logofs_flush;
  #endif

  //
  // Now that we know the remote proxy version
  // check if we need to disable the render
  // extension because:
  // 
  // 1. The remote proxy assumes that we will
  //    do that.
  //
  // 2. The remote proxy doesn't have support
  //    for the alpha channel.
  //

  if (control -> ProxyMode == proxy_server)
  {
    if (control -> SessionMode == session_x &&
            control -> isProtoStep4() == 0)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Not using render extension "
              << "due to lack of alpha support.\n"
              << logofs_flush;
      #endif

      control -> PersistentCacheLoadRender = 0;

      control -> AgentHideRender = 1;
    }

    //
    // TODO: There are problems when running agent sessions
    // with the RENDER extension on the Solaris client. They
    // look to be caused by differences in the implementation
    // of RENDER on the Sun X server shipped by default.
    // We did not test NX with the XFree86 X server, but it
    // seems that, when using the Sun X server, even plain X
    // clients don't make use of the extension.
    //

    #ifdef __sun

    if (control -> SessionMode == session_x &&
            control -> AgentHideRender == 0)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Not using render extension "
              << "due to incompatible agent libraries.\n"
              << logofs_flush;
      #endif

      control -> AgentHideRender = 1;
    }

    #endif
  }

  //
  // Finally determine the default flush policy.
  //

  if (control -> FrameTimeout == 0 ||
          control -> FlushTimeout == 0)
  {
    control -> FlushPolicy = policy_immediate;
  }
  else
  {
    control -> FlushPolicy = policy_deferred;
  }

  #if defined(TEST) || defined(INFO)
  *logofs << "Loop: Using default flush policy '"
          << PolicyLabel(control -> FlushPolicy)
          << "'.\n" << logofs_flush;
  #endif

  return 1;
}

//
// Parameters for MODEM 28.8/33.6/56 Kbps.
//

int SetLinkModem()
{
  #ifdef TEST
  *logofs << "Loop: Setting parameters for MODEM.\n"
          << logofs_flush;
  #endif

  ParseControlParameters("modem");

  control -> LinkMode = LINK_TYPE_MODEM;

  return 1;
}

//
// Parameters for ISDN 64/128 Kbps.
//

int SetLinkIsdn()
{
  #ifdef TEST
  *logofs << "Loop: Setting parameters for ISDN.\n"
          << logofs_flush;
  #endif

  ParseControlParameters("isdn");

  control -> LinkMode = LINK_TYPE_ISDN;

  return 1;
}

//
// Parameters for ADSL 256 Kbps.
//

int SetLinkAdsl()
{
  #ifdef TEST
  *logofs << "Loop: Setting parameters for ADSL.\n"
          << logofs_flush;
  #endif

  ParseControlParameters("adsl");

  control -> LinkMode = LINK_TYPE_ADSL;

  return 1;
}

//
// Parameters for XDSL/FDDI/ATM 1/2/34 Mbps WAN.
//

int SetLinkWan()
{
  #ifdef TEST
  *logofs << "Loop: Setting parameters for WAN.\n"
          << logofs_flush;
  #endif

  ParseControlParameters("wan");

  control -> LinkMode = LINK_TYPE_WAN;

  return 1;
}

//
// Parameters for LAN 10/100 Mbps.
//

int SetLinkLan()
{
  #ifdef TEST
  *logofs << "Loop: Setting parameters for LAN.\n"
          << logofs_flush;
  #endif

  ParseControlParameters("lan");

  control -> LinkMode = LINK_TYPE_LAN;

  //
  // Don't use the coalescence timeout.
  //

  control -> FrameTimeout = 0;
  control -> FlushTimeout = 0;

  return 1;
}

//
// Identify requested link characteristics
// and set control parameters accordingly.
//

int SetCompression()
{
  if (strcasecmp(linkSpeedName, "modem") == 0)
  {
    SetCompressionModem();
  }
  else if (strcasecmp(linkSpeedName, "isdn") == 0)
  {
    SetCompressionIsdn();
  }
  else if (strcasecmp(linkSpeedName, "adsl") == 0)
  {
    SetCompressionAdsl();
  }
  else if (strcasecmp(linkSpeedName, "wan") == 0)
  {
    SetCompressionWan();
  }
  else if (strcasecmp(linkSpeedName, "lan") == 0)
  {
    SetCompressionLan();
  }
  else
  {
    return -1;
  }

  if (control -> LocalDeltaCompression < 0)
  {
    control -> LocalDeltaCompression = 1;
  }

  //
  // If we didn't set remote delta compression
  // (as it should always be the case at client
  // side) assume value of local side.
  //

  if (control -> RemoteDeltaCompression < 0)
  {
    control -> RemoteDeltaCompression =
        control -> LocalDeltaCompression;
  }

  //
  // If we didn't set remote compression levels
  // assume values of local side.
  //

  if (control -> RemoteStreamCompression < 0)
  {
    control -> RemoteStreamCompressionLevel =
        control -> LocalStreamCompressionLevel;

    if (control -> RemoteStreamCompressionLevel > 0)
    {
      control -> RemoteStreamCompression = 1;
    }
    else
    {
      control -> RemoteStreamCompression = 0;
    }
  }

  if (control -> RemoteDataCompression < 0)
  {
    control -> RemoteDataCompressionLevel =
        control -> LocalDataCompressionLevel;

    if (control -> RemoteDataCompressionLevel > 0)
    {
      control -> RemoteDataCompression = 1;
    }
    else
    {
      control -> RemoteDataCompression = 0;
    }
  }

  return 1;
}

//
// Compression for MODEM.
//

int SetCompressionModem()
{
  if (control -> LocalDataCompression < 0)
  {
    control -> LocalDataCompression      = 1;
    control -> LocalDataCompressionLevel = 6;
  }

  if (control -> LocalDataCompressionThreshold < 0)
  {
    control -> LocalDataCompressionThreshold = 32;
  }

  if (control -> LocalStreamCompression < 0)
  {
    control -> LocalStreamCompression      = 1;
    control -> LocalStreamCompressionLevel = 9;
  }

  return 1;
}

//
// Compression for ISDN.
//

int SetCompressionIsdn()
{
  if (control -> LocalDataCompression < 0)
  {
    control -> LocalDataCompression      = 1;
    control -> LocalDataCompressionLevel = 6;
  }

  if (control -> LocalDataCompressionThreshold < 0)
  {
    control -> LocalDataCompressionThreshold = 32;
  }

  if (control -> LocalStreamCompression < 0)
  {
    control -> LocalStreamCompression      = 1;
    control -> LocalStreamCompressionLevel = 9;
  }

  return 1;
}

//
// Compression for ADSL.
//

int SetCompressionAdsl()
{
  if (control -> LocalDataCompression < 0)
  {
    control -> LocalDataCompression      = 1;
    control -> LocalDataCompressionLevel = 3;
  }

  if (control -> LocalDataCompressionThreshold < 0)
  {
    control -> LocalDataCompressionThreshold = 32;
  }

  if (control -> LocalStreamCompression < 0)
  {
    control -> LocalStreamCompression      = 1;
    control -> LocalStreamCompressionLevel = 6;
  }

  return 1;
}

//
// Compression for WAN.
//

int SetCompressionWan()
{
  if (control -> LocalDataCompression < 0)
  {
    control -> LocalDataCompression      = 1;
    control -> LocalDataCompressionLevel = 1;
  }

  if (control -> LocalDataCompressionThreshold < 0)
  {
    control -> LocalDataCompressionThreshold = 32;
  }

  if (control -> LocalStreamCompression < 0)
  {
    control -> LocalStreamCompression      = 0;
    control -> LocalStreamCompressionLevel = 0;
  }

  return 1;
}

//
// Compression for LAN.
//

int SetCompressionLan()
{
  //
  // Disable delta compression if not
  // explicitly enabled.
  //

  if (control -> LocalDeltaCompression < 0)
  {
    control -> LocalDeltaCompression = 0;
  }

  if (control -> LocalDataCompression < 0)
  {
    control -> LocalDataCompression      = 0;
    control -> LocalDataCompressionLevel = 0;
  }

  if (control -> LocalDataCompressionThreshold < 0)
  {
    control -> LocalDataCompressionThreshold = 0;
  }

  if (control -> LocalStreamCompression < 0)
  {
    control -> LocalStreamCompression      = 0;
    control -> LocalStreamCompressionLevel = 0;
  }

  return 1;
}

int SetLimits()
{
  //
  // Adjust CPU limit. Set limit to 0 if
  // user explicitly disabled limits.
  //

  if (control -> LocalProcessorLimit == -1)
  {
    if (control -> LocalBitrateLimit == 0)
    {
      control -> LocalProcessorLimit = 0;
    }
    else
    {
      if (control -> ProxyMode == proxy_client)
      {
        control -> LocalProcessorLimit =
            control -> ClientProcessorLimit;
      }
      else
      {
        control -> LocalProcessorLimit =
            control -> ServerProcessorLimit;
      }
    }
  }

  //
  // Check bitrate limit.
  //

  if (control -> LocalBitrateLimit == -1)
  {
    if (control -> ProxyMode == proxy_client)
    {
      control -> LocalBitrateLimit =
          control -> ClientBitrateLimit;
    }
    else
    {
      control -> LocalBitrateLimit =
          control -> ServerBitrateLimit;
    }
  }
  else if (control -> LocalBitrateLimit > 0)
  {
    //
    // If bandwidth limit was explicitly set by
    // user, make CPU limit more restrictive at
    // X client side.
    //

    if (control -> ProxyMode == proxy_client)
    {
      control -> LocalProcessorLimit *= 2;

      if (control -> LocalProcessorLimit == 0)
      {
        control -> LocalProcessorLimit = 1;
      }
    }
  }

  //
  // Finally make CPU limits less restrictive
  // in case of RFB or RDP sessions.
  //

  if (control -> LocalProcessorLimit > 1 &&
          (control -> SessionMode == session_rdp ||
              control -> SessionMode == session_rfb))
  {
    control -> LocalProcessorLimit /= 2;
  }    

  return 1;
}

//
// Compression for connection to localhost.
//

int ParseControlParameters(const char *link)
{
  //
  // Open file and get configuration parameters.
  //

  char file[DEFAULT_STRING_LENGTH];

  strcpy(file, "control");

  strcat(file, "-");

  strcat(file, link);

  istream *stream = NULL;

  if (OpenInputFile(file, stream) < 0)
  {
    delete stream;

    HandleCleanup();
  }

  char buffer[CONTROL_PARAMETERS_FILE_LENGTH];

  int parameters[CONTROL_PARAMETERS_FILE_ENTRIES];

  int count = 0;

  //
  // A quite rudimentary way to parse a configuration
  // file. Control files, however, are just intended
  // for developers. Reading them from disk should be
  // always disabled on production machines.
  //

  while (stream -> fail() == 0 &&
             stream -> getline(buffer, DEFAULT_STRING_LENGTH))
  {
    #ifdef TEST
    *logofs << "Loop: Parsing line '" << buffer << "'.\n"
            << logofs_flush;
    #endif

    if (count < CONTROL_PARAMETERS_FILE_ENTRIES)
    {
      if (strlen(buffer) > 0)
      {
        parameters[count] = atoi(buffer + 4);
      }
      else
      {
        parameters[count] = 0;
      }
    }

    count++;
  }

  if (count > 0 && count != CONTROL_PARAMETERS_FILE_ENTRIES)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Bad control file '" << file
            << "'. Read " << count << " values instead of "
            << CONTROL_PARAMETERS_FILE_ENTRIES << ".\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Bad control file '" << file
         << "'. Read " << count << " values instead of "
         << CONTROL_PARAMETERS_FILE_ENTRIES << ".\n";

    delete stream;

    HandleCleanup();
  }

  delete stream;

  //
  // If we didn't locate the file then
  // get buffer from control class.
  //

  if (count == 0)
  {
    *file = '\0';

    //
    // LOCAL basic configuration is
    // used for both 'wan' and 'lan'.
    //

    if (!strcmp(link, "modem"))
    {
      strcpy(buffer, ControlModem);
    }
    else if (!strcmp(link, "isdn"))
    {
      strcpy(buffer, ControlIsdn);
    }
    else if (!strcmp(link, "adsl"))
    {
      strcpy(buffer, ControlAdsl);
    }
    else if (!strcmp(link, "wan"))
    {
      strcpy(buffer, ControlLocal);
    }
    else if (!strcmp(link, "lan"))
    {
      strcpy(buffer, ControlLocal);
    }

    char *line = strtok(buffer, "\n");

    while (line != NULL)
    {
      #ifdef TEST
      *logofs << "Loop: Parsing line '" << line << "'.\n"
              << logofs_flush;
      #endif

      if (strlen(line) > 0)
      {
        parameters[count] = atoi(line + 4);
      }
      else
      {
        parameters[count] = 0;
      }

      line = strtok(NULL, "\n");

      count++;
    }

    if (count != CONTROL_PARAMETERS_FILE_ENTRIES)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Bad control parameters file. "
              << "Read " << count << " values instead of "
              << CONTROL_PARAMETERS_FILE_ENTRIES << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Bad control parameters file. "
           << "Read " << count << " values instead of "
           << CONTROL_PARAMETERS_FILE_ENTRIES << ".\n";

      HandleCleanup();
    }
  }
  else
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Reading control parameters from '"
            << file << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Reading control parameters from '"
         << file << "'.\n";
  }

  //
  // Scheduling parameters.
  //

  control -> TokenBytes = parameters[7];
  control -> TokenLimit = parameters[8];

  //
  // Split options.
  //

  control -> SplitMode                  = parameters[12];
  control -> SplitTimeout               = parameters[13];
  control -> SplitDataThreshold         = parameters[14];
  control -> SplitDataPacketLimit       = parameters[15];
  control -> SplitTotalStorageSizeLimit = parameters[16];

  //
  // Timeout parameters.
  //

  if (useFlush != -1)
  {
    if (useFlush > 1)
    {
      control -> FrameTimeout = useFlush / 5;
      control -> FlushTimeout = useFlush;
    }
    else
    {
      control -> FrameTimeout = parameters[20];
      control -> FlushTimeout = parameters[21];
    }
  }
  else
  {
    control -> FrameTimeout = parameters[20];
    control -> FlushTimeout = parameters[21];
  }

  control -> MotionTimeout  = parameters[22];
  control -> PendingTimeout = parameters[23];
  control -> ShmemTimeout   = parameters[24];
  control -> PingTimeout    = parameters[25];
  control -> StartupTimeout = parameters[26];

  //
  // Transport parameters.
  //

  control -> TransportWriteThreshold         = parameters[30];
  control -> TransportXBufferThreshold       = parameters[31];
  control -> TransportProxyBufferThreshold   = parameters[32];
  control -> TransportGenericBufferThreshold = parameters[33];

  //
  // Read buffers parameters.
  //

  control -> ClientInitialReadSize  = parameters[37];
  control -> ClientMaximumReadSize  = parameters[38];
  control -> ServerInitialReadSize  = parameters[39];
  control -> ServerMaximumReadSize  = parameters[40];
  control -> ProxyInitialReadSize   = parameters[41];
  control -> ProxyMaximumReadSize   = parameters[42];
  control -> GenericInitialReadSize = parameters[43];
  control -> GenericMaximumReadSize = parameters[44];

  //
  // Socket options.
  //

  control -> OptionProxySendBuffer    = parameters[48];
  control -> OptionProxyReceiveBuffer = parameters[49];

  //
  // Set TCP_NODELAY according to the user's wish.
  //

  if (useNoDelay != -1)
  {
    control -> OptionProxyClientNoDelay = useNoDelay;
    control -> OptionProxyServerNoDelay = useNoDelay;
  }
  else
  {
    control -> OptionProxyClientNoDelay = parameters[50];
    control -> OptionProxyServerNoDelay = parameters[51];
  }

  //
  // Agent control parameters.
  //

  control -> AgentKarmaSize = parameters[55];

  if (packMethod == -1 || packQuality == -1)
  {
    if (packMethod == -1)
    {
      packMethod = parameters[56];
    }

    if (packQuality == -1)
    {
      packQuality = parameters[57];
    }

    if (ParsePackMethod(packMethod, packQuality) < 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Unrecognized pack method id "
              << packMethod << " with quality " << packQuality
              << ".\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Unrecognized pack method id "
           << packMethod << " with quality " << packQuality
           << ".\n";

      HandleCleanup();
    }
  }

  control -> AgentFlushQueued   = parameters[58];
  control -> AgentFlushPriority = parameters[59];

  if (useTaint != -1)
  {
    control -> AgentTaintReplies = (useTaint == 1);
  }
  else
  {
    control -> AgentTaintReplies = parameters[60];
  }

  control -> AgentTaintThreshold = parameters[61];

  //
  // Cleanup of buffers.
  //

  control -> AgentCleanGet    = parameters[65];
  control -> AgentCleanAlloc  = parameters[66];
  control -> AgentCleanFlush  = parameters[67];
  control -> AgentCleanSend   = parameters[68];
  control -> AgentCleanImages = parameters[69];

  //
  // Image parameters.
  //

  control -> AgentImageSplit       = parameters[73];
  control -> AgentImageMask        = parameters[74];
  control -> AgentImageFrame       = parameters[75];
  control -> AgentImageSplitMethod = parameters[76];
  control -> AgentImageMaskMethod  = parameters[77];

  //
  // Apply the new values.
  //

  if (proxy != NULL)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Configuring proxy according to file '"
            << file << "'.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Configuring proxy according to file '"
         << file << "'.\n";

    proxy -> handleLinkConfiguration();
  }

  return 1;
}

//
// These functions are used to parse literal
// values provided by the user and set the
// control parameters accordingly.
//

int ParseCacheOption(const char *opt)
{
  int size = ParseArg("", "cache", opt);

  if (size < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Invalid value '"
            << opt << "' for option 'cache'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Invalid value '"
         << opt << "' for option 'cache'.\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "Loop: Setting size of cache to "
          << size << " bytes.\n" << logofs_flush;
  #endif

  control -> ClientTotalStorageSizeLimit = size;
  control -> ServerTotalStorageSizeLimit = size;

  strcpy(cacheSizeName, opt);

  if (size == 0)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Disabling NX delta compression.\n"
            << logofs_flush;
    #endif

    control -> LocalDeltaCompression = 0;

    #ifdef WARNING
    *logofs << "Loop: WARNING! Disabling use of NX persistent cache.\n"
            << logofs_flush;
    #endif

    control -> PersistentCacheEnableLoad = 0;
    control -> PersistentCacheEnableSave = 0;
  }

  return 1;
}

int ParseImagesOption(const char *opt)
{
  int size = ParseArg("", "images", opt);

  if (size < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Invalid value '"
            << opt << "' for option 'images'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Invalid value '"
         << opt << "' for option 'images'.\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "Loop: Setting size of images cache to "
          << size << " bytes.\n" << logofs_flush;
  #endif

  control -> ImageCacheDiskLimit = size;

  strcpy(imagesSizeName, opt);

  return 1;
}

int ParseShmemOption(const char *opt)
{
  int size = ParseArg("", "shmem", opt);

  if (size < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Invalid value '"
            << opt << "' for option 'shmem'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Invalid value '"
         << opt << "' for option 'shmem'.\n";

    return -1;
  }

  control -> AgentShmemClientSize = size;
  control -> AgentShmemServerSize = size;

  #if defined(TEST) || defined(INFO)
  *logofs << "Loop: Set shared memory size to "
          << control -> AgentShmemServerSize
          << " bytes.\n" << logofs_flush;
  #endif

  strcpy(shmemSizeName, opt);

  return 1;
}

int ParseLimitOption(const char *opt)
{
  int bitrate = ParseArg("", "limit", opt);

  if (bitrate < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Invalid value '"
            << opt << "' for option 'limit'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Invalid value '"
         << opt << "' for option 'limit'.\n";

    return -1;
  }

  strcpy(bitrateLimitName, opt);

  if (bitrate == 0)
  {
    #ifdef TEST
    *logofs << "Loop: Disabling bitrate limit on proxy link.\n"
            << logofs_flush;
    #endif

    control -> LocalBitrateLimit = 0;
  }
  else
  {
    #ifdef TEST
    *logofs << "Loop: Setting bitrate to " << bitrate
            << " bits per second.\n" << logofs_flush;
    #endif

    //
    // Internal representation is in bytes
    // per second.
    //

    control -> LocalBitrateLimit = bitrate >> 3;
  }

  return 1;
}

int ParseHostOption(const char *opt, char *host, int &port)
{
  #ifdef TEST
  *logofs << "Loop: Trying to parse options string '" << opt
          << "' as a remote NX host.\n" << logofs_flush;
  #endif

  if (opt == NULL || *opt == '\0')
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! No host parameter provided.\n"
            << logofs_flush;
    #endif

    return 0;
  }
  else if (strlen(opt) >= DEFAULT_STRING_LENGTH)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Host parameter exceeds length of "
            << DEFAULT_STRING_LENGTH << " characters.\n"
            << logofs_flush;
    #endif

    return 0;
  }

  //
  // Look for a host name followed
  // by a colon followed by port.
  //

  int newPort = port;

  const char *separator = rindex(opt, ':');

  if (separator != NULL)
  {
    const char *check = separator + 1;

    while (*check != '\0' && *check != ',' &&
               *check != '=' && isdigit(*check) != 0)
    {
      check++;
    }

    newPort = atoi(separator + 1);

    if (newPort < 0 || *check != '\0')
    {
      #ifdef TEST
      *logofs << "Loop: Can't identify remote NX port in string '"
              << separator << "'.\n" << logofs_flush;
      #endif

      return 0;
    }
  }
  else if (newPort < 0)
  {
    //
    // Complain if port was not passed
    // by other means.
    //

    #ifdef TEST
    *logofs << "Loop: Can't identify remote NX port in string '"
            << opt << "'.\n" << logofs_flush;
    #endif

    return 0;
  }
  else
  {
    separator = opt + strlen(opt);
  }

  char newHost[DEFAULT_STRING_LENGTH] = { 0 };

  strncpy(newHost, opt, strlen(opt) - strlen(separator));

  *(newHost + strlen(opt) - strlen(separator)) = '\0';

  const char *check = newHost;

  while (*check != '\0' && *check != ',' &&
             *check != '=')
  {
    check++;
  }

  if (*check != '\0')
  {
    #ifdef TEST
    *logofs << "Loop: Can't identify remote NX host in string '"
            << newHost << "'.\n" << logofs_flush;
    #endif

    return 0;
  }
  else if (*acceptHost != '\0')
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Can't manage to connect and accept connections "
            << "at the same time.\n" << logofs_flush;

    *logofs << "Loop: PANIC! Refusing remote NX host with string '"
            << opt << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Can't manage to connect and accept connections "
         << "at the same time.\n";

    cerr << "Error" << ": Refusing remote NX host with string '"
         << opt << "'.\n";

    return -1;
  }

  if (*host != '\0' && strcmp(host, newHost) != 0)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Overriding remote NX host '"
            << host << "' with new value '" << newHost
            << "'.\n" << logofs_flush;
    #endif
  }

  strcpy(host, newHost);

  if (port != -1 && port != newPort)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Overriding remote NX port '"
            << port << "' with new value '" << newPort
            << "'.\n" << logofs_flush;
    #endif
  }

  #ifdef TEST
  *logofs << "Loop: Parsed options string '" << opt
          << "' with host '" << newHost << "' and port '"
          << newPort << "'.\n" << logofs_flush;
  #endif

  port = newPort;

  return 1;
}

int ParseListenOption(int &address)
{
  if (*listenHost == '\0')
  {
    //
    // On the X client side listen on any address.
    // On the X server side listen to the forwarder
    // on localhost.
    //

    if (control -> ProxyMode == proxy_server)
    {
      address = (int) inet_addr("127.0.0.1");
    }
    else
    {
      address = htonl(INADDR_ANY);
    }
  }
  else
  {
    address = inet_addr(listenHost);
  }

  return 1;
}

int OpenInputFile(char *name, istream *&stream)
{
  if (name == NULL || *name == '\0')
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Cannot determine name of NX control file.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot determine name of NX control file.\n";

    return -1;
  }
  else if (*name != '/' && *name != '.')
  {
    char *filePath = GetRootPath();

    if (filePath == NULL)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Cannot determine directory of NX control file.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Cannot determine directory of NX control file.\n";

      return -1;
    }
    else if (strlen(filePath) + strlen("/") +
                 strlen(name) + 1 > DEFAULT_STRING_LENGTH)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Full name of NX file '" << name
              << " would exceed length of " << DEFAULT_STRING_LENGTH
              << " characters.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Full name of NX file '" << name
           << " would exceed length of " << DEFAULT_STRING_LENGTH
           << " characters.\n";

      return -1;
    }

    char *file = new char[strlen(filePath) + strlen("/") +
                              strlen(name) + 1];

    //
    // Transform name in a fully qualified name.
    //

    strcpy(file, filePath);
    strcat(file, "/");
    strcat(file, name);

    strcpy(name, file);

    delete [] filePath;
    delete [] file;
  }

  for (;;)
  {
    if ((stream = new ifstream(name, ios::in)) != NULL)
    {
      break;
    }

    usleep(200000);
  }

  return 1;
}

int OpenOutputFile(char *name, ostream *&stream)
{
  if (name == NULL || *name == '\0')
  {
    #ifdef TEST
    *logofs << "Loop: WARNING! No name provided for output. Using standard error.\n"
            << logofs_flush;
    #endif

    if (stream == NULL)
    {
      stream = &cerr;
    }

    return 1;
  }

  if (stream == NULL || stream == &cerr)
  {
    if (*name != '/' && *name != '.')
    {
      char *filePath = GetSessionPath();

      if (filePath == NULL)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Cannot determine directory of NX session file.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Cannot determine directory of NX session file.\n";

        return -1;
      }

      if (strlen(filePath) + strlen("/") +
              strlen(name) + 1 > DEFAULT_STRING_LENGTH)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Full name of NX file '" << name
                << " would exceed length of " << DEFAULT_STRING_LENGTH
                << " characters.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Full name of NX file '" << name
             << " would exceed length of " << DEFAULT_STRING_LENGTH
             << " characters.\n";

        return -1;
      }

      char *file = new char[strlen(filePath) + strlen("/") +
                                strlen(name) + 1];

      //
      // Transform name in a fully qualified name.
      //

      strcpy(file, filePath);
      strcat(file, "/");
      strcat(file, name);

      strcpy(name, file);

      delete [] filePath;
      delete [] file;
    }

    for (;;)
    {
      if ((stream = new ofstream(name, ios::app)) != NULL)
      {
        break;
      }

      usleep(200000);
    }
  }
  else
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Bad stream provided for output.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Bad stream provided for output.\n";

    return -1;
  }

  return 1;
}

int ReopenOutputFile(char *name, ostream *&stream, int limit)
{
  if (*name != '\0' && limit >= 0)
  {
    struct stat fileStat;

    if (limit > 0)
    {
      //
      // This is used for the log file, if the
      // size exceeds the limit.
      //

      if (stat(name, &fileStat) != 0)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Can't get stats of file '"
                << name  << "'. Error is " << EGET() 
                << " '" << ESTR() << "'.\n" << logofs_flush;
        #endif

        return 0;
      }
      else if (fileStat.st_size < (long) limit)
      {
        return 0;
      }

      #ifdef TEST
      *logofs << "Loop: Deleting file '" << name
              << "' with size " << fileStat.st_size
              << ".\n" << logofs_flush;
      #endif

      *stream << flush;

      delete stream;

      unlink(name);
    }
    else
    {
      //
      // This is used to truncate the statistics
      // output. Trying to delete the file and
      // then reopening it doesn't seem to work
      // on recent Cygwin installs.
      //

      *stream << flush;

      delete stream;
    }

    for (;;)
    {
      if ((stream = new ofstream(name, ios::out)) != NULL)
      {
        break;
      }

      usleep(200000);
    }

    #ifdef TEST
    *logofs << "Loop: Reopened file '" << name
            << "'.\n" << logofs_flush;
    #endif
  }

  return 1;
}

void PrintProcessInfo()
{
  if (agentFD[1] == -1)
  {
    cerr << "NXPROXY - Version " << control -> LocalVersionMajor
         << "." << control -> LocalVersionMinor << "."
         << control -> LocalVersionPatch << "\n\n";

    cerr << "Copyright (C) 2001, 2005 NoMachine.\n"
         << "See http://www.nomachine.com/ for more information.\n\n";
  }

  //
  // People get confused by the fact that client
  // mode is running on NX server and viceversa.
  // Let's adopt an user-friendly naming conven-
  // tion here.
  //

  cerr << "Info: Proxy running in "
       << (control -> ProxyMode == proxy_client ? "server" : "client")
       << " mode with pid '" << getpid() << "'.\n";

  #ifdef TEST

  if (*logFileName != '\0')
  {
    cerr << "Info" << ": Using log file '" << logFileName << "'.\n";
  }

  if (*statFileName != '\0')
  {
    cerr << "Info" << ": Using stat file '" << statFileName << "'.\n";
  }

  #endif
}

void PrintConnectionInfo()
{
  if (control -> ProxyMode == proxy_client)
  {
    if (control -> LocalDeltaCompression == 1)
    {
      cerr << "Info" << ": Using cache parameters "
           << control -> MinimumMessageSizeThreshold
           << "/" << control -> MaximumMessageSizeThreshold
           << "/" << control -> LocalTotalStorageSizeLimit / 1024 << "KB"
           << "/" << control -> RemoteTotalStorageSizeLimit / 1024 << "KB"
           << ".\n";

      if (control -> ImageCacheEnableLoad > 0 ||
              control -> ImageCacheEnableSave > 0)
      {
        cerr << "Info" << ": Using image cache parameters "
             << control -> ImageCacheEnableLoad
             << "/" << control -> ImageCacheEnableSave
             << "/" << control -> ImageCacheDiskLimit / 1024 << "KB"
             << ".\n";
      }

      cerr << "Info" << ": Using split parameters "
           << control -> SplitTimeout
           << "/" << control -> SplitDataThreshold
           << "/" << control -> SplitDataPacketLimit
           << "/" << control -> SplitTotalStorageSizeLimit / 1024 << "KB"
           << ".\n";
    }
    else
    {
      cerr << "Info" << ": Not using NX delta compression.\n";
    }

    cerr << "Info" << ": Using "
         << linkSpeedName << " link parameters "
         << control -> TokenBytes
         << "/" << control -> TokenLimit
         << "/" << control -> FrameTimeout
         << "/" << control -> FlushTimeout
         << ".\n";

    cerr << "Info" << ": Using agent parameters "
         << control -> AgentKarmaSize
         << "/" << control -> AgentFlushQueued
         << "/" << control -> AgentFlushPriority
         << ".\n";
  }
  else
  {
    if (control -> LocalDeltaCompression == 1)
    {
      cerr << "Info" << ": Using cache parameters "
           << control -> MinimumMessageSizeThreshold
           << "/" << control -> MaximumMessageSizeThreshold
           << "/" << control -> ClientTotalStorageSizeLimit / 1024 << "KB"
           << "/" << control -> ServerTotalStorageSizeLimit / 1024 << "KB"
           << ".\n";

      if (control -> ImageCacheEnableLoad > 0 ||
              control -> ImageCacheEnableSave > 0)
      {
        cerr << "Info" << ": Using image cache parameters "
             << control -> ImageCacheEnableLoad
             << "/" << control -> ImageCacheEnableSave
             << "/" << control -> ImageCacheDiskLimit / 1024 << "KB"
             << ".\n";
      }
    }
    else
    {
      cerr << "Info" << ": Not using NX delta compression.\n";
    }

    cerr << "Info" << ": Using "
         << linkSpeedName << " link parameters "
         << control -> TokenBytes
         << "/" << control -> TokenLimit
         << "/" << control -> FrameTimeout
         << "/" << control -> FlushTimeout
         << ".\n";
  }

  cerr << "Info" << ": Using pack method '"
       << packMethodName << "' with session '"
       << sessionType << "'.\n";

  if (control -> LocalDeltaCompression == 1)
  {
    if (control -> LocalDataCompression == 1)
    {
      cerr << "Info" << ": Using ZLIB data compression level "
           << control -> LocalDataCompressionLevel << ".\n";

      cerr << "Info" << ": Using ZLIB data threshold set to " 
           << control -> LocalDataCompressionThreshold << ".\n";
    }
    else
    {
      cerr << "Info" << ": Not using ZLIB data compression.\n";
    }
  }

  if (control -> LocalStreamCompression == 1)
  {
    cerr << "Info" << ": Using ZLIB stream compression level "
         << control -> LocalStreamCompressionLevel << ".\n";
  }
  else
  {
    cerr << "Info" << ": Not using ZLIB stream compression.\n";
  }

  if (control -> LocalDeltaCompression == 1)
  {
    if (control -> RemoteDataCompression == 1)
    {
      cerr << "Info" << ": Using remote ZLIB data compression level "
           << control -> RemoteDataCompressionLevel << ".\n";
    }
    else
    {
      cerr << "Info" << ": Not using remote ZLIB data compression.\n";
    }
  }

  if (control -> RemoteStreamCompression == 1)
  {
    cerr << "Info" << ": Using remote ZLIB stream compression level "
         << control -> RemoteStreamCompressionLevel << ".\n";
  }
  else
  {
    cerr << "Info" << ": Not using remote ZLIB stream compression.\n";
  }

  if (control -> LocalBitrateLimit > 0)
  {
    cerr << "Info" << ": Using bandwidth limit of "
         << bitrateLimitName << " bits per second.\n";
  }

  if (control -> LocalProcessorLimit > 0)
  {
    cerr << "Info" << ": Using processor idle time ratio of "
         << control -> LocalProcessorLimit << ".\n";
  }
  else if (control -> ProxyMode == proxy_client)
  {
    cerr << "Info" << ": Not using processor load limit.\n";
  }

  if (control -> PersistentCacheName != NULL)
  {
    cerr << "Info" << ": Using cache file '"
         << control -> PersistentCachePath << "/"
         << control -> PersistentCacheName << "'.\n";
  }
  else
  {
    if (control -> PersistentCacheEnableLoad == 0 ||
            control -> LocalDeltaCompression == 0)
    {
      cerr << "Info" << ": Not using persistent cache.\n";
    }
    else
    {
      cerr << "Info" << ": No suitable cache file found.\n";
    }
  }

  if (WE_INITIATE_CONNECTION)
  {
    cerr << "Info" << ": Using remote server '" << connectHost
         << ":" << DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort
         << "'.\n" << logofs_flush;
  }
  else if (WE_LISTEN_FORWARDER)
  {
    cerr << "Info" << ": Using remote server connected "
         << "on port '" << listenPort << "'.\n"
         << logofs_flush;
  }

  if (control -> ProxyMode == proxy_client &&
          useCupsSocket > 0 && cupsPort > 0)
  {
    cerr << "Info" << ": Listening for cups connections "
         << "on port '" << cupsPort << "'.\n";
  }
  else if (control -> ProxyMode == proxy_server &&
               cupsPort > 0)
  {
    cerr << "Info" << ": Forwarding cups connections "
         << "to port '" << cupsPort << "'.\n";
  }

  if (control -> ProxyMode == proxy_client &&
          useKeybdSocket > 0 && keybdPort > 0)
  {
    cerr << "Info" << ": Listening for embedded keyboard connections "
         << "on port '" << keybdPort << "'.\n";
  }
  else if (control -> ProxyMode == proxy_server &&
               keybdPort > 0)
  {
    cerr << "Info" << ": Forwarding embedded keyboard connections "
         << "to port '" << keybdPort << "'.\n";
  }

  if (control -> ProxyMode == proxy_client &&
          useSambaSocket > 0 && sambaPort > 0)
  {
    cerr << "Info" << ": Listening for SMB connections "
         << "on port '" << sambaPort << "'.\n";
  }
  else if (control -> ProxyMode == proxy_server &&
               sambaPort > 0)
  {
    cerr << "Info" << ": Forwarding SMB connections "
         << "to port '" << sambaPort << "'.\n";
  }

  if (control -> ProxyMode == proxy_client &&
          useMediaSocket > 0 && mediaPort > 0)
  {
    cerr << "Info" << ": Listening for multimedia connections "
         << "on port '" << mediaPort << "'.\n";
  }
  else if (control -> ProxyMode == proxy_server &&
               mediaPort > 0)
  {
    cerr << "Info" << ": Forwarding multimedia connections "
         << "to port '" << mediaPort << "'.\n";
  }

  if (control -> ProxyMode == proxy_client &&
          useHttpSocket > 0 && httpPort > 0)
  {
    cerr << "Info" << ": Listening for HTTP connections "
         << "on port '" << httpPort << "'.\n";
  }
  else if (control -> ProxyMode == proxy_server &&
               httpPort > 0)
  {
    cerr << "Info" << ": Forwarding HTTP connections "
         << "to port '" << httpPort << "'.\n";
  }
}

void PrintResetInfo()
{
  if (WE_INITIATE_CONNECTION)
  {
    cerr << "Info" << ": Using remote server '" << connectHost
         << ":" << DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort
         << "'.\n" << logofs_flush;
  }
  else if (WE_LISTEN_FORWARDER)
  {
    cerr << "Info" << ": Using remote server connected "
         << "on port '" << listenPort << "'.\n"
         << logofs_flush;
  }
  else
  {
    cerr << "Info" << ": Using local port '"
         << DEFAULT_NX_PROXY_PORT_OFFSET + proxyPort
         << "'.\n" << logofs_flush;
  }

  if (control -> PersistentCacheName != NULL)
  {
    cerr << "Info" << ": Using cache file '"
         << control -> PersistentCachePath << "/"
         << control -> PersistentCacheName << "'.\n";
  }
  else
  {
    if (control -> PersistentCacheEnableLoad == 0 ||
            control -> LocalDeltaCompression == 0)
    {
      cerr << "Info" << ": Not using persistent cache.\n";
    }
    else
    {
      cerr << "Info" << ": No suitable cache file found.\n";
    }
  }
}

void PrintVersionInfo()
{
  cerr << "NXPROXY - " << "Version "
       << control -> LocalVersionMajor << "."
       << control -> LocalVersionMinor << "."
       << control -> LocalVersionPatch;

  cerr << endl;
}

void PrintCopyrightInfo()
{
  cerr << endl;

  PrintVersionInfo();

  cerr << endl;

  cerr << GetCopyrightInfo();

  //
  // Print third party's copyright info.
  //

  cerr << endl;

  cerr << GetOtherCopyrightInfo();

  cerr << endl;
}

const char *GetOptions(const char *options)
{
  if (options != NULL)
  {
    if (strncasecmp(options, "nx/nx,", 6) != 0 &&
            strncasecmp(options, "nx,", 3) != 0 &&
                strncasecmp(options, "nx:", 3) != 0)
    {
      #ifdef TEST
      *logofs << "Loop: PANIC! Display options string '" << options
              << "' must start with 'nx' or 'nx/nx' prefix.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Display options string '" << options
           << "' must start with 'nx' or 'nx/nx' prefix.\n";

      HandleCleanup();
    }
  }
  else
  {
    options = getenv("DISPLAY");
  }

  return options;
}

const char *GetArg(int &argi, int argc, const char **argv)
{
  //
  // Skip "-" and flag character.
  //

  const char *arg = argv[argi] + 2;

  if (*arg == 0)
  {
    if (argi + 1 == argc)
    {
      return NULL;
    }
    else
    {
      argi++;

      return (*argv[argi] == '-' ? NULL : argv[argi]);
    }
  }
  else
  {
    return (*arg == '-' ? NULL : arg);
  }
}

int CheckArg(const char *type, const char *name, const char *value)
{
  #ifdef TEST
  *logofs << "Loop: Parsing " << type << " option '" << name
          << "' with value '" << (value ? value : "(null)")
          << "'.\n" << logofs_flush;
  #endif

  if (value == NULL || strstr(value, "=") != NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error in " << type << " option '"
            << name << "'. No value found.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error in " << type << " option '"
         << name << "'. No value found.\n";

    return -1;
  }
  else if (strstr(name, ",") != NULL)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Parse error at " << type << " option '"
            << name << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Parse error at " << type << " option '"
         << name << "'.\n";

    return -1;
  }
  else if (strlen(value) >= DEFAULT_STRING_LENGTH)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Value '" << value << "' of "
            << type << " option '" << name << "' exceeds length of "
            << DEFAULT_STRING_LENGTH << " characters.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Value '" << value << "' of "
         << type << " option '" << name << "' exceeds length of "
         << DEFAULT_STRING_LENGTH << " characters.\n";

    return -1;
  }

  return 1;
}

int ParseArg(const char *type, const char *name, const char *value)
{
  if (!strcasecmp(value, "0"))
  {
    return 0;
  }

  //
  // Find the base factor.
  //

  double base;

  const char *id = value + strlen(value) - 1;

  if (!strcasecmp(id, "g"))
  {
    base = 1024 * 1024 * 1024;
  }
  else if (!strcasecmp(id, "m"))
  {
    base = 1024 * 1024;
  }
  else if (!strcasecmp(id, "k"))
  {
    base = 1024;
  }
  else if (!strcasecmp(id, "b") || isdigit(*id))
  {
    base = 1;
  }
  else
  {
    return -1;
  }

  char *string = new char[strlen(value)];

  strncpy(string, value, strlen(value) - 1);

  *(string + (strlen(value) - 1)) = '\0';

  #ifdef TEST

  *logofs << "Loop: Parsing integer option '" << name
          << "' from string '" << string << "' with base set to ";

  switch (tolower(*id))
  {
    case 'k':
    case 'm':
    case 'g':
    {
      *logofs << (char) toupper(*id);
    }
    break;
  }

  *logofs << ".\n" << logofs_flush;

  #endif

  double result = atof(string) * base;

  if (result < 0 || result > (((unsigned) -1) >> 1))
  {
    delete [] string;

    return -1;
  }

  delete [] string;

  return (int) result;
}

int CheckSignal(int signal)
{
  //
  // Return 1 if the signal needs to be handled
  // by the proxy, 2 if the signal just needs to
  // be blocked to avoid interrupting a system
  // call.
  //

  switch (signal)
  {
    case SIGCHLD:
    case SIGUSR1:
    case SIGUSR2:
    case SIGHUP:
    case SIGINT:
    case SIGTERM:
    case SIGPIPE:
    case SIGALRM:
    {
      return 1;
    }
    case SIGVTALRM:
    case SIGWINCH:
    case SIGIO:
    case SIGTSTP:
    case SIGTTIN:
    case SIGTTOU:
    {
      return 2;
    }
    default:
    {
      #ifdef __CYGWIN32__

      //
      // This signal can be raised by the Cygwin
      // library.
      //

      if (signal == 12)
      {
        return 1;
      }

      #endif

      return 0;
    }
  }
}

static void PrintUsageInfo(const char *option, int error)
{
  if (error == 1)
  {
    cerr << "Error" << ": Invalid command line option '" << option << "'.\n";
  }

  cerr << GetUsageInfo();

  if (error == 1)
  {
    cerr << "Error" << ": NX transport initialization failed.\n";
  }
}

static void handleCheckSessionInLoop()
{
  //
  // The kill flag can be:
  //
  // 0: No shutdown procedure has started yet.
  //
  // 1: Cleanup procedure has started and we
  //    are waiting for all channels to be
  //    closed.
  //
  // 2: All channels have been successfully
  //    closed.
  //
  // NX server should send the cleanup signal
  // only after this message has been printed
  // in the session log.
  //

  if (lastKill == 1 && proxy -> getChannels(channel_x) == 0)
  {
    if (control -> CleanupTimeout > 0)
    {
      #ifdef TEST
      *logofs << "Loop: Waiting for cleanup timeout to complete.\n"
              << logofs_flush;
      #endif

      cerr << "Info" << ": Waiting for cleanup timeout to complete.\n";
    }
    else
    {
      #ifdef TEST
      *logofs << "Loop: Waiting for a further signal to complete.\n"
              << logofs_flush;
      #endif

      cerr << "Info" << ": Waiting for a further signal to complete.\n";
    }

    //
    // Don't print messages in the
    // session log a second time.
    //

    lastKill = 2;
  }

  //
  // Let client proxy find out if the agent's channel
  // is gone. This is the normal procedure in case of
  // internal connections with any of X, RDP and RFB
  // agents. 
  // 

  int cleanup = 0;

  if (control -> ProxyMode == proxy_client &&
          useAgentSocket == 1 && proxy -> getType(agentFD[1]) ==
              channel_none && lastKill == 0)
  {
    #ifdef TEST
    *logofs << "Loop: End of session requested by agent "
            << "termination.\n" << logofs_flush;
    #endif

    cerr << "Info" << ": End of session requested by "
         << "agent termination.\n";
 
    #ifdef TEST
    *logofs << "Loop: Bytes sent so far are "
            << (unsigned long long) control -> getBytesOut()
            << ".\n" << logofs_flush;
    #endif

    if (control -> getBytesOut() < 1024)
    {
      cerr << "Info" << ": Your session has died before reaching "
           << "an usable state.\n";
      cerr << "Info" << ": This can be due to remote X server "
           << "refusing access to the client.\n";
      cerr << "Info" << ": Please check authorization provided "
           << "by your X application.\n";
    }

    cleanup = 1;
  }

  //
  // Check if the user requested a session cleanup by
  // sending a signal to the proxy. All signals are
  // handled in the main loop so we need to reset the
  // value to get ready for the next iteration.
  //

  int signal = 0;

  if (lastSignal != 0)
  {
    switch (lastSignal)
    {
      case SIGCHLD:
      case SIGUSR1:
      case SIGUSR2:
      {
        break;
      }
      case SIGHUP:
      {
        if (control -> EnableShutdownOnSighup == 1)
        {
          signal = lastSignal;

          cleanup = 1;
        }

        break;
      }
      default:
      {
        signal = lastSignal;

        cleanup = 1;

        break;
      }
    }

    lastSignal = 0;
  }

  if (cleanup == 1)
  {
    //
    // The first time termination signal is received
    // disable all further connections, close down
    // any X channel and wait for a second signal.
    //

    if (lastKill == 0)
    {
      //
      // Don't print a message if cleanup is
      // due to normal termination of agent.
      //

      if (signal != 0)
      {
        #ifdef TEST
        *logofs << "Loop: End of session requested by signal '"
                << signal << "' '" << SignalLabel(signal)
                << "'.\n" << logofs_flush;
        #endif

        cerr << "Info" << ": End of session requested by signal '"
             << signal << "' '" << SignalLabel(signal)
             << "'.\n";
      }

      //
      // Disable any further connection.
      //

      if (useUnixSocket == 1 && *unixSocketName != '\0')
      {
        #ifdef TEST
        *logofs << "Loop: Going to remove the Unix domain socket '"
                << unixSocketName << "'.\n" << logofs_flush;
        #endif

        unlink(unixSocketName);
      }

      useUnixSocket  = 0;
      useTCPSocket   = 0;
      useCupsSocket  = 0;
      useKeybdSocket = 0;
      useSambaSocket = 0;
      useMediaSocket = 0;
      useHttpSocket  = 0;

      //
      // Close all the remaining X channels and
      // let proxies save their persistent cache
      // on disk.
      //

      proxy -> handleCloseAllXConnections();

      //
      // Run a watchdog process so we can finally
      // give up at the time signal is delivered.
      //

      if (lastWatchdog == 0)
      {
        if (control -> CleanupTimeout > 0)
        {
          #ifdef TEST
          *logofs << "Loop: Starting watchdog process with timeout of "
                  << control -> CleanupTimeout / 1000 << " seconds.\n"
                  << logofs_flush;
          #endif

          lastWatchdog = NXTransWatchdog(control -> CleanupTimeout / 1000);

          if (lastWatchdog < 0)
          {
            #ifdef PANIC
            *logofs << "Loop: PANIC! Can't start the NX watchdog process.\n"
                    << logofs_flush;
            #endif

            lastWatchdog = 0;
          }
          #ifdef TEST
          else
          {
            *logofs << "Loop: Watchdog started with pid '"
                    << lastWatchdog << "'.\n" << logofs_flush;
          }
          #endif
        }
      }
      else
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Previous watchdog detected "
                << "in shutdown with pid '" << lastWatchdog
                << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Previous watchdog detected "
             << "in shutdown with pid '" << lastWatchdog
             << "'.\n";

        HandleCleanup();
      }

      lastKill = 1;
    }
    else
    {
      #ifdef TEST
      *logofs << "Loop: Finalized the local proxy shutdown.\n"
              << logofs_flush;
      #endif

      cerr << "Info" << ": Shutting down the link and exiting.\n";

      proxy -> handleShutdown();

      HandleCleanup();
    }
  }
}

static void handleCheckLimitsInLoop()
{
  static long int slept = 0;

  #ifdef TEST
  *logofs << "Loop: Bitrate is " << control -> getBitrateInShortFrame()
          << " B/s and " << control -> getBitrateInLongFrame()
          << " B/s in " << control -> ShortBitrateTimeFrame / 1000
          << "/" << control -> LongBitrateTimeFrame / 1000
          << " seconds timeframes.\n" << logofs_flush;
  #endif

  #ifdef TEST
  *logofs << "Loop: CPU time is " << control -> getIdleTime()
          << " Ms in select and " << control -> getReadTime()
          << " in loop.\n" << logofs_flush;
  #endif

  //
  // Ensure time spent in select is higher
  // than time spent handling messages.
  //

  if (control -> isStartup() == 0)
  {
    //
    // Very preliminary. It should track CPU time in
    // short and long timeframes and not only total
    // time since beginning of session.
    //

    if (control -> LocalProcessorLimit > 0)
    {
      #ifdef TEST
      *logofs << "Loop: Calculating CPU usage with limit "
              << control -> LocalProcessorLimit << ".\n"
              << logofs_flush;
      #endif

      double ratio = control -> getIdleTime() /
                         control -> getReadTime();

      if (ratio < control -> LocalProcessorLimit)
      {
        double offset = control -> LocalProcessorLimit + 1 - ratio;

        if (offset > 1.1)
        {
          offset = 1.1;
        }

        slept += (unsigned int) (pow(50000, offset) / 1000);

        if (slept > 2000)
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! Sleeping due to "
                  << "select/loop CPU time ratio of "
                  << ratio << ".\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Sleeping due to "
               << "select/loop CPU time ratio of "
               << ratio << ".\n";

          slept %= 2000;
        }

        T_timestamp idleTs = getTimestamp();

        usleep((unsigned int) pow(50000, offset));

        int diffTs = diffTimestamp(idleTs, getTimestamp());

        control -> addIdleTime(diffTs);

        control -> subReadTime(diffTs);

        if (control -> CollectStatistics)
        {
          statistics -> addIdleTime(diffTs);

          statistics -> subReadTime(diffTs);
        }
      }
    }

    //
    // Very preliminary. Because of buffering, we don't
    // jump out of select often enough to guarantee
    // accuracy.
    //

    if (control -> LocalBitrateLimit > 0)
    {
      #ifdef TEST
      *logofs << "Loop: Calculating bandwidth usage with limit "
              << control -> LocalBitrateLimit << ".\n"
              << logofs_flush;
      #endif

      int reference = (control -> getBitrateInLongFrame() +
                           control -> getBitrateInShortFrame()) / 2;

      if (reference > control -> LocalBitrateLimit)
      {
        double ratio = ((double) reference) /
                           ((double) control -> LocalBitrateLimit);

        if (ratio > 1.2)
        {
          ratio = 1.2;
        }

        slept += (unsigned int) (pow(50000, ratio) / 1000);

        if (slept > 2000)
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! Sleeping due to "
                  << "reference bitrate of " << reference
                  << " B/s.\n" << logofs_flush;
          #endif

          cerr << "Warning" << ": Sleeping due to "
               << "reference bitrate of " << reference
               << " B/s.\n";

          slept %= 2000;
        }

        T_timestamp idleTs = getTimestamp();

        usleep((unsigned int) pow(50000, ratio));

        int diffTs = diffTimestamp(idleTs, getTimestamp());

        control -> addIdleTime(diffTs);

        control -> subReadTime(diffTs);

        if (control -> CollectStatistics)
        {
          statistics -> addIdleTime(diffTs);

          statistics -> subReadTime(diffTs);
        }
      }
    }
  }
}

#if defined(TEST) || defined(INFO)

static void handleCheckStateInLoop(int &setFDs)
{
  int fdLength;
  int fdPending;
  int fdSplits;

  for (int j = 0; j < setFDs; j++)
  {
    if (j != proxyFD)
    {
      fdPending = proxy -> getPending(j);

      if (fdPending > 0)
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Buffer for X descriptor FD#"
                << j << " has pending bytes to read.\n"
                << logofs_flush;
        #endif

        HandleCleanup();
      }

      fdLength = proxy -> getLength(j);

      if (fdLength > 0)
      {
        #ifdef TEST
        *logofs << "Loop: WARNING! Buffer for X descriptor FD#"
                << j << " has " << fdLength << " bytes to write.\n"
                << logofs_flush;
        #endif
      }
    }
  }

  fdPending = proxy -> getPending(proxyFD);

  if (fdPending > 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Buffer for proxy descriptor FD#"
            << proxyFD << " has pending bytes to read.\n"
            << logofs_flush;
    #endif

    HandleCleanup();
  }

  fdLength = proxy -> getLength(proxyFD) +
                 proxy -> getFlushable(proxyFD);

  if (fdLength > 0)
  {
    if (control -> FlushPolicy == policy_immediate &&
            proxy -> getBlocked(proxyFD) == 0)
    {
      #ifdef PANIC
      *logofs << "Loop: PANIC! Buffer for proxy descriptor FD#"
              << proxyFD << " has " << fdLength << " bytes "
              << "to write with policy 'immediate'.\n"
              << logofs_flush;
      #endif

      HandleCleanup();
    }
    else
    {
      #ifdef TEST
      *logofs << "Loop: WARNING! Buffer for proxy descriptor FD#"
              << proxyFD << " has " << fdLength << " bytes "
              << "to write.\n" << logofs_flush;
      #endif
    }
  }

  fdSplits = proxy -> getSplits(proxyFD);

  if (fdSplits > 0)
  {
    #ifdef WARNING
    *logofs << "Loop: WARNING! Proxy descriptor FD#" << proxyFD
            << " has " << fdSplits << " messages to split.\n"
            << logofs_flush;
    #endif
  }
}

static void handleCheckSelectInLoop(int &setFDs, fd_set &readSet,
                                        fd_set &writeSet, T_timestamp selectTs)
{
  #if defined(TEST) || defined(INFO)
  *logofs << "Loop: Maximum descriptors is ["
          << setFDs << "] at " << strMsTimestamp()
          << ".\n" << logofs_flush;
  #endif

  int i;

  if (setFDs > 0)
  {
    i = 0;

    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Selected for read are ";
    #endif

    for (int j = 0; j < setFDs; j++)
    {
      if (FD_ISSET(j, &readSet))
      {
        #if defined(TEST) || defined(INFO)
        *logofs << "[" << j << "]" << logofs_flush;
        #endif

        i++;
      }
    }

    if (i > 0)
    {
      #if defined(TEST) || defined(INFO)
      *logofs << ".\n" << logofs_flush;
      #endif
    }
    else
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "[none].\n" << logofs_flush;
      #endif
    }

    i = 0;

    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Selected for write are ";
    #endif

    for (int j = 0; j < setFDs; j++)
    {
      if (FD_ISSET(j, &writeSet))
      {
        #if defined(TEST) || defined(INFO)
        *logofs << "[" << j << "]" << logofs_flush;
        #endif

        i++;
      }
    }

    if (i > 0)
    {
      #if defined(TEST) || defined(INFO)
      *logofs << ".\n" << logofs_flush;
      #endif
    }
    else
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "[none].\n" << logofs_flush;
      #endif
    }
  }

  #if defined(TEST) || defined(INFO)
  *logofs << "Loop: Select timeout is "
          << selectTs.tv_sec << " S and "
          << (double) selectTs.tv_usec / 1000
          << " Ms.\n" << logofs_flush;
  #endif
}

static void handleCheckResultInLoop(int &resultFDs, int &errorFDs, int &setFDs, fd_set &readSet,
                                        fd_set &writeSet, struct timeval &selectTs,
                                            struct timeval &startTs)
{
  int diffTs = diffTimestamp(startTs, getTimestamp());

  #if defined(TEST) || defined(INFO)

  if (diffTs >= (control -> PingTimeout -
                     (control -> LatencyTimeout * 4)))
  {
    *logofs << "Loop: Select result is [" << resultFDs
            << "] at " << strMsTimestamp() << " with no "
            << "communication within " << diffTs
            << " Ms.\n" << logofs_flush;
  }
  else
  {
    *logofs << "Loop: Select result is [" << resultFDs
            << "] error is [" << errorFDs << "] at "
            << strMsTimestamp() << " after " << diffTs
            << " Ms.\n" << logofs_flush;
  }

  #endif

  int i;

  if (resultFDs > 0)
  {
    i = 0;

    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Selected for read are ";
    #endif

    for (int j = 0; j < setFDs; j++)
    {
      if (FD_ISSET(j, &readSet))
      {
        #if defined(TEST) || defined(INFO)
        *logofs << "[" << j << "]" << logofs_flush;
        #endif

        i++;
      }
    }

    if (i > 0)
    {
      #if defined(TEST) || defined(INFO)
      *logofs << ".\n" << logofs_flush;
      #endif
    }
    else
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "[none].\n" << logofs_flush;
      #endif
    }

    i = 0;

    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Selected for write are ";
    #endif

    for (int j = 0; j < setFDs; j++)
    {
      if (FD_ISSET(j, &writeSet))
      {
        #if defined(TEST) || defined(INFO)
        *logofs << "[" << j << "]" << logofs_flush;
        #endif

        i++;
      }
    }

    if (i > 0)
    {
      #if defined(TEST) || defined(INFO)
      *logofs << ".\n" << logofs_flush;
      #endif
    }
    else
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "[none].\n" << logofs_flush;
      #endif
    }
  }
}

#endif

static void handleCheckSessionInConnect()
{
  #ifdef TEST
  *logofs << "Loop: Going to check session in connect.\n"
          << logofs_flush;
  #endif

  if (control -> ProxyMode == proxy_client)
  {
    HandleAlert(FAILED_PROXY_CONNECTION_CLIENT_ALERT, 1);
  }
  else if (lastDialog == 0)
  {
    HandleAlert(FAILED_PROXY_CONNECTION_SERVER_ALERT, 1);
  }

  handleAlertInLoop();
}

static void handleStatisticsInLoop()
{
  if (lastSignal == 0)
  {
    return;
  }

  int mode = NO_STATS;

  if (lastSignal == SIGUSR1)
  {
    //
    // Print overall statistics.
    //

    mode = TOTAL_STATS;
  }
  else if (lastSignal == SIGUSR2)
  {
    //
    // Print partial statistics.
    //

    mode = PARTIAL_STATS;
  }

  #ifdef TEST
  *logofs << "Loop: Going to check the statistics "
          << "with signal " << SignalLabel(lastSignal)
          << ".\n" << logofs_flush;
  #endif

  if (mode == TOTAL_STATS || mode == PARTIAL_STATS)
  {
    if (control -> CollectStatistics &&
            proxy != NULL && statistics != NULL)
    {
      if (ReopenOutputFile(statFileName, statofs, 0) < 0)
      {
        HandleCleanup();
      }

      proxy -> handleStatistics(mode, statofs);
    }
  }
}

static void handleResetInLoop()
{
  //
  // You should override this if you just want
  // to reload control parameters on the fly.
  //

  if (lastSignal == SIGHUP)
  {
    if (control -> EnableRestartOnSighup == 1)
    {
      if (control -> EnableCoreDumpOnAbort == 0 &&
              control -> EnableRestartOnFailure == 1)
      {
        #ifdef WARNING
        *logofs << "Loop: WARNING! Received signal SIGHUP. "
                << "Aborting proxy connection.\n"
                << logofs_flush;
        #endif

        cerr << "Warning" << ": Received signal SIGHUP. "
             << "Aborting proxy connection.\n";

        HandleAbort();
      }
    }
    else if (control -> EnableReconfigOnSighup == 1)
    {
      #ifdef WARNING
      *logofs << "Loop: WARNING! Received signal SIGHUP. "
              << "Reading new control parameters.\n"
              << logofs_flush;
      #endif

      cerr << "Warning" << ": Received signal SIGHUP. "
           << "Reading new control parameters.\n";

      SetParameters();

      lastSignal = 0;
    }
    else if (control -> EnableShutdownOnSighup == 0)
    {
      #ifdef TEST
      *logofs << "Loop: Ignoring signal SIGHUP.\n"
              << logofs_flush;
      #endif

      lastSignal = 0;
    }
    #ifdef TEST
    else
    {
      *logofs << "Loop: Received signal SIGHUP "
              << "to close the proxy connection.\n"
              << logofs_flush;
    }
    #endif
  }
}

static void handleNegotiationInLoop(int &setFDs, fd_set &readSet,
                                        fd_set &writeSet, T_timestamp &selectTs)
{
  int yield = 0;

  while (yield == 0)
  {
    #ifdef TEST
    *logofs << "Loop: Going to run a new negotiation loop "
            << "with stage " << control -> ProxyStage << ".\n"
            << logofs_flush;
    #endif

    switch (control -> ProxyStage)
    {
      case stage_undefined:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_undefined" << "'.\n"
                << logofs_flush;
        #endif

        control -> ProxyStage = stage_initializing;

        break;
      }
      case stage_initializing:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_initializing" << "'.\n"
                << logofs_flush;
        #endif

        InitBeforeNegotiation();

        control -> ProxyStage = stage_connecting;

        break;
      }
      case stage_connecting:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_connecting" << "'.\n"
                << logofs_flush;
        #endif

        SetupProxyConnection();

        control -> ProxyStage = stage_connected;

        break;
      }
      case stage_connected:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_connected" << "'.\n"
                << logofs_flush;
        #endif

        //
        // Server side proxy must always be the one that
        // sends its version and options first, so, in
        // some way, client side can be the the one that
        // has the last word on the matter.
        //

        if (control -> ProxyMode == proxy_server)
        {
          //
          // Check if we have been listening for a
          // forwarder. In this case it will have to
          // authenticate itself.
          //

          if (WE_LISTEN_FORWARDER)
          {
            control -> ProxyStage = stage_waiting_forwarder_version;

            break;
          }

          control -> ProxyStage = stage_sending_proxy_options;
        }
        else
        {
          //
          // The X client side is the side that has to wait
          // for the authorization cookie and any remote
          // option.
          //

          control -> ProxyStage = stage_waiting_proxy_version;
        }

        break;
      }
      case stage_sending_proxy_options:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_sending_proxy_options" << "'.\n"
                << logofs_flush;
        #endif

        if (SendProxyOptions(proxyFD) < 0)
        {
          goto handleNegotiationInLoopError;
        }

        if (control -> ProxyMode == proxy_server)
        {
          control -> ProxyStage = stage_waiting_proxy_version;
        }
        else
        {
          control -> ProxyStage = stage_sending_proxy_caches;
        }

        break;
      }
      case stage_waiting_forwarder_version:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_waiting_forwarder_version" << "'.\n"
                << logofs_flush;
        #endif

        int result = ReadForwarderVersion(proxyFD);

        if (result == 0)
        {
          yield = 1;
        }
        else if (result == 1)
        {
          control -> ProxyStage = stage_waiting_forwarder_options;
        }
        else
        {
          goto handleNegotiationInLoopError;
        }

        break;
      }
      case stage_waiting_forwarder_options:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_waiting_forwarder_options" << "'.\n"
                << logofs_flush;
        #endif

        int result = ReadForwarderOptions(proxyFD);

        if (result == 0)
        {
          yield = 1;
        }
        else if (result == 1)
        {
          control -> ProxyStage = stage_sending_proxy_options;
        }
        else
        {
          goto handleNegotiationInLoopError;
        }

        break;
      }
      case stage_waiting_proxy_version:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_waiting_proxy_version" << "'.\n"
                << logofs_flush;
        #endif

        int result = ReadProxyVersion(proxyFD);

        if (result == 0)
        {
          yield = 1;
        }
        else if (result == 1)
        {
          control -> ProxyStage = stage_waiting_proxy_options;
        }
        else
        {
          goto handleNegotiationInLoopError;
        }

        break;
      }
      case stage_waiting_proxy_options:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_waiting_proxy_options" << "'.\n"
                << logofs_flush;
        #endif

        int result = ReadProxyOptions(proxyFD);

        if (result == 0)
        {
          yield = 1;
        }
        else if (result == 1)
        {
          if (control -> ProxyMode == proxy_server)
          {
            control -> ProxyStage = stage_waiting_proxy_caches;
          }
          else
          {
            control -> ProxyStage = stage_sending_proxy_options;
          }
        }
        else
        {
          goto handleNegotiationInLoopError;
        }

        break;
      }
      case stage_sending_proxy_caches:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_sending_proxy_caches" << "'.\n"
                << logofs_flush;
        #endif

        if (SendProxyCaches(proxyFD) < 0)
        {
          goto handleNegotiationInLoopError;
        }

        if (control -> ProxyMode == proxy_server)
        {
          control -> ProxyStage = stage_operational;
        }
        else
        {
          control -> ProxyStage = stage_waiting_proxy_caches;
        }

        break;
      }
      case stage_waiting_proxy_caches:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_waiting_proxy_caches" << "'.\n"
                << logofs_flush;
        #endif

        int result = ReadProxyCaches(proxyFD);

        if (result == 0)
        {
          yield = 1;
        }
        else if (result == 1)
        {
          if (control -> ProxyMode == proxy_server)
          {
            control -> ProxyStage = stage_sending_proxy_caches;
          }
          else
          {
            control -> ProxyStage = stage_operational;
          }
        }
        else
        {
          goto handleNegotiationInLoopError;
        }

        break;
      }
      case stage_operational:
      {
        #ifdef TEST
        *logofs << "Loop: Handling negotiation with '"
                << "stage_operational" << "'.\n"
                << logofs_flush;
        #endif

        InitAfterNegotiation();

        yield = 1;

        break;
      }
      default:
      {
        #ifdef PANIC
        *logofs << "Loop: PANIC! Unmanaged case '" << control -> ProxyStage
                << "' while handling negotiation.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Unmanaged case '" << control -> ProxyStage
             << "' while handling negotiation.\n";

        HandleCleanup();
      }
    }
  }

  //
  // Check if the user requested the end of
  // the session.
  //
  
  if (CheckAbort() != 0)
  {
    HandleCleanup();
  }

  //
  // Select the proxy descriptor so that we
  // can proceed negotiating the session.
  //

  FD_SET(proxyFD, &readSet);

  if (proxyFD >= setFDs)
  {
    setFDs = proxyFD + 1;
  }

  setMinTimestamp(selectTs, control -> PingTimeout);

  #ifdef TEST
  *logofs << "Loop: Selected proxy FD#" << proxyFD << " in negotiation "
          << "phase with timeout of " << selectTs.tv_sec << " S and "
          << selectTs.tv_usec << " Ms.\n" << logofs_flush;
  #endif

  return;

handleNegotiationInLoopError:

  #ifdef PANIC
  *logofs << "Loop: PANIC! Failure negotiating the session in stage '"
          << control -> ProxyStage << "'.\n" << logofs_flush;

  if (control -> ProxyStage == stage_waiting_proxy_version)
  {
    *logofs << "Loop: PANIC! This means that we probably didn't provide "
            << "a valid cookie.\n" << logofs_flush;
  }

  *logofs << "Loop: PANIC! The remote NX proxy closed the connection.\n"
          << logofs_flush;
  #endif

  cerr << "Error" << ": Failure negotiating the session in stage '"
       << control -> ProxyStage << "'.\n";

  if (control -> ProxyStage == stage_waiting_proxy_version)
  {
    cerr << "Error" << ": This means that we probably didn't provide "
         << "a valid cookie.\n";
  }

  cerr << "Error" << ": The remote NX proxy closed the connection.\n";

  HandleCleanup();
}

static void handleAlertInLoop()
{
  //
  // Don't try to show the dialogs that are
  // not recognized by the remote proxy.
  //
    
  if (lastAlert.code == 0)
  {
    return;
  }

  if (lastAlert.code > ABORT_PROXY_CONNECTION_ALERT &&
          control -> isProtoStep6() == 0)
  {
    //
    // Ignore the alert request.
    //

    #ifdef WARNING
    *logofs << "Loop: WARNING! Ignoring unsupported alert "
            << "with code '" << lastAlert.code << "'.\n"
            << logofs_flush;
    #endif
  }
  else if (lastAlert.local == 0)
  {
    if (proxy != NULL)
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "Loop: Requesting a remote alert with code '"
              << lastAlert.code << "'.\n" << logofs_flush;
      #endif

      if (proxy -> handleAlert(lastAlert.code) < 0)
      {
        HandleShutdown();
      }
    }
  }
  else
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Handling a local alert with code '"
            << lastAlert.code << "'.\n" << logofs_flush;
    #endif

    if (control -> ProxyMode == proxy_client)
    {
      //
      // If we are at X client side and server
      // proxy is not responding, we don't have
      // any possibility to interact with user.
      //

      if (lastAlert.code != CLOSE_DEAD_PROXY_CONNECTION_CLIENT_ALERT &&
              lastAlert.code != RESTART_DEAD_PROXY_CONNECTION_CLIENT_ALERT &&
                  lastAlert.code != FAILED_PROXY_CONNECTION_CLIENT_ALERT)
      {
        //
        // Let the server proxy show the dialog.
        //

        if (proxy != NULL &&
                proxy -> handleAlert(lastAlert.code) < 0)
        {
          HandleShutdown();
        }
      }
    }
    else
    {
      char caption[DEFAULT_STRING_LENGTH];

      strcpy(caption, ALERT_CAPTION_PREFIX);

      int length = strlen(sessionId);

      //
      // Get rid of the trailing MD5 from session id.
      //

      if (length > (MD5_LENGTH * 2 + 1) &&
              *(sessionId + (length - (MD5_LENGTH * 2 + 1))) == '-')
      {
        strncat(caption, sessionId, length - (MD5_LENGTH * 2 + 1));
      }
      else
      {
        strcat(caption, sessionId);
      }

      //
      // Use the display to which we are forwarding
      // the remote X connections.
      // 

      char *display = displayHost;

      int replace = 1;
      int local   = 1;

      char *message;
      char *type;

      switch (lastAlert.code)
      {
        case CLOSE_DEAD_X_CONNECTION_CLIENT_ALERT:
        {
          message = CLOSE_DEAD_X_CONNECTION_CLIENT_ALERT_STRING;
          type    = CLOSE_DEAD_X_CONNECTION_CLIENT_ALERT_TYPE;

          break;
        }
        case CLOSE_DEAD_X_CONNECTION_SERVER_ALERT:
        {
          message = CLOSE_DEAD_X_CONNECTION_SERVER_ALERT_STRING;
          type    = CLOSE_DEAD_X_CONNECTION_SERVER_ALERT_TYPE;

          break;
        }
        case CLOSE_DEAD_PROXY_CONNECTION_SERVER_ALERT:
        {
          message = CLOSE_DEAD_PROXY_CONNECTION_SERVER_ALERT_STRING;
          type    = CLOSE_DEAD_PROXY_CONNECTION_SERVER_ALERT_TYPE;

          break;
        }
        case RESTART_DEAD_PROXY_CONNECTION_SERVER_ALERT:
        {
          message = RESTART_DEAD_PROXY_CONNECTION_SERVER_ALERT_STRING;
          type    = RESTART_DEAD_PROXY_CONNECTION_SERVER_ALERT_TYPE;

          break;
        }
        case CLOSE_UNRESPONSIVE_X_SERVER_ALERT:
        {
          message = CLOSE_UNRESPONSIVE_X_SERVER_ALERT_STRING;
          type    = CLOSE_UNRESPONSIVE_X_SERVER_ALERT_TYPE;

          break;
        }
        case WRONG_PROXY_VERSION_ALERT:
        {
          message = WRONG_PROXY_VERSION_ALERT_STRING;
          type    = WRONG_PROXY_VERSION_ALERT_TYPE;

          break;
        }
        case FAILED_PROXY_CONNECTION_SERVER_ALERT:
        {
          message = FAILED_PROXY_CONNECTION_SERVER_ALERT_STRING;
          type    = FAILED_PROXY_CONNECTION_SERVER_ALERT_TYPE;

          break;
        }
        case MISSING_PROXY_CACHE_ALERT:
        {
          message = MISSING_PROXY_CACHE_ALERT_STRING;
          type    = MISSING_PROXY_CACHE_ALERT_TYPE;

          break;
        }
        case ABORT_PROXY_CONNECTION_ALERT:
        {
          message = ABORT_PROXY_CONNECTION_ALERT_STRING;
          type    = ABORT_PROXY_CONNECTION_ALERT_TYPE;

          break;
        }
        case DISPLACE_MESSAGE_ALERT:
        {
          message = DISPLACE_MESSAGE_ALERT_STRING;
          type    = DISPLACE_MESSAGE_ALERT_TYPE;

          break;
        }
        case GREETING_MESSAGE_ALERT:
        {
          message = GREETING_MESSAGE_ALERT_STRING;
          type    = GREETING_MESSAGE_ALERT_TYPE;

          break;
        }
        case START_RESUME_SESSION_ALERT:
        {
          message = START_RESUME_SESSION_ALERT_STRING;
          type    = START_RESUME_SESSION_ALERT_TYPE;

          break;
        }
        case FAILED_RESUME_DISPLAY_ALERT:
        {
          message = FAILED_RESUME_DISPLAY_ALERT_STRING;
          type    = FAILED_RESUME_DISPLAY_ALERT_TYPE;

          break;
        }
        case FAILED_RESUME_DISPLAY_BROKEN_ALERT:
        {
          message = FAILED_RESUME_DISPLAY_BROKEN_STRING;
          type    = FAILED_RESUME_DISPLAY_BROKEN_TYPE;

          break;
        }
        case FAILED_RESUME_VISUALS_ALERT:
        {
          message = FAILED_RESUME_VISUALS_ALERT_STRING;
          type    = FAILED_RESUME_VISUALS_ALERT_TYPE;

          break;
        }
        case FAILED_RESUME_COLORMAPS_ALERT:
        {
          message = FAILED_RESUME_COLORMAPS_ALERT_STRING;
          type    = FAILED_RESUME_COLORMAPS_ALERT_TYPE;

          break;
        }
        case FAILED_RESUME_PIXMAPS_ALERT:
        {
          message = FAILED_RESUME_PIXMAPS_ALERT_STRING;
          type    = FAILED_RESUME_PIXMAPS_ALERT_TYPE;

          break;
        }
        case FAILED_RESUME_DEPTHS_ALERT:
        {
          message = FAILED_RESUME_DEPTHS_ALERT_STRING;
          type    = FAILED_RESUME_DEPTHS_ALERT_TYPE;

          break;
        }
        case FAILED_RESUME_RENDER_ALERT:
        {
          message = FAILED_RESUME_RENDER_ALERT_STRING;
          type    = FAILED_RESUME_RENDER_ALERT_TYPE;

          break;
        }
        case FAILED_RESUME_FONTS_ALERT:
        {
          message = FAILED_RESUME_FONTS_ALERT_STRING;
          type    = FAILED_RESUME_FONTS_ALERT_TYPE;

          break;
        }
        case INTERNAL_ERROR_ALERT:
        {
          message = INTERNAL_ERROR_ALERT_STRING;
          type    = INTERNAL_ERROR_ALERT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_NOT_FOUND_ALERT:
        {
          message = REMOTE_SERVER_RDP_NOT_FOUND_STRING;
          type    = REMOTE_SERVER_RDP_NOT_FOUND_TYPE;

          break;
        }
        case REMOTE_SERVER_RFB_NOT_FOUND_ALERT:
        {
          message = REMOTE_SERVER_RFB_NOT_FOUND_STRING;
          type    = REMOTE_SERVER_RFB_NOT_FOUND_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_REFUSED_ALERT:
        {
          message = REMOTE_SERVER_RDP_REFUSED_STRING;
          type    = REMOTE_SERVER_RDP_REFUSED_TYPE;

          break;
        }
        case REMOTE_SERVER_RFB_REFUSED_ALERT:
        {
          message = REMOTE_SERVER_RFB_REFUSED_STRING;
          type    = REMOTE_SERVER_RFB_REFUSED_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_CONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_CONNECT_STRING;
          type    = REMOTE_SERVER_RDP_CONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RFB_CONNECT_ALERT:
        {
          message = REMOTE_SERVER_RFB_CONNECT_STRING;
          type    = REMOTE_SERVER_RFB_CONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_SHUTDOWN_ALERT:
        {
          message = REMOTE_SERVER_RDP_SHUTDOWN_STRING;
          type    = REMOTE_SERVER_RDP_SHUTDOWN_TYPE;

          break;
        }
        case REMOTE_SERVER_RFB_SHUTDOWN_ALERT:
        {
          message = REMOTE_SERVER_RFB_SHUTDOWN_STRING;
          type    = REMOTE_SERVER_RFB_SHUTDOWN_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_PROTOCOL_ALERT:
        {
          message = REMOTE_SERVER_RDP_PROTOCOL_STRING;
          type    = REMOTE_SERVER_RDP_PROTOCOL_TYPE;

          break;
        }
        case REMOTE_SERVER_RFB_PROTOCOL_ALERT:
        {
          message = REMOTE_SERVER_RFB_PROTOCOL_STRING;
          type    = REMOTE_SERVER_RFB_PROTOCOL_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_AUTHENTICATION_ALERT:
        {
          message = REMOTE_SERVER_RDP_AUTHENTICATION_STRING;
          type    = REMOTE_SERVER_RDP_AUTHENTICATION_TYPE;

          break;
        }
        case REMOTE_SERVER_RFB_AUTHENTICATION_ALERT:
        {
          message = REMOTE_SERVER_RFB_AUTHENTICATION_STRING;
          type    = REMOTE_SERVER_RFB_AUTHENTICATION_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_COLOR_LIMIT_ALERT:
        {
          message = REMOTE_SERVER_RDP_COLOR_LIMIT_STRING;
          type    = REMOTE_SERVER_RDP_COLOR_LIMIT_TYPE;

          break;
        }
        case REMOTE_SERVER_RFB_COLOR_LIMIT_ALERT:
        {
          message = REMOTE_SERVER_RFB_COLOR_LIMIT_STRING;
          type    = REMOTE_SERVER_RFB_COLOR_LIMIT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_GEOMETRY_LIMIT_ALERT:
        {
          message = REMOTE_SERVER_RDP_GEOMETRY_LIMIT_STRING;
          type    = REMOTE_SERVER_RDP_GEOMETRY_LIMIT_TYPE;

          break;
        }
        case REMOTE_SERVER_RFB_GEOMETRY_LIMIT_ALERT:
        {
          message = REMOTE_SERVER_RFB_GEOMETRY_LIMIT_STRING;
          type    = REMOTE_SERVER_RFB_GEOMETRY_LIMIT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_TERMINATED_ALERT:
        {
          message = REMOTE_SERVER_RDP_TERMINATED_STRING;
          type    = REMOTE_SERVER_RDP_TERMINATED_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_MANDATED_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_MANDATED_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_MANDATED_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_LOGOFF_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_LOGOFF_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_LOGOFF_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_IDLE_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_IDLE_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_IDLE_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_LOGON_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_LOGON_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_LOGON_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_ADMIN_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_ADMIN_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_ADMIN_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_MEMORY_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_MEMORY_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_MEMORY_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_DENIED_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_DENIED_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_DENIED_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_SECURITY_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_SECURITY_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_SECURITY_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_FORBIDDEN_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_FORBIDDEN_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_FORBIDDEN_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_PROTOCOL_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_PROTOCOL_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_PROTOCOL_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_INVALID_LICENSE_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_INVALID_LICENSE_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_INVALID_LICENSE_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_NETWORK_LICENSE_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_NETWORK_LICENSE_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_NETWORK_LICENSE_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_NO_LICENSE_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_NO_LICENSE_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_NO_LICENSE_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_LICENSE_MESSAGE_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_LICENSE_MESSAGE_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_LICENSE_MESSAGE_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_LICENSE_SYSTEM_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_LICENSE_SYSTEM_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_LICENSE_SYSTEM_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_LICENSE_ERROR_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_LICENSE_ERROR_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_LICENSE_ERROR_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_LICENSE_PROTO_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_LICENSE_PROTO_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_LICENSE_PROTO_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_LICENSE_ABORT_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_LICENSE_ABORT_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_LICENSE_ABORT_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_LICENSE_FORMAT_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_LICENSE_FORMAT_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_LICENSE_FORMAT_DISCONNECT_TYPE;

          break;
        }
        case REMOTE_SERVER_RDP_LICENSE_UPGRADE_DISCONNECT_ALERT:
        {
          message = REMOTE_SERVER_RDP_LICENSE_UPGRADE_DISCONNECT_STRING;
          type    = REMOTE_SERVER_RDP_LICENSE_UPGRADE_DISCONNECT_TYPE;

          break;
        }
        default:
        {
          #ifdef WARNING
          *logofs << "Loop: WARNING! An unrecognized alert type '"
                  << lastAlert.code << "' was requested.\n"
                  << logofs_flush;
          #endif

          cerr << "Warning" << ": An unrecognized alert type '"
               << lastAlert.code << "' was requested.\n";

          message = NULL;
          type    = NULL;

          replace = 0;

          break;
        }
      }

      if (replace == 1 && lastDialog > 0)
      {
        #if defined(TEST) || defined(INFO)
        *logofs << "Loop: Killing the previous dialog with pid '"
                << lastDialog << "'.\n" << logofs_flush;
        #endif

        //
        // The client ignores the TERM signal
        // on Windows.
        //

        #ifdef __CYGWIN32__

        KillProcess(lastDialog, "dialog", SIGKILL, 0);

        #else

        KillProcess(lastDialog, "dialog", SIGTERM, 0);

        #endif

        lastDialog = 0;
      }

      if (message != NULL && type != NULL)
      {
        lastDialog = NXTransDialog(caption, message, 0, type, local, display);

        if (lastDialog < 0)
        {
          #ifdef PANIC
          *logofs << "Loop: PANIC! Can't start the NX dialog process.\n"
                  << logofs_flush;
          #endif

          lastDialog = 0;
        }
        #if defined(TEST) || defined(INFO)
        else
        {
          *logofs << "Loop: Dialog started with pid '"
                  << lastDialog << "'.\n" << logofs_flush;
        }
        #endif
      }
      #if defined(TEST) || defined(INFO)
      else
      {
        *logofs << "Loop: No new dialog required for code '"
                << lastAlert.code << "'.\n" << logofs_flush;
      }
      #endif
    }
  }

  //
  // Reset state.
  //

  lastAlert.code  = 0;
  lastAlert.local = 0;
}

static inline void handleSetAgentInLoop(int &setFDs, fd_set &readSet,
                                            fd_set &writeSet, struct timeval &selectTs)
{
  #ifdef TEST
  *logofs << "Loop: Preparing the masks for the agent descriptors.\n"
          << logofs_flush;
  #endif

  agent -> saveChannelState();

  agent -> saveReadMask(&readSet);
  agent -> saveWriteMask(&writeSet);

  if (control -> ProxyStage == stage_operational)
  {
    //
    // We don't need to care about the local
    // side. It will be handled as it was a
    // channel having data pending.
    //

    if (agent -> remoteCanRead(&readSet) ||
            agent -> remoteCanWrite(&writeSet) ||
                agent -> localCanRead() ||
                    agent -> proxyCanRead())
    {
      #ifdef TEST
      *logofs << "Loop: Setting a null timeout with agent descriptors ready.\n"
              << logofs_flush;
      #endif

      //
      // Force a null timeout so we'll bail out
      // of the select immediately. We will ac-
      // comodate the result code later.
      //

      selectTs.tv_sec  = 0;
      selectTs.tv_usec = 0;

      agent -> setSelected();
    }
  }

  #ifdef TEST
  *logofs << "Loop: Clearing the read and write agent descriptors.\n"
          << logofs_flush;
  #endif

  agent -> clearReadMask(&readSet);
  agent -> clearWriteMask(&writeSet);
}

static inline void handleAgentInLoop(int &resultFDs, int &errorFDs, int &setFDs, fd_set &readSet,
                                         fd_set &writeSet, struct timeval &selectTs)
{
  #if defined(TEST) || defined(INFO)
  *logofs << "Loop: Setting proxy and local agent descriptors.\n"
          << logofs_flush;
  #endif

  //
  // We skipped the select, if possible.
  //

  agent -> resetSelected();

  //
  // Save if the proxy can read from the
  // the agent descriptor.
  //

  agent -> saveChannelState();

  //
  // Check if I/O is possible on the local
  // agent or the proxy descriptor.
  //
  
  #if defined(TEST) || defined(INFO)
  *logofs << "Loop: Values were resultFDs " << resultFDs
          << " errorFDs " << errorFDs << " setFDs "
          << setFDs << ".\n" << logofs_flush;
  #endif

  int localRead = 0;
  int proxyRead = 0;

  int total = 0;

  if (agent -> localCanRead() == 1)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Setting agent descriptor FD#" << agent ->
               getLocalFd() << " as ready to read.\n"
            << logofs_flush;
    #endif

    localRead = 1;

    total++;
  }

  if (agent -> proxyCanRead() == 1)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Setting proxy descriptor FD#" << agent ->
               getProxyFd() << " as ready to read.\n"
            << logofs_flush;
    #endif

    proxyRead = 1;

    total++;
  }

  if (total != 0)
  {
    if (resultFDs <= 0)
    {
      //
      // Need to clear the remaining descriptors
      // or will cause the caller to read from
      // descriptors that were set in the origi-
      // nal masks.
      //

      FD_ZERO(&readSet);
      FD_ZERO(&writeSet);

      resultFDs = 0;
      errorFDs  = 0;
      setFDs    = 0;
    }

    if (localRead == 1)
    {
      agent -> setLocalRead();
    }
    else
    {
      agent -> clearLocalRead(&readSet);
    }

    if (proxyRead == 1)
    {
      agent -> setProxyRead();
    }
    else
    {
      agent -> clearProxyRead(&readSet);
    }
  }

  #if defined(TEST) || defined(INFO)
  *logofs << "Loop: Values are now resultFDs " << resultFDs
          << " errorFDs " << errorFDs << " setFDs "
          << setFDs << ".\n" << logofs_flush;
  #endif
}

static inline void handleAgentLateInLoop(int &resultFDs, int &errorFDs, int &setFDs, fd_set &readSet,
                                             fd_set &writeSet, struct timeval &selectTs)
{
  #if defined(TEST) || defined(INFO)
  *logofs << "Loop: Setting remote agent descriptors.\n"
          << logofs_flush;
  #endif

  //
  // We reset the masks before calling our select.
  // We now set the descriptors that are ready but
  // only if they were set in the original mask.
  // We do this after having executed our loop as
  // the agent descriptors may have become reada-
  // ble or writable in the meanwhile.
  //
  
  #if defined(TEST) || defined(INFO)
  *logofs << "Loop: Values were resultFDs " << resultFDs
          << " errorFDs " << errorFDs << " setFDs "
          << setFDs << ".\n" << logofs_flush;
  #endif

  //
  // Save if the proxy can read from the
  // the agent descriptor.
  //

  agent -> saveChannelState();

  int remoteRead  = 0;
  int remoteWrite = 0;

  int total = 0;

  if (agent -> remoteCanRead(agent ->
          getSavedReadMask()) == 1)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Setting agent descriptor FD#" << agent ->
               getRemoteFd() << " as ready to read.\n"
            << logofs_flush;
    #endif

    remoteRead = 1;

    total++;
  }

  if (agent -> remoteCanWrite(agent ->
          getSavedWriteMask()) == 1)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Loop: Setting agent descriptor FD#" << agent ->
               getRemoteFd() << " as ready to write.\n"
            << logofs_flush;
    #endif

    remoteWrite = 1;

    total++;
  }

  if (total != 0)
  {
    if (resultFDs <= 0)
    {
      //
      // Need to clear the remaining descriptors
      // or will cause the caller to read from
      // descriptors that were set in the origi-
      // nal masks.
      //

      FD_ZERO(&readSet);
      FD_ZERO(&writeSet);

      resultFDs = 0;
      errorFDs  = 0;
      setFDs    = 0;
    }

    if (remoteRead == 1)
    {
      agent -> setRemoteRead(&setFDs, &readSet);

      resultFDs++;
    }
    else
    {
      agent -> clearRemoteRead(&readSet);
    }

    if (remoteWrite == 1)
    {
      agent -> setRemoteWrite(&setFDs, &writeSet);

      resultFDs++;
    }
    else
    {
      agent -> clearRemoteWrite(&writeSet);
    }
  }

  #if defined(TEST) || defined(INFO)
  *logofs << "Loop: Values are now resultFDs " << resultFDs
          << " errorFDs " << errorFDs << " setFDs "
          << setFDs << ".\n" << logofs_flush;
  #endif
}

static void handlePingInLoop(int &diffTs)
{
  #ifdef TEST
  if (diffTs >= (control -> PingTimeout -
                     control -> LatencyTimeout))
  {
    *logofs << "Loop: No communication within nearly "
            << control -> PingTimeout / 1000 << " seconds.\n"
            << logofs_flush;
  }
  #endif

  if (proxy -> getShutdown() > 0)
  {
    #ifdef TEST
    *logofs << "Loop: End of session requested by "
            << "remote proxy.\n" << logofs_flush;
    #endif

    cerr << "Info" << ": End of session requested by "
         << "remote proxy.\n";
 
    HandleShutdown();
  }
  else if (proxy -> handlePing() < 0)
  {
    #ifdef TEST
    *logofs << "Loop: Proxy failure in handlePing().\n"
            << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static inline void handleReadableInLoop(int &resultFDs, fd_set &readSet)
{
  if (resultFDs > 0)
  {
    if (tcpFD != -1 && FD_ISSET(tcpFD, &readSet))
    {
      handleAcceptTcpConnectionInLoop(tcpFD);

      resultFDs--;
    }

    if (unixFD != -1 && FD_ISSET(unixFD, &readSet))
    {
      handleAcceptUnixConnectionInLoop(unixFD);

      resultFDs--;
    }

    if (cupsFD != -1 && FD_ISSET(cupsFD, &readSet))
    {
      handleAcceptCupsConnectionInLoop(cupsFD);

      resultFDs--;
    }

    if (keybdFD != -1 && FD_ISSET(keybdFD, &readSet))
    {
      handleAcceptKeybdConnectionInLoop(keybdFD);

      resultFDs--;
    }

    if (sambaFD != -1 && FD_ISSET(sambaFD, &readSet))
    {
      handleAcceptSambaConnectionInLoop(sambaFD);

      resultFDs--;
    }

    if (mediaFD != -1 && FD_ISSET(mediaFD, &readSet))
    {
      handleAcceptMediaConnectionInLoop(mediaFD);

      resultFDs--;
    }

    if (httpFD != -1 && FD_ISSET(httpFD, &readSet))
    {
      handleAcceptHttpConnectionInLoop(httpFD);

      resultFDs--;
    }
  }

  //
  // We must let proxy check each channel even
  // if no descriptor is set as there might be
  // channels with data pending from previous
  // reads.
  //

  #ifdef DEBUG
  *logofs << "Loop: Going to check the readable descriptors.\n"
          << logofs_flush;
  #endif

  if (proxy -> handleRead(resultFDs, readSet) < 0)
  {
    #ifdef TEST
    *logofs << "Loop: Failure reading from descriptors "
            << "for proxy FD#" << proxyFD << ".\n"
            << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static inline void handleWritableInLoop(int &resultFDs, fd_set &writeSet)
{
  #ifdef DEBUG
  *logofs << "Loop: Going to check the writable descriptors.\n"
          << logofs_flush;
  #endif

  if (resultFDs > 0 && proxy -> handleFlush(resultFDs, writeSet) < 0)
  {
    #ifdef TEST
    *logofs << "Loop: Failure writing to descriptors "
            << "for proxy FD#" << proxyFD << ".\n"
            << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static inline void handleChannelEventsInLoop()
{
  #ifdef DEBUG
  *logofs << "Loop: Going to check channel events "
          << "for proxy FD#" << proxyFD << ".\n"
          << logofs_flush;
  #endif

  if (proxy -> handleEvents() < 0)
  {
    #ifdef TEST
    *logofs << "Loop: Failure handling channel events "
            << "for proxy FD#" << proxyFD << ".\n"
            << logofs_flush;
    #endif

    HandleShutdown();
  }
}

static void handleLogReopenInLoop(T_timestamp &logsTs, T_timestamp &nowTs)
{
  if (diffTimestamp(logsTs, nowTs) > control -> FileSizeCheckTimeout)
  {
    #ifdef DEBUG
    *logofs << "Loop: Checking size of log file '"
            << logFileName << "'.\n" << logofs_flush;
    #endif

    #ifndef MIXED

    if (ReopenOutputFile(logFileName, logofs, control -> LogFileSizeLimit) < 0)
    {
      HandleShutdown();
    }

    #endif

    //
    // Reset to current timestamp.
    //

    logsTs = nowTs;
  }
}

static void handleAcceptTcpConnectionInLoop(int &fd)
{
  #ifdef TEST
  *logofs << "Loop: Going to accept new connection on TCP socket FD#"
          << fd << ".\n" << logofs_flush;
  #endif

  sockaddr_in newAddr;

  size_t addrLen = sizeof(sockaddr_in);

  int newFD = accept(tcpFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed. Error is "
            << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed. Error is "
         << EGET() << " '" << ESTR() << "'.\n";

    return;
  }

  if (proxy -> handleNewXConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new X connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new X connection.\n";

    close(newFD);

    HandleCleanup();
  }
}

static void handleAcceptUnixConnectionInLoop(int &fd)
{
  #ifdef TEST
  *logofs << "Loop: Going to accept new connection on UNIX socket FD#"
          << fd << ".\n" << logofs_flush;
  #endif

  sockaddr_un newAddr;

  size_t addrLen = sizeof(sockaddr_un);

  int newFD = accept(unixFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed. Error is "
            << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed. Error is "
         << EGET() << " '" << ESTR() << "'.\n";

    return;
  }

  if (proxy -> handleNewXConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new X connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new X connection.\n";

    close(newFD);

    HandleCleanup();
  }
}

//
// Same copy-paste. Write a generic function.
//

static void handleAcceptCupsConnectionInLoop(int &fd)
{
  #ifdef TEST
  *logofs << "Loop: Going to accept new cups connection "
          << "on TCP socket FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  sockaddr_in newAddr;

  size_t addrLen = sizeof(sockaddr_in);

  int newFD = accept(cupsFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed for cups. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed for cups. "
         << "Error is " << EGET() << " '" << ESTR() << "'.\n";

    return;
  }

  if (proxy -> handleNewCupsConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new cups connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new cups connection.\n";

    close(newFD);
  }
}

static void handleAcceptKeybdConnectionInLoop(int &fd)
{
  #ifdef TEST
  *logofs << "Loop: Going to accept new embedded keyboard connection "
          << "on TCP socket FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  sockaddr_in newAddr;

  size_t addrLen = sizeof(sockaddr_in);

  int newFD = accept(keybdFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed for embedded keyboard. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed for embedded keyboard. "
         << "Error is " << EGET() << " '" << ESTR() << "'.\n";

    return;
  }

  //
  // Starting from version 1.5.0 we create real X
  // connections for the keyboard channel, so they
  // can use the fake authorization cookie. This
  // means that there is not such a thing like a
  // channel_keybd anymore.
  //

  if (proxy -> handleNewXConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new embedded keyboard connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new embedded keyboard connection.\n";

    close(newFD);
  }
}

static void handleAcceptSambaConnectionInLoop(int &fd)
{
  #ifdef TEST
  *logofs << "Loop: Going to accept new SMB connection "
          << "on TCP socket FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  sockaddr_in newAddr;

  size_t addrLen = sizeof(sockaddr_in);

  int newFD = accept(sambaFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed for SMB. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed for SMB. "
         << "Error is " << EGET() << " '" << ESTR() << "'.\n";

    return;
  }

  if (proxy -> handleNewSambaConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new SMB connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new SMB connection.\n";

    close(newFD);
  }
}

static void handleAcceptMediaConnectionInLoop(int &fd)
{
  #ifdef TEST
  *logofs << "Loop: Going to accept new multimedia connection "
          << "on TCP socket FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  sockaddr_in newAddr;

  size_t addrLen = sizeof(sockaddr_in);

  int newFD = accept(mediaFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed for multimedia. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed for multimedia. "
         << "Error is " << EGET() << " '" << ESTR() << "'.\n";

    return;
  }

  if (proxy -> handleNewMediaConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new multimedia connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new multimedia connection.\n";

    close(newFD);
  }
}

static void handleAcceptHttpConnectionInLoop(int &fd)
{
  #ifdef TEST
  *logofs << "Loop: Going to accept new http connection "
          << "on TCP socket FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  sockaddr_in newAddr;

  size_t addrLen = sizeof(sockaddr_in);

  int newFD = accept(httpFD, (sockaddr *) &newAddr, (socklen_t *) &addrLen);

  if (newFD < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Call to accept failed for HTTP. "
            << "Error is " << EGET() << " '" << ESTR() << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Call to accept failed for HTTP. "
         << "Error is " << EGET() << " '" << ESTR() << "'.\n";

    return;
  }

  if (proxy -> handleNewHttpConnection(newFD) < 0)
  {
    #ifdef PANIC
    *logofs << "Loop: PANIC! Error creating new HTTP connection.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Error creating new HTTP connection.\n";

    close(newFD);
  }
}

static inline void handleSetReadInLoop(fd_set &readSet, int &setFDs, struct timeval &selectTs)
{
  proxy -> setReadDescriptors(&readSet, setFDs, selectTs);
}

static inline void handleSetWriteInLoop(fd_set &writeSet, int &setFDs, struct timeval &selectTs)
{
  proxy -> setWriteDescriptors(&writeSet, setFDs, selectTs);
}

static void handleSetListenersInLoop(fd_set &readSet, int &setFDs)
{
  //
  // Set descriptors of listening sockets.
  //

  if (useTCPSocket)
  {
    FD_SET(tcpFD, &readSet);

    if (tcpFD >= setFDs)
    {
      setFDs = tcpFD + 1;
    }

    #ifdef DEBUG
    *logofs << "Loop: Selected listener tcpFD " << tcpFD
            << " with setFDs " << setFDs << ".\n"
            << logofs_flush;
    #endif
  }

  if (useUnixSocket)
  {
    FD_SET(unixFD, &readSet);

    if (unixFD >= setFDs)
    {
      setFDs = unixFD + 1;
    }

    #ifdef DEBUG
    *logofs << "Loop: Selected listener unixFD " << unixFD
            << " with setFDs " << setFDs << ".\n"
            << logofs_flush;
    #endif
  }

  if (useCupsSocket)
  {
    FD_SET(cupsFD, &readSet);

    if (cupsFD >= setFDs)
    {
      setFDs = cupsFD + 1;
    }

    #ifdef DEBUG
    *logofs << "Loop: Selected listener cupsFD " << cupsFD
            << " with setFDs " << setFDs << ".\n"
            << logofs_flush;
    #endif
  }

  if (useKeybdSocket)
  {
    FD_SET(keybdFD, &readSet);

    if (keybdFD >= setFDs)
    {
      setFDs = keybdFD + 1;
    }

    #ifdef DEBUG
    *logofs << "Loop: Selected listener keybdFD " << keybdFD
            << " with setFDs " << setFDs << ".\n"
            << logofs_flush;
    #endif
  }

  if (useSambaSocket)
  {
    FD_SET(sambaFD, &readSet);

    if (sambaFD >= setFDs)
    {
      setFDs = sambaFD + 1;
    }

    #ifdef DEBUG
    *logofs << "Loop: Selected listener sambaFD " << sambaFD
            << " with setFDs " << setFDs << ".\n"
            << logofs_flush;
    #endif
  }

  if (useMediaSocket)
  {
    FD_SET(mediaFD, &readSet);

    if (mediaFD >= setFDs)
    {
      setFDs = mediaFD + 1;
    }

    #ifdef DEBUG
    *logofs << "Loop: Selected listener mediaFD " << mediaFD
            << " with setFDs " << setFDs << ".\n"
            << logofs_flush;
    #endif
  }

  if (useHttpSocket)
  {
    FD_SET(httpFD, &readSet);

    if (httpFD >= setFDs)
    {
      setFDs = httpFD + 1;
    }

    #ifdef DEBUG
    *logofs << "Loop: Selected listener httpFD " << httpFD
            << " with setFDs " << setFDs << ".\n"
            << logofs_flush;
    #endif
  }
}
