/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdio.h>
#include <fstream.h>
#include <unistd.h>
#include <string.h>

#include <algorithm>

//
// We need channel's cache data.
//

#include "Message.h"

#include "EncodeBuffer.h"
#include "DecodeBuffer.h"

//
// Set the verbosity level. You also
// need to define DUMP in Misc.cpp
// if DUMP is defined here.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG
#undef  DUMP

//
// Define this to log when messages
// are allocated and deallocated.
//

#undef  REFERENCES

//
// Keep track of how many bytes
// are taken by cache.
//

int MessageStore::totalLocalStorageSize_  = 0;
int MessageStore::totalRemoteStorageSize_ = 0;

//
// These are used for reference count.
//

#ifdef REFERENCES

int Message::references_        = 0;
int MessageStore::references_   = 0;

#endif

//
// Here are the methods to handle cached messages.
//

MessageStore::MessageStore(Compressor *compressor,
                               Decompressor *decompressor)

  : compressor_(compressor), decompressor_(decompressor)
{
  //
  // Public members.
  //

  enableCache    = MESSAGE_ENABLE_CACHE;
  enableData     = MESSAGE_ENABLE_DATA;
  enableSplit    = MESSAGE_ENABLE_SPLIT;
  enableCompress = MESSAGE_ENABLE_COMPRESS;

  dataLimit  = MESSAGE_DATA_LIMIT;
  dataOffset = MESSAGE_DATA_OFFSET;

  cacheSlots          = MESSAGE_CACHE_SLOTS;
  cacheThreshold      = MESSAGE_CACHE_THRESHOLD;
  cacheLowerThreshold = MESSAGE_CACHE_LOWER_THRESHOLD;

  #ifdef TEST
  *logofs << "MessageStore: Compressor is at " << compressor_
          << " decompressor is at " << decompressor_ << ".\n"
          << logofs_flush;
  #endif

  md5_state_ = new md5_state_t();

  #ifdef DEBUG
  *logofs << "MessageStore: Created MD5 state for object at " 
          << this << ".\n" << logofs_flush;
  #endif

  lastAdded   = cacheSlots;
  lastHit     = 0;
  lastRemoved = 0;
  lastRated   = nothing;

  lastResize = 0;
  lastStatus = is_discarded;

  //
  // Private members.
  //

  localStorageSize_  = 0;
  remoteStorageSize_ = 0;

  #ifdef TEST
  *logofs << "MessageStore: Size of total cache is " 
          << totalLocalStorageSize_ << " bytes at local side and " 
          << totalRemoteStorageSize_ << " bytes at remote side.\n" 
          << logofs_flush;
  #endif

  messages_  = new T_messages();
  checksums_ = new T_checksums();

  temporary_ = NULL;

  timestamp_ = nullTimestamp();

  kept_ = 0;

  #ifdef REFERENCES

  references_++;

  *logofs << "MessageStore: Created new store at " 
          << this << "out of " << references_
          << " allocated stores.\n" << logofs_flush;

  #endif
}

MessageStore::~MessageStore()
{
  //
  // The virtual destructor of specialized class
  // must get rid of both messages in container
  // and temporary.
  //

  #ifdef DEBUG
  *logofs << "MessageStore: Deleting MD5 state for object at "
          << this << ".\n" << logofs_flush;
  #endif

  delete md5_state_;

  delete messages_;
  delete checksums_;

  //
  // Update the static members tracking
  // size of total memory allocated for
  // all stores.
  //

  totalLocalStorageSize_  -= localStorageSize_;
  totalRemoteStorageSize_ -= remoteStorageSize_;

  #ifdef TEST
  *logofs << "MessageStore: Size of total cache is " 
          << totalLocalStorageSize_ << " bytes at local side and " 
          << totalRemoteStorageSize_ << " bytes at remote side.\n" 
          << logofs_flush;
  #endif

  #ifdef REFERENCES

  references_--;

  *logofs << "MessageStore: Deleted store at " 
          << this << " out of " << references_
          << " allocated stores.\n" << logofs_flush;

  #endif
}

//
// Here are the methods to handle messages' content.
//

int MessageStore::parse(Message *message, const unsigned char *buffer,
                            unsigned int size, T_checksum_action checksumAction,
                                T_data_action dataAction, int bigEndian)
{
  if (size < 4 || size > MESSAGE_DATA_LIMIT)
  {
    #ifdef PANIC
    *logofs << name() << ": PANIC! Size of message cannot be "
            << size << " bytes.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Size of message cannot be " << size
         << " bytes.\n" << logofs_flush;

    cerr << "Error" << ": Assuming failure decoding message "
         << "of type " << name() << ".\n";

    HandleAbort();
  }

  //
  // Save the message size as received on the link.
  // This information will be used to create an
  // appropriate buffer at the time message will be
  // unparsed.
  //
  // The identity size can be later modified by the
  // parseIdentity() function.
  //

  int offset = identitySize(buffer, size);

  message -> size_   = size;
  message -> i_size_ = offset;
  message -> c_size_ = 0;

  //
  // Do a cast to allow function to change the data.
  // This is made outdated starting from step 3 of
  // the protocol.
  //

  parseBegin(message, (unsigned char *) buffer, size, bigEndian);

  if (checksumAction == use_checksum)
  {
    beginChecksum(message);

    parseIdentity(message, buffer, size, bigEndian);

    identityChecksum(message, buffer, size, bigEndian);

    parseData(message, buffer, size, checksumAction, dataAction, bigEndian);

    endChecksum(message);
  }
  else
  {
    parseIdentity(message, buffer, size, bigEndian);

    parseData(message, buffer, size, checksumAction, dataAction, bigEndian);
  }

  parseEnd(message, (unsigned char *) buffer, size, bigEndian);

  return true;
}

int MessageStore::parse(Message *message, const unsigned char *buffer,
                            unsigned int size, const unsigned char *compressedData,
                                const unsigned int compressedDataSize,
                                    T_checksum_action checksumAction,
                                        T_data_action dataAction, int bigEndian)
{
  if (size < 4 || size > MESSAGE_DATA_LIMIT)
  {
    #ifdef PANIC
    *logofs << name() << ": PANIC! Size of message cannot be "
            << size << " bytes.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Size of message cannot be " << size
         << " bytes.\n" << logofs_flush;

    cerr << "Error" << ": Assuming failure decoding message "
         << "of type " << name() << ".\n";

    HandleAbort();
  }

  //
  // The identity size can be later modified
  // by the parseIdentity() function.
  //

  int offset = identitySize(buffer, size);

  message -> size_   = size;
  message -> i_size_ = offset;
  message -> c_size_ = compressedDataSize + offset;

  parseBegin(message, (unsigned char *) buffer, size, bigEndian);

  if (checksumAction == use_checksum)
  {
    beginChecksum(message);

    parseIdentity(message, buffer, size, bigEndian);

    identityChecksum(message, buffer, size, bigEndian);

    parseData(message, buffer, size, compressedData, compressedDataSize,
                  checksumAction, dataAction, bigEndian);

    endChecksum(message);
  }
  else
  {
    parseIdentity(message, buffer, size, bigEndian);

    parseData(message, buffer, size, compressedData, compressedDataSize,
                  checksumAction, dataAction, bigEndian);
  }

  parseEnd(message, (unsigned char *) buffer, size, bigEndian);

  return true;
}

int MessageStore::parseData(Message *message, const unsigned char *buffer,
                                unsigned int size, T_checksum_action checksumAction,
                                    T_data_action dataAction, int bigEndian)
{
  if ((int) size > message -> i_size_)
  {
    unsigned int dataSize = size - message -> i_size_;

    if (checksumAction == use_checksum)
    {
      #ifdef DEBUG
      *logofs << name() << ": Calculating checksum of object at "
              << message << " with data size " << dataSize
              << ".\n" << logofs_flush;
      #endif

      dataChecksum(message, buffer, size, bigEndian);
    }

    if (dataAction == discard_data)
    {
      #ifdef DEBUG
      *logofs << name() << ": Discarded " << dataSize
              << " bytes of plain data. Real size is "
              << message -> size_ << " compressed size is "
              << message -> c_size_ << ".\n" << logofs_flush;
      #endif

      return true;
    }

    //
    // Accept anyway data beyond the limit
    // but issue a warning.
    //

    #ifdef WARNING
    if (dataSize > (unsigned int) dataLimit)
    {
      *logofs << name() << ": WARNING! Data is " << dataSize
              << " bytes. Ignoring the established limit!\n"
              << logofs_flush;
    }
    #endif

    if (dataSize != message -> data_.size())
    {
      #ifdef DEBUG
      *logofs << name() << ": Data will be resized from "
              << message -> data_.size() << " to hold a plain buffer of "
              << dataSize << " bytes.\n" << logofs_flush;
      #endif

      try
      {
        message -> data_.clear();

        message -> data_.resize(dataSize);
      }
      catch (exception e)
      {
        #ifdef PANIC
        *logofs << name() << ": PANIC! Exception caught resizing data."
                << " Error is '" << e.what() << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Exception caught resizing data."
             << " Error is '" << e.what() << "'.\n";

        return false;
      }
    }

    memcpy(message -> data_.begin(), buffer + message -> i_size_, dataSize);

    #ifdef DEBUG
    *logofs << name() << ": Parsed " << dataSize
            << " bytes of plain data. Real size is "
            << message -> size_ << " compressed size is "
            << message -> c_size_ << ".\n" << logofs_flush;
    #endif
  }

  return true;
}

//
// Store the data part in compressed format.
//

int MessageStore::parseData(Message *message, const unsigned char *buffer,
                                unsigned int size, const unsigned char *compressedData,
                                    const unsigned int compressedDataSize,
                                        T_checksum_action checksumAction,
                                             T_data_action dataAction, int bigEndian)
{
  if ((int) size > message -> i_size_)
  {
    unsigned int dataSize = size - message -> i_size_;

    if (checksumAction == use_checksum)
    {
      #ifdef DEBUG
      *logofs << name() << ": Calculating checksum of object at "
              << message << " with data size " << dataSize
              << ".\n" << logofs_flush;
      #endif

      dataChecksum(message, buffer, size, bigEndian);
    }

    if (dataAction == discard_data)
    {
      #ifdef DEBUG
      *logofs << name() << ": Discarded " << dataSize
              << " bytes of compressed data. Real size is "
              << message -> size_ << " compressed size is "
              << message -> c_size_ << ".\n" << logofs_flush;
      #endif

      return true;
    }

    #ifdef WARNING
    if (dataSize > (unsigned int) dataLimit)
    {
      *logofs << name() << ": WARNING! Data is " << dataSize
              << " bytes. Ignoring the established limit!\n"
              << logofs_flush;
    }
    #endif

    dataSize = compressedDataSize;

    if (dataSize != message -> data_.size())
    {
      #ifdef DEBUG
      *logofs << name() << ": Data will be resized from "
              << message -> data_.size() << " to hold a compressed buffer of "
              << dataSize << " bytes.\n" << logofs_flush;
      #endif

      try
      {
        message -> data_.clear();

        message -> data_.resize(compressedDataSize);
      }
      catch (exception e)
      {
        #ifdef PANIC
        *logofs << name() << ": PANIC! Exception caught resizing data."
                << " Error is '" << e.what() << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Exception caught resizing data."
             << " Error is '" << e.what() << "'.\n";

        return false;
      }
    }

    memcpy(message -> data_.begin(), compressedData, compressedDataSize);

    #ifdef DEBUG
    *logofs << name() << ": Parsed " << dataSize
            << " bytes of compressed data. Real size is "
            << message -> size_ << " compressed size is "
            << message -> c_size_ << ".\n" << logofs_flush;
    #endif
  }

  return true;
}

int MessageStore::unparseData(const Message *message, unsigned char *buffer,
                                  unsigned int size, int bigEndian)
{
  //
  // Copy data, if any, to the buffer.
  //

  if ((int) size > message -> i_size_)
  {
    //
    // Check if message has been stored
    // in compressed format.
    //

    if (message -> c_size_ == 0)
    {
      memcpy(buffer + message -> i_size_, message -> data_.begin(), size - message -> i_size_);

      #ifdef DEBUG
      *logofs << name() << ": Unparsed " << message -> size_ - message -> i_size_
              << " bytes of data to a buffer of " << message -> size_ - message -> i_size_
              << ".\n" << logofs_flush;
      #endif
    }
    else
    {
      #ifdef DEBUG
      *logofs << name() << ": Using decompressor at " << (void *) decompressor_
              << ".\n" << logofs_flush;
      #endif

      if (decompressor_ ->
              decompressBuffer(buffer + message -> i_size_,
                                   size - message -> i_size_,
                                       message -> data_.begin(),
                                           message -> c_size_ - message -> i_size_) < 0)
      {
        #ifdef PANIC
        *logofs << name() << ": PANIC! Data decompression failed.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Data decompression failed.\n";

        return -1;
      }

      #ifdef DEBUG
      *logofs << name() << ": Unparsed " << message -> c_size_ - message -> i_size_
              << " bytes of compressed data to a buffer of "
              << message -> size_ - message -> i_size_ << ".\n" << logofs_flush;
      #endif
    }
  }

  //
  // We could write size to the buffer but this
  // is something the channel class is doing by
  // itself.
  //
  // PutUINT(size >> 2, buffer + 2, bigEndian);
  //

  return true;
}

void MessageStore::dumpData(const Message *message) const
{
  #ifdef DUMP

  *logofs << name() << ": Dumping enumerated data:\n" << logofs_flush; 

  DumpData(message -> data_.begin(), message -> data_.size());

  #endif

  #ifdef DUMP

  *logofs << name() << ": Dumping checksum data:\n" << logofs_flush; 

  DumpData(message -> md5_digest_, MD5_LENGTH);

  #endif
}

int MessageStore::find(const Message *message, 
                           T_checksums::iterator &found) const
{
  md5_byte_t *checksum = getChecksum(message);

  #ifdef TEST
  *logofs << name() << ": Searching checksum ["
          << DumpChecksum(checksum) << "] in repository.\n"
          << logofs_flush;
  #endif

  found = checksums_ -> find(checksum);

  if (found != checksums_ -> end())
  {
    #ifdef TEST
    *logofs << name() << ": Message found in cache at "
            << "position " << found -> second << ".\n"
            << logofs_flush;
    #endif

    #ifdef TEST
    *logofs << name() << ": Checksum for object is ["
            << DumpChecksum(found -> first) << "].\n"
            << logofs_flush;
    #endif

    return found -> second;
  }

  #ifdef TEST
  *logofs << name() << ": End of repository reached "
          << "without any match.\n" << logofs_flush;
  #endif

  return nothing;
}

int MessageStore::clean(T_checksum_action checksumAction)
{
  int position = lastRemoved + 1;

  if (position >= cacheSlots)
  {
    position = 0;
  }

  #ifdef DEBUG
  *logofs << name() << ": Searching a message to remove "
          << "starting at position " << position
          << " with " << checksums_ -> size()
          << " elements in cache.\n"
          << logofs_flush;
  #endif

  while (position != lastRemoved)
  {
    #ifdef DEBUG
    *logofs << name() << ": Examining position "
            << position << ".\n" << logofs_flush;
    #endif

    if ((*messages_)[position] != NULL)
    {
      if (getRate(position, rate_for_clean) == 0)
      {
        break;
      }
      else
      {
        untouch(position);
      }
    }

    if (++position == cacheSlots)
    {
      #ifdef DEBUG
      *logofs << name() << ": Rolled position at "
              << strTimestamp() << ".\n"
              << logofs_flush;
      #endif

      position = 0;
    }
  }

  //
  // If no message is a good candidate,
  // then try the object at the next slot
  // in respect to last element removed.
  //

  if (position == lastRemoved)
  {
    position = lastRemoved + 1;

    if (position >= cacheSlots)
    {
      position = 0;
    }

    if ((*messages_)[position] == NULL ||
            (*messages_)[position] -> locks_ != 0)
    {
      #ifdef DEBUG
      *logofs << name() << ": WARNING! No message found "
            << "to be actually removed.\n"
            << logofs_flush;
      #endif

      return nothing;
    }
    else if ((*messages_)[position] -> keep_ != 0 &&
                  (getRemoteStorageSize() <
                       (control -> RemoteTotalStorageSizeLimit / 100 * cacheThreshold)) &&
                           (getLocalStorageSize() <
                                (control -> LocalTotalStorageSizeLimit / 100 * cacheThreshold)))
    {
      #ifdef WARNING
      *logofs << name() << ": WARNING! Skipping removal "
              << "of startup message.\n"
              << logofs_flush;
      #endif

      return nothing;
    }

    #ifdef DEBUG
    *logofs << name() << ": WARNING! Assuming object "
            << "at position " << position << ".\n"
            << logofs_flush;
    #endif
  }

  #ifdef TEST

  if ((*messages_)[position] -> keep_ != 0)
  {
    #ifdef WARNING
    *logofs << name() << ": WARNING! Cleanup of position "
            << position << " is going to remove a startup "
            << "message.\n" << logofs_flush;
    #endif
  }

  #endif

  #ifdef DEBUG
  *logofs << name() << ": Going to clean up message "
          << "at position " << position << ".\n" 
          << logofs_flush;
  #endif

  return remove(position, checksumAction);
}

int MessageStore::keep(const int position)
{
  Message *message = (*messages_)[position];

  if (message == NULL)
  {
    return 0;
  }

  //
  // Never retain more than 20% of total
  // available messages in repository.
  //

  if (kept_ < cacheSlots / 5)
  {
    #ifdef TEST
    *logofs << name() << ": Set keep flag of object at "
            << strTimestamp(timestamp_) << " with "
            << kept_ << " objects marked.\n"
            << logofs_flush;
    #endif

    kept_++;

    return (message -> keep_ = 1);
  }
  else
  {
    #ifdef TEST
    *logofs << name() << ": WARNING! Can't keep object with "
            << kept_ << " messages already marked.\n"
            << logofs_flush;
    #endif

    return 0;
  }
}

//
// This is the insertion method used at local side 
// side. Cache at remote side side will be kept in
// sync by telling the to other party where to 
// store the message. 
//

int MessageStore::findOrAdd(Message *message, T_checksum_action checksumAction,
                                int &added, int &locked)
{
  if (checksumAction != use_checksum)
  {
    #ifdef PANIC
    *logofs << name() << ": PANIC! Internal error in context [A]. "
            << "Cannot find or add message to repository "
            << "without using checksum.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Internal error in context [A]. "
         << "Cannot find or add message to repository "
         << "without using checksum.\n";

    return nothing;
  }

  //
  // Set added to true only if message
  // is inserted in cache.
  //

  added  = false;
  locked = false;

  //
  // First of all figure out where to
  // store this object.
  //

  #ifdef DEBUG
  *logofs << name() << ": Searching an empty slot "
          << "with last rated " << lastRated << " and "
          << "last added " << lastAdded << ".\n"
          << logofs_flush;
  #endif

  int position = lastRated;

  if (position == nothing)
  {
    position = lastAdded + 1;

    if (position >= cacheSlots)
    {
      position = 0;
    }

    #ifdef DEBUG
    *logofs << name() << ": Searching an empty slot "
            << "starting at position " << position
            << " with " << checksums_ -> size()
            << " elements in cache.\n"
            << logofs_flush;
    #endif

    while (position != lastAdded)
    {
      #ifdef DEBUG
      *logofs << name() << ": Examining position "
              << position << ".\n" << logofs_flush;
      #endif

      if ((*messages_)[position] == NULL)
      {
        break;
      }
      else if (getRate(position, rate_for_insert) == 0)
      {
        break;
      }
      else
      {
        untouch(position);
      }

      if (++position == cacheSlots)
      {
        #ifdef DEBUG
        *logofs << name() << ": Rolled position at "
                << strTimestamp() << ".\n"
                << logofs_flush;
        #endif

        position = 0;
      }
    }
  }
  #ifdef DEBUG
  else
  {
    *logofs << name() << ": Using last rated position "
            << position << ".\n" << logofs_flush;
  }
  #endif

  //
  // If we made an extensive check but did not
  // find neither a free slot or a message to
  // replace, assume slot at next position in
  // respect to last added. This can happen if
  // all objects in repository have got an hit
  // recently.
  //

  if (position == lastAdded)
  {
    position = lastAdded + 1;

    if (position >= cacheSlots)
    {
      position = 0;
    }

    #ifdef DEBUG
    *logofs << name() << ": WARNING! Assuming slot "
            << "at position " << position << ".\n"
            << logofs_flush;
    #endif
  }
  #ifdef DEBUG
  else
  {
    *logofs << name() << ": Found candidate slot "
            << "at position " << position << ".\n"
            << logofs_flush;
  }
  #endif

  //
  // Save the search result so if message
  // is found in cache, we use the slot
  // at next run.
  //

  lastRated = position;

  if ((*messages_)[position] != NULL)
  {
    if ((*messages_)[position] -> locks_ != 0)
    {
      #ifdef WARNING
      *logofs << name() << ": WARNING! Insertion at position "
              << position << " would replace a locked message. "
              << "Forcing channel to discard the message.\n"
              << logofs_flush;
      #endif

      #ifdef TEST
      *logofs << name() << ": Invalidating rating of object "
              << "at position " << position << ".\n"
              << logofs_flush;
      #endif

      return (lastRated = nothing);
    }
    else if ((*messages_)[position] -> keep_ != 0 &&
                  (getRemoteStorageSize() <
                       (control -> RemoteTotalStorageSizeLimit / 100 * cacheThreshold)) &&
                           (getLocalStorageSize() <
                                (control -> LocalTotalStorageSizeLimit / 100 * cacheThreshold)))
    {
      //
      // Skip removal of this message. This
      // will implicitly reserve more space
      // for startup images.
      //

      #ifdef WARNING
      *logofs << name() << ": WARNING! Insertion at position "
              << position << " would replace a startup message. "
              << "Forcing channel to discard the message.\n"
              << logofs_flush;
      #endif

      #ifdef TEST
      *logofs << name() << ": Invalidating rating of object "
              << "at position " << position << ".\n"
              << logofs_flush;
      #endif

      return (lastRated = nothing);
    }
  }

  #if defined(TEST) || defined(INFO)

  if ((*messages_)[position] != NULL &&
          (*messages_)[position] -> keep_ != 0)
  {
    #ifdef WARNING
    *logofs << name() << ": WARNING! Insertion at position "
            << position << " is going to replace a startup "
            << "message.\n" << logofs_flush;
    #endif
  }

  if ((*messages_)[position] != NULL &&
          (opcode() == X_PutImage ||
              opcode() == X_NXPutPackedImage))
  {
    #ifdef WARNING
    *logofs << name() << ": WARNING! Discarding image object "
            << "of type " << name() << "at position "
            << position << ".\n" << logofs_flush;
    #endif
  }

  #endif

  if (checksumAction == use_checksum)
  {
    md5_byte_t *checksum = getChecksum(message);

    #ifdef TEST
    *logofs << name() << ": Searching checksum ["
            << DumpChecksum(checksum) << "] in repository.\n"
            << logofs_flush;

    #endif

    pair<T_checksums::iterator, bool> result;

    try
    {
      result = checksums_ -> insert(T_checksums::value_type(checksum, position));
    }
    catch (exception e)
    {
      #ifdef PANIC
      *logofs << name() << ": PANIC! Exception caught. Error is '"
              << e.what() << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Exception caught. Error is '"
           << e.what() << "'.\n";

      #ifdef DEBUG
      *logofs << name() << ": Cleaning up the object from container.\n";
      #endif

      return nothing;
    }

    //
    // Message was found in cache or
    // insertion couldn't take place.
    //

    if (result.second == false)
    {
      if (result.first == checksums_ -> end())
      {
        #ifdef PANIC
        *logofs << name() << ": PANIC! Failed to insert object "
                << "in the cache.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Failed to insert object of type "
             << name() << " in the cache.\n";

        return nothing;
      }

      //
      // Message is in cache.
      //

      #ifdef TEST
      *logofs << name() << ": Object is already in cache "
              << "at position " << (result.first) -> second
              << ".\n" << logofs_flush;
      #endif

      #ifdef DEBUG

      printStorageSize();

      #endif

      //
      // Message is locked, probably because
      // it has not completely recomposed at
      // remote side after a split.
      //

      if ((*messages_)[(result.first) -> second] -> locks_ != 0)
      {
        #ifdef TEST
        *logofs << name() << ": WARNING! Object at position "
                << (result.first) -> second << " is locked.\n"
                << logofs_flush;
        #endif

        locked = true;
      }

      //
      // Object got a hit, so prevent
      // its removal.
      //

      if (lastRated == (result.first) -> second)
      {
        #ifdef TEST
        *logofs << name() << ": Resetting rating of object "
                << "at position " << (result.first) -> second
                << ".\n" << logofs_flush;
        #endif

        lastRated = nothing;
      }

      return (result.first) -> second;
    }

    #ifdef DEBUG
    *logofs << name() << ": Could not find message in cache.\n"
            << logofs_flush;
    #endif
  }

  //
  // Message not found in hash table (or insertion
  // of checksum in hash table was not requested).
  // Message was added to cache.
  //

  added = true;

  if ((*messages_)[position] != NULL)
  {
    #ifdef DEBUG
    *logofs << name() << ": The message replaces "
            << "the old one at position " << position
            << ".\n" << logofs_flush;
    #endif

    remove(position, checksumAction);
  }

  (*messages_)[position] = message;

  //
  // We used the slot. Perform a new
  // search at next run.
  //

  lastRated = nothing;

  #ifdef TEST
  *logofs << name() << ": Stored message object of size "
          << plainSize(position) << " (" << message -> size_
          << "/" << message -> c_size_ << ") at position "
          << position << ".\n" << logofs_flush;
  #endif

  unsigned int localSize;
  unsigned int remoteSize;

  storageSize(message, localSize, remoteSize);

  localStorageSize_  += localSize;
  remoteStorageSize_ += remoteSize;

  totalLocalStorageSize_  += localSize;
  totalRemoteStorageSize_ += remoteSize;

  #ifdef DEBUG

  printStorageSize();

  #endif

  //
  // Set hits and timestamp at insertion in cache.
  //

  message -> hits_ = control -> StoreHitsAddBonus;
  message -> last_ = timestamp_.tv_sec;

  message -> locks_ = 0;
  message -> keep_  = 0;

  #ifdef DEBUG
  *logofs << name() << ": Set last hit of object at " 
          << strTimestamp(timestamp_) << " with a bonus of "
          << message -> hits_ << ".\n" << logofs_flush;
  #endif

  return position;
}

//
// Add a parsed message to repository. It is normally used
// at decoding side or at encoding side when we load store
// from disk. To handle messages coming from network, the
// encoding side uses the optimized method findOrAdd().
//

int MessageStore::add(Message *message, const int position,
                          T_checksum_action checksumAction)
{
  if (position < 0 || position >= cacheSlots)
  {
    #ifdef PANIC
    *logofs << name() << ": PANIC! Cannot add a message "
            << "at non existing position " << position
            << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot add a message "
         << "at non existing position " << position
         << ".\n";

    HandleAbort();
  }

  if ((*messages_)[position] != NULL)
  {
    #ifdef DEBUG
    *logofs << name() << ": The message will replace "
            << "the old one at position " << position
            << ".\n" << logofs_flush;
    #endif

    remove(position, checksumAction);
  }

  #ifdef DEBUG
  *logofs << name() << ": Inserting object in repository at position "
          << position << ".\n" << logofs_flush;
  #endif

  (*messages_)[position] = message;

  //
  // Get the object's checksum value
  // and insert it in the table.
  //

  if (checksumAction == use_checksum)
  {
    #ifdef DEBUG
    *logofs << name() << ": Inserting object's checksum in repository.\n";
    #endif

    md5_byte_t *checksum = getChecksum(message);

    try
    {
      checksums_ -> insert(T_checksums::value_type(checksum, position));
    }
    catch (exception e)
    {
      #ifdef PANIC
      *logofs << name() << ": PANIC! Exception caught. Error is '" 
              << e.what() << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Exception caught. Error is '"
           << e.what() << "'.\n";

      #ifdef DEBUG
      *logofs << name() << ": Cleaning up the object from container.\n";
      #endif

      (*messages_)[position] = NULL;

      return nothing;
    }
  }

  #ifdef DEBUG
  *logofs << name() << ": Stored message object of size "
          << plainSize(position) << " (" << message -> size_ 
          << "/" << message -> c_size_ << ") at position "
          << position << ".\n" << logofs_flush;
  #endif

  unsigned int localSize;
  unsigned int remoteSize;

  storageSize(message, localSize, remoteSize);

  localStorageSize_  += localSize;
  remoteStorageSize_ += remoteSize;

  totalLocalStorageSize_  += localSize;
  totalRemoteStorageSize_ += remoteSize;

  #ifdef DEBUG

  printStorageSize();

  #endif

  //
  // Set hits and timestamp at insertion in cache.
  //

  message -> hits_ = control -> StoreHitsAddBonus;
  message -> last_ = timestamp_.tv_sec;

  message -> locks_ = 0;
  message -> keep_  = 0;

  #ifdef DEBUG
  *logofs << name() << ": Set last hit of object at " 
          << strTimestamp(timestamp_) << " with a bonus of "
          << message -> hits_ << ".\n" << logofs_flush;
  #endif

  return position;
}

//
// This doesn't modify data, so is supposed
// to be called at encoding side. 
//

void MessageStore::updateData(const int position, unsigned int dataSize,
                                  unsigned int compressedDataSize)
{
  Message *message = (*messages_)[position];

  if (compressedDataSize != 0)
  {
    unsigned int localSize;
    unsigned int remoteSize;

    storageSize(message, localSize, remoteSize);

    localStorageSize_  -= localSize;
    remoteStorageSize_ -= remoteSize;

    totalLocalStorageSize_  -= localSize;
    totalRemoteStorageSize_ -= remoteSize;

    message -> c_size_ = compressedDataSize + message -> i_size_;

    #ifdef TEST

    if (message -> size_ != (int) (dataSize + message -> i_size_))
    {
      #ifdef PANIC
      *logofs << name() << ": PANIC! Size of object looks "
              << message -> size_ << " bytes while it "
              << "should be " << dataSize + message -> i_size_
              << ".\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Size of object looks "
           << message -> size_ << " bytes while it "
           << "should be " << dataSize + message -> i_size_
           << ".\n";
    }

    #endif

    storageSize(message, localSize, remoteSize);

    localStorageSize_  += localSize;
    remoteStorageSize_ += remoteSize;

    totalLocalStorageSize_  += localSize;
    totalRemoteStorageSize_ += remoteSize;

    #ifdef DEBUG

    printStorageSize();

    #endif
  }
}

//
// This replace the data and update information
// about size. It is supposed to be called at
// decoding side, after data is compressed or
// a split message is recomposed.
//

void MessageStore::updateData(const int position, const unsigned char *newData,
                                  unsigned int dataSize, unsigned int compressedDataSize)
{
  Message *message = (*messages_)[position];

  #ifdef TEST

  if (message -> size_ != (int) (dataSize + message -> i_size_))
  {
    #ifdef PANIC
    *logofs << name() << ": PANIC! Data of object looks "
            << dataSize << " bytes while it " << "should be "
            << message -> size_ - message -> i_size_
            << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Data of object looks "
         << dataSize << " bytes while it " << "should be "
         << message -> size_ - message -> i_size_
         << ".\n";
  }

  #endif

  //
  // A compressed data size of 0 means that
  // message's data was not compressed.
  //

  if (compressedDataSize != 0)
  {
    unsigned int localSize;
    unsigned int remoteSize;

    storageSize(message, localSize, remoteSize);

    localStorageSize_  -= localSize;
    remoteStorageSize_ -= remoteSize;

    totalLocalStorageSize_  -= localSize;
    totalRemoteStorageSize_ -= remoteSize;

    #ifdef TEST
    *logofs << name() << ": Data of message at position "
            << position << " will be shrunk from " << dataSize
            << " to " << compressedDataSize << " bytes.\n"
            << logofs_flush;
    #endif

    try
    {
      message -> data_.clear();

      message -> data_.resize(compressedDataSize);
    }
    catch (exception e)
    {
      #ifdef PANIC
      *logofs << name() << ": PANIC! Buffer shrink failed for "
              << "message at position " << position << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Buffer shrink failed for "
           << "message at position " << position << ".\n";

      return;
    }

    memcpy(message -> data_.begin(), newData, compressedDataSize);

    #ifdef TEST
    *logofs << name() << ": Data of message at position "
            << position << " shrunk to size " << message -> data_.size()
            << " and capacity " << message -> data_.capacity() << ".\n"
            << logofs_flush;
    #endif

    message -> c_size_ = compressedDataSize + message -> i_size_;

    storageSize(message, localSize, remoteSize);

    localStorageSize_  += localSize;
    remoteStorageSize_ += remoteSize;

    totalLocalStorageSize_  += localSize;
    totalRemoteStorageSize_ += remoteSize;

    #ifdef DEBUG

    printStorageSize();

    #endif
  }
  else
  {
    #ifdef TEST
    *logofs << name() << ": No changes to data size for message "
            << "at position " << position << ".\n" << logofs_flush;
    #endif

    memcpy(message -> data_.begin(), newData, dataSize);
  }
}

int MessageStore::remove(const int position, T_checksum_action checksumAction)
{
  Message *message;

  if (position < 0 || position >= cacheSlots ||
          (message = (*messages_)[position]) == NULL)
  {
    #ifdef PANIC
    *logofs << name() << ": PANIC! Cannot remove "
            << "a non existing message at position "
            << position << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot remove "
         << "a non existing message at position "
         << position << ".\n";

    return nothing;
  }

  //
  // Checksum is only stored at encoding side.
  //

  if (checksumAction == use_checksum)
  {
    #ifdef DEBUG
    *logofs << name() << ": Removing checksum for object at "
            << "position " << position << ".\n" << logofs_flush;
    #endif

    T_checksums::iterator found;

    int foundPosition = find(message, found);

    if (foundPosition == nothing)
    {
      #ifdef PANIC
      *logofs << name() << ": PANIC! No checksum found for "
              << "object at position " << position << ".\n" 
              << logofs_flush;
      #endif

      cerr << "Error" << ": No checksum found for "
           << "object at position " << position << ".\n";

      return nothing;
    }

    #ifdef TEST

    else if (position != foundPosition)
    {
      #ifdef PANIC
      *logofs << name() << ": PANIC! Value of position for object "
              << "doesn't match position " << position << ".\n" 
              << logofs_flush;
      #endif

      cerr << "Error" << ": Value of position for object "
           << "doesn't match position " << position << ".\n";

      return nothing;
    }

    #endif

    checksums_ -> erase(found);
  }

  #ifdef DEBUG
  *logofs << name() << ": Removing message at position "
          << position << " of size " << plainSize(position)
          << " (" << message -> size_ << "/" << message -> c_size_
          << ").\n" << logofs_flush;
  #endif

  unsigned int localSize;
  unsigned int remoteSize;

  storageSize(message, localSize, remoteSize);

  #ifdef TEST

  if (opcode() == X_PutImage || opcode() == X_NXPutPackedImage)
  {
    *logofs << name() << ": WARNING! Removing message at position "
            << position << " of size " << plainSize(position)
            << " (" << message -> size_ << "/" << message -> c_size_
            << ").\n" << logofs_flush;

    printStorageSize();
  }

  #endif

  localStorageSize_  -= localSize;
  remoteStorageSize_ -= remoteSize;

  totalLocalStorageSize_  -= localSize;
  totalRemoteStorageSize_ -= remoteSize;

  recycle(message);

  (*messages_)[position] = NULL;

  #ifdef DEBUG

  printStorageSize();

  #endif

  return position;
}

//
// This should only be called at encoding side.
// Decoding side can't rely on hits counter as
// it is decremented at the time repository is
// searched for a message to remove.
//

int MessageStore::getRate(const int position, T_rate type) const
{
  Message *message = (*messages_)[position];

  if (message == NULL)
  {
    #ifdef PANIC
    *logofs << name() << ": PANIC! Cannot get rate of message "
            << "at non existing position " << position
            << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot get rate of message "
         << "at non existing position " << position
         << ".\n";

    return -1;
  }

  if (message -> locks_ != 0 || message -> keep_ != 0)
  {
    #ifdef TEST
    if (message -> locks_ != 0)
    {
      *logofs << name() << ": Rate set to -1 as locks of object are "
              << (int) message -> locks_ << ".\n"
              << logofs_flush;
    }
    else if (message -> keep_ != 0)
    {
      *logofs << name() << ": Rate set to -1 as keep flag is "
              << (int) message -> keep_ << ".\n"
              << logofs_flush;
    }
    #endif

    return -1;
  }
  else if ((type == rate_for_clean ||
                (int) checksums_ -> size() == cacheSlots) &&
                    message -> hits_ <= control -> StoreHitsLoadBonus)
  {
    //
    // We don't have any free slot or we exceeded the
    // available storage size. This is likely to happen
    // after having loaded objects from persistent cache.
    // It's not a bad idea to discard some messages that
    // were restored but never referenced.
    //

    #ifdef TEST

    if (type == rate_for_clean)
    {
      *logofs << name() << ": Rate set to 0 with hits "
              << message -> hits_ << " as maximum storage size "
              << "was exceeded.\n" << logofs_flush;
    }
    else
    {
      *logofs << name() << ": Rate set to 0 with hits "
              << message -> hits_ << " as there are no available "
              << "slots in store.\n" << logofs_flush;
    }

    #endif

    return 0;
  }
  else if (type == rate_for_clean &&
               timestamp_.tv_sec - message -> last_ >=
                   control -> StoreTimeLimit)
  {
    #ifdef TEST
    *logofs << name() << ": Rate set to 0 as last hit of object was "
            << timestamp_.tv_sec - message -> last_ << " seconds ago "
            << "with limit set to " << control -> StoreTimeLimit
            << ".\n" << logofs_flush;
    #endif

    return 0;
  }
  else
  {
    #ifdef TEST
    if (message -> hits_ < 0)
    {
      *logofs << name() << ": PANIC! Rate of object shouldn't be "
              << message -> hits_ << ".\n" << logofs_flush;

      cerr << "Error" << ": Rate of object of type " << name()
           << " shouldn't be " << message -> hits_ << ".\n";

      HandleCleanup();
    }
    #endif

    #ifdef TEST
    *logofs << name() << ": Rate of object is " << message -> hits_
            << " with last hit " << timestamp_.tv_sec - message -> last_
            << " seconds ago.\n" << logofs_flush;
    #endif

    return message -> hits_;
  }
}

int MessageStore::lock(const int position) const
{
  Message *message = (*messages_)[position];

  if (message == NULL)
  {
    #ifdef PANIC
    *logofs << name() << ": PANIC! Can't lock the null "
            << "object at position " << position
            << ".\n" << logofs_flush;
    #endif

    return -1;
  }

  #ifdef DEBUG
  *logofs << name() << ": Increasing locks of object to " 
          << (int) message -> locks_ + 1 << ".\n"
          << logofs_flush;
  #endif

  return ++(message -> locks_);
}

int MessageStore::unlock(const int position) const
{
  Message *message = (*messages_)[position];

  if (message == NULL)
  {
    #ifdef PANIC
    *logofs << name() << ": PANIC! Can't unlock the null "
            << "object at position " << position
            << ".\n" << logofs_flush;
    #endif

    return -1;
  }

  #ifdef DEBUG
  *logofs << name() << ": Decreasing locks of object to " 
          << (int) message -> locks_ - 1 << ".\n"
          << logofs_flush;
  #endif

  return --(message -> locks_);
}

int MessageStore::saveStore(ostream *cachefs, md5_state_t *md5StateStream,
                                md5_state_t *md5StateClient, T_checksum_action checksumAction,
                                    T_data_action dataAction, int bigEndian)
{
  Message *message;

  #ifdef TEST
  *logofs << name() << ": Opcode of this store is "
          << (unsigned int) opcode() << " default size of "
          << "identity is " << dataOffset << ".\n"
          << logofs_flush;
  #endif

  //
  // Protocol versions older than 3 don't have
  // a specific store for render extension.
  //

  if ((control -> isProtoStep3() == 0 &&
           opcode() == X_NXInternalRenderExtension) ||
               (control -> isProtoStep4() == 0 &&
                    (opcode() == X_NXSetUnpackGeometry ||
                         opcode() == X_NXSetUnpackAlpha)) ||
                             (control -> isProtoStep5() == 0 &&
                                  opcode() == X_CreatePixmap))
  {
    #ifdef TEST
    *logofs << name() << ": Skipping save of store for opcode "
            << (unsigned int) opcode() << ".\n"
            << logofs_flush;
    #endif

    return 1;
  }

  unsigned char *identityBuffer = new unsigned char[dataOffset];
  unsigned char *sizeBuffer     = new unsigned char[4 * 2];
  unsigned char *positionBuffer = new unsigned char[4];
  unsigned char *opcodeBuffer   = new unsigned char[4];

  #ifdef DUMP

  char *md5ClientDump = new char[dataOffset * 2 + 128];

  #endif

  unsigned char value;

  int offset;

  int failed = 0;

  for (int position = 0; position < cacheSlots; position++)
  {
    message = (*messages_)[position];

    //
    // Don't save split messages.
    //

    if (message != NULL && message -> locks_ == 0)
    {
      //
      // Use the total size if offset is
      // beyond the real end of message.
      //

      offset = dataOffset;

      if (offset > message -> size_)
      {
        offset = message -> size_;
      }
        
      #ifdef TEST
      *logofs << name() << ": Going to save message at position "
              << position << ".\n" << logofs_flush;
      #endif

      value = 1;

      PutULONG(position, positionBuffer, bigEndian);
      PutULONG(opcode(), opcodeBuffer, bigEndian);

      md5_append(md5StateClient, positionBuffer, 4);
      md5_append(md5StateClient, opcodeBuffer, 4);

      #ifdef DUMP

      *logofs << "Name=" << name() << logofs_flush;

      sprintf(md5ClientDump," Pos=%d Op=%d\n", position, opcode());

      *logofs << md5ClientDump << logofs_flush;

      #endif

      if (PutData(cachefs, &value, 1) < 0)
      {
        #ifdef DEBUG
        *logofs << name() << ": PANIC! Failure writing " << 1
                << " bytes.\n" << logofs_flush;
        #endif

        failed = 1;

        break;
      }

      md5_append(md5StateStream, &value, 1);

      PutULONG(message -> size_, sizeBuffer, bigEndian);
      PutULONG(message -> c_size_, sizeBuffer + 4, bigEndian);

      //
      // Note that the identity size is not saved with
      // the message and will be determined from the
      // data read when restoring the identity.
      //

      if (PutData(cachefs, sizeBuffer, 4 * 2) < 0)
      {
        #ifdef DEBUG
        *logofs << name() << ": PANIC! Failure writing " << 4 * 2
                << " bytes.\n" << logofs_flush;
        #endif

        failed = 1;

        break;
      }

      md5_append(md5StateStream, sizeBuffer, 4 * 2);
      md5_append(md5StateClient, sizeBuffer, 4 * 2);

      #ifdef DUMP

      sprintf(md5ClientDump, "size = %d c_size = %d\n",
                  message -> size_, message -> c_size_);

      *logofs << md5ClientDump << logofs_flush;

      #endif

      //
      // Prepare a clean buffer for unparse.
      //

      memset(identityBuffer, 0, offset);

      unparseIdentity(message, identityBuffer, offset, bigEndian);

      if (PutData(cachefs, identityBuffer, offset) < 0)
      {
        #ifdef DEBUG
        *logofs << name() << ": PANIC! Failure writing " << offset
                << " bytes.\n" << logofs_flush;
        #endif

        failed = 1;

        break;
      }

      md5_append(md5StateStream, identityBuffer, offset);
      md5_append(md5StateClient, identityBuffer, offset);

      #ifdef DUMP

      for (int i = 0; i < offset; i++)
      {
        sprintf(md5ClientDump + (i * 2), "%02X", identityBuffer[i]);
      }

      *logofs << "Identity = " << md5ClientDump << "\n" << logofs_flush;

      #endif

      //
      // Set the real identity size before
      // saving the data.
      //

      offset = message -> i_size_;

      if (offset > message -> size_)
      {
        offset = message -> size_;
      }

      if (checksumAction == use_checksum)
      {
        if (PutData(cachefs, message -> md5_digest_, MD5_LENGTH) < 0)
        {
          #ifdef DEBUG
          *logofs << name() << ": PANIC! Failure writing " << MD5_LENGTH
                  << " bytes.\n" << logofs_flush;
          #endif

          failed = 1;

          break;
        }

        md5_append(md5StateStream, message -> md5_digest_, MD5_LENGTH);
      }
      else if (dataAction == use_data)
      {
        int dataSize = (message -> c_size_ == 0 ?
                             message -> size_ - offset :
                                 message -> c_size_ - offset);
        if (dataSize > 0)
        {
          if (PutData(cachefs, message -> data_.begin(), dataSize) < 0)
          {
            #ifdef DEBUG
            *logofs << name() << ": PANIC! Failure writing " << dataSize
                    << " bytes.\n" << logofs_flush;
            #endif

            failed = 1;

            break;
          }

          md5_append(md5StateStream, message -> data_.begin(), dataSize);
        }
      }
    }
    else
    {
      #ifdef TEST
      *logofs << name() << ": Not saving message at position "
              << position << ".\n" << logofs_flush;
      #endif

      value = 0;

      if (PutData(cachefs, &value, 1) < 0)
      {
        #ifdef DEBUG
        *logofs << name() << ": PANIC! Failure writing " << 1
                << " bytes.\n" << logofs_flush;
        #endif

        failed = 1;

        break;
      }

      md5_append(md5StateStream, &value, 1);
    }
  }

  if (failed == 1)
  {
    #ifdef PANIC
    *logofs << name() << ": PANIC! Write to persistent cache file failed.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Write to persistent cache file failed.\n";
  }

  delete [] identityBuffer;
  delete [] sizeBuffer;
  delete [] positionBuffer;
  delete [] opcodeBuffer;

  #ifdef DUMP

  delete [] md5ClientDump;

  #endif

  return (failed == 0 ? 1 : -1);
}

int MessageStore::loadStore(istream *cachefs, md5_state_t *md5StateStream,
                                T_checksum_action checksumAction, T_data_action dataAction,
                                    int bigEndian)
{
  Message *message;

  #ifdef TEST
  *logofs << name() << ": Opcode of this store is "
          << (unsigned int) opcode() << " default size of "
          << "identity is " << dataOffset << " slots are "
          << cacheSlots << ".\n" << logofs_flush;
  #endif

  //
  // Protocol versions older than 3 don't have a
  // specific store for render extension, so skip
  // this to avoid unpredictable results. Loading
  // recent caches with an old proxy version will
  // gracefully fail when checksum is compared.
  //

  if ((control -> isProtoStep3() == 0 &&
           opcode() == X_NXInternalRenderExtension) ||
               (control -> isProtoStep4() == 0 &&
                    (opcode() == X_NXSetUnpackGeometry ||
                         opcode() == X_NXSetUnpackAlpha)) ||
                             ((control -> isProtoStep5() == 0 ||
                                  control -> isCacheStep2() == 0) &&
                                      opcode() == X_CreatePixmap))
  {
    #ifdef TEST
    *logofs << name() << ": Skipping load of store for opcode "
            << (unsigned int) opcode() << ".\n"
            << logofs_flush;
    #endif

    return 1;
  }

  //
  // If packed images or the render extension has been
  // disabled we don't need to restore these messages
  // in the cache. Encoding of RENDER in 1.4.0 is also
  // changed so we want to skip messages saved using
  // the old format. We want to restore all the other
  // messages so we'll need to skip these one by one.
  //

  int skip = 0;

  if ((opcode() == X_NXPutPackedImage &&
          control -> PersistentCacheLoadPacked == 0) ||
              (opcode() == X_NXInternalRenderExtension &&
                  control -> PersistentCacheLoadRender == 0) ||
                      (opcode() == X_NXInternalRenderExtension &&
                          control -> isProtoStep5() == 1 &&
                              control -> isCacheStep2() == 0))
  {
    #ifdef TEST
    *logofs << name() << ": All messages for OPCODE#"
            << (unsigned int) opcode() << " will be discarded.\n"
            << logofs_flush;
    #endif

    skip = 1;
  }

  unsigned char *identityBuffer = new unsigned char[dataOffset];
  unsigned char *sizeBuffer     = new unsigned char[4 * 2];

  unsigned char value;

  int offset;

  int failed = 0;

  //
  // Get a new timestamp for all messages
  // to be loaded.
  //

  newTimestamp();

  for (int position = 0; position < cacheSlots; position++)
  {
    if (GetData(cachefs, &value, 1) < 0)
    {
      #ifdef DEBUG
      *logofs << name() << ": PANIC! Failure reading " << 1
              << " bytes.\n" << logofs_flush;
      #endif

      failed = 1;

      break;
    }

    md5_append(md5StateStream, &value, 1);

    if (value == 1)
    {
      #ifdef TEST
      *logofs << name() << ": Going to load message at position "
              << position << ".\n" << logofs_flush;
      #endif

      if (GetData(cachefs, sizeBuffer, 4 * 2) < 0)
      {
        #ifdef DEBUG
        *logofs << name() << ": PANIC! Failure reading " << 4 * 2
                << " bytes.\n" << logofs_flush;
        #endif

        failed = 1;

        break;
      }

      md5_append(md5StateStream, sizeBuffer, 4 * 2);

      message = getTemporary();

      if (message == NULL)
      {
        #ifdef PANIC
        *logofs << name() << ": PANIC! Can't access temporary storage "
                << "for message in context [B].\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't access temporary storage "
             << "for message in context [B].\n";

        failed = 1;

        break;
      }

      message -> size_   = GetULONG(sizeBuffer, bigEndian);
      message -> c_size_ = GetULONG(sizeBuffer + 4, bigEndian);

      #ifdef DEBUG
      *logofs << name() << ": Size is " << message -> size_
              << " compressed size is " << message -> c_size_
              << ".\n" << logofs_flush;
      #endif

      //
      // Use the total size if offset is
      // beyond the real end of message.
      //

      offset = dataOffset;

      if (offset > message -> size_)
      {
        offset = message -> size_;
      }

      if (GetData(cachefs, identityBuffer, offset) < 0)
      {
        #ifdef DEBUG
        *logofs << name() << ": PANIC! Failure reading " << offset
                << " bytes.\n" << logofs_flush;
        #endif

        failed = 1;

        break;
      }

      md5_append(md5StateStream, identityBuffer, offset);

      //
      // We need to use the old offset if cache was
      // saved by the 1.3.X. We'll later discard the
      // message.
      //

      if (opcode() == X_NXInternalRenderExtension &&
              control -> isProtoStep5() == 1 &&
                  control -> isCacheStep2() == 1)
      {
        //
        // Get the real identity size based on the
        // identity data read from the buffer.
        //

        offset = identitySize(identityBuffer, offset);

        if (offset > message -> size_)
        {
          offset = message -> size_;
        }
      }

      message -> i_size_ = offset;
        
      //
      // Get identity of message from the buffer we just
      // created. Don't calculate neither checksum nor
      // data, restore them from stream. Don't pass the
      // message's size but the default size of identity.
      //

      parseIdentity(message, identityBuffer, offset, bigEndian);

      if (checksumAction == use_checksum)
      {
        if (message -> md5_digest_ == NULL)
        {
          message -> md5_digest_ = new md5_byte_t[MD5_LENGTH];
        }

        if (GetData(cachefs, message -> md5_digest_, MD5_LENGTH) < 0)
        {
          #ifdef DEBUG
          *logofs << name() << ": PANIC! Failure reading " << MD5_LENGTH
                  << " bytes.\n" << logofs_flush;
          #endif

          failed = 1;

          break;
        }

        //
        // Add message's checksum to checksum that will
        // be saved together with this cache. Checksum
        // will be verified when cache file is restored
        // to ensure file is not corrupted.
        //

        md5_append(md5StateStream, message -> md5_digest_, MD5_LENGTH);

        if (skip == 1)
        {
          #ifdef TEST
          *logofs << name() << ": Discarding message for OPCODE#"
                  << (unsigned int) opcode() << ".\n"
                  << logofs_flush;
          #endif
          
          continue;
        }
      }
      else if (dataAction == use_data)
      {
        //
        // Restore messages's data.
        //

        int dataSize = (message -> c_size_ == 0 ?
                             message -> size_ - offset :
                                 message -> c_size_ - offset);

        if (dataSize < 0 || dataSize > MESSAGE_DATA_LIMIT)
        {
          #ifdef PANIC
          *logofs << name() << ": PANIC! Bad data size "
                  << dataSize << " loading persistent cache.\n"
                  << logofs_flush;
          #endif

          cerr << "Error" << ": Bad data size " << dataSize
               << " loading persistent cache.\n";

          failed = 1;

          break;
        }
        else if (dataSize > 0)
        {
          //
          // If need to skip the message let anyway
          // it to be part of the calculated MD5.
          //

          if (skip == 1)
          {
            unsigned char *dummy = new unsigned char[dataSize];

            if (dummy == NULL)
            {
              #ifdef PANIC
              *logofs << name() << ": PANIC! Can't allocate dummy buffer "
                      << "of size " << dataSize << " loading cache.\n"
                      << logofs_flush;
              #endif

              cerr << "Error" << ": Can't allocate dummy buffer "
                   << "of size " << dataSize << " loading cache.\n";

              failed = 1;

              break;
            }

            if (GetData(cachefs, dummy, dataSize) < 0)
            {
              #ifdef DEBUG
              *logofs << name() << ": PANIC! Failure reading " << dataSize
                      << " bytes.\n" << logofs_flush;
              #endif

              failed = 1;

              break;
            }

            md5_append(md5StateStream, dummy, dataSize);

            delete [] dummy;

            #ifdef TEST
            *logofs << name() << ": Discarding message for OPCODE#"
                    << (unsigned int) opcode() << ".\n"
                    << logofs_flush;
            #endif

            continue;
          }
          else
          {
            try
            {
              message -> data_.clear();

              message -> data_.resize(dataSize);
            }
            catch (exception e)
            {
              #ifdef PANIC
              *logofs << name() << ": PANIC! Exception caught resizing data."
                      << " Error is '" << e.what() << "'.\n" << logofs_flush;
              #endif

              cerr << "Error" << ": Exception caught resizing data."
                   << " Error is '" << e.what() << "'.\n";

              failed = 1;

              break;
            }

            if (GetData(cachefs, message -> data_.begin(), dataSize) < 0)
            {
              #ifdef DEBUG
              *logofs << name() << ": PANIC! Failure reading " << dataSize
                      << " bytes.\n" << logofs_flush;
              #endif

              failed = 1;

              break;
            }

            //
            // Add message's data to cache checksum.
            //

            md5_append(md5StateStream, message -> data_.begin(), dataSize);
          }
        }
        else
        {
          //
          // We are here if data part is zero.
          //

          if (skip == 1)
          {
            #ifdef TEST
            *logofs << name() << ": Discarding message for OPCODE#"
                    << (unsigned int) opcode() << ".\n"
                    << logofs_flush;
            #endif

            continue;
          }
        }
      }

      int added;

      added = add(message, position, checksumAction);

      if (added != position)
      {
        #ifdef PANIC
        *logofs << name() << ": PANIC! Can't store message "
                << "in the cache at position " << position
                << ".\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Can't store message "
             << "in the cache at position " << position
             << ".\n";

        failed = 1;

        break;
      }
      else
      {
        //
        // Replace default value of hits set by add
        // function. Messages read from cache start
        // with a lower bonus than fresh messages
        // inserted.
        //

        message -> hits_ = control -> StoreHitsLoadBonus;

        #ifdef DEBUG
        *logofs << name() << ": Updated last hit of object at " 
                << strTimestamp(timestamp_) << " with a bonus of "
                << message -> hits_ << ".\n" << logofs_flush;
        #endif

        resetTemporary();
      }
    }
    else if ((*messages_)[position] != NULL)
    {
      #ifdef TEST
      *logofs << name() << ": Going to remove message at position "
              << position << ".\n" << logofs_flush;
      #endif

      int removed;

      removed = remove(position, checksumAction);

      if (removed != position)
      {
        #ifdef PANIC
        *logofs << name() << ": PANIC! Can't remove message from cache "
                << "at position " << position << ".\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Can't remove message from cache "
             << "at position " << position << ".\n";

        failed = 1;

        break;
      }
    }
    #ifdef TEST
    else
    {
      *logofs << name() << ": Not loading message at position "
              << position << ".\n" << logofs_flush;
    }
    #endif
  }

  if (failed == 1)
  {
    #ifdef PANIC
    *logofs << name() << ": PANIC! Read from persistent cache file failed.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Read from persistent cache file failed.\n";
  }

  delete [] identityBuffer;
  delete [] sizeBuffer;

  return (failed == 0 ? 1 : -1);
}

void MessageStore::storageSize(const Message *message, unsigned int &local,
                                   unsigned int &remote) const
{
  local = remote = storage();

  //
  // Encoding side includes 48 bytes for
  // the map of checksums and 24 bytes
  // of adjustment for total overhead.
  //

  local += MD5_LENGTH + 48 + 24;

  //
  // At decoding side we include size of
  // data part and 24 bytes of adjustment
  // for total overhead.
  //

  if (message -> c_size_ == 0)
  {
    remote += message -> size_ + 24;
  }
  else
  {
    remote += message -> c_size_ + 24;
  }

  if (message -> md5_digest_ == NULL)
  {
    unsigned int t = local;
    local = remote;
    remote = t;
  }
}

void MessageStore::printStorageSize()
{
  #ifdef TEST

  *logofs << name() << ": There are "
          << checksums_ -> size() << " checksums in this store "
          << "out of " << cacheSlots << " slots.\n"
          << logofs_flush;

  *logofs << name() << ": Size of this store is "
          << localStorageSize_ << " bytes at local side and "
          << remoteStorageSize_ << " bytes at remote side.\n"
          << logofs_flush;

  *logofs << name() << ": Size of total cache is "
          << totalLocalStorageSize_ << " bytes at local side and "
          << totalRemoteStorageSize_ << " bytes at remote side.\n"
          << logofs_flush;

  #endif
}
