/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com.          */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef NX_H
#define NX_H

#ifdef __cplusplus
extern "C" {
#endif

#include <sys/uio.h>

#define NX_FD_ANY                      -1

#define NX_MODE_ANY                    -1
#define NX_MODE_CLIENT                  1
#define NX_MODE_SERVER                  2

#define NX_DISPLAY_ANY                  NULL

#define NX_SIGNAL_ANY                  -1
#define NX_SIGNAL_ENABLE                1
#define NX_SIGNAL_DISABLE               2
#define NX_SIGNAL_RAISE                 3
#define NX_SIGNAL_FORWARD               4

#define NX_POLICY_IMMEDIATE             1
#define NX_POLICY_DEFERRED              2

#define NX_ALERT_REMOTE                 0
#define NX_ALERT_LOCAL                  1

/*
 * The following are the new interfaces to the NX transport. The
 * NX proxy software is now intended to be run as a library of a
 * higher level communication manager (nxssh, nxhttp, nxrtp, etc,
 * not only nxproxy). This is a work-in-progress, so expect these
 * interfaces to change in future. At the present moment, as an
 * example, there is no provision for creating and managing mul-
 * tiple proxy connections.
 */

/*
 * Attach a NX transport to the provided descriptor. This should be
 * done after creating a pair of connected sockets.
 */

extern int NXTransCreate(int fd, int mode, const char* options);

/*
 * Tell the proxy to use the second descriptor as its own end of
 * the internal connection to the NX agent. The NX agent will use
 * the first descriptor. Setting an agent connection will have the
 * effect of disabling further X client connections and, if it is
 * possible, will trigger the use of the memory-to-memory transport.
 */

extern int NXTransAgent(int fd[2]);

/*
 * Prepare the file sets and the timeout for a later execution of
 * the select(). The masks and the timeout must persist across all
 * the calls, so if you don't need any of the values, it is requi-
 * red that you create empty masks and a default timeout. To save
 * a check at each run, all the functions below assume that valid
 * pointers are passed.
 */

extern int NXTransPrepare(int *maxfds, fd_set *readfds,
                              fd_set *writefds, struct timeval *timeout);

/*
 * Call select() to find out the descriptors in the sets having
 * pending data.
 */

extern int NXTransSelect(int *result, int *error, int *maxfds, fd_set *readfds,
                             fd_set *writefds, struct timeval *timeout);

/*
 * Perform the required I/O on all the NX descriptors having pen-
 * ding data. This can include reading and writing to the NX chan-
 * nels, encoding and decoding the proxy data or managing any of
 * the other NX resources.
 */

extern int NXTransExecute(int *result, int *error, int *maxfds, fd_set *readfds,
                              fd_set *writefds, struct timeval *timeout);

/*
 * Restore the proxy connection in case of failure.
 */

extern int NXTransRestore(void);

/*
 * Close down the transport and free the allocated NX resources.
 */

extern int NXTransDestroy(void);

/*
 * Return true if the NX transport is running.
 */

extern int NXTransRunning(void);

/*
 * Run an empty loop, giving to the NX transport
 * a chance to check its descriptors.
 */

extern int NXTransContinue(struct timeval *timeout);

/*
 * Perform I/O on the given descriptors. If memory-to-memory trans-
 * port has been activated and the descriptor is recognized as a
 * valid agent connection, then the functions will read and write
 * the data directly to the proxy buffer, otherwise the correspond-
 * ing network operation will be performed.
 */

extern int NXTransRead(int fd, char *data, int size);
extern int NXTransWrite(int fd, char *data, int size);
extern int NXTransReadable(int fd, long int *readable);

extern int NXTransReadVector(int fd, struct iovec *iovdata, int iovsize);
extern int NXTransWriteVector(int fd, struct iovec *iovdata, int iovsize);

/*
 * Tell to the proxy how to handle the standard POSIX signals. For
 * example, given the SIGINT signal, the caller can specify any of
 * the following actions:
 *
 * NX_SIGNAL_ENABLE:  A signal handler will have to be installed by
 *                    the library, so that it can be intercepted by
 *                    the proxy.
 *
 * NX_SIGNAL_DISABLE: The signal will be handled by the caller and,
 *                    eventually, forwarded to the proxy by calling
 *                    NXTransSignal() explicitly.
 *
 * NX_SIGNAL_RAISE:   The signal must be handled now, as if it had
 *                    been delivered by the operating system. This
 *                    function can be called by the agent with the
 *                    purpose of propagating a signal to the proxy.
 *
 * NX_SIGNAL_FORWARD: A signal handler will have to be installed by
 *                    the library but the library will have to call
 *                    the original signal handler when the signal
 *                    is received.
 *
 * As a rule of thumb, agents should let the proxy handle SIGUSR1
 * and SIGUSR2, used for producing the NX protocol statistics, and
 * SIGHUP, used for disconnecting the NX transport.
 *
 * The following signals are blocked by default upon creation of the
 * NX transport:
 *
 * SIGCHLD    These signals should be always put under the control
 * SIGUSR1    of the proxy. If agents are intercepting them, agents
 * SIGUSR2    should later call NXTransSignal(..., NX_SIGNAL_RAISE)
 * SIGHUP     to forward the signal to the proxy. As an alternative
 *            they can specify a NX_SIGNAL_FORWARD action, so they,
 *            in turn, can be notified about the signal. This can
 *            be especially useful for SIGCHLD.
 *
 * SIGINT     These signals should be intercepted by agents. Agents
 * SIGTERM    should ensure that NXTransDestroy() is called before
 *            exiting, to give the proxy a chance to shut down the
 *            NX transport.
 *
 * SIGPIPE    This signal is blocked by the proxy, but not used to
 *            implement any functionality. It can be handled by the
 *            NX agent without affecting the proxy.
 *
 * SIGALRM    This is now used by the proxy and agents should not
 *            redefine it. Agents can use the signal to implement
 *            their own timers but should not interleave calls to
 *            the NX transport and should restore the old handler
 *            when the timeout is raised.
 *
 * SIGVTALRM  These signals are not used but may be used in future
 * SIGWINCH   versions of the library.
 * SIGIO
 * SIGTSTP
 * SIGTTIN
 * SIGTTOU
 *
 * By calling NXTransSignal(..., NX_SIGNAL_DISABLE) nxcomp will res-
 * tore the signal handler that was saved at the time the proxy hand-
 * ler was installed. This means that you should call the function
 * just after the XOpenDisplay() or any other function used to init-
 * ialize the NX transport.
 */
 
extern int NXTransSignal(int signal, int action);

/*
 * Return true if the transport is in congestion state.
 */

extern int NXTransCongestion(void);

/*
 * Set the policy to be used by the NX transport to write data to the
 * proxy link:
 *
 * NX_POLICY_IMMEDIATE: When set to immediate, the proxy will try to
 *                      write the data just after having encoded it.
 *
 * NX_POLICY_DEFERRED:  When policy is set to deferred, data will be
 *                      accumulated in a buffer and written to the
 *                      remote proxy when NXTransFlush() is called by
 *                      the agent.
 */

extern int NXTransPolicy(int type);

/*
 * Query the number of bytes that have been accumulated for a deferred
 * flush.
 */

extern int NXTransFlushable(void);

/*
 * Tell to the NX transport to write all the accumulated data to the
 * remote proxy. If the sync flag is set, the proxy will force the
 * remote peer to flush all the data until past the sequence number
 * matching the sequence number of the last request written for the
 * agent. It's advisable to call the function with the sync flag set
 * whenever the flush policy is set to deferred writes and the agent
 * is going to block waiting for a reply.
 */

extern int NXTransFlush(int sync);

/*
 * Other interfaces to the internal transport functions.
 */

extern int NXTransProxy(int fd, int mode, const char* display);

extern int NXTransClient(const char* display);

extern int NXTransDialog(const char *caption, const char *message,
                           const char *window, const char *type, int local,
                               const char* display);

extern int NXTransAlert(int code, int local);

extern int NXTransWatchdog(int timeout);

extern int NXTransKeeper(int caches, int images, const char *root);

extern void NXTransExit(int code);

extern int NXTransParseCommandLine(int argc, const char **argv);
extern int NXTransParseEnvironment(const char *env, int force);

extern void NXTransCleanup(void) __attribute__((noreturn));

#ifdef __cplusplus
}
#endif

#endif /* NX_H */
