/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <fstream.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include "NXalert.h"

#include "Proxy.h"

#include "Socket.h"
#include "Channel.h"
#include "Statistics.h"

//
// We need to adjust some values related
// to these messages at the time cache
// is reconfigured.
//

#include "PutImage.h"
#include "ChangeGC.h"
#include "PolyFillRectangle.h"
#include "PutPackedImage.h"

//
// Set the verbosity level. You also need
// to define DUMP in Misc.cpp if DUMP is
// defined here.
//

#define WARNING
#define PANIC
#undef  OPCODES
#undef  TEST
#undef  DEBUG
#undef  DUMP

//
// Log the important tracepoints related
// to writing packets to the peer proxy.
//

#undef  FLUSH

//
// Log operations related to adjusting
// the timeouts.
//

#undef  ADJUST

//
// Log operations related to handling
// the synchronization requests.
//

#undef  SYNC

//
// Log the operations related to splits.
//

#undef  SPLIT

//
// Needed to test the synchronous flush.
//

#if defined(TEST) || defined(INFO) || defined(SYNC)

#include "ClientChannel.h"

#endif

Proxy::Proxy(int fd)

  : transport_(new ProxyTransport(fd)), fd_(fd), readBuffer_(transport_)
{
  for (int channelId = 0;
           channelId < CONNECTIONS_LIMIT;
               channelId++)
  {
    channels_[channelId]    = NULL;
    transports_[channelId]  = NULL;
    congestions_[channelId] = 0;
  }

  inputChannel_   = -1;
  outputChannel_  = -1;

  lowerChannel_ = 0;
  upperChannel_ = 0;
  firstChannel_ = 0;

  activeChannels_ = 0;

  controlLength_ = 0;

  operation_ = operation_in_negotiation;

  pending_    = 0;
  priority_   = 0;
  shutdown_   = 0;
  reset_      = 0;
  congestion_ = 0;
  timer_      = 0;
  tokens_     = 0;

  //
  // Set null timeouts. It will require
  // a new link configuration.
  //

  timeouts_.frame = 0;
  timeouts_.flush = 0;
  timeouts_.split = 0;

  lastBytesInTs_  = getTimestamp();
  lastBytesOutTs_ = getTimestamp();

  lastLoopTs_  = getTimestamp();
  lastPingTs_  = getTimestamp();

  lastAlertTs_ = nullTimestamp();
  lastSplitTs_ = nullTimestamp();
  lastFlushTs_ = nullTimestamp();
  lastFrameTs_ = nullTimestamp();

  notifiedSync_ = -1;

  notifiedCongestion_ = 0;

  notifiedBytes_ = 0;

  deferredFrames_ = 0;
  deferredBytes_  = 0;

  statisticsStream_ = NULL;

  //
  // Create compressor and decompressor
  // for image and data payload.
  //

  compressor_ = new Compressor(control -> LocalDataCompressionLevel,
                                   control -> LocalDataCompressionThreshold);

  decompressor_ = new Decompressor();

  //
  // Create object storing NX specific
  // opcodes.
  //

  opcodeStore_ = new OpcodeStore();

  //
  // Create the message stores.
  //

  clientStore_ = new ClientStore(compressor_, decompressor_);
  serverStore_ = new ServerStore(compressor_, decompressor_);

  lastLoadTs_ = nullTimestamp();

  if (control -> isProtoStep3() == 1)
  {
    clientCache_ = new ClientCache();
    serverCache_ = new ServerCache();

    if (clientCache_ == NULL || serverCache_ == NULL)
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! Failed to create encode caches.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Failed to create encode caches.\n";

      HandleCleanup();
    }
  }
  else
  {
    clientCache_ = NULL;
    serverCache_ = NULL;
  }

  #ifdef DEBUG
  *logofs << "Proxy: Created new object at " << this
          << ".\n" << logofs_flush;
  #endif
}

Proxy::~Proxy()
{
  resetTimer();

  for (int channelId = 0;
           channelId < CONNECTIONS_LIMIT;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      deallocateTransport(channelId);

      delete channels_[channelId];
      channels_[channelId] = NULL;
    }
  }

  delete transport_;

  delete compressor_;
  delete decompressor_;

  //
  // Delete storage shared among channels.
  //

  delete opcodeStore_;

  delete clientStore_;
  delete serverStore_;

  delete clientCache_;
  delete serverCache_;

  //
  // Get ready for the next run.
  //

  Channel::resetAgent();

  #ifdef DEBUG
  *logofs << "Proxy: Deleted proxy object at " << this
          << ".\n" << logofs_flush;
  #endif
}

int Proxy::setOperational()
{
  #ifdef TEST
  *logofs << "Proxy: Entering operational mode.\n"
          << logofs_flush;
  #endif

  operation_ = operation_in_messages;

  return 1;
}

int Proxy::setReadDescriptors(fd_set *fdSet, int &fdMax, T_timestamp &tsMax)
{
  //
  // Initial timeout used to ping the remote proxy.
  //

  #ifdef TEST
  *logofs << "Proxy: Initial timeout is " << tsMax.tv_sec
          << " S and " << (double) tsMax.tv_usec /
             1000 << " Ms.\n" << logofs_flush;
  #endif

  setMinTimestamp(tsMax, control -> PingTimeout);

  //
  // Loop through the valid channels and set
  // the descriptors selected for read and
  // the timeout.
  //

  int fd    = -1;
  int limit = -1;
  int split = -1;

  if (isTimeToRead() == 1)
  {
    for (int channelId = lowerChannel_;
             channelId <= upperChannel_;
                 channelId++)
    {
      if (channels_[channelId] == NULL)
      {
        continue;
      }

      fd = getFd(channelId);

      if (channels_[channelId] -> needLimit())
      {
        //
        // Will later cause the proxy to be not
        // selected for read.
        //

        #ifdef TEST
        *logofs << "Proxy: WARNING! Channel for descriptor FD#"
                << fd << " is beyond its buffer limit.\n"
                << logofs_flush;
        #endif

        limit = fd;
      }

      if (channels_[channelId] -> getFinish() == 0 &&
              congestions_[channelId] == 0)
      {
        FD_SET(fd, fdSet);

        #ifdef TEST
        *logofs << "Proxy: Descriptor FD#" << fd
                << " selected for read with buffer length "
                << transports_[channelId] -> length()
                << ".\n" << logofs_flush;
        #endif

        #ifdef DEBUG
        *logofs << "Proxy: Including descriptor FD#" << fd
                << " in count of total descriptors.\n"
                << logofs_flush;
        #endif

        if (fd >= fdMax)
        {
          fdMax = fd + 1;
        }

        //
        // We should never have data left in the
        // read buffer.
        //

        #if defined(TEST) || defined(INFO)

        if (channels_[channelId] -> getPending() == 1)
        {
          #ifdef PANIC
          *logofs << "Proxy: PANIC! Channel with descriptor FD#"
                  << fd << " has pending messages.\n"
                  << logofs_flush;
          #endif

          cerr << "Error" << ": Channel with descriptor FD#"
               << fd << " has pending messages.\n";

          HandleCleanup();
        }

        #endif

        //
        // Restart the proxy if there are motion
        // events to flush.
        //

        if (channels_[channelId] -> needMotion() == 1)
        {
          #ifdef TEST
          *logofs << "Proxy: Requesting timeout of "
                  << control -> MotionTimeout << " Ms as FD#"
                  << fd << " has motion events to flush.\n"
                  << logofs_flush;
          #endif

          setMinTimestamp(tsMax, control -> MotionTimeout);
        }

        if (split == -1 && channels_[channelId] -> needSplit() == 1)
        {
          int diffTs = getTimeToNextSplit();

          #if defined(TEST) || defined(INFO) || defined(FLUSH)

          if (diffTimestamp(lastSplitTs_,
                  getTimestamp()) > timeouts_.split)
          {
            *logofs << "Proxy: FLUSH! WARNING! Running with "
                    << diffTimestamp(lastSplitTs_, getTimestamp())
                    << " Ms elapsed since the last split.\n"
                    << logofs_flush;
          }

          *logofs << "Proxy: FLUSH! Requesting timeout of " << diffTs
                  << " Ms as FD#" << fd << " has splits to send.\n"
                  << logofs_flush;

          #endif

          setMinTimestamp(tsMax, diffTs);

          //
          // Set the timeout only once for all
          // channels.
          //

          split = 0;
        }
      }
    }
  }
  #if defined(TEST) || defined(INFO)
  else
  {
    *logofs << "Proxy: WARNING! Disabled reading from X connections.\n"
            << logofs_flush;

    *logofs << "Proxy: WARNING! Reset is " << reset_ << " congestion "
            << congestion_ << " pending " << pending_ << " blocked "
            << transport_ -> blocked() << " length " << transport_ ->
               length() << ".\n" << logofs_flush;
  }
  #endif

  //
  // Sanity check on enqued data to X connections.
  // If it exceeds the buffer limit then stop
  // reading from proxy.
  //

  if (limit == -1)
  {
    FD_SET(fd_, fdSet);

    #ifdef TEST

    int fdMaxLength = 0;

    for (int channelId = lowerChannel_;
             channelId <= upperChannel_;
                 channelId++)
    {
      if (channels_[channelId] != NULL &&
              transports_[channelId] -> length() > fdMaxLength)
      {
        fdMaxLength = transports_[channelId] -> length();
      }
    }

    *logofs << "Proxy: Proxy descriptor FD#" << fd_ << " selected "
            << "for read. Maximum size of buffers is " << fdMaxLength
            << ".\n" << logofs_flush;

    #endif

    #ifdef DEBUG
    *logofs << "Proxy: Including proxy descriptor FD#" << fd_
            << " in count of total descriptors.\n"
            << logofs_flush;
    #endif

    if (fd_ >= fdMax)
    {
      fdMax = fd_ + 1;
    }

    #if defined(TEST) || defined(INFO)

    if (pending_ == 1)
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! Descriptor for proxy FD#" << fd_
              << " has pending messages.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Descriptor for proxy FD#" << fd_
           << " has pending messages.\n";

      HandleCleanup();
    }

    #endif
  }
  #ifdef TEST
  else
  {
    *logofs << "Proxy: WARNING! Disabled reading from proxy link "
            << "FD#" << fd_ << " with offending channel "
            << limit << ".\n" << logofs_flush;
  }
  #endif

  return 1;
}


//
// Add to the mask the file descriptors of all
// X connections to write to.
//

int Proxy::setWriteDescriptors(fd_set *fdSet, int &fdMax, T_timestamp &tsMax)
{
  int fd = -1;

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      fd = getFd(channelId);

      if (transports_[channelId] -> length() > 0)
      {
        FD_SET(fd, fdSet);

        #ifdef TEST
        *logofs << "Proxy: Descriptor FD#" << fd << " selected "
                << "for write with blocked " << transports_[channelId] ->
                   blocked() << " and length " << transports_[channelId] ->
                   length() << ".\n" << logofs_flush;
        #endif

        if (fd >= fdMax)
        {
          fdMax = fd + 1;
        }
      }
      #ifdef TEST
      else
      {
        *logofs << "Proxy: Descriptor FD#" << fd << " not selected "
                << "for write with blocked " << transports_[channelId] ->
                   blocked() << " and length " << transports_[channelId] ->
                   length() << ".\n" << logofs_flush;
      }
      #endif

      #if defined(TEST) || defined(INFO)

      if (transports_[channelId] -> getType() !=
              transport_agent && transports_[channelId] ->
                   length() > 0 && transports_[channelId] ->
                       blocked() != 1)
      {
        *logofs << "Proxy: PANIC! Descriptor FD#" << fd
                << " has data to write but blocked flag is "
                << transports_[channelId] -> blocked()
                << ".\n" << logofs_flush;

        cerr << "Error" << ": Descriptor FD#" << fd
             << " has data to write but blocked flag is "
             << transports_[channelId] -> blocked()
             << ".\n";

        HandleCleanup();
      }

      #endif
    }
  }

  //
  // Check if the proxy link has data to be
  // written from a previous blocking write.
  //
  // Should we reset the flush timeout here?
  //

  if (transport_ -> blocked() == 1)
  {
    FD_SET(fd_, fdSet);

    #ifdef TEST
    *logofs << "Proxy: Proxy descriptor FD#"
            << fd_ << " selected for write. Blocked is "
            << transport_ -> blocked() << " length is "
            << transport_ -> length() << ".\n"
            << logofs_flush;
    #endif

    if (fd_ >= fdMax)
    {
      fdMax = fd_ + 1;
    }
  }
  else
  {
    #ifdef TEST
    *logofs << "Proxy: Proxy descriptor FD#"
            << fd_ << " not selected for write. Blocked is "
            << transport_ -> blocked() << " length is "
            << transport_ -> length() << ".\n"
            << logofs_flush;
    #endif

    //
    // Setup a timeout so that we can flush
    // the data we have accumulated so far.
    //

    if (canFlush() == 1)
    {
      #if defined(TEST) || defined(INFO)

      if (control -> FlushPolicy == policy_immediate)
      {
        #ifdef PANIC
        *logofs << "Proxy: PANIC! Proxy descriptor FD#" << fd_
                << " has data to flush with policy set to "
                << "'immediate'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Proxy descriptor FD#" << fd_
             << " has data to flush with policy set to "
             << "'immediate'.\n";

        HandleCleanup();
      }

      #endif

      int diffTs = getTimeToNextFlush();

      #if defined(TEST) || defined(INFO) || defined(FLUSH)

      if (diffTimestamp(lastFrameTs_, getTimestamp()) >
              timeouts_.frame || diffTimestamp(lastFlushTs_,
                  getTimestamp()) > timeouts_.flush)
      {
        *logofs << "Proxy: FLUSH! WARNING! Running with "
                << diffTimestamp(lastFrameTs_, getTimestamp())
                << " and " << diffTimestamp(lastFlushTs_, getTimestamp())
                << " Ms elapsed since the last flush.\n"
                << logofs_flush;
      }

      *logofs << "Proxy: FLUSH! Requesting timeout of " << diffTs
              << " Ms as proxy FD#" << fd_ << " has data "
              << "to flush.\n" << logofs_flush;

      #endif

      setMinTimestamp(tsMax, diffTs);
    }
  }

  //
  // We are entering the main select. Save
  // the timestamp of the last loop so that
  // we can detect the clock drifts.
  //

  lastLoopTs_ = getTimestamp();

  return 1;
}

int Proxy::getChannels(T_channel_type type)
{
  int channels = 0;

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            (type == channel_none ||
                 type == channels_[channelId] ->
                     getType()))
    {
      channels++;
    }
  }

  return channels;
}

T_channel_type Proxy::getType(int fd)
{
  int channelId = getChannel(fd);

  if (channelId < 0 || channels_[channelId] == NULL)
  {
    return channel_none;
  }

  return channels_[channelId] -> getType();
}

//
// Handle data from channels selected for read
// or having pending messages in their buffer.
//

int Proxy::handleRead(int &resultFds, fd_set &readSet)
{
  #ifdef DEBUG
  *logofs << "Proxy: Checking descriptors selected for read.\n"
          << logofs_flush;
  #endif

  if ((resultFds > 0 && FD_ISSET(fd_, &readSet)) || pending_ == 1)
  {
    #ifdef DEBUG
    *logofs << "Proxy: Going to read messages from proxy FD#"
            << fd_ << " with pending " << pending_ << " and "
            << "FD_ISSET() " << (int) FD_ISSET(fd_, &readSet)
            << ".\n" << logofs_flush;
    #endif

    if (handleRead() < 0)
    {
      #ifdef TEST
      *logofs << "Proxy: Failure reading from proxy FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      return -1;
    }

    //
    // We handled the descriptor so we can clear the
    // mask. This will avoid that the agent will get
    // confused if it tries to read from our own
    // descriptors.
    //

    if (FD_ISSET(fd_, &readSet))
    {
      #ifdef DEBUG
      *logofs << "Proxy: Clearing the read descriptor "
              << "for proxy FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      FD_CLR(fd_, &readSet);

      resultFds--;
    }
  }

  //
  // Can't presently use the value of the select
  // result because there might be descriptors
  // set as pending.
  // 

  for (int j = (firstChannel_ + 1 > upperChannel_ ?
                    lowerChannel_ : firstChannel_ + 1);
                        ; (j + 1 > upperChannel_ ?
                               j = lowerChannel_ : j++))
  {
    #ifdef DEBUG
    *logofs << "Proxy: Looping with first "
            << firstChannel_ << " and current "
            << j << ".\n" << logofs_flush;
    #endif

    int fd = getFd(j);

    if (fd >= 0 && (getPending(fd) > 0 ||
            (resultFds > 0 && FD_ISSET(fd, &readSet))))
    {
      #ifdef DEBUG
      *logofs << "Proxy: Going to read messages from FD#"
              << fd << " with pending " << getPending(fd)
              << " and FD_ISSET() " << (int) FD_ISSET(fd, &readSet)
              << ".\n" << logofs_flush;
      #endif

      int result = handleRead(fd);

      if (result < 0)
      {
        #ifdef TEST
        *logofs << "Proxy: Failure reading messages from FD#"
                << fd << ".\n" << logofs_flush;
        #endif

        return -1;
      }

      if (FD_ISSET(fd, &readSet))
      {
        #ifdef DEBUG
        *logofs << "Proxy: Clearing the read descriptor "
                << "for FD#" << fd << ".\n" << logofs_flush;
        #endif

        FD_CLR(fd, &readSet);

        resultFds--;
      }
    }

    if (j == firstChannel_)
    {
      firstChannel_ = (firstChannel_ + 1 > upperChannel_ ?
                           lowerChannel_ : firstChannel_ + 1);
      break;
    }
  }

  return 1;
}

//
// Perform flush on descriptors selected for write.
//

int Proxy::handleFlush(int &resultFds, fd_set &writeSet)
{
  #ifdef DEBUG
  *logofs << "Proxy: Checking descriptors selected for write.\n"
          << logofs_flush;
  #endif

  if (resultFds > 0 && FD_ISSET(fd_, &writeSet))
  {
    #ifdef TEST
    *logofs << "Proxy: FLUSH! Proxy descriptor FD#" << fd_
            << " reported to be writable.\n"
            << logofs_flush;
    #endif

    if (handleFlush(flush_if_any) < 0)
    {
      #ifdef TEST
      *logofs << "Proxy: Failure flushing the writable "
              << "proxy FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      return -1;
    }

    #ifdef DEBUG
    *logofs << "Proxy: Clearing the write descriptor "
            << "for proxy FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    FD_CLR(fd_, &writeSet);

    resultFds--;
  }

  for (int j = (firstChannel_ + 1 > upperChannel_ ?
                    lowerChannel_ : firstChannel_ + 1);
                        resultFds > 0; (j + 1 > upperChannel_ ?
                            j = lowerChannel_ : j++))
  {
    #ifdef DEBUG
    *logofs << "Proxy: Looping with first "
            << firstChannel_ << " and current "
            << j << ".\n" << logofs_flush;
    #endif

    int fd = getFd(j);

    if (fd >= 0 && FD_ISSET(fd, &writeSet))
    {
      #ifdef TEST
      *logofs << "Proxy: X descriptor FD#" << fd
              << " reported to be writable.\n"
              << logofs_flush;
      #endif

      //
      // It can happen that, in handling reads, we have
      // destroyed the buffer associated to a closed
      // socket so don't complain about errors.
      //

      handleFlush(flush_if_any, fd);

      //
      // Clear the descriptor from the mask so
      // we don't confuse an eventual agent not
      // checking only its own descriptors.
      //

      #ifdef DEBUG
      *logofs << "Proxy: Clearing the write descriptor "
              << "for FD#" << fd << ".\n"
              << logofs_flush;
      #endif

      FD_CLR(fd, &writeSet);

      resultFds--;
    }

    if (j == firstChannel_)
    {
      firstChannel_ = (firstChannel_ + 1 > upperChannel_ ?
                           lowerChannel_ : firstChannel_ + 1);
      break;
    }
  }

  return 1;
}

int Proxy::handleRead()
{
  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Handling data from proxy on FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // Decode all the available messages from
  // the remote proxy until is not possible
  // to read more.
  //

  for (;;)
  {
    int result = readBuffer_.readMessage();

    #if defined(TEST) || defined(DEBUG) || defined(INFO)
    *logofs << "Proxy: Read result on proxy FD#" << fd_
            << " is " << result << " with pending "
            << pending_ << ".\n" << logofs_flush;
    #endif

    if (result < 0)
    {
      if (shutdown_ == 0)
      {
        #ifdef PANIC
        *logofs << "Proxy: PANIC! Lost connection to peer proxy on FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Lost connection to peer proxy on FD#"
             << fd_ << ".\n";
      }
      #ifdef TEST
      else
      {
        *logofs << "Proxy: Closure of the proxy link detected "
                << "after clean shutdown.\n" << logofs_flush;
      }
      #endif

      return -1;
    }
    else if (result == 0 && pending_ == 0)
    {
      #if defined(TEST) || defined(DEBUG) || defined(INFO)
      *logofs << "Proxy: No data read from proxy FD#"
              << fd_ << "\n" << logofs_flush;
      #endif

      return 0;
    }

    //
    // Set time of last read from remote
    // proxy.
    //

    lastBytesInTs_ = getTimestamp();

    //
    // Jump out of proxy congestion state.
    //

    if (control -> ProxyMode == proxy_server)
    {
      congestion_ = 0;
    }

    //
    // Show the 'no data received' dialog
    // at next timeout.
    //

    lastAlertTs_ = nullTimestamp();

    #if defined(TEST) || defined(OPCODES) || defined(INFO)
    *logofs << "Proxy: Getting messages from proxy FD#" << fd_
            << " with " << readBuffer_.getLength() << " bytes "
            << "in the read buffer.\n" << logofs_flush;
    #endif

    unsigned int controlLength;
    unsigned int dataLength;

    const unsigned char *message;

    while ((message = readBuffer_.getMessage(controlLength, dataLength)) != NULL)
    {
      if (control -> CollectStatistics)
      {
        statistics -> addFrameIn();
      }

      if (controlLength == 3 && *message == 0 &&
              *(message + 1) < code_last_tag)
      {
        if (handleControlFromProxy(message) < 0)
        {
          return -1;
        }
      }
      else if (operation_ == operation_in_messages)
      {
        int channelId = inputChannel_;

        #if defined(TEST) || defined(OPCODES)
        *logofs << "Proxy: Identified message of " << dataLength
                << " bytes for FD#" << getFd(channelId) << " channel ID#"
                 << channelId << ".\n" << logofs_flush;
        #endif

        if (channelId >= 0 && channelId < CONNECTIONS_LIMIT &&
                channels_[channelId] != NULL)
        {
          int finish = channels_[channelId] -> getFinish();

          #ifdef WARNING

          if (finish == 1)
          {
            *logofs << "Proxy: WARNING! Handling data for finishing "
                    << "FD#" << getFd(channelId) << " channel ID#"
                    << channelId << ".\n" << logofs_flush;
          }

          #endif

          int result = channels_[channelId] -> handleWrite(message, dataLength);

          //
          // Check if this is the first time that
          // failure on descriptor was detected.
          //

          if (result < 0 && finish == 0)
          {
            #ifdef TEST
            *logofs << "Proxy: Failed to write proxy data to FD#"
                    << getFd(channelId) << " channel ID#"
                    << channelId << ".\n" << logofs_flush;
            #endif

            if (handleFinish(channelId) < 0)
            {
              return -1;
            }
          }

          //
          // Check if we have user input and
          // so decrease the flush timeout.
          //

          if (channels_[channelId] -> getInput() > 0)
          {
            #if defined(TEST) || defined(INFO) || defined(ADJUST)
            *logofs << "Proxy: ADJUST! Propagating input to proxy "
                    << "FD#" << fd_ << " at " << strMsTimestamp()
                    << " because of FD#" << getFd(channelId)
                    << ".\n" << logofs_flush;
            #endif

            handleInput();

            channels_[channelId] -> clearInput();
          }
        }
        #ifdef WARNING
        else
        {
          *logofs << "Proxy: WARNING! Ignoring data received for "
                  << "invalid channel ID#" << channelId << ".\n"
                  << logofs_flush;
        }
        #endif
      }
      else if (operation_ == operation_in_statistics)
      {
        #ifdef TEST
        *logofs << "Proxy: Received statistics data from remote proxy.\n"
                << logofs_flush;
        #endif

        if (handleStatisticsFromProxy(message, dataLength) < 0)
        {
          return -1;
        }

        operation_ = operation_in_messages;
      }
      else if (operation_ == operation_in_negotiation)
      {
        #ifdef TEST
        *logofs << "Proxy: Received new negotiation data from remote proxy.\n"
                << logofs_flush;
        #endif

        if (handleNegotiationFromProxy(message, dataLength) < 0)
        {
          return -1;
        }
      }

      //
      // if (controlLength == 3 && *message == 0 && ...) ...
      // else if (operation_ == operation_in_statistics) ...
      // else if (operation_ == operation_in_messages) ...
      // else if (operation_ == operation_in_negotiation) ...
      // else ...
      //

      else
      {
        #ifdef PANIC
        *logofs << "Proxy: PANIC! Unrecognized message received on proxy FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Unrecognized message received on proxy FD#"
             << fd_ << ".\n";

        return -1;
      }

    } // while ((message = readBuffer_.getMessage(controlLength, dataLength)) != NULL) ...

    //
    // No more complete messages are available
    // in the read buffer.
    //

    pending_ = 0;

    //
    // Wait for a complete message to
    // become available.
    //

    if (readBuffer_.getRemaining() > 0)
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "Proxy: WARNING! Waiting a complete "
              << "message with " << readBuffer_.getLength()
              << " bytes and " << readBuffer_.getRemaining()
              << " remaining.\n" << logofs_flush;
      #endif

      readBuffer_.waitMessage(control -> RetryTimeout);
    }

    //
    // Reset the read buffer.
    //

    readBuffer_.partialReset();

    //
    // Ensure that we haven't left data in the
    // encode buffer.
    //

    #if defined(TEST) || defined(INFO)

    if (encodeBuffer_.getLength() + controlLength_ > 0)
    {
      *logofs << "Proxy: PANIC! Data is still present in "
              << "the encode buffer.\n" << logofs_flush;

      HandleCleanup();
    }

    #endif

  } // End of for (;;) ...
}

int Proxy::handleControlFromProxy(const unsigned char *message)
{
  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Received message '" << CodeLabel(*(message + 1))
          << "' at " << strMsTimestamp() << " with data ID#"
          << (int) *(message + 2) << ".\n" << logofs_flush;
  #endif

  switch (*(message + 1))
  {
    case code_switch_connection:
    {
      int channelId = *(message + 2);

      //
      // If channel is invalid further messages will
      // be ignored. The acknowledged shutdown of
      // channels should prevent this.
      //

      inputChannel_ = channelId;

      break;
    }
    case code_begin_congestion:
    {
      //
      // Set congestion state for channel
      // reported by remote proxy.
      //

      int channelId = *(message + 2);

      if (channels_[channelId] != NULL)
      {
        congestions_[channelId] = 1;
      }
      #ifdef WARNING
      else
      {
        *logofs << "Proxy: WARNING! Received a begin congestion message "
                << "for invalid channel id ID#" << channelId
                << " with FD#" << getFd(channelId) << ".\n"
                << logofs_flush;
      }
      #endif

      break;
    }
    case code_end_congestion:
    {
      //
      // Attend again to channel.
      //

      int channelId = *(message + 2);

      if (channels_[channelId] != NULL)
      {
        congestions_[channelId] = 0;
      }
      #ifdef WARNING
      else
      {
        *logofs << "Proxy: WARNING! Received an end congestion message "
                << "for invalid channel id ID#" << channelId
                << " with FD#" << getFd(channelId) << ".\n"
                << logofs_flush;
      }
      #endif

      break;
    }
    case code_token_request:
    {
      if (handleTokenFromProxy() < 0)
      {
        return -1;
      }

      break;
    }
    case code_token_reply:
    {
      //
      // Increment the available tokens.
      //

      tokens_++;

      if (tokens_ >  control -> TokenLimit)
      {
        #ifdef PANIC
        *logofs << "Proxy: PANIC! Token overflow handling messages.\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Token overflow handling messages.\n";

        HandleCleanup();
      }

      //
      // Jump out of congestion state.
      //

      congestion_ = 0;

      break;
    }
    case code_new_x_connection:
    case code_new_cups_connection:
    case code_new_keybd_connection:
    case code_new_samba_connection:
    case code_new_media_connection:
    case code_new_http_connection:
    {
      int result;

      int channelId = *(message + 2);

      if (*(message + 1) == code_new_x_connection)
      {
        result = handleNewXConnectionFromProxy(channelId);
      }
      else if (*(message + 1) == code_new_cups_connection)
      {
        result = handleNewCupsConnectionFromProxy(channelId);
      }
      else if (*(message + 1) == code_new_keybd_connection)
      {
        result = handleNewKeybdConnectionFromProxy(channelId);
      }
      else if (*(message + 1) == code_new_samba_connection)
      {
        result = handleNewSambaConnectionFromProxy(channelId);
      }
      else if (*(message + 1) == code_new_media_connection)
      {
        result = handleNewMediaConnectionFromProxy(channelId);
      }
      else
      {
        result = handleNewHttpConnectionFromProxy(channelId);
      }

      if (result < 0)
      {
        //
        // Realization of new channel failed.
        // Send channel shutdown message to
        // the peer proxy.
        //

        if (handleControl(code_drop_connection, channelId) < 0)
        {
          return -1;
        }
      }

      break;
    }
    case code_drop_connection:
    {
      int channelId = *(message + 2);

      if (channelId >= 0 && channelId < CONNECTIONS_LIMIT &&
              channels_[channelId] != NULL)
      {
        handleFinishFromProxy(channelId);

        //
        // Check if it's time to save content
        // of message stores. Don't abort the
        // connection if can't write to disk.
        //

        handleCheckSave();
      }
      #ifdef WARNING
      else
      {
        *logofs << "Proxy: WARNING! Received a drop message for "
                << "invalid channel id ID#" << channelId
                << " with FD#" << getFd(channelId) << ".\n"
                << logofs_flush;
      }
      #endif

      break;
    }
    case code_finish_connection:
    {
      //
      // Prepare the channel for closing connection and
      // send channel shutdown message to the peer proxy.
      //

      int channelId = *(message + 2);

      int channelValid = 0;

      if (channelId >= 0 && channelId < CONNECTIONS_LIMIT &&
              channels_[channelId] != NULL)
      {
        //
        // Force finish state on channel.
        //

        channels_[channelId] -> handleFinish();

        handleFinishFromProxy(channelId);

        channelValid = 1;
      }
      #ifdef WARNING
      else
      {
        *logofs << "Proxy: WARNING! Received a finish message for "
                << "invalid channel id ID#" << channelId
                << " with FD#" << getFd(channelId) << ".\n"
                << logofs_flush;
      }
      #endif

      //
      // Send anyway the channel shutdown
      // message to the peer proxy.
      //

      if (handleControl(code_drop_connection, channelId) < 0)
      {
        return -1;
      }

      //
      // If we actually dropped the channel
      // check if it's time to save content
      // of message stores to disk.
      //

      if (channelValid == 1 && handleCheckSave() < 0)
      {
        return -1;
      }

      break;
    }
    case code_reset_request:
    {
      //
      // Remote proxy completed its reset.
      //

      reset_ = 0;

      break;
    }
    case code_shutdown_request:
    {
      //
      // Time to rest in peace.
      //

      shutdown_ = 1;

      break;
    }
    case code_load_request:
    {
      if (handleLoadFromProxy() < 0)
      {
        return -1;
      }

      break;
      }
    case code_save_request:
    {
      //
      // Don't abort the connection
      // if can't write to disk.
      //

      handleSaveFromProxy();

      break;
    }
    case code_statistics_request:
    {
      int type = *(message + 2);

      if (handleStatisticsFromProxy(type) < 0)
      {
        return -1;
      }

      break;
    }
    case code_statistics_reply:
    {
      operation_ = operation_in_statistics;

      break;
    }
    case code_alert_request:
    {
      HandleAlert(*(message + 2), 1);

      break;
    }
    case code_sync_request:
    {
      if (handleSyncFromProxy() < 0)
      {
        return -1;
      }

      break;
    }
    case code_sync_reply:
    {
      //
      // The previous frame contained the
      // prioritized replies. No action
      // is needed here.
      //

      #if defined(TEST) || defined(INFO) || defined(SYNC)
      *logofs << "Proxy: SYNC! Received a synchronization "
              << "reply from the remote proxy.\n"
              << logofs_flush;
      #endif

      #if defined(TEST) || defined(INFO) || defined(SYNC)

      if (notifiedSync_ != 1)
      {
        *logofs << "Proxy: SYNC! PANIC! No synchronization "
                << "request was supposed to be pending.\n"
                << logofs_flush;

        HandleCleanup();
      }

      #endif

      notifiedSync_ = -1;

      break;
    }
    default:
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! Received bad control message number "
              << (unsigned int) *(message + 1) << " with attribute "
              << (unsigned int) *(message + 2) << ".\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Received bad control message number "
           << (unsigned int) *(message + 1) << " with attribute "
           << (unsigned int) *(message + 2) << ".\n";

      HandleCleanup();
    }

  } // End of switch (*(message + 1)) ...

  return 1;
}

int Proxy::handleRead(int fd)
{
  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Handling data from connection on FD#"
          << fd << ".\n" << logofs_flush;
  #endif

  if (canRead(fd) == 0)
  {
    #if defined(TEST) || defined(INFO)

    if (getPending(fd) == 1)
    {
      *logofs << "Proxy: PANIC! Can't read from FD#" << fd
            << " with data being pending.\n"
            << logofs_flush;

      HandleCleanup();
    }
    else if (getChannel(fd) < 0)
    {
      *logofs << "Proxy: PANIC! Can't read from invalid FD#"
              << fd << ".\n" << logofs_flush;

      HandleCleanup();
    }
    else
    {
      *logofs << "Proxy: WARNING! Read method called for FD#"
              << fd << " but operation is not possible.\n"
              << logofs_flush;
    }

    #endif

    return 0;
  }

  //
  // Let the channel object read all the new data from
  // its file descriptor, isolate messages, compress
  // those messages, and append the compressed form to
  // the encode buffer.
  //

  int channelId = getChannel(fd);

  #if defined(TEST) || defined(OPCODES)
  *logofs << "Proxy: Reading messages from FD#" << fd
          << " channel ID#" << channelId << ".\n"
          << logofs_flush;
  #endif

  int result = channels_[channelId] -> handleRead(encodeBuffer_);

  //
  // Even in case of failure write produced data to the
  // proxy connection. To keep stores synchronized remote
  // side needs to decode any encoded message, also if X
  // socket was closed in the meanwhile. If this is the
  // case, the decompressed output will not be passed to
  // transport and will be silently discarded.
  //

  if (result < 0)
  {
    #ifdef TEST
    *logofs << "Proxy: Failed to read data from X connection FD#"
            << fd << " channel ID#" << channelId << ".\n"
            << logofs_flush;
    #endif

    encodeBuffer_.fullReset();

    if (handleFinish(channelId) < 0)
    {
      return -1;
    }
  }
  else if (encodeBuffer_.getLength() + controlLength_ > 0)
  {
    //
    // Normal clients get a flush whenever
    // there is a reply pending. The same
    // happens to the agent at startup.
    //

    if (channels_[channelId] -> getPriority() > 0)
    {
      if (control -> AgentFlushPriority == 1)
      {
        #if defined(TEST) || defined(INFO) || defined(ADJUST)
        *logofs << "Proxy: ADJUST! Propagating priority to proxy "
                << "FD#" << fd_ << " at " << strMsTimestamp()
                << " because of FD#" << fd << ".\n"
                << logofs_flush;
        #endif

        handlePriority();
      }

      channels_[channelId] -> clearPriority();
    }

    //
    // Check if we have user input and
    // so decrease the flush timeout.
    //

    if (channels_[channelId] -> getInput() > 0)
    {
      #if defined(TEST) || defined(INFO) || defined(ADJUST)
      *logofs << "Proxy: ADJUST! Propagating input to proxy "
              << "FD#" << fd_ << " at " << strMsTimestamp()
              << " because of FD#" << fd << ".\n"
              << logofs_flush;
      #endif

      handleInput();

      channels_[channelId] -> clearInput();
    }

    if (handleSwitch(channelId) < 0)
    {
      return -1;
    }
    else if (handleFrame(frame_data) < 0)
    {
      return -1;
    }
  }

  //
  // Maybe we have splits to send now.
  //

  if (channels_[channelId] -> needSplit() == 1 &&
        isTimestamp(lastSplitTs_) == 0)
  {
    //
    // TODO: Presently we do not always reset
    // the timeout as soon as the split store
    // has become empty.
    //

    lastSplitTs_ = getTimestamp();
  }

  //
  // Maybe we got past the sequence that the
  // remote peer requested to synchronize.
  //

  if (control -> ProxyMode == proxy_server &&
          notifiedSync_ >= 0)
  {
    if (handleSync(channelId) < 0)
    {
      return -1;
    }
  }

  //
  // Maybe the X client channel has given up
  // because there was data available for
  // the proxy. Try to consume the available
  // data.
  //

  if (control -> ProxyMode == proxy_client &&
          transport_ -> readable() > 0)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Proxy: WARNING! Going to asynchronously "
            << "read from the proxy FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    if (handleRead() < 0)
    {
      return -1;
    }
  }

  return 1;
}

int Proxy::handleEvents()
{
  #ifdef TEST
  *logofs << "Proxy: Going to check events on channels.\n"
          << logofs_flush;
  #endif

  //
  // There are two types of congestion events:
  //
  // 1. A X congestion event is sent to X channels if the
  //    client proxy can't write its data to the remote
  //    peer. X channels should stop producing data until
  //    a new event is received, ceasing the previous
  //    condition.
  //
  // 2. Proxy sends to the remote peer a begin congestion
  //    control code in the case the local end of the
  //    channel is not consuming the data accumulated in
  //    its buffer. This allows the remote proxy to stop
  //    accepting more data.
  //

  if (control -> ProxyMode == proxy_client &&
          congestion_ != notifiedCongestion_)
  {
    #ifdef TEST
    *logofs << "Proxy: Handling congestion notify "
            << "on channels for proxy FD#" << fd_
            << ".\n" << logofs_flush;
    #endif

    if (congestion_ == 1)
    {
      handleNotify(notify_begin_congestion);
    }
    else
    {
      handleNotify(notify_end_congestion);
    }

    //
    // Save the current state.
    //

    notifiedCongestion_ = congestion_;
  }

  //
  // Check if we can safely write to the
  // proxy link.
  //

  int read = isTimeToRead();

  //
  // Will be updated later, if it is time
  // to send more splits.
  //

  int split = -1;

  //
  // Loop on channels and send the pending
  // events.
  //

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] == NULL ||
            channels_[channelId] -> getFinish() == 1)
    {
      #ifdef TEST

      if (channels_[channelId] != NULL &&
              channels_[channelId] -> getFinish() == 1)
      {
        *logofs << "Proxy: Skipping finishing "
                << "descriptor FD#" << getFd(channelId)
                  << " channel ID#" << channelId << ".\n"
                  << logofs_flush;
      }

      #endif

      continue;
    }

    //
    // Handle congestion state in channels even
    // if we cannot presently write to the proxy
    // link.
    //
    // The channel will report if a transition in
    // congestion state has occurred. Proxy shall
    // get the new value and send an appropriate
    // control message.
    //

    if (channels_[channelId] -> handleCongestion() == 1)
    {
      int state = channels_[channelId] -> getCongestion();

      T_proxy_code controlCode;

      if (state == 1)
      {
        controlCode = code_begin_congestion;
      }
      else
      {
        controlCode = code_end_congestion;
      }

      if (handleControl(controlCode, channelId) < 0)
      {
        return -1;
      }
    }

    //
    // Now check if we can safely produce
    // more data for the channel.
    //

    if (read == 0 || congestions_[channelId] == 1)
    {
      #ifdef TEST

      if (read == 0)
      {
        *logofs << "Proxy: Can't handle events for FD#"
                << getFd(channelId) << " channel ID#"
                << channelId << " with proxy not available.\n"
                << logofs_flush;
      }
      else
      {
        *logofs << "Proxy: Can't handle events for FD#"
                << getFd(channelId) << " channel ID#"
                << channelId << " with channel congested.\n"
                << logofs_flush;
      }

      #endif

      continue;
    }

    //
    // Handle the timeouts on channel operations.
    //

    int result = 0;

    //
    // Handle the motion events.
    //

    if (result >= 0 && channels_[channelId] -> needMotion() == 1)
    {
      #ifdef TEST
      *logofs << "Proxy: Going to send motion events for FD#"
              << getFd(channelId) << ".\n" << logofs_flush;
      #endif

      result = channels_[channelId] -> handleMotion(encodeBuffer_, 0);

      #ifdef TEST

      if (result < 0)
      {
        *logofs << "Proxy: Failed to handle motion events for FD#"
                << getFd(channelId) << " channel ID#" << channelId
                << ".\n" << logofs_flush;
      }

      #endif
    }

    if (result >= 0 && channels_[channelId] -> needSplit() == 1)
    {
      //
      // Check if it is time to send more splits
      // and how many bytes are going to be sent.
      //

      if (split == -1)
      {
        if (isTimeToSplit() == 1 ||
                channels_[channelId] -> needAbort() == 1)
        {
          #if defined(TEST) || defined(INFO) || defined(FLUSH)

          if (isTimeToSplit() == 1)
          {
            *logofs << "Proxy: FLUSH! Split timeout expired with split "
                    << "at " << diffTimestamp(lastSplitTs_, getTimestamp())
                    << ".\n" << logofs_flush;
          }
          else
          {
            *logofs << "Proxy: FLUSH! Aborting split with a pending abort "
                    << "at " << diffTimestamp(lastSplitTs_, getTimestamp())
                    << ".\n" << logofs_flush;
          }

          #endif

          split = control -> SplitDataPacketLimit;
        }
        else
        {
          #if defined(TEST) || defined(INFO) || defined(FLUSH)

          if (diffTimestamp(lastSplitTs_,
                  getTimestamp()) > timeouts_.split)
          {
            *logofs << "Proxy: FLUSH! WARNING! Running with "
                    << diffTimestamp(lastSplitTs_, getTimestamp())
                    << " Ms elapsed since the last split.\n"
                    << logofs_flush;
          }

          #endif

          split = 0;
        }
      }

      if (split > 0)
      {
        #if defined(TEST) || defined(INFO) || defined(FLUSH) || \
                defined(ADJUST) || defined(SPLIT)

        *logofs << "Proxy: ADJUST! FLUSH! SPLIT! Encoding splits for FD#"
                << getFd(channelId) << " at " << strMsTimestamp()
                << " with " << clientStore_ -> getSplitStore() ->
                   getSize() << " elements and " << split
                << " bytes to write.\n" << logofs_flush;

        #endif

        result = channels_[channelId] -> handleSplit(encodeBuffer_, split);

        if (result >= 0)
        {
          //
          // Set or reset the timeout to next split.
          //
          // TODO: Presently we do not always reset
          // the timeout as soon as the split store
          // has become empty.
          //

          if (channels_[channelId] -> needSplit() == 1)
          {
            lastSplitTs_ = getTimestamp();
          }
          else
          {
            lastSplitTs_ = nullTimestamp();
          }

          //
          // All channels currently share the same split
          // store so we send splits only once.
          //

          split = 0;
        }
        #ifdef TEST
        else
        {
          *logofs << "Proxy: Failed to handle splits for FD#"
                  << getFd(channelId) << " channel ID#" << channelId
                  << ".\n" << logofs_flush;
        }
        #endif
      }
    }

    //
    // Finally write the data produce in the
    // encode buffer to the remote proxy.
    //

    if (result >= 0)
    {
      if (encodeBuffer_.getLength() + controlLength_ > 0)
      {
        if (handleSwitch(channelId) < 0)
        {
          return -1;
        }
        else if (handleFrame(frame_data) < 0)
        {
          return -1;
        }
      }
    }
    else
    {
      encodeBuffer_.fullReset();

      if (handleFinish(channelId) < 0)
      {
        return -1;
      }
    }
  }

  //
  // Check if it is now time to flush the
  // proxy link.
  //

  if (canFlush() == 1 && isTimeToFlush() == 1)
  {
    #if defined(TEST) || defined(INFO) || defined(FLUSH)
    *logofs << "Proxy: FLUSH! Flush timeout expired with "
            << "frame at " << diffTimestamp(lastFrameTs_, getTimestamp())
            << " and flush at " << diffTimestamp(lastFlushTs_, getTimestamp())
            << ".\n" << logofs_flush;
    #endif

    if (handleFlush(flush_if_any) < 0)
    {
      return -1;
    }
  }
  #if defined(TEST) || defined(INFO) || defined(FLUSH)

  else if (canFlush() == 1)
  {
    if (diffTimestamp(lastFrameTs_, getTimestamp()) >
            timeouts_.frame || diffTimestamp(lastFlushTs_,
                getTimestamp()) > timeouts_.flush)
    {
      *logofs << "Proxy: FLUSH! WARNING! Running with "
              << diffTimestamp(lastFrameTs_, getTimestamp())
              << " and " << diffTimestamp(lastFlushTs_, getTimestamp())
              << " Ms elapsed since the last flush.\n"
              << logofs_flush;
    }
  }

  #endif

  return 1;
}

int Proxy::handleAsyncSplit(int fd)
{
  int channelId = getChannel(fd);

  int result = channels_[channelId] -> handleAbortSplit(encodeBuffer_);

  if (result < 0)
  {
    //
    // Finish on channel must be called in
    // the parent procedure.
    //

    #if defined(TEST) || defined(INFO)
    *logofs << "ServerProxy: Failed to handle abort splits "
            << "for FD#" << getFd(channelId) << " channel ID#"
            << channelId << ".\n" << logofs_flush;
    #endif

    encodeBuffer_.fullReset();

    return -1;
  }

  if (encodeBuffer_.getLength() + controlLength_ > 0)
  {
    if (handleSwitch(channelId) < 0)
    {
      return -1;
    }
    else if (handleFrame(frame_data) < 0)
    {
      return -1;
    }
  }

  return 1;
}

int Proxy::handleAsyncCongestion(int fd)
{
  //
  // This is a callback, used by channels
  // to notify congestion changes earlier.
  //

  #ifdef TEST
  *logofs << "Proxy: Sending congestion notification "
          << "request for FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  int channelId = getChannel(fd);

  if (channels_[channelId] -> getFinish())
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Congestion notification on "
            << "finishing descriptor FD#" << getFd(channelId)
            << " channel ID#" << channelId << ".\n"
            << logofs_flush;
    #endif

    return -1;
  }

  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Sending immediate code_begin_congestion "
          << "message for FD#" << fd << " at " << strMsTimestamp()
          << ".\n" << logofs_flush;
  #endif

  if (handleControl(code_begin_congestion, channelId) < 0)
  {
    return -1;
  }

  return 1;
}

int Proxy::handleFrame(T_frame_type type)
{
  //
  // Write any outstanding control message, followed by the
  // content of the encode buffer to the proxy transport.
  //
  // This code assumes that encode buffer data is at a loca-
  // tion offset several bytes from start of the buffer, so
  // that the length header and any necessary control bytes
  // can be inserted in front of data already in the buffer.
  // This is the easiest way to encapsulate header and data
  // together in a single frame.
  //
  // The way framing is done is inherently limited and does
  // not allow for getting the best performance, especially
  // when running over a fast link. Framing should be re-
  // written to include the length of the packets in a fixed
  // size header and, furthermore, to incapsulate the control
  // messages and the channel's data in a pseudo X protocol
  // message, so that the proxy itself would be treated like
  // any other channel.
  //

  int token = 0;

  if (control -> ProxyMode == proxy_client &&
          type == frame_ping)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Proxy: Sending a new ping with "
            << notifiedBytes_ + 3 << " bytes accumulated and "
            << tokens_ << " tokens " << "available at "
            << strMsTimestamp() << ".\n" << logofs_flush;
    #endif

    if (handleToken(frame_ping) < 0)
    {
      return -1;
    }

    token = 1;
  }

  unsigned int dataLength = encodeBuffer_.getLength();

  if (dataLength + controlLength_ == 0)
  {
    #if defined(TEST) || defined(INFO)

    *logofs << "Proxy: PANIC! A new frame was requested "
            << "but there is no data to write.\n"
            << logofs_flush;

    HandleCleanup();

    #endif

    return 0;
  }

  //
  // Check if this frame needs to carry a new
  // token request.
  //

  if (control -> ProxyMode == proxy_client &&
          type == frame_data)
  {
    #if defined(TEST) || defined(INFO)

    if (congestion_ != (tokens_ == 0))
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! Congestion is " << congestion_
              << " but have " << tokens_ << " tokens available.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Congestion is " << congestion_
           << " but have " << tokens_ << " tokens available.\n";

      HandleCleanup();
    }
    else if (congestion_ == 1)
    {
      //
      // This can happen because there may be control
      // messages to send, like a proxy shutdown mes-
      // sage or a statistics request. All the other
      // cases should be considered an error.
      //

      #ifdef PANIC
      *logofs << "Proxy: PANIC! Data is to be sent while "
              << "congestion is " << congestion_ << ".\n"
              << logofs_flush;
      #endif
    }

    #endif

    //
    // Send the message but don't send a new token
    // if we are in congestion state.
    //

    int bytes = dataLength + controlLength_ + 3;

    if (congestion_ == 0 && notifiedBytes_ +
            bytes >= control -> TokenBytes)
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "Proxy: Sending a new token with "
              << notifiedBytes_ + bytes << " bytes accumulated "
              << "and " << tokens_ << " tokens " << "available at "
              << strMsTimestamp() << ".\n" << logofs_flush;
      #endif

      if (handleToken(frame_data) < 0)
      {
        return -1;
      }

      token = 1;
    }
  }

  #ifdef DEBUG
  *logofs << "Proxy: Adding a new frame for the remote proxy.\n"
          << logofs_flush;
  #endif

  unsigned char temp[5];

  unsigned int lengthLength = 0;
  unsigned int shift = dataLength;

  while (shift)
  {
    temp[lengthLength++] = (unsigned char) (shift & 0x7f);

    shift >>= 7;
  }

  unsigned char *data = encodeBuffer_.getData();

  unsigned char *outputMessage = data - (controlLength_ + lengthLength);

  unsigned char *nextDest = outputMessage;

  for (int i = 0; i < controlLength_; i++)
  {
    *nextDest++ = controlCodes_[i];
  }

  for (int j = lengthLength - 1; j > 0; j--)
  {
    *nextDest++ = (temp[j] | 0x80);
  }

  if (lengthLength)
  {
    *nextDest++ = temp[0];
  }

  int outputLength = dataLength + controlLength_ + lengthLength;

  #if defined(TEST) || defined(OPCODES) || defined(INFO)
  *logofs << "Proxy: Produced plain output for " << dataLength << "+"
          << controlLength_ << "+" << lengthLength << " out of "
          << outputLength << " bytes.\n" << logofs_flush;
  #endif

  deferredFrames_ += 1;
  deferredBytes_  += outputLength;

  //
  // Reset the counter if we sent the token.
  //

  if (control -> ProxyMode == proxy_client)
  {
    if (token == 1)
    {
      notifiedBytes_ = 0;
    }
    else
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "Proxy: Accumulated " << outputLength
              << " + " << notifiedBytes_ << " bytes with total "
              << notifiedBytes_ + outputLength << " and " << tokens_
              << " tokens " << "available at " << strMsTimestamp()
              << ".\n" << logofs_flush;
      #endif

      notifiedBytes_ += outputLength;
    }
  }

  //
  // Determine if we need to enqueue data to
  // the transport or write it immediately.
  //

  T_write when = write_delayed;

  //
  // By checking if timeouts are expired here
  // we'd end up writing more partial packets.
  //
  // TODO: We write the data immediately if
  // there is already something queued to the
  // network. It must be better evaluated if
  // this provides any benefit.
  //

  if (mustFlush() == 1 || control ->
          FlushPolicy == policy_immediate ||
              queuedFlush() == 1)
  {
    #if defined(TEST) || defined(INFO) || defined(FLUSH)

    if ((mustFlush() == 0 && control ->
            FlushPolicy != policy_immediate) &&
                transport_ -> queued() > 0)
    {
     *logofs << "Proxy: FLUSH! Have " << transport_ ->
                queued() << " bytes queued to the socket for "
             << "proxy FD#" << fd_ << ".\n" << logofs_flush;
    }

    #endif

    #if defined(TEST) || defined(INFO) || defined(FLUSH)
    *logofs << "Proxy: FLUSH! Immediate with blocked " << transport_ ->
               blocked() << " length " << transport_ -> length()
            << " new " << outputLength << " flushable " << transport_ ->
               flushable() << " tokens " << tokens_ << " frames "
            << deferredFrames_ << " bytes " << deferredBytes_
            << ".\n" << logofs_flush;

    *logofs << "Proxy: FLUSH! Immediate flush to proxy FD#" << fd_
            << " of " << outputLength << " bytes at " << strMsTimestamp()
            << " with priority " << priority_ << ".\n" << logofs_flush;

    *logofs << "Proxy: FLUSH! Current bitrate is "
            << control -> getBitrateInShortFrame() << " with "
            << control -> getBitrateInLongFrame() << " in the "
            << "long frame and top " << control -> getTopBitrate()
            << ".\n" << logofs_flush;
    #endif

    when = write_immediate;

    if (control -> CollectStatistics)
    {
      statistics -> addWriteOut();
    }
  }
  else
  {
    T_timestamp now = getTimestamp();

    #if defined(TEST) || defined(INFO) || defined(FLUSH)
    *logofs << "Proxy: FLUSH! Setting last frame timestamp at "
            << strMsTimestamp(now) << ".\n" << logofs_flush;
    #endif

    lastFrameTs_ = now;

    if (isTimestamp(lastFlushTs_) == 0)
    {
      #if defined(TEST) || defined(INFO) || defined(FLUSH)
      *logofs << "Proxy: FLUSH! Setting last flush timestamp at "
              << strMsTimestamp(now) << ".\n" << logofs_flush;
      #endif

      lastFlushTs_ = now;
    }
    
    #if defined(TEST) || defined(INFO) || defined(FLUSH)

    if (diffTimestamp(lastFrameTs_, getTimestamp()) >
            timeouts_.frame || diffTimestamp(lastFlushTs_,
                getTimestamp()) > timeouts_.flush)
    {
      *logofs << "Proxy: FLUSH! WARNING! Running with "
              << diffTimestamp(lastFrameTs_, getTimestamp())
              << " and " << diffTimestamp(lastFlushTs_, getTimestamp())
              << " Ms elapsed since the last flush.\n"
              << logofs_flush;
    }

    if (control -> ProxyMode == proxy_client &&
               isTimeToFlush() == 1)
    {
      *logofs << "Proxy: FLUSH! Delaying partial token "
              << "with length " << transport_ -> length() << " new "
              << outputLength << " flushable " << transport_ ->
                 flushable() << " tokens " << tokens_ << " frames "
              << deferredFrames_ << " bytes " << deferredBytes_
              << ".\n" << logofs_flush;
    }

    #endif
  }

  int result = transport_ -> write(when, outputMessage, outputLength);

  #if defined(TEST) || defined(INFO)

  if (canFlush() == 1 && when == write_immediate)
  {
    *logofs << "Proxy: PANIC! Proxy FD#" << fd_ << " has length "
            << transport_ -> length() << " flushable " << transport_ ->
               flushable() << " blocked " << transport_ -> blocked()
            << " and write immediate.\n" << logofs_flush;

    HandleCleanup();
  }

  #endif

  #ifdef DUMP
  *logofs << "Proxy: Sent " << outputLength << " bytes of data "
          << "with checksum ";

  DumpChecksum(outputMessage, outputLength);

  *logofs << " on proxy FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  #ifdef DUMP
  *logofs << "Proxy: Partial checksums are:\n";

  DumpBlockChecksums(outputMessage, outputLength, 256);

  *logofs << logofs_flush;
  #endif

  //
  // Clean up the encode buffer and
  // revert to the initial size.
  //

  encodeBuffer_.fullReset();

  if (result < 0)
  {
    #ifdef TEST
    *logofs << "Proxy: Failed write on proxy FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    return -1;
  }

  if (control -> CollectStatistics)
  {
    //
    // Account for the data frame and the
    // framing overhead.
    //

    if (dataLength > 0)
    {
      statistics -> addFrameOut();
    }

    statistics -> addFramingBits((controlLength_ + lengthLength) << 3);
  }

  controlLength_ = 0;

  //
  // Update the timestamp of the last frame
  // and recalculate the current bitrate.
  //

  control -> updateBitrate(outputLength);

  //
  // Adjust the timeouts according to the
  // bitrate.
  //

  handleBitrate();

  //
  // If we actually wrote the data to the link
  // then reset all buffers, counters and the
  // priority flag on proxy and the channels.
  //

  if (when == write_immediate)
  {
    handleResetFlush();
  }

  return result;
}

int Proxy::handleFlush(T_flush type)
{
  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Flush requested on proxy FD#" << fd_
          << " with type '" << FlushLabel(type)
          << "'.\n" << logofs_flush;
  #endif

  //
  // Run a bunch of internal coherency tests to
  // verify that we didn't do something wrong.
  //

  #if defined(TEST) || defined(INFO)

  if (type != flush_if_any && type != flush_if_sync)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Flush of proxy descriptor "
            << "FD#" << fd_ << " called with invalid type '"
            << type << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Flush of proxy descriptor "
         << "FD#" << fd_ << " called with invalid type '"
         << type << "'.\n";

    HandleCleanup();
  }

  #endif

  //
  // If this is a request for a synchronous flush
  // then initiate the sync procedure. If the re-
  // mote doesn't support our protocol level, be
  // sure that all data is written to the socket.
  //

  if (type == flush_if_sync)
  {
    if (control -> isProtoStep6() == 1)
    {
      return handleSync();
    }
    else if (transport_ -> length() + transport_ ->
                 flushable() == 0)
    {
      return 0;
    }

    type = flush_if_any;
  }

  //
  // This is not a synchronous flush. Check
  // if we actually have something to write.
  //

  #if defined(TEST) || defined(INFO)

  if (transport_ -> length() + transport_ ->
          flushable() == 0)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Proxy: PANIC! Nothing to flush on the proxy link "
            << "with type '" << FlushLabel(type) << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Nothing to flush on the proxy link "
         << "with type '" << FlushLabel(type) << "'.\n";

    HandleCleanup();
  }

  if (control -> FlushPolicy == policy_immediate &&
          transport_ -> blocked() == 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Proxy descriptor FD#" << fd_
            << " has data to flush with policy set to "
            << "'immediate'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Proxy descriptor FD#" << fd_
         << " has data to flush with policy set to "
         << "'immediate'.\n";

    HandleCleanup();
  }

  //
  // We should always write data as soon
  // as it becomes available if the 'must
  // flush' condition is true.
  //

  if (mustFlush() == 1 && transport_ -> blocked() == 0)
  {
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Proxy descriptor FD#"
            << fd_ << " has the 'must flush' condition "
            << "being true with length " << transport_ ->
               length() << " flushable " << transport_ ->
               flushable() << " priority " << priority_
            << " congestion " << congestion_
            << " and blocked " << transport_ -> blocked()
            << ".\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Flush of proxy descriptor "
         << "FD#" << fd_ << " has the 'must flush' "
         << "condition being true.\n";
  }

  if (transport_ -> blocked() == 0 && isTimeToFlush() == 0)
  {
    #ifdef WARNING
    *logofs << "Proxy: WARNING! It was not time to flush the "
            << "proxy FD#" << fd_ << " at " << strMsTimestamp()
            << " with length " << transport_ -> length()
            << " flushable " << transport_ -> flushable()
            << ".\n" << logofs_flush;
    #endif
  }

  #endif

  #if defined(TEST) || defined(INFO) || defined(FLUSH)
  *logofs << "Proxy: FLUSH! Deferred with blocked " << transport_ ->
             blocked() << " length " << transport_ -> length()
          << " flushable " << transport_ -> flushable() << " tokens "
          << tokens_ << " frames " << deferredFrames_ << " bytes "
          << deferredBytes_ << ".\n" << logofs_flush;

  *logofs << "Proxy: FLUSH! Deferred flush to proxy FD#" << fd_
          << " of " << transport_ -> length() + transport_ ->
             flushable() << " bytes at " << strMsTimestamp()
          << " with priority " << priority_ << ".\n"
          << logofs_flush;

  *logofs << "Proxy: FLUSH! Current bitrate is "
          << control -> getBitrateInShortFrame() << " with "
          << control -> getBitrateInLongFrame() << " in the "
          << "long frame and top " << control -> getTopBitrate()
          << ".\n" << logofs_flush;
  #endif

  if (control -> CollectStatistics)
  {
    statistics -> addWriteOut();
  }

  int result = transport_ -> flush();

  if (result < 0)
  {
    return -1;
  }

  //
  // Reset the counters and update the
  // timestamp of the last write.
  //

  handleResetFlush();

  return result;
}

//
// Type is ignored for channel flushes.
//

int Proxy::handleFlush(T_flush type, int fd)
{
  #if defined(TEST) || defined(INFO)

  if (type != flush_if_any)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Flush of channel descriptor "
            << "FD#" << fd << " called with invalid type '"
            << type << "'.\n" << logofs_flush;
    #endif

    HandleCleanup();
  }

  #endif

  int channelId = getChannel(fd);

  if (channelId < 0 || channels_[channelId] == NULL)
  {
    #ifdef TEST
    *logofs << "Proxy: WARNING! Skipping flush on invalid "
            << "descriptor FD#" << fd << " channel ID#"
            << channelId << ".\n" << logofs_flush;
    #endif

    return 0;
  }
  else if (channels_[channelId] -> getFinish() == 1)
  {
    #ifdef TEST
    *logofs << "Proxy: Skipping flush on finishing "
            << "descriptor FD#" << fd << " channel ID#"
            << channelId << ".\n" << logofs_flush;
    #endif

    return 0;
  }

  #ifdef TEST
  *logofs << "Proxy: Going to flush FD#" << fd
          << " with blocked " << transports_[channelId] -> blocked()
          << " length " << transports_[channelId] -> length()
          << ".\n" << logofs_flush;
  #endif

  int result = transports_[channelId] -> flush();

  if (result < 0)
  {
    #ifdef TEST
    *logofs << "Proxy: Failed to flush data to FD#"
            << getFd(channelId) << " channel ID#" << channelId
            << ".\n" << logofs_flush;
    #endif

    handleFinish(channelId);

    return -1;
  }

  //
  // Reset channel's transport buffers.
  //

  transports_[channelId] -> partialReset();

  return result;
}

int Proxy::handleStatistics(int type, ostream *stream)
{
  if (control -> CollectStatistics == 0 ||
          statistics == NULL || stream == NULL)
  {
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Cannot produce statistics "
            << " for proxy FD#" << fd_ << ". Invalid settings "
            << "for statistics or stream.\n" << logofs_flush;
    #endif

    return 0;
  }
  else if (statisticsStream_ != NULL)
  {
    //
    // Need to update the stream pointer as the
    // previous one could have been destroyed.
    //
 
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Replacing stream while producing "
            << "statistics in stream at " << statisticsStream_
            << " for proxy FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif
  }

  statisticsStream_ = stream;

  //
  // Get statistics of remote peer.
  //

  if (handleControl(code_statistics_request, type) < 0)
  {
    return -1;
  }

  return 1;
}

int Proxy::handleStatisticsFromProxy(int type)
{
  if (control -> CollectStatistics != 0 &&
          statistics != NULL)
  {
    //
    // Allocate buffer for output.
    //

    char *buffer = new char[STATISTICS_LENGTH];
 
    *buffer = '\0';

    if (control -> ProxyMode == proxy_client)
    {
      #ifdef TEST
      *logofs << "Proxy: Producing "
              << (type == TOTAL_STATS ? "total" : "partial")
              << " client statistics for proxy FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      statistics -> getClientProtocolStats(type, buffer);

      statistics -> getClientOverallStats(type, buffer);
    }
    else
    {
      #ifdef TEST
      *logofs << "Proxy: Producing "
              << (type == TOTAL_STATS ? "total" : "partial")
              << " server statistics for proxy FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      statistics -> getServerProtocolStats(type, buffer);
    }

    if (type == PARTIAL_STATS)
    {
      statistics -> resetPartialStats();
    }

    unsigned int length = strlen((char *) buffer) + 1;

    encodeBuffer_.encodeValue(type, 8);

    encodeBuffer_.encodeValue(length, 32);

    #ifdef TEST
    *logofs << "Proxy: Encoding " << length
            << " bytes of statistics data for proxy FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif

    encodeBuffer_.encodeMemory((unsigned char *) buffer, length);

    //
    // Account statistics data as framing bits.
    //

    if (control -> CollectStatistics)
    {
      statistics -> addFramingBits(length << 3);
    }

    delete [] buffer;
  }
  else
  {
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Got statistics request "
            << "but local statistics are disabled.\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Got statistics request "
         << "but local statistics are disabled.\n";

    type = NO_STATS;

    encodeBuffer_.encodeValue(type, 8);

    #ifdef TEST
    *logofs << "Proxy: Sending error code to remote proxy on FD#"
            << fd_ << ".\n" << logofs_flush;
    #endif
  }

  //
  // The next write will flush the statistics
  // data and the control message.
  //

  if (handleControl(code_statistics_reply, type) < 0)
  {
    return -1;
  }

  return 1;
}

int Proxy::handleStatisticsFromProxy(const unsigned char *message, unsigned int length)
{
  if (statisticsStream_ == NULL)
  {
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Unexpected statistics data received "
            << "from remote proxy on FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Unexpected statistics data received "
         << "from remote proxy.\n";

    return 0;
  }

  //
  // Allocate the decode buffer and at least
  // the 'type' field to see if there was an
  // error.
  //

  DecodeBuffer decodeBuffer(message, length);

  unsigned int type;

  decodeBuffer.decodeValue(type, 8);

  if (type == NO_STATS)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Couldn't get statistics from remote "
            << "proxy on FD#" << fd_ << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Couldn't get statistics from remote proxy.\n";
  }
  else if (type != TOTAL_STATS && type != PARTIAL_STATS)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Cannot produce statistics "
            << "with qualifier '" << type << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot produce statistics "
         << "with qualifier '" << type << "'.\n";

    return -1;
  }
  else
  {
    unsigned int size;

    decodeBuffer.decodeValue(size, 32);

    char *buffer = new char[STATISTICS_LENGTH];

    *buffer = '\0';

    if (control -> CollectStatistics != 0 &&
            statistics != NULL)
    {
      if (control -> ProxyMode == proxy_client)
      {
        #ifdef TEST
        *logofs << "Proxy: Finalizing "
                << (type == TOTAL_STATS ? "total" : "partial")
                << " client statistics for proxy FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        statistics -> getClientCacheStats(type, buffer);

        #ifdef TEST
        *logofs << "Proxy: Decoding " << size
                << " bytes of statistics data for proxy FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        strncat(buffer, (char *) decodeBuffer.decodeMemory(size), size);

        statistics -> getClientProtocolStats(type, buffer);

        statistics -> getClientOverallStats(type, buffer);
      }
      else
      {
        #ifdef TEST
        *logofs << "Proxy: Finalizing "
                << (type == TOTAL_STATS ? "total" : "partial")
                << " server statistics for proxy FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        statistics -> getServerCacheStats(type, buffer);

        statistics -> getServerProtocolStats(type, buffer);

        #ifdef TEST
        *logofs << "Proxy: Decoding " << size
                << " bytes of statistics data for proxy FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        strncat(buffer, (char *) decodeBuffer.decodeMemory(size), size);
      }

      if (type == PARTIAL_STATS)
      {
        statistics -> resetPartialStats();
      }

      *statisticsStream_ << buffer;

      //
      // Mark the end of text to help external parsing.
      //

      *statisticsStream_ << '\4';

      *statisticsStream_ << flush;
    }
    else
    {
      //
      // It can be that statistics were enabled at the time
      // we issued the request (otherwise we could not have
      // set the stream), but now they have been disabled
      // by user. We must decode statistics data if we want
      // to keep the connection.
      //

      #ifdef TEST
      *logofs << "Proxy: Discarding " << size
              << " bytes of statistics data for proxy FD#"
              << fd_ << ".\n" << logofs_flush;
      #endif

      strncat(buffer, (char *) decodeBuffer.decodeMemory(size), size);
    }

    delete [] buffer;
  }

  statisticsStream_ = NULL;

  return 1;
}

int Proxy::handleNegotiation(const unsigned char *message, unsigned int length)
{
  #ifdef PANIC
  *logofs << "Proxy: PANIC! Writing data during proxy "
          << "negotiation is not implemented.\n"
          << logofs_flush;
  #endif

  cerr << "Error" << ": Writing data during proxy "
       << "negotiation is not implemented.\n";

  return -1;
}

int Proxy::handleNegotiationFromProxy(const unsigned char *message, unsigned int length)
{
  #ifdef PANIC
  *logofs << "Proxy: PANIC! Reading data during proxy "
          << "negotiation is not implemented.\n"
          << logofs_flush;
  #endif

  cerr << "Error" << ": Reading data during proxy "
       << "negotiation is not implemented.\n";

  return -1;
}

int Proxy::handleAlert(int alert)
{
  if (alert > ABORT_PROXY_CONNECTION_ALERT &&
          control -> isProtoStep6() == 0)
  {
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Ignoring unsupported alert "
            << "with code '" << alert << "'.\n"
            << logofs_flush;
    #endif

    return 0;
  }

  if (handleControl(code_alert_request, alert) < 0)
  {
    return -1;
  }

  return 1;
}

int Proxy::handleCloseConnection(int clientFd)
{
  #ifdef TEST
  *logofs << "Proxy: Closing down the channel for FD#"
          << clientFd << ".\n" << logofs_flush;
  #endif

  int channelId = getChannel(clientFd);

  if (channels_[channelId] != NULL &&
          channels_[channelId] -> getFinish() == 0)
  {
    shutdown(clientFd, SHUT_RD);

    if (handleFinish(channelId) < 0)
    {
      return -1;
    }

    return 1;
  }

  return 0;
}

int Proxy::handleCloseAllXConnections()
{
  #ifdef TEST
  *logofs << "Proxy: Closing down all the X channels.\n"
          << logofs_flush;
  #endif

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> getType() == channel_x &&
                channels_[channelId] -> getFinish() == 0)
    {
      shutdown(getFd(channelId), SHUT_RD);

      if (handleFinish(channelId) < 0)
      {
        return -1;
      }
    }
  }

  return 1;
}

int Proxy::handleFinish(int channelId)
{
  //
  // Do any finalization needed on channel
  // and send channel shutdown message to
  // the peer proxy.
  //

  if (channels_[channelId] -> getFinish() == 0)
  {
    channels_[channelId] -> handleFinish();

    if (handleControl(code_finish_connection, channelId) < 0)
    {
      return -1;
    }

    return 1;
  }

  return 0;
}

//
// Close the channel and deallocate resources.
//

int Proxy::handleFinishFromProxy(int channelId)
{
  if (channelId < 0 || channelId >= CONNECTIONS_LIMIT ||
          channels_[channelId] == NULL)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Trying to destroy an invalid "
            << "channel id ID#" << channelId << " with FD#"
            << getFd(channelId) << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Trying to destroy an invalid "
         << "channel id ID#" << channelId << ".\n";

    return -1;
  }

  #ifdef TEST
  *logofs << "Proxy: Handling finish from proxy for FD#"
          << getFd(channelId) << " channel ID#" << channelId
          << ".\n" << logofs_flush;
  #endif

  //
  // Get rid of channel.
  //

  char *channelType;

  switch (channels_[channelId] -> getType())
  {
    case channel_cups:
    {
      channelType = "cups";

      break;
    }
    case channel_samba:
    {
      channelType = "samba";

      break;
    }
    case channel_media:
    {
      channelType = "media";

      break;
    }
    case channel_http:
    {
      channelType = "http";

      break;
    }
    default:
    {
      channelType = NULL;

      break;
    }
  }

  if (channelType != NULL)
  {
    #ifdef TEST
    *logofs << "Proxy: Closed connection to "
            << channelType << " server.\n"
            << logofs_flush;
    #endif

    cerr << "Info" << ": Closed connection to "
         << channelType << " server.\n";
  }

  delete channels_[channelId];
  channels_[channelId] = NULL;

  cleanupChannelMap(channelId);

  //
  // Get rid of the transport.
  //

  deallocateTransport(channelId);

  congestions_[channelId] = 0;

  decreaseChannels(channelId);

  return 1;
}

//
// Send an empty message to the remote peer
// just to verify if link is alive and help
// the remote proxy to detect congestion.
//

int Proxy::handlePing()
{
  T_timestamp nowTs = getTimestamp();

  #ifdef DEBUG

  *logofs << "Proxy: Checking ping at "
          << strMsTimestamp(nowTs) << logofs_flush;

  *logofs << " with last loop at "
          << strMsTimestamp(lastLoopTs_) << ".\n"
          << logofs_flush;

  *logofs << "Proxy: Last bytes in at "
          << strMsTimestamp(lastBytesInTs_) << logofs_flush;

  *logofs << " last bytes out at "
          << strMsTimestamp(lastBytesOutTs_) << ".\n"
          << logofs_flush;

  *logofs << "Proxy: Last ping at "
          << strMsTimestamp(lastPingTs_) << ".\n"
          << logofs_flush;

  #endif

  //
  // Be sure we take into account any clock drift. This
  // can be caused by the user changing the system timer
  // or by small adjustments introduced by the operating
  // system making the clock go backward.
  //

  if (checkDiffTimestamp(lastLoopTs_, nowTs) == 0)
  {
    #ifdef WARNING
    *logofs << "Proxy: WARNING! Detected drift in system "
            << "timer. Resetting to current time.\n"
            << logofs_flush;
    #endif

    lastPingTs_ = nowTs;
    lastBytesInTs_ = nowTs;
    lastBytesOutTs_ = nowTs;
  }

  //
  // Check timestamp of last read from remote proxy. It can
  // happen that we stayed in the main loop long enough to
  // have idle timeout expired, for example if proxy was
  // stopped and restarted or because of an extremely high
  // load of the system. If this is the case do not complain
  // if there is something to read from the proxy socket.
  //

  int diffIn = diffTimestamp(lastBytesInTs_, nowTs);

  if (diffIn >= ((control -> PingTimeout * 2) -
          control -> LatencyTimeout) &&
              transport_ -> readable() == 0)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Proxy: Detected congestion with "
            << diffIn / 1000 << " seconds elapsed since "
            << "the last bytes in.\n"  << logofs_flush;
    #endif

    //
    // Switch to proxy congestion state. Tokens are
    // only used at client side. At X server side we
    // must return to read data from channels after
    // a while, because we need to read the key seq-
    // uence CTRL+ALT+SHIFT+ESC.
    //

    if (control -> ProxyMode == proxy_server)
    {
      if (congestion_ == 0)
      {
        congestion_ = 1;
      }
      else
      {
        congestion_ = 0;
      }
    }

    if (control -> ProxyTimeout > 0 &&
            diffIn >= (control -> ProxyTimeout -
                control -> LatencyTimeout))
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! No data received from "
              << "remote proxy on FD#" << fd_ << " within "
              << (diffIn + control -> LatencyTimeout) / 1000
              << " seconds.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": No data received from remote proxy within "
           << (diffIn + control -> LatencyTimeout) / 1000
           << " seconds.\n";

      HandleAbort();
    }
    else
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "Proxy: WARNING! No data received from "
              << "remote proxy on FD#" << fd_ << " since "
              << diffIn << " Ms.\n" << logofs_flush;
      #endif

      if (control -> ProxyTimeout > 0 &&
              isTimestamp(lastAlertTs_) == 0 &&
                  diffIn >= (control -> ProxyTimeout -
                      control -> LatencyTimeout) / 4)
      {
        cerr << "Warning" << ": No data received from remote proxy within "
             << (diffIn + control -> LatencyTimeout) / 1000
             << " seconds.\n";

        if (control -> ProxyMode == proxy_client)
        {
          HandleAlert(CLOSE_DEAD_PROXY_CONNECTION_CLIENT_ALERT, 1);
        }
        else
        {
          HandleAlert(CLOSE_DEAD_PROXY_CONNECTION_SERVER_ALERT, 1);
        }

        lastAlertTs_ = nowTs;
      }
    }
  }

  //
  // Send a new token only if we didn't receive any
  // data from the remote for longer than the ping
  // timeout. Client side sends a ping, the server
  // side must respond with a reply.
  //

  if (control -> ProxyMode == proxy_client)
  {
    int diffOut = diffTimestamp(lastBytesOutTs_, nowTs);

    //
    // We need to send a ping even if we didn't receive
    // anything from the remote within the timeout. The
    // server side doesn't originate the ping so, if no
    // event is to be sent, we would enter in congestion
    // state.
    //

    if (diffIn >= (control -> PingTimeout -
            control -> LatencyTimeout * 5) ||
                diffOut >= (control -> PingTimeout -
                    control -> LatencyTimeout * 5))
    {
      int diffPing = diffTimestamp(lastPingTs_, nowTs);

      if (diffPing >= (control -> PingTimeout -
              control -> LatencyTimeout * 5))
      {
        #if defined(TEST) || defined(INFO)

        if (congestion_ != (tokens_ == 0))
        {
          #ifdef PANIC
          *logofs << "Proxy: PANIC! Congestion is " << congestion_
                  << " but have " << tokens_ << " tokens available.\n"
                  << logofs_flush;
          #endif

          cerr << "Error" << ": Congestion is " << congestion_
               << " but have " << tokens_ << " tokens available.\n";

          HandleCleanup();
        }

        #endif

        if (congestion_ == 0)
        {
          if (handleFrame(frame_ping) < 0)
          {
            return -1;
          }

          lastPingTs_ = nowTs;
        }
        #if defined(TEST) || defined(INFO)
        else
        {
          *logofs << "Proxy: WARNING! Not sending a new ping "
                  << "with " << tokens_ << " tokens available "
                  << "difference in " << diffIn << " out "
                  << diffOut << " ping " << diffPing
                  << ".\n" << logofs_flush;
        }
        #endif
      }
      #if defined(TEST) || defined(INFO)
      else
      {
        *logofs << "Proxy: WARNING! Not sending a new ping "
                << "with difference in " << diffIn << " out "
                << diffOut << " ping " << diffPing
                << ".\n" << logofs_flush;
      }
      #endif
    }
  }

  return 1;
}

int Proxy::handleSync()
{
  //
  // Send the control message to the remote, so that
  // the local proxy will flush all the data. The re-
  // mote will send back the reply (and will flush)
  // as soon as it will get past encoding events with
  // the marked sequence.
  //
  // We check if there is already a synchronization
  // request pending because the request is origina-
  // ted by entering in _XWaitForReadable() and we
  // may enter that function multiple times for the
  // same reply, because of events coming previous
  // of the reply.
  //

  if (notifiedSync_ == -1)
  {
    #if defined(TEST) || defined(INFO) || defined(SYNC)

    if (control -> ProxyMode == proxy_client)
    {
      int fd = Channel::getAgent();

      if (fd >= 0)
      {
        int opcode = ((ClientChannel *) channels_
                          [getChannel(fd)]) -> getRequest();

        int sequence = channels_[getChannel(fd)] -> getClientSequence();

        *logofs << "Proxy: SYNC! Sending a synchronization request "
                << "with OPCODE#" << opcode << " (" << DumpOpcode(opcode)
                << ") and sequence " << sequence << ".\n"
                << logofs_flush;
      }
      else
      {
        *logofs << "Proxy: SYNC! WARNING! Sending a synchronization "
                << "request without an X opcode.\n"
                << logofs_flush;
      }
    }

    #endif

    if (handleControl(code_sync_request) < 0)
    {
      return -1;
    }

    notifiedSync_ = 1;
  }
  #if defined(TEST) || defined(INFO) || defined(SYNC)
  else
  {
    *logofs << "Proxy: SYNC! WARNING! A synchronization "
            << "request is already pending.\n"
            << logofs_flush;
  }
  #endif

  return 1;
}

int Proxy::handleSyncFromProxy()
{
  #if defined(TEST) || defined(INFO) || defined(SYNC)
  *logofs << "Proxy: SYNC! Received a synchronization "
          << "request from the remote proxy.\n"
          << logofs_flush;
  #endif

  int fd = Channel::getAgent();

  if (fd < 0)
  {
    //
    // Actually this makes the first reply
    // from the X server to come later, af-
    // ter the synchronization reply.
    //

    #if defined(TEST) || defined(INFO) || defined(SYNC)
    *logofs << "Proxy: SYNC! WARNING! No agent channel "
            << "available for the synchronization.\n"
            << logofs_flush;
    #endif

    if (handleControl(code_sync_reply) < 0)
    {
      return -1;
    }

    return 1;
  }

  int channelId = getChannel(fd);

  if (channels_[channelId] -> getFinish() == 1)
  {
    #if defined(TEST) || defined(INFO) || defined(SYNC)
    *logofs << "Proxy: SYNC! WARNING! Can't request "
            << "synchronization on finishing "
            << "channel for FD#" << fd << ".\n"
            << logofs_flush;
    #endif

    if (handleControl(code_sync_reply) < 0)
    {
      return -1;
    }

    return 1;
  }

  notifiedSync_ = channels_[channelId] -> getClientSequence();

  #if defined(TEST) || defined(INFO) || defined(SYNC)
  *logofs << "Proxy: SYNC! Gotten sequence " << notifiedSync_
          << " from channel for FD#" << fd << ".\n"
          << logofs_flush;
  #endif

  return 1;
}

int Proxy::handleSync(int channelId)
{
  int fd = Channel::getAgent();

  if (fd < 0)
  {
    #if defined(TEST) || defined(INFO) || defined(SYNC)
    *logofs << "Proxy: SYNC! WARNING! No agent channel "
            << "available for the synchronization.\n"
            << logofs_flush;
    #endif

    if (handleControl(code_sync_reply) < 0)
    {
      return -1;
    }

    return 0;
  }

  if (getChannel(fd) == channelId)
  {
    if (channels_[channelId] -> getFinish() == 0 ||
            shouldSync(channels_[channelId] -> getServerSequence()))
    {
      #if defined(TEST) || defined(INFO) || defined(SYNC)
      *logofs << "Proxy: SYNC! Synchronized sequence "
              << notifiedSync_ << " for FD#" << fd << " with "
              << "current sequence " << channels_[channelId] ->
                 getServerSequence() << ".\n" << logofs_flush;
      #endif

      if (handleControl(code_sync_reply) < 0)
      {
        return -1;
      }

      notifiedSync_ = -1;

      return 1;
    }
  }

  return 0;
}

int Proxy::handleTokenFromProxy()
{
  //
  // We are going to send a new frame to the
  // remote proxy anyway, so flush the motion
  // events waiting for a timeout.
  //

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL &&
            channels_[channelId] -> getFinish() == 0 &&
                channels_[channelId] -> needMotion() == 1)
    {
      #if defined(TEST) || defined(INFO) || defined(FLUSH)
      *logofs << "Proxy: WARNING! Going to send pending motion "
              << "events for FD#" << getFd(channelId) << ".\n"
              << logofs_flush;
      #endif

      if (channels_[channelId] -> handleMotion(encodeBuffer_, 1) > 0)
      {
        if (handleSwitch(channelId) < 0)
        {
          return -1;
        }
        else if (handleFrame(frame_data) < 0)
        {
          return -1;
        }
      }
      else
      {
        #ifdef TEST
        *logofs << "Proxy: Failed to handle motion events for FD#"
                << getFd(channelId) << " channel ID#" << channelId
                << ".\n" << logofs_flush;
        #endif

        encodeBuffer_.fullReset();

        if (handleFinish(channelId) < 0)
        {
          return -1;
        }
      }
    }
  }

  //
  // Add our token reply.
  //

  if (handleControl(code_token_reply) < 0)
  {
    return -1;
  }

  return 1;
}

void Proxy::handleResetFlush()
{
  #ifdef TEST
  *logofs << "Proxy: Going to reset flush counters "
          << "for proxy FD#" << fd_ << ".\n"
          << logofs_flush;
  #endif

  //
  // Reset timers to the next flush.
  //

  lastFrameTs_ = nullTimestamp();
  lastFlushTs_ = nullTimestamp();

  #if defined(TEST) || defined(INFO) || defined(FLUSH)
  *logofs << "Proxy: FLUSH! Reset last frame and flush "
          << "timestamps at " << strMsTimestamp()
          << ".\n" << logofs_flush;
  #endif

  initialTimeout(timeouts_.frame, control -> FrameTimeout);
  initialTimeout(timeouts_.flush, control -> FlushTimeout);
  initialTimeout(timeouts_.split, control -> SplitTimeout);

  #if defined(TEST) || defined(INFO) || defined(FLUSH)
  *logofs << "Proxy: FLUSH! Initial timeouts are frame "
          << timeouts_.frame << " flush " << timeouts_.flush
          << " split " << timeouts_.split << ".\n"
          << logofs_flush;
  #endif

  //
  // Restore buffers to their initial
  // size.
  //

  transport_ -> partialReset();

  //
  // Reset the counters used to track
  // the number of frames and the
  // bytes to be written.
  //

  deferredFrames_ = 0;
  deferredBytes_  = 0;

  //
  // Reset the proxy priority flag.
  //

  priority_ = 0;

  //
  // Update the timestamp of the last
  // write operation performed on the
  // socket.
  //

  lastBytesOutTs_ = getTimestamp();
}

int Proxy::handleReset()
{
  #ifdef TEST
  *logofs << "Proxy: Resetting proxy for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  //
  // Need to reset:
  //
  // 1. Channels, including sending an X error
  //    to clients for any pending reply and
  //    restarting agent's clients stopped
  //    because of splits.
  //
  // 2. Internal state, resetting congestion
  //    state for all channels.
  //
  // 3. Message stores, including their own
  //    encode caches.
  //
  // 4. Channel caches, only in protocol 3
  //    or greater.
  //
  // 5. Split stores, including encode caches.
  //
  // 6. Channels' encode caches.
  //
  // 7. Proxy ZLIB streams in any of compressor,
  //    decompressor and transport.
  //
  // 8. Any unreliable data accumulated in
  //    proxy read buffer.
  //
  // 9. If client proxy, tell finally to remote
  //    proxy to load message stores from last
  //    negotiated persistent cache.
  //
  // Stores are recreated as soon as reset message
  // is received from remote proxy.
  //

  //
  // Reset the timer.
  //

  resetTimer();

  //
  // Reset channels.
  //

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      if (channels_[channelId] -> handleReset() < 0)
      {
        #ifdef PANIC
        *logofs << "Proxy: PANIC! Failed to reset channel for FD#"
                << getFd(channelId) << ".\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Failed to reset channel for FD#"
             << getFd(channelId) << ".\n";
      }
    }
  }

  //
  // Reset ZLIB structures in compressor
  // and transport and reset proxy read
  // buffer.
  //

  compressor_ -> fullReset();
  decompressor_ -> fullReset();

  transport_ -> fullReset();

  readBuffer_.fullReset();

  encodeBuffer_.fullReset();

  //
  // Reset internal state.
  //

  controlLength_ = 0;

  operation_ = operation_in_negotiation;

  pending_    = 0;
  priority_   = 0;
  shutdown_   = 0;
  reset_      = 1;
  congestion_ = 0;
  timer_      = 0;
  tokens_     = 0;

  timeouts_.frame = 0;
  timeouts_.flush = 0;
  timeouts_.split = 0;

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    congestions_[channelId] = 0;
  }

  lastBytesInTs_  = getTimestamp();
  lastBytesOutTs_ = getTimestamp();

  lastLoopTs_  = getTimestamp();
  lastPingTs_  = getTimestamp();

  lastAlertTs_ = nullTimestamp();
  lastSplitTs_ = nullTimestamp();

  statisticsStream_ = NULL;

  //
  // Reset message stores and tell remote
  // proxy to reload content from disk.
  //

  if (handleResetOpcodes() < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Failed to reset opcodes store.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  if (handleResetStores() < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Failed to reset message stores.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  if (handleResetCaches() < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Failed to reset channel caches.\n"
            << logofs_flush;
    #endif

    return -1;
  }

  //
  // Force load of last negotiated persistent cache
  // if one or more channels already exist. It will
  // actually produce a load request if we are at
  // client side.
  //

  if (handleCheckLoad(load_if_any) < 0)
  {
    return -1;
  }

  if (handleControl(code_reset_request) < 0)
  {
    return -1;
  }

  //
  // Void the sync request.
  //

  notifiedSync_ = -1;

  //
  // Force a decongestion notification
  // when the proxy becomes writable.
  //

  notifiedCongestion_ = 1;

  //
  // Reset the flush counters.
  //

  notifiedBytes_ = 0;

  deferredFrames_ = 0;
  deferredBytes_  = 0;

  #ifdef TEST
  *logofs << "Proxy: Completed reset of proxy for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // Write any data produced for the remote
  // proxy, including any load request and
  // the reset message.
  //

  if (handleFrame(frame_data) < 0)
  {
    return -1;
  }

  return 1;
}


int Proxy::handleResetPersistentCache()
{
  char *fullName = new char[strlen(control -> PersistentCachePath) +
                                strlen(control -> PersistentCacheName) + 2];

  strcpy(fullName, control -> PersistentCachePath);
  strcat(fullName, "/");
  strcat(fullName, control -> PersistentCacheName);

  #ifdef TEST
  *logofs << "Proxy: Going to remove persistent cache file '"
          << fullName << "'\n" << logofs_flush;
  #endif

  remove(fullName);

  delete [] fullName;

  delete [] control -> PersistentCacheName;

  control -> PersistentCacheName = NULL;

  return 1;
}

int Proxy::handleResetOpcodes()
{
  //
  // Nothing to be done. Opcodes are
  // not changing given that we remain
  // connected to the same X server.
  //

  return 1;
}

int Proxy::handleResetStores()
{
  //
  // Recreate the message stores.
  //

  delete clientStore_;
  delete serverStore_;

  clientStore_ = new ClientStore(compressor_, decompressor_);
  serverStore_ = new ServerStore(compressor_, decompressor_);

  lastLoadTs_ = nullTimestamp();

  //
  // Replace message stores in channels.
  //

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      if (channels_[channelId] -> setStores(clientStore_, serverStore_) < 0)
      {
        #ifdef PANIC
        *logofs << "Proxy: PANIC! Failed to replace message stores in "
                << "channel for FD#" << getFd(channelId) << ".\n"
                << logofs_flush;
        #endif

        cerr << "Error" << ": Failed to replace message stores in "
             << "channel for FD#" << getFd(channelId) << ".\n";

        return -1;
      }
      #ifdef TEST
      else
      {
        *logofs << "Proxy: Replaced message stores in channel "
                << "for FD#" << getFd(channelId) << ".\n"
                << logofs_flush;
      }
      #endif
    }
  }

  return 1;
}

int Proxy::handleResetCaches()
{
  //
  // If protocol is 3 or newer then recreate the
  // channel caches. In older protocol versions
  // caches are not shared and they are reset in
  // the channel specific method.
  //

  if (control -> isProtoStep3() == 1)
  {
    delete clientCache_;
    delete serverCache_;

    clientCache_ = new ClientCache();
    serverCache_ = new ServerCache();

    if (clientCache_ == NULL || serverCache_ == NULL)
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! Failed to create channel's caches.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Failed to create channel's caches.\n";

      HandleCleanup();
    }

    //
    // Replace channel caches in channels.
    //

    for (int channelId = lowerChannel_;
             channelId <= upperChannel_;
                 channelId++)
    {
      if (channels_[channelId] != NULL)
      {
        if (channels_[channelId] -> setCaches(clientCache_, serverCache_) < 0)
        {
          #ifdef PANIC
          *logofs << "Proxy: PANIC! Failed to replace channel caches in "
                  << "channel for FD#" << getFd(channelId) << ".\n"
                  << logofs_flush;
          #endif

          cerr << "Error" << ": Failed to replace channel caches in "
               << "channel for FD#" << getFd(channelId) << ".\n";

          return -1;
        }
        #ifdef TEST
        else
        {
          *logofs << "Proxy: Replaced encode caches in channel "
                  << "for FD#" << getFd(channelId) << ".\n"
                  << logofs_flush;
        }
        #endif
      }
    }
  }

  return 1;
}

int Proxy::handleShutdown()
{
  //
  // Send shutdown message to remote proxy.
  //

  shutdown_ = 1;

  handleControl(code_shutdown_request);

  //
  // Ensure data is flushed.
  //

  for (int i = 0; i < 10; i++)
  {
    if (canFlush() == 1)
    {
      handleFlush(flush_if_any);
    }

    if (transport_ -> queued() <= 0)
    {
      break;
    }

    usleep(200000);
  }

  //
  // Give time to the remote end
  // to read the shutdown message.
  //

  for (int i = 0; i < 10; i++)
  {
    if (transport_ -> readable() < 0)
    {
      break;
    }

    usleep(200000);
  }

  return 1;
}

int Proxy::handleSocketConfiguration()
{
  //
  // Set linger mode on proxy to correctly
  // get shutdown notification.
  //

  SetLingerTimeout(fd_, 30);

  //
  // Set keep-alive on socket so that if remote link
  // terminates abnormally (as killed hard or because
  // of a power-off) process will get a SIGPIPE. In
  // practice this is useless as proxies already ping
  // each other every few seconds.
  //

  if (control -> OptionProxyKeepAlive == 1)
  {
    SetKeepAlive(fd_);
  }

  //
  // Set 'priority' flag at TCP layer for path
  // proxy-to-proxy. Look at IPTOS_LOWDELAY in
  // man 7 ip.
  //

  if (control -> OptionProxyLowDelay == 1)
  {
    SetLowDelay(fd_);
  }

  //
  // Update size of TCP send and receive buffers.
  //

  if (control -> OptionProxySendBuffer != -1)
  {
    SetSendBuffer(fd_, control -> OptionProxySendBuffer);
  }

  if (control -> OptionProxyReceiveBuffer != -1)
  {
    SetReceiveBuffer(fd_, control -> OptionProxyReceiveBuffer);
  }

  //
  // Update TCP_NODELAY settings. Note that on old Linux
  // kernels turning off the Nagle algorithm didn't work
  // when proxy was run through a PPP link. Trying to do
  // so caused the kernel to stop delivering data to us
  // if a serious network congestion was encountered.
  //

  if (control -> ProxyMode == proxy_client)
  {
    if (control -> OptionProxyClientNoDelay != -1)
    {
      SetNoDelay(fd_, control -> OptionProxyClientNoDelay);
    }
  }
  else
  {
    if (control -> OptionProxyServerNoDelay != -1)
    {
      SetNoDelay(fd_, control -> OptionProxyServerNoDelay);
    }
  }

  return 1;
}

int Proxy::handleLinkConfiguration()
{
  #ifdef TEST
  *logofs << "Proxy: Propagating parameters to "
          << "channels' read buffers.\n"
          << logofs_flush;
  #endif

  for (int channelId = lowerChannel_;
           channelId <= upperChannel_;
               channelId++)
  {
    if (channels_[channelId] != NULL)
    {
      channels_[channelId] -> handleConfiguration();
    }
  }

  #ifdef TEST
  *logofs << "Proxy: Propagating parameters to "
          << "proxy buffers.\n"
          << logofs_flush;
  #endif

  readBuffer_.setSize(control -> ProxyInitialReadSize,
                          control -> ProxyMaximumReadSize);

  encodeBuffer_.setSize(control -> TransportProxyBufferSize,
                            control -> TransportProxyBufferLimit,
                                control -> TransportMaximumBufferSize);

  transport_ -> setSize(control -> TransportProxyBufferSize,
                            control -> TransportProxyBufferLimit,
                                control -> TransportMaximumBufferSize);

  #ifdef TEST
  *logofs << "Proxy: Setting the available tokens to "
          << control -> TokenLimit << ".\n"
          << logofs_flush;
  #endif

  tokens_ = control -> TokenLimit;

  return 1;
}

int Proxy::handleCacheConfiguration()
{
  #ifdef TEST
  *logofs << "Proxy: Configuring cache according to pack parameters.\n"
          << logofs_flush;
  #endif

  //
  // Further adjust cache parameters. If packing
  // of images is enabled, reduce size available
  // for plain images. At the moment this doesn't
  // apply do normal X applications as this has
  // not been built yet inside Xlib.
  //

  if (control -> SessionMode != session_application)
  {
    if (control -> AgentPackMethod != NO_PACK)
    {
      clientStore_ -> getRequestStore(X_PutImage) ->
          cacheThreshold = PUTIMAGE_CACHE_THRESHOLD_IF_PACKED;

      clientStore_ -> getRequestStore(X_PutImage) ->
          cacheLowerThreshold = PUTIMAGE_CACHE_LOWER_THRESHOLD_IF_PACKED;
    }
  }

  //
  // If this is a windows session (i.e. using any of
  // RDP pack methods), increase size of X_ChangeGC
  // and X_PolyFillRectangle cache.
  //

  if (control -> SessionMode == session_rdp)
  {
    clientStore_ -> getRequestStore(X_ChangeGC) ->
        cacheThreshold = CHANGEGC_CACHE_THRESHOLD_IF_PACKED_RDP;

    clientStore_ -> getRequestStore(X_ChangeGC) ->
        cacheLowerThreshold = CHANGEGC_CACHE_LOWER_THRESHOLD_IF_PACKED_RDP;

    clientStore_ -> getRequestStore(X_PolyFillRectangle) ->
        cacheThreshold = POLYFILLRECTANGLE_CACHE_THRESHOLD_IF_PACKED_RDP;

    clientStore_ -> getRequestStore(X_PolyFillRectangle) ->
        cacheLowerThreshold = POLYFILLRECTANGLE_CACHE_LOWER_THRESHOLD_IF_PACKED_RDP;

    clientStore_ -> getRequestStore(X_NXPutPackedImage) ->
        cacheThreshold = PUTPACKEDIMAGE_CACHE_THRESHOLD_IF_PACKED_RDP;

    clientStore_ -> getRequestStore(X_NXPutPackedImage) ->
        cacheLowerThreshold = PUTPACKEDIMAGE_CACHE_LOWER_THRESHOLD_IF_PACKED_RDP;
  }

  return 1;
}

int Proxy::handleSave()
{
  //
  // Save content of stores on disk.
  //

  char *cacheToAdopt = NULL;

  if (control -> PersistentCacheEnableSave)
  {
    #ifdef TEST
    *logofs << "Proxy: Going to save content of client store.\n"
            << logofs_flush;
    #endif

    cacheToAdopt = handleSaveStores(control -> PersistentCachePath);
  }
  #ifdef TEST
  else
  {
    if (control -> ProxyMode == proxy_client)
    {
      *logofs << "Proxy: Saving persistent cache to disk disabled.\n"
              << logofs_flush;
    }
    else
    {
      *logofs << "Proxy: PANIC! Protocol violation in command save.\n"
              << logofs_flush;

      cerr << "Error" << ": Protocol violation in command save.\n";

      HandleCleanup();
    }
  }
  #endif

  if (cacheToAdopt != NULL)
  {
    //
    // Do we have a cache already?
    //

    if (control -> PersistentCacheName != NULL)
    {
      //
      // Check if old and new cache are the same.
      // In this case don't remove the old cache.
      //

      if (strcasecmp(control -> PersistentCacheName, cacheToAdopt) != 0)
      {
        handleResetPersistentCache();
      }

      delete [] control -> PersistentCacheName;
    }

    #ifdef TEST
    *logofs << "Proxy: Setting current persistent cache file to '"
            << cacheToAdopt << "'\n" << logofs_flush;
    #endif

    control -> PersistentCacheName = cacheToAdopt;

    return 1;
  }
  #ifdef TEST
  else
  {
    *logofs << "Proxy: No cache file produced from message stores.\n"
            << logofs_flush;
  }
  #endif

  //
  // It can be that we didn't generate a new cache
  // because store was too small or persistent cache
  // was disabled. This is not an error.
  //

  return 0;
}

int Proxy::handleLoad()
{
  //
  // Restore content of client store from disk
  // if any valid cache was negotiated between
  // proxies.
  //

  if (control -> PersistentCacheEnableLoad == 1 &&
          control -> PersistentCachePath != NULL &&
              control -> PersistentCacheName != NULL)
  {
    #ifdef TEST
    *logofs << "Proxy: Going to load content of client store.\n"
            << logofs_flush;
    #endif

    //
    // Returns the same string passed as name of
    // the cache, or NULL if it was not possible
    // to load the cache from disk.
    //

    if (handleLoadStores(control -> PersistentCachePath,
                             control -> PersistentCacheName) == NULL)
    {
      //
      // Corrupted cache should have been removed
      // from disk. Get rid of reference to it so
      // we don't try to load it again in case of
      // proxy reset.
      //

      if (control -> PersistentCacheName != NULL)
      {
        delete [] control -> PersistentCacheName;
      }

      control -> PersistentCacheName = NULL;
 
      return -1;
    }

    //
    // Set timestamp of last time cache
    // was loaded from data on disk.
    //

    lastLoadTs_ = getTimestamp();

    return 1;
  }
  #ifdef TEST
  else
  {
    if (control -> ProxyMode == proxy_client)
    {
      *logofs << "Proxy: Loading of cache disabled or no cache file selected.\n"
              << logofs_flush;
    }
    else
    {
      *logofs << "Proxy: PANIC! Protocol violation in command load.\n"
              << logofs_flush;

      cerr << "Error" << ": Protocol violation in command load.\n";

      HandleCleanup();
    }
  }
  #endif

  return 0;
}

int Proxy::handleControl(T_proxy_code code, int data)
{
  //
  // Send the given control messages
  // to the remote proxy.
  //

  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Sending message '" << CodeLabel(code)
          << "' at " << strMsTimestamp() << " with data ID#"
          << data << ".\n" << logofs_flush;
  #endif

  if (controlLength_ + 3 >= CONTROL_CODES_LENGTH)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Overflow in control message length "
            << "while sending code '" << code << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Overflow in control message length "
         << "while sending code '" << code << "'.\n";

    HandleCleanup();
  }

  controlCodes_[controlLength_++] = 0;
  controlCodes_[controlLength_++] = (unsigned char) code;
  controlCodes_[controlLength_++] = (unsigned char) (data == -1 ? 0 : data);

  //
  // Account for the control frame.
  //

  if (controlLength_ > 0)
  {
    statistics -> addFrameOut();
  }

  //
  // Flush the control message immediately.
  //

  priority_ = 1;

  if (handleFrame(frame_data) < 0)
  {
    return -1;
  }

  return 1;
}

int Proxy::handleSwitch(int channelId)
{
  //
  // If data is for a different channel than last
  // selected for output, prepend to data the new
  // channel id.
  //

  if (channelId != outputChannel_)
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Proxy: Sending message '"
            << CodeLabel(code_switch_connection) << "' at "
            << strMsTimestamp() << " with FD#" << getFd(channelId)
            << " channel ID#" << channelId << ".\n"
            << logofs_flush;
    #endif

    if (controlLength_ + 3 >= CONTROL_CODES_LENGTH)
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! Overflow in control message length "
              << "while switching channel.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Overflow in control message length "
           << "while switching channel.\n";

      HandleCleanup();
    }

    outputChannel_ = channelId;

    controlCodes_[controlLength_++] = 0;
    controlCodes_[controlLength_++] = (unsigned char) code_switch_connection;
    controlCodes_[controlLength_++] = (unsigned char) channelId;

    //
    // Account for the control frame.
    //

    if (controlLength_ > 0)
    {
      statistics -> addFrameOut();
    }
  }

  return 1;
}

int Proxy::handleToken(T_frame_type type)
{
  #if defined(TEST) || defined(INFO)
  *logofs << "Proxy: Sending message '"
          << CodeLabel(code_token_request) << "' at "
          << strMsTimestamp() << " with " << tokens_ - 1
          << " remaining.\n" << logofs_flush;
  #endif

  #if defined(TEST) || defined(INFO)

  if (controlLength_ + 3 >= CONTROL_CODES_LENGTH)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Overflow in control message length "
            << "while handling the token.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Overflow in control message length "
         << "while handling the token.\n";

    HandleCleanup();
  }

  #endif
  
  controlCodes_[controlLength_++] = 0;
  controlCodes_[controlLength_++] = code_token_request;
  controlCodes_[controlLength_++] = 0;

  //
  // Account for the control frame.
  //

  if (controlLength_ > 0)
  {
    statistics -> addFrameOut();
  }

  tokens_--;

  if (tokens_ < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Token underflow handling messages.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Token underflow handling messages.\n";

    HandleCleanup();
  }
  else if (tokens_ == 0)
  {
    //
    // Enter in congestion state.
    //

    congestion_ = 1;

    priority_ = 1;
  }

  //
  // Set the priority flag only if the frame is
  // a ping.
  //
  // An alternative implementation may require
  // that we write the data as soon as we have
  // accumulated enough for a token. This has
  // been evaluated, but it seems to lower both
  // the throughput and the compression ratio.
  //

  if (type == frame_ping)
  {
    priority_ = 1;
  }

  return 1;
}

void Proxy::handleFailOnSave(const char *fullName, const char *failContext) const
{
  #ifdef PANIC
  *logofs << "Proxy: PANIC! Error saving stores to cache file "
          << "in context [" << failContext << "].\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Error saving stores to cache file "
       << "in context [" << failContext << "].\n";

  #ifdef PANIC
  *logofs << "Proxy: PANIC! Removing corrupted cache '"
          << fullName << "'.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Removing corrupted cache '"
       << fullName << "'.\n";

  remove(fullName);
}

void Proxy::handleFailOnLoad(const char *fullName, const char *failContext) const
{
  #ifdef PANIC
  *logofs << "Proxy: PANIC! Error loading stores from cache file "
          << "in context [" << failContext << "].\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Error loading stores from cache file "
       << "in context [" << failContext << "].\n";

  #ifdef PANIC
  *logofs << "Proxy: PANIC! Removing corrupted cache '"
          << fullName << "'.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Removing corrupted cache '"
       << fullName << "'.\n";

  remove(fullName);
}

int Proxy::handleSaveVersion(unsigned char *buffer, int &major,
                                 int &minor, int &patch) const
{
  if (control -> isProtoStep4() == 1)
  {
    if (control -> isProtoStep5() == 1)
    {
      //
      // Identify the caches as the 1.4.0 version
      // so that they are not discarded when the
      // software is upgraded.
      //
      // major = control -> LocalVersionMajor;
      // minor = control -> LocalVersionMinor;
      // patch = control -> LocalVersionPatch;
      //

      major = 1;
      minor = 4;
      patch = 0;
    }
    else
    {
      //
      // Save cache in a format compatible
      // with version 1.3.2.
      //

      major = 1;
      minor = 3;
      patch = 2;
    }

    *(buffer + 0) = major;
    *(buffer + 1) = minor;

    PutUINT(patch, buffer + 2, storeBigEndian());
  }
  else if (control -> isProtoStep3() == 1)
  {
    //
    // We'll save cache in a compatible
    // format.
    //

    major = 1;
    minor = 2;
    patch = 2;

    *(buffer + 0) = major;
    *(buffer + 1) = minor;

    PutUINT(patch, buffer + 2, storeBigEndian());
  }
  else
  {
    major = control -> LocalVersionMajor;

    minor = 0;
    patch = 0;

    *((int *) buffer) = major;
  }

  return 1;
}

int Proxy::handleLoadVersion(const unsigned char *buffer, int &major,
                                 int &minor, int &patch) const
{
  if (control -> isProtoStep3() == 1)
  {
    major = *(buffer + 0);
    minor = *(buffer + 1);

    patch = GetUINT(buffer + 2, storeBigEndian());
  }
  else
  {
    major = *((int *) buffer);

    minor = 0;
    patch = 0;
  }

  if (major != control -> LocalVersionMajor)
  {
    return -1;
  }

  //
  // Set the cache version. Will manage to
  // discard the RENDER extension if cache
  // was saved by a version less than 1.4.0
  // and we are using protocol step 5.
  //

  if (minor < 4)
  {
    control -> setCacheStep1();
  }
  else
  {
    control -> setCacheStep2();
  }

  if (control -> isProtoStep3() == 1)
  {
    //
    // Refuse to load caches with a minor version
    // greater than local, while different patch
    // versions are assumed to be compatible.
    //

    if (minor > control -> LocalVersionMinor)
    {
      return -1;
    }

    //
    // Caches from 1.2.1 are not compatible while
    // caches from version 1.2.2 are managed in
    // a compatible way...
    //

    if (minor == 2 && patch <= 1)
    {
      return -1;
    }

    //
    // ...But not if both sides use protocol 4.
    //

    if (control -> isProtoStep4() == 1)
    {
      if (minor == 2 && patch == 2)
      {
        return -1;
      }
    }
  }

  return 1;
}

char *Proxy::handleSaveStores(const char *savePath) const
{
  int cumulativeSize = MessageStore::getCumulativeTotalStorageSize();

  if (cumulativeSize < control -> PersistentCacheThreshold)
  {
    #ifdef TEST
    *logofs << "Proxy: Cache not saved as size is "
            << cumulativeSize << " with threshold set to "
            << control -> PersistentCacheThreshold
            << ".\n" << logofs_flush;
    #endif

    return NULL;
  }
  else if (savePath == NULL)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! No name provided for save path.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": No name provided for save path.\n";

    return NULL;
  }

  #ifdef TEST
  *logofs << "Proxy: Going to save content of message stores.\n"
          << logofs_flush;
  #endif

  //
  // Our parent process is likely going to terminate.
  // Until we finish saving cache we must ignore its
  // SIGIPE.
  //

  DisableSignals();

  ofstream *cachefs = NULL;

  md5_state_t *md5StateStream = NULL;
  md5_byte_t  *md5DigestStream = NULL;

  md5_state_t *md5StateClient = NULL;
  md5_byte_t  *md5DigestClient = NULL;

  char *tempName = NULL;

  char md5_string[MD5_LENGTH * 2 + 2];

  char fullName[strlen(savePath) + MD5_LENGTH * 2 + 4];

  if (control -> ProxyMode == proxy_client)
  {
    tempName = tempnam(savePath, "Z-C-");
  }
  else
  {
    tempName = tempnam(savePath, "Z-S-");
  }

  cachefs = new ofstream(tempName, ios::out | ios::binary);

  if (tempName == NULL || cachefs == NULL)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Can't create temporary file in '"
            << savePath << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Can't create temporary file in '"
         << savePath << "'.\n";

    EnableSignals();

    if (tempName != NULL)
    {
      free(tempName);
    }

    if (cachefs != NULL)
    {
      delete cachefs;
    }

    return NULL;
  }

  md5StateStream  = new md5_state_t();
  md5DigestStream = new md5_byte_t[MD5_LENGTH];

  md5_init(md5StateStream);

  //
  // First write nxproxy version. In protocol
  // level 3 we write all version information
  // not just the major number.
  //

  unsigned char version[4];

  int major;
  int minor;
  int patch;

  handleSaveVersion(version, major, minor, patch);

  #ifdef TEST
  *logofs << "Proxy: Saving cache using version '"
          << major << "." << minor << "." << patch
          << "'.\n" << logofs_flush;
  #endif

  if (PutData(cachefs, version, 4) < 0)
  {
    handleFailOnSave(tempName, "A");

    delete cachefs;

    delete md5StateStream;
    delete [] md5DigestStream;

    EnableSignals();

    return NULL;
  }

  //
  // Don't include version in checksum. This
  // will allow proxy to load caches produced
  // by a different version.
  //

  if (control -> isProtoStep3() == 0)
  {
    md5_append(md5StateStream, version, 4);
  }

  //
  // Make space for the calculated MD5 so we
  // can later rewind the file and write it
  // at this position.
  //

  if (PutData(cachefs, md5DigestStream, MD5_LENGTH) < 0)
  {
    handleFailOnSave(tempName, "B");

    delete cachefs;

    delete md5StateStream;
    delete [] md5DigestStream;

    EnableSignals();

    return NULL;
  }

  md5StateClient  = new md5_state_t();
  md5DigestClient = new md5_byte_t[MD5_LENGTH];

  md5_init(md5StateClient);

  #ifdef DUMP

  ofstream *cache_dump = NULL;

  ofstream *tempfs = (ofstream*) logofs;

  if (control -> ProxyMode == proxy_client)
  {
    cache_dump = new ofstream("/tmp/nxproxy-client-cache.dump", ios::out);
  }
  else
  {
    cache_dump = new ofstream("/tmp/nxproxy-server-cache.dump", ios::out);
  }

  logofs = cache_dump;

  #endif

  //
  // Use the virtual method of the concrete proxy class.
  //

  int allSaved = handleSaveStores(cachefs, md5StateStream, md5StateClient);

  #ifdef DUMP

  logofs = tempfs;

  delete cache_dump;

  #endif

  if (allSaved == 0)
  {
    handleFailOnSave(tempName, "C");

    delete cachefs;

    delete md5StateStream;
    delete [] md5DigestStream;

    delete md5StateClient;
    delete [] md5DigestClient;

    EnableSignals();

    return NULL;
  }

  md5_finish(md5StateClient, md5DigestClient);

  for (unsigned int i = 0; i < MD5_LENGTH; i++)
  {
    sprintf(md5_string + (i * 2), "%02X", md5DigestClient[i]);
  }

  strcpy(fullName, (control -> ProxyMode == proxy_client) ? "C-" : "S-");

  strcat(fullName, md5_string);

  md5_append(md5StateStream, (const md5_byte_t *) fullName, strlen(fullName));
  md5_finish(md5StateStream, md5DigestStream);

  //
  // Go to the beginning of file plus
  // the integer where we wrote our
  // proxy version.
  //

  cachefs -> seekp(4);

  if (PutData(cachefs, md5DigestStream, MD5_LENGTH) < 0)
  {
    handleFailOnSave(tempName, "D");

    delete cachefs;

    delete md5StateStream;
    delete [] md5DigestStream;

    delete md5StateClient;
    delete [] md5DigestClient;

    EnableSignals();

    return NULL;
  }

  delete cachefs;

  //
  // Save finally the resulting cache name
  // without the path.
  //

  char *cacheName = new char[MD5_LENGTH * 2 + 4];

  strcpy(cacheName, fullName);

  strcpy(fullName, savePath);
  strcat(fullName, (control -> ProxyMode == proxy_client) ? "/C-" : "/S-");
  strcat(fullName, md5_string);

  #ifdef TEST
  *logofs << "Proxy: Renaming cache file to "
          << fullName << "'.\n" << logofs_flush;
  #endif

  rename(tempName, fullName);

  delete md5StateStream;
  delete [] md5DigestStream;

  delete md5StateClient;
  delete [] md5DigestClient;

  free(tempName);

  //
  // Restore the original handlers.
  //

  EnableSignals();

  #ifdef TEST
  *logofs << "Proxy: Successfully saved cache file '"
          << cacheName << "'.\n" << logofs_flush;
  #endif

  return cacheName;
}

const char *Proxy::handleLoadStores(const char *loadPath, const char *loadName) const
{
  #ifdef TEST
  *logofs << "Proxy: Going to load content of message stores.\n"
          << logofs_flush;
  #endif

  //
  // Until we finish loading cache we
  // must at least ignore any SIGIPE.
  //

  DisableSignals();

  if (loadPath == NULL || loadName == NULL)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! No path or no file name provided for cache to restore.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": No path or no file name provided for cache to restore.\n";

    EnableSignals();

    return NULL;
  }
  else if (strlen(loadName) != MD5_LENGTH * 2 + 2)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Bad file name provided for cache to restore.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Bad file name provided for cache to restore.\n";

    EnableSignals();

    return NULL;
  }

  istream *cachefs = NULL;
  char md5_string[(MD5_LENGTH * 2) + 2];
  md5_byte_t md5_from_file[MD5_LENGTH];

  char *cacheName = new char[strlen(loadPath) + strlen(loadName) + 3];

  strcpy(cacheName, loadPath);
  strcat(cacheName, "/");
  strcat(cacheName, loadName);

  #ifdef TEST
  *logofs << "Proxy: Name of cache file is '"
          << cacheName << "'.\n" << logofs_flush;
  #endif

  cachefs = new ifstream(cacheName, ios::in | ios::binary);

  unsigned char version[4];

  if (cachefs == NULL || GetData(cachefs, version, 4) < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Can't read cache file '"
            << cacheName << "'.\n" << logofs_flush;;
    #endif

    handleFailOnLoad(cacheName, "A");

    delete cachefs;

    delete [] cacheName;

    EnableSignals();

    return NULL;
  }

  int major;
  int minor;
  int patch;

  if (handleLoadVersion(version, major, minor, patch) < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Incompatible version '"
            << major << "." << minor << "." << patch
            << "' in cache file '" << cacheName
            << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Incompatible version '"
         << major << "." << minor << "." << patch
         << "' in cache file '" << cacheName
         << "'.\n" << logofs_flush;

    handleFailOnLoad(cacheName, "B");

    delete cachefs;

    delete [] cacheName;

    EnableSignals();

    return NULL;
  }

  #ifdef TEST
  *logofs << "Proxy: Reading from cache file version '"
          << major << "." << minor << "." << patch
          << "'.\n" << logofs_flush;
  #endif

  if (GetData(cachefs, md5_from_file, MD5_LENGTH) < 0)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! No checksum in cache file '"
            << loadName << "'.\n" << logofs_flush;
    #endif

    handleFailOnLoad(cacheName, "C");

    delete cachefs;

    delete [] cacheName;

    EnableSignals();

    return NULL;
  }

  md5_state_t *md5StateStream = NULL;
  md5_byte_t  *md5DigestStream = NULL;

  md5StateStream  = new md5_state_t();
  md5DigestStream = new md5_byte_t[MD5_LENGTH];

  md5_init(md5StateStream);

  //
  // Don't include version in checksum.
  //

  if (control -> isProtoStep3() == 0)
  {
    md5_append(md5StateStream, version, 4);
  }

  //
  // Use the virtual method of the concrete proxy class.
  //

  if (handleLoadStores(cachefs, md5StateStream) < 0)
  {
    handleFailOnLoad(cacheName, "D");

    delete cachefs;

    delete md5StateStream;
    delete [] md5DigestStream;

    delete [] cacheName;

    EnableSignals();

    return NULL;
  }

  md5_append(md5StateStream, (const md5_byte_t *) loadName, strlen(loadName));
  md5_finish(md5StateStream, md5DigestStream);

  for (int i = 0; i < MD5_LENGTH; i++)
  {
    if (md5DigestStream[i] != md5_from_file[i])
    {
      #ifdef PANIC

      *logofs << "Proxy: PANIC! Bad checksum for cache file '"
              << cacheName << "'.\n" << logofs_flush;

      for (unsigned int i = 0; i < MD5_LENGTH; i++)
      {
        sprintf(md5_string + (i * 2), "%02X", md5_from_file[i]);
      }

      *logofs << "Proxy: PANIC! Saved checksum is '"
              << md5_string << "'.\n" << logofs_flush;

      for (unsigned int i = 0; i < MD5_LENGTH; i++)
      {
        sprintf(md5_string + (i * 2),"%02X", md5DigestStream[i]);
      }

      *logofs << "Proxy: PANIC! Calculated checksum is '"
              << md5_string << "'.\n" << logofs_flush;

      #endif

      handleFailOnLoad(cacheName, "E");

      delete cachefs;

      delete md5StateStream;
      delete [] md5DigestStream;

      delete [] cacheName;

      EnableSignals();

      return NULL;
    }
  }

  delete cachefs;

  delete md5StateStream;
  delete [] md5DigestStream;

  delete [] cacheName;

  //
  // Restore the original handlers.
  //

  EnableSignals();

  #ifdef TEST
  *logofs << "Proxy: Successfully loaded cache file '"
          << loadName << "'.\n" << logofs_flush;
  #endif

  //
  // Return the string provided by caller.
  //

  return loadName;
}

void Proxy::increaseChannels(int channelId)
{
  activeChannels_++;

  if (channelId > upperChannel_)
  {
    upperChannel_ = channelId;

    while (channels_[lowerChannel_] == NULL &&
               lowerChannel_ < upperChannel_)
    {
      lowerChannel_++;
    }
  }

  if (channelId < lowerChannel_)
  {
    lowerChannel_ = channelId;

    while (channels_[upperChannel_] == NULL &&
               upperChannel_ > lowerChannel_)
    {
      upperChannel_--;
    }
  }

  if (firstChannel_ > upperChannel_)
  {
    firstChannel_ = upperChannel_;
  }
  else if (firstChannel_ < lowerChannel_)
  {
    firstChannel_ = lowerChannel_;
  }

  #ifdef TEST
  *logofs << "Proxy: Active channels are "
          << activeChannels_ << " lower channel is "
          << lowerChannel_ << " upper channel is "
          << upperChannel_ << " first channel is "
          << firstChannel_ << ".\n" << logofs_flush;
  #endif
}

void Proxy::decreaseChannels(int channelId)
{
  activeChannels_--;

  while (channels_[upperChannel_] == NULL &&
             upperChannel_ > lowerChannel_)
  {
    upperChannel_--;
  }

  while (channels_[lowerChannel_] == NULL &&
             lowerChannel_ < upperChannel_)
  {
    lowerChannel_++;
  }

  if (firstChannel_ > upperChannel_)
  {
    firstChannel_ = upperChannel_;
  }
  else if (firstChannel_ < lowerChannel_)
  {
    firstChannel_ = lowerChannel_;
  }

  #ifdef TEST
  *logofs << "Proxy: Active channels are "
          << activeChannels_ << " lower channel is "
          << lowerChannel_ << " upper channel is "
          << upperChannel_ << " first channel is "
          << firstChannel_ << ".\n" << logofs_flush;
  #endif
}

int Proxy::allocateTransport(int channelFd, int channelId)
{
  if (transports_[channelId] == NULL)
  {
    transports_[channelId] = new Transport(channelFd);

    if (transports_[channelId] == NULL)
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! Can't allocate transport for "
              << "channel id " << channelId << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Can't allocate transport for "
           << "channel id " << channelId << ".\n";

      return -1;
    }
  }
  else if (transports_[channelId] -> getType() !=
               transport_agent)
  {
    #ifdef PANIC
    *logofs << "Proxy: PANIC! Transport for channel id "
            << channelId << " should be null.\n"
            << logofs_flush;
    #endif

    cerr << "Error" << ": Transport for channel id "
         << channelId << " should be null.\n";

    return -1;
  }

  return 1;
}

int Proxy::deallocateTransport(int channelId)
{
  //
  // Transport for the agent connection
  // is passed from the outside when
  // creating the channel.
  //

  if (transports_[channelId] -> getType() !=
          transport_agent)
  {
    delete transports_[channelId];
  }

  transports_[channelId] = NULL;

  return 1;
}
