/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef Proxy_H
#define Proxy_H

#include <sys/types.h>

#ifdef _AIX
#include <sys/select.h>
#endif

#include "Misc.h"
#include "Timestamp.h"

#include "Channel.h"
#include "Transport.h"
#include "EncodeBuffer.h"
#include "ProxyReadBuffer.h"

//
// Forward declaration as we
// need a pointer.
//

class Agent;

//
// Set the verbosity level.
//

#define WARNING
#define PANIC
#undef  TEST
#undef  DEBUG

//
// Log the important tracepoints related
// to writing packets to the peer proxy.
//

#undef  FLUSH

//
// Log operations related to adjusting
// the timeouts.
//

#undef  ADJUST

//
// Codes used for control messages in
// proxy-to-proxy protocol.
//
// The following codes are currently
// unused.
//
// code_alert_reply,
// code_reset_reply,
// code_load_reply,
// code_save_reply,
// code_shutdown_reply,
// code_configuration_request,
// code_configuration_reply.
//

typedef enum
{
  code_new_x_connection,
  code_new_cups_connection,
  code_new_keybd_connection,
  code_new_samba_connection,
  code_new_media_connection,
  code_switch_connection,
  code_drop_connection,
  code_finish_connection,
  code_begin_congestion,
  code_end_congestion,
  code_alert_request,
  code_alert_reply,
  code_reset_request,
  code_reset_reply,
  code_load_request,
  code_load_reply,
  code_save_request,
  code_save_reply,
  code_shutdown_request,
  code_shutdown_reply,
  code_token_request,
  code_token_reply,
  code_configuration_request,
  code_configuration_reply,
  code_statistics_request,
  code_statistics_reply,
  code_new_http_connection,
  code_sync_request,
  code_sync_reply,
  code_last_tag

} T_proxy_code;

typedef enum
{
  operation_in_negotiation,
  operation_in_messages,
  operation_in_configuration,
  operation_in_statistics,
  operation_last_tag

} T_proxy_operation;

typedef enum
{
  frame_data,
  frame_ping,
  frame_last_tag

} T_frame_type;

class Proxy
{
  public:

  //
  // Maximum number of X connections supported.
  //

  static const int CONNECTIONS_LIMIT = 256;

  //
  // This makes possible 3 control messages in
  // a single frame.
  //

  static const int CONTROL_CODES_LENGTH = 9;

  Proxy(int fd);

  virtual ~Proxy();

  //
  // These must be called just after initialization to
  // tell to the proxy where the network connections
  // have to be forwarded.
  //

  virtual void setDisplayInfo(const char *xServerDisplay, int xServerAddrFamily,
                                  sockaddr * xServerAddr, unsigned int xServerAddrLength) = 0;

  virtual void setPortInfo(int syncServerPort, int keybdServerPort,
                               int sambaServerPort, int mediaServerPort,
                                   int httpServerPort) = 0;

  //
  // Inform the proxy that the negotiation phase is
  // completed and that it can start handling binary
  // messages.
  //

  int setOperational();

  int getOperational()
  {
    return (operation_ != operation_in_negotiation);
  }

  int setReadDescriptors(fd_set *fdSet, int &fdMax, T_timestamp &tsMax);

  int setWriteDescriptors(fd_set *fdSet, int &fdMax, T_timestamp &tsMax);

  //
  // Perform operations on proxy link
  // or its own channels.
  //

  int handleRead(int &resultFds, fd_set &fdSet);

  int handleFlush(int &resultFds, fd_set &fdSet);

  int handleRead();

  int handleRead(int fd);

  int handleEvents();

  int handleFlush(T_flush type);

  int handleFlush(T_flush type, int fd);

  int handlePing();

  int handleSync();

  int handleReset();

  int handleShutdown();

  int handleStatistics(int type, ostream *statofs);

  int handleAlert(int alert);

  virtual int handleNotify(T_notification_type type) = 0;

  int handleSocketConfiguration();

  int handleLinkConfiguration();

  int handleCacheConfiguration();

  virtual int handleNewXConnection(int clientFd) = 0;

  virtual int handleNewXConnectionFromProxy(int channelId) = 0;

  virtual int handleNewCupsConnection(int clientFd) = 0;

  virtual int handleNewCupsConnectionFromProxy(int channelId) = 0;

  virtual int handleNewKeybdConnection(int clientFd) = 0;

  virtual int handleNewKeybdConnectionFromProxy(int channelId) = 0;

  virtual int handleNewSambaConnection(int clientFd) = 0;

  virtual int handleNewSambaConnectionFromProxy(int channelId) = 0;

  virtual int handleNewMediaConnection(int clientFd) = 0;

  virtual int handleNewMediaConnectionFromProxy(int channelId) = 0;

  virtual int handleNewHttpConnection(int clientFd) = 0;

  virtual int handleNewHttpConnectionFromProxy(int channelId) = 0;

  //
  // Special version using memory-to-memory transport.
  //

  virtual int handleNewAgentConnection(Agent *agent) = 0;

  //
  // Force closure of channels.
  //

  int handleCloseConnection(int clientFd);

  int handleCloseAllXConnections();

  protected:

  int handleFrame(T_frame_type type);

  int handleFinish(int channelId);

  int handleFinishFromProxy(int channelId);

  int handleStatisticsFromProxy(int type);

  int handleStatisticsFromProxy(const unsigned char *message, unsigned int length);

  int handleNegotiation(const unsigned char *message, unsigned int length);

  int handleNegotiationFromProxy(const unsigned char *message, unsigned int length);

  int handleToken(T_frame_type type);

  int handleTokenFromProxy();

  int handleSyncFromProxy();

  int handleLoad();

  int handleSave();

  int handleSwitch(int channelId);

  int handleControl(T_proxy_code code, int data = -1);

  int handleControlFromProxy(const unsigned char *message);

  int handleSync(int channelId);

  //
  // Timer related functions.
  //

  protected:

  void setTimer(int value)
  {
    SetTimer(value);
  }

  void resetTimer()
  {
    ResetTimer();

    timer_ = 0;
  }

  public:

  void handleTimer()
  {
    timer_ = 1;
  }

  int getTimer()
  {
    return timer_;
  }

  //
  // Can the channel consume data and the
  // proxy produce more output?
  //

  int canRead(int fd) const
  {
    return (isTimeToRead() == 1 &&
                isTimeToRead(getChannel(fd)) == 1);
  }

  //
  // Can the proxy read more data from its
  // peer?
  //

  int canRead() const
  {
    return (transport_ -> readable() > 0);
  }

  int canFlush() const
  {
    return (transport_ -> length() +
                transport_ -> flushable() > 0 &&
                    transport_ -> blocked() == 0);
  }

  int mustFlush() const
  {
    #if defined(TEST) || defined(INFO)
    *logofs << "Proxy: Must flush is " << (priority_ == 1 ||
               congestion_ == 1 || transport_ -> blocked() == 1)
            << " with priority " << priority_ << " congestion "
            << congestion_ << " and blocked " << transport_ ->
               blocked() << ".\n" << logofs_flush;
    #endif

    return (priority_ == 1 || congestion_ == 1 ||
                transport_ -> blocked() == 1);
  }

  int queuedFlush() const
  {
    return (control -> AgentFlushQueued > 0 &&
                transport_ -> queued() > 0);
  }

  int shouldFlush() const
  {
    //
    // It is advisable to keep encoding data as long as
    // it is made available, even when this will cause
    // writing data in excess, compared to the amount
    // scheduled for the token, or when the flush time-
    // out is already expired. By writing all the data
    // we'll ensure that the drawing is performed by the
    // X server according to the original intentions of
    // the client.
    //
    //
    // #if defined(TEST) || defined(INFO)
    // *logofs << "Proxy: Should flush is " << (notifiedBytes_ +
    //            (int) encodeBuffer_.getLength() + controlLength_ +
    //            3 >= control -> TokenBytes || isTimeToFlush() == 1)
    //         << " with " << (notifiedBytes_ + (int) encodeBuffer_.
    //            getLength() + controlLength_ + 3) << " bytes "
    //         << "to notify and " << getTimeToNextFlush() << " Ms "
    //         << "to the next flush.\n" << logofs_flush;
    // #endif
    //
    // if (notifiedBytes_ + (int) encodeBuffer_.getLength() +
    //         controlLength_ + 3 >= control -> TokenBytes ||
    //             isTimeToFlush() == 1)
    // {
    //   return 1;
    // }
    //

    return 0;
  }

  int shouldSync(int sequence) const
  {
    return (notifiedSync_ >= 0 && (sequence >=
                notifiedSync_ || notifiedSync_ -
                    sequence >= 32768));
  }

  int getFd() const
  {
    return fd_;
  }

  int getFlushable(int fd) const
  {
    if (fd == fd_)
    {
      return (transport_ -> length() +
                  transport_ -> flushable());
    }

    return 0;
  }

  int getSplits(int fd)
  {
    return (clientStore_ -> getSplitStore() ->
                getSize());
  }

  //
  // Returns the number of active channels
  // that currently managed by this proxy.
  //

  int getChannels(T_channel_type type);

  //
  // If descriptor corresponds to a valid
  // channel, returns the type of traffic
  // carried by it.
  //

  T_channel_type getType(int fd);

  //
  // True if proxy is presently recovering
  // from a reset.
  //

  int getReset() const
  {
    return reset_;
  }

  //
  // Did remote peer signaled a shutdown?
  //

  int getShutdown() const
  {
    return shutdown_;
  }

  //
  // Remap these calls to the IO buffers.
  //

  int getLength(int fd) const
  {
    if (fd == fd_)
    {
      return transport_ -> length();
    }

    int channelId = getChannel(fd);

    if (channelId < 0 || channels_[channelId] == NULL)
    {
      return 0;
    }

    return transports_[channelId] -> length();
  }

  int getPending(int fd) const
  {
    if (fd == fd_)
    {
      return pending_;
    }

    int channelId = getChannel(fd);

    if (channelId < 0 || channels_[channelId] == NULL)
    {
      return 0;
    }

    return channels_[channelId] -> getPending();
  }

  int setPending(int fd)
  {
    if (fd == fd_)
    {
      pending_ = 1;
    }
    else
    {
      int channelId = getChannel(fd);

      if (channelId < 0 || channels_[channelId] == NULL)
      {
        #ifdef TEST
        *logofs << "Proxy: PANIC! Can't set buffer for X "
                << "descriptor FD#" << fd_ << " to be pending.\n"
                << logofs_flush;
        #endif

        return -1;
      }

      #ifdef TEST
      *logofs << "Proxy: Setting buffer for X descriptor FD#"
              << fd_ << " to be pending.\n" << logofs_flush;
      #endif

      channels_[channelId] -> setPending();
    }

    return 1;
  }

  //
  // Check if proxy or given channel has data
  // in buffers because of a blocking write.
  //

  int getBlocked(int fd) const
  {
    if (fd == fd_)
    {
      return transport_ -> blocked();
    }

    int channelId = getChannel(fd);

    if (channelId < 0 || channels_[channelId] == NULL)
    {
      return 0;
    }

    return transports_[channelId] -> blocked();
  }

  //
  // Check if an immediate flush of proxy link
  // is requested by channels. This is the case
  // when an X connection is waiting a reply.
  //

  int getPriority(int fd) const
  {
    if (fd == fd_)
    {
      return priority_;
    }

    int channelId = getChannel(fd);

    if (channelId < 0 || channels_[channelId] == NULL)
    {
      return 0;
    }

    return channels_[channelId] -> getPriority();
  }

  int getCongestion(int fd) const
  {
    if (fd == fd_)
    {
      return congestion_;
    }

    int channelId = getChannel(fd);

    if (channelId < 0 || channels_[channelId] == NULL)
    {
      return 0;
    }

    return channels_[channelId] -> getCongestion();
  }

  //
  // Let statistics class to get info from
  // message stores.
  //

  const ClientStore * const getClientStore() const
  {
    return clientStore_;
  }

  const ServerStore * const getServerStore() const
  {
    return serverStore_;
  }

  //
  // These can be called asynchronously by
  // channels during their read or write
  // loop.
  //

  int canAsyncRead() const
  {
    return canRead();
  }

  int canAsyncFlush() const
  {
    return shouldFlush();
  }

  int canAsyncSync(int sequence) const
  {
    return shouldSync(sequence);
  }

  int handleAsyncRead(int fd)
  {
    return handleRead(fd);
  }

  int handleAsyncCongestion(int fd);

  int handleAsyncSplit(int fd);

  //
  // Internal interfaces used to verify the
  // availability of channels and the proxy
  // link.
  //

  protected:

  int isTimeToRead() const
  {
    if (reset_ == 0 && congestion_ == 0 &&
                transport_ -> blocked() == 0)
    {
      return 1;
    }
    else
    {
      #if defined(TEST) || defined(INFO)
      *logofs << "Proxy: Can't read from channels "
              << "with reset " << reset_ << " congestion "
              << congestion_ << " blocked " << transport_ ->
                 blocked() << ".\n" << logofs_flush;
      #endif

      return 0;
    }
  }

  int isTimeToRead(int channelId) const
  {
    if (channelId >= 0 && channelId < CONNECTIONS_LIMIT &&
            channels_[channelId] != NULL && channels_[channelId] ->
                getFinish() == 0 && congestions_[channelId] == 0)
    {
      return 1;
    }
    else
    {
      #if defined(TEST) || defined(INFO)

      if (channelId < 0 || channelId >= CONNECTIONS_LIMIT ||
              channels_[channelId] == NULL)
      {
        *logofs << "Proxy: WARNING! No valid channel for ID#"
                << channelId << ".\n" << logofs_flush;
      }
      else if (channels_[channelId] -> getFinish())
      {
        *logofs << "Proxy: Can't read from finishing "
                << "descriptor FD#" << getFd(channelId)
                << " channel ID#" << channelId << ".\n"
                << logofs_flush;
      }
      else if (congestions_[channelId] == 1)
      {
        *logofs << "Proxy: Can't read from congested "
                << "descriptor FD#" << getFd(channelId)
                << " channel ID#" << channelId << ".\n"
                << logofs_flush;
      }

      #endif

      return 0;
    }
  }

  //
  // Handle the flush and split timeouts.
  // All these functions should round up
  // to the value of the latency timeout
  // to save a further loop.
  //
  //

  protected:

  int isTimeToSplit() const
  {
    return (getTimeToNextSplit() <=
                control -> LatencyTimeout ? 1 : 0);
  }

  int isTimeToFlush() const
  {
    return (getTimeToNextFlush() <=
                control -> LatencyTimeout ? 1 : 0);
  }

  int getTimeToNextSplit() const
  {
    #if defined(TEST) || defined(INFO) || defined(FLUSH)

    if (isTimestamp(lastSplitTs_) == 0)
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! No split timeout was set "
              << "for proxy FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": No split timeout was set "
           << "for proxy FD#" << fd_ << ".\n";

      HandleCleanup();
    }

    #endif

    int diffTs = timeouts_.split -
                     diffTimestamp(lastSplitTs_, getTimestamp());

    #if defined(TEST) || defined(INFO) || defined(FLUSH)

    if (diffTs <= control -> LatencyTimeout)
    {
      *logofs << "Proxy: FLUSH! WARNING! The split timeout is already expired.\n"
              << logofs_flush;
    }

    #endif

    return (diffTs > 0 ? diffTs : 0);
  }

  int getTimeToNextFlush() const
  {
    #if defined(TEST) || defined(INFO)

    if (isTimestamp(lastFrameTs_) == 0)
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! No frame timeout was set "
              << "for proxy FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": No frame timeout was set "
           << "for proxy FD#" << fd_ << ".\n";

      HandleCleanup();
    }
    else if (isTimestamp(lastFlushTs_) == 0)
    {
      #ifdef PANIC
      *logofs << "Proxy: PANIC! No flush timeout was set "
              << "for proxy FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": No flush timeout was set "
           << "for proxy FD#" << fd_ << ".\n";

      HandleCleanup();
    }

    #endif

    T_timestamp now = getTimestamp();

    int diffFrame = timeouts_.frame -
                        diffTimestamp(lastFrameTs_, now);

    #if defined(TEST) || defined(INFO)
    *logofs << "Proxy: Difference from frame timeout " << diffFrame
            << " with elapsed " << diffTimestamp(lastFrameTs_, now)
            << " and timeout " << timeouts_.frame << ".\n"
            << logofs_flush;
    #endif

    int diffFlush = timeouts_.flush -
                        diffTimestamp(lastFlushTs_, now);

    #if defined(TEST) || defined(INFO)
    *logofs << "Proxy: Difference from flush timeout " << diffFlush
            << " with elapsed " << diffTimestamp(lastFlushTs_, now)
            << " and timeout " << timeouts_.flush << ".\n"
            << logofs_flush;
    #endif

    int diffTs = (diffFrame < diffFlush ? diffFrame : diffFlush);

    #if defined(TEST) || defined(INFO)
    *logofs << "Proxy: Selecting minimum difference of " << diffTs
            << " Ms.\n" << logofs_flush;
    #endif

    return (diffTs > 0 ? diffTs : 0);
  }

  //
  // Implement persistence of cache on disk.
  //

  protected:

  typedef enum
  {
    load_if_any,
    load_if_first

  } T_proxy_load;

  virtual int handleCheckLoad(T_proxy_load type) = 0;

  virtual int handleCheckSave() = 0;

  virtual int handleLoadFromProxy() = 0;

  virtual int handleSaveFromProxy() = 0;

  char *handleSaveStores(const char *savePath) const;

  virtual int handleSaveStores(ostream *cachefs, md5_state_t *md5StateStream,
                                   md5_state_t *md5StateClient) const = 0;

  int handleSaveVersion(unsigned char *buffer, int &major, int &minor, int &patch) const;

  void handleFailOnSave(const char *fullName, const char *failContext) const;

  const char *handleLoadStores(const char *loadPath, const char *loadName) const;

  virtual int handleLoadStores(istream *cachefs, md5_state_t *md5StateStream) const = 0;

  int handleLoadVersion(const unsigned char *buffer, int &major, int &minor, int &patch) const;

  void handleFailOnLoad(const char *fullName, const char *failContext) const;

  int handleResetPersistentCache();

  int handleResetOpcodes();

  int handleResetStores();

  int handleResetCaches();

  //
  // Reset counters used to determine
  // the best time to flush the proxy
  // link.
  //

  void handleResetFlush();

  //
  // Adjust the timeouts according to
  // the network events.
  //

  void handleInput()
  {
    decreaseTimeout(timeouts_.frame, control -> FrameTimeout);
    decreaseTimeout(timeouts_.flush, control -> FlushTimeout);

    #if defined(TEST) || defined(INFO) || defined(ADJUST)
    *logofs << "Proxy: ADJUST! Decreased flush and frame timeout to "
            << timeouts_.frame << " and " << timeouts_.flush
            << " Ms.\n" << logofs_flush;
    #endif
  }

  void handlePriority()
  {
    priority_ = 1;
  }

  void handleBitrate()
  {
    int bitrate = control -> getBitrateInShortFrame();

    #if defined(TEST) || defined(INFO) || defined(ADJUST)
    *logofs << "Proxy: ADJUST! Propagating bitrate of "
            << bitrate << " B/S  at " << strMsTimestamp()
            << ".\n" << logofs_flush;
    #endif

    if (bitrate > control -> TokenBytes)
    {
      increaseTimeout(timeouts_.frame, control -> FrameTimeout);
      increaseTimeout(timeouts_.flush, control -> FlushTimeout);

      #if defined(TEST) || defined(INFO) || defined(ADJUST)
      *logofs << "Proxy: ADJUST! Increased flush and frame timeout to "
              << timeouts_.frame << " and " << timeouts_.flush
              << " Ms.\n" << logofs_flush;
      #endif

      increaseTimeout(timeouts_.split, control -> SplitTimeout);

      #if defined(TEST) || defined(INFO) || defined(ADJUST)
      *logofs << "Proxy: ADJUST! Increased split timeout to "
              << timeouts_.split << " Ms.\n"
              << logofs_flush;
      #endif
    }
    else if (bitrate < control -> TokenBytes >> 2)
    {
      decreaseTimeout(timeouts_.split, control -> SplitTimeout);

      #if defined(TEST) || defined(INFO) || defined(ADJUST)
      *logofs << "Proxy: ADJUST! Decreased split timeout to "
              << timeouts_.split << " Ms.\n"
              << logofs_flush;
      #endif
    }
  }

  protected:

  virtual int getFd(int channelId) const = 0;
  virtual int getChannel(int fd) const = 0;

  virtual void cleanupChannelMap(int channelId) = 0;

  void increaseChannels(int channelId);
  void decreaseChannels(int channelId);

  int allocateTransport(int channelFd, int channelId);
  int deallocateTransport(int channelId);

  void initialTimeout(int &timeout, int base)
  {
    timeout = base >> 1;
  }

  void increaseTimeout(int &timeout, int base)
  {
    //
    // This is an alternate version that
    // that must be evaluated.
    //
    // timeout <<= 1;
    //
    // if (timeout > base)
    // {
    //   timeout = base;
    // }
    //

    timeout = base;
  }

  void decreaseTimeout(int &timeout, int base)
  {
    //
    // This is an alternate version that
    // that must be evaluated.
    //
    // base >>= 2;
    //
    // timeout >>= 1;
    //
    // if (timeout < base)
    // {
    //   timeout = base;
    // }
    //

    timeout = base >> 2;
  }

  //
  // Proxy create its own transport.
  //

  ProxyTransport *transport_;

  //
  // Compressor and decompressor are shared
  // between channels and message stores.
  //

  Compressor   *compressor_;
  Decompressor *decompressor_;

  //
  // Map NX specific opcodes.
  //

  OpcodeStore *opcodeStore_;

  //
  // Stores are shared between channels.
  //

  ClientStore *clientStore_;
  ServerStore *serverStore_;

  //
  // Starting from protocol level 3 even
  // client and server caches are shared
  // between channels.
  //

  ClientCache *clientCache_;
  ServerCache *serverCache_;

  //
  // Used for IO operations on proxy link.
  //

  int fd_;

  int inputChannel_;
  int outputChannel_;

  //
  // Used to read data sent from peer proxy.
  //

  ProxyReadBuffer readBuffer_;

  //
  // Used to send data to peer proxy.
  //

  EncodeBuffer encodeBuffer_;

  //
  // Control messages' array.
  //

  int controlLength_;

  unsigned char controlCodes_[CONTROL_CODES_LENGTH];

  //
  // Table of channel classes taking
  // care of open X connections.
  //

  Channel *channels_[CONNECTIONS_LIMIT];

  //
  // Table of open sockets.
  //

  Transport *transports_[CONNECTIONS_LIMIT];

  //
  // Lower and upper limit of active
  // and next channel to be handled
  // first.
  //

  int lowerChannel_;
  int upperChannel_;
  int firstChannel_;

  //
  // The number of active connections.
  //

  int activeChannels_;

  //
  // Used to adapt the timeouts to the
  // link conditions.
  //

  typedef struct
  {
    int frame;
    int flush;
    int split;

  } T_proxy_timeouts;

  T_proxy_timeouts timeouts_;

  //
  // Proxy can be decoding messages,
  // handling a link reconfiguration,
  // or decoding statistics.
  //

  int operation_;

  //
  // True if there are pending messages
  // in proxy's read buffer.
  //

  int pending_;

  //
  // Force flush because of prioritized
  // control messages to send.
  //

  int priority_;

  //
  // Remote peer requested shutdown.
  //

  int shutdown_;

  //
  // We are in the middle of a reset.
  //

  int reset_;

  //
  // We are in the middle of a network
  // congestion in the path to remote
  // proxy.
  //

  int congestion_;

  //
  // Is there any channel at remote end
  // that's not consuming enough data?
  //

  int congestions_[CONNECTIONS_LIMIT];

  //
  // Is the flush timer expired.
  //

  int timer_;

  //
  // Number of remaining tokens.
  //

  int tokens_;

  //
  // Timestamp of last packet received
  // or sent to remote proxy. Used to
  // detect lost connection.
  //

  T_timestamp lastBytesInTs_;
  T_timestamp lastBytesOutTs_;

  //
  // Timestamp of last loop completed by
  // the proxy
  //

  T_timestamp lastLoopTs_;

  //
  // Timestamp of last ping request sent
  // to the remote peer.
  //

  T_timestamp lastPingTs_;

  //
  // Timestamp of last 'no data received'
  // alert dialog shown to user.
  //

  T_timestamp lastAlertTs_;

  //
  // Were message stores populated from
  // data on disk.
  //

  T_timestamp lastLoadTs_;

  //
  // Timestamp of the last operation of
  // this kind performed on the proxy.
  //

  T_timestamp lastSplitTs_;
  T_timestamp lastFlushTs_;
  T_timestamp lastFrameTs_;

  //
  // Sequence counter of the last X request
  // decoded by agent connection at the time
  // the sync control code was received by
  // this proxy.
  //

  int notifiedSync_;
  
  //
  // Type of last congestion notification
  // sent to the X agent.
  //

  int notifiedCongestion_;

  //
  // Amount of data accumulated before
  // sending the next token.
  //

  int notifiedBytes_;

  //
  // Number of proxy frames and number of bytes
  // that are going to be sent in the next flush.
  //

  int deferredFrames_;
  int deferredBytes_;

  //
  // Pointer to stream descriptor where
  // proxy is producing statistics.
  //

  ostream *statisticsStream_;
};

#endif /* Proxy_H */
