/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include "NXrender.h"

#include "RenderExtension.h"

#include "RenderGenericRequest.h"
#include "RenderCreatePicture.h"
#include "RenderChangePicture.h"
#include "RenderFreePicture.h"
#include "RenderSetPictureClipRectangles.h"
#include "RenderCreateGlyphSet.h"
#include "RenderAddGlyphs.h"
#include "RenderComposite.h"
#include "RenderCompositeGlyphs.h"
#include "RenderFillRectangles.h"

#include "ClientCache.h"

#include "EncodeBuffer.h"
#include "DecodeBuffer.h"

#include "WriteBuffer.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

//
// Constructor and destructor.
//

RenderExtensionStore::RenderExtensionStore(Compressor *compressor, Decompressor *decompressor)

  : MessageStore(compressor, decompressor)
{
  if (control -> isProtoStep5() == 1)
  {
    enableCache    = RENDEREXTENSION_ENABLE_CACHE_IF_PROTO_STEP_5;
    enableData     = RENDEREXTENSION_ENABLE_DATA_IF_PROTO_STEP_5;
    enableSplit    = RENDEREXTENSION_ENABLE_SPLIT_IF_PROTO_STEP_5;
    enableCompress = RENDEREXTENSION_ENABLE_COMPRESS_IF_PROTO_STEP_5;
  }
  else
  {
    enableCache    = RENDEREXTENSION_ENABLE_CACHE;
    enableData     = RENDEREXTENSION_ENABLE_DATA;
    enableSplit    = RENDEREXTENSION_ENABLE_SPLIT;
    enableCompress = RENDEREXTENSION_ENABLE_COMPRESS;
  }

  generic_ = new RenderGenericRequestStore();

  for (int i = 0; i < RENDEREXTENSION_MINOR_OPCODE_LIMIT; i++)
  {
    minors_[i] = generic_;
  }

  if (control -> isProtoStep5() == 1)
  {
    minors_[X_RenderCreatePicture]            = new RenderCreatePictureStore();
    minors_[X_RenderChangePicture]            = new RenderChangePictureStore();
    minors_[X_RenderFreePicture]              = new RenderFreePictureStore();
    minors_[X_RenderSetPictureClipRectangles] = new RenderSetPictureClipRectanglesStore();
    minors_[X_RenderCreateGlyphSet]           = new RenderCreateGlyphSetStore();
    minors_[X_RenderAddGlyphs]                = new RenderAddGlyphsStore();
    minors_[X_RenderComposite]                = new RenderCompositeStore();
    minors_[X_RenderCompositeGlyphs8]         = new RenderCompositeGlyphsStore();
    minors_[X_RenderCompositeGlyphs16]        = new RenderCompositeGlyphsStore();
    minors_[X_RenderCompositeGlyphs32]        = new RenderCompositeGlyphsStore();
    minors_[X_RenderFillRectangles]           = new RenderFillRectanglesStore();
  }

  dataLimit  = RENDEREXTENSION_DATA_LIMIT;
  dataOffset = RENDEREXTENSION_DATA_OFFSET;

  cacheSlots          = RENDEREXTENSION_CACHE_SLOTS;
  cacheThreshold      = RENDEREXTENSION_CACHE_THRESHOLD;
  cacheLowerThreshold = RENDEREXTENSION_CACHE_LOWER_THRESHOLD;

  opcode_ = X_NXInternalRenderExtension;

  messages_ -> resize(cacheSlots);

  for (T_messages::iterator i = messages_ -> begin();
           i < messages_ -> end(); i++)
  {
    *i = NULL;
  }

  temporary_ = NULL;
}

RenderExtensionStore::~RenderExtensionStore()
{
  for (int i = 0; i < RENDEREXTENSION_MINOR_OPCODE_LIMIT; i++)
  {
    if (minors_[i] != generic_)
    {
      delete minors_[i];
    }
  }

  delete generic_;

  for (T_messages::iterator i = messages_ -> begin();
           i < messages_ -> end(); i++)
  {
    destroy(*i);
  }

  destroy(temporary_);
}

//
// Here are the methods to handle the messages' content.
//

int RenderExtensionStore::identitySize(const unsigned char *buffer, unsigned int size)
{
  return minors_[*(buffer + 1)] -> identitySize(buffer, size);
}

int RenderExtensionStore::encodeIdentity(EncodeBuffer &encodeBuffer, const unsigned char *buffer,
                                             const unsigned int size, int bigEndian,
                                                 ChannelCache *channelCache) const
{
  if (control -> isProtoStep5() == 1)
  {
    encodeBuffer.encodeOpcodeValue(*(buffer + 1),
                       ((ClientCache *) channelCache) -> renderOpcodeCache);
  }
  else
  {
    encodeBuffer.encodeCachedValue(*(buffer + 1), 8,
                       ((ClientCache *) channelCache) -> renderTypeCache);
  }

  minors_[*(buffer + 1)] -> encodeMessage(encodeBuffer, buffer, size,
                                              bigEndian, channelCache);

  return 1;
}

int RenderExtensionStore::decodeIdentity(DecodeBuffer &decodeBuffer, unsigned char *&buffer,
                                             unsigned int &size, int bigEndian, WriteBuffer *writeBuffer,
                                                 ChannelCache *channelCache) const
{
  unsigned char type;

  if (control -> isProtoStep5() == 1)
  {
    decodeBuffer.decodeOpcodeValue(type,
                       ((ClientCache *) channelCache) -> renderOpcodeCache);
  }
  else
  {
    decodeBuffer.decodeCachedValue(type, 8,
                       ((ClientCache *) channelCache) -> renderTypeCache);
  }

  minors_[type] -> decodeMessage(decodeBuffer, buffer, size, type,
                                     bigEndian, writeBuffer, channelCache);

  return 1;
}

int RenderExtensionStore::parseIdentity(Message *message, const unsigned char *buffer,
                                            unsigned int size, int bigEndian) const
{
  return minors_[*(buffer + 1)] -> parseIdentity(message, buffer, size, bigEndian);
}

int RenderExtensionStore::unparseIdentity(const Message *message, unsigned char *buffer,
                                              unsigned int size, int bigEndian) const
{
  return minors_[((RenderExtensionMessage *) message) -> type] ->
                     unparseIdentity(message, buffer, size, bigEndian);
}

void RenderExtensionStore::identityChecksum(const Message *message, const unsigned char *buffer,
                                               unsigned int size, int bigEndian) const
{
  minors_[*(buffer + 1)] -> identityChecksum(message, buffer, size, md5_state_, bigEndian);
}

void RenderExtensionStore::updateIdentity(EncodeBuffer &encodeBuffer, const Message *message,
                                              const Message *cachedMessage,
                                                   ChannelCache *channelCache) const
{
  minors_[((RenderExtensionMessage *) message) -> type] ->
              updateIdentity(encodeBuffer, message, cachedMessage, channelCache);
}

void RenderExtensionStore::updateIdentity(DecodeBuffer &decodeBuffer, const Message *message,
                                              ChannelCache *channelCache) const
{
  minors_[((RenderExtensionMessage *) message) -> type] ->
              updateIdentity(decodeBuffer, message, channelCache);
}

void RenderExtensionStore::dumpIdentity(const Message *message) const
{
  #ifdef PANIC
  *logofs << name() << ": PANIC! Dump of identity not implemented.\n"
          << logofs_flush;
  #endif
}

//
// TODO: The following encoding and decoding functions
// could be generalized further, for example by passing
// the pointer to the data cache, the number of caches
// made available by the caller and the first cache to
// iterate through.
//

void RenderMinorExtensionStore::encodeLongData(EncodeBuffer &encodeBuffer, const unsigned char *buffer,
                                                   unsigned int offset, unsigned int size, int bigEndian,
                                                       ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  for (unsigned int i = offset, c = (offset - 4) % 16; i < size; i += 4)
  {
    #ifdef DEBUG
    *logofs << name() << ": Encoding int with i = " << i << " c = "
            << c << ".\n" << logofs_flush;
    #endif

    encodeBuffer.encodeCachedValue(GetULONG(buffer + i, bigEndian), 32,
                       *clientCache -> renderDataCache[c]);

    if (++c == 16) c = 0;
  }
}

void RenderMinorExtensionStore::encodeIntData(EncodeBuffer &encodeBuffer, const unsigned char *buffer,
                                                  unsigned int offset, unsigned int size, int bigEndian,
                                                      ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  for (unsigned int i = offset, c = (offset - 4) % 16; i < size; i += 2)
  {
    #ifdef DEBUG
    *logofs << name() << ": Encoding int with i = " << i << " c = "
            << c << ".\n" << logofs_flush;
    #endif

    encodeBuffer.encodeCachedValue(GetUINT(buffer + i, bigEndian), 16,
                       *clientCache -> renderDataCache[c]);

    if (++c == 16) c = 0;
  }
}

void RenderMinorExtensionStore::encodeCharData(EncodeBuffer &encodeBuffer, const unsigned char *buffer,
                                                   unsigned int offset, unsigned int size, int bigEndian,
                                                       ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  clientCache -> renderTextCompressor.reset();

  const unsigned char *next = buffer + offset;

  for (unsigned int i = offset; i < size; i++)
  {
    #ifdef DEBUG
    *logofs << name() << ": Encoding char with i = " << i
            << ".\n" << logofs_flush;
    #endif

    clientCache -> renderTextCompressor.
          encodeChar(*next++, encodeBuffer);
  }
}

void RenderMinorExtensionStore::decodeLongData(DecodeBuffer &decodeBuffer, unsigned char *buffer,
                                                   unsigned int offset, unsigned int size, int bigEndian,
                                                       ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  unsigned int value;

  for (unsigned int i = offset, c = (offset - 4) % 16; i < size; i += 4)
  {
    #ifdef DEBUG
    *logofs << name() << ": Decoding int with i = " << i << " c = "
            << c << ".\n" << logofs_flush;
    #endif

    decodeBuffer.decodeCachedValue(value, 32,
                       *clientCache -> renderDataCache[c]);

    PutULONG(value, buffer + i, bigEndian);

    if (++c == 16) c = 0;
  }
}

void RenderMinorExtensionStore::decodeIntData(DecodeBuffer &decodeBuffer, unsigned char *buffer,
                                                  unsigned int offset, unsigned int size, int bigEndian,
                                                      ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  unsigned int value;

  for (unsigned int i = offset, c = (offset - 4) % 16; i < size; i += 2)
  {
    #ifdef DEBUG
    *logofs << name() << ": Decoding int with i = " << i << " c = "
            << c << ".\n" << logofs_flush;
    #endif

    decodeBuffer.decodeCachedValue(value, 16,
                       *clientCache -> renderDataCache[c]);

    PutUINT(value, buffer + i, bigEndian);

    if (++c == 16) c = 0;
  }
}

void RenderMinorExtensionStore::decodeCharData(DecodeBuffer &decodeBuffer, unsigned char *buffer,
                                                   unsigned int offset, unsigned int size, int bigEndian,
                                                       ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  clientCache -> renderTextCompressor.reset();

  unsigned char *next = buffer + offset;

  for (unsigned int i = offset; i < size; i++)
  {
    #ifdef DEBUG
    *logofs << name() << ": Decoding char with i = " << i
            << ".\n" << logofs_flush;
    #endif

    *next++ = clientCache -> renderTextCompressor.
                    decodeChar(decodeBuffer);
  }
}

void RenderMinorExtensionStore::parseIntData(const Message *message, const unsigned char *buffer,
                                                 unsigned int offset, unsigned int size,
                                                     int bigEndian) const
{
  RenderExtensionMessage *renderExtension = (RenderExtensionMessage *) message;

  unsigned int last = ((unsigned) message -> i_size_ > size ? size : message -> i_size_);

  for (unsigned int i = offset, c = (offset - 4) % 16; i < last; i += 2)
  {
    #ifdef DEBUG
    *logofs << name() << ": Parsing int with i = " << i << " c = "
            << c << ".\n" << logofs_flush;
    #endif

    renderExtension -> raw_data.short_data[c] = GetUINT(buffer + i, bigEndian);

    if (++c == 16) c = 0;
  }
}

void RenderMinorExtensionStore::parseCharData(const Message *message, const unsigned char *buffer,
                                                  unsigned int offset, unsigned int size,
                                                      int bigEndian) const
{
  RenderExtensionMessage *renderExtension = (RenderExtensionMessage *) message;

  const unsigned char *next = buffer + offset;

  unsigned int last = ((unsigned) message -> i_size_ > size ? size : message -> i_size_);

  for (unsigned int i = offset, c = 0; i < last; i++, c++)
  {
    #ifdef DEBUG
    *logofs << name() << ": Parsing char with i = " << i << ".\n"
            << logofs_flush;
    #endif

    renderExtension -> raw_data.char_data[c] = *next++;
  }
}

void RenderMinorExtensionStore::unparseIntData(const Message *message, unsigned char *buffer,
                                                   unsigned int offset, unsigned int size,
                                                       int bigEndian) const
{
  RenderExtensionMessage *renderExtension = (RenderExtensionMessage *) message;

  unsigned int last = ((unsigned) message -> i_size_ > size ? size : message -> i_size_);

  for (unsigned int i = offset, c = (offset - 4) % 16; i < last; i += 2)
  {
    #ifdef DEBUG
    *logofs << name() << ": Unparsing int with i = " << i << " c = "
            << c << ".\n" << logofs_flush;
    #endif

    PutUINT(renderExtension -> raw_data.short_data[c], buffer + i, bigEndian);

    if (++c == 16) c = 0;
  }
}

void RenderMinorExtensionStore::unparseCharData(const Message *message, unsigned char *buffer,
                                                    unsigned int offset, unsigned int size,
                                                        int bigEndian) const
{
  RenderExtensionMessage *renderExtension = (RenderExtensionMessage *) message;

  unsigned char *next = buffer + offset;

  unsigned int last = ((unsigned) message -> i_size_ > size ? size : message -> i_size_);

  for (unsigned int i = offset, c = 0; i < last; i++, c++)
  {
    #ifdef DEBUG
    *logofs << name() << ": Unparsing char with i = " << i
            << ".\n" << logofs_flush;
    #endif

    *next++ = renderExtension -> raw_data.char_data[c];
  }
}

void RenderMinorExtensionStore::updateIntData(EncodeBuffer &encodeBuffer, const Message *message,
                                                  const Message *cachedMessage, unsigned int offset,
                                                      unsigned int size, ChannelCache *channelCache) const
{
  RenderExtensionMessage *renderExtension       = (RenderExtensionMessage *) message;
  RenderExtensionMessage *cachedRenderExtension = (RenderExtensionMessage *) cachedMessage;

  ClientCache *clientCache = (ClientCache *) channelCache;

  unsigned int last = ((unsigned) message -> i_size_ > size ? size : message -> i_size_);

  for (unsigned int i = offset, c = (offset - 4) % 16; i < last; i += 2)
  {
    #ifdef DEBUG
    *logofs << name() << ": Encoding int update with i = " << i
            << " c = " << c << ".\n" << logofs_flush;
    #endif

    encodeBuffer.encodeCachedValue(renderExtension -> raw_data.short_data[c], 16,
                       *clientCache -> renderDataCache[c]);

    cachedRenderExtension -> raw_data.short_data[c] =
                renderExtension -> raw_data.short_data[c];

    if (++c == 16) c = 0;
  }
}

void RenderMinorExtensionStore::updateCharData(EncodeBuffer &encodeBuffer, const Message *message,
                                                   const Message *cachedMessage, unsigned int offset,
                                                       unsigned int size, ChannelCache *channelCache) const
{
  RenderExtensionMessage *renderExtension       = (RenderExtensionMessage *) message;
  RenderExtensionMessage *cachedRenderExtension = (RenderExtensionMessage *) cachedMessage;

  ClientCache *clientCache = (ClientCache *) channelCache;

  clientCache -> renderTextCompressor.reset();

  unsigned int last = ((unsigned) message -> i_size_ > size ? size : message -> i_size_);

  for (unsigned int i = offset, c = 0; i < last; i++, c++)
  {
    #ifdef DEBUG
    *logofs << name() << ": Encoding char update with i = " << i
            << ".\n" << logofs_flush;
    #endif

    clientCache -> renderTextCompressor.
          encodeChar(renderExtension -> raw_data.char_data[c],
                encodeBuffer);

    cachedRenderExtension -> raw_data.char_data[c] =
               renderExtension -> raw_data.char_data[c];
  }
}

void RenderMinorExtensionStore::updateIntData(DecodeBuffer &decodeBuffer, const Message *message,
                                                  unsigned int offset, unsigned int size,
                                                      ChannelCache *channelCache) const
{
  RenderExtensionMessage *renderExtension = (RenderExtensionMessage *) message;

  ClientCache *clientCache = (ClientCache *) channelCache;

  unsigned int last = ((unsigned) message -> i_size_ > size ? size : message -> i_size_);

  unsigned int value;

  for (unsigned int i = offset, c = (offset - 4) % 16; i < last; i += 2)
  {
    #ifdef DEBUG
    *logofs << name() << ": Decoding int update with i = " << i
            << " c = " << c << ".\n" << logofs_flush;
    #endif

    decodeBuffer.decodeCachedValue(value, 16,
                       *clientCache -> renderDataCache[c]);

    renderExtension -> raw_data.short_data[c] = value;

    if (++c == 16) c = 0;
  }
}

void RenderMinorExtensionStore::updateCharData(DecodeBuffer &decodeBuffer, const Message *message,
                                                   unsigned int offset, unsigned int size,
                                                       ChannelCache *channelCache) const
{
  RenderExtensionMessage *renderExtension = (RenderExtensionMessage *) message;

  ClientCache *clientCache = (ClientCache *) channelCache;

  clientCache -> renderTextCompressor.reset();

  unsigned int last = ((unsigned) message -> i_size_ > size ? size : message -> i_size_);

  for (unsigned int i = offset, c = 0; i < last; i++, c++)
  {
    #ifdef DEBUG
    *logofs << name() << ": Decoding char update with i = " << i << ".\n"
            << logofs_flush;
    #endif

    renderExtension -> raw_data.char_data[c] =
          clientCache -> renderTextCompressor.
                decodeChar(decodeBuffer);
  }
}
