/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include "NXrender.h"

#include "RenderExtension.h"
#include "RenderGenericRequest.h"

#include "ClientCache.h"

#include "EncodeBuffer.h"
#include "DecodeBuffer.h"

#include "WriteBuffer.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

//
// Here are the methods to handle the messages' content.
//

int RenderGenericRequestStore::encodeMessage(EncodeBuffer &encodeBuffer, const unsigned char *buffer,
                                                 const unsigned int size, int bigEndian,
                                                     ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  #ifdef DEBUG
  *logofs << name() << ": Encoding full message.\n"
          << logofs_flush;
  #endif

  unsigned char type = *(buffer + 1);

  encodeBuffer.encodeCachedValue(size >> 2, 16,
                     clientCache -> renderLengthCache, 5);

  unsigned int value;

  #ifdef DEBUG
  *logofs << name() << ": Encoding full message. Type is "
          << (unsigned int) type << " size is " << size
          << ".\n" << logofs_flush;
  #endif

  switch (type)
  {
    case X_RenderCreatePicture:
    {
      unsigned int picture = GetULONG(buffer + 4, bigEndian);

      encodeBuffer.encodeXidValue(picture - clientCache -> renderLastId,
                                      clientCache -> renderPictureCache);

      clientCache -> renderLastId = picture;

      encodeBuffer.encodeXidValue(GetULONG(buffer + 8, bigEndian),
                         clientCache -> drawableCache);

      encodeBuffer.encodeCachedValue(GetULONG(buffer + 12, bigEndian), 32,
                         clientCache -> renderFormatCache);

      encodeBuffer.encodeCachedValue(GetULONG(buffer + 16, bigEndian), 32,
                         clientCache -> renderMaskCache);

      encodeCharData(encodeBuffer, buffer, 20, size,
                         bigEndian, clientCache);
      break;
    }
    case X_RenderChangePicture:
    {
      encodeBuffer.encodeXidValue(GetULONG(buffer + 4, bigEndian),
                                      clientCache -> renderPictureCache);

      encodeBuffer.encodeCachedValue(GetULONG(buffer + 8, bigEndian), 32,
                         clientCache -> renderMaskCache);

      encodeCharData(encodeBuffer, buffer, 12, size,
                         bigEndian, clientCache);
      break;
    }
    case X_RenderFreePicture:
    {
      encodeBuffer.encodeXidValue(GetULONG(buffer + 4, bigEndian),
                                      clientCache -> renderPictureCache);

      encodeCharData(encodeBuffer, buffer, 8, size,
                         bigEndian, clientCache);
      break;
    }
    case X_RenderComposite:
    {
      encodeBuffer.encodeCachedValue(*(buffer + 4), 8,
                         clientCache -> renderOpCache);

      encodeBuffer.encodeXidValue(GetULONG(buffer + 8, bigEndian),
                         clientCache -> renderPictureCache);

      encodeBuffer.encodeXidValue(GetULONG(buffer + 12, bigEndian),
                         clientCache -> renderPictureCache);

      encodeBuffer.encodeXidValue(GetULONG(buffer + 16, bigEndian),
                         clientCache -> renderPictureCache);

      value = GetUINT(buffer + 20, bigEndian);

      encodeBuffer.encodeCachedValue(value - clientCache -> renderLastX, 16,
                         clientCache -> renderXCache, 8);

      clientCache -> renderLastX = value;

      value = GetUINT(buffer + 22, bigEndian);

      encodeBuffer.encodeCachedValue(value - clientCache -> renderLastY, 16,
                         clientCache -> renderYCache, 8);

      clientCache -> renderLastY = value;

      encodeIntData(encodeBuffer, buffer, 24, size,
                        bigEndian, clientCache);
      break;
    }
    case X_RenderAddGlyphs:
    {
      encodeBuffer.encodeCachedValue(GetULONG(buffer + 4, bigEndian), 29,
                         clientCache -> renderGlyphsetCache);

      encodeBuffer.encodeCachedValue(GetULONG(buffer + 8, bigEndian), 32,
                         clientCache -> renderNumGlyphsCache);

      encodeCharData(encodeBuffer, buffer, 12, size,
                         bigEndian, clientCache);
      break;
    }
    case X_RenderSetPictureClipRectangles:
    {
      encodeBuffer.encodeXidValue(GetULONG(buffer + 4, bigEndian),
                         clientCache -> renderPictureCache);

      value = GetUINT(buffer + 8, bigEndian);

      encodeBuffer.encodeCachedValue(value - clientCache -> renderLastX, 16,
                         clientCache -> renderXCache, 8);

      clientCache -> renderLastX = value;

      value = GetUINT(buffer + 10, bigEndian);

      encodeBuffer.encodeCachedValue(value - clientCache -> renderLastY, 16,
                         clientCache -> renderYCache, 8);

      clientCache -> renderLastY = value;

      encodeIntData(encodeBuffer, buffer, 12, size,
                        bigEndian, clientCache);

      break;
    }
    case X_RenderCompositeGlyphs8:
    case X_RenderCompositeGlyphs16:
    case X_RenderCompositeGlyphs32:
    {
      encodeBuffer.encodeCachedValue(*(buffer + 4), 8,
                         clientCache -> renderOpCache);

      encodeBuffer.encodeXidValue(GetULONG(buffer + 8, bigEndian),
                         clientCache -> renderPictureCache);

      encodeBuffer.encodeXidValue(GetULONG(buffer + 12, bigEndian),
                         clientCache -> renderPictureCache);

      encodeBuffer.encodeCachedValue(GetULONG(buffer + 16, bigEndian), 32,
                         clientCache -> renderFormatCache);

      encodeBuffer.encodeCachedValue(GetULONG(buffer + 20, bigEndian), 29,
                         clientCache -> renderGlyphsetCache);

      value = GetUINT(buffer + 24, bigEndian);

      encodeBuffer.encodeCachedValue(value - clientCache -> renderLastX, 16,
                         clientCache -> renderXCache, 8);

      clientCache -> renderLastX = value;

      value = GetUINT(buffer + 26, bigEndian);

      encodeBuffer.encodeCachedValue(value - clientCache -> renderLastY, 16,
                         clientCache -> renderYCache, 8);

      clientCache -> renderLastY = value;

      encodeCharData(encodeBuffer, buffer, 28, size,
                         bigEndian, clientCache);
      break;
    }
    case X_RenderFillRectangles:
    {
      encodeBuffer.encodeCachedValue(*(buffer + 4), 8,
                         clientCache -> renderOpCache);

      encodeBuffer.encodeXidValue(GetULONG(buffer + 8, bigEndian),
                         clientCache -> renderPictureCache);

      encodeIntData(encodeBuffer, buffer, 12, size,
                        bigEndian, clientCache);

      break;
    }
    default:
    {
      #ifdef DEBUG
      *logofs << name() << ": Encoding unhandled render type "
              << (unsigned int) type << " with size " << size
              << ".\n" << logofs_flush;
      #endif

      encodeIntData(encodeBuffer, buffer, 4, size,
                        bigEndian, clientCache);
      break;
    }
  }

  #ifdef DEBUG
  *logofs << name() << ": Encoded full message.\n"
          << logofs_flush;
  #endif

  return 1;
}

int RenderGenericRequestStore::decodeMessage(DecodeBuffer &decodeBuffer, unsigned char *&buffer,
                                                 unsigned int &size, unsigned char type, int bigEndian,
                                                     WriteBuffer *writeBuffer, ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  #ifdef DEBUG
  *logofs << name() << ": Decoding full message.\n"
          << logofs_flush;
  #endif

  decodeBuffer.decodeCachedValue(size, 16,
                     clientCache -> renderLengthCache, 5);

  size <<= 2;

  buffer = writeBuffer -> addMessage(size);

  *(buffer + 1) = type;

  unsigned int value;

  #ifdef DEBUG
  *logofs << name() << ": Decoding full message. Type is "
          << (unsigned int) type << " size is " << size
          << ".\n" << logofs_flush;
  #endif

  switch (type)
  {
    case X_RenderCreatePicture:
    {
      decodeBuffer.decodeXidValue(value, clientCache -> renderPictureCache);

      clientCache -> renderLastId += value;
      clientCache -> renderLastId &= 0x1fffffff;

      PutULONG(clientCache -> renderLastId, buffer + 4, bigEndian);

      decodeBuffer.decodeXidValue(value, clientCache -> drawableCache);

      PutULONG(value, buffer + 8, bigEndian);

      decodeBuffer.decodeCachedValue(value, 32,
                         clientCache -> renderFormatCache);

      PutULONG(value, buffer + 12, bigEndian);

      decodeBuffer.decodeCachedValue(value, 32,
                         clientCache -> renderMaskCache);

      PutULONG(value, buffer + 16, bigEndian);

      decodeCharData(decodeBuffer, buffer, 20, size,
                         bigEndian, clientCache);
      break;
    }
    case X_RenderChangePicture:
    {
      decodeBuffer.decodeXidValue(value, clientCache -> renderPictureCache);

      PutULONG(value, buffer + 4, bigEndian);

      decodeBuffer.decodeCachedValue(value, 32,
                         clientCache -> renderMaskCache);

      PutULONG(value, buffer + 8, bigEndian);

      decodeCharData(decodeBuffer, buffer, 12, size,
                         bigEndian, clientCache);
      break;
    }
    case X_RenderFreePicture:
    {
      decodeBuffer.decodeXidValue(value, clientCache -> renderPictureCache);

      PutULONG(value, buffer + 4, bigEndian);

      decodeCharData(decodeBuffer, buffer, 8, size,
                         bigEndian, clientCache);
      break;
    }
    case X_RenderComposite:
    {
      decodeBuffer.decodeCachedValue(*(buffer + 4), 8,
                         clientCache -> renderOpCache);

      decodeBuffer.decodeXidValue(value, clientCache -> renderPictureCache);

      PutULONG(value, buffer + 8, bigEndian);

      decodeBuffer.decodeXidValue(value, clientCache -> renderPictureCache);

      PutULONG(value, buffer + 12, bigEndian);

      decodeBuffer.decodeXidValue(value, clientCache -> renderPictureCache);

      PutULONG(value, buffer + 16, bigEndian);

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderXCache, 8);

      clientCache -> renderLastX += value;
      clientCache -> renderLastX &= 0xffff;

      PutUINT(clientCache -> renderLastX, buffer + 20, bigEndian);

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderYCache, 8);

      clientCache -> renderLastY += value;
      clientCache -> renderLastY &= 0xffff;

      PutUINT(clientCache -> renderLastY, buffer + 22, bigEndian);

      decodeIntData(decodeBuffer, buffer, 24, size,
                        bigEndian, clientCache);
      break;
    }
    case X_RenderAddGlyphs:
    {
      decodeBuffer.decodeCachedValue(value, 29,
                         clientCache -> renderGlyphsetCache);

      PutULONG(value, buffer + 4, bigEndian);

      decodeBuffer.decodeCachedValue(value, 32,
                         clientCache -> renderNumGlyphsCache);

      PutULONG(value, buffer + 8, bigEndian);

      decodeCharData(decodeBuffer, buffer, 12, size,
                         bigEndian, clientCache);
      break;
    }
    case X_RenderSetPictureClipRectangles:
    {
      decodeBuffer.decodeXidValue(value, clientCache -> renderPictureCache);

      PutULONG(value, buffer + 4, bigEndian);

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderXCache, 8);

      clientCache -> renderLastX += value;
      clientCache -> renderLastX &= 0xffff;

      PutUINT(clientCache -> renderLastX, buffer + 8, bigEndian);

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderYCache, 8);

      clientCache -> renderLastY += value;
      clientCache -> renderLastY &= 0xffff;

      PutUINT(clientCache -> renderLastY, buffer + 10, bigEndian);

      decodeIntData(decodeBuffer, buffer, 12, size,
                        bigEndian, clientCache);
      break;
    }
    case X_RenderCompositeGlyphs8:
    case X_RenderCompositeGlyphs16:
    case X_RenderCompositeGlyphs32:
    {
      decodeBuffer.decodeCachedValue(*(buffer + 4), 8,
                         clientCache -> renderOpCache);

      decodeBuffer.decodeXidValue(value, clientCache -> renderPictureCache);

      PutULONG(value, buffer + 8, bigEndian);

      decodeBuffer.decodeXidValue(value, clientCache -> renderPictureCache);

      PutULONG(value, buffer + 12, bigEndian);

      decodeBuffer.decodeCachedValue(value, 32,
                         clientCache -> renderFormatCache);

      PutULONG(value, buffer + 16, bigEndian);

      decodeBuffer.decodeCachedValue(value, 29,
                         clientCache -> renderGlyphsetCache);

      PutULONG(value, buffer + 20, bigEndian);

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderXCache, 8);

      clientCache -> renderLastX += value;
      clientCache -> renderLastX &= 0xffff;

      PutUINT(clientCache -> renderLastX, buffer + 24, bigEndian);

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderYCache, 8);

      clientCache -> renderLastY += value;
      clientCache -> renderLastY &= 0xffff;

      PutUINT(clientCache -> renderLastY, buffer + 26, bigEndian);

      decodeCharData(decodeBuffer, buffer, 28, size,
                         bigEndian, clientCache);
      break;
    }
    case X_RenderFillRectangles:
    {
      decodeBuffer.decodeCachedValue(*(buffer + 4), 8,
                         clientCache -> renderOpCache);

      decodeBuffer.decodeXidValue(value, clientCache -> renderPictureCache);

      PutULONG(value, buffer + 8, bigEndian);

      decodeIntData(decodeBuffer, buffer, 12, size,
                        bigEndian, clientCache);
      break;
    }
    default:
    {
      #ifdef DEBUG
      *logofs << name() << ": Decoding unhandled render type "
              << (unsigned int) type << " with size " << size
              << ".\n" << logofs_flush;
      #endif

      decodeIntData(decodeBuffer, buffer, 4, size,
                        bigEndian, clientCache);
      break;
    }
  }

  #ifdef DEBUG
  *logofs << name() << ": Decoded full message.\n"
          << logofs_flush;
  #endif

  return 1;
}

void RenderGenericRequestStore::encodeData(EncodeBuffer &encodeBuffer, const unsigned char *buffer,
                                               unsigned int size, int bigEndian,
                                                   ChannelCache *channelCache) const
{
}

void RenderGenericRequestStore::decodeData(DecodeBuffer &decodeBuffer, unsigned char *buffer,
                                               unsigned int size, int bigEndian,
                                                   ChannelCache *channelCache) const
{
}

int RenderGenericRequestStore::parseIdentity(Message *message, const unsigned char *buffer,
                                                 unsigned int size, int bigEndian) const
{
  #ifdef DEBUG
  *logofs << name() << ": Parsing identity for message at "
          << this << ".\n" << logofs_flush;
  #endif

  RenderExtensionMessage *renderExtension = (RenderExtensionMessage *) message;

  unsigned char type = *(buffer + 1);

  renderExtension -> type = type;

  #ifdef DEBUG
  *logofs << name() << ": Parsing identity. Type is "
          << (unsigned int) renderExtension -> type << " size is "
          << renderExtension -> size_ << ".\n" << logofs_flush;
  #endif

  switch (type)
  {
    case X_RenderCreatePicture:
    {
      renderExtension -> src_id = GetULONG(buffer + 4,  bigEndian);
      renderExtension -> dst_id = GetULONG(buffer + 8,  bigEndian);

      renderExtension -> format = GetULONG(buffer + 12,  bigEndian);
      renderExtension -> mask   = GetULONG(buffer + 16,  bigEndian);

      parseCharData(message, buffer, 20, size, bigEndian);

      break;
    }
    case X_RenderChangePicture:
    {
      renderExtension -> src_id = GetULONG(buffer + 4,  bigEndian);
      renderExtension -> mask   = GetULONG(buffer + 8,  bigEndian);

      parseCharData(message, buffer, 12, size, bigEndian);

      break;
    }
    case X_RenderFreePicture:
    {
      renderExtension -> src_id = GetULONG(buffer + 4,  bigEndian);

      parseCharData(message, buffer, 8, size, bigEndian);

      break;
    }
    case X_RenderComposite:
    {
      renderExtension -> op = *(buffer + 4);

      renderExtension -> src_id = GetULONG(buffer + 8,  bigEndian);
      renderExtension -> msk_id = GetULONG(buffer + 12, bigEndian);
      renderExtension -> dst_id = GetULONG(buffer + 16, bigEndian);

      renderExtension -> src_x = GetUINT(buffer + 20, bigEndian);
      renderExtension -> src_y = GetUINT(buffer + 22, bigEndian);

      parseIntData(message, buffer, 24, size, bigEndian);

      break;
    }
    case X_RenderAddGlyphs:
    {
      renderExtension -> glyphset = GetULONG(buffer + 4, bigEndian);

      renderExtension -> num_glyphs = GetULONG(buffer + 8,  bigEndian);

      parseCharData(message, buffer, 12, size, bigEndian);

      break;
    }
    case X_RenderSetPictureClipRectangles:
    {
      renderExtension -> src_id = GetULONG(buffer + 4,  bigEndian);

      renderExtension -> src_x = GetUINT(buffer + 8,  bigEndian);
      renderExtension -> src_y = GetUINT(buffer + 10, bigEndian);

      parseIntData(message, buffer, 12, size, bigEndian);

      break;
    }
    case X_RenderCompositeGlyphs8:
    case X_RenderCompositeGlyphs16:
    case X_RenderCompositeGlyphs32:
    {
      renderExtension -> op = *(buffer + 4);

      renderExtension -> src_id = GetULONG(buffer + 8,  bigEndian);
      renderExtension -> dst_id = GetULONG(buffer + 12, bigEndian);

      renderExtension -> format   = GetULONG(buffer + 16, bigEndian);
      renderExtension -> glyphset = GetULONG(buffer + 20, bigEndian);

      renderExtension -> src_x = GetUINT(buffer + 24, bigEndian);
      renderExtension -> src_y = GetUINT(buffer + 26, bigEndian);

      parseCharData(message, buffer, 28, size, bigEndian);

      break;
    }
    case X_RenderFillRectangles:
    {
      renderExtension -> op = *(buffer + 4);

      renderExtension -> dst_id = GetULONG(buffer + 8, bigEndian);

      parseIntData(message, buffer, 12, size, bigEndian);

      break;
    }
    default:
    {
      #ifdef DEBUG
      *logofs << name() << ": Parsing unhandled render type "
              << (unsigned int) type << " with size " << size
              << ".\n" << logofs_flush;
      #endif

      parseIntData(message, buffer, 4, size, bigEndian);

      break;
    }
  }

  #ifdef DEBUG
  *logofs << name() << ": Parsed identity for message at "
          << this << ".\n" << logofs_flush;
  #endif

  return 1;
}

int RenderGenericRequestStore::unparseIdentity(const Message *message, unsigned char *buffer,
                                                   unsigned int size, int bigEndian) const
{
  #ifdef DEBUG
  *logofs << name() << ": Unparsing identity for message at "
          << this << ".\n" << logofs_flush;
  #endif

  RenderExtensionMessage *renderExtension = (RenderExtensionMessage *) message;

  unsigned char type = renderExtension -> type;

  *(buffer + 1) = type;

  #ifdef DEBUG
  *logofs << name() << ": Unparsing identity. Type is "
          << (unsigned int) renderExtension -> type << " size is "
          << renderExtension -> size_ << ".\n" << logofs_flush;
  #endif

  switch (type)
  {
    case X_RenderCreatePicture:
    {
      PutULONG(renderExtension -> src_id, buffer + 4,  bigEndian);
      PutULONG(renderExtension -> dst_id, buffer + 8, bigEndian);

      PutULONG(renderExtension -> format, buffer + 12, bigEndian);
      PutULONG(renderExtension -> mask,   buffer + 16, bigEndian);

      unparseCharData(message, buffer, 20, size, bigEndian);

      break;
    }
    case X_RenderChangePicture:
    {
      PutULONG(renderExtension -> src_id, buffer + 4,  bigEndian);
      PutULONG(renderExtension -> mask,   buffer + 8, bigEndian);

      unparseCharData(message, buffer, 12, size, bigEndian);

      break;
    }
    case X_RenderFreePicture:
    {
      PutULONG(renderExtension -> src_id, buffer + 4,  bigEndian);

      unparseCharData(message, buffer, 8, size, bigEndian);

      break;
    }
    case X_RenderComposite:
    {
      *(buffer + 4) = renderExtension -> op;

      PutULONG(renderExtension -> src_id, buffer + 8,  bigEndian);
      PutULONG(renderExtension -> msk_id, buffer + 12, bigEndian);
      PutULONG(renderExtension -> dst_id, buffer + 16, bigEndian);

      PutUINT(renderExtension -> src_x, buffer + 20, bigEndian);
      PutUINT(renderExtension -> src_y, buffer + 22, bigEndian);

      unparseIntData(message, buffer, 24, size, bigEndian);

      break;
    }
    case X_RenderAddGlyphs:
    {
      PutULONG(renderExtension -> glyphset, buffer + 4,  bigEndian);

      PutULONG(renderExtension -> num_glyphs, buffer + 8,  bigEndian);

      unparseCharData(message, buffer, 12, size, bigEndian);

      break;
    }
    case X_RenderSetPictureClipRectangles:
    {
      PutULONG(renderExtension -> src_id, buffer + 4,  bigEndian);

      PutUINT(renderExtension -> src_x, buffer + 8,  bigEndian);
      PutUINT(renderExtension -> src_y, buffer + 10, bigEndian);

      unparseIntData(message, buffer, 12, size, bigEndian);

      break;
    }
    case X_RenderCompositeGlyphs8:
    case X_RenderCompositeGlyphs16:
    case X_RenderCompositeGlyphs32:
    {
      *(buffer + 4) = renderExtension -> op;

      PutULONG(renderExtension -> src_id, buffer + 8,  bigEndian);
      PutULONG(renderExtension -> dst_id, buffer + 12, bigEndian);

      PutULONG(renderExtension -> format,   buffer + 16, bigEndian);
      PutULONG(renderExtension -> glyphset, buffer + 20, bigEndian);

      PutUINT(renderExtension -> src_x, buffer + 24, bigEndian);
      PutUINT(renderExtension -> src_y, buffer + 26, bigEndian);

      unparseCharData(message, buffer, 28, size, bigEndian);

      break;
    }
    case X_RenderFillRectangles:
    {
      *(buffer + 4) = renderExtension -> op;

      PutULONG(renderExtension -> dst_id, buffer + 8,  bigEndian);

      unparseIntData(message, buffer, 12, size, bigEndian);

      break;
    }
    default:
    {
      #ifdef DEBUG
      *logofs << name() << ": Unparsing unhandled render type "
              << (unsigned int) type << " with size " << size
              << ".\n" << logofs_flush;
      #endif

      unparseIntData(message, buffer, 4, size, bigEndian);

      break;
    }
  }

  #ifdef DEBUG
  *logofs << name() << ": Unparsed identity for message at "
          << this << ".\n" << logofs_flush;
  #endif

  return 1;
}

void RenderGenericRequestStore::identityChecksum(const Message *message, const unsigned char *buffer,
                                                     unsigned int size, md5_state_t *md5_state,
                                                         int bigEndian) const
{
  //
  // Include minor opcode in the checksum. As data
  // offset can be beyond the real end of message,
  // we need to include size or we will match any
  // message of size less or equal to data offset.
  //

  md5_append(md5_state, buffer + 1,  3);
}

void RenderGenericRequestStore::updateIdentity(EncodeBuffer &encodeBuffer, const Message *message,
                                                   const Message *cachedMessage,
                                                       ChannelCache *channelCache) const
{
  //
  // Encode the variant part.
  //

  #ifdef DEBUG
  *logofs << name() << ": Updating identity for message at "
          << this << ".\n" << logofs_flush;
  #endif

  RenderExtensionMessage *renderExtension       = (RenderExtensionMessage *) message;
  RenderExtensionMessage *cachedRenderExtension = (RenderExtensionMessage *) cachedMessage;

  ClientCache *clientCache = (ClientCache *) channelCache;

  unsigned char type = renderExtension -> type;

  #ifdef DEBUG
  *logofs << name() << ": Encoding update. Type is "
          << (unsigned int) renderExtension -> type << " size is "
          << renderExtension -> size_ << ".\n" << logofs_flush;
  #endif

  switch (type)
  {
    case X_RenderCreatePicture:
    {
      unsigned int picture = renderExtension -> src_id;

      encodeBuffer.encodeXidValue(picture - clientCache -> renderLastId,
                                      clientCache -> renderPictureCache);

      clientCache -> renderLastId = picture;

      cachedRenderExtension -> src_id = picture;

      encodeBuffer.encodeXidValue(renderExtension -> dst_id,
                         clientCache -> drawableCache);

      cachedRenderExtension -> dst_id = renderExtension -> dst_id;

      encodeBuffer.encodeCachedValue(renderExtension -> format, 32,
                         clientCache -> renderFormatCache);

      cachedRenderExtension -> format = renderExtension -> format;

      encodeBuffer.encodeCachedValue(renderExtension -> mask, 32,
                         clientCache -> renderMaskCache);

      cachedRenderExtension -> mask = renderExtension -> mask;

      updateCharData(encodeBuffer, message, cachedMessage, 20,
                         renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderChangePicture:
    {
      encodeBuffer.encodeXidValue(renderExtension -> src_id,
                         clientCache -> renderPictureCache);

      cachedRenderExtension -> src_id = renderExtension -> src_id;

      encodeBuffer.encodeCachedValue(renderExtension -> mask, 32,
                         clientCache -> renderMaskCache);

      cachedRenderExtension -> mask = renderExtension -> mask;

      updateCharData(encodeBuffer, message, cachedMessage, 12,
                         renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderFreePicture:
    {
      encodeBuffer.encodeXidValue(renderExtension -> src_id,
                         clientCache -> renderPictureCache);

      cachedRenderExtension -> src_id = renderExtension -> src_id;

      updateCharData(encodeBuffer, message, cachedMessage, 8,
                         renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderComposite:
    {
      encodeBuffer.encodeCachedValue(renderExtension -> op, 8,
                         clientCache -> renderOpCache);

      cachedRenderExtension -> op = renderExtension -> op;

      encodeBuffer.encodeXidValue(renderExtension -> src_id,
                         clientCache -> renderPictureCache);

      cachedRenderExtension -> src_id = renderExtension -> src_id;

      encodeBuffer.encodeXidValue(renderExtension -> msk_id,
                         clientCache -> renderPictureCache);

      cachedRenderExtension -> msk_id = renderExtension -> msk_id;

      encodeBuffer.encodeXidValue(renderExtension -> dst_id,
                         clientCache -> renderPictureCache);

      cachedRenderExtension -> dst_id = renderExtension -> dst_id;

      unsigned short int diff_x = renderExtension -> src_x - cachedRenderExtension -> src_x;

      encodeBuffer.encodeCachedValue(diff_x, 16,
                         clientCache -> renderXCache, 8);

      cachedRenderExtension -> src_x = renderExtension -> src_x;

      unsigned short int diff_y = renderExtension -> src_y - cachedRenderExtension -> src_y;

      encodeBuffer.encodeCachedValue(diff_y, 16,
                         clientCache -> renderYCache, 8);

      cachedRenderExtension -> src_y = renderExtension -> src_y;

      updateIntData(encodeBuffer, message, cachedMessage, 24,
                        renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderAddGlyphs:
    {
      encodeBuffer.encodeCachedValue(renderExtension -> glyphset, 29,
                         clientCache -> renderGlyphsetCache);

      cachedRenderExtension -> glyphset = renderExtension -> glyphset;

      encodeBuffer.encodeCachedValue(renderExtension -> num_glyphs, 32,
                         clientCache -> renderNumGlyphsCache);

      cachedRenderExtension -> num_glyphs = renderExtension -> num_glyphs;

      updateCharData(encodeBuffer, message, cachedMessage, 12,
                         renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderSetPictureClipRectangles:
    {
      encodeBuffer.encodeXidValue(renderExtension -> src_id,
                         clientCache -> renderPictureCache);

      cachedRenderExtension -> src_id = renderExtension -> src_id;

      unsigned short int diff_x = renderExtension -> src_x - cachedRenderExtension -> src_x;

      encodeBuffer.encodeCachedValue(diff_x, 16,
                         clientCache -> renderXCache, 8);

      cachedRenderExtension -> src_x = renderExtension -> src_x;

      unsigned short int diff_y = renderExtension -> src_y - cachedRenderExtension -> src_y;

      encodeBuffer.encodeCachedValue(diff_y, 16,
                         clientCache -> renderYCache, 8);

      cachedRenderExtension -> src_y = renderExtension -> src_y;

      updateIntData(encodeBuffer, message, cachedMessage, 12,
                        renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderCompositeGlyphs8:
    case X_RenderCompositeGlyphs16:
    case X_RenderCompositeGlyphs32:
    {
      encodeBuffer.encodeCachedValue(renderExtension -> op, 8,
                         clientCache -> renderOpCache);

      cachedRenderExtension -> op = renderExtension -> op;

      encodeBuffer.encodeXidValue(renderExtension -> src_id,
                         clientCache -> renderPictureCache);

      cachedRenderExtension -> src_id = renderExtension -> src_id;

      encodeBuffer.encodeXidValue(renderExtension -> dst_id,
                         clientCache -> renderPictureCache);

      cachedRenderExtension -> dst_id = renderExtension -> dst_id;

      encodeBuffer.encodeCachedValue(renderExtension -> format, 32,
                         clientCache -> renderFormatCache);

      cachedRenderExtension -> format = renderExtension -> format;

      encodeBuffer.encodeCachedValue(renderExtension -> glyphset, 29,
                         clientCache -> renderGlyphsetCache);

      cachedRenderExtension -> glyphset = renderExtension -> glyphset;

      unsigned short int diff_x = renderExtension -> src_x - cachedRenderExtension -> src_x;

      encodeBuffer.encodeCachedValue(diff_x, 16,
                         clientCache -> renderXCache, 8);

      cachedRenderExtension -> src_x = renderExtension -> src_x;

      unsigned short int diff_y = renderExtension -> src_y - cachedRenderExtension -> src_y;

      encodeBuffer.encodeCachedValue(diff_y, 16,
                         clientCache -> renderYCache, 8);

      cachedRenderExtension -> src_y = renderExtension -> src_y;

      updateCharData(encodeBuffer, message, cachedMessage, 28,
                         renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderFillRectangles:
    {
      encodeBuffer.encodeCachedValue(renderExtension -> op, 8,
                         clientCache -> renderOpCache);

      cachedRenderExtension -> op = renderExtension -> op;

      encodeBuffer.encodeXidValue(renderExtension -> dst_id,
                         clientCache -> renderPictureCache);

      cachedRenderExtension -> dst_id = renderExtension -> dst_id;

      updateIntData(encodeBuffer, message, cachedMessage, 12,
                        renderExtension -> size_, channelCache);
      break;
    }
    default:
    {
      #ifdef DEBUG
      *logofs << name() << ": Updating unhandled render type "
              << (unsigned int) type << " with size "
              << renderExtension -> size_ << ".\n"
              << logofs_flush;
      #endif

      updateIntData(encodeBuffer, message, cachedMessage, 4,
                        renderExtension -> size_, channelCache);
      break;
    }
  }

  #ifdef DEBUG
  *logofs << name() << ": Updated identity for message at "
          << this << ".\n" << logofs_flush;
  #endif
}

void RenderGenericRequestStore::updateIdentity(DecodeBuffer &decodeBuffer, const Message *message,
                                                   ChannelCache *channelCache) const
{
  #ifdef DEBUG
  *logofs << name() << ": Updating identity for message at "
          << this << ".\n" << logofs_flush;
  #endif

  RenderExtensionMessage *renderExtension = (RenderExtensionMessage *) message;

  ClientCache *clientCache = (ClientCache *) channelCache;

  unsigned int value;

  unsigned char type = renderExtension -> type;

  #ifdef DEBUG
  *logofs << name() << ": Decoding update. Type is "
          << (unsigned int) renderExtension -> type << " size is "
          << renderExtension -> size_ << ".\n" << logofs_flush;
  #endif

  switch (type)
  {
    case X_RenderCreatePicture:
    {
      decodeBuffer.decodeXidValue(value, clientCache -> renderPictureCache);

      clientCache -> renderLastId += value;
      clientCache -> renderLastId &= 0x1fffffff;

      renderExtension -> src_id = clientCache -> renderLastId;

      decodeBuffer.decodeXidValue(renderExtension -> dst_id,
                         clientCache -> drawableCache);

      decodeBuffer.decodeCachedValue(renderExtension -> format, 32,
                         clientCache -> renderFormatCache);

      decodeBuffer.decodeCachedValue(renderExtension -> mask, 32,
                         clientCache -> renderMaskCache);

      updateCharData(decodeBuffer, message, 20,
                         renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderChangePicture:
    {
      decodeBuffer.decodeXidValue(renderExtension -> src_id,
                         clientCache -> renderPictureCache);

      decodeBuffer.decodeCachedValue(renderExtension -> mask, 32,
                         clientCache -> renderMaskCache);

      updateCharData(decodeBuffer, message, 12,
                         renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderFreePicture:
    {
      decodeBuffer.decodeXidValue(renderExtension -> src_id,
                         clientCache -> renderPictureCache);

      updateCharData(decodeBuffer, message, 8,
                         renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderComposite:
    {
      decodeBuffer.decodeCachedValue(renderExtension -> op, 8,
                         clientCache -> renderOpCache);

      decodeBuffer.decodeXidValue(renderExtension -> src_id,
                         clientCache -> renderPictureCache);

      decodeBuffer.decodeXidValue(renderExtension -> msk_id,
                         clientCache -> renderPictureCache);

      decodeBuffer.decodeXidValue(renderExtension -> dst_id,
                         clientCache -> renderPictureCache);

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderXCache, 8);

      renderExtension -> src_x += value;
      renderExtension -> src_x &= 0xffff;

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderYCache, 8);

      renderExtension -> src_y += value;
      renderExtension -> src_y &= 0xffff;

      updateIntData(decodeBuffer, message, 24,
                        renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderAddGlyphs:
    {
      decodeBuffer.decodeCachedValue(renderExtension -> glyphset, 29,
                         clientCache -> renderGlyphsetCache);

      decodeBuffer.decodeCachedValue(renderExtension -> num_glyphs, 32,
                         clientCache -> renderNumGlyphsCache);

      updateCharData(decodeBuffer, message, 12,
                         renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderSetPictureClipRectangles:
    {
      decodeBuffer.decodeXidValue(renderExtension -> src_id,
                         clientCache -> renderPictureCache);

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderXCache, 8);

      renderExtension -> src_x += value;
      renderExtension -> src_x &= 0xffff;

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderYCache, 8);

      renderExtension -> src_y += value;
      renderExtension -> src_y &= 0xffff;

      updateIntData(decodeBuffer, message, 12,
                        renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderCompositeGlyphs8:
    case X_RenderCompositeGlyphs16:
    case X_RenderCompositeGlyphs32:
    {
      decodeBuffer.decodeCachedValue(renderExtension -> op, 8,
                         clientCache -> renderOpCache);

      decodeBuffer.decodeXidValue(renderExtension -> src_id,
                         clientCache -> renderPictureCache);

      decodeBuffer.decodeXidValue(renderExtension -> dst_id,
                         clientCache -> renderPictureCache);

      decodeBuffer.decodeCachedValue(renderExtension -> format, 32,
                         clientCache -> renderFormatCache);

      decodeBuffer.decodeCachedValue(renderExtension -> glyphset, 29,
                         clientCache -> renderGlyphsetCache);

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderXCache, 8);

      renderExtension -> src_x += value;
      renderExtension -> src_x &= 0xffff;

      decodeBuffer.decodeCachedValue(value, 16,
                   clientCache -> renderYCache, 8);

      renderExtension -> src_y += value;
      renderExtension -> src_y &= 0xffff;

      updateCharData(decodeBuffer, message, 28,
                         renderExtension -> size_, channelCache);
      break;
    }
    case X_RenderFillRectangles:
    {
      decodeBuffer.decodeCachedValue(renderExtension -> op, 8,
                         clientCache -> renderOpCache);

      decodeBuffer.decodeXidValue(renderExtension -> dst_id,
                         clientCache -> renderPictureCache);

      updateIntData(decodeBuffer, message, 12,
                        renderExtension -> size_, channelCache);
      break;
    }
    default:
    {
      #ifdef DEBUG
      *logofs << name() << ": Updating unhandled render type "
              << (unsigned int) type << " with size "
              << renderExtension -> size_ << ".\n"
              << logofs_flush;
      #endif

      updateIntData(decodeBuffer, message, 4,
                        renderExtension -> size_, channelCache);
      break;
    }
  }

  #ifdef DEBUG
  *logofs << name() << ": Updated identity for message at "
          << this << ".\n" << logofs_flush;
  #endif
}
