/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include "SetUnpackAlpha.h"

#include "ClientCache.h"

#include "EncodeBuffer.h"
#include "DecodeBuffer.h"

#include "WriteBuffer.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG
#undef  DUMP

//
// Constructors and destructors.
//

SetUnpackAlphaStore::SetUnpackAlphaStore(Compressor *compressor, Decompressor *decompressor)

  : MessageStore(compressor, decompressor)
{
  enableCache    = SETUNPACKALPHA_ENABLE_CACHE;
  enableData     = SETUNPACKALPHA_ENABLE_DATA;
  enableSplit    = SETUNPACKALPHA_ENABLE_SPLIT;
  enableCompress = SETUNPACKALPHA_ENABLE_COMPRESS;

  dataLimit  = SETUNPACKALPHA_DATA_LIMIT;
  dataOffset = SETUNPACKALPHA_DATA_OFFSET;

  cacheSlots          = SETUNPACKALPHA_CACHE_SLOTS;
  cacheThreshold      = SETUNPACKALPHA_CACHE_THRESHOLD;
  cacheLowerThreshold = SETUNPACKALPHA_CACHE_LOWER_THRESHOLD;

  messages_ -> resize(cacheSlots);

  for (T_messages::iterator i = messages_ -> begin();
           i < messages_ -> end(); i++)
  {
    *i = NULL;
  }

  temporary_ = NULL;
}

SetUnpackAlphaStore::~SetUnpackAlphaStore()
{
  for (T_messages::iterator i = messages_ -> begin();
           i < messages_ -> end(); i++)
  {
    destroy(*i);
  }

  destroy(temporary_);
}

//
// Here are the methods to handle messages' content.
//

int SetUnpackAlphaStore::encodeIdentity(EncodeBuffer &encodeBuffer, const unsigned char *buffer,
                                            const unsigned int size, int bigEndian,
                                                ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  #ifdef DEBUG
  *logofs << name() << ": Encoding full message identity.\n" << logofs_flush;
  #endif

  // Client.
  encodeBuffer.encodeCachedValue(*(buffer + 1), 8,
                     clientCache -> resourceCache);
  // Entries.
  encodeBuffer.encodeValue(GetULONG(buffer + 4, bigEndian), 32, 9);

  #ifdef DEBUG
  *logofs << name() << ": Encoded full message identity.\n" << logofs_flush;
  #endif

  return 1;
}

int SetUnpackAlphaStore::decodeIdentity(DecodeBuffer &decodeBuffer, unsigned char *&buffer,
                                            unsigned int &size, int bigEndian, WriteBuffer *writeBuffer,
                                                ChannelCache *channelCache) const
{
  ClientCache *clientCache = (ClientCache *) channelCache;

  #ifdef DEBUG
  *logofs << name() << ": Decoding full message identity.\n" << logofs_flush;
  #endif

  unsigned int  value;
  unsigned char cValue;

  // Client.
  decodeBuffer.decodeCachedValue(cValue, 8,
                     clientCache -> resourceCache);
  // Entries.
  decodeBuffer.decodeValue(value, 32, 9);

  size = RoundUp4(value) + 8;

  buffer = writeBuffer -> addMessage(size);

  *(buffer + 1) = cValue;

  PutULONG(value, buffer + 4, bigEndian);

  #ifdef DEBUG
  *logofs << name() << ": Decoded full message identity.\n" << logofs_flush;
  #endif

  return 1;
}

int SetUnpackAlphaStore::parseIdentity(Message *message, const unsigned char *buffer,
                                           unsigned int size, int bigEndian) const
{
  SetUnpackAlphaMessage *setUnpackAlpha = (SetUnpackAlphaMessage *) message;

  setUnpackAlpha -> client = *(buffer + 1);

  setUnpackAlpha -> entries = GetULONG(buffer + 4, bigEndian);

  #ifdef DEBUG
  *logofs << name() << ": Parsed identity for message at " << message << ".\n" << logofs_flush;
  #endif

  return true;
}

int SetUnpackAlphaStore::unparseIdentity(const Message *message, unsigned char *buffer,
                                             unsigned int size, int bigEndian) const
{
  SetUnpackAlphaMessage *setUnpackAlpha = (SetUnpackAlphaMessage *) message;

  *(buffer + 1) = setUnpackAlpha -> client;

  PutULONG(setUnpackAlpha -> entries, buffer + 4, bigEndian);

  #ifdef DEBUG
  *logofs << name() << ": Unparsed identity for message at " << message << ".\n" << logofs_flush;
  #endif

  return true;
}

void SetUnpackAlphaStore::dumpIdentity(const Message *message) const
{
  #ifdef DUMP

  SetUnpackAlphaMessage *setUnpackAlpha = (SetUnpackAlphaMessage *) message;

  *logofs << name() << ": Identity client "
          << (unsigned int) setUnpackAlpha -> client << " entries "
          << setUnpackAlpha -> entries << " size "
          << setUnpackAlpha -> size_ << ".\n";

  #endif
}

void SetUnpackAlphaStore::identityChecksum(const Message *message, const unsigned char *buffer,
                                               unsigned int size, int bigEndian) const
{
  md5_append(md5_state_, buffer + 4, 4);
}

void SetUnpackAlphaStore::updateIdentity(EncodeBuffer &encodeBuffer, const Message *message,
                                             const Message *cachedMessage,
                                                 ChannelCache *channelCache) const
{
  SetUnpackAlphaMessage *setUnpackAlpha       = (SetUnpackAlphaMessage *) message;
  SetUnpackAlphaMessage *cachedSetUnpackAlpha = (SetUnpackAlphaMessage *) cachedMessage;

  ClientCache *clientCache = (ClientCache *) channelCache;

  encodeBuffer.encodeCachedValue(setUnpackAlpha -> client, 8,
                     clientCache -> resourceCache);

  cachedSetUnpackAlpha -> client = setUnpackAlpha -> client;

  //
  // TODO: Should not send entries as
  // they are included in the identity
  // checksum.
  //

  if (cachedSetUnpackAlpha -> entries != setUnpackAlpha -> entries)
  {
    #ifdef TEST
    *logofs << name() << ": Encoding value " << setUnpackAlpha -> entries
            << " as entries field.\n" << logofs_flush;
    #endif

    encodeBuffer.encodeBool(1);

    encodeBuffer.encodeValue(setUnpackAlpha -> entries, 32, 9);

    cachedSetUnpackAlpha -> entries = setUnpackAlpha -> entries;
  }
  else
  {
    encodeBuffer.encodeBool(0);
  }
}

void SetUnpackAlphaStore::updateIdentity(DecodeBuffer &decodeBuffer, const Message *message,
                                             ChannelCache *channelCache) const
{
  SetUnpackAlphaMessage *setUnpackAlpha = (SetUnpackAlphaMessage *) message;

  ClientCache *clientCache = (ClientCache *) channelCache;

  unsigned int value;

  decodeBuffer.decodeCachedValue(setUnpackAlpha -> client, 8,
                     clientCache -> resourceCache);

  decodeBuffer.decodeBool(value);

  if (value)
  {
    decodeBuffer.decodeValue(value, 32, 9);

    setUnpackAlpha -> entries = value;

    #ifdef DEBUG
    *logofs << name() << ": Decoded value " << setUnpackAlpha -> entries
            << " as entries field.\n" << logofs_flush;
    #endif
  }
}
