/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <fstream.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <utime.h>

#include "Split.h"

#include "Control.h"
#include "Statistics.h"

#include "Compressor.h"
#include "Decompressor.h"

//
// We need message store info.
//

#include "Message.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

//
// Define this to know how many messages
// are allocated and deallocated.
//

#undef  REFERENCES

//
// Counters used for store control.
//

int SplitStore::totalSplitStorageSize_;

//
// This is used for reference count.
//

#ifdef REFERENCES

int Split::references_ = 0;

#endif

Split *SplitStore::push(MessageStore *store, int client,  int position,
                            md5_byte_t *checksum, const unsigned char *buffer,
                                const int size)
{
  #ifdef TEST
  *logofs << "SplitStore: Adding message OPCODE#"
          << (unsigned int) store -> opcode() << " with client "
          << client << " position " << position << " and checksum ["
          << DumpChecksum(checksum) << "]" << ".\n"
          << logofs_flush;
  #endif

  Split *split = new Split();

  split -> store_    = store;
  split -> client_   = client;
  split -> position_ = position;

  //
  // If checksum is provided save it
  // together with message.
  //

  if (checksum != NULL)
  {
    split -> checksum_ = new md5_byte_t[MD5_LENGTH];

    memcpy(split -> checksum_, checksum, MD5_LENGTH);
  }

  split -> size_ = size - store -> dataOffset;

  try
  {
    split -> data_.resize(split -> size_);
  }
  catch (exception e)
  {
    #ifdef PANIC
    *logofs << "SplitStore: PANIC! Exception caught resizing data."
            << " Error is '" << e.what() << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Exception caught resizing data."
         << " Error is '" << e.what() << "'.\n";

    return NULL;
  }

  memcpy(split -> data_.begin(), buffer + store -> dataOffset, 
             size - store -> dataOffset);

  splits_ -> push_back(split);

  splitStorageSize_ += getNodeSize(split);

  totalSplitStorageSize_ += getNodeSize(split);

  #ifdef TEST
  *logofs << "SplitStore: Size of split store after push is "
          << splitStorageSize_ << "/" << totalSplitStorageSize_
          << ".\n" << logofs_flush;
  #endif

  return split;
}

//
// This is called at the encoding side.
//

Split *SplitStore::add(MessageStore *store, int client, int position,
                           const unsigned char *buffer, const int size)
{
  Split *split = push(store, client, position, NULL, buffer, size);

  if (split == NULL)
  {
    return NULL;
  }

  if (control -> CollectStatistics)
  {
    statistics -> addSplit();
  }

  return split;
}

//
// This is called at decoding side. If checksum
// is provided, the message can be searched on
// disk. If message is found, an abort event is
// later sent to interrupt message streaming.
//

Split *SplitStore::add(MessageStore *store, int position, md5_byte_t *checksum,
                           unsigned char *buffer, const int size)
{
  //
  // TODO: We could avoid to cleanup the buffer
  // at receiving side but in this case we would
  // be unable to detect a failure in unsplit.
  //

  memset(buffer + store -> dataOffset, 0x88, size - store -> dataOffset);

  Split *split = push(store, nothing, position, checksum, buffer, size);

  if (split == NULL)
  {
    return NULL;
  }

  //
  // Set flags to give channel a chance
  // to load data from disk and send the
  // abort message.
  //

  if (checksum != NULL)
  {
    split -> abort_ = 1;
  }

  if (control -> CollectStatistics)
  {
    statistics -> addSplit();
  }

  return split;
}

int SplitStore::send(EncodeBuffer &encodeBuffer, int packetSize)
{
  //
  // No split completed so far.
  //

  lastClient_   = nothing;
  lastRequest_  = nothing;
  lastPosition_ = nothing;

  lastCompletion_ = 0;

  if (splits_ -> size() == 0)
  {
    #ifdef TEST
    *logofs << "SplitStore: WARNING! No splits are currently available.\n" 
            << logofs_flush;
    #endif

    return 0;
  }

  //
  // TODO: Start must always be executed on split, even
  // in case split will later be aborted. This gives a
  // chance to the encoding side to calculate the size
  // of compressed messages. In theory this could be
  // avoided, and decoding side could include such info
  // in the abort event. On the other hand, keeping the
  // start procedure on both sides simplifies noticeably
  // the algorithm and reduces the risk of incompatibi-
  // lities with the previous implementation.
  // 

  if (current_ == splits_ -> end())
  {
    start(encodeBuffer, packetSize);
  }

  //
  // If we have matched the checksum received from
  // the remote side then we must abort the current
  // split, else we can send another block of data
  // to the remote peer.
  //

  Split *split = *current_;

  unsigned int abort = 0;

  if (control -> isProtoStep3() == 1 &&
          (control -> ImageCacheEnableLoad > 0 ||
               control -> ImageCacheEnableSave > 0))
  {
    if (split -> abort_ == 1)
    {
      abort = 1;
    }

    encodeBuffer.encodeBool(abort);
  }

  if (abort == 1)
  {
    #ifdef TEST
    *logofs << "SplitStore: Aborting split for OPCODE#"
            << (unsigned int) split -> store_ -> opcode()
            << " at position " << split -> position_ << " with "
            << (split -> data_.size() - split -> next_)
            << " bytes to go out of " << split -> data_.size()
            << ".\n" << logofs_flush;
    #endif

    if (control -> CollectStatistics)
    {
      statistics -> addSplitAborted();

      statistics -> addSplitAbortedBytesOut(split -> data_.size() - split -> next_);
    }

    split -> next_ = split -> data_.size();
  }
  else
  {
    unsigned int count = packetSize;

    if (split -> next_ + packetSize > ((int) split -> data_.size()))
    {
      count = split -> data_.size() - split -> next_;
    }

    #ifdef TEST
    *logofs << "SplitStore: Going to send split of " << count
            << " bytes " << "for position " << split -> position_
            << ". Data size is " << split -> data_.size() << " ("
            << split -> size_ << "/" << split -> c_size_ << "), "
            << split -> data_.size() - (split -> next_ + count)
            << " to go.\n" << logofs_flush;
    #endif

    encodeBuffer.encodeValue(count, 32, 10);

    encodeBuffer.encodeMemory(split -> data_.begin() + split -> next_, count);

    split -> next_ += count;
  }

  //
  // Was data completely transferred? We are the
  // sending side. We must update the message in
  // store, even if split was aborted.
  //

  if (split -> next_ == ((int) split -> data_.size()))
  {
    #ifdef TEST
    *logofs << "SplitStore: Going to update size of object "
            << "at position " << split -> position_
            << ".\n" << logofs_flush;
    #endif

    split -> store_ -> updateData(split -> position_, split -> size_,
                                      split -> c_size_);

    #ifdef TEST
    *logofs << "SplitStore: Going to remove split from the list.\n"
            << logofs_flush;
    #endif

    //
    // Remove the split from list.
    //

    splits_ -> pop_front();

    splitStorageSize_ -= getNodeSize(split);

    totalSplitStorageSize_ -= getNodeSize(split);

    #ifdef TEST
    *logofs << "SplitStore: Size of split store after send is "
            << splitStorageSize_ << "/" << totalSplitStorageSize_
            << ".\n" << logofs_flush;
    #endif

    #ifdef TEST

    if (splits_ -> size() == 0)
    {
      if (splitStorageSize_ != 0)
      {
        #ifdef PANIC
        *logofs << "SplitStore: PANIC! Wrong calculation for split "
                << "data size. It is " << splitStorageSize_ 
                << " while should be 0.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Wrong calculation for split "
             << "data size. It is " << splitStorageSize_
             << " while should be 0.\n";
      }
    }

    #endif

    //
    // Reset current position to the
    // end of repository.
    //

    current_ = splits_ -> end();

    //
    // Set end of split information so
    // it can be retrieved by channel.
    //

    lastClient_   = split -> client_;
    lastRequest_  = split -> store_ -> opcode();
    lastPosition_ = split -> position_;

    //
    // Set if split was marked as the
    // last one handled in a start/stop
    // split sequence.
    //

    lastCompletion_ = split -> completion_;

    delete split;
  }

  return 1;
}

int SplitStore::start(EncodeBuffer &encodeBuffer, int packetSize)
{
  #ifdef TEST
  *logofs << "SplitStore: Going to send a new split to the remote side.\n"
          << logofs_flush;
  #endif

  //
  // Get element at top of the list.
  //

  current_ = splits_ -> begin();

  Split *split = *current_;

  //
  // See if data compression is enabled.
  //

  if (split -> store_ -> enableCompress)
  {
    //
    // TODO: This routine must be reviewed. There is
    // no need to send a boolean and the new size as
    // such information is already sent by the static
    // compressor if data is reduced in size.
    //

    unsigned int compressedDataSize = 0;
    unsigned char *compressedData   = NULL;

    if (control -> LocalDataCompression &&
            (compressor_ -> compressBuffer(split -> data_.begin(), split -> size_,
                                               compressedData, compressedDataSize)))
    {
      //
      // Replace data with the one in compressed form.
      //

      #ifdef TEST
      *logofs << "SplitStore: Splitted data of size " << split -> size_
              << " has been compressed to " << compressedDataSize 
              << " bytes.\n" << logofs_flush;
      #endif

      try
      {
        split -> data_.clear();

        split -> data_.resize(compressedDataSize);

        memcpy(split -> data_.begin(), compressedData, compressedDataSize);

        split -> c_size_ = compressedDataSize;
      }
      catch (exception e)
      {
        #ifdef PANIC
        *logofs << "SplitStore: PANIC! Exception caught resizing data."
                << " Error is '" << e.what() << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Exception caught resizing data."
             << " Error is '" << e.what() << "'.\n";

        return -1;
      }

      //
      // Inform peer that data is compressed
      // and send the new size.
      //

      encodeBuffer.encodeBool(1);

      int diffSize = split -> c_size_ - split -> store_ -> lastResize;

      split -> store_ -> lastResize = split -> c_size_;

      encodeBuffer.encodeValue(diffSize, 32, 14);

      #ifdef TEST
      *logofs << "SplitStore: Signaled " << split -> c_size_
              << " bytes of compressed data for this message.\n" << logofs_flush;
      #endif
    }
    else
    {
      encodeBuffer.encodeBool(0);
    }
  }

  return 1;
}

int SplitStore::start(DecodeBuffer &decodeBuffer)
{
  #ifdef TEST
  *logofs << "SplitStore: Going to receive a new split from the remote side.\n"
          << logofs_flush;
  #endif

  //
  // Get element at top of the list.
  //

  current_ = splits_ -> begin();

  Split *split = *current_;

  //
  // TODO: See comment on corresponding
  // routine at encoding side.
  //

  if (split -> store_ -> enableCompress)
  {
    //
    // Find out if data has been compressed.
    //

    unsigned int compressed;

    decodeBuffer.decodeBool(compressed);

    if (compressed)
    {
      //
      // Get compressed size.
      //

      unsigned int diffSize;

      decodeBuffer.decodeValue(diffSize, 32, 14);

      split -> store_ -> lastResize += diffSize;

      #ifdef TEST
      *logofs << "SplitStore: Splitted data of size " << split -> size_
              << " was compressed to " << split -> store_ -> lastResize
              << " bytes.\n" << logofs_flush;
      #endif

      try
      {
        split -> data_.clear();

        split -> data_.resize(split -> store_ -> lastResize);

        split -> c_size_ = split -> store_ -> lastResize;
      }
      catch (exception e)
      {
        #ifdef PANIC
        *logofs << "SplitStore: PANIC! Exception caught resizing data."
                << " Error is '" << e.what() << "'.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Exception caught resizing data."
             << " Error is '" << e.what() << "'.\n";

        return -1;
      }
    }
  }

  return 1;
}

int SplitStore::receive(DecodeBuffer &decodeBuffer)
{
  //
  // No split completed so far.
  //

  lastClient_   = nothing;
  lastRequest_  = nothing;
  lastPosition_ = nothing;

  lastCompletion_ = 0;

  if (splits_ -> size() == 0)
  {
    #ifdef WARNING

    *logofs << "SplitStore: WARNING! Called receive function with no splits available.\n" 
            << logofs_flush;

    cerr << "Warning" << ": Called receive function with no splits available.\n";

    #endif

    return 0;
  }

  if (current_ == splits_ -> end())
  {
    start(decodeBuffer);
  }

  //
  // Check first if split was aborted, else add
  // any new data to message being recomposed.
  //

  Split *split = *current_;

  unsigned int abort = 0;

  if (control -> isProtoStep3() == 1 &&
          (control -> ImageCacheEnableLoad > 0 ||
               control -> ImageCacheEnableSave > 0))
  {
    decodeBuffer.decodeBool(abort);
  }

  if (abort == 1)
  {
    #ifdef TEST
    *logofs << "SplitStore: Aborting split for OPCODE#"
            << (unsigned int) split -> store_ -> opcode()
            << " at position " << split -> position_ << " with "
            << (split -> data_.size() - split -> next_)
            << " bytes to go out of " << split -> data_.size()
            << ".\n" << logofs_flush;
    #endif

    load(split);

    if (control -> CollectStatistics)
    {
      statistics -> addSplitAborted();

      statistics -> addSplitAbortedBytesOut(split -> data_.size() - split -> next_);
    }

    split -> next_ = split -> data_.size();
  }
  else
  {
    //
    // Get size of packet.
    //

    unsigned int count;

    decodeBuffer.decodeValue(count, 32, 10);

    if (split -> next_ + count > split -> data_.size())
    {
      #ifdef PANIC
      *logofs << "SplitStore: PANIC! Provided data is larger "
              << "than the allocated buffer.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Provided data is larger "
           << "than the allocated buffer.\n";

      return -1;
    }

    //
    // Decode the packet and update
    // the split.
    //

    #ifdef TEST
    *logofs << "SplitStore: Going to receive " << count
            << " bytes " << "for position " << split -> position_
            << ". Data size is " << split -> data_.size() << " ("
            << split -> size_ << "/" << split -> c_size_ << "), "
            << split -> data_.size() - (split -> next_ + count)
            << " to go.\n" << logofs_flush;
    #endif

    memcpy(split -> data_.begin() + split -> next_, 
               decodeBuffer.decodeMemory(count), count);

    split -> next_ += count;
  }

  //
  // Is unsplit complete?
  //

  if (split -> next_ == ((int) split -> data_.size()))
  {
    //
    // If split was aborted, we should have
    // message both on disk and in store in
    // its final format.
    //

    if (abort == 0)
    {
      //
      // If persistent cache is enabled and we
      // have a valid checksum, save split data
      // on disk.
      //

      save(split);

      //
      // Commit data to the message store.
      //

      finish(split);
    }

    //
    // End of current split reached.
    //

    lastClient_   = split -> client_;
    lastRequest_  = split -> store_ -> opcode();
    lastPosition_ = split -> position_;

    #ifdef TEST
    *logofs << "SplitStore: Going to remove split from the list.\n"
            << logofs_flush;
    #endif

    //
    // Remove the split from list.
    //

    splits_ -> pop_front();

    splitStorageSize_ -= getNodeSize(split);

    totalSplitStorageSize_ -= getNodeSize(split);

    #ifdef TEST
    *logofs << "SplitStore: Size of split store after receive is "
            << splitStorageSize_ << "/" << totalSplitStorageSize_
            << ".\n" << logofs_flush;
    #endif

    delete split;

    #ifdef TEST

    if (splits_ -> size() == 0)
    {
      if (splitStorageSize_ != 0)
      {
        #ifdef PANIC
        *logofs << "SplitStore: PANIC! Wrong calculation for split "
                << " data size. It is " << splitStorageSize_ 
                << " while should be 0.\n" << logofs_flush;
        #endif

        cerr << "Error" << ": Wrong calculation for split "
             << " data size. It is " << splitStorageSize_
             << " while should be 0.\n";
      }
    }

    #endif

    //
    // Reset current position to the
    // end of repository.
    //

    current_ = splits_ -> end();
  }

  return 1;
}

int SplitStore::pop()
{
  lastClient_   = nothing;
  lastRequest_  = nothing;
  lastPosition_ = nothing;

  lastCompletion_ = 0;

  if (splits_ -> size() == 0)
  {
    return 0;
  }

  if (current_ == splits_ -> end())
  {
    current_ = splits_ -> begin();
  }

  Split *split = *current_;

  //
  // Remove top element from list. Leave data of message
  // in store unchanged. This is likely to be the dummy
  // data that was saved by channel at the time split
  // was started.
  //

  #ifdef TEST
  *logofs << "SplitStore: Going to remove object at position "
          << split -> position_ << ".\n" << logofs_flush;
  #endif

  splits_ -> pop_front();

  splitStorageSize_ -= getNodeSize(split);

  totalSplitStorageSize_ -= getNodeSize(split);

  #ifdef TEST
  *logofs << "SplitStore: Size of split store after pop is "
          << splitStorageSize_ << "/" << totalSplitStorageSize_
          << ".\n" << logofs_flush;
  #endif

  #ifdef TEST

  if (splits_ -> size() == 0)
  {
    if (splitStorageSize_ != 0)
    {
      #ifdef PANIC
      *logofs << "SplitStore: PANIC! Wrong calculation for split "
              << " data size. It is " << splitStorageSize_ 
              << " while should be 0.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Wrong calculation for split "
           << " data size. It is " << splitStorageSize_
           << " while should be 0.\n";
    }
  }

  #endif

  //
  // Reset current position to the
  // end of repository.
  //

  current_ = splits_ -> end();

  //
  // End of current split reached.
  //

  lastClient_   = split -> client_;
  lastRequest_  = split -> store_ -> opcode();
  lastPosition_ = split -> position_;

  //
  // Set if split was marked as the
  // last one handled in a start/stop
  // split sequence.
  //

  lastCompletion_ = split -> completion_;

  delete split;

  #ifdef TEST
  *logofs << "SplitStore: Removed split at head of the list. "
          << "Client is " << lastClient_ << " request is "
          << lastRequest_ << " position is " << lastPosition_
          << " completion is " << lastCompletion_ << ".\n"
          << logofs_flush;
  #endif

  return 1;
}

int SplitStore::finish(Split *split)
{
  //
  // We recomposed the full message so we
  // can finally update message stored in
  // cache.
  //

  #ifdef TEST
  *logofs << "SplitStore: Going to update object at position "
          << split -> position_ << " with data size "
          << split -> size_ << " and compressed data size "
          << split -> c_size_ << ".\n" << logofs_flush;
  #endif

  split -> store_ -> updateData(split -> position_, split -> data_.begin(), 
                                   split -> size_, split -> c_size_);

  return 1;
}

const char *SplitStore::name(const md5_byte_t *checksum)
{
  if (checksum == NULL)
  {
    return NULL;
  }

  char *pathName = control -> ImageCachePath;

  if (pathName == NULL)
  {
    #ifdef PANIC
    *logofs << "SplitStore: PANIC! Cannot determine directory of "
            << "NX image files.\n" << logofs_flush;
    #endif

    return NULL;
  }

  int pathSize = strlen(pathName);

  //
  // File name is "[path][/I-c/I-][checksum][\0]",
  // where c is the first hex digit of checksum.
  //

  int nameSize = pathSize + 7 + MD5_LENGTH * 2 + 1;

  char *fileName = new char[nameSize];

  if (fileName == NULL)
  {
    #ifdef PANIC
    *logofs << "SplitStore: PANIC! Cannot allocate space for "
            << "NX image file name.\n" << logofs_flush;
    #endif

    return NULL;
  }

  strcpy(fileName, pathName);

  sprintf(fileName + pathSize, "/I-%1X/I-",
              *((unsigned char *) checksum) >> 4);

  for (unsigned int i = 0; i < MD5_LENGTH; i++)
  {
    sprintf(fileName + pathSize + 7 + (i * 2), "%02X",
                ((unsigned char *) checksum)[i]);
  }

  return fileName;
}

int SplitStore::find(Split *split)
{
  if (control -> ImageCacheEnableLoad == 0)
  {
    return 0;
  }

  const char *fileName = name(split -> checksum_);

  if (fileName == NULL)
  {
    return 0;
  }

  #ifdef DEBUG
  *logofs << "SplitStore: Going to find split OPCODE#"
          << (unsigned int) split -> store_ -> opcode()
          << " in file '" << fileName << "'.\n"
          << logofs_flush;
  #endif

  //
  // Check if file exists and change, at the same
  // time, file's access and modification time to
  // prevent its deletion.
  //

  if (utime(fileName, NULL) == 0)
  {
    #ifdef TEST
    *logofs << "SplitStore: Found split OPCODE#"
            << (unsigned int) split -> store_ -> opcode()
            << " in file '" << fileName << "'.\n"
            << logofs_flush;
    #endif

    delete [] fileName;

    return 1;
  }

  #ifdef TEST
  *logofs << "SplitStore: Can't find file '" << fileName
          << "' on disk.\n" << logofs_flush;
  #endif

  delete [] fileName;

  return 0;
}

int SplitStore::save(Split *split)
{
  if (control -> ImageCacheEnableSave == 0)
  {
    return 0;
  }

  const char *fileName = name(split -> checksum_);

  if (fileName == NULL)
  {
    return 0;
  }

  #ifdef DEBUG
  *logofs << "SplitStore: Going to save split OPCODE#"
          << (unsigned int) split -> store_ -> opcode()
          << " to file '" << fileName << "' with size "
          << split -> size_ << " and compressed size "
          << split -> c_size_ << ".\n" << logofs_flush;
  #endif

  //
  // Check if file already exists and that
  // split is big enough to be worth saving
  // it on disk.
  //

  struct stat fileStat;

  if (split -> size_ < control -> SplitDataThreshold ||
          (split -> c_size_ > 0 && split -> c_size_ <
               control -> SplitDataThreshold) ||
                   stat(fileName, &fileStat) == 0)
  {
    #ifdef TEST
    *logofs << "SplitStore: Skipping save as split is too small "
            << "or already present on disk.\n" << logofs_flush;
    #endif

    delete [] fileName;

    return 0;
  }

  ostream *fileStream = new ofstream(fileName, ios::out | ios::binary);

  if (fileStream == NULL || fileStream -> fail())
  {
    #ifdef PANIC
    *logofs << "SplitStore: PANIC! Cannot open file '" << fileName
            << "' for output.\n" << logofs_flush;
    #endif

    delete [] fileName;

    return -1;
  }

  unsigned char *fileHeader = new unsigned char[SPLIT_HEADER_SIZE];

  if (fileHeader == NULL)
  {
    #ifdef PANIC
    *logofs << "SplitStore: PANIC! Cannot allocate space for "
            << "NX image header.\n" << logofs_flush;
    #endif

    delete fileStream;

    delete [] fileName;

    return -1;
  }

  //
  // Leave 3 bytes for future use. Consider
  // also that, on some CPU, we can't write
  // integers if they are not word-aligned.
  //

  *fileHeader = split -> store_ -> opcode();

  *(fileHeader + 1) = 0;
  *(fileHeader + 2) = 0;
  *(fileHeader + 3) = 0;

  PutULONG(split -> size_,   fileHeader + 4, false);
  PutULONG(split -> c_size_, fileHeader + 8, false);

  int size = (split -> c_size_ > 0 ? split -> c_size_ : split -> size_);

  if (PutData(fileStream, fileHeader, SPLIT_HEADER_SIZE) < 0 ||
          PutData(fileStream, split -> data_.begin(), size) < 0)
  {
    #ifdef PANIC
    *logofs << "SplitStore: PANIC! Cannot write to NX image file '"
            << fileName << "'.\n" << logofs_flush;
    #endif

    delete fileStream;

    unlink(fileName);

    delete [] fileName;
    delete [] fileHeader;

    return -1;
  }

  #ifdef TEST
  *logofs << "SplitStore: Saved split to file '" << fileName
          << "' with data size " << split -> size_
          << " and compressed data size " << split -> c_size_
          << ".\n" << logofs_flush;
  #endif

  delete fileStream;

  delete [] fileName;
  delete [] fileHeader;

  return 1;
}

int SplitStore::load(Split *split)
{
  if (control -> ImageCacheEnableLoad == 0)
  {
    return 0;
  }

  const char *fileName = name(split -> checksum_);

  if (fileName == NULL)
  {
    return 0;
  }

  #ifdef DEBUG
  *logofs << "SplitStore: Going to load split OPCODE#"
          << (unsigned int) split -> store_ -> opcode()
          << " from file '" << fileName << "'.\n"
          << logofs_flush;
  #endif

  istream *fileStream = new ifstream(fileName, ios::in | ios::binary);

  if (fileStream == NULL || fileStream -> fail())
  {
    #ifdef PANIC
    *logofs << "SplitStore: PANIC! Can't find image file '"
            << fileName  << "' on disk.\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Can't find image file '"
         << fileName << "' on disk.\n";

    delete [] fileName;

    return -1;
  }

  unsigned char *fileHeader = new unsigned char[SPLIT_HEADER_SIZE];

  if (fileHeader == NULL)
  {
    #ifdef PANIC
    *logofs << "SplitStore: PANIC! Cannot allocate space for "
            << "NX image header.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Cannot allocate space for "
         << "NX image header.\n";

    delete fileStream;

    delete [] fileName;

    return -1;
  }

  if (GetData(fileStream, fileHeader, SPLIT_HEADER_SIZE) < 0)
  {
    #ifdef PANIC
    *logofs << "SplitStore: PANIC! Cannot read header from "
            << "NX image file '" << fileName << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Cannot read header from "
         << "NX image file '" << fileName << "'.\n";

    delete fileStream;

    unlink(fileName);

    delete [] fileName;
    delete [] fileHeader;

    return -1;
  }

  unsigned char fileOpcode;

  unsigned int fileSize;
  unsigned int fileCSize;

  fileOpcode = *fileHeader;

  fileSize  = GetULONG(fileHeader + 4, false);
  fileCSize = GetULONG(fileHeader + 8, false);

  if (fileOpcode != split -> store_ -> opcode() ||
          fileSize != (unsigned int) split -> size_ ||
              fileSize > MESSAGE_DATA_LIMIT ||
                  fileCSize != (unsigned int) split -> c_size_ ||
                      fileCSize > MESSAGE_DATA_LIMIT)
  {
    #ifdef TEST
    *logofs << "SplitStore: PANIC! Corrupted image file '" << fileName
            << "'. Expected " << (unsigned int) split -> store_ -> opcode()
            << "/" << split -> size_ << "/" << split -> c_size_ << " found "
            << (unsigned int) fileOpcode << "/" << fileSize << "/"
            << fileCSize << ".\n" << logofs_flush;
    #endif

    cerr << "Warning" << ": Corrupted image file '" << fileName
         << "'. Expected " << (unsigned int) split -> store_ -> opcode()
         << "/" << split -> size_ << "/" << split -> c_size_ << " found "
         << (unsigned int) fileOpcode << "/" << fileSize << "/"
         << fileCSize << ".\n";

    delete fileStream;

    unlink(fileName);

    delete [] fileName;
    delete [] fileHeader;

    return -1;
  }

  unsigned int dataSize;

  if (fileCSize > 0)
  {
    dataSize = fileCSize;
  }
  else
  {
    dataSize = fileSize;
  }

  unsigned char *fileData = new unsigned char[dataSize];

  if (fileData == NULL || GetData(fileStream, fileData, dataSize) < 0)
  {
    #ifdef PANIC
    *logofs << "SplitStore: PANIC! Cannot read data from "
            << "NX image file '" << fileName << "'.\n"
            << logofs_flush;
    #endif

    cerr << "Warning" << ": Cannot read data from "
         << "NX image file '" << fileName << "'.\n";

    delete fileStream;

    unlink(fileName);

    delete [] fileName;
    delete [] fileHeader;

    return -1;
  }

  #ifdef TEST
  *logofs << "SplitStore: Going to update object at position "
          << split -> position_ << " with data size " << fileSize
          << " and compressed data size " << fileCSize
          << ".\n" << logofs_flush;
  #endif

  split -> store_ -> updateData(split -> position_, fileData, fileSize, fileCSize);

  delete fileStream;

  delete [] fileData;

  delete [] fileName;
  delete [] fileHeader;

  return 1;
}
