/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef Statistics_H
#define Statistics_H

#include "Misc.h"

class Proxy;

//
// Opcode 255 is for generic requests, 1 is for generic
// replies (those which haven't a specific differential
// encoding), opcode 0 is for any message from sync,
// samba, http, multimedia and keyboard channels.
//

#define STATISTICS_OPCODE_MAX       256

//
// Length of buffer to allocate to get
// statistics output.
//

#define STATISTICS_LENGTH           16384

//
// Query type.
//

#define TOTAL_STATS                 1
#define PARTIAL_STATS               2
#define NO_STATS                    3

//
// Log level.
//

#undef  TEST
#undef  DEBUG

class Statistics
{
  public:

  //
  // Use proxy to access message stores.
  //

  Statistics(Proxy *proxy);

  ~Statistics();

  void addIdleTime(unsigned int numMs)
  {
    transportPartial_.idleTime_ += numMs;
    transportTotal_.idleTime_ += numMs;
  }

  void subIdleTime(unsigned int numMs)
  {
    transportPartial_.idleTime_ -= numMs;
    transportTotal_.idleTime_ -= numMs;
  }

  void addReadTime(unsigned int numMs)
  {
    transportPartial_.readTime_ += numMs;
    transportTotal_.readTime_ += numMs;
  }

  void subReadTime(unsigned int numMs)
  {
    transportPartial_.readTime_ -= numMs;
    transportTotal_.readTime_ -= numMs;
  }

  void addWriteTime(unsigned int numMs)
  {
    transportPartial_.writeTime_ += numMs;
    transportTotal_.writeTime_ += numMs;
  }

  void subWriteTime(unsigned int numMs)
  {
    transportPartial_.writeTime_ -= numMs;
    transportTotal_.writeTime_ -= numMs;
  }

  void addBytesIn(unsigned int numBytes)
  {
    transportPartial_.proxyBytesIn_ += numBytes;
    transportTotal_.proxyBytesIn_ += numBytes;
  }

  void addBytesOut(unsigned int numBytes)
  {
    transportPartial_.proxyBytesOut_ += numBytes;
    transportTotal_.proxyBytesOut_ += numBytes;
  }

  void addFrameIn()
  {
    transportPartial_.proxyFramesIn_++;
    transportTotal_.proxyFramesIn_++;

    #ifdef TEST
    *logofs << "Statistics: Updated total proxy frames in to "
            << transportTotal_.proxyFramesIn_ << " at "
            << strMsTimestamp() << ".\n" << logofs_flush;
    #endif
  }

  void addFrameOut()
  {
    transportPartial_.proxyFramesOut_++;
    transportTotal_.proxyFramesOut_++;

    #ifdef TEST
    *logofs << "Statistics: Updated total proxy frames out to "
            << transportTotal_.proxyFramesOut_ << " at "
            << strMsTimestamp() << ".\n"
            << logofs_flush;
    #endif
  }

  void addWriteOut()
  {
    transportPartial_.proxyWritesOut_++;
    transportTotal_.proxyWritesOut_++;

    #ifdef TEST
    *logofs << "Statistics: Updated total proxy writes out to "
            << transportTotal_.proxyWritesOut_ << " at "
            << strMsTimestamp() << ".\n"
            << logofs_flush;
    #endif
  }

  void addCompressedBytes(unsigned int bytesIn, unsigned int bytesOut)
  {
    transportPartial_.compressedBytesIn_ += bytesIn;
    transportTotal_.compressedBytesIn_ += bytesIn;

    transportPartial_.compressedBytesOut_ += bytesOut;
    transportTotal_.compressedBytesOut_ += bytesOut;
  }

  void addDecompressedBytes(unsigned int bytesIn, unsigned int bytesOut)
  {
    transportPartial_.decompressedBytesIn_ += bytesIn;
    transportTotal_.decompressedBytesIn_ += bytesIn;

    transportPartial_.decompressedBytesOut_ += bytesOut;
    transportTotal_.decompressedBytesOut_ += bytesOut;
  }

  void addFramingBits(unsigned int numBits)
  {
    transportPartial_.framingBitsOut_ += numBits;
    transportTotal_.framingBitsOut_ += numBits;
  }

  void addCachedRequest(unsigned int opcode)
  {
    protocolPartial_.requestCached_[opcode]++;
    protocolTotal_.requestCached_[opcode]++;
  }

  void addRepliedRequest(unsigned int opcode)
  {
    protocolPartial_.requestReplied_[opcode]++;
    protocolTotal_.requestReplied_[opcode]++;
  }

  void addCachedReply(unsigned int opcode)
  {
    protocolPartial_.replyCached_[opcode]++;
    protocolTotal_.replyCached_[opcode]++;
  }

  void addRequestBits(unsigned int opcode, unsigned int bitsIn, unsigned int bitsOut)
  {
    #ifdef DEBUG
    *logofs << "Statistcs: Added " << bitsIn << " bits in and "
            << bitsOut << " bits out to opcode " << opcode
            << ".\n" << logofs_flush;
    #endif

    protocolPartial_.requestCount_[opcode]++;
    protocolTotal_.requestCount_[opcode]++;

    protocolPartial_.requestBitsIn_[opcode] += bitsIn;
    protocolTotal_.requestBitsIn_[opcode] += bitsIn;

    protocolPartial_.requestBitsOut_[opcode] += bitsOut;
    protocolTotal_.requestBitsOut_[opcode] += bitsOut;
  }

  void addReplyBits(unsigned int opcode, unsigned int bitsIn, unsigned int bitsOut)
  {
    protocolPartial_.replyCount_[opcode]++;
    protocolTotal_.replyCount_[opcode]++;

    protocolPartial_.replyBitsIn_[opcode] += bitsIn;
    protocolTotal_.replyBitsIn_[opcode] += bitsIn;

    protocolPartial_.replyBitsOut_[opcode] += bitsOut;
    protocolTotal_.replyBitsOut_[opcode] += bitsOut;
  }

  void addEventBits(unsigned int opcode, unsigned int bitsIn, unsigned int bitsOut)
  {
    protocolPartial_.eventCount_[opcode]++;
    protocolTotal_.eventCount_[opcode]++;

    protocolPartial_.eventBitsIn_[opcode] += bitsIn;
    protocolTotal_.eventBitsIn_[opcode] += bitsIn;

    protocolPartial_.eventBitsOut_[opcode] += bitsOut;
    protocolTotal_.eventBitsOut_[opcode] += bitsOut;
  }

  void addSyncBits(unsigned int bitsIn, unsigned int bitsOut)
  {
    protocolPartial_.syncCount_++;
    protocolTotal_.syncCount_++;

    protocolPartial_.syncBitsIn_ += bitsIn;
    protocolTotal_.syncBitsIn_ += bitsIn;

    protocolPartial_.syncBitsOut_ += bitsOut;
    protocolTotal_.syncBitsOut_ += bitsOut;
  }

  void addKeybdBits(unsigned int bitsIn, unsigned int bitsOut)
  {
    protocolPartial_.keybdCount_++;
    protocolTotal_.keybdCount_++;

    protocolPartial_.keybdBitsIn_ += bitsIn;
    protocolTotal_.keybdBitsIn_ += bitsIn;

    protocolPartial_.keybdBitsOut_ += bitsOut;
    protocolTotal_.keybdBitsOut_ += bitsOut;
  }

  void addSambaBits(unsigned int bitsIn, unsigned int bitsOut)
  {
    protocolPartial_.sambaCount_++;
    protocolTotal_.sambaCount_++;

    protocolPartial_.sambaBitsIn_ += bitsIn;
    protocolTotal_.sambaBitsIn_ += bitsIn;

    protocolPartial_.sambaBitsOut_ += bitsOut;
    protocolTotal_.sambaBitsOut_ += bitsOut;
  }

  void addMediaBits(unsigned int bitsIn, unsigned int bitsOut)
  {
    protocolPartial_.mediaCount_++;
    protocolTotal_.mediaCount_++;

    protocolPartial_.mediaBitsIn_ += bitsIn;
    protocolTotal_.mediaBitsIn_ += bitsIn;

    protocolPartial_.mediaBitsOut_ += bitsOut;
    protocolTotal_.mediaBitsOut_ += bitsOut;
  }

  void addHttpBits(unsigned int bitsIn, unsigned int bitsOut)
  {
    protocolPartial_.httpCount_++;
    protocolTotal_.httpCount_++;

    protocolPartial_.httpBitsIn_ += bitsIn;
    protocolTotal_.httpBitsIn_ += bitsIn;

    protocolPartial_.httpBitsOut_ += bitsOut;
    protocolTotal_.httpBitsOut_ += bitsOut;
  }

  void addPackedBytesIn(unsigned int numBytes)
  {
    packedPartial_.packedBytesIn_ += numBytes;
    packedTotal_.packedBytesIn_ += numBytes;
  }

  void addPackedBytesOut(unsigned int numBytes)
  {
    packedPartial_.packedBytesOut_ += numBytes;
    packedTotal_.packedBytesOut_ += numBytes;
  }

  void addSplit()
  {
    splitPartial_.splitCount_++;
    splitTotal_.splitCount_++;
  }

  void addSplitAborted()
  {
    splitPartial_.splitAborted_++;
    splitTotal_.splitAborted_++;
  }

  void addSplitAbortedBytesOut(unsigned int numBytes)
  {
    splitPartial_.splitAbortedBytesOut_ += numBytes;
    splitTotal_.splitAbortedBytesOut_ += numBytes;
  }

  //
  // Produce statistics on provided buffer.
  //

  int getClientCacheStats(int type, char *&buffer);
  int getClientProtocolStats(int type, char *&buffer);
  int getClientOverallStats(int type, char *&buffer);

  int getServerCacheStats(int type, char *&buffer);
  int getServerProtocolStats(int type, char *&buffer);
  int getServerOverallStats(int type, char *&buffer);

  int resetPartialStats();

  private:

  int getTimeStats(int type, char *&buffer);
  int getServicesStats(int type, char *&buffer);
  int getFramingStats(int type, char *&buffer);
  int getBitrateStats(int type, char *&buffer);
  int getStreamStats(int type, char *&buffer);
  int getSplitStats(int type, char *&buffer);

  struct T_protocolData
  {
    double requestCached_[STATISTICS_OPCODE_MAX];
    double requestReplied_[STATISTICS_OPCODE_MAX];
    double requestCount_[STATISTICS_OPCODE_MAX];
    double requestBitsIn_[STATISTICS_OPCODE_MAX];
    double requestBitsOut_[STATISTICS_OPCODE_MAX];

    double replyCached_[STATISTICS_OPCODE_MAX];
    double replyCount_[STATISTICS_OPCODE_MAX];
    double replyBitsIn_[STATISTICS_OPCODE_MAX];
    double replyBitsOut_[STATISTICS_OPCODE_MAX];

    double eventCached_[STATISTICS_OPCODE_MAX];
    double eventCount_[STATISTICS_OPCODE_MAX];
    double eventBitsIn_[STATISTICS_OPCODE_MAX];
    double eventBitsOut_[STATISTICS_OPCODE_MAX];

    double syncCount_;
    double syncBitsIn_;
    double syncBitsOut_;

    double keybdCount_;
    double keybdBitsIn_;
    double keybdBitsOut_;

    double sambaCount_;
    double sambaBitsIn_;
    double sambaBitsOut_;

    double mediaCount_;
    double mediaBitsIn_;
    double mediaBitsOut_;

    double httpCount_;
    double httpBitsIn_;
    double httpBitsOut_;
  };

  struct T_transportData
  {
    double idleTime_;
    double readTime_;
    double writeTime_;

    double proxyBytesIn_;
    double proxyBytesOut_;

    double proxyFramesIn_;
    double proxyFramesOut_;
    double proxyWritesOut_;

    double compressedBytesIn_;
    double compressedBytesOut_;

    double decompressedBytesIn_;
    double decompressedBytesOut_;

    double framingBitsOut_;
  };

  struct T_packedData
  {
    double packedBytesIn_;
    double packedBytesOut_;
  };

  struct T_splitData
  {
    double splitCount_;
    double splitAborted_;
    double splitAbortedBytesOut_;
  };

  struct T_overallData
  {
    double overallBytesIn_;
    double overallBytesOut_;
  };

  T_protocolData protocolPartial_;
  T_protocolData protocolTotal_;

  T_transportData transportPartial_;
  T_transportData transportTotal_;

  T_packedData packedPartial_;
  T_packedData packedTotal_;

  T_splitData splitPartial_;
  T_splitData splitTotal_;

  T_overallData overallPartial_;
  T_overallData overallTotal_;

  Proxy *proxy_;
};

#endif /* Statistics_H */
