/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <X11/Xmd.h>

#include <unistd.h>
#include <string.h>
#include <zlib.h>

#include "Tight.h"

#include "NXproto.h"

#include "Misc.h"
#include "Message.h"
#include "Transport.h"
#include "ServerChannel.h"

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

#define RGB_TO_PIXEL(bpp,r,g,b)                                 \
  (((CARD##bpp)(r) & srcRedMax) << srcRedShift |                \
   ((CARD##bpp)(g) & srcGreenMax) << srcGreenShift |            \
   ((CARD##bpp)(b) & srcBlueMax) << srcBlueShift)

#define RGB24_TO_PIXEL(bpp,r,g,b)                               \
   ((((CARD##bpp)(r) & 0xFF) * srcRedMax + 127) / 255           \
    << srcRedShift |                                            \
    (((CARD##bpp)(g) & 0xFF) * srcGreenMax + 127) / 255         \
    << srcGreenShift |                                          \
    (((CARD##bpp)(b) & 0xFF) * srcBlueMax + 127) / 255          \
    << srcBlueShift)

#define RGB24_TO_PIXEL32(r,g,b)                                 \
  (((CARD32)(r) & 0xFF) << srcRedShift |                        \
   ((CARD32)(g) & 0xFF) << srcGreenShift |                      \
   ((CARD32)(b) & 0xFF) << srcBlueShift)

extern int Unpack32To32(const T_colormask *colormask, const unsigned int *data,
                            unsigned int *out, unsigned int *end);

extern int Unpack24To24(const T_colormask *colormask, const unsigned char *data,
                            unsigned char *out, unsigned char *end);

extern int Unpack16To16(const T_colormask *colormask, const unsigned char *data,
                            unsigned char *out, unsigned char *end);

Tight::Tight()
{
  for (int i = 0; i < 4; i ++)
  {
    zlibStreamActive[i] = 0;
  }

  cutZeros        = 0;
  finalSize       = 0;
  rectWidth       = 0;
  rectColors      = 0;
  tightDecompSize = 0;

  finalBuffer       = NULL;
  tightPalette      = NULL;
  tightDecompBuffer = NULL;

  srcRedMax   = 0;
  srcGreenMax = 0;
  srcBlueMax  = 0;

  srcRedShift   = 0;
  srcGreenShift = 0;
  srcBlueShift  = 0;

  srcDepth = 0;
}

Tight::~Tight()
{
  CleanFinal();
  CleanDecomp();

  //
  // - When UnpackRFBTight is used, tightPalette is
  //   just assigned to the lastColormap_->data attrib
  //   from ServerChannel and UnpackRFBTight sets this
  //   palette to NULL when returning.
  //
  // - When UnpackTight is used, the memory for this
  //   palette is allocated internally so it can be
  //   safely freed.
  //

  if (tightPalette != NULL)
  {
    delete [] tightPalette;
  }
}

int Tight::ResizeFinal(unsigned int size)
{
  if (size == 0)
  {
    return -1;
  }

  if (finalSize < size)
  {
    if (finalSize > 0)
    {
      delete [] finalBuffer;
    }

    finalSize = size;

    finalBuffer = new unsigned char [size];
  }

  return 1;
}

unsigned char *Tight::ResizeDecomp(unsigned int size)
{
  if (size == 0)
  {
    return NULL;
  }

  if (tightDecompSize < size)
  {
    if (tightDecompSize > 0)
    {     
      delete [] tightDecompBuffer;
    }

    tightDecompSize = size;

    tightDecompBuffer = new unsigned char [size];
  }

  return tightDecompBuffer;
}

void Tight::CleanDecomp()
{
  if (tightDecompSize > 0)
  {
    delete [] tightDecompBuffer;

    tightDecompSize = 0;
  }
}

void Tight::CleanFinal()
{
  if (finalSize > 0)
  {
    delete [] finalBuffer;

    finalSize = 0;
  }
}


//
// This method is used to unpack images coming from
// native VNC servers. It is unreasonably complex as
// it needs to unpack images that can be larger than
// the maximum size allowed for a single X request
// and it was not designed to do so at the beginning.
// Parameters grew in number as long as we found that
// we needed some more information. It should be, at
// least, redesigned or use BIG-REQUESTS.
//

int Tight::UnpackTight(unsigned char method, T_colormap *new_colormap,
                           unsigned char *tightData, unsigned int src_length,
                               unsigned short src_height, unsigned short src_width,
                                   unsigned char *dstData, unsigned int dst_length,
                                       unsigned short dst_height, unsigned short dst_width,
                                           int dstBitsPerPixel, unsigned int drawable,
                                               unsigned int gcontext, short int dst_x, short int dst_y,
                                                   unsigned char dst_depth, WriteBuffer *writeBuffer,
                                                       Transport *transport, int bigEndian,
                                                           unsigned char *buffer)
{
  //
  // Check if the image at least contains the header.
  //

  if (src_length < TIGHT_PACK_HEADER)
  {
    #ifdef PANIC
    *logofs << "UnpackTight: size of TightEncoded image " << src_length
            << " smaller than the TightHeader " << TIGHT_PACK_HEADER
            << "\n" << logofs_flush;
    #endif

    tightPalette = NULL;

    return -1;
  }

  //
  // Initialisation of the parameters from
  // the tightData header
  //

  bool  zlibComp         = (method == PACK_RFB_TIGHT_COMPRESSED);

  CARD8 tightFilter      =  *tightData;
  CARD8 filteredBpp      =  *(tightData + 1);
  CARD8 sourceBpp        =  *(tightData + 2);
  srcDepth               =  *(tightData + 3);
  srcRedMax              =  *(CARD16 *)(tightData + 4);
  srcGreenMax            =  *(CARD16 *)(tightData + 6);
  srcBlueMax             =  *(CARD16 *)(tightData + 8);
  srcRedShift            =  *(tightData + 10);
  srcGreenShift          =  *(tightData + 11);
  srcBlueShift           =  *(tightData + 12);
  unsigned char comp_ctl =  *(tightData + 13);

  if (new_colormap != NULL)
  {
    tightPalette = new_colormap -> data;
  }

  //
  // Make sure that the filter is between 0..2.
  //

  if (tightFilter > 2)
  {
    #ifdef PANIC
    *logofs << "UnpackTight: Unsupported filter type " << tightFilter
            << "\n" << logofs_flush;
    #endif

    tightPalette = NULL;

    return -1;
  }

  if (tightFilter == 1)
  {
    if (sourceBpp == 32 && srcDepth == 24 && srcRedMax == 0xFF &&
           srcGreenMax == 0xFF && srcBlueMax == 0xFF)
    {
      rectColors = new_colormap->entries;
    }
    else
    {
      rectColors = new_colormap->entries / (sourceBpp / 8);
    }
  }

  //
  // Flush the zlib buffers if needed
  //

  int stream_id = 0;

  for (; stream_id < 4; stream_id++)
  {
    if ((comp_ctl & 1) && zlibStreamActive[stream_id])
    {
      if (inflateEnd (&zlibStream[stream_id]) != Z_OK && zlibStream[stream_id].msg != NULL)
      {
        #ifdef PANIC
        *logofs << "UnpackTight: InflateEnd in stream " << stream_id
                << " error " << zlibStream[stream_id].msg << "] \n" << logofs_flush ;
        #endif
      }
      zlibStreamActive[stream_id] = 0;
    }
    comp_ctl >>= 1;
  }

  //
  // Decompress the data if it is zlib
  // compressed.
  //

  unsigned char *tightDecompData;
  unsigned int  tightSize;

  tightData += TIGHT_PACK_HEADER;

  if (zlibComp)
  {
    z_stream *zs;
    int err, compressedSize;

    tightSize = src_height * int((src_width * filteredBpp + 7) / 8);
    compressedSize = src_length - TIGHT_PACK_HEADER;
    tightDecompData = ResizeDecomp(tightSize);

    if (tightDecompData == NULL)
    {
      #ifdef PANIC
      *logofs << "UnpackTight: Cannot allocate " << tightSize
              << " bytes for the decompressed image\n" << logofs_flush;
      #endif

      tightPalette = NULL;

      return -1;
    }

    #ifdef DEBUG
    *logofs << "UnpackTight: Compressed size [" << compressedSize
            << "] expected size [" << tightSize << "] address ["
            << (void *)tightDecompData << "].\n"
            << logofs_flush ;
    #endif

    //
    // Initialization of the stream that
    // will be used to decompress.
    //

    stream_id = comp_ctl & 0x03;

    if (stream_id < 0 || stream_id > 3)
    {
      #ifdef PANIC
      *logofs << "UnpackTight: Invalid zlib stream number "
              << stream_id << ". It should be in [0..3]\n"
              << logofs_flush;
      #endif

      tightPalette = NULL;

      return -1;
    }

    zs = &zlibStream[stream_id];
    if (!zlibStreamActive[stream_id])
    {
      zs->zalloc = Z_NULL;
      zs->zfree = Z_NULL;
      zs->opaque = Z_NULL;
      err = inflateInit(zs);
      if (err != Z_OK)
      {
        if (zs->msg != NULL)
        {
          #ifdef PANIC
          *logofs << "UnpackTight: InflateInit error in stream " << stream_id
                  << " error " << zs->msg << "\n" << logofs_flush;
          #endif
        }

        tightPalette = NULL;

        return -1;
      }
      zlibStreamActive[stream_id] = 1;
    }

    //
    // Decompress the buffer
    //

    zs->next_in = (Bytef *)tightData;
    zs->avail_in = compressedSize;
    zs->next_out = (Bytef *)tightDecompData;
    zs->avail_out = tightSize;

    err = inflate(zs, Z_SYNC_FLUSH);
    if (err != Z_OK && err != Z_STREAM_END)
    {
      if (zs->msg != NULL)
      {
        #ifdef PANIC
        *logofs << "UnpackTight: Inflate error in stream " << stream_id
                << " error " << zs->msg
                << "\n" << logofs_flush;
        #endif
      }
      else
      {
        #ifdef PANIC
        *logofs << "UnpackTight: Inflate error in stream " << stream_id
                << " error " << err
                << "\n" << logofs_flush;
                #endif
      }

      tightPalette = NULL;

      return -1;
    }
  }
  else
  {
    tightSize = src_length - TIGHT_PACK_HEADER;
    tightDecompData = tightData;
  }

  //
  // Apply the appropriate filter
  //

  if (ResizeFinal(int(src_width * sourceBpp / 8) * src_height) < 0
         || finalBuffer == NULL)
  {
    #ifdef PANIC
    *logofs << "UnpackTight: Cannot allocate " << int(src_width * sourceBpp / 8) * src_height
            << " bytes for the filtered data\n" << logofs_flush;
    #endif

    tightPalette = NULL;

    return -1;
  }

  #ifdef DEBUG
  *logofs << "UnpackTight: Allocated " << src_height * src_width * sourceBpp / 8
          << " space\n" << logofs_flush;
  *logofs << "UnpackTight: Source address " << (void *) tightDecompData
          << "\n" << logofs_flush;
  *logofs << "UnpackTight: Source Bpp " << int(sourceBpp)
          << "\n" << logofs_flush;
  #endif

  switch(sourceBpp)
  {
    case 8:
    {
      HandleTight8(tightFilter, tightDecompData, src_width, src_height);
      break;
    }
    case 16:
    {
      HandleTight16(tightFilter, tightDecompData, src_width, src_height);
      break;
    }
    case 32:
    {
      HandleTight32(tightFilter, tightDecompData, src_width, src_height);
      break;
    }
    default:
    {
      #ifdef PANIC
      *logofs << "UnpackTight: PANIC! Unsupported Bpp value " << sourceBpp
              << "for the tight encoded image.\n" << logofs_flush;
      #endif

      tightPalette = NULL;

      return -1;

      break;
    }
  }

  //
  // Just move the data into the destination
  // buffer with the correct padding
  //

  unsigned int row;
  unsigned char *dstBuff, *srcBuff;

  dstBuff = dstData;
  srcBuff = finalBuffer;

  #ifdef DEBUG
  *logofs << "UnpackTight: Dest Bpp " << dstBitsPerPixel
          << " Source Bpp " << int(sourceBpp)
          << "\n" << logofs_flush;

  *logofs << "UnpackTight: height " << dst_height
          << " width " << dst_width
          << "\n" << logofs_flush;
  #endif

  for (row = 0; row < dst_height; row++)
  {
    memcpy(dstBuff, srcBuff, dst_width * sourceBpp / 8);

    dstBuff += RoundUp4(dst_width * dstBitsPerPixel / 8);
    srcBuff += dst_width * sourceBpp / 8;
  }

  //
  // If the data part of the final image is
  // bigger than MESSAGE_DATA_LIMIT, cut it
  // and generate more X_PutImage messages.
  //

  unsigned int dst_size = dst_height * RoundUp4(dst_width * dstBitsPerPixel / 8);

  if (dst_size > MESSAGE_DATA_LIMIT - sz_xPutImageReq)
  {
    unsigned int data_offset = sz_xPutImageReq;
    unsigned int pieceSize = MESSAGE_DATA_LIMIT - sz_xPutImageReq;
    unsigned int pieces = 0, currPiece = 0;

    if (dst_size % pieceSize == 0)
    {
      pieces = dst_size / pieceSize;
    }
    else
    {
      pieces = dst_size / pieceSize + 1;
    }

    unsigned int lineSize = dst_size / dst_height;
    unsigned int linesPerPiece = pieceSize * dst_height/ dst_size;
    unsigned short nrLines = 0;

    unsigned int outputLen;
    unsigned char *cpDstData = dstData, *wBuffer = NULL;

    while(pieces > 0)
    {
      if (dst_height < linesPerPiece)
      {
        nrLines = dst_height;
      }
      else
      {
        nrLines = linesPerPiece;
      }

      outputLen = nrLines * lineSize + data_offset;

      if (currPiece == 0)
      {
        wBuffer = buffer;

        #ifdef DEBUG
        *logofs << "UnpackTight: Using main scratch buffer "
                << (void *) wBuffer << " with size " << outputLen
                << ".\n" << logofs_flush;
        #endif
      }
      else
      {
        wBuffer = writeBuffer -> addScratchMessage(outputLen);

        #ifdef DEBUG
        *logofs << "UnpackTight: Using new scratch buffer "
                << (void *) wBuffer << " with size " << outputLen
                << ".\n" << logofs_flush;
        #endif
      }

      *(wBuffer) = (unsigned char) X_PutImage;
      *(wBuffer + 1) = ZPixmap;
      PutUINT(outputLen >> 2, wBuffer + 2, bigEndian);
      PutULONG(drawable, wBuffer + 4, bigEndian);
      PutULONG(gcontext, wBuffer + 8, bigEndian);
      PutUINT(dst_width, wBuffer + 12, bigEndian);
      PutUINT(nrLines, wBuffer + 14, bigEndian);
      PutUINT(dst_x, wBuffer + 16, bigEndian);
      PutUINT(dst_y, wBuffer + 18, bigEndian);
      *(wBuffer + 20) = 0;
      *(wBuffer + 21) = dst_depth;

      memcpy(wBuffer + data_offset, dstData, nrLines * lineSize);

      dst_y += nrLines;
      dst_height -= nrLines;
      dstData += nrLines * lineSize;
      pieces--;
      currPiece++;

      if (FlushBuffer(writeBuffer, transport) < 0)
      {
        return -1;
      }
    }

    delete [] cpDstData;

    tightPalette = NULL;

    return 1;
  }
  else
  {
    #ifdef DEBUG
    *logofs << "UnpackTight: Returning scratch buffer "
            << (void *) buffer << " with size " << dst_size
            << ".\n" << logofs_flush;
    #endif

    tightPalette = NULL;

    return 1;
  }
}

int Tight::FlushBuffer(WriteBuffer *writeBuffer, Transport *transport)
{
  int writeLength   = 0;
  int scratchLength = writeBuffer -> getScratchLength();

  if (scratchLength > 0)
  {
    #ifdef TEST
    *logofs << "FlushBuffer: Going to flush " << scratchLength
            << " bytes of data " << "to FD#" << transport -> fd()
            << ".\n" << logofs_flush;
    #endif

    writeLength = transport -> write(write_immediate,
                                         writeBuffer -> getScratchData(),
                                             scratchLength);
    #ifdef PANIC

    if (writeLength < 0)
    {
      *logofs << "FlushBuffer: PANIC! Error writing data to FD#"
              << transport -> fd() << ".\n" << logofs_flush;
    }

    #endif

    writeBuffer -> removeScratchMessage();
  }

  return writeLength;
}

//
// This method is used to unpack images that were
// compressed by the NX X agent (using the compext
// library) with the Tight algorithm. Currently it
// is not used as it doesn't offer much advantages
// compared to PNG.
//

int Tight::UnpackTight(T_geometry *geometry, T_colormap *new_colormap,
                           unsigned char method, unsigned char *src_data,
                               int src_size, int dst_depth, int dst_bpp, int
                                   dst_width, int dst_height, unsigned char *dst_data,
                                       int dst_size)
{
  //
  // Initialize the filter, visual and palette
  //

  CARD8 tightFilter = *(CARD8 *)src_data;

  srcDepth      = dst_depth;
  srcRedShift   = ffs(geometry->red_mask) - 1;
  srcGreenShift = ffs(geometry->green_mask) - 1;
  srcBlueShift  = ffs(geometry->blue_mask) - 1;
  srcRedMax     = geometry->red_mask >> srcRedShift;
  srcGreenMax   = geometry->green_mask >> srcGreenShift;
  srcBlueMax    = geometry->blue_mask >> srcBlueShift;

  #ifdef DEBUG
  *logofs << "UnpackTight : filter " << (int)tightFilter
          << " rm " << (int)srcRedMax
          << " gm " << (int)srcGreenMax
          << " bm " << (int)srcBlueMax
          << " rs " << (int)srcRedShift
          << " gs " << (int)srcGreenShift
          << " bs " << (int)srcBlueShift
          << "\n" << logofs_flush;
  #endif

  int tightSize = 0;
  int colorsHeader = 0;

  if (tightFilter == 1)
  {
    int nrColors = *(CARD32 *)(src_data + 1);
    int i = 0;
    unsigned char *cpData = src_data + 5;

    if (nrColors <= 0)
    {
      #ifdef PANIC
      *logofs << "UnpackTight: PANIC! Error using filter palette. "
              << " Invalid number of colors in the palette " << new_colormap -> entries
              << ".\n" << logofs_flush;
      #endif

      return -1;
    }

    if (rectColors < nrColors)
    {
      delete [] tightPalette;

      tightPalette = new unsigned int[nrColors];
    }

    rectColors = nrColors;

    if(dst_bpp == 24)
    {
      if (src_size - 5 < rectColors * 4)
      {
        #ifdef PANIC
        *logofs << "UnpackTight: PANIC! Error using filter palette. "
                << " Size of the data part " << src_size - 5
                << " expected at least " << rectColors * 4
                << " as size of the header containing the palette"
                << ".\n" << logofs_flush;
        #endif

        return -1;
      }
    }
    else
    {
      if (src_size - 5 < rectColors * dst_bpp / 8)
      {
        #ifdef PANIC
        *logofs << "UnpackTight: PANIC! Error using filter palette. "
                << " Size of the data part " << src_size - 5
                << " expected at least " << rectColors * dst_bpp / 8
                << " as size of the header containing the palette"
                << ".\n" << logofs_flush;
        #endif

        return -1;
      }
    }

    switch(dst_bpp)
    {
      case 8:
      {
        for(i = 0; i < rectColors; i++)
        {
          tightPalette[i] = *(CARD8 *)cpData;

          cpData ++;
        }
        break;
      }
      case 16:
      {
        for(i = 0; i < rectColors; i++)
        {
          tightPalette[i] = *(CARD16 *)cpData;

          cpData += 2;
        }
        break;
      }
      default:
      {
        for(i = 0; i < rectColors; i++)
        {
          tightPalette[i] = *(CARD32 *)cpData;

          cpData += 4;
        }
        break;
      }
    }

    if(dst_bpp == 24)
    {
      colorsHeader = rectColors * 4 + 4;
    }
    else
    {
      colorsHeader = rectColors * dst_bpp / 8 + 4;
    }


    if(rectColors == 2)
    {
      tightSize = (dst_width + 7) / 8;
      tightSize *= dst_height;
    }
    else
      tightSize = dst_height * dst_width;
  }
  else if (tightFilter == 2)
  {
    if (dst_bpp == 32 && srcDepth == 24 && srcRedMax == 0xFF &&
           srcGreenMax == 0xFF && srcBlueMax == 0xFF)
    {
      tightSize = dst_height * dst_width * 3;
    }
    else
    {
      tightSize = dst_height * dst_width * dst_bpp / 8;
    }
  }
  else if (tightFilter == 0)
  {
    if (dst_bpp == 32 && srcDepth == 24 && srcRedMax == 0xFF &&
           srcGreenMax == 0xFF && srcBlueMax == 0xFF)
    {
      tightSize = dst_height * dst_width * 3;
    }
    else
    {
      tightSize = dst_height * dst_width * dst_bpp / 8;
    }
  }
  else
  {
    #ifdef PANIC
    *logofs << "UnpackTight: PANIC! Filter error. "
            << " Invalid filter type " << (int)tightFilter
            << ".\n" << logofs_flush;
    #endif

    return -1;
  }

  //
  // Apply the correct filter
  //

  unsigned char *tightDecompData = src_data + 1 + colorsHeader;

  if (ResizeFinal(int(dst_width * dst_bpp / 8) * dst_height) < 0 || finalBuffer == NULL)
  {
    #ifdef PANIC
    *logofs << "UnpackTight : Cannot allocate " << int(dst_width * dst_bpp / 8) * dst_height
            << " bytes for the filtered data\n" << logofs_flush;
    #endif

    return -1;
  }

  switch(dst_bpp)
  {
    case 8:
    {
      HandleTight8(tightFilter, tightDecompData, dst_width, dst_height);
      break;
    }
    case 16:
    {
      HandleTight16(tightFilter, tightDecompData, dst_width, dst_height);
      break;
    }
    case 24:
    {
      HandleTight24(tightFilter, tightDecompData, dst_width, dst_height);
      break;
    }
    case 32:
    {
      HandleTight32(tightFilter, tightDecompData, dst_width, dst_height);
      break;
    }
    default:
    {
      #ifdef PANIC
      *logofs << "UnpackTight: PANIC! Unsupported Bpp value " << dst_bpp
              << "for the tight encoded image.\n" << logofs_flush;
      #endif

      return -1;

      break;
    }
  }

  //
  // Just move the data into the destination
  // buffer with the correct padding
  //

  int row;
  unsigned char *dstBuff, *srcBuff;

  dstBuff = dst_data;
  srcBuff = finalBuffer;

  for (row = 0; row < dst_height; row++)
  {
    memcpy(dstBuff, srcBuff, dst_width * dst_bpp / 8);

    dstBuff += RoundUp4(dst_width * dst_bpp / 8);
    srcBuff += dst_width * dst_bpp / 8;
  }

  //
  // Apply the correction for the brightness
  //

  int maskMethod;

  switch(method)
  {
    case PACK_TIGHT_8_COLORS:
    {
      maskMethod = MASK_8_COLORS;
      break;
    }
    case PACK_TIGHT_64_COLORS:
    {
      maskMethod = MASK_64_COLORS;
      break;
    }
    case PACK_TIGHT_256_COLORS:
    {
      maskMethod = MASK_256_COLORS;
      break;
    }
    case PACK_TIGHT_512_COLORS:
    {
      maskMethod = MASK_512_COLORS;
      break;
    }
    case PACK_TIGHT_4K_COLORS:
    {
      maskMethod = MASK_4K_COLORS;
      break;
    }
    case PACK_TIGHT_32K_COLORS:
    {
      maskMethod = MASK_32K_COLORS;
      break;
    }
    case PACK_TIGHT_64K_COLORS:
    {
      maskMethod = MASK_64K_COLORS;
      break;
    }
    case PACK_TIGHT_256K_COLORS:
    {
      maskMethod = MASK_256K_COLORS;
      break;
    }
    case PACK_TIGHT_2M_COLORS:
    {
      maskMethod = MASK_2M_COLORS;
      break;
    }
    case PACK_TIGHT_16M_COLORS:
    {
      maskMethod = MASK_16M_COLORS;
      break;
    }
    default:
    {
      return 1;
    }
  }

  const T_colormask *colorMask = GetColorMask(maskMethod);

  dstBuff = dst_data;

  switch (dst_bpp)
  {
    case 16:
    {
      Unpack16To16(colorMask, dstBuff, dstBuff, dstBuff + dst_size);
      break;
    }
    case 24:
    {
      break;
    }
    case 32:
    {
      Unpack32To32(colorMask, (unsigned int *)dstBuff, (unsigned int *)dstBuff,
                       (unsigned int *)(dstBuff + dst_size));
      break;
    }
    default:
    {
      return 1;
    }
  }

  return 1;

}

void Tight::HandleTight8 (int filterType, unsigned char *buffer, int rw, int rh)
{
  switch(filterType)
  {
    case 0:
    {
      rectWidth = rw;
      FilterCopy8(rh, buffer, (CARD8 *)finalBuffer);
      break;
    }
    case 1:
    {
      rectWidth = rw;
      FilterPalette8(rh, buffer, (CARD8 *)finalBuffer);
      break;
    }
    case 2:
    {
      rectWidth = rw;
      if (cutZeros)
        memset(tightPrevRow, 0, rw * 3);
      else
        memset(tightPrevRow, 0, rw * 3 * sizeof(CARD16));

      FilterGradient8(rh, buffer, (CARD8 *)finalBuffer);
      break;
    }
  }
}

void Tight::HandleTight16 (int filterType, unsigned char *buffer, int rw, int rh)
{
  switch(filterType)
  {
    case 0:
    {
      rectWidth = rw;
      FilterCopy16(rh, buffer, (CARD16 *)finalBuffer);
      break;
    }
    case 1:
    {
      rectWidth = rw;
      FilterPalette16(rh, buffer, (CARD16 *)finalBuffer);
      break;
    }
    case 2:
    {
      rectWidth = rw;
      if (cutZeros)
        memset(tightPrevRow, 0, rw * 3);
      else
        memset(tightPrevRow, 0, rw * 3 * sizeof(CARD16));

      FilterGradient16(rh, buffer, (CARD16 *)finalBuffer);
      break;
    }
  }
}

void Tight::HandleTight24 (int filterType, unsigned char *buffer, int rw, int rh)
{
  switch(filterType)
  {
    case 0:
    {
      rectWidth = rw;
      FilterCopy24(rh, buffer, (CARD8 *)finalBuffer);
      break;
    }
    case 1:
    {
      rectWidth = rw;
      FilterPalette24(rh, buffer, (CARD8 *)finalBuffer);
      break;
    }
    case 2:
    {
      rectWidth = rw;
      if (cutZeros)
        memset(tightPrevRow, 0, rw * 3);
      else
        memset(tightPrevRow, 0, rw * 3 * sizeof(CARD16));

      FilterGradient24bpp(rh, buffer, (CARD8 *)finalBuffer);
      break;
    }
  }
}

void Tight::HandleTight32 (int filterType, unsigned char *buffer, int rw, int rh)
{
  switch(filterType)
  {
    case 0:
    {
      rectWidth = rw;
      if (srcDepth == 24 && srcRedMax == 0xFF && srcGreenMax == 0xFF && srcBlueMax == 0xFF)
      {
        cutZeros = 1;
      }
      else
      {
        cutZeros = 0;
      }
      FilterCopy32(rh, buffer, (CARD32 *)finalBuffer);
      break;
    }
    case 1:
    {
      rectWidth = rw;
      FilterPalette32(rh, buffer, (CARD32 *)finalBuffer);
      break;
    }
    case 2:
    {
      rectWidth = rw;
      if (srcDepth == 24 && srcRedMax == 0xFF && srcGreenMax == 0xFF && srcBlueMax == 0xFF)
      {
        cutZeros = 1;
      }
      else
      {
        cutZeros = 0;
      }
      if (cutZeros)
        memset(tightPrevRow, 0, rw * 3);
      else
        memset(tightPrevRow, 0, rw * 3 * sizeof(CARD16));

      FilterGradient32(rh, buffer, (CARD32 *)finalBuffer);
      break;
    }
  }
}


//
// Functions that realize the filtering
//


void Tight::FilterCopy8 (int numRows, unsigned char *buffer, CARD8 *dst)
{
  memcpy (dst, buffer, numRows * rectWidth);
}

void Tight::FilterCopy16 (int numRows, unsigned char *buffer, CARD16 *dst)
{
  memcpy (dst, buffer, numRows * rectWidth * 2);
}

void Tight::FilterCopy24 (int numRows, unsigned char *buffer, CARD8 *dst)
{
  memcpy (dst, buffer, numRows * rectWidth * 3);
}

void Tight::FilterCopy32 (int numRows, unsigned char *buffer, CARD32 *dst)
{
  int x, y;

  if (cutZeros)
  {
    for (y = 0; y < numRows; y++)
    {
      for (x = 0; x < rectWidth; x++)
      {
        dst[y*rectWidth+x] = RGB24_TO_PIXEL32(buffer[(y*rectWidth+x)*3],
                                              buffer[(y*rectWidth+x)*3+1],
                                              buffer[(y*rectWidth+x)*3+2]);
      }
    }
    return;
  }

  memcpy (dst, buffer, numRows * rectWidth * 4);
}

void Tight::FilterGradient8 (int numRows, unsigned char *buffer, CARD8 *dst)
{
  int x, y, c;
  CARD8 *src = (CARD8 *)buffer;
  CARD16 *thatRow = (CARD16 *)tightPrevRow;
  CARD16 thisRow[2048*3];
  CARD16 pix[3];
  CARD16 max[3];
  int shift[3];
  int est[3];

  max[0] = srcRedMax;
  max[1] = srcGreenMax;
  max[2] = srcBlueMax;

  shift[0] = srcRedShift;
  shift[1] = srcGreenShift;
  shift[2] = srcBlueShift;

  for (y = 0; y < numRows; y++)
  {
    //
    // First pixel in a row
    //

    for (c = 0; c < 3; c++)
    {
      pix[c] = (CARD16)((src[y*rectWidth] >> shift[c]) + thatRow[c] & max[c]);
      thisRow[c] = pix[c];
    }
    dst[y*rectWidth] = RGB_TO_PIXEL(8, pix[0], pix[1], pix[2]);

    //
    // Remaining pixels of a row
    //

    for (x = 1; x < rectWidth; x++)
    {
      for (c = 0; c < 3; c++)
      {
        est[c] = (int)thatRow[x*3+c] + (int)pix[c] - (int)thatRow[(x-1)*3+c];

        if (est[c] > (int)max[c])
        {
          est[c] = (int)max[c];
        }
        else if (est[c] < 0)
        {
          est[c] = 0;
        }

        pix[c] = (CARD16)((src[y*rectWidth+x] >> shift[c]) + est[c] & max[c]);
        thisRow[x*3+c] = pix[c];
      }
      dst[y*rectWidth+x] = RGB_TO_PIXEL(8, pix[0], pix[1], pix[2]);
    }
    memcpy(thatRow, thisRow, rectWidth * 3 * sizeof(CARD16));
  }
}

void Tight::FilterGradient16 (int numRows, unsigned char *buffer, CARD16 *dst)
{
  int x, y, c;
  CARD16 *src = (CARD16 *)buffer;
  CARD16 *thatRow = (CARD16 *)tightPrevRow;
  CARD16 thisRow[2048*3];
  CARD16 pix[3];
  CARD16 max[3];
  int shift[3];
  int est[3];

  max[0] = srcRedMax;
  max[1] = srcGreenMax;
  max[2] = srcBlueMax;

  shift[0] = srcRedShift;
  shift[1] = srcGreenShift;
  shift[2] = srcBlueShift;

  for (y = 0; y < numRows; y++)
  {
    //
    // First pixel in a row
    //

    for (c = 0; c < 3; c++)
    {
      pix[c] = (CARD16)((src[y*rectWidth] >> shift[c]) + thatRow[c] & max[c]);
      thisRow[c] = pix[c];
    }
    dst[y*rectWidth] = RGB_TO_PIXEL(16, pix[0], pix[1], pix[2]);

    //
    // Remaining pixels of a row
    //

    for (x = 1; x < rectWidth; x++)
    {
      for (c = 0; c < 3; c++)
      {
        est[c] = (int)thatRow[x*3+c] + (int)pix[c] - (int)thatRow[(x-1)*3+c];

        if (est[c] > (int)max[c])
        {
          est[c] = (int)max[c];
        }
        else if (est[c] < 0)
        {
          est[c] = 0;
        }

        pix[c] = (CARD16)((src[y*rectWidth+x] >> shift[c]) + est[c] & max[c]);
        thisRow[x*3+c] = pix[c];
      }
      dst[y*rectWidth+x] = RGB_TO_PIXEL(16, pix[0], pix[1], pix[2]);
    }
    memcpy(thatRow, thisRow, rectWidth * 3 * sizeof(CARD16));
  }
}

void Tight::FilterGradient24 (int numRows, unsigned char *buffer, CARD32 *dst)
{
  int x, y, c;
  CARD8 thisRow[2048*3];
  CARD8 pix[3];
  int est[3];

  for (y = 0; y < numRows; y++)
  {
    //
    // First pixel in a row
    //

    for (c = 0; c < 3; c++)
    {
      pix[c] = tightPrevRow[c] + buffer[y*rectWidth*3+c];
      thisRow[c] = pix[c];
    }
    dst[y*rectWidth] = RGB24_TO_PIXEL32(pix[0], pix[1], pix[2]);

    //
    // Remaining pixels of a row
    //

    for (x = 1; x < rectWidth; x++)
    {
      for (c = 0; c < 3; c++)
      {
        est[c] = (int)tightPrevRow[x*3+c] + (int)pix[c] - (int)tightPrevRow[(x-1)*3+c];
        if (est[c] > 0xFF)
        {
          est[c] = 0xFF;
        }
        else if (est[c] < 0x00)
        {
          est[c] = 0x00;
        }
        pix[c] = (CARD8)est[c] + buffer[(y*rectWidth+x)*3+c];
        thisRow[x*3+c] = pix[c];
      }
      dst[y*rectWidth+x] = RGB24_TO_PIXEL32(pix[0], pix[1], pix[2]);
    }

    memcpy(tightPrevRow, thisRow, rectWidth * 3);
  }
}

void Tight::FilterGradient24bpp (int numRows, unsigned char *buffer, CARD8 *dst)
{
  int x, y, c;
  CARD8 thisRow[2048*3];
  CARD8 pix[3];
  int est[3];

  for (y = 0; y < numRows; y++)
  {
    //
    // First pixel in a row
    //

    for (c = 0; c < 3; c++)
    {
      pix[c] = tightPrevRow[c] + buffer[y*rectWidth*3+c];
      thisRow[c] = pix[c];
      dst[y*rectWidth*3+c] = pix[c];
    }

    //
    // Remaining pixels of a row
    //

    for (x = 1; x < rectWidth; x++)
    {
      for (c = 0; c < 3; c++)
      {
        est[c] = (int)tightPrevRow[x*3+c] + (int)pix[c] - (int)tightPrevRow[(x-1)*3+c];
        if (est[c] > 0xFF)
        {
          est[c] = 0xFF;
        }
        else if (est[c] < 0x00)
        {
          est[c] = 0x00;
        }
        pix[c] = (CARD8)est[c] + buffer[(y*rectWidth+x)*3+c];
        thisRow[x*3+c] = pix[c];

        dst[(y*rectWidth+x)*3+c] = pix[c];
      }
    }

    memcpy(tightPrevRow, thisRow, rectWidth * 3);
  }
}


void Tight::FilterGradient32 (int numRows, unsigned char *buffer, CARD32 *dst)
{
  int x, y, c;
  CARD32 *src = (CARD32 *)buffer;
  CARD16 *thatRow = (CARD16 *)tightPrevRow;
  CARD16 thisRow[2048*3];
  CARD16 pix[3];
  CARD16 max[3];
  int shift[3];
  int est[3];

  if (cutZeros)
  {
    FilterGradient24(numRows, buffer, dst);
    return;
  }

  max[0] = srcRedMax;
  max[1] = srcGreenMax;
  max[2] = srcBlueMax;

  shift[0] = srcRedShift;
  shift[1] = srcGreenShift;
  shift[2] = srcBlueShift;

  for (y = 0; y < numRows; y++)
  {

    //
    // First pixel in a row
    //

    for (c = 0; c < 3; c++)
    {
      pix[c] = (CARD16)((src[y*rectWidth] >> shift[c]) + thatRow[c] & max[c]);
      thisRow[c] = pix[c];
    }
    dst[y*rectWidth] = RGB_TO_PIXEL(32, pix[0], pix[1], pix[2]);

    //
    // Remaining pixels of a row
    //

    for (x = 1; x < rectWidth; x++)
    {
      for (c = 0; c < 3; c++)
      {
        est[c] = (int)thatRow[x*3+c] + (int)pix[c] - (int)thatRow[(x-1)*3+c];
        if (est[c] > (int)max[c])
        {
          est[c] = (int)max[c];
        }
        else if (est[c] < 0)
        {
          est[c] = 0;
        }
        pix[c] = (CARD16)((src[y*rectWidth+x] >> shift[c]) + est[c] & max[c]);
        thisRow[x*3+c] = pix[c];
      }
      dst[y*rectWidth+x] = RGB_TO_PIXEL(32, pix[0], pix[1], pix[2]);
    }
    memcpy(thatRow, thisRow, rectWidth * 3 * sizeof(CARD16));
  }
}

void Tight::FilterPalette8 (int numRows, unsigned char *buffer, CARD8 *dst)
{
  int x, y, b, w;
  CARD8 *src = (CARD8 *)buffer;

  if (rectColors == 2)
  {
    w = (rectWidth + 7) / 8;

    for (y = 0; y < numRows; y++)
    {
      for (x = 0; x < rectWidth / 8; x++)
      {
        for (b = 7; b >= 0; b--)
        {
          dst[y*rectWidth+x*8+7-b] = tightPalette[src[y*w+x] >> b & 1];
        }
      }

      for (b = 7; b >= 8 - rectWidth % 8; b--)
      {
        dst[y*rectWidth+x*8+7-b] = tightPalette[src[y*w+x] >> b & 1];
      }
    }
  }
  else
  {
    for (y = 0; y < numRows; y++)
    {
      for (x = 0; x < rectWidth; x++)
      {
        dst[y*rectWidth+x] = tightPalette[(int)src[y*rectWidth+x]];
      }
    }
  }
}

void Tight::FilterPalette16 (int numRows, unsigned char *buffer, CARD16 *dst)
{
  int x, y, b, w;
  CARD8 *src = (CARD8 *)buffer;

  if (rectColors == 2)
  {
    w = (rectWidth + 7) / 8;

    for (y = 0; y < numRows; y++)
    {
      for (x = 0; x < rectWidth / 8; x++)
      {
        for (b = 7; b >= 0; b--)
        {
          dst[y*rectWidth+x*8+7-b] = tightPalette[src[y*w+x] >> b & 1];
        }
      }

      for (b = 7; b >= 8 - rectWidth % 8; b--)
      {
        dst[y*rectWidth+x*8+7-b] = tightPalette[src[y*w+x] >> b & 1];
      }
    }
  }
  else
  {
    for (y = 0; y < numRows; y++)
    {
      for (x = 0; x < rectWidth; x++)
      {
        dst[y*rectWidth+x] = tightPalette[(int)src[y*rectWidth+x]];
      }
    }
  }
}

void Tight::FilterPalette24 (int numRows, unsigned char *buffer, CARD8 *dst)
{
  int x, y, b, w;
  CARD8 *src = (CARD8 *)buffer;

  if (rectColors == 2)
  {
    w = (rectWidth + 7) / 8;

    for (y = 0; y < numRows; y++)
    {
      for (x = 0; x < rectWidth / 8; x++)
      {
        for (b = 7; b >= 0; b--)
        {
          dst[(y*rectWidth+x*8+7-b)*3]   = (tightPalette[src[y*w+x] >> b & 1] & 0x00FF0000) >> 16;
          dst[(y*rectWidth+x*8+7-b)*3+1] = (tightPalette[src[y*w+x] >> b & 1] & 0x0000FF00) >> 8;
          dst[(y*rectWidth+x*8+7-b)*3+2] = tightPalette[src[y*w+x] >> b & 1] & 0x000000FF;
        }
      }

      for (b = 7; b >= 8 - rectWidth % 8; b--)
      {
        dst[(y*rectWidth+x*8+7-b)*3]   = (tightPalette[src[y*w+x] >> b & 1] & 0x00FF0000) >> 16;
        dst[(y*rectWidth+x*8+7-b)*3+1] = (tightPalette[src[y*w+x] >> b & 1] & 0x0000FF00) >> 8;
        dst[(y*rectWidth+x*8+7-b)*3+2] = tightPalette[src[y*w+x] >> b & 1] & 0x000000FF;
      }
    }
  }
  else
  {
    for (y = 0; y < numRows; y++)
    {
      for (x = 0; x < rectWidth; x++)
      {
        dst[(y*rectWidth+x)*3]   = (tightPalette[(int)src[y*rectWidth+x]] & 0x00FF0000) >> 16;
        dst[(y*rectWidth+x)*3+1] = (tightPalette[(int)src[y*rectWidth+x]] & 0x0000FF00) >> 8;
        dst[(y*rectWidth+x)*3+2] = tightPalette[(int)src[y*rectWidth+x]] & 0x000000FF;
      }
    }
  }
}


void Tight::FilterPalette32 (int numRows, unsigned char *buffer, CARD32 *dst)
{
  int x, y, b, w;
  CARD8 *src = (CARD8 *)buffer;

  if (rectColors == 2)
  {
    w = (rectWidth + 7) / 8;

    for (y = 0; y < numRows; y++)
    {
      for (x = 0; x < rectWidth / 8; x++)
      {
        for (b = 7; b >= 0; b--)
        {
          dst[y*rectWidth+x*8+7-b] = tightPalette[src[y*w+x] >> b & 1];
        }
      }

      for (b = 7; b >= 8 - rectWidth % 8; b--)
      {
        dst[y*rectWidth+x*8+7-b] = tightPalette[src[y*w+x] >> b & 1];
      }
    }
  }
  else
  {
    for (y = 0; y < numRows; y++)
    {
      for (x = 0; x < rectWidth; x++)
      {
        dst[y*rectWidth+x] = tightPalette[(int)src[y*rectWidth+x]];
      }
    }
  }
}

