/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2005 NoMachine, http://www.nomachine.com/.         */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <sys/socket.h>

#include "NXalert.h"

#include "Transport.h"

#include "Statistics.h"

//
// Set the verbosity level. You also
// need to define DUMP in Misc.cpp
// if DUMP is defined here.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG
#undef  INSPECT
#undef  DUMP

//
// Define this to lock and unlock the
// memory-to-memory transport buffers
// before they are accessed. The code
// is outdated and doesn't work with
// the current pthread library.
//

#undef  THREADS

//
// Define this to get logging all the
// operations performed by the parent
// thread, the one that enqueues and
// dequeues data.
//

#define PARENT

//
// Define this to know when a channel
// is created or destroyed.
//

#undef  REFERENCES

//
// Reference count for allocated buffers.
//

#ifdef REFERENCES

int Transport::references_;
int ProxyTransport::references_;
int InternalTransport::references_;

#endif

//
// This is the base class providing methods for read
// and write buffering.
//

Transport::Transport(int fd) : fd_(fd)
{
  #ifdef TEST
  *logofs << "Transport: Going to create base transport "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  type_ = transport_base;

  //
  // Set up the write buffer.
  //

  w_buffer_.length_ = 0;
  w_buffer_.start_  = 0;

  initialSize_   = TRANSPORT_BUFFER_DEFAULT_SIZE;
  thresholdSize_ = TRANSPORT_BUFFER_DEFAULT_SIZE << 1;
  maximumSize_   = TRANSPORT_BUFFER_DEFAULT_SIZE << 4;

  w_buffer_.data_.resize(initialSize_);

  //
  // Set non-blocking IO on socket.
  //

  SetNonBlocking(fd_, 1);

  blocked_ = 0;
  forced_  = 0;
  finish_  = 0;

  #ifdef REFERENCES
  *logofs << "Transport: Created new object at " 
          << this << " out of " << ++references_ 
          << " allocated references.\n" << logofs_flush;
  #endif
}

Transport::~Transport()
{
  #ifdef TEST
  *logofs << "Transport: Going to destroy base class "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  ::close(fd_);

  #ifdef REFERENCES
  *logofs << "Transport: Deleted object at " 
          << this << " out of " << --references_ 
          << " allocated references.\n" << logofs_flush;
  #endif
}

//
// Read data from its file descriptor.
//

int Transport::read(unsigned char *data, unsigned int size)
{
  #ifdef DEBUG
  *logofs << "Transport: Going to read " << size << " bytes from " 
          << "FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // Get any data from socket.
  //

  int result = ::read(fd_, data, size);

  if (result < 0)
  {
    if (EGET() == EAGAIN)
    {
      #ifdef TEST
      *logofs << "Transport: WARNING! Read of " << size << " bytes from "
              << "FD#" << fd_ << " would block.\n" << logofs_flush;
      #endif

      return 0;
    }
    else if (EGET() == EINTR)
    {
      #ifdef TEST
      *logofs << "Transport: Read of " << size << " bytes from "
              << "FD#" << fd_ << " was interrupted.\n"
              << logofs_flush;
      #endif

      return 0;
    }
    else
    {
      #ifdef TEST
      *logofs << "Transport: Error reading from " 
              << "FD#" << fd_ << ".\n" << logofs_flush;
      #endif

      finish();

      return -1;
    }
  }
  else if (result == 0)
  {
    #ifdef TEST
    *logofs << "Transport: No data read from " 
            << "FD#" << fd_ << ".\n" << logofs_flush;
    #endif

    finish();

    return -1;
  }

  #ifdef TEST
  *logofs << "Transport: Read " << result << " bytes out of "
          << size << " from FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  #ifdef DUMP

  *logofs << "Transport: Dumping content of read data.\n"
          << logofs_flush;

  DumpData(data, result);

  #endif

  return result;
}

//
// Write as many bytes as possible to socket. 
// Append the remaining data bytes to the end 
// of the buffer and update length to reflect
// changes.
//

int Transport::write(T_write type, const unsigned char *data, const unsigned int size)
{
  //
  // If an immediate write was requested then
  // flush the enqueued data first.
  //

  if (w_buffer_.length_ > 0 && blocked_ == 0 &&
          type == write_immediate)
  {
    #ifdef TEST
    *logofs << "Transport: Writing " << w_buffer_.length_
            << " bytes of previous data to FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    int result = Transport::flush();

    if (result < 0)
    {
      return -1;
    }
  }

  //
  // If nothing is remained, write immediately
  // to the socket.
  //

  unsigned int written = 0;

  if (w_buffer_.length_ == 0 && blocked_ == 0 &&
          type == write_immediate)
  {
    //
    // Limit the amount of data sent.
    //

    unsigned int toWrite = ((int) size > control -> TransportWriteLimit ?
                                 control -> TransportWriteLimit : size);

    #ifdef DUMP

    *logofs << "Transport: Going to write " << toWrite
            << " bytes to FD#" << fd_ << " with checksum ";

    DumpChecksum(data, size);

    *logofs << ".\n" << logofs_flush;

    #endif

    T_timestamp writeTs;

    int diffTs;

    while (written < toWrite)
    {
      //
      // Trace system time spent writing data.
      //

      writeTs = getTimestamp();

      int result = ::write(fd_, data + written, toWrite - written);

      diffTs = diffTimestamp(writeTs, getTimestamp());

      control -> addWriteTime(diffTs);

      if (control -> CollectStatistics)
      {
        statistics -> addWriteTime(diffTs);
      }

      if (result <= 0)
      {
        if (EGET() == EAGAIN)
        {
          #ifdef TEST
          *logofs << "Transport: Write of " << toWrite - written
                  << " bytes on FD#" << fd_ << " would block.\n"
                  << logofs_flush;
          #endif

          blocked_ = 1;

          break;
        }
        else if (EGET() == EINTR)
        {
          #ifdef TEST
          *logofs << "Transport: Write of " << toWrite - written
                  << " bytes on FD#" << fd_ << " was interrupted.\n"
                  << logofs_flush;
          #endif

          continue;
        }
        else
        {
          #ifdef TEST
          *logofs << "Transport: Write on " << "FD#" 
                  << fd_ << " failed.\n" << logofs_flush;
          #endif

          finish();

          return -1;
        }
      }
      else
      {
        #ifdef TEST
        *logofs << "Transport: Immediately written " << result
                << " bytes on " << "FD#" << fd_ << ".\n"
                << logofs_flush;
        #endif

        written += result;
      }
    }

    #ifdef DUMP

    if (written > 0)
    {
      *logofs << "Transport: Dumping content of immediately written data.\n"
              << logofs_flush;

      DumpData(data, written);
    }

    #endif
  }

  if (written == size)
  {
    //
    // We will not affect the write buffer.
    //

    return written;
  }

  #ifdef DEBUG
  *logofs << "Transport: Going to append " << size - written
          << " bytes to write buffer for " << "FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  if (resize(w_buffer_, size - written) < 0)
  {
    return -1;
  }

  memmove(w_buffer_.data_.begin() + w_buffer_.start_ + w_buffer_.length_,
              data + written, size - written);

  w_buffer_.length_ += size - written;

  #ifdef TEST
  *logofs << "Transport: Write buffer for FD#" << fd_
          << " has data for " << w_buffer_.length_ << " bytes.\n"
          << logofs_flush;

  *logofs << "Transport: Start is " << w_buffer_.start_ 
          << " length is " << w_buffer_.length_ << " size is "
          << w_buffer_.data_.size() << " capacity is "
          << w_buffer_.data_.capacity() << ".\n"
          << logofs_flush;
  #endif

  //
  // Note that this function always returns the whole
  // size of buffer that was provided, either if not
  // all the data could be actually written.
  //

  return size;
}

//
// Write pending data to its file descriptor.
//

int Transport::flush()
{
  if (w_buffer_.length_ == 0)
  {
    #ifdef TEST
    *logofs << "Transport: No data to flush on "
            << "FD#" << fd_ << ".\n" << logofs_flush;
    #endif

    #ifdef WARNING
    if (blocked_ != 0)
    {
      *logofs << "Transport: Blocked flag is " << blocked_
              << " with no data to flush on FD#" << fd_
              << ".\n" << logofs_flush;
    }
    #endif

    return 0;
  }

  //
  // It's time to move data from the 
  // write buffer to the real link. 
  //

  int written = 0;

  int toWrite = (w_buffer_.length_ > control -> TransportWriteLimit ?
                      control -> TransportWriteLimit : w_buffer_.length_);

  //
  // We will do our best to write any available
  // data to the socket, so let's say we start
  // from a clean state.
  //

  blocked_ = 0;

  #ifdef TEST
  *logofs << "Transport: Going to flush " << toWrite
          << " bytes on FD#" << fd_ << ".\n"
          << logofs_flush;
  #endif

  T_timestamp writeTs;

  int diffTs;

  while (written < toWrite)
  {
    writeTs = getTimestamp();

    int result = ::write(fd_, w_buffer_.data_.begin() + w_buffer_.start_ +
                             written, toWrite - written);

    diffTs = diffTimestamp(writeTs, getTimestamp());

    control -> addWriteTime(diffTs);

    if (control -> CollectStatistics)
    {
      statistics -> addWriteTime(diffTs);
    }

    if (result <= 0)
    {
      if (EGET() == EAGAIN)
      {
        #ifdef TEST
        *logofs << "Transport: Write of " << toWrite - written
                << " bytes on FD#" << fd_ << " would block.\n"
                << logofs_flush;
        #endif

        blocked_ = 1;

        break;
      }
      else if (EGET() == EINTR)
      {
        #ifdef TEST
        *logofs << "Transport: Write of " << toWrite - written
                << " bytes on FD#" << fd_ << " was interrupted.\n"
                << logofs_flush;
        #endif

        continue;
      }
      else
      {
        #ifdef TEST
        *logofs << "Transport: Write on " << "FD#" 
                << fd_ << " failed.\n" << logofs_flush;
        #endif

        finish();

        return -1;
      }
    }
    else
    {
      #ifdef TEST
      *logofs << "Transport: Flushed " << result << " bytes on " 
              << "FD#" << fd_ << ".\n" << logofs_flush;
      #endif

      written += result;
    }
  }

  if (written > 0)
  {
    #ifdef DUMP

    *logofs << "Transport: Dumping content of flushed data.\n"
            << logofs_flush;

    DumpData(w_buffer_.data_.begin() + w_buffer_.start_, written);

    #endif

    //
    // Update buffer status.
    //

    w_buffer_.length_ -= written;

    if (w_buffer_.length_ == 0)
    {
      w_buffer_.start_ = 0;
    }
    else
    {
      w_buffer_.start_ += written;
    }
  }

  //
  // It can be that we wrote less bytes than
  // available because of the write limit.
  //

  if (w_buffer_.length_ > 0)
  {
    #ifdef TEST
    *logofs << "Transport: There are still " << w_buffer_.length_ 
            << " bytes in write buffer for " << "FD#" 
            << fd_ << ".\n" << logofs_flush;
    #endif

    blocked_ = 1;
  }

  #ifdef TEST
  *logofs << "Transport: Write buffer for FD#" << fd_
          << " has data for " << w_buffer_.length_ << " bytes.\n"
          << logofs_flush;

  *logofs << "Transport: Start is " << w_buffer_.start_ 
          << " length is " << w_buffer_.length_ << " size is "
          << w_buffer_.data_.size() << " capacity is "
          << w_buffer_.data_.capacity() << ".\n"
          << logofs_flush;
  #endif

  //
  // No new data was produced for the link except
  // any outstanding data from previous writes.
  //

  return 0;
}

//
// Forcibly write data to file descriptor until
// data left in buffer is below the limit.
//

int Transport::force(int limit)
{
  if (w_buffer_.length_ <= limit)
  {
    return 0;
  }

  forced_ = 1;

  //
  // Write only enough data to return in
  // the limit.
  //

  int toWrite = w_buffer_.length_ - limit;

  int written = 0;

  #ifdef TEST
  *logofs << "Transport: WARNING! Having to force " << toWrite
          << " bytes on FD#" << fd_ << " with limit set to "
          << limit << ".\n" << logofs_flush;
  #endif

  return 1;

  T_timestamp startTs = getTimestamp();

  T_timestamp selectTs;
  T_timestamp writeTs;
  T_timestamp idleTs;
  T_timestamp nowTs;

  int diffTs;

  fd_set writeSet;

  FD_ZERO(&writeSet);
  FD_SET(fd_, &writeSet);

  while (written < toWrite)
  {
    writeTs = getTimestamp();

    int result = ::write(fd_, w_buffer_.data_.begin() + w_buffer_.start_ +
                             written, toWrite - written);

    nowTs = getTimestamp();

    diffTs = diffTimestamp(writeTs, nowTs);

    control -> addWriteTime(diffTs);

    if (control -> CollectStatistics)
    {
      statistics -> addWriteTime(diffTs);
    }

    if (result <= 0)
    {
      if (EGET() == EAGAIN)
      {
        #ifdef TEST
        *logofs << "Transport: Write of " << toWrite - written
                << " bytes on FD#" << fd_ << " would block.\n"
                << logofs_flush;
        #endif

        //
        // Wait for descriptor to become writable.
        //

        setTimestamp(selectTs, control -> ChannelTimeout / 2);

        //
        // Save a system call using timestamp
        // got at the end of write.
        //

        idleTs = nowTs;

        select(fd_ + 1, NULL, &writeSet, NULL, &selectTs);

        nowTs = getTimestamp();

        diffTs = diffTimestamp(idleTs, nowTs);

        control -> addIdleTime(diffTs);

        control -> subReadTime(diffTs);

        if (control -> CollectStatistics)
        {
          statistics -> addIdleTime(diffTs);

          statistics -> subReadTime(diffTs);
        }

        diffTs = diffTimestamp(startTs, nowTs);

        if (diffTs >= (control -> ChannelTimeout -
                            (control -> LatencyTimeout * 10)))
        {
          int diffSec = (diffTs + control -> LatencyTimeout * 10) / 1000;

          #ifdef PANIC
          *logofs << "Transport: PANIC! Can't force write to FD#"
                  << fd_ << " since " << diffSec << " seconds.\n"
                  << logofs_flush;
          #endif

          cerr << "Error" << ": Can't write to connection on FD#"
               << fd_ << " since " << diffSec << " seconds.\n";

          if (control -> ProxyMode == proxy_client)
          {
            HandleAlert(CLOSE_DEAD_X_CONNECTION_CLIENT_ALERT, 1);
          }
          else
          {
            HandleAlert(CLOSE_DEAD_X_CONNECTION_SERVER_ALERT, 1);
          }

          #ifdef PANIC
          *logofs << "Transport: PANIC! Shutting down dead connection "
                  << "on FD#" << fd_ << ".\n" << logofs_flush;
          #endif

          cerr << "Error" << ": Shutting down dead connection "
               << "on FD#" << fd_ << ".\n";

          shutdown(fd_, SHUT_RDWR);

          break;
        }
        #ifdef WARNING
        else if (diffTs >= (control -> PingTimeout -
                                 (control -> LatencyTimeout * 10)))
        {
          int diffSec = (diffTs + control -> LatencyTimeout * 10) / 1000;

          *logofs << "Transport: WARNING: Couldn't force write to FD#"
                  << fd_ << " for " << diffSec << " seconds.\n"
                  << logofs_flush;
        }
        #endif
      }
      else if (EGET() == EINTR)
      {
        #ifdef TEST
        *logofs << "Transport: Write of " << toWrite - written
                << " bytes on FD#" << fd_ << " was interrupted.\n"
                << logofs_flush;
        #endif

        continue;
      }
      else
      {
        #ifdef TEST
        *logofs << "Transport: Write on " << "FD#" 
                << fd_ << " failed.\n" << logofs_flush;
        #endif

        finish();

        return -1;
      }
    }
    else
    {
      #ifdef TEST
      *logofs << "Transport: Forced flush of " << result
              << " bytes on " << "FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      written += result;
    }
  }

  if (written > 0)
  {
    #ifdef DUMP

    *logofs << "Transport: Dumping content of flushed data.\n"
            << logofs_flush;

    DumpData(w_buffer_.data_.begin() + w_buffer_.start_, written);

    #endif

    //
    // Update buffer status.
    //

    w_buffer_.length_ -= written;

    if (w_buffer_.length_ == 0)
    {
      w_buffer_.start_ = 0;

      blocked_ = 0;
    }
    else
    {
      w_buffer_.start_ += written;

      #ifdef TEST
      *logofs << "Transport: There are still " << w_buffer_.length_ 
              << " bytes in write buffer for " << "FD#" 
              << fd_ << ".\n" << logofs_flush;
      #endif

      blocked_ = 1;
    }
  }
  #ifdef TEST
  else
  {
    *logofs << "Transport: WARNING! No data written on FD#" << fd_
            << " with " << toWrite  << " bytes to force and limit "
            << "set to " << limit << ".\n" << logofs_flush;
  }
  #endif

  #ifdef TEST
  *logofs << "Transport: Write buffer for FD#" << fd_
          << " has data for " << w_buffer_.length_ << " bytes.\n"
          << logofs_flush;

  *logofs << "Transport: Start is " << w_buffer_.start_ 
          << " length is " << w_buffer_.length_ << " size is "
          << w_buffer_.data_.size() << " capacity is "
          << w_buffer_.data_.capacity() << ".\n"
          << logofs_flush;
  #endif

  return 1;
}

int Transport::wait(int timeout) const
{
  T_timestamp startTs = getTimestamp();

  T_timestamp idleTs;
  T_timestamp selectTs;

  T_timestamp nowTs = startTs;

  long available = 0;
  int  result    = 0;

  int diffTs;

  fd_set readSet;

  FD_ZERO(&readSet);
  FD_SET(fd_, &readSet);

  for (;;)
  {
    available = readable();

    diffTs = diffTimestamp(startTs, nowTs);

    if (available != 0 || timeout == 0 ||
            (diffTs + timeout / 10) >= timeout)
    {
      #ifdef TEST
      *logofs << "Transport: There are " << available
              << " bytes on FD#" << fd_ << " after "
              << diffTimestamp(startTs, getTimestamp())
              << " Ms.\n" << logofs_flush;
      #endif

      return (int) available;
    }
    else if (available == 0 && result > 0)
    {
      #ifdef TEST
      *logofs << "Transport: Read on " << "FD#" 
              << fd_ << " failed.\n" << logofs_flush;
      #endif

      return -1;
    }

    //
    // TODO: Should subtract the time
    // already spent in select.
    //

    selectTs.tv_sec  = 0;
    selectTs.tv_usec = timeout * 1000;

    idleTs = getTimestamp();

    //
    // Wait for descriptor to become readable.
    //

    result = select(fd_ + 1, &readSet, NULL, NULL, &selectTs);

    nowTs = getTimestamp();

    diffTs = diffTimestamp(idleTs, nowTs);

    control -> addIdleTime(diffTs);

    control -> subReadTime(diffTs);

    if (control -> CollectStatistics)
    {
      statistics -> addIdleTime(diffTs);

      statistics -> subReadTime(diffTs);
    }

    if (result < 0)
    {
      if (EGET() == EINTR)
      {
        #ifdef TEST
        *logofs << "Transport: Select on FD#" << fd_
                << " was interrupted.\n" << logofs_flush;
        #endif

        continue;
      }
      else
      {
        #ifdef TEST
        *logofs << "Transport: Select on " << "FD#" 
                << fd_ << " failed.\n" << logofs_flush;
        #endif

        return -1;
      }
    }
    #ifdef TEST
    else if (result == 0)
    {
      *logofs << "Transport: No data available on FD#" << fd_
              << " after " << diffTimestamp(startTs, getTimestamp())
              << " Ms.\n" << logofs_flush;
    }
    else
    {
      *logofs << "Transport: Data became available on FD#" << fd_
              << " after " << diffTimestamp(startTs, getTimestamp())
              << " Ms.\n" << logofs_flush;
    }
    #endif
  }
}

void Transport::setSize(unsigned int initialSize, unsigned int thresholdSize,
                            unsigned int maximumSize)
{
  initialSize_   = initialSize;
  thresholdSize_ = thresholdSize;
  maximumSize_   = maximumSize;

  #ifdef TEST
  *logofs << "Transport: Set buffer sizes for FD#" << fd_
          << " to " << initialSize_ << "/" << thresholdSize_
          << "/" << maximumSize_ << ".\n" << logofs_flush;
  #endif
}

int Transport::fullReset()
{
  blocked_ = 0;
  forced_  = 0;
  finish_  = 0;

  return fullReset(w_buffer_);
}

int Transport::resize(T_buffer &buffer, const int &size)
{
  if ((int) buffer.data_.size() >= (buffer.length_ + size) &&
          (buffer.start_ + buffer.length_ + size) >
               (int) buffer.data_.size())
  {
    //
    // There is enough space in buffer but we need
    // to move existing data at the beginning.
    //

    #ifdef TEST
    *logofs << "Transport: Moving " << buffer.length_
            << " bytes of data for " << "FD#" << fd_
            << " to make room in the buffer.\n"
            << logofs_flush;
    #endif

    memmove(buffer.data_.begin(), buffer.data_.begin() +
                buffer.start_, buffer.length_);

    buffer.start_ = 0;

    #ifdef DEBUG
    *logofs << "Transport: Made room for " 
            << buffer.data_.size() - buffer.start_
            << " bytes in buffer for " << "FD#" 
            << fd_ << ".\n" << logofs_flush;
    #endif
  }
  else if ((buffer.length_ + size) > (int) buffer.data_.size())
  {
    //
    // Not enough space, so increase
    // size of the buffer.
    //

    if (buffer.start_ != 0)
    {
      #ifdef TEST
      *logofs << "Transport: Moving " << buffer.length_
              << " bytes of data for " << "FD#" << fd_
              << " to resize the buffer.\n"
              << logofs_flush;
      #endif

      memmove(buffer.data_.begin(), buffer.data_.begin() +
                  buffer.start_, buffer.length_);
    }

    buffer.start_ = 0;

    unsigned int newSize = thresholdSize_;

    while (newSize < (unsigned int) buffer.length_ + size)
    {
      newSize <<= 1;

      if (newSize > maximumSize_)
      {
        newSize = buffer.length_ + size + initialSize_;
      }
    }

    #ifdef DEBUG
    *logofs << "Transport: Buffer for " << "FD#" << fd_
            << " will be enlarged from " << buffer.data_.size()
            << " to at least " << buffer.length_ + size
            << " bytes.\n" << logofs_flush;
    #endif

    try
    {
      buffer.data_.resize(newSize);
    }
    catch (exception e)
    {
      #ifdef PANIC
      *logofs << "Transport: PANIC! Buffer resize for FD#" << fd_
              << " from " << buffer.data_.size() << " to " << newSize
              << " bytes failed.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Buffer resize for FD#" << fd_
           << " from " << buffer.data_.size() << " to " << newSize
           << " bytes failed.\n";

      finish();

      return -1;
    }

    #ifdef TEST
    if (newSize >= maximumSize_)
    {
      *logofs << "Transport: WARNING! Buffer for FD#" << fd_
              << " grown to reach size of " << newSize
              << " bytes.\n" << logofs_flush;
    }
    #endif

    #ifdef TEST
    *logofs << "Transport: Data buffer for " << "FD#" 
            << fd_ << " has now size " << buffer.data_.size() 
            << " and capacity " << buffer.data_.capacity()
            << ".\n" << logofs_flush;
    #endif
  }

  return (buffer.length_ + size);
}

int Transport::fullReset(T_buffer &buffer)
{
  //
  // Force deallocation and allocation
  // of the initial size.
  //

  buffer.start_  = 0;
  buffer.length_ = 0;

  if (buffer.data_.size() == (unsigned int) initialSize_ &&
          buffer.data_.capacity() == (unsigned int) initialSize_)
  {
    return 0;
  }

  try
  {
    buffer.data_.clear();

    buffer.data_.resize(initialSize_);
  }
  catch (exception e)
  {
    #ifdef PANIC
    *logofs << "Transport: PANIC! Buffer shrink failed for "
            << "FD#" << fd_ << ".\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Buffer shrink failed for "
         << "FD#" << fd_ << ".\n";

    finish();

    return -1;
  }

  #ifdef TEST
  *logofs << "Transport: Data buffer for " << "FD#"
          << fd_ << " shrunk to size " << buffer.data_.size()
          << " and capacity " << buffer.data_.capacity()
          << ".\n" << logofs_flush;
  #endif

  return 1;
}

ProxyTransport::ProxyTransport(int fd) : Transport(fd)
{
  #ifdef TEST
  *logofs << "ProxyTransport: Going to create proxy transport "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  type_ = transport_proxy;

  //
  // Set up the read buffer.
  //

  r_buffer_.length_ = 0;
  r_buffer_.start_  = 0;

  r_buffer_.data_.resize(initialSize_);

  //
  // Set up ZLIB compression.
  //

  int result;

  r_stream_.zalloc = NULL;
  r_stream_.zfree  = NULL;
  r_stream_.opaque = NULL;

  r_stream_.next_in  = NULL;
  r_stream_.avail_in = 0;

  if ((result = inflateInit2(&r_stream_, 15)) != Z_OK)
  {
    #ifdef PANIC
    *logofs << "ProxyTransport: PANIC! Failed initialization of ZLIB read stream. "
            << "Error is '" << zError(result) << "'.\n" << logofs_flush;
    #endif

    cerr << "Error" << ": Failed initialization of ZLIB read stream. "
         << "Error is '" << zError(result) << "'.\n";

    HandleCleanup();
  }

  if (control -> LocalStreamCompression)
  {
    w_stream_.zalloc = NULL;
    w_stream_.zfree  = NULL;
    w_stream_.opaque = NULL;

    if ((result = deflateInit2(&w_stream_, control -> LocalStreamCompressionLevel, Z_DEFLATED,
                                   15, 9, Z_DEFAULT_STRATEGY)) != Z_OK)
    {
      #ifdef PANIC
      *logofs << "ProxyTransport: PANIC! Failed initialization of ZLIB write stream. "
              << "Error is '" << zError(result) << "'.\n" << logofs_flush;
      #endif

      cerr << "Error" << ": Failed initialization of ZLIB write stream. "
           << "Error is '" << zError(result) << "'.\n";

      HandleCleanup();
    }
  }

  //
  // No ZLIB stream to flush yet.
  //

  flush_ = 0;

  #ifdef REFERENCES
  *logofs << "ProxyTransport: Created new object at " 
          << this << " out of " << ++references_ 
          << " allocated references.\n" << logofs_flush;
  #endif
}

ProxyTransport::~ProxyTransport()
{
  #ifdef TEST
  *logofs << "ProxyTransport: Going to destroy derived class "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // Free ZLIB dynamic structures.
  //

  inflateEnd(&r_stream_);

  if (control -> LocalStreamCompression)
  {
    deflateEnd(&w_stream_);
  }

  #ifdef REFERENCES
  *logofs << "ProxyTransport: Deleted object at " 
          << this << " out of " << --references_ 
          << " allocated references.\n" << logofs_flush;
  #endif
}

//
// Read data from its file descriptor.
//

int ProxyTransport::read(unsigned char *data, unsigned int size)
{
  //
  // If remote peer is not compressing the stream
  // then just return any byte read from socket.
  //

  if (control -> RemoteStreamCompression == 0)
  {
    int result = Transport::read(data, size);

    if (result <= 0)
    {
      return result;
    }

    control -> addBytesIn(result);

    if (control -> CollectStatistics)
    {
      statistics -> addBytesIn(result);
    }

    return result;
  }

  //
  // Return any pending data first.
  //

  if (r_buffer_.length_ > 0)
  {
    int copied = (r_buffer_.length_ > ((int) size) ?
                      ((int) size) : r_buffer_.length_);

    memcpy(data, r_buffer_.data_.begin() + r_buffer_.start_, copied);

    //
    // Update buffer status.
    //

    #ifdef DEBUG
    *logofs << "ProxyTransport: Going to immediately return " << copied
            << " bytes from proxy FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    r_buffer_.length_ -= copied;

    if (r_buffer_.length_ == 0)
    {
      r_buffer_.start_ = 0;
    }
    else
    {
      r_buffer_.start_ += copied;

      #ifdef TEST
      *logofs << "ProxyTransport: There are still " << r_buffer_.length_ 
              << " bytes in read buffer for proxy " << "FD#" 
              << fd_ << ".\n" << logofs_flush;
        #endif
    }

    return copied;
  }

  //
  // Read data in the user buffer.
  //

  int result = Transport::read(data, size);

  if (result <= 0)
  {
    return result;
  }

  control -> addBytesIn(result);

  if (control -> CollectStatistics)
  {
    statistics -> addBytesIn(result);
  }

  //
  // Uncompress data into the read buffer.
  //

  #ifdef DEBUG
  *logofs << "ProxyTransport: Going to uncompress data for " 
          << "proxy FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  int saveTotalIn  = r_stream_.total_in;
  int saveTotalOut = r_stream_.total_out;

  int oldTotalIn  = saveTotalIn;
  int oldTotalOut = saveTotalOut;

  int diffTotalIn;
  int diffTotalOut;

  #ifdef INSPECT
  *logofs << "ProxyTransport: oldTotalIn = " << oldTotalIn
          << ".\n" << logofs_flush;
  #endif

  #ifdef INSPECT
  *logofs << "ProxyTransport: oldTotalOut = " << oldTotalOut
          << ".\n" << logofs_flush;
  #endif

  r_stream_.next_in  = (Bytef *) data;
  r_stream_.avail_in = result;

  //
  // Initial buffer will be twice the
  // amount of data read from socket.
  //

  int newAvailOut = (result < 1024 ? 2048 : result * 2);

  #ifdef TEST
  *logofs << "ProxyTransport: Initial decompress buffer is "
          << newAvailOut << " bytes.\n" << logofs_flush;
  #endif

  for (;;)
  {
    #ifdef INSPECT
    *logofs << "\nProxyTransport: Running uncompress loop.\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_buffer_.length_ = " << r_buffer_.length_
            << ".\n" << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_buffer_.data_.size() = " << r_buffer_.data_.size()
            << ".\n" << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: newAvailOut = " << newAvailOut
            << ".\n" << logofs_flush;
    #endif

    if (resize(r_buffer_, newAvailOut) < 0)
    {
      return -1;
    }

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_buffer_.data_.size() = "
            << r_buffer_.data_.size() << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_stream_.next_in = "
            << (void *) r_stream_.next_in << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_stream_.avail_in = "
            << r_stream_.avail_in << ".\n"
            << logofs_flush;
    #endif

    r_stream_.next_out  = r_buffer_.data_.begin() + r_buffer_.start_ +
                              r_buffer_.length_;

    r_stream_.avail_out = newAvailOut;

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_stream_.next_out = "
            << (void *) r_stream_.next_out << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_stream_.avail_out = "
            << r_stream_.avail_out << ".\n"
            << logofs_flush;
    #endif

    int result = inflate(&r_stream_, Z_SYNC_FLUSH);

    #ifdef INSPECT
    *logofs << "ProxyTransport: Called inflate() result is "
            << result << ".\n" << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_stream_.avail_in = "
            << r_stream_.avail_in << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_stream_.avail_out = "
            << r_stream_.avail_out << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_stream_.total_in = "
            << r_stream_.total_in << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_stream_.total_out = "
            << r_stream_.total_out << ".\n"
            << logofs_flush;
    #endif

    diffTotalIn  = r_stream_.total_in  - oldTotalIn;
    diffTotalOut = r_stream_.total_out - oldTotalOut;

    #ifdef INSPECT
    *logofs << "ProxyTransport: diffTotalIn = "
            << diffTotalIn << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: diffTotalOut = "
            << diffTotalOut << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_buffer_.length_ = "
            << r_buffer_.length_ << ".\n"
            << logofs_flush;
    #endif

    r_buffer_.length_ += diffTotalOut;

    #ifdef INSPECT
    *logofs << "ProxyTransport: r_buffer_.length_ = "
            << r_buffer_.length_ << ".\n"
            << logofs_flush;
    #endif

    oldTotalIn  = r_stream_.total_in;
    oldTotalOut = r_stream_.total_out;

    if (result == Z_OK)
    {
      if (r_stream_.avail_in == 0 && r_stream_.avail_out > 0)
      {
        break;
      }
    }
    else if (result == Z_BUF_ERROR && r_stream_.avail_out > 0 &&
                 r_stream_.avail_in == 0)
    {
      #ifdef TEST
      *logofs << "ProxyTransport: WARNING! Raised Z_BUF_ERROR decompressing data.\n"
              << logofs_flush;
      #endif

      break;
    }
    else
    {
      #ifdef PANIC
      *logofs << "ProxyTransport: PANIC! Decompression of data failed. "
              << "Error is '" << zError(result) << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Decompression of data failed. Error is '"
           << zError(result) << "'.\n";

      finish();

      return -1;
    }

    #ifdef TEST
    *logofs << "ProxyTransport: Need to add " << newAvailOut
            << " bytes to decompress buffer.\n"
            << logofs_flush;
    #endif
  }

  diffTotalIn  = r_stream_.total_in  - saveTotalIn;
  diffTotalOut = r_stream_.total_out - saveTotalOut;

  #ifdef DEBUG
  *logofs << "ProxyTransport: Decompressed data from "
          << diffTotalIn << " to " << diffTotalOut
          << " bytes.\n" << logofs_flush;
  #endif

  if (control -> CollectStatistics)
  {
    statistics -> addDecompressedBytes(diffTotalIn, diffTotalOut);
  }

  //
  // Copy the uncompressed data to the provided buffer.
  //

  int copied = (r_buffer_.length_ > ((int) size) ?
                    ((int) size) : r_buffer_.length_);

  #ifdef DEBUG
  *logofs << "ProxyTransport: Going to return " << copied
          << " bytes from proxy FD#" << fd_ << ".\n"
          << logofs_flush;
  #endif

  memcpy(data, r_buffer_.data_.begin() + r_buffer_.start_, copied);

  //
  // Update buffer status.
  //

  r_buffer_.length_ -= copied;

  if (r_buffer_.length_ == 0)
  {
    r_buffer_.start_ = 0;
  }
  else
  {
    r_buffer_.start_ += copied;

    #ifdef TEST
    *logofs << "ProxyTransport: There are still " << r_buffer_.length_ 
            << " bytes in read buffer for proxy FD#" << fd_
            << ".\n" << logofs_flush;
    #endif
  }

  return copied;
}

//
// If required compress data, else write it to socket.
//

int ProxyTransport::write(T_write type, const unsigned char *data, const unsigned int size)
{
  #ifdef TEST
  if (size == 0)
  {
    *logofs << "ProxyTransport: WARNING! Write called for FD#" 
            << fd_ << " without any data to write.\n"
            << logofs_flush;

    return 0;
  }
  #endif

  //
  // If there is no compression revert to
  // plain socket management.
  //

  if (control -> LocalStreamCompression == 0)
  {
    int result = Transport::write(type, data, size);

    if (result <= 0)
    {
      return result;
    }

    control -> addBytesOut(result);

    if (control -> CollectStatistics)
    {
      statistics -> addBytesOut(result);
    }

    return result;
  }

  #ifdef DEBUG
  *logofs << "ProxyTransport: Going to compress " << size 
          << " bytes to write buffer for proxy FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  //
  // Compress data into the write buffer.
  //

  int saveTotalIn  = w_stream_.total_in;
  int saveTotalOut = w_stream_.total_out;

  int oldTotalIn  = saveTotalIn;
  int oldTotalOut = saveTotalOut;

  int diffTotalIn;
  int diffTotalOut;

  #ifdef INSPECT
  *logofs << "ProxyTransport: oldTotalIn = " << oldTotalIn
          << ".\n" << logofs_flush;
  #endif

  #ifdef INSPECT
  *logofs << "ProxyTransport: oldTotalOut = " << oldTotalOut
          << ".\n" << logofs_flush;
  #endif

  w_stream_.next_in  = (Bytef *) data;
  w_stream_.avail_in = size;

  //
  // Initial buffer will be the amount of data to
  // compress plus the data to flush plus overhead.
  //

  int newAvailOut = (size + flush_) + ((size + flush_) / 100) + 12;

  #ifdef TEST
  *logofs << "ProxyTransport: Initial compress buffer is "
          << newAvailOut << " bytes.\n" << logofs_flush;
  #endif

  for (;;)
  {
    #ifdef INSPECT
    *logofs << "\nProxyTransport: Running compress loop.\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_buffer_.length_ = "
            << w_buffer_.length_ << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_buffer_.data_.size() = "
            << w_buffer_.data_.size() << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: newAvailOut = "
            << newAvailOut << ".\n"
            << logofs_flush;
    #endif

    if (resize(w_buffer_, newAvailOut) < 0)
    {
      return -1;
    }

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_buffer_.data_.size() = "
            << w_buffer_.data_.size() << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.next_in = "
            << (void *) w_stream_.next_in << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.avail_in = "
            << w_stream_.avail_in << ".\n"
            << logofs_flush;
    #endif

    w_stream_.next_out  = w_buffer_.data_.begin() + w_buffer_.start_ +
                              w_buffer_.length_;

    w_stream_.avail_out = newAvailOut;

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.next_out = "
            << (void *) w_stream_.next_out << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.avail_out = "
            << w_stream_.avail_out << ".\n"
            << logofs_flush;
    #endif

    int result = deflate(&w_stream_, (type == write_delayed ?
                             Z_NO_FLUSH : Z_SYNC_FLUSH));

    #ifdef INSPECT
    *logofs << "ProxyTransport: Called deflate() result is "
            << result << ".\n" << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.avail_in = "
            << w_stream_.avail_in << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.avail_out = "
            << w_stream_.avail_out << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.total_in = "
            << w_stream_.total_in << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.total_out = "
            << w_stream_.total_out << ".\n"
            << logofs_flush;
    #endif

    diffTotalOut = w_stream_.total_out - oldTotalOut;
    diffTotalIn  = w_stream_.total_in  - oldTotalIn;

    #ifdef INSPECT
    *logofs << "ProxyTransport: diffTotalIn = "
            << diffTotalIn << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: diffTotalOut = "
            << diffTotalOut << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_buffer_.length_ = "
            << w_buffer_.length_ << ".\n"
            << logofs_flush;
    #endif

    w_buffer_.length_ += diffTotalOut;

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_buffer_.length_ = "
            << w_buffer_.length_ << ".\n"
            << logofs_flush;
    #endif

    oldTotalOut = w_stream_.total_out;
    oldTotalIn  = w_stream_.total_in;

    if (result == Z_OK)
    {
      if (w_stream_.avail_in == 0 && w_stream_.avail_out > 0)
      {
        break;
      }
    }
    else if (result == Z_BUF_ERROR && w_stream_.avail_out > 0 &&
                 w_stream_.avail_in == 0)
    {
      #ifdef TEST
      *logofs << "ProxyTransport: WARNING! Raised Z_BUF_ERROR compressing data.\n"
              << logofs_flush;
      #endif

      break;
    }
    else
    {
      #ifdef PANIC
      *logofs << "ProxyTransport: PANIC! Compression of data failed. "
              << "Error is '" << zError(result) << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Compression of data failed. Error is '"
           << zError(result) << "'.\n";

      finish();

      return -1;
    }

    #ifdef TEST
    *logofs << "ProxyTransport: Need to add " << newAvailOut
            << " bytes to compress buffer.\n"
            << logofs_flush;
    #endif
  }

  diffTotalIn  = w_stream_.total_in  - saveTotalIn;
  diffTotalOut = w_stream_.total_out - saveTotalOut;

  #ifdef TEST

  *logofs << "ProxyTransport: Compressed data from "
          << diffTotalIn << " to " << diffTotalOut
          << " bytes.\n" << logofs_flush;

  if (diffTotalIn != (int) size)
  {
    #ifdef PANIC
    *logofs << "ProxyTransport: PANIC! Bytes provided to ZLIB stream "
            << "should be " << size << " but they look to be "
            << diffTotalIn << ".\n" << logofs_flush;
    #endif
  }

  #endif

  //
  // We added bytes to write buffer. Data is not
  // written to network yet, but it's convenient
  // to update the counters at this stage to get
  // the current bitrate earlier.
  //

  control -> addBytesOut(diffTotalOut);

  if (control -> CollectStatistics)
  {
    statistics -> addCompressedBytes(diffTotalIn, diffTotalOut);

    statistics -> addBytesOut(diffTotalOut);
  }

  //
  // Find out what to do with the
  // produced data.
  //

  if (type == write_immediate)
  {
    //
    // If user requested an immediate write we
    // flushed the ZLIB buffer. We can now reset
    // the counter and write data to socket.
    //

    flush_ = 0;

    #ifdef TEST
    *logofs << "ProxyTransport: Write buffer for proxy FD#" << fd_
            << " has data for " << w_buffer_.length_ << " bytes.\n"
            << logofs_flush;

    *logofs << "ProxyTransport: Start is " << w_buffer_.start_ 
            << " length is " << w_buffer_.length_ << " flush is "
            << flush_ << " size is " << w_buffer_.data_.size()
            << " capacity is " << w_buffer_.data_.capacity()
            << ".\n" << logofs_flush;
    #endif

    if (w_buffer_.length_ > 0 && blocked_ == 0)
    {
      #ifdef TEST
      *logofs << "ProxyTransport: Writing " << w_buffer_.length_
              << " bytes of produced data to FD#" << fd_ << ".\n"
              << logofs_flush;
      #endif

      int result = Transport::flush();

      if (result < 0)
      {
        return -1;
      }
    }
  }
  else
  {
    //
    // We haven't flushed the ZLIB compression
    // buffer, so user will have to call proxy
    // transport's flush explicitly.
    //

    flush_ += diffTotalIn;
  }

  #ifdef TEST
  *logofs << "ProxyTransport: Write buffer for proxy FD#" << fd_
          << " has data for " << w_buffer_.length_ << " bytes.\n"
          << logofs_flush;

  *logofs << "ProxyTransport: Start is " << w_buffer_.start_ 
          << " length is " << w_buffer_.length_ << " flush is "
          << flush_ << " size is " << w_buffer_.data_.size()
          << " capacity is " << w_buffer_.data_.capacity()
          << ".\n" << logofs_flush;
  #endif

  return size;
}

//
// Write data to its file descriptor.
//

int ProxyTransport::flush()
{
  //
  // If there is no compression or we already compressed
  // outgoing data and just need to write it to socket
  // because of previous incomplete writes then revert
  // to plain socket management.
  //

  if (flush_ == 0 || control -> LocalStreamCompression == 0)
  {
    int result = Transport::flush();

    if (result < 0)
    {
      return -1;
    }

    return result;
  }

  #ifdef DEBUG
  *logofs << "ProxyTransport: Going to flush compression on "
          << "proxy FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  #ifdef TEST
  *logofs << "ProxyTransport: Flush counter for proxy FD#" << fd_
          << " is " << flush_ << " bytes.\n" << logofs_flush;
  #endif

  //
  // Flush ZLIB stream into the write buffer.
  //

  int saveTotalIn  = w_stream_.total_in;
  int saveTotalOut = w_stream_.total_out;

  int oldTotalIn  = saveTotalIn;
  int oldTotalOut = saveTotalOut;

  int diffTotalOut;
  int diffTotalIn;

  #ifdef INSPECT
  *logofs << "ProxyTransport: oldTotalIn = " << oldTotalIn
          << ".\n" << logofs_flush;
  #endif

  #ifdef INSPECT
  *logofs << "ProxyTransport: oldTotalOut = " << oldTotalOut
          << ".\n" << logofs_flush;
  #endif

  w_stream_.next_in  = w_buffer_.data_.begin() + w_buffer_.start_ + w_buffer_.length_;
  w_stream_.avail_in = 0;

  //
  // Initial buffer will be the amount
  // of data to flush plus overhead.
  //

  int newAvailOut = flush_ + (flush_ / 100) + 12;

  #ifdef DEBUG
  *logofs << "ProxyTransport: Initial flush buffer is "
          << newAvailOut << " bytes.\n" << logofs_flush;
  #endif

  for (;;)
  {
    #ifdef INSPECT
    *logofs << "\nProxyTransport: Running flush loop.\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_buffer_.length_ = "
            << w_buffer_.length_ << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_buffer_.data_.size() = "
            << w_buffer_.data_.size() << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: newAvailOut = "
            << newAvailOut << ".\n"
            << logofs_flush;
    #endif

    if (resize(w_buffer_, newAvailOut) < 0)
    {
      return -1;
    }

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_buffer_.data_.size() = "
            << w_buffer_.data_.size() << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.next_in = "
            << (void *) w_stream_.next_in << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.avail_in = "
            << w_stream_.avail_in << ".\n"
            << logofs_flush;
    #endif

    w_stream_.next_out  = w_buffer_.data_.begin() + w_buffer_.start_ +
                              w_buffer_.length_;

    w_stream_.avail_out = newAvailOut;

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.next_out = "
            << (void *) w_stream_.next_out << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.avail_out = "
            << w_stream_.avail_out << ".\n"
            << logofs_flush;
    #endif

    int result = deflate(&w_stream_, Z_SYNC_FLUSH);

    #ifdef INSPECT
    *logofs << "ProxyTransport: Called deflate() result is "
            << result << ".\n" << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.avail_in = "
            << w_stream_.avail_in << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.avail_out = "
            << w_stream_.avail_out << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.total_in = "
            << w_stream_.total_in << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_stream_.total_out = "
            << w_stream_.total_out << ".\n"
            << logofs_flush;
    #endif

    diffTotalOut = w_stream_.total_out - oldTotalOut;
    diffTotalIn  = w_stream_.total_in  - oldTotalIn;

    #ifdef INSPECT
    *logofs << "ProxyTransport: diffTotalIn = "
            << diffTotalIn << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: diffTotalOut = "
            << diffTotalOut << ".\n"
            << logofs_flush;
    #endif

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_buffer_.length_ = "
            << w_buffer_.length_ << ".\n"
            << logofs_flush;
    #endif

    w_buffer_.length_ += diffTotalOut;

    #ifdef INSPECT
    *logofs << "ProxyTransport: w_buffer_.length_ = "
            << w_buffer_.length_ << ".\n"
            << logofs_flush;
    #endif

    oldTotalOut = w_stream_.total_out;
    oldTotalIn  = w_stream_.total_in;

    if (result == Z_OK)
    {
      if (w_stream_.avail_in == 0 && w_stream_.avail_out > 0)
      {
        break;
      }
    }
    else if (result == Z_BUF_ERROR && w_stream_.avail_out > 0 &&
                 w_stream_.avail_in == 0)
    {
      #ifdef TEST
      *logofs << "ProxyTransport: WARNING! Raised Z_BUF_ERROR flushing data.\n"
              << logofs_flush;
      #endif

      break;
    }
    else
    {
      #ifdef PANIC
      *logofs << "ProxyTransport: PANIC! Flush of compressed data failed. "
              << "Error is '" << zError(result) << "'.\n"
              << logofs_flush;
      #endif

      cerr << "Error" << ": Flush of compressed data failed. Error is '"
           << zError(result) << "'.\n";

      finish();

      return -1;
    }

    #ifdef TEST
    *logofs << "ProxyTransport: Need to add " << newAvailOut
            << " bytes to compress buffer in flush.\n"
            << logofs_flush;
    #endif
  }

  diffTotalIn  = w_stream_.total_in  - saveTotalIn;
  diffTotalOut = w_stream_.total_out - saveTotalOut;

  #ifdef TEST
  *logofs << "ProxyTransport: Compressed flush data from "
          << diffTotalIn << " to " << diffTotalOut
          << " bytes.\n" << logofs_flush;
  #endif

  control -> addBytesOut(diffTotalOut);

  if (control -> CollectStatistics)
  {
    statistics -> addCompressedBytes(diffTotalIn, diffTotalOut);

    statistics -> addBytesOut(diffTotalOut);
  }

  //
  // Time to move data from the write
  // buffer to the real link. 
  //

  #ifdef DEBUG
  *logofs << "ProxyTransport: Reset flush counter for proxy FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  flush_ = 0;

  #ifdef TEST
  *logofs << "ProxyTransport: Write buffer for proxy FD#" << fd_
          << " has data for " << w_buffer_.length_ << " bytes.\n"
          << logofs_flush;

  *logofs << "ProxyTransport: Start is " << w_buffer_.start_ 
          << " length is " << w_buffer_.length_ << " flush is "
          << flush_ << " size is " << w_buffer_.data_.size()
          << " capacity is " << w_buffer_.data_.capacity()
          << ".\n" << logofs_flush;
  #endif

  int result = Transport::flush();

  if (result < 0)
  {
    return -1;
  }

  return result;
}

int ProxyTransport::force(int limit)
{
  #ifdef PANIC
  *logofs << "ProxyTransport: PANIC! Called force() for "
          << "proxy transport on " << "FD#" << fd_ << ".\n"
          << logofs_flush;
  #endif

  cerr << "Error" << ": Called force() for proxy transport "
       << "on " << "FD#" << fd_ << ".\n";

  HandleAbort();

  return -1;
}

int ProxyTransport::fullReset()
{
  blocked_ = 0;
  forced_  = 0;
  finish_  = 0;
  flush_   = 0;

  int result = 0;

  if (control -> RemoteStreamCompression &&
          inflateReset(&r_stream_) == Z_OK)
  {
    result++;
  }

  if (control -> LocalStreamCompression &&
          deflateReset(&w_stream_) == Z_OK)
  {
    result++;
  }

  if (Transport::fullReset(r_buffer_) >= 0)
  {
    result++;
  }

  if (Transport::fullReset(w_buffer_) >= 0)
  {
    result++;
  }

  return (result == 4);
}

AgentTransport::AgentTransport(int fd) : Transport(fd)
{
  #ifdef TEST
  *logofs << "AgentTransport: Going to create agent transport "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  type_ = transport_agent;

  //
  // Set up the read buffer.
  //

  r_buffer_.length_ = 0;
  r_buffer_.start_  = 0;

  r_buffer_.data_.resize(initialSize_);

  //
  // Set up the mutexes.
  //

  #ifdef THREADS

  pthread_mutexattr_t m_attributes;

  pthread_mutexattr_init(&m_attributes);

  //
  // Interfaces in pthread to handle mutex
  // type do not work in current version. 
  //

  m_attributes.__mutexkind = PTHREAD_MUTEX_ERRORCHECK_NP;

  if (pthread_mutex_init(&m_read_, &m_attributes) != 0)
  {
    #ifdef TEST
    *logofs << "AgentTransport: Child: Creation of read mutex failed. "
            << "Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif
  }

  if (pthread_mutex_init(&m_write_, &m_attributes) != 0)
  {
    #ifdef TEST
    *logofs << "AgentTransport: Child: Creation of write mutex failed. "
            << "Error is " << EGET() << " '" << ESTR()
            << "'.\n" << logofs_flush;
    #endif
  }

  #endif

  #ifdef REFERENCES
  *logofs << "AgentTransport: Child: Created new object at " 
          << this << " out of " << ++references_ 
          << " allocated references.\n" << logofs_flush;
  #endif
}

AgentTransport::~AgentTransport()
{
  #ifdef TEST
  *logofs << "AgentTransport: Going to destroy derived class "
          << "for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // Unlock and free all mutexes.
  //

  #ifdef THREADS

  pthread_mutex_unlock(&m_read_);
  pthread_mutex_unlock(&m_write_);

  pthread_mutex_destroy(&m_read_);
  pthread_mutex_destroy(&m_write_);

  #endif

  #ifdef REFERENCES
  *logofs << "AgentTransport: Child: Deleted object at " 
          << this << " out of " << --references_ 
          << " allocated references.\n" << logofs_flush;
  #endif
}

//
// Read data enqueued by the other thread.
//

int AgentTransport::read(unsigned char *data, unsigned int size)
{
  #ifdef THREADS

  lockRead();

  #endif

  #ifdef DEBUG
  *logofs << "AgentTransport: Child: Going to read " << size
          << " bytes from " << "FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  int copied = 0;

  if (r_buffer_.length_ > 0)
  {
    copied = (r_buffer_.length_ > ((int) size) ?
                  ((int) size) : r_buffer_.length_);

    memcpy(data, r_buffer_.data_.begin() + r_buffer_.start_, copied);

    //
    // Update buffer status.
    //

    #ifdef TEST
    *logofs << "AgentTransport: Child: Going to immediately return "
            << copied << " bytes from FD#" << fd_ << ".\n" << logofs_flush;
    #endif

    #ifdef DUMP

    *logofs << "AgentTransport: Child: Dumping content of read data.\n"
            << logofs_flush;

    DumpData(data, copied);

    #endif

    r_buffer_.length_ -= copied;

    if (r_buffer_.length_ == 0)
    {
      r_buffer_.start_ = 0;
    }
    else
    {
      r_buffer_.start_ += copied;

      #ifdef TEST
      *logofs << "AgentTransport: Child: There are still " << r_buffer_.length_ 
              << " bytes in read buffer for proxy " << "FD#" << fd_
              << ".\n" << logofs_flush;
      #endif
    }
  }
  else
  {
    #ifdef DEBUG
    *logofs << "AgentTransport: Child: No data can be gotten "
            << "from read buffer for FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    ESET(EAGAIN);

    #ifdef THREADS

    unlockRead();

    #endif

    return 0;
  }

  #ifdef THREADS

  unlockRead();

  #endif

  return copied;
}

//
// Write data to buffer so that the other
// thread can get it.
//

int AgentTransport::write(T_write type, const unsigned char *data, const unsigned int size)
{
  #ifdef THREADS

  lockWrite();

  #endif

  //
  // Just append data to socket's write buffer.
  // Note that we presently don't care if buffer
  // exceeds the limit.
  //

  #ifdef TEST
  *logofs << "AgentTransport: Child: Going to append " << size
          << " bytes to write buffer for " << "FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  if (resize(w_buffer_, size) < 0)
  {
    finish();

    #ifdef THREADS

    unlockWrite();

    #endif

    return -1;
  }

  memmove(w_buffer_.data_.begin() + w_buffer_.start_ + w_buffer_.length_, data, size);

  w_buffer_.length_ += size;

  #ifdef DUMP

  *logofs << "AgentTransport: Child: Dumping content of written data.\n"
          << logofs_flush;

  DumpData(data, size);

  #endif

  #ifdef TEST
  *logofs << "AgentTransport: Child: Write buffer for FD#" << fd_
          << " has data for " << w_buffer_.length_ << " bytes.\n"
          << logofs_flush;

  *logofs << "AgentTransport: Child: Start is " << w_buffer_.start_ 
          << " length is " << w_buffer_.length_ << " size is "
          << w_buffer_.data_.size() << " capacity is "
          << w_buffer_.data_.capacity() << ".\n"
          << logofs_flush;
  #endif

  //
  // Let child to access the read buffer.
  //

  #ifdef THREADS

  unlockWrite();

  #endif

  return size;
}

int AgentTransport::flush()
{
  //
  // In case of memory-to-memory transport
  // this function should never be called.
  //

  #ifdef WARNING
  *logofs << "AgentTransport: Child: WARNING! Called flush() for "
          << "memory to memory transport on " << "FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  cerr << "Warning" << ": Called flush() for "
       << "memory to memory transport on " << "FD#"
       << fd_ << ".\n";

  return 0;
}

int AgentTransport::force(int limit)
{
  //
  // This function does nothing in the case of
  // memory-to-memory transport. Data has been
  // enqueued for the agent to read but agent
  // could require multiple loops to read it
  // all.
  //

  #ifdef TEST
  *logofs << "AgentTransport: Child: WARNING! Called force() for "
          << "memory to memory transport on " << "FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  return 0;
}

int AgentTransport::writable() const
{
  #if defined(PARENT) && (defined(TEST) || defined(INFO))
  *logofs << "AgentTransport: Child: WARNING! Called for "
          << "FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  int writable = control -> TransportXBufferThreshold -
                     w_buffer_.length_;

  if (writable < 0)
  {
    writable = 0;
  }

  #if defined(PARENT) && (defined(TEST) || defined(INFO))
  if (writable <= 0)
  {
    *logofs << "AgentTransport: Child: WARNING! No data is "
            << "writable on write buffer for FD#" << fd_
            << ".\n" << logofs_flush;
  }
  #endif

  return writable;
}

int AgentTransport::fullReset()
{
  #ifdef THREADS

  lockRead();
  lockWrite();

  #endif

  blocked_ = 0;
  forced_  = 0;
  finish_  = 0;

  int result = 0;

  if (Transport::fullReset(r_buffer_) >= 0)
  {
    result++;
  }

  if (Transport::fullReset(w_buffer_) >= 0)
  {
    result++;
  }

  return (result == 2);
}

int AgentTransport::enqueue(const char *data, const int size)
{
  #ifdef THREADS

  lockRead();

  #endif

  //
  // Don't allow the agent to write more data
  // than the threshold.
  //

  int toPut = control -> TransportXBufferThreshold -
                   r_buffer_.length_;

  if (toPut <= 0)
  {
    #if defined(PARENT) && defined(TEST)
    *logofs << "AgentTransport: Parent: WARNING! Read buffer for FD#"
            << fd_ << " has already reached the limit of "
            << control -> TransportXBufferThreshold
            << " bytes.\n" << logofs_flush;
    #endif

    ESET(EAGAIN);

    #ifdef THREADS

    unlockRead();

    #endif

    return -1;
  }

  toPut = (toPut > size ? size : toPut);

  #if defined(PARENT) && defined(TEST)
  *logofs << "AgentTransport: Parent: Going to put " << toPut
          << " bytes into " << "read buffer for FD#" << fd_
          << ". Buffer length is " << r_buffer_.length_
          << ".\n" << logofs_flush;
  #endif

  if (resize(r_buffer_, toPut) < 0)
  {
    finish();

    #ifdef THREADS

    unlockRead();

    #endif

    return -1;
  }

  memcpy(r_buffer_.data_.begin() + r_buffer_.start_ + r_buffer_.length_, data, toPut);

  r_buffer_.length_ += toPut;

  #if defined(DUMP) && defined(PARENT)

  *logofs << "AgentTransport: Parent: Dumping content of enqueued data.\n"
          << logofs_flush;

  DumpData(data, toPut);

  #endif

  #if defined(PARENT) && defined(TEST)
  *logofs << "AgentTransport: Parent: Read buffer for FD#" << fd_
          << " has now data for " << r_buffer_.length_
          << " bytes.\n" << logofs_flush;

  *logofs << "AgentTransport: Parent: Start is " << r_buffer_.start_ 
          << " length is " << r_buffer_.length_ << " size is "
          << r_buffer_.data_.size() << " capacity is "
          << r_buffer_.data_.capacity() << ".\n"
          << logofs_flush;
  #endif

  #ifdef THREADS

  unlockRead();

  #endif

  return toPut;
}

int AgentTransport::dequeue(char *data, int size)
{
  #ifdef THREADS

  lockWrite();

  #endif

  if (w_buffer_.length_ == 0)
  {
    #if defined(PARENT) && defined(TEST)
    *logofs << "AgentTransport: Parent: No data can be gotten "
            << "from write buffer for FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

    ESET(EAGAIN);

    #ifdef THREADS

    unlockWrite();

    #endif

    return -1;
  }

  //
  // Return as many bytes as possible.
  //

  int toGet;

  toGet = ((int) size > w_buffer_.length_ ? w_buffer_.length_ : size);

  #if defined(PARENT) && defined(TEST)
  *logofs << "AgentTransport: Parent: Going to get " << toGet
          << " bytes from write buffer for FD#" << fd_ << ".\n"
          << logofs_flush;
  #endif

  memcpy(data, w_buffer_.data_.begin() + w_buffer_.start_, toGet);

  w_buffer_.start_  += toGet;
  w_buffer_.length_ -= toGet;

  #if defined(DUMP) && defined(PARENT)

  *logofs << "AgentTransport: Parent: Dumping content of dequeued data.\n"
          << logofs_flush;

  DumpData(data, toGet);

  #endif

  #if defined(PARENT) && defined(TEST)
  *logofs << "AgentTransport: Parent: Write buffer for FD#" << fd_
          << " has now data for " << length() << " bytes.\n"
          << logofs_flush;

  *logofs << "AgentTransport: Parent: Start is " << w_buffer_.start_ 
          << " length is " << w_buffer_.length_ << " size is "
          << w_buffer_.data_.size() << " capacity is "
          << w_buffer_.data_.capacity() << ".\n"
          << logofs_flush;
  #endif

  #ifdef THREADS

  unlockWrite();

  #endif

  return toGet;
}

int AgentTransport::queuable()
{
  #ifdef THREADS

  lockRead();

  #endif

  int queuable = control -> TransportXBufferThreshold -
                     r_buffer_.length_;

  if (queuable < 0)
  {
    queuable = 0;
  }

  #if defined(PARENT) && defined(TEST)
  if (queuable <= 0)
  {
    *logofs << "AgentTransport: Parent: No data is queuable "
            << "on read buffer for FD#" << fd_ << ".\n"
            << logofs_flush;
  }
  #endif

  #ifdef THREADS

  unlockRead();

  #endif

  return queuable;
}

int AgentTransport::dequeuable()
{
  #ifdef THREADS

  lockWrite();

  #endif

  #if defined(PARENT) && defined(TEST)
  if (w_buffer_.length_ == 0)
  {
    *logofs << "AgentTransport: Parent: No data is dequeuable "
            << "on read buffer for FD#" << fd_ << ".\n"
            << logofs_flush;
  }
  #endif

  #ifdef THREADS

  unlockWrite();

  #endif

  return w_buffer_.length_;
}

#ifdef THREADS

int AgentTransport::lockRead()
{
  for (;;)
  {
    int result = pthread_mutex_lock(&m_read_);

    if (result == 0)
    {
      #ifdef DEBUG
      *logofs << "AgentTransport: Read mutex locked by thread id "
              << pthread_self() << ".\n" << logofs_flush;
      #endif

      return 0;
    }
    else if (EGET() == EINTR)
    {
      continue;
    }
    else
    {
      #ifdef WARNING
      *logofs << "AgentTransport: WARNING! Locking of read mutex by thread id "
              << pthread_self() << " returned " << result << ". Error is '"
              << ESTR() << "'.\n" << logofs_flush;
      #endif

      return result;
    }
  }
}

int AgentTransport::lockWrite()
{
  for (;;)
  {
    int result = pthread_mutex_lock(&m_write_);

    if (result == 0)
    {
      #ifdef DEBUG
      *logofs << "AgentTransport: Write mutex locked by thread id "
              << pthread_self() << ".\n" << logofs_flush;
      #endif

      return 0;
    }
    else if (EGET() == EINTR)
    {
      continue;
    }
    else
    {
      #ifdef WARNING
      *logofs << "AgentTransport: WARNING! Locking of write mutex by thread id "
              << pthread_self() << " returned " << result << ". Error is '"
              << ESTR() << "'.\n" << logofs_flush;
      #endif

      return result;
    }
  }
}

int AgentTransport::unlockRead()
{
  for (;;)
  {
    int result = pthread_mutex_unlock(&m_read_);

    if (result == 0)
    {
      #ifdef DEBUG
      *logofs << "AgentTransport: Read mutex unlocked by thread id "
              << pthread_self() << ".\n" << logofs_flush;
      #endif

      return 0;
    }
    else if (EGET() == EINTR)
    {
      continue;
    }
    else
    {
      #ifdef WARNING
      *logofs << "AgentTransport: WARNING! Unlocking of read mutex by thread id "
              << pthread_self() << " returned " << result << ". Error is '"
              << ESTR() << "'.\n" << logofs_flush;
      #endif

      return result;
    }
  }
}

int AgentTransport::unlockWrite()
{
  for (;;)
  {
    int result = pthread_mutex_unlock(&m_write_);

    if (result == 0)
    {
      #ifdef DEBUG
      *logofs << "AgentTransport: Write mutex unlocked by thread id "
              << pthread_self() << ".\n" << logofs_flush;
      #endif

      return 0;
    }
    else if (EGET() == EINTR)
    {
      continue;
    }
    else
    {
      #ifdef WARNING
      *logofs << "AgentTransport: WARNING! Unlocking of write mutex by thread id "
              << pthread_self() << " returned " << result << ". Error is '"
              << ESTR() << "'.\n" << logofs_flush;
      #endif

      return result;
    }
  }
}

#endif
